<?php

namespace App\Console\Commands;

use App\Services\SapBusinessPartnersService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class GetSapCustomerByCardCode extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sap:get-customer {cardCode : The CardCode of the customer to retrieve} {--expand= : Optional expand parameter for related data}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Get a specific customer from SAP B1 by CardCode';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle(SapBusinessPartnersService $sapService)
    {
        $cardCode = $this->argument('cardCode');
        $expand = $this->option('expand');
        
        $this->info("Fetching customer with CardCode: {$cardCode}");
        
        try {
            // Get the business partner
            $customer = $sapService->getBusinessPartner($cardCode, $expand);
            
            if (!$customer) {
                $this->error("Customer with CardCode '{$cardCode}' not found.");
                return 1;
            }
            
            // Check if it's a customer type
            if (isset($customer['CardType']) && $customer['CardType'] !== 'cCustomer') {
                $this->warn("Note: This CardCode belongs to a {$customer['CardType']} (not a customer).");
            }
            
            // Display customer information
            $this->info("\n=== Customer Details ===");
            $this->displayCustomerInfo($customer);
            
            // Optionally save raw JSON to file
        
            
            $sapService->logout();
            return 0;
            
        } catch (\Exception $e) {
            $this->error("Failed to retrieve customer: " . $e->getMessage());
            Log::error("SAP Get Customer Failed for CardCode {$cardCode}: " . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Display customer information in a formatted way
     */
    protected function displayCustomerInfo($customer)
    {
        // Basic Information
        $this->line("<info>Basic Information:</info>");
        $this->table(
            ['Field', 'Value'],
            [
                ['CardCode', $customer['CardCode'] ?? 'N/A'],
                ['CardName', $customer['CardName'] ?? 'N/A'],
                ['CardType', $customer['CardType'] ?? 'N/A'],
                ['GroupCode', $customer['GroupCode'] ?? 'N/A'],
                ['Currency', $customer['Currency'] ?? 'N/A'],
                ['Federal Tax ID', $customer['FederalTaxID'] ?? 'N/A'],
            ]
        );
        
        // Contact Information
        $this->line("\n<info>Contact Information:</info>");
        $this->table(
            ['Field', 'Value'],
            [
                ['Phone 1', $customer['Phone1'] ?? 'N/A'],
                ['Phone 2', $customer['Phone2'] ?? 'N/A'],
                ['Cellular', $customer['Cellular'] ?? 'N/A'],
                ['Fax', $customer['Fax'] ?? 'N/A'],
                ['Email', $customer['EmailAddress'] ?? 'N/A'],
                ['Website', $customer['Website'] ?? 'N/A'],
            ]
        );
        
        // Address Information
        $this->line("\n<info>Address Information:</info>");
        $this->table(
            ['Field', 'Value'],
            [
                ['Address', $customer['Address'] ?? 'N/A'],
                ['Street', $customer['StreetNo'] ?? 'N/A'],
                ['Block', $customer['Block'] ?? 'N/A'],
                ['City', $customer['City'] ?? 'N/A'],
                ['County/State', $customer['County'] ?? 'N/A'],
                ['Country', $customer['Country'] ?? 'N/A'],
                ['Zip Code', $customer['ZipCode'] ?? 'N/A'],
            ]
        );
        
        // Financial Information
        $this->line("\n<info>Financial Information:</info>");
        $this->table(
            ['Field', 'Value'],
            [
                ['Credit Limit', number_format($customer['CreditLimit'] ?? 0, 2)],
                ['Current Balance', number_format($customer['CurrentAccountBalance'] ?? 0, 2)],
                ['Total Orders Balance', number_format($customer['OpenOrdersBalance'] ?? 0, 2)],
                ['Total Deliveries Balance', number_format($customer['OpenDeliveryNotesBalance'] ?? 0, 2)],
                ['Price List', $customer['PriceListNum'] ?? 'N/A'],
                ['Payment Terms', $customer['PaymentGroupCode'] ?? 'N/A'],
            ]
        );
        
        // Sales Information
        $this->line("\n<info>Sales Information:</info>");
        $this->table(
            ['Field', 'Value'],
            [
                ['Sales Person Code', $customer['SalesPersonCode'] ?? 'N/A'],
                ['Territory', $customer['Territory'] ?? 'N/A'],
                ['Industry', $customer['Industry'] ?? 'N/A'],
            ]
        );
        
        // System Information
        $this->line("\n<info>System Information:</info>");
        $this->table(
            ['Field', 'Value'],
            [
                ['Valid', ($customer['Valid'] ?? 'N') == 'Y' ? 'Yes' : 'No'],
                ['Frozen', ($customer['Frozen'] ?? 'N') == 'Y' ? 'Yes' : 'No'],
                ['Create Date', $customer['CreateDate'] ?? 'N/A'],
                ['Update Date', $customer['UpdateDate'] ?? 'N/A'],
            ]
        );
        
        // Additional Details
        if (!empty($customer['Notes'])) {
            $this->line("\n<info>Notes:</info>");
            $this->line($customer['Notes']);
        }
        
        // Show contact persons if expanded
        if (isset($customer['ContactEmployees']) && is_array($customer['ContactEmployees'])) {
            $this->line("\n<info>Contact Persons:</info>");
            foreach ($customer['ContactEmployees'] as $contact) {
                $name = $contact['Name'] ?? 'N/A';
                $position = $contact['Position'] ?? 'N/A';
                $phone = $contact['Phone1'] ?? 'N/A';
                $email = $contact['E_Mail'] ?? 'N/A';
                $this->line("- {$name} ({$position}) - {$phone} - {$email}");
            }
        }
        
        // Show addresses if expanded
        if (isset($customer['BPAddresses']) && is_array($customer['BPAddresses'])) {
            $this->line("\n<info>Additional Addresses:</info>");
            foreach ($customer['BPAddresses'] as $address) {
                $type = $address['AddressType'] ?? 'N/A';
                $street = $address['Street'] ?? '';
                $city = $address['City'] ?? '';
                $country = $address['Country'] ?? '';
                $this->line("- {$type}: {$street} {$city} {$country}");
            }
        }
    }
}