<?php

namespace App\Console\Commands;

use App\Services\SapItemsService;
use App\Services\SapBusinessPartnersService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class SapSyncBatched extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sap:sync-batched 
                            {type : Type of sync - contacts, products, or all}
                            {--full : Perform a full sync, overwriting existing data}
                            {--batch-size=100 : Number of records to process in each batch}
                            {--max-batches=0 : Maximum number of batches to process (0 = unlimited)}
                            {--delay=1 : Delay in seconds between batches}
                            {--resume : Resume from last cursor position}
                            {--reset-cursor : Reset cursor before starting}
                            {--progress : Show progress bar}
                            {--customers-only : Sync customers only (for contacts)}
                            {--suppliers-only : Sync suppliers only (for contacts)}
                            {--test=0 : Test mode - limit total records}
                            {--details : Show detailed progress}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Enhanced SAP sync command with batched processing to handle large datasets';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $type = $this->argument('type');
        $allowedTypes = ['contacts', 'products', 'all'];
        
        if (!in_array($type, $allowedTypes)) {
            $this->error("Invalid type. Allowed types: " . implode(', ', $allowedTypes));
            return 1;
        }

        try {
            $this->info("🚀 بدء مزامنة SAP المحسّنة");
            $this->info("النوع: " . $this->getTypeLabel($type));
            $this->info("حجم الدفعة: " . $this->option('batch-size'));
            $this->info("مزامنة كاملة: " . ($this->option('full') ? 'نعم' : 'لا'));
            
            if ($this->option('reset-cursor')) {
                $this->resetCursors($type);
            }
            
            $startTime = microtime(true);
            
            switch ($type) {
                case 'contacts':
                    return $this->syncContacts();
                case 'products':
                    return $this->syncProducts();
                case 'all':
                    $contactsResult = $this->syncContacts();
                    if ($contactsResult !== 0) return $contactsResult;
                    return $this->syncProducts();
                default:
                    return 1;
            }
            
        } catch (\Exception $e) {
            $this->error("خطأ في المزامنة: " . $e->getMessage());
            Log::error("SAP Batched Sync Error: " . $e->getMessage());
            return 1;
        }
    }

    /**
     * Sync contacts in batches
     */
    protected function syncContacts()
    {
        $this->info("\n📞 بدء مزامنة جهات الاتصال...");
        
        $sapService = app(SapBusinessPartnersService::class);
        $batchSize = min($this->option('batch-size'), 200);
        $maxBatches = $this->option('max-batches');
        $delay = $this->option('delay');
        $testMode = $this->option('test');
        $customersOnly = $this->option('customers-only');
        $suppliersOnly = $this->option('suppliers-only');
        
        $types = [];
        if ($customersOnly) {
            $types = ['customer'];
        } elseif ($suppliersOnly) {
            $types = ['supplier'];
        } else {
            $types = ['customer', 'supplier'];
        }
        
        $totalStats = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0,
            'batches' => 0
        ];
        
        foreach ($types as $type) {
            $this->info("\n🔄 معالجة {$this->getContactTypeLabel($type)}...");
            
            $stats = $this->processContactBatches($sapService, $type, $batchSize, $maxBatches, $delay, $testMode);
            
            $totalStats['total'] += $stats['total'];
            $totalStats['created'] += $stats['created'];
            $totalStats['updated'] += $stats['updated'];
            $totalStats['failed'] += $stats['failed'];
            $totalStats['skipped'] += $stats['skipped'];
            $totalStats['batches'] += $stats['batches'];
            
            if ($testMode && $totalStats['total'] >= $testMode) {
                $this->warn("وصل إلى حد وضع الاختبار ({$testMode})");
                break;
            }
        }
        
        $this->displayFinalStats('جهات الاتصال', $totalStats);
        return 0;
    }

    /**
     * Sync products in batches
     */
    protected function syncProducts()
    {
        $this->info("\n📦 بدء مزامنة المنتجات...");
        
        $sapService = app(SapItemsService::class);
        $batchSize = min($this->option('batch-size'), 100);
        $maxBatches = $this->option('max-batches');
        $delay = $this->option('delay');
        $testMode = $this->option('test');
        
        $stats = $this->processProductBatches($sapService, $batchSize, $maxBatches, $delay, $testMode);
        
        $this->displayFinalStats('المنتجات', $stats);
        return 0;
    }

    /**
     * Process contact batches
     */
    protected function processContactBatches($sapService, $type, $batchSize, $maxBatches, $delay, $testMode)
    {
        $stats = ['total' => 0, 'created' => 0, 'updated' => 0, 'failed' => 0, 'skipped' => 0, 'batches' => 0];
        
        $cursorKey = "sap_sync_cursor_{$type}";
        $lastCardCode = $this->option('resume') ? Cache::get($cursorKey, '') : '';
        
        $baseFilter = $type === 'customer' ? "CardType eq 'cCustomer'" : "CardType eq 'cSupplier'";
        $batchCount = 0;
        
        $progressBar = $this->option('progress') ? $this->output->createProgressBar(0) : null;
        
        while (true) {
            $batchCount++;
            
            if ($maxBatches > 0 && $batchCount > $maxBatches) {
                $this->info("وصل إلى الحد الأقصى للدفعات ({$maxBatches})");
                break;
            }
            
            try {
                // بناء الفلتر
                $filter = $baseFilter;
                if ($lastCardCode) {
                    $filter .= " and CardCode gt '{$lastCardCode}'";
                }
                
                $queryParams = [
                    '$filter' => $filter,
                    '$orderby' => 'CardCode asc',
                    '$top' => $batchSize,
                    '$select' => 'CardCode,CardName,CardType,GroupCode,Address,Phone1,Phone2,EmailAddress,UpdateDate'
                ];
                
                                 if ($this->option('details')) {
                     $this->line("الدفعة {$batchCount}: Filter = {$filter}");
                 }
                
                $result = $sapService->get('BusinessPartners', $queryParams, 60000);
                
                if (empty($result['value'])) {
                    $this->info("انتهت البيانات للنوع {$type}");
                    break;
                }
                
                $batchData = $result['value'];
                $batchTotal = count($batchData);
                
                $this->line("الدفعة {$batchCount}: معالجة {$batchTotal} " . $this->getContactTypeLabel($type));
                
                // معالجة كل عنصر في الدفعة
                foreach ($batchData as $partner) {
                    try {
                        $processResult = $sapService->processSapPartner($partner, $type);
                        
                        switch ($processResult) {
                            case 'created':
                                $stats['created']++;
                                break;
                            case 'updated':
                                $stats['updated']++;
                                break;
                            case 'skipped':
                                $stats['skipped']++;
                                break;
                        }
                        
                        $stats['total']++;
                        
                        if ($progressBar) {
                            $progressBar->advance();
                        }
                        
                        // اختبار الحد الأقصى
                        if ($testMode && $stats['total'] >= $testMode) {
                            break 2;
                        }
                        
                    } catch (\Exception $e) {
                        $stats['failed']++;
                                                 if ($this->option('details')) {
                             $this->error("خطأ في معالجة {$partner['CardCode']}: " . $e->getMessage());
                         }
                    }
                }
                
                // تحديث المؤشر
                $lastCardCode = end($batchData)['CardCode'];
                Cache::put($cursorKey, $lastCardCode, 3600); // حفظ لساعة واحدة
                
                $stats['batches']++;
                
                // تأخير بين الدفعات
                if ($delay > 0 && $batchData) {
                    $this->line("تأخير {$delay} ثانية...");
                    sleep($delay);
                }
                
                // التحقق من انتهاء البيانات يعتمد على nextLink وليس حجم الدفعة
                if (!isset($result['odata.nextLink'])) {
                    $this->info("انتهت البيانات - لا يوجد nextLink");
                    if ($this->option('details')) {
                        $this->line("حجم الدفعة: {$batchTotal}/{$batchSize}");
                    }
                    break;
                } else {
                    if ($this->option('details')) {
                        $this->line("يوجد المزيد من البيانات - nextLink موجود");
                    }
                }
                
            } catch (\Exception $e) {
                $this->error("خطأ في الدفعة {$batchCount}: " . $e->getMessage());
                
                // في حالة الخطأ، نحاول المتابعة
                if ($batchCount > 3) {
                    break;
                }
            }
        }
        
        if ($progressBar) {
            $progressBar->finish();
            $this->line('');
        }
        
        // مسح المؤشر عند الانتهاء
        Cache::forget($cursorKey);
        
        return $stats;
    }

    /**
     * Process product batches
     */
    protected function processProductBatches($sapService, $batchSize, $maxBatches, $delay, $testMode)
    {
        $stats = ['total' => 0, 'created' => 0, 'updated' => 0, 'failed' => 0, 'skipped' => 0, 'batches' => 0];
        
        $cursorKey = "sap_sync_cursor_products";
        $lastItemCode = $this->option('resume') ? Cache::get($cursorKey, '') : '';
        
        $batchCount = 0;
        $progressBar = $this->option('progress') ? $this->output->createProgressBar(0) : null;
        
        while (true) {
            $batchCount++;
            
            if ($maxBatches > 0 && $batchCount > $maxBatches) {
                $this->info("وصل إلى الحد الأقصى للدفعات ({$maxBatches})");
                break;
            }
            
            try {
                // بناء الفلتر
                $filter = null;
                if ($lastItemCode) {
                    $filter = "ItemCode gt '{$lastItemCode}'";
                }
                
                $queryParams = [
                    '$orderby' => 'ItemCode asc',
                    '$top' => $batchSize,
                    '$select' => 'ItemCode,ItemName,ItemType,ItemsGroupCode,InventoryItem,UpdateDate'
                ];
                
                if ($filter) {
                    $queryParams['$filter'] = $filter;
                }
                
                                 if ($this->option('details')) {
                     $this->line("دفعة المنتجات {$batchCount}: Filter = " . ($filter ?: 'none'));
                 }
                
                $result = $sapService->get('Items', $queryParams, 60000);
                
                if (empty($result['value'])) {
                    $this->info("انتهت البيانات للمنتجات");
                    break;
                }
                
                $batchData = $result['value'];
                $batchTotal = count($batchData);
                
                $this->line("دفعة المنتجات {$batchCount}: معالجة {$batchTotal} منتج");
                
                // معالجة كل منتج في الدفعة
                foreach ($batchData as $item) {
                    try {
                        $processResult = $sapService->processSapItem($item, $this->option('full'));
                        
                        switch ($processResult) {
                            case 'created':
                                $stats['created']++;
                                break;
                            case 'updated':
                                $stats['updated']++;
                                break;
                            case 'skipped':
                                $stats['skipped']++;
                                break;
                        }
                        
                        $stats['total']++;
                        
                        if ($progressBar) {
                            $progressBar->advance();
                        }
                        
                        // اختبار الحد الأقصى
                        if ($testMode && $stats['total'] >= $testMode) {
                            break 2;
                        }
                        
                    } catch (\Exception $e) {
                        $stats['failed']++;
                                                 if ($this->option('details')) {
                             $this->error("خطأ في معالجة {$item['ItemCode']}: " . $e->getMessage());
                         }
                    }
                }
                
                // تحديث المؤشر
                $lastItemCode = end($batchData)['ItemCode'];
                Cache::put($cursorKey, $lastItemCode, 3600);
                
                $stats['batches']++;
                
                // تأخير بين الدفعات
                if ($delay > 0 && $batchData) {
                    $this->line("تأخير {$delay} ثانية...");
                    sleep($delay);
                }
                
                // التحقق من انتهاء البيانات يعتمد على nextLink وليس حجم الدفعة
                if (!isset($result['odata.nextLink'])) {
                    $this->info("انتهت البيانات - لا يوجد nextLink");
                    if ($this->option('details')) {
                        $this->line("حجم الدفعة: {$batchTotal}/{$batchSize}");
                    }
                    break;
                } else {
                    if ($this->option('details')) {
                        $this->line("يوجد المزيد من البيانات - nextLink موجود");
                    }
                }
                
            } catch (\Exception $e) {
                $this->error("خطأ في دفعة المنتجات {$batchCount}: " . $e->getMessage());
                
                if ($batchCount > 3) {
                    break;
                }
            }
        }
        
        if ($progressBar) {
            $progressBar->finish();
            $this->line('');
        }
        
        // مسح المؤشر عند الانتهاء
        Cache::forget($cursorKey);
        
        return $stats;
    }

    /**
     * Reset cursors
     */
    protected function resetCursors($type)
    {
        if ($type === 'all' || $type === 'contacts') {
            Cache::forget('sap_sync_cursor_customer');
            Cache::forget('sap_sync_cursor_supplier');
            $this->info("✓ تم إعادة تعيين مؤشرات جهات الاتصال");
        }
        
        if ($type === 'all' || $type === 'products') {
            Cache::forget('sap_sync_cursor_products');
            $this->info("✓ تم إعادة تعيين مؤشرات المنتجات");
        }
    }

    /**
     * Display final statistics
     */
    protected function displayFinalStats($type, $stats)
    {
        $this->info("\n" . str_repeat('=', 50));
        $this->info("📊 إحصائيات نهائية - {$type}");
        $this->info(str_repeat('=', 50));
        $this->info("إجمالي المعالج: " . number_format($stats['total']));
        $this->info("تم إنشاؤها: " . number_format($stats['created']));
        $this->info("تم تحديثها: " . number_format($stats['updated']));
        $this->info("تم تخطيها: " . number_format($stats['skipped']));
        $this->info("فشل في: " . number_format($stats['failed']));
        $this->info("عدد الدفعات: " . number_format($stats['batches']));
        
        if ($stats['failed'] > 0) {
            $this->warn("⚠️  يوجد {$stats['failed']} أخطاء - راجع ملف الـ logs");
        }
        
        $this->info("✅ اكتملت مزامنة {$type} بنجاح!");
    }

    /**
     * Get type label in Arabic
     */
    protected function getTypeLabel($type)
    {
        $labels = [
            'contacts' => 'جهات الاتصال',
            'products' => 'المنتجات',
            'all' => 'جميع البيانات'
        ];
        
        return $labels[$type] ?? $type;
    }

    /**
     * Get contact type label in Arabic
     */
    protected function getContactTypeLabel($type)
    {
        $labels = [
            'customer' => 'العملاء',
            'supplier' => 'الموردون'
        ];
        
        return $labels[$type] ?? $type;
    }
} 