<?php

namespace App\Console\Commands;

use App\Transaction;
use App\Jobs\SyncTransactionToSap;
use App\Services\SapInvoicesService;
use Illuminate\Console\Command;

class SyncPendingTransactionsToSap extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sap:sync-transactions 
                            {--business_id= : The ID of the business to sync transactions for}
                            {--limit=50 : Maximum number of transactions to process}
                            {--days=7 : Only process transactions from the last X days}
                            {--force : Force sync even for already synced transactions}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync pending transactions (sales and quotations) to SAP B1';

    /**
     * The SAP Invoices Service
     * 
     * @var \App\Services\SapInvoicesService
     */
    protected $sapService;

    /**
     * Create a new command instance.
     *
     * @param  \App\Services\SapInvoicesService  $sapService
     * @return void
     */
    public function __construct(SapInvoicesService $sapService)
    {
        parent::__construct();
        $this->sapService = $sapService;
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        // Check if SAP integration is enabled
        if (!config('sap.enabled', false)) {
            $this->error('SAP integration is not enabled in configuration.');
            return 1;
        }

        $businessId = $this->option('business_id');
        $limit = (int) $this->option('limit');
        $days = (int) $this->option('days');
        $force = $this->option('force');

        $this->info("Starting SAP transaction sync...");
        
        // Build the query
        $query = Transaction::where(function($q) {
                $q->where('status', 'final')
                  ->orWhere('is_quotation', 1)
                  ->orWhere('sub_status', 'quotation');
            })
            ->where('created_at', '>=', now()->subDays($days));
            
        if (!$force) {
            $query->where(function($q) {
                $q->whereNull('sap_sync_status')
                  ->orWhere('sap_sync_status', 'failed');
            });
        }
        
        if ($businessId) {
            $query->where('business_id', $businessId);
        }
        
        $count = $query->count();
        $this->info("Found {$count} transactions to process. Processing up to {$limit}...");
        
        // Process transactions
        $transactions = $query->limit($limit)->get();
        $bar = $this->output->createProgressBar(count($transactions));
        $bar->start();
        
        $processed = 0;
        $queued = 0;
        
        foreach ($transactions as $transaction) {
            if (config('sap.use_queue', true)) {
                // Pass the transaction ID instead of the entire object
                SyncTransactionToSap::dispatch($transaction->id);
                $queued++;
            } else {
                try {
                    // Process synchronously
                    $result = $this->sapService->pushTransactionToSap($transaction);
                    if ($result['success']) {
                        $processed++;
                    } else {
                        $this->error("Error processing transaction #{$transaction->id}: {$result['message']}");
                    }
                } catch (\Exception $e) {
                    $this->error("Error processing transaction #{$transaction->id}: {$e->getMessage()}");
                }
            }
            
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine();
        
        if (config('sap.use_queue', true)) {
            $this->info("Queued {$queued} transactions for processing.");
        } else {
            $this->info("Successfully processed {$processed} transactions.");
        }
        
        return 0;
    }
}