<?php

namespace App\Console\Commands;

use App\Services\SapBusinessPartnersService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SyncSapContacts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sap:sync-contacts 
                            {--full : Perform a full sync, overwriting existing data}
                            {--batch-size=200 : Number of records to process in each batch}
                            {--timeout=120 : Timeout in seconds for each API call}
                            {--test : Test mode - process only first 10 contacts}
                            {--check-connection : Only test SAP connection}
                            {--stats : Show sync statistics only}
                            {--customers-only : Sync customers only}
                            {--suppliers-only : Sync suppliers only}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync contacts from SAP B1 to local database with enhanced options';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle(SapBusinessPartnersService $sapService)
    {
        try {
            // Test connection only
            if ($this->option('check-connection')) {
                return $this->checkConnection($sapService);
            }
            
            // Show statistics only
            if ($this->option('stats')) {
                return $this->showStats();
            }
            
            $this->info('بدء مزامنة جهات الاتصال من SAP...');
            
            $isFullSync = $this->option('full');
            $batchSize = $this->option('batch-size');
            $timeout = $this->option('timeout');
            $testMode = $this->option('test');
            $customersOnly = $this->option('customers-only');
            $suppliersOnly = $this->option('suppliers-only');
            
            if ($isFullSync) {
                $this->info("إجراء مزامنة كاملة كما طُلب.");
            } else {
                $this->info("إجراء مزامنة تدريجية.");
            }
            
            $this->info("إعدادات المزامنة:");
            $this->info("- حجم الدفعة: {$batchSize}");
            $this->info("- مهلة الاتصال: {$timeout} ثانية");
            
            if ($testMode) {
                $this->info("وضع الاختبار: معالجة أول 10 جهات اتصال فقط");
            }
            
            if ($customersOnly) {
                $this->info("مزامنة العملاء فقط");
                $stats = $sapService->syncCustomersOnly($isFullSync);
            } elseif ($suppliersOnly) {
                $this->info("مزامنة الموردين فقط");
                $stats = $sapService->syncSuppliersOnly($isFullSync);
            } else {
                $stats = $sapService->syncAllContacts($isFullSync);
            }
            
            $this->info("انتهت المزامنة.");
            $this->info("إجمالي جهات الاتصال: {$stats['total']}");
            $this->info("تم إنشاؤها: {$stats['created']}");
            $this->info("تم تحديثها: {$stats['updated']}");
            $this->info("تم تخطيها: {$stats['skipped']}");
            $this->info("فشل في: {$stats['failed']}");
            
            // Display additional info if available
            if (isset($stats['batches_processed'])) {
                $this->info("عدد الدفعات المعالجة: {$stats['batches_processed']}");
            }
            
            $sapService->logout();
            return 0;
        } catch (\Exception $e) {
            $this->error("فشلت المزامنة: " . $e->getMessage());
            Log::error("SAP Contact Sync Failed: " . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Check SAP connection
     */
    private function checkConnection($sapService)
    {
        try {
            $this->info('اختبار الاتصال بـ SAP...');
            
            // Try to get a small sample of data
            $result = $sapService->getBusinessPartners(null, 'CardCode,CardName', 1, 0, 30000);
            
            if (!empty($result) && !empty($result['value'])) {
                $this->info('✓ الاتصال بـ SAP ناجح');
                $this->info("عينة من البيانات: " . $result['value'][0]['CardName']);
                return 0;
            } else {
                $this->error('✗ لا توجد بيانات من SAP');
                return 1;
            }
        } catch (\Exception $e) {
            $this->error('✗ فشل الاتصال بـ SAP: ' . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Show sync statistics
     */
    private function showStats()
    {
        try {
            $this->info('إحصائيات مزامنة جهات الاتصال:');
            
            $totalContacts = \App\Contact::count();
            $syncedContacts = \App\Contact::whereNotNull('card_code')->count();
            $lastSync = \App\Contact::whereNotNull('sap_last_sync')
                ->orderBy('sap_last_sync', 'desc')
                ->first();
            
            $this->info("إجمالي جهات الاتصال: {$totalContacts}");
            $this->info("جهات الاتصال المتزامنة: {$syncedContacts}");
            $this->info("آخر مزامنة: " . ($lastSync ? $lastSync->sap_last_sync : 'لم تتم مزامنة من قبل'));
            
            // Customer/Supplier breakdown
            $customers = \App\Contact::where('type', 'customer')->count();
            $suppliers = \App\Contact::where('type', 'supplier')->count();
            
            $this->info("العملاء: {$customers}");
            $this->info("الموردون: {$suppliers}");
            
            return 0;
        } catch (\Exception $e) {
            $this->error('خطأ في جلب الإحصائيات: ' . $e->getMessage());
            return 1;
        }
    }
}