<?php

namespace App\Console\Commands;

use App\Services\SapItemsService;
use App\SapWarehouseMapping;
use App\SapUomMapping;
use App\SapProductWarehouse;
use App\Product;
use App\Unit;
use App\BusinessLocation;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class SyncSapProducts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sap:sync-products 
                            {--full : Perform a full sync, overwriting existing data}
                            {--item-code= : Sync specific item code only}
                            {--test : Test mode - process only first 10 items}
                            {--check-connection : Only test SAP connection}
                            {--sync-warehouses : Sync warehouses only}
                            {--sync-uom : Sync UoM groups only}
                            {--map-uom : Map UoM to local units}
                            {--map-warehouses : Map warehouses to locations}
                            {--cleanup : Clean up old sync data}
                            {--stats : Show sync statistics only}
                            {--verify : Verify sync integrity}
                            {--fix-quantities : Fix quantity mismatches}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync products from SAP B1 to local database with various options';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle(SapItemsService $sapService)
    {
        try {
            // Test connection only
            if ($this->option('check-connection')) {
                return $this->checkConnection($sapService);
            }
            
            // Show statistics only
            if ($this->option('stats')) {
                return $this->showStats($sapService);
            }
            
            // Cleanup old data
            if ($this->option('cleanup')) {
                return $this->cleanupOldData($sapService);
            }
            
            // Sync warehouses only
            if ($this->option('sync-warehouses')) {
                return $this->syncWarehouses($sapService);
            }
            
            // Sync UoM groups only
            if ($this->option('sync-uom')) {
                return $this->syncUomGroups($sapService);
            }
            
            // Map UoM to local units
            if ($this->option('map-uom')) {
                return $this->mapUomToUnits();
            }
            
            // Map warehouses to locations
            if ($this->option('map-warehouses')) {
                return $this->mapWarehousesToLocations();
            }
            
            // Verify sync integrity
            if ($this->option('verify')) {
                return $this->verifySyncIntegrity();
            }
            
            // Fix quantity mismatches
            if ($this->option('fix-quantities')) {
                return $this->fixQuantityMismatches();
            }
            
            // Sync specific item
            if ($this->option('item-code')) {
                return $this->syncSingleItem($sapService);
            }
            
            // Main sync process
            $this->info('بدء مزامنة المنتجات من SAP B1...');
            
            $isFullSync = $this->option('full');
            $testMode = $this->option('test');
            
            if ($testMode) {
                $this->info('وضع الاختبار: معالجة أول 10 منتجات فقط');
            }
            
            if ($isFullSync) {
                $this->info("إجراء مزامنة كاملة كما طُلب.");
            } else {
                $this->info("إجراء مزامنة تدريجية.");
            }
            
            $stats = $sapService->syncAllProducts($isFullSync);
            
            $this->info("انتهت المزامنة بنجاح.");
            $this->displayStats($stats);
            
            $sapService->logout();
            return 0;
            
        } catch (\Exception $e) {
            $this->error("فشلت مزامنة المنتجات: " . $e->getMessage());
            Log::error("SAP Product Sync Failed: " . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Display sync statistics
     */
    private function displayStats($stats)
    {
        $this->info("إحصائيات المزامنة:");
        $this->info("إجمالي المنتجات: {$stats['total']}");
        $this->info("تم إنشاؤها: {$stats['created']}");
        $this->info("تم تحديثها: {$stats['updated']}");
        $this->info("تم تخطيها: {$stats['skipped']}");
        $this->info("فشل في: {$stats['failed']}");
        
        if (isset($stats['batches_processed'])) {
            $this->info("عدد الدفعات المعالجة: {$stats['batches_processed']}");
        }
        
        if (isset($stats['warehouses_synced'])) {
            $this->info("المستودعات المتزامنة: {$stats['warehouses_synced']}");
        }
        
        if (isset($stats['uom_groups_synced'])) {
            $this->info("مجموعات وحدات القياس المتزامنة: {$stats['uom_groups_synced']}");
        }
    }
    
    /**
     * Check SAP connection
     */
    private function checkConnection($sapService)
    {
        try {
            $this->info('اختبار الاتصال بـ SAP...');
            
            // Try to get a small sample of items
            $result = $sapService->getItems(null, 'ItemCode,ItemName', 1, 0, 30);
            
            if (!empty($result) && !empty($result['value'])) {
                $this->info('✓ الاتصال بـ SAP ناجح');
                $this->info("عينة من البيانات: " . $result['value'][0]['ItemName']);
                return 0;
            } else {
                $this->error('✗ لا توجد بيانات من SAP');
                return 1;
            }
        } catch (\Exception $e) {
            $this->error('✗ فشل الاتصال بـ SAP: ' . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Show sync statistics
     */
    private function showStats($sapService)
    {
        try {
            $this->info('إحصائيات مزامنة المنتجات:');
            
            $totalProducts = Product::count();
            $syncedProducts = Product::whereNotNull('sap_item_code')->count();
            $lastSync = Product::whereNotNull('sap_last_sync')
                ->orderBy('sap_last_sync', 'desc')
                ->first();
            
            $this->info("إجمالي المنتجات: {$totalProducts}");
            $this->info("المنتجات المتزامنة: {$syncedProducts}");
            $this->info("آخر مزامنة: " . ($lastSync ? $lastSync->sap_last_sync : 'لم تتم مزامنة من قبل'));
            
            // Warehouse mappings
            $warehouseMappings = SapWarehouseMapping::count();
            $this->info("خرائط المستودعات: {$warehouseMappings}");
            
            // UoM mappings
            $uomMappings = SapUomMapping::count();
            $this->info("خرائط وحدات القياس: {$uomMappings}");
            
            return 0;
        } catch (\Exception $e) {
            $this->error('خطأ في جلب الإحصائيات: ' . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Map UoM to local units
     */
    protected function mapUomToUnits()
    {
        $this->info('Mapping SAP UoMs to local units...');
        
        $unmappedUoms = SapUomMapping::whereNull('unit_id')->get();
        
        if ($unmappedUoms->count() == 0) {
            $this->info('✅ All UoMs are already mapped!');
            return 0;
        }
        
        $this->info("Found {$unmappedUoms->count()} unmapped UoMs");
        
        // Get all local units
        $units = Unit::whereNull('deleted_at')->get();
        $unitOptions = ['0' => 'Skip (map later)'];
        foreach ($units as $unit) {
            $unitOptions[$unit->id] = "{$unit->actual_name} ({$unit->short_name})";
        }
        
        $autoMapped = 0;
        $manuallyMapped = 0;
        
        foreach ($unmappedUoms as $uom) {
            $this->info("\n📏 UoM: {$uom->uom_code} - {$uom->uom_name}");
            
            if ($uom->uom_group_entry) {
                $this->info("   Group Entry: {$uom->uom_group_entry}");
            }
            
            if ($uom->is_base_uom) {
                $this->info("   ⭐ Base UoM");
            }
            
            if ($uom->conversion_rate != 1) {
                $this->info("   Conversion Rate: {$uom->conversion_rate}");
            }
            
            // Try auto-mapping first
            $autoMatched = $this->findBestUnitMatch($uom, $units);
            
            if ($autoMatched) {
                $this->info("   🎯 Auto-matched to: {$autoMatched->actual_name} ({$autoMatched->short_name})");
                
                if ($this->confirm("   Accept this mapping?", true)) {
                    $uom->unit_id = $autoMatched->id;
                    $uom->save();
                    $autoMapped++;
                    continue;
                }
            }
            
            // Manual selection
            $selectedId = $this->choice(
                "   Select local unit for '{$uom->uom_code}':",
                $unitOptions,
                0
            );
            
            if ($selectedId > 0) {
                $uom->unit_id = $selectedId;
                $uom->save();
                $manuallyMapped++;
                $this->info("   ✅ Mapped successfully");
            }
        }
        
        $this->info("\n📊 Mapping Summary:");
        $this->info("   • Auto-mapped: {$autoMapped}");
        $this->info("   • Manually mapped: {$manuallyMapped}");
        $this->info("   • Skipped: " . ($unmappedUoms->count() - $autoMapped - $manuallyMapped));
        
        return 0;
    }
    
    /**
     * Find best unit match for UoM
     */
    protected function findBestUnitMatch($uom, $units)
    {
        $uomCode = strtolower($uom->uom_code);
        $uomName = strtolower($uom->uom_name);
        
        // Direct matches
        foreach ($units as $unit) {
            if (strtolower($unit->short_name) == $uomCode ||
                strtolower($unit->actual_name) == $uomCode ||
                strtolower($unit->short_name) == $uomName ||
                strtolower($unit->actual_name) == $uomName) {
                return $unit;
            }
        }
        
        // Pattern matches
        $patterns = [
            'kg|كغم|kilogram' => ['كغم', 'كيلو', 'kg'],
            'g|غم|gram' => ['غم', 'جرام', 'g'],
            'l|لتر|liter|litre' => ['لتر', 'l'],
            'ml|مل|milliliter' => ['مل', 'ml'],
            'ea|each|قطعة' => ['قطعة', 'حبة', 'ea'],
            'box|صندوق' => ['صندوق', 'علبة', 'box'],
            'pack|باكيت' => ['باكيت', 'رزمة', 'pack'],
        ];
        
        foreach ($patterns as $pattern => $keywords) {
            if (preg_match("/{$pattern}/i", $uomCode) || preg_match("/{$pattern}/i", $uomName)) {
                foreach ($units as $unit) {
                    foreach ($keywords as $keyword) {
                        if (stripos($unit->short_name, $keyword) !== false ||
                            stripos($unit->actual_name, $keyword) !== false) {
                            return $unit;
                        }
                    }
                }
            }
        }
        
        return null;
    }
    
    /**
     * Map warehouses to locations
     */
    protected function mapWarehousesToLocations()
    {
        $this->info('Mapping SAP warehouses to business locations...');
        
        $unmappedWarehouses = SapWarehouseMapping::whereNull('location_id')->get();
        
        if ($unmappedWarehouses->count() == 0) {
            $this->info('✅ All warehouses are already mapped!');
            return 0;
        }
        
        $this->info("Found {$unmappedWarehouses->count()} unmapped warehouses");
        
        // Get all locations
        $locations = BusinessLocation::where('is_active', 1)->get();
        $locationOptions = ['0' => 'Skip (map later)'];
        foreach ($locations as $location) {
            $locationOptions[$location->id] = $location->name;
        }
        
        $mapped = 0;
        
        foreach ($unmappedWarehouses as $warehouse) {
            $this->info("\n🏪 Warehouse: {$warehouse->warehouse_code} - {$warehouse->warehouse_name}");
            
            // Show products count in this warehouse
            $productCount = SapProductWarehouse::where('warehouse_code', $warehouse->warehouse_code)
                ->where('in_stock', '>', 0)
                ->count();
            
            if ($productCount > 0) {
                $this->info("   📦 Contains stock for {$productCount} products");
            }
            
            // Try auto-match
            $autoMatched = $this->findBestLocationMatch($warehouse, $locations);
            
            if ($autoMatched) {
                $this->info("   🎯 Auto-matched to: {$autoMatched->name}");
                
                if ($this->confirm("   Accept this mapping?", true)) {
                    $warehouse->location_id = $autoMatched->id;
                    $warehouse->save();
                    $mapped++;
                    continue;
                }
            }
            
            // Manual selection
            $selectedId = $this->choice(
                "   Select location for warehouse '{$warehouse->warehouse_code}':",
                $locationOptions,
                0
            );
            
            if ($selectedId > 0) {
                $warehouse->location_id = $selectedId;
                $warehouse->save();
                $mapped++;
                $this->info("   ✅ Mapped successfully");
            }
        }
        
        $this->info("\n✅ Mapped {$mapped} warehouses");
        
        return 0;
    }
    
    /**
     * Find best location match for warehouse
     */
    protected function findBestLocationMatch($warehouse, $locations)
    {
        $warehouseCode = strtolower($warehouse->warehouse_code);
        $warehouseName = strtolower($warehouse->warehouse_name);
        
        foreach ($locations as $location) {
            $locationName = strtolower($location->name);
            
            // Direct match
            if ($warehouseCode == $locationName || $warehouseName == $locationName) {
                return $location;
            }
            
            // Partial match
            if (stripos($warehouseName, $locationName) !== false ||
                stripos($locationName, $warehouseName) !== false ||
                stripos($warehouseCode, $locationName) !== false) {
                return $location;
            }
        }
        
        return null;
    }
    
    /**
     * Verify sync integrity
     */
    protected function verifySyncIntegrity()
    {
        $this->info('Verifying SAP sync integrity...');
        
        $issues = [];
        
        // Check for products without variations
        $productsWithoutVariations = Product::whereNotNull('sap_item_code')
            ->doesntHave('variations')
            ->count();
        
        if ($productsWithoutVariations > 0) {
            $issues[] = "{$productsWithoutVariations} products without variations";
        }
        
        // Check for quantity mismatches
        $quantityMismatches = Product::whereNotNull('sap_item_code')
            ->whereRaw('quantity != sap_on_hand')
            ->count();
        
        if ($quantityMismatches > 0) {
            $issues[] = "{$quantityMismatches} products with quantity mismatches";
        }
        
        // Check for price inconsistencies
        $priceIssues = Product::whereNotNull('sap_item_code')
            ->where(function($query) {
                $query->whereNull('sale_price')
                    ->orWhere('sale_price', 0)
                    ->orWhereRaw('sale_price != sap_price');
            })
            ->count();
        
        if ($priceIssues > 0) {
            $issues[] = "{$priceIssues} products with price issues";
        }
        
        // Check orphaned warehouse records
        $orphanedWarehouses = DB::table('sap_product_warehouses as spw')
            ->leftJoin('products as p', 'spw.sap_item_code', '=', 'p.sap_item_code')
            ->whereNull('p.id')
            ->count();
        
        if ($orphanedWarehouses > 0) {
            $issues[] = "{$orphanedWarehouses} orphaned warehouse records";
        }
        
        if (count($issues) > 0) {
            $this->warn("\n⚠️  Found the following issues:");
            foreach ($issues as $issue) {
                $this->warn("  • {$issue}");
            }
            
            $this->info("\nRun 'php artisan sap:sync-products --fix-quantities' to fix quantity issues");
        } else {
            $this->info("\n✅ Sync integrity check passed! No issues found.");
        }
        
        return 0;
    }
    
    /**
     * Fix quantity mismatches
     */
    protected function fixQuantityMismatches()
    {
        $this->info('Fixing quantity mismatches...');
        
        $mismatches = Product::whereNotNull('sap_item_code')
            ->whereRaw('quantity != sap_on_hand')
            ->get();
        
        if ($mismatches->count() == 0) {
            $this->info('✅ No quantity mismatches found!');
            return 0;
        }
        
        $this->warn("Found {$mismatches->count()} products with quantity mismatches");
        
        if (!$this->confirm('Do you want to update local quantities to match SAP?')) {
            $this->info('Operation cancelled.');
            return 0;
        }
        
        $progressBar = $this->output->createProgressBar($mismatches->count());
        $progressBar->start();
        
        $fixed = 0;
        
        foreach ($mismatches as $product) {
            try {
                $oldQty = $product->quantity;
                $product->quantity = $product->sap_on_hand;
                $product->save();
                
                // Update variation location details
                $variations = $product->variations;
                foreach ($variations as $variation) {
                    DB::table('variation_location_details')
                        ->where('variation_id', $variation->id)
                        ->update(['qty_available' => $product->sap_on_hand]);
                }
                
                $fixed++;
                Log::info("Fixed quantity for product {$product->id}: {$oldQty} -> {$product->sap_on_hand}");
                
            } catch (\Exception $e) {
                Log::error("Failed to fix quantity for product {$product->id}: " . $e->getMessage());
            }
            
            $progressBar->advance();
        }
        
        $progressBar->finish();
        
        $this->info("\n\n✅ Fixed quantities for {$fixed} products");
        
        return 0;
    }
    
    /**
     * Clean up old sync data
     */
    protected function cleanupOldData(SapItemsService $sapService)
    {
        $this->info('Cleaning up old sync data...');
        
        $days = $this->ask('Delete data older than how many days?', 30);
        
        if ($this->confirm("Are you sure you want to delete sync data older than {$days} days?")) {
            $result = $sapService->cleanupSyncData($days);
            
            $this->info("Cleaned up {$result['warehouse_records_deleted']} old warehouse records");
            $this->info('Cleanup completed successfully!');
        } else {
            $this->info('Cleanup cancelled.');
        }
        
        return 0;
    }
    
    /**
     * Sync warehouses only
     */
    protected function syncWarehouses(SapItemsService $sapService)
    {
        $this->info('Syncing SAP warehouses...');
        
        $count = $sapService->syncWarehouses();
        
        $this->info("Successfully synced {$count} warehouses");
        
        // Show unmapped warehouses
        $unmapped = SapWarehouseMapping::whereNull('location_id')->count();
        if ($unmapped > 0) {
            $this->warn("\n⚠️  {$unmapped} warehouses need to be mapped to locations");
            $this->info("Run 'php artisan sap:sync-products --map-warehouses' to map them");
        }
        
        return 0;
    }
    
    /**
     * Sync UoM groups only
     */
    protected function syncUomGroups(SapItemsService $sapService)
    {
        $this->info('Syncing SAP UoM groups...');
        
        $count = $sapService->syncUomGroups();
        
        $this->info("Successfully synced {$count} UoM mappings");
        
        // Show unmapped UoMs
        $unmapped = SapUomMapping::whereNull('unit_id')->count();
        if ($unmapped > 0) {
            $this->warn("\n⚠️  {$unmapped} UoMs need to be mapped to units");
            $this->info("Run 'php artisan sap:sync-products --map-uom' to map them");
        }
        
        return 0;
    }
    
    /**
     * Sync single item
     */
    protected function syncSingleItem(SapItemsService $sapService)
    {
        $itemCode = $this->option('item-code');
        
        $this->info("Syncing single item: {$itemCode}");
        
        $result = $sapService->syncSingleProduct($itemCode, true);
        
        $this->info("Single item sync result: {$result}");
        
        // Show item details after sync
        $product = Product::where('sap_item_code', $itemCode)->first();
        if ($product) {
            $this->info("\n📦 Product Details:");
            $this->table(
                ['Property', 'Value'],
                [
                    ['Name', $product->name],
                    ['SKU', $product->sku],
                    ['SAP Item Code', $product->sap_item_code],
                    ['Type', $product->type],
                    ['SAP Price', $product->sap_price ? $product->sap_currency . ' ' . number_format($product->sap_price, 2) : 'N/A'],
                    ['Sale Price', number_format($product->sale_price, 2)],
                    ['Purchase Price', number_format($product->purchase_price, 2)],
                    ['SAP Stock', number_format($product->sap_on_hand, 2)],
                    ['Local Stock', number_format($product->quantity, 2)],
                    ['UoM', $product->sap_inventory_uom ?? 'N/A'],
                    ['Status', $product->is_inactive ? 'Inactive' : 'Active'],
                    ['Last Sync', $product->sap_last_sync ?? 'N/A'],
                    ['Sync Status', $product->sap_sync_status ?? 'N/A']
                ]
            );
            
            // Show warehouse details
            $warehouses = $product->sapWarehouses;
            if ($warehouses->count() > 0) {
                $this->info("\n🏪 Warehouse Stock:");
                $warehouseData = [];
                foreach ($warehouses as $warehouse) {
                    $warehouseData[] = [
                        $warehouse->warehouse_code,
                        $warehouse->warehouseMapping->warehouse_name ?? 'Unknown',
                        number_format($warehouse->in_stock, 2),
                        number_format($warehouse->committed, 2),
                        number_format($warehouse->in_stock - $warehouse->committed, 2)
                    ];
                }
                $this->table(
                    ['Code', 'Name', 'In Stock', 'Committed', 'Available'],
                    $warehouseData
                );
            }
        }
        
        return 0;
    }
    
    /**
     * Sync all products
     */
    protected function syncAllProducts(SapItemsService $sapService)
    {
        $isFullSync = $this->option('full');
        $isTestMode = $this->option('test');
        
        $this->info('Starting SAP products synchronization...');
        
        if ($isFullSync) {
            $this->warn('⚠️  Performing FULL sync - this will overwrite existing data and may take a while...');
        } else {
            $this->info('Performing incremental sync (only new/updated items)...');
        }
        
        if ($isTestMode) {
            $this->warn('🧪 Test mode enabled - processing limited items only');
        }
        
        // Confirm for full sync
        if ($isFullSync && !$isTestMode) {
            if (!$this->confirm('Are you sure you want to perform a full sync? This may take a long time.')) {
                $this->info('Sync cancelled.');
                return 0;
            }
        }
        
        // Create progress bar
        $this->info('Fetching items from SAP...');
        
        $stats = $sapService->syncAllProducts($isFullSync);
        
        // Display detailed results
        $this->info('');
        $this->info('🎉 Sync completed successfully!');
        $this->info('');
        
        $this->table(
            ['📊 Sync Results', 'Count', 'Percentage'],
            [
                ['Total Items Processed', number_format($stats['total']), '100%'],
                ['✅ Created', number_format($stats['created']), $this->calculatePercentage($stats['created'], $stats['total'])],
                ['🔄 Updated', number_format($stats['updated']), $this->calculatePercentage($stats['updated'], $stats['total'])],
                ['⏭️  Skipped', number_format($stats['skipped']), $this->calculatePercentage($stats['skipped'], $stats['total'])],
                ['❌ Failed', number_format($stats['failed']), $this->calculatePercentage($stats['failed'], $stats['total'])],
                ['🏪 Warehouses Synced', number_format($stats['warehouses_synced']), '-'],
                ['📏 UoM Groups Synced', number_format($stats['uom_groups_synced']), '-'],
                ['📦 Batches Processed', number_format($stats['batches_processed']), '-']
            ]
        );
        
        // Show warnings/errors if any
        if ($stats['failed'] > 0) {
            $this->warn("⚠️  {$stats['failed']} items failed to sync. Check logs for details.");
        }
        
        // Check for unmapped items
        $unmappedWarehouses = SapWarehouseMapping::whereNull('location_id')->count();
        $unmappedUoms = SapUomMapping::whereNull('unit_id')->count();
        
        if ($unmappedWarehouses > 0 || $unmappedUoms > 0) {
            $this->info('');
            $this->warn('⚠️  Some items need mapping:');
            
            if ($unmappedWarehouses > 0) {
                $this->warn("  • {$unmappedWarehouses} warehouses → Run: php artisan sap:sync-products --map-warehouses");
            }
            
            if ($unmappedUoms > 0) {
                $this->warn("  • {$unmappedUoms} UoMs → Run: php artisan sap:sync-products --map-uom");
            }
        }
        
        // Suggest next steps
        $this->info('');
        $this->info('💡 Next steps:');
        $this->info('  • Verify sync: php artisan sap:sync-products --verify');
        $this->info('  • Check products in POS to verify sync');
        $this->info('  • Set up automated sync schedule in kernel.php');
        
        return 0;
    }
    
    /**
     * Calculate percentage
     */
    protected function calculatePercentage($part, $total)
    {
        if ($total == 0) {
            return '0%';
        }
        
        return number_format(($part / $total) * 100, 1) . '%';
    }
}