<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\SapItemsService;
use App\Services\SapBusinessPartnersService;
use App\SapWarehouseMapping;  // Add this
use App\SapUomMapping;        // Add this
use App\BusinessLocation;  // Add this
use App\Unit;  // Make sure this is the correct namespace
use Illuminate\Support\Facades\Log; // Add this


class SapSyncController extends Controller
{
    protected $sapItemsService;
    protected $sapBusinessPartnersService;
    
    public function __construct(SapItemsService $sapItemsService, SapBusinessPartnersService $sapBusinessPartnersService)
    {
        $this->sapItemsService = $sapItemsService;
        $this->sapBusinessPartnersService = $sapBusinessPartnersService;
    }
    
    /**
     * Show SAP synchronization dashboard
     */
    public function index()
{
    $business_id = request()->session()->get('user.business_id');
    
    // Get all warehouse mappings
    $warehouseMappings = SapWarehouseMapping::all();
    
    // Get all UoM mappings
    $uomMappings = SapUomMapping::all();
    
    // Get all business locations for mapping
    $businessLocations = BusinessLocation::forDropdown($business_id);
    
    // Get all units for mapping
    $units = Unit::forDropdown($business_id);
    
    return view('sap.dashboard', compact(
        'warehouseMappings',
        'uomMappings',
        'businessLocations',
        'units'
    ));
}
    

    

    
    /**
     * Sync warehouses from SAP
     */
    public function syncWarehouses()
    {
        try {
            $count = $this->sapItemsService->syncWarehouses();
            
            return response()->json([
                'success' => true,
                'message' => $count . ' warehouses synced successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error syncing warehouses: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Sync UOM groups from SAP
     */
    public function syncUomGroups()
    {
        try {
            $count = $this->sapItemsService->syncUomGroups();
            
            return response()->json([
                'success' => true,
                'message' => $count . ' UOM mappings synced successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error syncing UOM groups: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Map SAP warehouses to local business locations
     */
    public function mapWarehouses(Request $request)
    {
        $mappings = $request->input('mappings', []);
        
        try {
            foreach ($mappings as $warehouseCode => $locationId) {
                if (empty($locationId)) {
                    continue;
                }
                
                \App\SapWarehouseMapping::where('warehouse_code', $warehouseCode)
                    ->update(['location_id' => $locationId]);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Warehouse mappings updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating warehouse mappings: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Map SAP UOMs to local units
     */
    public function mapUoms(Request $request)
    {
        $mappings = $request->input('mappings', []);
        
        try {
            foreach ($mappings as $uomId => $unitId) {
                if (empty($unitId)) {
                    continue;
                }
                
                \App\SapUomMapping::where('id', $uomId)
                    ->update(['unit_id' => $unitId]);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'UOM mappings updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating UOM mappings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get sync status and statistics - محسّنة
     */
    public function getSyncStatus()
    {
        try {
            $productStats = [
                'total_products' => \App\Product::count(),
                'synced_products' => \App\Product::whereNotNull('sap_item_code')->count(),
                'unsynced_products' => \App\Product::whereNull('sap_item_code')->count(),
                'last_sync' => \App\Product::whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first()?->sap_last_sync,
                'last_sync_ago' => $this->getTimeDifference(\App\Product::whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first()?->sap_last_sync)
            ];
            
            $contactStats = [
                'total_contacts' => \App\Contact::count(),
                'synced_contacts' => \App\Contact::whereNotNull('card_code')->count(),
                'unsynced_contacts' => \App\Contact::whereNull('card_code')->count(),
                'customers' => \App\Contact::where('type', 'customer')->count(),
                'suppliers' => \App\Contact::where('type', 'supplier')->count(),
                'synced_customers' => \App\Contact::where('type', 'customer')
                    ->whereNotNull('card_code')->count(),
                'synced_suppliers' => \App\Contact::where('type', 'supplier')
                    ->whereNotNull('card_code')->count(),
                'last_sync' => \App\Contact::whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first()?->sap_last_sync,
                'last_sync_ago' => $this->getTimeDifference(\App\Contact::whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first()?->sap_last_sync)
            ];
            
            $systemInfo = [
                'warehouse_mappings' => \App\SapWarehouseMapping::count(),
                'uom_mappings' => \App\SapUomMapping::count(),
                'unmapped_warehouses' => \App\SapWarehouseMapping::whereNull('location_id')->count(),
                'unmapped_uoms' => \App\SapUomMapping::whereNull('unit_id')->count(),
                'sync_method' => 'cursor-based pagination',
                'recommended_action' => $this->getRecommendedAction($contactStats, $productStats)
            ];
            
            return response()->json([
                'success' => true,
                'data' => [
                    'products' => $productStats,
                    'contacts' => $contactStats,
                    'system' => $systemInfo
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطأ في جلب إحصائيات المزامنة: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get time difference in human readable format
     */
    private function getTimeDifference($date)
    {
        if (!$date) {
            return 'لم تتم مزامنة من قبل';
        }
        
        $now = new \DateTime();
        $syncDate = new \DateTime($date);
        $diff = $now->diff($syncDate);
        
        if ($diff->days > 0) {
            return $diff->days . ' يوم مضت';
        } elseif ($diff->h > 0) {
            return $diff->h . ' ساعة مضت';
        } elseif ($diff->i > 0) {
            return $diff->i . ' دقيقة مضت';
        } else {
            return 'منذ قليل';
        }
    }
    
    /**
     * Get recommended action based on sync status
     */
    private function getRecommendedAction($contactStats, $productStats)
    {
        $recommendations = [];
        
        if ($contactStats['unsynced_contacts'] > 0) {
            $recommendations[] = "يوجد {$contactStats['unsynced_contacts']} جهة اتصال غير متزامنة";
        }
        
        if ($productStats['unsynced_products'] > 0) {
            $recommendations[] = "يوجد {$productStats['unsynced_products']} منتج غير متزامن";
        }
        
        if (empty($recommendations)) {
            return 'جميع البيانات متزامنة بنجاح';
        }
        
        return implode('. ', $recommendations) . '. يُنصح بإجراء مزامنة كاملة.';
    }

    /**
     * Force full sync for all data
     */
    public function fullSync(Request $request)
    {
        try {
            $results = [];
            
            // مزامنة المستودعات أولاً
            Log::info("بدء المزامنة الكاملة - المستودعات");
            $results['warehouses'] = $this->sapItemsService->syncWarehouses();
            
            // مزامنة وحدات القياس
            Log::info("بدء المزامنة الكاملة - وحدات القياس");
            $results['uom_groups'] = $this->sapItemsService->syncUomGroups();
            
            // مزامنة جهات الاتصال
            Log::info("بدء المزامنة الكاملة - جهات الاتصال");
            $results['contacts'] = $this->sapBusinessPartnersService->syncAllContacts(true);
            
            // مزامنة المنتجات
            Log::info("بدء المزامنة الكاملة - المنتجات");
            $results['products'] = $this->sapItemsService->syncAllProducts(true);
            
            return response()->json([
                'success' => true,
                'message' => 'تم إجراء المزامنة الكاملة بنجاح',
                'data' => $results
            ]);
        } catch (\Exception $e) {
            Log::error("خطأ في المزامنة الكاملة: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطأ في المزامنة الكاملة: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Sync contacts in small batches for browser - محسّنة للمتصفح
     */
    public function syncContactsBatched(Request $request)
    {
        // زيادة حدود التنفيذ
        set_time_limit(300); // 5 دقائق
        ini_set('memory_limit', '512M');
        
        $fullSync = $request->input('full_sync', false);
        $batchSize = min($request->input('batch_size', 100), 100); // حد أقصى 100
        $maxBatches = $request->input('max_batches', 5); // حد أقصى 5 دفعات في الطلب الواحد
        
        try {
            Log::info("بدء مزامنة جهات الاتصال على دفعات للمتصفح", [
                'full_sync' => $fullSync,
                'batch_size' => $batchSize,
                'max_batches' => $maxBatches
            ]);
            
            $totalStats = [
                'total' => 0,
                'created' => 0,
                'updated' => 0,
                'failed' => 0,
                'skipped' => 0,
                'batches_processed' => 0,
                'has_more' => false,
                'next_cursor' => null
            ];
            
            // البدء بالعملاء
            $customerResult = $this->processContactBatch('customer', $fullSync, $batchSize, $maxBatches);
            $totalStats['total'] += $customerResult['total'];
            $totalStats['created'] += $customerResult['created'];
            $totalStats['updated'] += $customerResult['updated'];
            $totalStats['failed'] += $customerResult['failed'];
            $totalStats['skipped'] += $customerResult['skipped'];
            $totalStats['batches_processed'] += $customerResult['batches_processed'];
            
            // البدء بالموردين إذا انتهينا من العملاء
            if (!$customerResult['has_more']) {
                $supplierResult = $this->processContactBatch('supplier', $fullSync, $batchSize, $maxBatches);
                $totalStats['total'] += $supplierResult['total'];
                $totalStats['created'] += $supplierResult['created'];
                $totalStats['updated'] += $supplierResult['updated'];
                $totalStats['failed'] += $supplierResult['failed'];
                $totalStats['skipped'] += $supplierResult['skipped'];
                $totalStats['batches_processed'] += $supplierResult['batches_processed'];
                $totalStats['has_more'] = $supplierResult['has_more'];
            } else {
                $totalStats['has_more'] = true;
            }
            
            return response()->json([
                'success' => true,
                'message' => "تم معالجة {$totalStats['batches_processed']} دفعة بنجاح",
                'data' => $totalStats
            ]);
            
        } catch (\Exception $e) {
            Log::error("خطأ في مزامنة جهات الاتصال على دفعات: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطأ في مزامنة جهات الاتصال: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Sync products in small batches for browser - محسّنة للمتصفح
     */
    public function syncProductsBatched(Request $request)
    {
        // زيادة حدود التنفيذ
        set_time_limit(300); // 5 دقائق
        ini_set('memory_limit', '512M');
        
        $fullSync = $request->input('full_sync', false);
        $batchSize = min($request->input('batch_size', 50), 50); // حد أقصى 50 للمنتجات
        $maxBatches = $request->input('max_batches', 3); // حد أقصى 3 دفعات للمنتجات
        
        try {
            Log::info("بدء مزامنة المنتجات على دفعات للمتصفح", [
                'full_sync' => $fullSync,
                'batch_size' => $batchSize,
                'max_batches' => $maxBatches
            ]);
            
            $stats = $this->processProductBatch($fullSync, $batchSize, $maxBatches);
            
            return response()->json([
                'success' => true,
                'message' => "تم معالجة {$stats['batches_processed']} دفعة من المنتجات بنجاح",
                'data' => $stats
            ]);
            
        } catch (\Exception $e) {
            Log::error("خطأ في مزامنة المنتجات على دفعات: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطأ في مزامنة المنتجات: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Process contact batch
     */
    private function processContactBatch($type, $fullSync, $batchSize, $maxBatches)
    {
        $stats = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0,
            'batches_processed' => 0,
            'has_more' => false
        ];
        
        // إعداد الفلتر
        $baseFilter = $type === 'customer' ? "CardType eq 'cCustomer'" : "CardType eq 'cSupplier'";
        
        // جلب البيانات على دفعات صغيرة
        $processedInThisCall = 0;
        $lastCardCode = session()->get("sap_sync_last_cardcode_{$type}", '');
        
        while ($processedInThisCall < $maxBatches) {
            try {
                // بناء الفلتر مع cursor
                $filter = $baseFilter;
                if ($lastCardCode) {
                    $filter .= " and CardCode gt '{$lastCardCode}'";
                }
                
                $queryParams = [
                    '$filter' => $filter,
                    '$orderby' => 'CardCode asc',
                    '$top' => $batchSize,
                    '$select' => 'CardCode,CardName,CardType,GroupCode,Address,Phone1,Phone2,EmailAddress,UpdateDate'
                ];
                
                $result = $this->sapBusinessPartnersService->get('BusinessPartners', $queryParams, 60000);
                
                if (empty($result['value'])) {
                    // انتهت البيانات
                    session()->forget("sap_sync_last_cardcode_{$type}");
                    break;
                }
                
                $batchCount = count($result['value']);
                
                // معالجة كل سجل في الدفعة
                foreach ($result['value'] as $partner) {
                    try {
                        $processResult = $this->sapBusinessPartnersService->processSapPartner($partner, $type);
                        
                        if ($processResult === 'created') {
                            $stats['created']++;
                        } elseif ($processResult === 'updated') {
                            $stats['updated']++;
                        } elseif ($processResult === 'skipped') {
                            $stats['skipped']++;
                        }
                        
                        $stats['total']++;
                    } catch (\Exception $e) {
                        $stats['failed']++;
                    }
                }
                
                // تحديث آخر CardCode
                $lastCardCode = end($result['value'])['CardCode'];
                session()->put("sap_sync_last_cardcode_{$type}", $lastCardCode);
                
                $stats['batches_processed']++;
                $processedInThisCall++;
                
                // التحقق من انتهاء البيانات يعتمد على nextLink وليس حجم الدفعة
                if (!isset($result['odata.nextLink'])) {
                    session()->forget("sap_sync_last_cardcode_{$type}");
                    break;
                } else {
                    $stats['has_more'] = true;
                }
                
            } catch (\Exception $e) {
                Log::error("خطأ في معالجة دفعة {$type}: " . $e->getMessage());
                break;
            }
        }
        
        return $stats;
    }
    
    /**
     * Process product batch
     */
    private function processProductBatch($fullSync, $batchSize, $maxBatches)
    {
        $stats = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0,
            'batches_processed' => 0,
            'has_more' => false
        ];
        
        // جلب المنتجات على دفعات صغيرة
        $processedInThisCall = 0;
        $lastItemCode = session()->get('sap_sync_last_itemcode', '');
        
        while ($processedInThisCall < $maxBatches) {
            try {
                // بناء الفلتر مع cursor
                $filter = null;
                if ($lastItemCode) {
                    $filter = "ItemCode gt '{$lastItemCode}'";
                }
                
                $queryParams = [
                    '$orderby' => 'ItemCode asc',
                    '$top' => $batchSize,
                    '$select' => 'ItemCode,ItemName,ItemType,ItemsGroupCode,InventoryItem,UpdateDate'
                ];
                
                if ($filter) {
                    $queryParams['$filter'] = $filter;
                }
                
                $result = $this->sapItemsService->get('Items', $queryParams, 60000);
                
                if (empty($result['value'])) {
                    // انتهت البيانات
                    session()->forget('sap_sync_last_itemcode');
                    break;
                }
                
                $batchCount = count($result['value']);
                
                // معالجة كل منتج في الدفعة (معالجة مبسطة للمتصفح)
                foreach ($result['value'] as $item) {
                    try {
                        $processResult = $this->sapItemsService->processSapItem($item, $fullSync);
                        
                        if ($processResult === 'created') {
                            $stats['created']++;
                        } elseif ($processResult === 'updated') {
                            $stats['updated']++;
                        } elseif ($processResult === 'skipped') {
                            $stats['skipped']++;
                        }
                        
                        $stats['total']++;
                    } catch (\Exception $e) {
                        $stats['failed']++;
                    }
                }
                
                // تحديث آخر ItemCode
                $lastItemCode = end($result['value'])['ItemCode'];
                session()->put('sap_sync_last_itemcode', $lastItemCode);
                
                $stats['batches_processed']++;
                $processedInThisCall++;
                
                // التحقق من انتهاء البيانات يعتمد على nextLink وليس حجم الدفعة
                if (!isset($result['odata.nextLink'])) {
                    session()->forget('sap_sync_last_itemcode');
                    break;
                } else {
                    $stats['has_more'] = true;
                }
                
            } catch (\Exception $e) {
                Log::error("خطأ في معالجة دفعة المنتجات: " . $e->getMessage());
                break;
            }
        }
        
        return $stats;
    }

    /**
     * Reset sync cursors - إعادة تعيين مؤشرات المزامنة
     */
    public function resetSyncCursors(Request $request)
    {
        try {
            $type = $request->input('type', 'all'); // all, contacts, products
            
            if ($type === 'all' || $type === 'contacts') {
                session()->forget('sap_sync_last_cardcode_customer');
                session()->forget('sap_sync_last_cardcode_supplier');
                Log::info("تم إعادة تعيين مؤشرات جهات الاتصال");
            }
            
            if ($type === 'all' || $type === 'products') {
                session()->forget('sap_sync_last_itemcode');
                Log::info("تم إعادة تعيين مؤشرات المنتجات");
            }
            
            return response()->json([
                'success' => true,
                'message' => 'تم إعادة تعيين مؤشرات المزامنة بنجاح'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطأ في إعادة تعيين المؤشرات: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Sync contacts from SAP - محسّنة مع حدود زمنية
     */
    public function syncContacts(Request $request)
    {
        $fullSync = $request->input('full_sync', false);
        $batchSize = $request->input('batch_size', 200);
        $timeout = $request->input('timeout', 120);
        $useBatched = $request->input('batched', false); // استخدام المعالجة على دفعات
        
        // إذا كان طلب من المتصفح، استخدم المعالجة على دفعات
        if ($useBatched || $request->ajax()) {
            return $this->syncContactsBatched($request);
        }
        
        // زيادة حدود التنفيذ للطريقة العادية
        set_time_limit(600); // 10 دقائق
        ini_set('memory_limit', '1G');
        
        try {
            Log::info("بدء مزامنة جهات الاتصال", [
                'full_sync' => $fullSync,
                'batch_size' => $batchSize,
                'timeout' => $timeout
            ]);
            
            $stats = $this->sapBusinessPartnersService->syncAllContacts($fullSync);
            
            return response()->json([
                'success' => true,
                'message' => 'تم مزامنة جهات الاتصال بنجاح',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            Log::error("خطأ في مزامنة جهات الاتصال: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطأ في مزامنة جهات الاتصال: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Sync products from SAP - محسّنة مع حدود زمنية
     */
    public function syncProducts(Request $request)
    {
        $fullSync = $request->input('full_sync', false);
        $batchSize = $request->input('batch_size', 200);
        $timeout = $request->input('timeout', 180);
        $useBatched = $request->input('batched', false); // استخدام المعالجة على دفعات
        
        // إذا كان طلب من المتصفح، استخدم المعالجة على دفعات
        if ($useBatched || $request->ajax()) {
            return $this->syncProductsBatched($request);
        }
        
        // زيادة حدود التنفيذ للطريقة العادية
        set_time_limit(900); // 15 دقيقة
        ini_set('memory_limit', '1G');
        
        try {
            Log::info("بدء مزامنة المنتجات", [
                'full_sync' => $fullSync,
                'batch_size' => $batchSize,
                'timeout' => $timeout
            ]);
            
            $stats = $this->sapItemsService->syncAllProducts($fullSync);
            
            return response()->json([
                'success' => true,
                'message' => 'تم مزامنة المنتجات بنجاح',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            Log::error("خطأ في مزامنة المنتجات: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطأ في مزامنة المنتجات: ' . $e->getMessage()
            ], 500);
        }
    }
}