<?php

namespace App\Http\Controllers;

use App\Transaction;
use App\Services\SapInvoicesService;
use App\Jobs\SyncTransactionToSap;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class SapTSyncController extends Controller
{
    /**
     * The SAP Invoices Service
     * 
     * @var \App\Services\SapInvoicesService
     */
    protected $sapService;

    /**
     * Create a new controller instance.
     * 
     * @param  \App\Services\SapInvoicesService  $sapService
     * @return void
     */
    public function __construct(SapInvoicesService $sapService)
    {
        $this->sapService = $sapService;
    }

    /**
     * Display SAP sync status page
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        
        
        return view('sap.index');
    }
    
    /**
     * Get transactions for datatable
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function getTransactions(Request $request)
    {
        
        
        if ($request->ajax()) {
            $business_id = $request->session()->get('user.business_id');
            
            $transactions = Transaction::where('business_id', $business_id)
                ->where(function($q) {
                    $q->where('status', 'final')
                      ->orWhere('is_quotation', 1)
                      ->orWhere('sub_status', 'quotation');
                })
                ->with(['contact', 'business_location'])
                ->select('transactions.*');
            
            return Datatables::of($transactions)
                ->addColumn('action', function ($row) {
                    $html = '<button class="btn btn-xs btn-primary sync-to-sap" data-id="'.$row->id.'">Sync to SAP</button>';
                    return $html;
                })
                ->editColumn('transaction_date', function ($row) {
                    return $row->transaction_date ? date('Y-m-d H:i', strtotime($row->transaction_date)) : '';
                })
                ->editColumn('sap_sync_status', function ($row) {
                    if ($row->sap_sync_status == 'success') {
                        return '<span class="label label-success">Success</span>';
                    } elseif ($row->sap_sync_status == 'failed') {
                        return '<span class="label label-danger">Failed</span>';
                    } else {
                        return '<span class="label label-warning">Pending</span>';
                    }
                })
                ->editColumn('sap_sync_date', function ($row) {
                    return $row->sap_sync_date ? date('Y-m-d H:i', strtotime($row->sap_sync_date)) : '';
                })
                ->editColumn('final_total', function ($row) {
                    return number_format($row->final_total, 2);
                })
                ->editColumn('status', function ($row) {
                    if ($row->status == 'final') {
                        return 'Final';
                    } elseif ($row->is_quotation == 1 || $row->sub_status == 'quotation') {
                        return 'Quotation';
                    } else {
                        return ucfirst($row->status);
                    }
                })
                ->addColumn('location', function ($row) {
                    return $row->business_location ? $row->business_location->name : '';
                })
                ->addColumn('customer', function ($row) {
                    return $row->contact ? $row->contact->name : '';
                })
                ->rawColumns(['action', 'sap_sync_status'])
                ->make(true);
        }
    }
    
    /**
     * Sync a transaction to SAP
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function syncTransaction(Request $request)
    {
        
        
        $transaction_id = $request->input('transaction_id');
        $transaction = Transaction::find($transaction_id);
        
        if (!$transaction) {
            return response()->json(['success' => false, 'message' => 'Transaction not found']);
        }
        
        try {
            if (config('sap.use_queue', true)) {
                SyncTransactionToSap::dispatch($transaction_id);
                return response()->json(['success' => true, 'message' => 'Transaction queued for synchronization']);
            } else {
                $result = $this->sapService->pushTransactionToSap($transaction);
                return response()->json(['success' => $result['success'], 'message' => $result['message']]);
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }
    
    /**
     * Queue multiple transactions for syncing
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function syncMultiple(Request $request)
    {
        
        
        $business_id = $request->session()->get('user.business_id');
        $transaction_ids = $request->input('transaction_ids', []);
        
        if (empty($transaction_ids)) {
            // Sync failed transactions from the last week
            $query = Transaction::where('business_id', $business_id)
                ->where(function($q) {
                    $q->where('status', 'final')
                      ->orWhere('is_quotation', 1)
                      ->orWhere('sub_status', 'quotation');
                })
                ->where(function($q) {
                    $q->whereNull('sap_sync_status')
                      ->orWhere('sap_sync_status', 'failed');
                })
                ->where('created_at', '>=', now()->subDays(7))
                ->limit(50);
                
            $transactions = $query->get();
        } else {
            $transactions = Transaction::whereIn('id', $transaction_ids)
                ->where('business_id', $business_id)
                ->get();
        }
        
        $count = 0;
        foreach ($transactions as $transaction) {
            if (config('sap.use_queue', true)) {
                SyncTransactionToSap::dispatch($transaction->id);
            } else {
                try {
                    $this->sapService->pushTransactionToSap($transaction);
                } catch (\Exception $e) {
                    // Log error and continue
                    \Log::error("Error syncing transaction #{$transaction->id}: {$e->getMessage()}");
                }
            }
            $count++;
        }
        
        $message = config('sap.use_queue', true)
            ? "{$count} transactions queued for synchronization"
            : "{$count} transactions processed";
            
        return response()->json(['success' => true, 'message' => $message]);
    }
}