<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class WhatsAppController extends Controller
{
    private $instanceId = '26795';
    private $token = 'egswpl1vlle5gjrt';
    private $baseUrl = 'https://api.ultramsg.com';

    /**
     * اختبار بسيط للتأكد من صحة Token
     */
    public function verifyToken()
    {
        try {
            // اختبار الاتصال بإرسال رسالة بسيطة
            $testPhone = '+9647710584672'; // رقم للاختبار
            $testMessage = 'Test token verification - ' . now()->format('H:i:s');
            
            $url = "{$this->baseUrl}/instance{$this->instanceId}/messages/chat?token={$this->token}";
            
            $postData = [
                'to' => $testPhone,
                'body' => $testMessage
            ];
            
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($postData),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 15,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/x-www-form-urlencoded',
                    'User-Agent: Mozilla/5.0 (compatible; COMBO-SAS/1.0)'
                ]
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            Log::info('Token verification attempt', [
                'http_code' => $httpCode,
                'curl_error' => $error,
                'response' => $response,
                'instance_id' => $this->instanceId,
                'token_length' => strlen($this->token)
            ]);
            
            if ($error) {
                return response()->json([
                    'success' => false,
                    'message' => 'خطأ في الاتصال',
                    'error' => 'cURL Error: ' . $error
                ], 500);
            }
            
            $responseData = json_decode($response, true);
            
            if ($httpCode === 200) {
                if (isset($responseData['error'])) {
                    return response()->json([
                        'success' => false,
                        'message' => 'خطأ في Token أو الإعدادات',
                        'error' => $responseData['error'],
                        'suggestions' => [
                            'تأكد من صحة Token',
                            'تأكد من صحة Instance ID',
                            'تأكد من أن WhatsApp متصل في UltraMsg'
                        ]
                    ], 400);
                } else {
                    return response()->json([
                        'success' => true,
                        'message' => 'Token صحيح والاتصال يعمل',
                        'response' => $responseData,
                        'configuration' => [
                            'instance_id' => $this->instanceId,
                            'base_url' => $this->baseUrl,
                            'token_length' => strlen($this->token) . ' characters'
                        ]
                    ]);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => "HTTP Error: {$httpCode}",
                    'error' => $response,
                    'http_code' => $httpCode
                ], 400);
            }
            
        } catch (\Exception $e) {
            Log::error('Token verification error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'خطأ في التحقق من Token',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال صورة الفاتورة - طريقة مُبسطة بـ Base64
     */
    public function sendInvoiceImage(Request $request)
    {
        // التحقق من صحة البيانات
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|min:10',
            'image_data' => 'required|string|min:100',
            'caption' => 'nullable|string|max:1000'
        ], [
            'phone.required' => 'رقم الهاتف مطلوب',
            'phone.min' => 'رقم الهاتف قصير جداً',
            'image_data.required' => 'بيانات الصورة مطلوبة',
            'image_data.min' => 'بيانات الصورة غير صحيحة',
            'caption.max' => 'نص الرسالة طويل جداً'
        ]);

        if ($validator->fails()) {
            Log::warning('Invoice validation failed', [
                'errors' => $validator->errors()->toArray()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'البيانات المرسلة غير صحيحة',
                'errors' => $validator->errors(),
                'details' => $validator->errors()->first()
            ], 422);
        }

        try {
            $phone = $request->input('phone');
            $imageData = $request->input('image_data');
            $caption = $request->input('caption', 'فاتورة مبيعات من COMBO SAS');

            Log::info('Starting invoice image send', [
                'phone' => $phone,
                'image_data_length' => strlen($imageData),
                'caption' => $caption
            ]);

            // تنظيف بيانات الصورة
            if (strpos($imageData, 'data:image') === 0) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
            }

            // إرسال الصورة مباشرة بـ Base64
            $response = $this->sendImageViaBase64($phone, $imageData, $caption);

            if ($response['success']) {
                Log::info('Invoice sent successfully', [
                    'phone' => $phone,
                    'response' => $response['data']
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'تم إرسال الفاتورة بنجاح',
                    'response' => $response['data']
                ]);
            } else {
                Log::error('Failed to send invoice', [
                    'phone' => $phone,
                    'error' => $response['error']
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'فشل في إرسال الفاتورة',
                    'error' => $response['error']
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('Send Invoice Image Exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getFile() . ':' . $e->getLine(),
                'phone' => $request->input('phone')
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطأ في الخادم',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال صورة مباشرة بـ Base64 - مُبسط
     */
    private function sendImageViaBase64($phone, $imageBase64, $caption)
    {
        try {
            // URL مع token
            $url = "{$this->baseUrl}/instance{$this->instanceId}/messages/image?token={$this->token}";

            // البيانات
            $postData = [
                'to' => $phone,
                'image' => $imageBase64,
                'caption' => $caption
            ];

            Log::info('Sending image via Base64', [
                'url_base' => "{$this->baseUrl}/instance{$this->instanceId}/messages/image",
                'phone' => $phone,
                'image_data_length' => strlen($imageBase64),
                'caption' => $caption,
                'token_length' => strlen($this->token)
            ]);

            // إرسال البيانات
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($postData),
                CURLOPT_TIMEOUT => 60,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/x-www-form-urlencoded',
                    'User-Agent: Mozilla/5.0 (compatible; COMBO-SAS/1.0)'
                ]
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);

            Log::info('Base64 send response', [
                'phone' => $phone,
                'http_code' => $httpCode,
                'curl_error' => $error,
                'response' => $response,
                'response_length' => strlen($response)
            ]);

            if ($error) {
                return [
                    'success' => false,
                    'error' => 'cURL Error: ' . $error
                ];
            }

            $responseData = json_decode($response, true);

            if ($httpCode === 200) {
                // فحص إذا كان هناك خطأ في الاستجابة
                if (isset($responseData['error'])) {
                    $errorMessage = is_array($responseData['error']) ? 
                        json_encode($responseData['error'], JSON_UNESCAPED_UNICODE) : 
                        $responseData['error'];
                        
                    Log::error('UltraMsg API Error', [
                        'error' => $responseData['error'],
                        'phone' => $phone,
                        'raw_response' => $response
                    ]);

                    return [
                        'success' => false,
                        'error' => 'خطأ API: ' . $errorMessage,
                        'api_error' => $responseData['error']
                    ];
                }

                // فحص حالات النجاح
                if (isset($responseData['sent']) && ($responseData['sent'] === "true" || $responseData['sent'] === true)) {
                    return [
                        'success' => true,
                        'data' => $responseData
                    ];
                } elseif (isset($responseData['id']) && !empty($responseData['id'])) {
                    return [
                        'success' => true,
                        'data' => $responseData
                    ];
                } elseif (isset($responseData['status']) && $responseData['status'] === 'sent') {
                    return [
                        'success' => true,
                        'data' => $responseData
                    ];
                } else {
                    // اعتبار HTTP 200 بدون خطأ كنجاح
                    Log::info('Assuming success based on HTTP 200');
                    return [
                        'success' => true,
                        'data' => $responseData,
                        'note' => 'Success assumed from HTTP 200'
                    ];
                }
            } else {
                return [
                    'success' => false,
                    'error' => "HTTP Error {$httpCode}: {$response}",
                    'http_code' => $httpCode
                ];
            }

        } catch (\Exception $e) {
            Log::error('Base64 send exception', [
                'error' => $e->getMessage(),
                'phone' => $phone
            ]);

            return [
                'success' => false,
                'error' => 'خطأ في إرسال الصورة: ' . $e->getMessage()
            ];
        }
    }

    /**
     * اختبار إرسال صورة تجريبية بـ Base64
     */
    public function testBase64ImageSend(Request $request)
    {
        try {
            $phone = $request->input('phone', '+9647710584672');

            // صورة تجريبية صغيرة (1x1 pixel PNG) بـ Base64
            $testImageBase64 = 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChAHgAuXJKgAAAABJRU5ErkJggg==';

            $response = $this->sendImageViaBase64(
                $phone,
                $testImageBase64,
                'اختبار إرسال صورة Base64 - ' . now()->format('Y-m-d H:i:s')
            );

            return response()->json([
                'success' => $response['success'],
                'message' => $response['success'] ? 'تم إرسال الصورة التجريبية بنجاح' : 'فشل إرسال الصورة التجريبية',
                'response' => $response
            ]);

        } catch (\Exception $e) {
            Log::error('Test Base64 Image Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطأ في اختبار الصورة',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * فحص معلمات الطلب قبل الإرسال (للتشخيص)
     */
    public function debugRequestParameters(Request $request)
    {
        try {
            $phone = $request->input('phone', '+9647710584672');
            $testMessage = 'Debug test - ' . now()->format('H:i:s');
            
            // اختبار إرسال نص أولاً
            $textUrl = "{$this->baseUrl}/instance{$this->instanceId}/messages/chat?token={$this->token}";
            $textData = [
                'to' => $phone,
                'body' => $testMessage
            ];
            
            Log::info('Debug - Text message test', [
                'url' => $textUrl,
                'data' => $textData
            ]);
            
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $textUrl,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($textData),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 15,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/x-www-form-urlencoded'
                ]
            ]);
            
            $textResponse = curl_exec($ch);
            $textHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $textError = curl_error($ch);
            curl_close($ch);
            
            // اختبار معلمات الصورة
            $imageUrl = "{$this->baseUrl}/instance{$this->instanceId}/messages/image?token={$this->token}";
            $imageData = [
                'to' => $phone,
                'image' => 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChAHgAuXJKgAAAABJRU5ErkJggg==',
                'caption' => 'Debug test image'
            ];
            
            Log::info('Debug - Image message test', [
                'url' => $imageUrl,
                'data_keys' => array_keys($imageData),
                'to_value' => $imageData['to'],
                'image_length' => strlen($imageData['image']),
                'caption' => $imageData['caption']
            ]);
            
            return response()->json([
                'success' => true,
                'debug_info' => [
                    'configuration' => [
                        'instance_id' => $this->instanceId,
                        'token_length' => strlen($this->token),
                        'base_url' => $this->baseUrl
                    ],
                    'text_test' => [
                        'url' => $textUrl,
                        'http_code' => $textHttpCode,
                        'curl_error' => $textError,
                        'response' => $textResponse
                    ],
                    'image_test_params' => [
                        'url' => $imageUrl,
                        'parameters' => $imageData,
                        'encoded_params' => http_build_query($imageData)
                    ]
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال صورة الفاتورة باستخدام cURL - مُصحح نهائياً
     */
    private function sendInvoiceImageWithCurl($phone, $imagePath, $caption)
    {
        try {
            // التأكد من وجود الملف
            if (!file_exists($imagePath)) {
                return [
                    'success' => false,
                    'error' => 'الملف غير موجود: ' . $imagePath
                ];
            }

            // URL مع token كـ GET parameter (كما يتطلب UltraMsg)
            $url = "{$this->baseUrl}/instance{$this->instanceId}/messages/image?token={$this->token}";

            // تحضير البيانات بدون token (لأنه في URL)
            $postFields = [
                'to' => $phone,
                'image' => new \CURLFile($imagePath, 'image/jpeg', 'invoice.jpg'),
                'caption' => $caption
            ];

            Log::info('Sending invoice image via cURL', [
                'url' => "{$this->baseUrl}/instance{$this->instanceId}/messages/image?token=***",
                'phone' => $phone,
                'file_path' => $imagePath,
                'file_size' => filesize($imagePath) . ' bytes',
                'caption' => $caption,
                'token_length' => strlen($this->token)
            ]);

            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => "",
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 60,
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_SSL_VERIFYPEER => 0,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => "POST",
                CURLOPT_POSTFIELDS => $postFields,
                CURLOPT_HTTPHEADER => [
                    "User-Agent: Mozilla/5.0 (compatible; COMBO-SAS/1.0)"
                ],
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);

            // تسجيل الاستجابة للتشخيص
            Log::info('cURL Response', [
                'phone' => $phone,
                'http_code' => $httpCode,
                'curl_error' => $error,
                'response' => $response,
                'response_length' => strlen($response)
            ]);

            if ($error) {
                return [
                    'success' => false,
                    'error' => 'cURL Error: ' . $error
                ];
            }

            // تحليل الاستجابة
            $responseData = json_decode($response, true);

            if ($httpCode === 200) {
                // فحص حالات النجاح المختلفة
                if (isset($responseData['sent']) && ($responseData['sent'] === "true" || $responseData['sent'] === true)) {
                    return [
                        'success' => true,
                        'data' => $responseData
                    ];
                } elseif (isset($responseData['id']) && !empty($responseData['id'])) {
                    return [
                        'success' => true,
                        'data' => $responseData
                    ];
                } elseif (isset($responseData['status']) && $responseData['status'] === 'sent') {
                    return [
                        'success' => true,
                        'data' => $responseData
                    ];
                } elseif (is_array($responseData) && !isset($responseData['error'])) {
                    // إذا كان HTTP 200 وليس هناك خطأ صريح، اعتبرها نجاح
                    Log::info('Assuming success based on HTTP 200 without error');
                    return [
                        'success' => true,
                        'data' => $responseData,
                        'note' => 'Success assumed from HTTP 200'
                    ];
                } else {
                    // فحص إذا كان هناك خطأ معين - إصلاح مشكلة Array to string conversion
                    if (isset($responseData['error'])) {
                        $errorMessage = is_array($responseData['error']) ? 
                            json_encode($responseData['error']) : 
                            $responseData['error'];
                            
                        Log::error('UltraMsg API Error', [
                            'error' => $responseData['error'],
                            'phone' => $phone,
                            'raw_response' => $response
                        ]);

                        return [
                            'success' => false,
                            'error' => 'خطأ API: ' . $errorMessage,
                            'api_error' => $responseData['error']
                        ];
                    } else {
                        Log::warning('Unexpected response format', [
                            'response' => $responseData,
                            'raw_response' => $response
                        ]);

                        return [
                            'success' => false,
                            'error' => 'استجابة غير متوقعة من خادم WhatsApp',
                            'response_data' => $responseData,
                            'raw_response' => $response
                        ];
                    }
                }
            } else {
                return [
                    'success' => false,
                    'error' => "HTTP Error {$httpCode}: {$response}",
                    'http_code' => $httpCode
                ];
            }

        } catch (\Exception $e) {
            Log::error('cURL Exception in sendInvoiceImageWithCurl', [
                'error' => $e->getMessage(),
                'phone' => $phone,
                'image_path' => $imagePath
            ]);

            return [
                'success' => false,
                'error' => 'خطأ في إرسال الصورة: ' . $e->getMessage()
            ];
        }
    }

    /**
     * إرسال صورة باستخدام طريقة بديلة (Base64)
     */
    public function sendInvoiceImageAlternative(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|min:10',
            'image_data' => 'required|string|min:100',
            'caption' => 'nullable|string|max:1000'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'البيانات المرسلة غير صحيحة',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $phone = $request->input('phone');
            $imageData = $request->input('image_data');
            $caption = $request->input('caption', 'فاتورة مبيعات من COMBO SAS');

            // تنظيف بيانات الصورة
            if (strpos($imageData, 'data:image') === 0) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
            }

            Log::info('Sending invoice image via Base64 method', [
                'phone' => $phone,
                'image_data_length' => strlen($imageData),
                'caption' => $caption
            ]);

            // URL مع token
            $url = "{$this->baseUrl}/instance{$this->instanceId}/messages/image?token={$this->token}";

            // إرسال البيانات كـ form-urlencoded
            $postData = [
                'to' => $phone,
                'image' => $imageData,
                'caption' => $caption
            ];

            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 60,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($postData),
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/x-www-form-urlencoded',
                    'User-Agent: Mozilla/5.0 (compatible; COMBO-SAS/1.0)'
                ]
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);

            Log::info('Base64 cURL Response', [
                'phone' => $phone,
                'http_code' => $httpCode,
                'curl_error' => $error,
                'response' => $response
            ]);

            if ($error) {
                return response()->json([
                    'success' => false,
                    'message' => 'خطأ في الاتصال',
                    'error' => $error
                ], 500);
            }

            $responseData = json_decode($response, true);

            if ($httpCode === 200) {
                if (isset($responseData['error'])) {
                    $errorMessage = is_array($responseData['error']) ? 
                        json_encode($responseData['error']) : 
                        $responseData['error'];
                        
                    return response()->json([
                        'success' => false,
                        'message' => 'فشل في إرسال الفاتورة',
                        'error' => $errorMessage,
                        'api_error' => $responseData['error']
                    ], 400);
                } else {
                    return response()->json([
                        'success' => true,
                        'message' => 'تم إرسال الفاتورة بنجاح (Base64)',
                        'response' => $responseData
                    ]);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => "HTTP Error: {$httpCode}",
                    'error' => $response
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('Alternative image send error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'خطأ في إرسال الفاتورة',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال رسالة نصية
     */
    public function sendTextMessage(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'message' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'بيانات غير صحيحة',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $postData = [
                'token' => $this->token,
                'to' => $request->input('phone'),
                'body' => $request->input('message')
            ];

            $response = $this->sendCurlRequest(
                "{$this->baseUrl}/instance{$this->instanceId}/messages/chat",
                $postData
            );

            if ($response['success']) {
                $data = json_decode($response['data'], true);

                if (isset($data['sent']) && $data['sent'] === "true") {
                    return response()->json([
                        'success' => true,
                        'message' => 'تم إرسال الرسالة بنجاح',
                        'response' => $data
                    ]);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'فشل في إرسال الرسالة',
                        'error' => $data
                    ], 400);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'خطأ في الطلب',
                    'error' => $response['error']
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('WhatsApp Text Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطأ في الخادم',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * اختبار الاتصال
     */
    public function testConnection(Request $request)
    {
        try {
            $phone = $request->input('phone', '+9647710584672');
            $message = 'اختبار اتصال من نظام POS - ' . now()->format('Y-m-d H:i:s');

            $postData = [
                'token' => $this->token,
                'to' => $phone,
                'body' => $message
            ];

            $response = $this->sendCurlRequest(
                "{$this->baseUrl}/instance{$this->instanceId}/messages/chat",
                $postData
            );

            if ($response['success']) {
                $data = json_decode($response['data'], true);

                return response()->json([
                    'success' => true,
                    'message' => 'تم اختبار الاتصال بنجاح',
                    'response' => $data
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'فشل اختبار الاتصال',
                    'error' => $response['error']
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('WhatsApp Test Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطأ في اختبار الاتصال',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * اختبار إرسال صورة تجريبية
     */
    public function testImageSend(Request $request)
    {
        try {
            $phone = $request->input('phone', '+9647710584672');

            // إنشاء صورة تجريبية صغيرة (1x1 pixel PNG)
            $testImageBase64 = 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChAHgAuXJKgAAAABJRU5ErkJggg==';

            // تحويل إلى ملف مؤقت
            $imageContent = base64_decode($testImageBase64);
            $tempFile = tempnam(sys_get_temp_dir(), 'test_') . '.png';
            file_put_contents($tempFile, $imageContent);

            // إرسال الصورة التجريبية
            $response = $this->sendInvoiceImageWithCurl(
                $phone,
                $tempFile,
                'اختبار إرسال صورة - ' . now()->format('Y-m-d H:i:s')
            );

            // حذف الملف المؤقت
            if (file_exists($tempFile)) {
                unlink($tempFile);
            }

            return response()->json([
                'success' => $response['success'],
                'message' => $response['success'] ? 'تم إرسال الصورة التجريبية بنجاح' : 'فشل إرسال الصورة التجريبية',
                'response' => $response
            ]);

        } catch (\Exception $e) {
            Log::error('Test Image Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطأ في اختبار الصورة',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * الحصول على حالة Instance
     */
    public function getInstanceStatus()
    {
        try {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => "{$this->baseUrl}/instance{$this->instanceId}/instance/status?token={$this->token}",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 15,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_HTTPHEADER => [
                    'User-Agent: Mozilla/5.0 (compatible; COMBO-SAS/1.0)'
                ]
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);

            if ($error) {
                return response()->json([
                    'success' => false,
                    'error' => 'cURL Error: ' . $error
                ], 500);
            }

            if ($httpCode === 200) {
                $status = json_decode($response, true);
                return response()->json([
                    'success' => true,
                    'status' => $status
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'error' => "HTTP Error: {$httpCode} - {$response}"
                ], 400);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * تشخيص مشاكل الإرسال
     */
    public function debugSendIssue(Request $request)
    {
        try {
            $phone = $request->input('phone', '+9647710584672');

            $debug = [
                'configuration' => [
                    'instance_id' => $this->instanceId,
                    'token_length' => strlen($this->token),
                    'base_url' => $this->baseUrl,
                    'full_url' => "{$this->baseUrl}/instance{$this->instanceId}/messages/image"
                ],
                'tests' => []
            ];

            // اختبار 1: إرسال رسالة نصية
            $textTest = $this->sendCurlRequest(
                "{$this->baseUrl}/instance{$this->instanceId}/messages/chat",
                [
                    'token' => $this->token,
                    'to' => $phone,
                    'body' => 'اختبار تشخيص - ' . now()->format('H:i:s')
                ]
            );

            $debug['tests']['text_message'] = [
                'success' => $textTest['success'],
                'data' => $textTest['success'] ? json_decode($textTest['data'], true) : $textTest['error']
            ];

            // اختبار 2: فحص حالة Instance
            $statusResponse = $this->getInstanceStatus();
            $debug['tests']['instance_status'] = $statusResponse->getData();

            // اختبار 3: إرسال صورة تجريبية صغيرة
            $imageTestResponse = $this->testImageSend($request);
            $debug['tests']['image_test'] = $imageTestResponse->getData();

            return response()->json([
                'success' => true,
                'debug_info' => $debug
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'debug_info' => [
                    'exception' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ], 500);
        }
    }

    /**
     * دالة مساعدة لإرسال طلبات cURL
     */
    private function sendCurlRequest($url, $postData)
    {
        try {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($postData),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/x-www-form-urlencoded',
                    'User-Agent: Mozilla/5.0 (compatible; COMBO-SAS/1.0)'
                ]
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);

            if ($error) {
                return [
                    'success' => false,
                    'error' => 'cURL Error: ' . $error
                ];
            }

            if ($httpCode === 200) {
                return [
                    'success' => true,
                    'data' => $response
                ];
            } else {
                return [
                    'success' => false,
                    'error' => "HTTP Error: {$httpCode} - {$response}"
                ];
            }

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * إرسال فاتورة تجريبية نصية
     */
    public function testInvoiceSend(Request $request)
    {
        try {
            $phone = $request->input('phone', '+9647710584672');

            $testMessage = "🧾 فاتورة تجريبية\n\n";
            $testMessage .= "رقم الفاتورة: TEST-" . time() . "\n";
            $testMessage .= "التاريخ: " . date('Y-m-d H:i:s') . "\n";
            $testMessage .= "العميل: عميل تجريبي\n\n";
            $testMessage .= "المنتجات:\n";
            $testMessage .= "1. منتج تجريبي - 1 قطعة - 100 د.ع\n\n";
            $testMessage .= "الإجمالي: 100 دينار عراقي\n\n";
            $testMessage .= "شكراً لكم\nفريق COMBO SAS";

            $response = $this->sendCurlRequest(
                "{$this->baseUrl}/instance{$this->instanceId}/messages/chat",
                [
                    'token' => $this->token,
                    'to' => $phone,
                    'body' => $testMessage
                ]
            );

            if ($response['success']) {
                return response()->json([
                    'success' => true,
                    'message' => 'تم إرسال الفاتورة التجريبية بنجاح',
                    'response' => json_decode($response['data'], true)
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'فشل في إرسال الفاتورة التجريبية',
                    'error' => $response['error']
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('Test invoice error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطأ في إرسال الفاتورة التجريبية',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال عرض سعر كامل مع صورة
     */
    public function sendQuote(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'customer_name' => 'required|string',
            'quote_text' => 'required|string',
            'image_data' => 'nullable|string',
            'additional_message' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'بيانات غير صحيحة',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $results = [];

            // 1. إرسال الرسالة النصية أولاً
            $textMessage = $request->input('quote_text');
            if (!empty($request->input('additional_message'))) {
                $textMessage .= "\n\n" . $request->input('additional_message');
            }
            $textMessage .= "\n\nشكراً لكم\nفريق COMBO SAS";

            $textData = [
                'token' => $this->token,
                'to' => $request->input('phone'),
                'body' => $textMessage
            ];

            $textResponse = $this->sendCurlRequest(
                "{$this->baseUrl}/instance{$this->instanceId}/messages/chat",
                $textData
            );

            $results['text_sent'] = $textResponse['success'];
            if ($textResponse['success']) {
                $results['text_response'] = json_decode($textResponse['data'], true);
            } else {
                $results['text_error'] = $textResponse['error'];
            }

            // 2. إرسال الصورة إذا وجدت
            if (!empty($request->input('image_data'))) {
                sleep(2); // انتظار ثانيتين بين الرسائل

                $imageResponse = $this->sendImageData(
                    $request->input('phone'),
                    $request->input('image_data'),
                    'عرض السعر المفصل - ' . $request->input('customer_name')
                );

                $results['image_sent'] = $imageResponse['success'];
                if ($imageResponse['success']) {
                    $results['image_response'] = $imageResponse['data'];
                } else {
                    $results['image_error'] = $imageResponse['error'];
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'تم إرسال عرض السعر',
                'results' => $results
            ]);

        } catch (\Exception $e) {
            Log::error('WhatsApp Quote Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطأ في إرسال عرض السعر',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال صورة بيانات
     */
    private function sendImageData($phone, $imageData, $caption)
    {
        try {
            // تنظيف بيانات الصورة
            if (strpos($imageData, 'data:image') === 0) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
            }

            $imageContent = base64_decode($imageData);
            if ($imageContent === false) {
                return [
                    'success' => false,
                    'error' => 'فشل في تحويل بيانات الصورة'
                ];
            }

            // التحقق من حجم الصورة
            $imageSizeMB = strlen($imageContent) / (1024 * 1024);
            if ($imageSizeMB > 5) {
                return [
                    'success' => false,
                    'error' => 'حجم الصورة كبير جداً (' . round($imageSizeMB, 2) . 'MB). الحد الأقصى 5MB'
                ];
            }

            $tempFile = tempnam(sys_get_temp_dir(), 'whatsapp_quote_') . '.jpg';
            $bytesWritten = file_put_contents($tempFile, $imageContent);

            if ($bytesWritten === false) {
                return [
                    'success' => false,
                    'error' => 'فشل في إنشاء الملف المؤقت'
                ];
            }

            // إرسال الصورة
            $response = $this->sendInvoiceImageWithCurl($phone, $tempFile, $caption);

            // حذف الملف المؤقت
            if (file_exists($tempFile)) {
                unlink($tempFile);
            }

            return $response;

        } catch (\Exception $e) {
            Log::error('Send Image Data Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}