<?php

namespace App\Jobs;

use App\Transaction;
use App\Services\SapInvoicesService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SyncTransactionToSap implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;
    
    /**
     * The number of seconds to wait before retrying the job.
     *
     * @var int
     */
    public $backoff = 30;

    /**
     * The transaction ID.
     *
     * @var int
     */
    protected $transactionId;

    /**
     * Create a new job instance.
     *
     * @param  int  $transactionId
     * @return void
     */
    public function __construct($transactionId)
    {
        $this->transactionId = $transactionId;
        
        // Set the queue based on configuration
        $this->onQueue(config('sap.queue_name', 'sap-sync'));
    }

    /**
     * Execute the job.
     *
     * @param  \App\Services\SapInvoicesService  $sapService
     * @return void
     */
    public function handle(SapInvoicesService $sapService)
    {
        // Load the transaction from the database using the ID
        $transaction = Transaction::find($this->transactionId);
        
        if (!$transaction) {
            Log::error("SyncTransactionToSap: Transaction #{$this->transactionId} not found");
            return;
        }
        
        // Only process final sales or quotations
        if ($transaction->status != 'final' && 
            $transaction->is_quotation != 1 && 
            $transaction->sub_status != 'quotation') {
            return;
        }
        
        // Push to SAP
        $result = $sapService->pushTransactionToSap($transaction);
        
        if (!$result['success']) {
            Log::error("SyncTransactionToSap: Failed to sync transaction #{$transaction->id}: {$result['message']}");
            
            // If max attempts reached, update transaction with failure status
            if ($this->attempts() >= $this->tries) {
                $transaction->sap_sync_status = 'failed';
                $transaction->save();
            } else {
                // Otherwise throw an exception to retry
                throw new \Exception("Failed to sync: {$result['message']}");
            }
        }
    }
    
    /**
     * The job failed to process.
     *
     * @param  \Exception  $exception
     * @return void
     */
    public function failed($exception)
    {
        // Load the transaction
        $transaction = Transaction::find($this->transactionId);
        
        if ($transaction) {
            // Update transaction status
            $transaction->sap_sync_status = 'failed';
            $transaction->save();
        }
        
        Log::error("SyncTransactionToSap: Job failed for transaction #{$this->transactionId}: {$exception->getMessage()}");
    }
}