<?php

namespace App;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class NumberingSeries extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'numbering_series';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'business_id',
        'prefix',
        'type',
        'last_number',
        'next_number',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'business_id' => 'integer',
        'last_number' => 'integer',
        'next_number' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the business that owns the numbering series.
     */
    public function business()
    {
        return $this->belongsTo(Business::class);
    }

    /**
     * Scope a query to only include series for a specific business.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $businessId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForBusiness($query, $businessId)
    {
        return $query->where('business_id', $businessId);
    }

    /**
     * Scope a query to only include series for a specific type.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  string  $type
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope a query to only include series with a specific prefix.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  string  $prefix
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeWithPrefix($query, $prefix)
    {
        return $query->where('prefix', $prefix);
    }

    /**
     * Get and increment the next number
     *
     * @return string
     */
    public function getNextNumber()
    {
        $number = $this->prefix . $this->next_number;
        
        // Update the numbers
        $this->last_number = $this->next_number;
        $this->next_number = $this->next_number + 1;
        $this->save();
        
        return $number;
    }

    /**
     * Get the next number without incrementing
     *
     * @return string
     */
    public function peekNextNumber()
    {
        return $this->prefix . $this->next_number;
    }

    /**
     * Reset the series to a specific number
     *
     * @param int $number
     * @return void
     */
    public function resetTo($number)
    {
        $this->last_number = $number - 1;
        $this->next_number = $number;
        $this->save();
    }

    /**
     * Check if a number is available
     *
     * @param int $number
     * @return bool
     */
    public function isNumberAvailable($number)
    {
        return $number >= $this->next_number;
    }

    /**
     * Skip to a specific number if it's higher than current
     *
     * @param int $number
     * @return void
     */
    public function skipTo($number)
    {
        if ($number > $this->next_number) {
            $this->last_number = $number;
            $this->next_number = $number + 1;
            $this->save();
        }
    }

    /**
     * Get formatted display of the series
     *
     * @return string
     */
    public function getDisplayFormat()
    {
        return sprintf(
            '%s (%s - %s): Next: %s%d',
            ucfirst($this->type),
            $this->prefix,
            $this->business->name ?? 'Business ' . $this->business_id,
            $this->prefix,
            $this->next_number
        );
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();

        // Ensure next_number is always greater than last_number
        static::saving(function ($model) {
            if ($model->next_number <= $model->last_number) {
                $model->next_number = $model->last_number + 1;
            }
        });
    }
}