<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use App\SapItemsService;
use App\SapProductWarehouse;

class Product extends Model
{
    /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    protected $appends = ['image_url'];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
      protected $casts = [
        'sub_unit_ids' => 'array',
        'sale_price' => 'decimal:4',
        'default_sell_price' => 'decimal:4',
        'purchase_price' => 'decimal:4',
        'default_purchase_price' => 'decimal:4',
        'sap_price' => 'decimal:4',
        'sap_moving_average_price' => 'decimal:4',
        'sap_avg_std_price' => 'decimal:4',
        'sap_on_hand' => 'decimal:4',
        'sap_quantity_on_stock' => 'decimal:4',
        'sap_quantity_ordered' => 'decimal:4',
        'sap_quantity_ordered_by_customers' => 'decimal:4',
        'sap_valid_from' => 'datetime',
        'sap_valid_to' => 'datetime',
        'sap_update_date' => 'datetime',
        'sap_create_date' => 'datetime',
        'sap_last_sync' => 'datetime',
        'sap_frozen' => 'boolean',
        'sap_inventory_item' => 'boolean',
        'sap_purchase_item' => 'boolean',
        'sap_sales_item' => 'boolean',
        'sap_manage_stock_by_warehouse' => 'boolean',
        'uom_prices' => 'array', // Support for JSON array of UoM prices
    ];

    /**
     * Accessor for sale unit
     * 
     * @return string|null
     */
    public function getSaleUnitAttribute($value)
    {
        return $value;
    }

    /**
     * Mutator for sale unit
     * 
     * @param string|null $value
     */
    public function setSaleUnitAttribute($value)
    {
        $this->attributes['sale_unit'] = $value ? trim($value) : null;
    }

    /**
     * Get formatted sale unit with translation support
     * 
     * @return string|null
     */
    public function getFormattedSaleUnitAttribute()
    {
        if (empty($this->sale_unit)) {
            return null;
        }

        // Mapping of common SAP UoM to readable names
        $uomTranslations = [
            'درزن' => 'Dozen',
            'قطعه' => 'Piece',
            'كارتون' => 'Carton',
            'علبة' => 'Box',
            'كغم' => 'Kilogram',
            'لتر' => 'Liter',
            'مل' => 'Milliliter'
        ];

        // Return translated version if exists, otherwise return original
        return $uomTranslations[$this->sale_unit] ?? $this->sale_unit;
    }

    /**
     * Get formatted UoM prices with detailed parsing
     */
    public function getFormattedUomPricesAttribute()
    {
        if (empty($this->uom_prices)) {
            return [];
        }

        return array_map(function($priceList) {
            return [
                'price_list' => $priceList['PriceList'] ?? null,
                'base_price' => $priceList['Price'] ?? null,
                'currency' => $priceList['Currency'] ?? null,
                'uom_prices' => array_map(function($uomPrice) {
                    return [
                        'uom_entry' => $uomPrice['UoMEntry'] ?? null,
                        'price' => $uomPrice['Price'] ?? null,
                        'currency' => $uomPrice['Currency'] ?? null,
                        'reduce_by' => $uomPrice['ReduceBy'] ?? null,
                        'additional_price' => $uomPrice['AdditionalPrice1'] ?? null,
                        'additional_currency' => $uomPrice['AdditionalCurrency1'] ?? null,
                    ];
                }, $priceList['UoMPrices'] ?? [])
            ];
        }, $this->uom_prices);
    }

    /**
     * Get the best selling price from UoM prices
     */
    public function getBestUomPriceAttribute()
    {
        $formattedPrices = $this->formatted_uom_prices;
        
        if (empty($formattedPrices)) {
            return null;
        }

        // Default to first price list
        $priceList = $formattedPrices[0];
        
        // Try to find the most detailed price
        foreach ($priceList['uom_prices'] as $uomPrice) {
            if (!empty($uomPrice['price']) && $uomPrice['price'] > 0) {
                return [
                    'price' => $uomPrice['price'],
                    'currency' => $uomPrice['currency'],
                    'uom_entry' => $uomPrice['uom_entry']
                ];
            }
        }

        // Fallback to base price
        return [
            'price' => $priceList['base_price'],
            'currency' => $priceList['currency'],
            'uom_entry' => null
        ];
    }

    /**
     * Get the primary UoM price from SAP
     */
    public function getPrimaryUomPriceAttribute()
    {
        $bestPrice = $this->best_uom_price;
        return $bestPrice ? $bestPrice['price'] : $this->sale_price;
    }

    /**
     * Get the products image.
     *
     * @return string
     */
    public function getImageUrlAttribute()
    {
        if (! empty($this->image)) {
            $image_url = asset('/uploads/img/'.rawurlencode($this->image));
        } else {
            $image_url = asset('/img/default.png');
        }

        return $image_url;
    }

    /**
     * Get the products image path.
     *
     * @return string
     */
    public function getImagePathAttribute()
    {
        if (! empty($this->image)) {
            $image_path = public_path('uploads').'/'.config('constants.product_img_path').'/'.$this->image;
        } else {
            $image_path = null;
        }

        return $image_path;
    }

    public function product_variations()
    {
        return $this->hasMany(\App\ProductVariation::class);
    }

    /**
     * Get the brand associated with the product.
     */
    public function brand()
    {
        return $this->belongsTo(\App\Brands::class);
    }

    

    /**
     * Get the unit associated with the product.
     */
    public function unit()
    {
        return $this->belongsTo(\App\Unit::class);
    }

    /**
     * Get the unit associated with the product.
     */
    public function second_unit()
    {
        return $this->belongsTo(\App\Unit::class, 'secondary_unit_id');
    }

    /**
     * Get category associated with the product.
     */
    public function category()
    {
        return $this->belongsTo(\App\Category::class);
    }

    /**
     * Get sub-category associated with the product.
     */
    public function sub_category()
    {
        return $this->belongsTo(\App\Category::class, 'sub_category_id', 'id');
    }

    /**
     * Get the tax associated with the product.
     */
    public function product_tax()
    {
        return $this->belongsTo(\App\TaxRate::class, 'tax', 'id');
    }

    /**
     * Get the variations associated with the product.
     */
    public function variations()
    {
        return $this->hasMany(\App\Variation::class);
    }

    /**
     * If product type is modifier get products associated with it.
     */
    public function modifier_products()
    {
        return $this->belongsToMany(\App\Product::class, 'res_product_modifier_sets', 'modifier_set_id', 'product_id');
    }

    /**
     * If product type is modifier get products associated with it.
     */
    public function modifier_sets()
    {
        return $this->belongsToMany(\App\Product::class, 'res_product_modifier_sets', 'product_id', 'modifier_set_id');
    }

    /**
     * Get the purchases associated with the product.
     */
    public function purchase_lines()
    {
        return $this->hasMany(\App\PurchaseLine::class);
    }

    /**
     * Scope a query to only include active products.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('products.is_inactive', 0);
    }

    /**
     * Scope a query to only include inactive products.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeInactive($query)
    {
        return $query->where('products.is_inactive', 1);
    }

    /**
     * Scope a query to only include products for sales.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeProductForSales($query)
    {
        return $query->where('not_for_selling', 0);
    }

    /**
     * Scope a query to only include products not for sales.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeProductNotForSales($query)
    {
        return $query->where('not_for_selling', 1);
    }

    public function product_locations()
    {
        return $this->belongsToMany(\App\BusinessLocation::class, 'product_locations', 'product_id', 'location_id');
    }

    /**
     * Scope a query to only include products available for a location.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForLocation($query, $location_id)
    {
        return $query->where(function ($q) use ($location_id) {
            $q->whereHas('product_locations', function ($query) use ($location_id) {
                $query->where('product_locations.location_id', $location_id);
            });
        });
    }

    /**
     * Get warranty associated with the product.
     */
    public function warranty()
    {
        return $this->belongsTo(\App\Warranty::class);
    }

    public function media()
    {
        return $this->morphMany(\App\Media::class, 'model');
    }

    public function rack_details()
    {
        return $this->hasMany(\App\ProductRack::class);
    }

    /**
     * Get warehouse details for this product with quantities
     */
    public function sapWarehouses()
    {
        return $this->hasMany(SapProductWarehouse::class, 'sap_item_code', 'sap_item_code');
    }

    /**
     * Get total quantity across all warehouses
     */
    public function getTotalWarehouseQuantityAttribute()
    {
        return $this->sapWarehouses()->sum('in_stock');
    }

    /**
     * Get warehouse quantities as array
     */
    public function getWarehouseQuantitiesAttribute()
    {
        return $this->sapWarehouses()
            ->with('warehouseMapping')
            ->get()
            ->mapWithKeys(function ($warehouse) {
                return [
                    $warehouse->warehouse_code => [
                        'name' => $warehouse->warehouseMapping->warehouse_name ?? $warehouse->warehouse_code,
                        'in_stock' => $warehouse->in_stock,
                        'committed' => $warehouse->committed,
                        'ordered' => $warehouse->ordered,
                        'available' => $warehouse->in_stock - $warehouse->committed,
                        'last_sync' => $warehouse->last_sync,
                    ]
                ];
            })
            ->toArray();
    }

    /**
     * Get available quantity for a specific warehouse
     */
    public function getWarehouseQuantity($warehouseCode)
    {
        $warehouse = $this->sapWarehouses()
            ->where('warehouse_code', $warehouseCode)
            ->first();
        
        return $warehouse ? $warehouse->in_stock : 0;
    }

    /**
     * Get available quantity for a specific warehouse (in stock - committed)
     */
    public function getAvailableWarehouseQuantity($warehouseCode)
    {
        $warehouse = $this->sapWarehouses()
            ->where('warehouse_code', $warehouseCode)
            ->first();
        
        return $warehouse ? ($warehouse->in_stock - $warehouse->committed) : 0;
    }

    /**
     * Check if product has stock in any warehouse
     */
    public function hasStockInAnyWarehouse()
    {
        return $this->sapWarehouses()->where('in_stock', '>', 0)->exists();
    }

    /**
     * Get warehouses with stock
     */
    public function getWarehousesWithStock()
    {
        return $this->sapWarehouses()
            ->with('warehouseMapping')
            ->where('in_stock', '>', 0)
            ->get();
    }

    /**
     * Static method to map SAP types to local types
     */
    public static function mapSapTypeToLocal($sapType)
    {
        $mapping = [
            'itItems' => 'single',
            'itService' => 'service',
        ];
        
        return $mapping[$sapType] ?? 'single';
    }

    /**
     * Push this product to SAP
     */
    public function pushToSap()
    {
        $sapService = app(SapItemsService::class);
        
        $itemData = [
            'ItemCode' => $this->sap_item_code ?? $this->sku,
            'ItemName' => $this->name,
            'ForeignName' => $this->sap_foreign_name ?? $this->name,
            'ItemType' => $this->getSapItemType(),
            'InventoryItem' => $this->enable_stock ? 'tYES' : 'tNO',
            'SalesItem' => $this->not_for_selling ? 'tNO' : 'tYES',
            'PurchaseItem' => 'tYES',
            'InventoryUOM' => $this->getUnitCode(),
            'DefaultPrice' => $this->default_sell_price ?? $this->sale_price,
        ];
        
        return $sapService->createOrUpdateItem($this->sap_item_code, $itemData);
    }

    /**
     * Get SAP item type based on local type
     */
    protected function getSapItemType()
    {
        $mapping = [
            'single' => 'itItems',
            'service' => 'itService',
        ];
        
        return $mapping[$this->type] ?? 'itItems';
    }

    /**
     * Get the unit code for SAP
     */
    protected function getUnitCode()
    {
        if ($this->unit && $this->unit->sapUomMapping) {
            return $this->unit->sapUomMapping->uom_code;
        }
        
        return 'EA'; // Default to Each if no mapping exists
    }

    /**
     * Get effective selling price (with fallback logic)
     */
    public function getEffectiveSellingPriceAttribute()
    {
        // Priority order: sale_price -> default_sell_price -> sap_price -> variation price -> 0
        if (!is_null($this->sale_price) && $this->sale_price > 0) {
            return $this->sale_price;
        }
        
        if (!is_null($this->default_sell_price) && $this->default_sell_price > 0) {
            return $this->default_sell_price;
        }
        
        if (!is_null($this->sap_price) && $this->sap_price > 0) {
            return $this->sap_price;
        }
        
        $firstVariation = $this->variations()->first();
        if ($firstVariation && $firstVariation->default_sell_price > 0) {
            return $firstVariation->default_sell_price;
        }
        
        return 0;
    }

    /**
     * Get effective purchase price (with fallback logic)
     */
    public function getEffectivePurchasePriceAttribute()
    {
        // Priority order: purchase_price -> default_purchase_price -> sap_moving_average_price -> variation price -> 0
        if (!is_null($this->purchase_price) && $this->purchase_price > 0) {
            return $this->purchase_price;
        }
        
        if (!is_null($this->default_purchase_price) && $this->default_purchase_price > 0) {
            return $this->default_purchase_price;
        }
        
        if (!is_null($this->sap_moving_average_price) && $this->sap_moving_average_price > 0) {
            return $this->sap_moving_average_price;
        }
        
        $firstVariation = $this->variations()->first();
        if ($firstVariation && $firstVariation->default_purchase_price > 0) {
            return $firstVariation->default_purchase_price;
        }
        
        return 0;
    }

    /**
     * Get profit margin percentage
     */
    public function getProfitMarginAttribute()
    {
        $sellPrice = $this->effective_selling_price;
        $purchasePrice = $this->effective_purchase_price;
        
        if ($purchasePrice == 0) {
            return 0;
        }
        
        return (($sellPrice - $purchasePrice) / $purchasePrice) * 100;
    }

    /**
     * Check if product is synced with SAP
     */
    public function isSyncedWithSap()
    {
        return !is_null($this->sap_item_code) && !is_null($this->sap_last_sync);
    }

    /**
     * Get SAP sync status
     */
    public function getSapSyncStatusAttribute()
    {
        if (is_null($this->sap_item_code)) {
            return 'not_synced';
        }
        
        if (is_null($this->sap_last_sync)) {
            return 'pending';
        }
        
        return $this->sap_sync_status ?? 'unknown';
    }

    /**
     * Get SAP sync status with human readable text
     */
    public function getSapSyncStatusTextAttribute()
    {
        $status = $this->sap_sync_status_attribute;
        
        $statusTexts = [
            'not_synced' => 'غير مزامن',
            'pending' => 'في الانتظار',
            'success' => 'تم بنجاح',
            'failed' => 'فشل',
            'unknown' => 'غير معروف'
        ];
        
        return $statusTexts[$status] ?? $status;
    }

    /**
     * Scope for products synced with SAP
     */
    public function scopeSyncedWithSap($query)
    {
        return $query->whereNotNull('sap_item_code')
                    ->whereNotNull('sap_last_sync');
    }

    /**
     * Scope for products not synced with SAP
     */
    public function scopeNotSyncedWithSap($query)
    {
        return $query->whereNull('sap_item_code');
    }

    /**
     * Scope for products with SAP sync errors
     */
    public function scopeWithSyncErrors($query)
    {
        return $query->where('sap_sync_status', 'failed')
                    ->orWhere('sap_sync_status', 'error');
    }

    /**
     * Get last sync date formatted
     */
    public function getLastSyncFormattedAttribute()
    {
        if (is_null($this->sap_last_sync)) {
            return 'لم يتم المزامنة';
        }
        
        return $this->sap_last_sync->format('Y-m-d H:i:s');
    }

    /**
     * Check if product needs sync (based on update times)
     */
    public function needsSapSync()
    {
        if (is_null($this->sap_last_sync)) {
            return true;
        }
        
        return $this->updated_at > $this->sap_last_sync;
    }

    /**
     * Mark product as needing sync
     */
    public function markForSync()
    {
        $this->update([
            'sap_sync_status' => 'pending',
            'sap_sync_message' => null
        ]);
    }

    /**
     * Update SAP sync status
     */
    public function updateSyncStatus($status, $message = null)
    {
        $this->update([
            'sap_sync_status' => $status,
            'sap_sync_message' => $message,
            'sap_last_sync' => $status === 'success' ? now() : $this->sap_last_sync
        ]);
    }


    public function getSapSyncBadgeAttribute()
{
    $status = $this->sap_sync_status_attribute;
    
    $badges = [
        'not_synced' => '<span class="badge badge-secondary">غير مزامن</span>',
        'pending' => '<span class="badge badge-warning">في الانتظار</span>',
        'success' => '<span class="badge badge-success">مزامن</span>',
        'failed' => '<span class="badge badge-danger">فشل</span>',
        'unknown' => '<span class="badge badge-dark">غير معروف</span>'
    ];
    
    return $badges[$status] ?? $badges['unknown'];
}

/**
 * Get formatted SAP price with currency
 */
public function getFormattedSapPriceAttribute()
{
    if (is_null($this->sap_price)) {
        return 'غير محدد';
    }
    
    $currency = $this->sap_currency ?: 'USD';
    return $currency . ' ' . number_format($this->sap_price, 2);
}

/**
 * Get stock status across all warehouses
 */
public function getOverallStockStatusAttribute()
{
    $totalStock = $this->sap_on_hand ?? 0;
    $totalCommitted = $this->sapWarehouses()->sum('committed');
    $availableStock = $totalStock - $totalCommitted;
    
    if ($totalStock <= 0) {
        return 'out_of_stock';
    }
    
    if ($availableStock <= 0) {
        return 'fully_committed';
    }
    
    $availablePercentage = ($availableStock / $totalStock) * 100;
    
    if ($availablePercentage < 20) {
        return 'low_stock';
    }
    
    return 'in_stock';
}

/**
 * Get warehouses summary for display
 */
public function getWarehousesSummaryAttribute()
{
    $warehouses = $this->sapWarehouses()->with('warehouseMapping')->get();
    
    $summary = [];
    foreach ($warehouses as $warehouse) {
        if ($warehouse->in_stock > 0) {
            $summary[] = sprintf(
                '%s: %.2f (متاح: %.2f)',
                $warehouse->warehouseMapping->warehouse_name ?? $warehouse->warehouse_code,
                $warehouse->in_stock,
                $warehouse->available_quantity
            );
        }
    }
    
    return implode(' | ', $summary);
}

/**
 * Get primary warehouse (with most stock)
 */
public function getPrimaryWarehouseAttribute()
{
    return $this->sapWarehouses()
        ->with('warehouseMapping')
        ->orderBy('in_stock', 'desc')
        ->first();
}

/**
 * Check if product needs reorder
 */
public function getNeedsReorderAttribute()
{
    // Check if stock is below minimum
    if ($this->alert_quantity && $this->sap_on_hand <= $this->alert_quantity) {
        return true;
    }
    
    // Check if below SAP minimum
    if ($this->min_inventory && $this->sap_on_hand <= $this->min_inventory) {
        return true;
    }
    
    return false;
}

/**
 * Get UoM display with unit name
 */
public function getUomDisplayAttribute()
{
    if (!$this->sap_inventory_uom) {
        return $this->unit->short_name ?? 'N/A';
    }
    
    // Try to find mapped unit
    $mapping = SapUomMapping::where('uom_code', $this->sap_inventory_uom)
        ->orWhere('uom_entry', $this->sap_inventory_uom)
        ->first();
    
    if ($mapping && $mapping->unit) {
        return $mapping->unit->short_name . ' (' . $this->sap_inventory_uom . ')';
    }
    
    return $this->sap_inventory_uom;
}

/**
 * Scope for products with SAP errors
 */
public function scopeWithSapErrors($query)
{
    return $query->where('sap_sync_status', 'failed')
        ->orWhereNotNull('sap_sync_message');
}

/**
 * Scope for products needing SAP sync
 */
public function scopeNeedingSapSync($query)
{
    return $query->where(function($q) {
        $q->whereNull('sap_last_sync')
          ->orWhere('sap_sync_status', 'pending')
          ->orWhereColumn('updated_at', '>', 'sap_last_sync');
    });
}



}