<?php

namespace App\Providers;

use App\Services\SapInvoicesService;
use App\Services\SapB1Service;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Event;
use App\Events\SellCreatedOrModified;
use App\Transaction;
use App\Jobs\SyncTransactionToSap;
use Illuminate\Support\Facades\Log;

class SapServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     *
     * @return void
     */
    public function register()
    {
        $this->app->singleton(SapInvoicesService::class, function ($app) {
            return new SapInvoicesService($app->make(SapB1Service::class));
        });
    }

    /**
     * Bootstrap services.
     *
     * @return void
     */
    public function boot()
    {
        // Listen for the SellCreatedOrModified event in a safer way
        // We'll use a closure that only references the transaction ID, not the object
        Event::listen(function (SellCreatedOrModified $event) {
            try {
                // Only use the transaction ID, not the object
                $transactionId = $event->transactionId;
                
                // Skip if no transaction ID
                if (empty($transactionId)) {
                    Log::warning('SapServiceProvider: Event received without a valid transaction ID');
                    return;
                }
                
                // Skip if SAP integration is disabled in config
                if (!config('sap.enabled', false)) {
                    return;
                }
                
                // Load the transaction separately rather than using the event's transaction
                $transaction = Transaction::find($transactionId);
                
                if (!$transaction) {
                    Log::warning("SapServiceProvider: Transaction #{$transactionId} not found");
                    return;
                }
                
                // Only process final sales, quotations, or drafts
                $is_draft = ($transaction->status == 'draft' && $transaction->is_quotation != 1 && $transaction->sub_status != 'quotation');
                
                Log::info("SapServiceProvider: Checking transaction #{$transactionId} for SAP sync", [
                    'status' => $transaction->status,
                    'is_quotation' => $transaction->is_quotation,
                    'sub_status' => $transaction->sub_status,
                    'is_draft' => $is_draft,
                    'will_process' => ($transaction->status == 'final' || $transaction->is_quotation == 1 || $transaction->sub_status == 'quotation' || $is_draft)
                ]);
                
                if ($transaction->status != 'final' && 
                    $transaction->is_quotation != 1 && 
                    $transaction->sub_status != 'quotation' &&
                    !$is_draft) {
                    Log::info("SapServiceProvider: Skipping transaction #{$transactionId} - does not meet sync criteria");
                    return;
                }
                
                // Check if we should sync this type of transaction
                $isQuotation = ($transaction->is_quotation == 1 || $transaction->sub_status == 'quotation');
                $isDraft = ($transaction->status == 'draft' && !$isQuotation);
                
                if ($isQuotation && !config('sap.auto_sync_quotations', true)) {
                    return;
                }
                
                if (!$isQuotation && !$isDraft && !config('sap.auto_sync_sales', true)) {
                    return;
                }
                
                // Enable auto sync for drafts (can be configured later if needed)
                if ($isDraft && !config('sap.auto_sync_drafts', true)) {
                    Log::info("SapServiceProvider: Draft sync disabled for transaction #{$transactionId}");
                    return;
                }
                
                // Use queue if enabled, otherwise process synchronously
                if (config('sap.use_queue', true)) {
                    // This will send just the transaction ID to the job
                    SyncTransactionToSap::dispatch($transactionId)
                        ->onQueue(config('sap.queue_name', 'sap-sync'));
                    
                    $docType = $isDraft ? 'draft' : ($isQuotation ? 'quotation' : 'final sale');
                    Log::info("SapServiceProvider: Queued {$docType} transaction #{$transactionId} for SAP sync");
                } else {
                    // Get the SAP service and push the transaction
                    $sapService = app(SapInvoicesService::class);
                    $result = $sapService->pushTransactionToSap($transaction);
                    
                    $docType = $isDraft ? 'draft' : ($isQuotation ? 'quotation' : 'final sale');
                    Log::info("SapServiceProvider: Processed {$docType} transaction #{$transactionId} for SAP sync. Result: " . 
                        ($result['success'] ? 'Success' : 'Failed - ' . $result['message']));
                }
            } catch (\Exception $e) {
                Log::error("SapServiceProvider: Exception handling event: " . $e->getMessage());
            }
        });
        
        // Also keep transaction observer for manual updates
        Transaction::updated(function (Transaction $transaction) {
            try {
                $isDraft = ($transaction->status == 'draft' && $transaction->is_quotation != 1 && $transaction->sub_status != 'quotation');
                $isQuotation = ($transaction->is_quotation == 1 || $transaction->sub_status == 'quotation');
                $isFinal = ($transaction->status == 'final');
                
                // For drafts, be more aggressive about syncing since sell_lines updates trigger touch()
                $shouldSync = false;
                
                if ($isFinal || $isQuotation) {
                    // Original logic for final and quotations
                    $fieldsToCheck = ['status', 'final_total', 'discount_amount', 'tax_amount', 'shipping_details', 'discount_type', 'additional_notes'];
                    $shouldSync = $transaction->isDirty($fieldsToCheck);
                } elseif ($isDraft) {
                    // For drafts, sync if updated_at changed (includes sell_lines updates via touch())
                    $shouldSync = $transaction->isDirty(['updated_at', 'status', 'final_total', 'discount_amount', 'tax_amount']);
                    
                    Log::info("SapServiceProvider Observer: Draft transaction #{$transaction->id} check", [
                        'isDirty_updated_at' => $transaction->isDirty(['updated_at']),
                        'isDirty_final_total' => $transaction->isDirty(['final_total']),
                        'will_sync' => $shouldSync,
                        'dirty_fields' => array_keys($transaction->getDirty())
                    ]);
                }
                
                if ($shouldSync) {
                    
                    // Skip if SAP integration is disabled in config
                    if (!config('sap.enabled', false)) {
                        return;
                    }
                    
                                    // Check if we should sync this type of transaction
                $isQuotation = ($transaction->is_quotation == 1 || $transaction->sub_status == 'quotation');
                if ($isQuotation && !config('sap.auto_sync_quotations', true)) {
                    return;
                }
                
                if (!$isQuotation && !$isDraft && !config('sap.auto_sync_sales', true)) {
                    return;
                }
                
                // Enable auto sync for drafts in observer too
                if ($isDraft && !config('sap.auto_sync_drafts', true)) {
                    return;
                }
                    
                    // Only use the transaction ID
                    $transactionId = $transaction->id;
                    
                    // Use queue if enabled, otherwise process synchronously
                    if (config('sap.use_queue', true)) {
                        SyncTransactionToSap::dispatch($transactionId)
                            ->onQueue(config('sap.queue_name', 'sap-sync'));
                    } else {
                        // Get the SAP service and push the transaction
                        $sapService = app(SapInvoicesService::class);
                        $sapService->pushTransactionToSap($transaction);
                    }
                }
            } catch (\Exception $e) {
                Log::error("SapServiceProvider: Exception in transaction observer: " . $e->getMessage());
            }
        });
    }
}