<?php
namespace App;

use Illuminate\Database\Eloquent\Model;

class SapProductWarehouse extends Model
{
    protected $fillable = [
        'sap_item_code', 
        'warehouse_code', 
        'in_stock', 
        'committed', 
        'ordered', 
        'last_sync'
    ];

    protected $casts = [
        'in_stock' => 'decimal:4',
        'committed' => 'decimal:4',
        'ordered' => 'decimal:4',
        'last_sync' => 'datetime',
    ];

    /**
     * Get the product that owns this warehouse record
     */
    public function product()
    {
        return $this->belongsTo(Product::class, 'sap_item_code', 'sap_item_code');
    }

    /**
     * Get the warehouse mapping
     */
    public function warehouseMapping()
    {
        return $this->belongsTo(SapWarehouseMapping::class, 'warehouse_code', 'warehouse_code');
    }

    /**
     * Get available quantity (in stock - committed)
     */
    public function getAvailableQuantityAttribute()
    {
        return $this->in_stock - $this->committed;
    }
    
    /**
     * Get available percentage
     */
    public function getAvailablePercentageAttribute()
    {
        if ($this->in_stock == 0) {
            return 0;
        }
        
        return round(($this->available_quantity / $this->in_stock) * 100, 2);
    }
    
    /**
     * Get stock status
     */
    public function getStockStatusAttribute()
    {
        if ($this->in_stock <= 0) {
            return 'out_of_stock';
        }
        
        if ($this->available_quantity <= 0) {
            return 'fully_committed';
        }
        
        if ($this->available_percentage < 20) {
            return 'low_stock';
        }
        
        return 'in_stock';
    }
    
    /**
     * Get stock status label
     */
    public function getStockStatusLabelAttribute()
    {
        $labels = [
            'out_of_stock' => 'نفذ المخزون',
            'fully_committed' => 'محجوز بالكامل',
            'low_stock' => 'مخزون منخفض',
            'in_stock' => 'متوفر'
        ];
        
        return $labels[$this->stock_status] ?? 'غير معروف';
    }

    /**
     * Scope to get only warehouses with stock
     */
    public function scopeWithStock($query)
    {
        return $query->where('in_stock', '>', 0);
    }

    /**
     * Scope to get warehouses for a specific product
     */
    public function scopeForProduct($query, $sapItemCode)
    {
        return $query->where('sap_item_code', $sapItemCode);
    }
    
    /**
     * Scope to get warehouses with available stock
     */
    public function scopeWithAvailableStock($query)
    {
        return $query->whereRaw('in_stock > committed');
    }
}