<?php

namespace App\Services;

use App\NumberingSeries;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class NumberingSeriesService
{
    /**
     * Get next number for contact
     */
    public function getNextContactNumber($businessId, $type = 'customer')
    {
        return DB::transaction(function () use ($businessId, $type) {
            // تحديد البادئة حسب النوع
            $prefix = $this->getPrefixForType($type);
            
            $series = NumberingSeries::where('business_id', $businessId)
                ->where('prefix', $prefix)
                ->where('type', $type)
                ->lockForUpdate()
                ->first();
            
            if (!$series) {
                // إنشاء سلسلة جديدة إذا لم تكن موجودة
                $startNumber = $this->getStartNumberForType($type, $prefix);
                
                $series = NumberingSeries::create([
                    'business_id' => $businessId,
                    'prefix' => $prefix,
                    'type' => $type,
                    'last_number' => $startNumber - 1,
                    'next_number' => $startNumber
                ]);
            }
            
            $nextNumber = $series->next_number;
            
            // تحديث الأرقام
            $series->last_number = $nextNumber;
            $series->next_number = $nextNumber + 1;
            $series->save();
            
            $formattedNumber = str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
            return $prefix . $nextNumber;
        });
    }
    
    /**
     * Get prefix based on contact type
     */
    private function getPrefixForType($type)
    {
        switch($type) {
            case 'supplier':
                return 'LV';
            case 'customer':
            case 'both':
            default:
                return 'IBG';
        }
    }
    
    /**
     * Get starting number based on type and prefix
     */
    private function getStartNumberForType($type, $prefix)
    {
        // يمكنك تخصيص هذه القيم حسب احتياجاتك
        $startNumbers = [
            'IBG' => 5077,  // للعملاء
            'LV' => 100,    // للموردين
        ];
        
        return $startNumbers[$prefix] ?? 1000;
    }
    
    /**
     * Sync with SAP numbering
     */
    public function syncWithSap($businessId, $type = 'customer')
    {
        try {
            $sapService = app(SapBusinessPartnersService::class);
            $prefix = 'IBG';
            
            // الحصول على آخر رقم من SAP
            $lastSapNumber = $sapService->getLastUsedNumber($prefix);
            
            if ($lastSapNumber) {
                $series = NumberingSeries::where('business_id', $businessId)
                    ->where('prefix', $prefix)
                    ->where('type', $type)
                    ->first();
                
                if ($series && $lastSapNumber >= $series->last_number) {
                    $series->last_number = $lastSapNumber;
                    $series->next_number = $lastSapNumber + 1;
                    $series->save();
                    
                    Log::info('Synced numbering series with SAP', [
                        'prefix' => $prefix,
                        'last_number' => $lastSapNumber,
                        'next_number' => $lastSapNumber + 1
                    ]);
                }
            }
        } catch (\Exception $e) {
            Log::error('Failed to sync numbering with SAP: ' . $e->getMessage());
        }
    }
}