<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class SapB1Service
{
    protected $baseUrl;
    protected $username;
    protected $password;
    protected $companyDb;
    protected $sessionTimeout = 300; // minutes

    public function __construct()
    {
        $this->baseUrl = config('sap.base_url');
        $this->username = config('sap.username');
        $this->password = config('sap.password');
        $this->companyDb = config('sap.company_db');
    }

    /**
     * Login to SAP B1 Service Layer
     */
    public function login()
    {
        // Check if we already have a valid session
        if (Cache::has('sap_session')) {
            return Cache::get('sap_session');
        }

        try {
            $response = Http::withoutVerifying() // Skip SSL verification - remove in production
                ->timeout(60) // 60 second timeout
                ->post("{$this->baseUrl}/Login", [
                    'CompanyDB' => $this->companyDb,
                    'UserName' => $this->username,
                    'Password' => $this->password
                ]);

            if ($response->successful()) {
                // Extract session ID from cookies
                $cookies = $response->cookies();
                $sessionId = null;
                
                foreach ($cookies as $cookie) {
                    if ($cookie->getName() === 'B1SESSION') {
                        $sessionId = $cookie->getValue();
                        break;
                    }
                }

                if ($sessionId) {
                    // Cache the session ID
                    Cache::put('sap_session', $sessionId, $this->sessionTimeout);
                    return $sessionId;
                }
            }
            
            Log::error('SAP Login Failed: ' . $response->body());
            return null;
        } catch (\Exception $e) {
            Log::error('SAP Login Exception: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Generic method to make GET requests to SAP B1
     * 
     * CHANGED: Changed from protected to public
     *
     * @param string $endpoint
     * @param array $params
     * @param int $timeout
     * @return array|null
     */
    public function get($endpoint, $params = [], $timeout = 60)
    {
        $sessionId = $this->login();
        if (!$sessionId) {
            throw new \Exception('Failed to authenticate with SAP B1 Service Layer');
        }

        try {
            $url = "{$this->baseUrl}/{$endpoint}";
            
            if (!empty($params)) {
                $url .= '?' . http_build_query($params);
            }
            
            Log::info("SAP GET Request", [
                'url' => $url,
                'params' => $params,
                'endpoint' => $endpoint
            ]);
            
            $response = Http::withoutVerifying()
                ->timeout($timeout)
                ->withCookies(['B1SESSION' => $sessionId], parse_url($this->baseUrl, PHP_URL_HOST))
                ->get($url);

            if ($response->successful()) {
                $responseData = $response->json();
                
                Log::info("SAP GET Response", [
                    'endpoint' => $endpoint,
                    'response_keys' => array_keys($responseData ?? []),
                    'value_count' => isset($responseData['value']) ? count($responseData['value']) : 0,
                    'has_nextLink' => isset($responseData['odata.nextLink']),
                    'has_count' => isset($responseData['@odata.count']),
                    'total_count' => $responseData['@odata.count'] ?? 'unknown'
                ]);
                
                return $responseData;
            }
            
            Log::error("Failed to get {$endpoint}", [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return null;
        } catch (\Exception $e) {
            Log::error("SAP GET Exception for {$endpoint}: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Generic method to make POST requests to SAP B1
     * 
     * CHANGED: Changed from protected to public
     *
     * @param string $endpoint
     * @param array $data
     * @param int $timeout
     * @return array|null
     */
    public function post($endpoint, $data, $timeout = 60)
{
    $sessionId = $this->login();
    if (!$sessionId) {
        throw new \Exception('Failed to authenticate with SAP B1 Service Layer');
    }

    try {
        $url = "{$this->baseUrl}/{$endpoint}";
        
        \Log::info("SAP POST Request", [
            'url' => $url,
            'data' => $data,
            'session' => substr($sessionId, 0, 10) . '...' // جزء من الجلسة للأمان
        ]);
        
        $response = Http::withoutVerifying()
            ->timeout($timeout)
            ->withCookies(['B1SESSION' => $sessionId], parse_url($this->baseUrl, PHP_URL_HOST))
            ->withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])
            ->post($url, $data);

        // تسجيل الاستجابة الخام
        \Log::info("SAP POST Response", [
            'status' => $response->status(),
            'headers' => $response->headers(),
            'body' => $response->body(),
            'successful' => $response->successful()
        ]);

        if ($response->successful()) {
            $jsonResponse = $response->json();
            
            // في حالة SAP، قد تكون الاستجابة في Location header
            if (empty($jsonResponse) && $response->status() === 201) {
                $locationHeader = $response->header('Location');
                \Log::info('SAP returned 201 with Location header', [
                    'location' => $locationHeader
                ]);
                
                // استخراج CardCode من Location header
                if ($locationHeader && preg_match("/BusinessPartners\('([^']+)'\)/", $locationHeader, $matches)) {
                    return ['CardCode' => $matches[1]];
                }
            }
            
            return $jsonResponse;
        }
        
        // في حالة الخطأ، نسجل التفاصيل
        $errorBody = $response->body();
        $errorJson = json_decode($errorBody, true);
        
        \Log::error("SAP POST Error", [
            'endpoint' => $endpoint,
            'status' => $response->status(),
            'error' => $errorJson ?? $errorBody
        ]);
        
        // رمي استثناء مع تفاصيل الخطأ
        if (isset($errorJson['error']['message']['value'])) {
            throw new \Exception("SAP Error: " . $errorJson['error']['message']['value']);
        }
        
        throw new \Exception("SAP Error: " . $errorBody);
        
    } catch (\Exception $e) {
        \Log::error("SAP POST Exception for {$endpoint}: " . $e->getMessage());
        throw $e;
    }
}

    /**
     * Generic method to make PATCH requests to SAP B1
     * 
     * CHANGED: Changed from protected to public
     *
     * @param string $endpoint
     * @param array $data
     * @param int $timeout
     * @return array|null
     */
    public function patch($endpoint, $data, $timeout = 60)
    {
        $sessionId = $this->login();
        if (!$sessionId) {
            throw new \Exception('Failed to authenticate with SAP B1 Service Layer');
        }

        try {
            $response = Http::withoutVerifying()
                ->timeout($timeout)
                ->withCookies(['B1SESSION' => $sessionId], parse_url($this->baseUrl, PHP_URL_HOST))
                ->patch("{$this->baseUrl}/{$endpoint}", $data);

            if ($response->successful()) {
                return $response->json();
            }
            
            Log::error("Failed to patch {$endpoint}: " . $response->body());
            return null;
        } catch (\Exception $e) {
            Log::error("SAP PATCH Exception for {$endpoint}: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Logout from SAP B1 Service Layer
     */
    public function logout()
    {
        $sessionId = Cache::get('sap_session');
        if (!$sessionId) {
            return true;
        }

        try {
            $response = Http::withoutVerifying()
                ->withCookies(['B1SESSION' => $sessionId], parse_url($this->baseUrl, PHP_URL_HOST))
                ->post("{$this->baseUrl}/Logout");

            Cache::forget('sap_session');
            return $response->successful();
        } catch (\Exception $e) {
            Log::error('SAP Logout Exception: ' . $e->getMessage());
            Cache::forget('sap_session');
            return false;
        }
    }
    public function createContactInSapAndGetCardCode($contact)
{
    try {
        // تحديد نوع البطاقة بدقة
        $cardType = $this->getCardTypeForSap($contact->type);
        
        // تحضير البيانات الأساسية مع إضافة CardCode يدوياً
        $bpData = [
            'CardCode' => $this->generateManualCardCode($contact, $cardType),
            'CardName' => $this->sanitizeForSap($contact->name ?? $contact->supplier_business_name),
            'CardType' => $cardType,
        ];

        // إضافة الحقول الإضافية
        $this->addOptionalFieldsToBusinessPartner($contact, $bpData);
        
        // إرسال إلى SAP
        $response = $this->createBusinessPartner($bpData);
        
        // معالجة الاستجابة
        $cardCode = $this->extractCardCodeFromResponse($response, $contact);
        
        if ($cardCode) {
            return [
                'CardCode' => $cardCode,
                'CardName' => $contact->name
            ];
        }
        
        throw new \Exception('لم يتم الحصول على CardCode من SAP');
        
    } catch (\Exception $e) {
        Log::error('فشل إنشاء جهة الاتصال في SAP: ' . $e->getMessage(), [
            'contact_id' => $contact->id,
            'contact_name' => $contact->name
        ]);
        
        throw $e;
    }
}

private function generateManualCardCode($contact, $cardType)
{
    // توليد CardCode يدوياً
    $prefix = $cardType == 'cSupplier' ? 'SUP' : 'CUS';
    $uniqueId = str_pad($contact->id, 6, '0', STR_PAD_LEFT);
    
    return $prefix . $uniqueId;
}
}