<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use App\Contact;

class SapBusinessPartnersService extends SapB1Service
{
    /**
     * Get business partners from SAP B1
     */
    public function getBusinessPartners($filter = null, $select = null, $top = 5000, $skip = 0, $timeout = 60000)
    {
        $queryParams = [];
        if ($filter) {
            $queryParams['$filter'] = $filter;
        }
        if ($select) {
            $queryParams['$select'] = $select;
        }
        if ($top > 0) {
            $queryParams['$top'] = $top;
        }
        if ($skip > 0) {
            $queryParams['$skip'] = $skip;
        }
        
        return $this->get('BusinessPartners', $queryParams, $timeout);
    }

    /**
     * Get a single business partner by CardCode
     */
    public function getBusinessPartner($cardCode, $expand = null)
    {
        $queryParams = [];
        if ($expand) {
            $queryParams['$expand'] = $expand;
        }
        
        return $this->get("BusinessPartners('{$cardCode}')", $queryParams);
    }

    /**
     * Create a new business partner in SAP B1
     */
   private function createBusinessPartner($bpData)
{
    try {
        // إرسال الطلب بدون CardCode
        $response = $this->post('BusinessPartners', $bpData);
        
        return $response;
    } catch (\Exception $e) {
        // سجل الخطأ
        Log::error('خطأ في إنشاء شريك العمل', [
            'error' => $e->getMessage(),
            'data' => $bpData
        ]);
        
        throw $e;
    }
}

private function generateManualCardCode($bpData)
{
    // تحديد البادئة بناءً على نوع العميل
    $prefix = $bpData['CardType'] == 'cSupplier' ? 'SUP' : 'IBG';
    
    // توليد رقم عشوائي بين 5000 و 10000
$randomNumber = rand(5000, 10000);
$formattedNumber = str_pad($randomNumber, 5, '0', STR_PAD_LEFT);
$cardCode = $prefix . $formattedNumber;    
    // إنشاء CardCode
    return $prefix . $randomNumber;
}

    /**
     * Update an existing business partner in SAP B1
     */
    public function updateBusinessPartner($cardCode, $bpData)
    {
        return $this->patch("BusinessPartners('{$cardCode}')", $bpData);
    }

    /**
     * جلب جميع العملاء من SAP مع حل مشكلة التوقف عند 2373
     */
    public function fetchAllCustomersFixed()
    {
        $allCustomers = [];
        $skip = 0;
        $top = 500; // حجم دفعة أصغر لضمان عدم فقدان البيانات
        $hasMore = true;
        $iteration = 0;
        $maxIterations = 100;
        $consecutiveEmptyBatches = 0;
        $maxEmptyBatches = 3;
        
        // فلتر للعملاء فقط
        $filter = "CardType eq 'cCustomer'";
        
        // تحديد الحقول الأساسية فقط لتقليل حجم الاستجابة
        $select = "CardCode,CardName,CardType,GroupCode,Address,Phone1,EmailAddress,UpdateDate";
        
        Log::info("بدء جلب جميع العملاء من SAP...");
        
        while ($hasMore && $iteration < $maxIterations) {
            $iteration++;
            
            try {
                Log::info("جلب الدفعة رقم {$iteration}: skip={$skip}, top={$top}");
                
                // محاولة جلب البيانات
                $result = $this->getBusinessPartners($filter, $select, $top, $skip, 180000); // 3 دقائق timeout
                
                if (empty($result) || empty($result['value'])) {
                    $consecutiveEmptyBatches++;
                    Log::warning("دفعة فارغة رقم {$consecutiveEmptyBatches} عند skip={$skip}");
                    
                    // إذا حصلنا على دفعات فارغة متتالية، نحاول القفز للأمام
                    if ($consecutiveEmptyBatches >= $maxEmptyBatches) {
                        // التحقق من وجود المزيد من البيانات بالقفز للأمام
                        $testSkip = $skip + ($top * 5); // قفزة اختبارية
                        Log::info("محاولة القفز للأمام للتحقق من وجود المزيد من البيانات عند skip={$testSkip}");
                        
                        $testResult = $this->getBusinessPartners($filter, $select, 10, $testSkip, 60000);
                        
                        if (!empty($testResult['value'])) {
                            // وجدنا بيانات، نستمر من هنا
                            $skip = $testSkip;
                            $consecutiveEmptyBatches = 0;
                            Log::info("وجدنا بيانات عند skip={$skip}, نستمر من هنا");
                            continue;
                        } else {
                            // لا توجد بيانات أخرى
                            Log::info("لا توجد المزيد من البيانات");
                            $hasMore = false;
                            break;
                        }
                    }
                    
                    $skip += $top;
                    continue;
                }

                $consecutiveEmptyBatches = 0; // إعادة تعيين العداد
                $batchCount = count($result['value']);
                $allCustomers = array_merge($allCustomers, $result['value']);
                
                Log::info("تم جلب {$batchCount} عميل في الدفعة {$iteration} (المجموع: " . count($allCustomers) . ")");
                
                // التحقق من وجود المزيد من البيانات
                if ($batchCount < $top) {
                    // إذا كانت الدفعة أقل من المطلوب، قد تكون الأخيرة
                    Log::info("الدفعة أقل من الحجم المطلوب، التحقق من وجود المزيد");
                    
                    // محاولة جلب دفعة صغيرة للتأكد
                    $testResult = $this->getBusinessPartners($filter, $select, 10, $skip + $top, 60000);
                    if (empty($testResult['value'])) {
                        $hasMore = false;
                        Log::info("تأكدنا من عدم وجود المزيد من البيانات");
                    } else {
                        $skip += $top;
                    }
                } else {
                    $skip += $top;
                }
                
                // إضافة تأخير لتجنب إرهاق الخادم
                usleep(300000); // 0.3 ثانية
                
            } catch (\Exception $e) {
                Log::error("خطأ في جلب الدفعة {$iteration}: " . $e->getMessage());
                
                if (strpos($e->getMessage(), 'timed out') !== false) {
                    // في حالة timeout، نقلل حجم الدفعة
                    if ($top > 100) {
                        $top = max(100, intval($top / 2));
                        Log::info("timeout - تقليل حجم الدفعة إلى {$top}");
                    } else {
                        // إذا كان حجم الدفعة صغير جداً، نتخطى هذه الدفعة
                        $skip += $top;
                        Log::warning("تخطي الدفعة بسبب timeout متكرر");
                    }
                } else {
                    // أخطاء أخرى
                    $skip += $top;
                }
            }
        }

        $totalFetched = count($allCustomers);
        Log::info("=== انتهى جلب العملاء ===");
        Log::info("إجمالي العملاء المجلوبين: {$totalFetched}");
        Log::info("عدد الدفعات: {$iteration}");
        
        return [
            'customers' => $allCustomers,
            'count' => $totalFetched,
            'iterations' => $iteration
        ];
    }

    /**
     * طريقة بديلة: جلب العملاء باستخدام OrderBy لضمان عدم فقدان أي سجل
     */
    public function fetchAllCustomersWithOrderBy()
    {
        $allCustomers = [];
        $lastCardCode = '';
        $batchSize = 500;
        $iteration = 0;
        $maxIterations = 100;
        
        Log::info("بدء جلب العملاء باستخدام OrderBy...");
        
        while ($iteration < $maxIterations) {
            $iteration++;
            
            try {
                // بناء الفلتر
                $filter = "CardType eq 'cCustomer'";
                if (!empty($lastCardCode)) {
                    $filter .= " and CardCode gt '{$lastCardCode}'";
                }
                
                // إضافة OrderBy للترتيب حسب CardCode
                $queryParams = [
                    '$filter' => $filter,
                    '$orderby' => 'CardCode',
                    '$top' => $batchSize
                ];
                
                Log::info("جلب الدفعة {$iteration} بعد CardCode: '{$lastCardCode}'");
                
                $result = $this->get('BusinessPartners', $queryParams, 180000);
                
                if (empty($result) || empty($result['value'])) {
                    Log::info("لا توجد المزيد من البيانات");
                    break;
                }
                
                $batchCount = count($result['value']);
                $allCustomers = array_merge($allCustomers, $result['value']);
                
                // تحديث آخر CardCode
                $lastCardCode = $result['value'][$batchCount - 1]['CardCode'];
                
                Log::info("تم جلب {$batchCount} عميل (المجموع: " . count($allCustomers) . ")");
                
                // إذا كانت الدفعة أقل من المطلوب، قد تكون الأخيرة
                if ($batchCount < $batchSize) {
                    Log::info("وصلنا لنهاية البيانات");
                    break;
                }
                
                usleep(300000); // تأخير 0.3 ثانية
                
            } catch (\Exception $e) {
                Log::error("خطأ في الدفعة {$iteration}: " . $e->getMessage());
                
                // في حالة الخطأ، نحاول تقليل حجم الدفعة
                if ($batchSize > 100) {
                    $batchSize = intval($batchSize / 2);
                    Log::info("تقليل حجم الدفعة إلى {$batchSize}");
                } else {
                    // إذا فشلت المحاولات، نتوقف
                    Log::error("فشلت المحاولات المتكررة، إيقاف العملية");
                    break;
                }
            }
        }
        
        $totalFetched = count($allCustomers);
        Log::info("انتهى جلب العملاء: {$totalFetched} عميل");
        
        return [
            'customers' => $allCustomers,
            'count' => $totalFetched,
            'iterations' => $iteration
        ];
    }

    /**
     * Get all business partners using cursor-based pagination - محسّنة للبيانات الكبيرة
     */
    public function getAllBusinessPartners($filter = null, $select = null)
    {
        // أولاً: جلب العدد الإجمالي من SAP
        try {
            $countParams = ['$count' => 'true', '$top' => 1];
            if ($filter) {
                $countParams['$filter'] = $filter;
            }
            
            Log::info("جلب العدد الإجمالي من SAP", ['filter' => $filter]);
            $countResult = $this->get('BusinessPartners', $countParams, 60000);
            
            $totalAvailable = isset($countResult['@odata.count']) ? $countResult['@odata.count'] : 'unknown';
            Log::info("العدد الإجمالي المتاح في SAP: {$totalAvailable}");
            
        } catch (\Exception $e) {
            Log::warning("فشل في جلب العدد الإجمالي: " . $e->getMessage());
            $totalAvailable = 'unknown';
        }
        
        $allPartners = [];
        $batchSize = 200; // حجم دفعة أصغر للبداية
        $iteration = 0;
        $maxIterations = 10000; // زيادة كبيرة للحد الأقصى
        $consecutiveFailures = 0;
        $maxConsecutiveFailures = 3;
        $totalFetched = 0;
        $lastCardCode = '';
        
        Log::info("بدء جلب جميع شركاء الأعمال من SAP باستخدام cursor pagination", [
            'filter' => $filter,
            'select' => $select,
            'batch_size' => $batchSize,
            'total_available_in_sap' => $totalAvailable
        ]);

        while ($iteration < $maxIterations) {
            $iteration++;
            
            try {
                // بناء فلتر مع cursor-based pagination
                $cursorFilter = $filter;
                if (!empty($lastCardCode)) {
                    $cursorCondition = "CardCode gt '{$lastCardCode}'";
                    $cursorFilter = $cursorFilter ? "({$filter}) and {$cursorCondition}" : $cursorCondition;
                }
                
                $queryParams = [
                    '$top' => $batchSize,
                    '$orderby' => 'CardCode asc',
                    '$count' => 'true' // للحصول على معلومات إضافية
                ];
                
                if ($cursorFilter) {
                    $queryParams['$filter'] = $cursorFilter;
                }
                
                if ($select) {
                    $queryParams['$select'] = $select;
                }
                
                Log::info("جلب الدفعة {$iteration} بعد CardCode: '{$lastCardCode}'", [
                    'filter' => $cursorFilter,
                    'batch_size' => $batchSize,
                    'total_fetched_so_far' => $totalFetched
                ]);
                
                $result = $this->get('BusinessPartners', $queryParams, 180000); // 3 دقائق timeout
                
                if (empty($result) || empty($result['value']) || count($result['value']) == 0) {
                    Log::info("لا توجد المزيد من البيانات، انتهاء التصفح", [
                        'total_fetched' => $totalFetched,
                        'iteration' => $iteration,
                        'result_empty' => empty($result),
                        'value_empty' => empty($result['value'] ?? null)
                    ]);
                    break;
                }

                $batchCount = count($result['value']);
                $allPartners = array_merge($allPartners, $result['value']);
                $totalFetched += $batchCount;
                $consecutiveFailures = 0; // إعادة تعيين عداد الفشل عند النجاح
                
                // تحديث آخر CardCode
                $lastCardCode = $result['value'][$batchCount - 1]['CardCode'];
                
                Log::info("تم جلب دفعة بنجاح", [
                    'batch_count' => $batchCount,
                    'iteration' => $iteration,
                    'total_fetched' => $totalFetched,
                    'last_card_code' => $lastCardCode,
                    'has_nextLink' => isset($result['odata.nextLink']),
                    'batch_vs_requested' => "{$batchCount}/{$batchSize}"
                ]);
                
                // التحقق من nextLink
                if (isset($result['odata.nextLink'])) {
                    Log::info("يوجد nextLink، المتابعة...", ['nextLink' => $result['odata.nextLink']]);
                } else {
                    Log::info("لا يوجد nextLink في الاستجابة");
                }
                
                // التحقق من انتهاء البيانات يعتمد على nextLink وليس حجم الدفعة
                if (!isset($result['odata.nextLink'])) {
                    Log::info("انتهت البيانات - لا يوجد nextLink", [
                        'batch_count' => $batchCount,
                        'batch_size' => $batchSize,
                        'has_nextLink' => false
                    ]);
                    break;
                } else {
                    Log::info("يوجد المزيد من البيانات - nextLink موجود", [
                        'batch_count' => $batchCount,
                        'nextLink' => substr($result['odata.nextLink'], -50) // آخر 50 حرف من الرابط
                    ]);
                }
                
                // تأخير قصير بين الدفعات لتجنب إرهاق الخادم
                if ($iteration % 10 == 0) {
                    Log::info("تأخير قصير بعد 10 دفعات...");
                    sleep(1);
                }
                
                // مراقبة الذاكرة
                if ($iteration % 20 == 0) {
                    $memoryMB = memory_get_usage(true) / 1024 / 1024;
                    Log::info("معلومات الذاكرة: {$memoryMB} MB");
                    if ($memoryMB > 512) { // أكثر من 512MB
                        Log::warning("استخدام ذاكرة عالي، تقليل حجم الدفعة");
                        $batchSize = max(100, intval($batchSize * 0.8));
                    }
                }
                
            } catch (\Exception $e) {
                $consecutiveFailures++;
                Log::error("خطأ في جلب دفعة شركاء الأعمال {$iteration} (بعد CardCode: {$lastCardCode}): " . $e->getMessage());
                
                // إذا كان timeout أو خطأ في الاتصال، نقلل حجم الدفعة
                if (strpos($e->getMessage(), 'timed out') !== false || 
                    strpos($e->getMessage(), 'timeout') !== false ||
                    strpos($e->getMessage(), 'Connection timed out') !== false ||
                    strpos($e->getMessage(), 'memory') !== false) {
                    
                    if ($batchSize > 50) {
                        $batchSize = max(50, intval($batchSize / 2));
                        Log::info("timeout أو مشكلة، تقليل حجم الدفعة إلى {$batchSize} وإعادة المحاولة");
                        // لا نغير lastCardCode، لإعادة المحاولة بنفس المكان
                        continue;
                    }
                }
                
                // للأخطاء الأخرى، نحاول الانتقال للدفعة التالية
                if ($consecutiveFailures < $maxConsecutiveFailures) {
                    Log::info("محاولة المتابعة بعد خطأ...");
                    
                    // محاولة تخطي هذا السجل المشكل
                    if (!empty($lastCardCode)) {
                        // الحصول على CardCode التالي لتخطي المشكل
                        try {
                            $skipFilter = $filter ? "({$filter}) and CardCode gt '{$lastCardCode}'" : "CardCode gt '{$lastCardCode}'";
                            $skipResult = $this->get('BusinessPartners', [
                                '$filter' => $skipFilter,
                                '$select' => 'CardCode',
                                '$top' => 1,
                                '$orderby' => 'CardCode asc'
                            ], 30000);
                            
                            if (!empty($skipResult['value'])) {
                                $lastCardCode = $skipResult['value'][0]['CardCode'];
                                Log::info("تخطي إلى CardCode التالي: {$lastCardCode}");
                            }
                        } catch (\Exception $skipError) {
                            Log::warning("فشل في العثور على CardCode التالي: " . $skipError->getMessage());
                        }
                    }
                    
                    // تأخير أطول بعد الخطأ
                    sleep(3);
                } else {
                    Log::error("فشل متكرر ({$consecutiveFailures} مرات)، إيقاف العملية");
                    break;
                    
                    // إذا لم نحصل على أي بيانات، نرمي الخطأ
                    if ($totalFetched == 0) {
                        throw new \Exception("فشل في جلب أي بيانات من SAP: " . $e->getMessage());
                    }
                }
            }
        }

        // تحقق من الوصول للحد الأقصى للتكرارات
        if ($iteration >= $maxIterations) {
            Log::warning("تم الوصول للحد الأقصى للتكرارات ({$maxIterations})، قد يكون هناك المزيد من البيانات");
        }

        Log::info("انتهى جلب شركاء الأعمال: {$totalFetched} شريك بعد {$iteration} تكرار");
        return $allPartners;
    }

    /**
     * Sync all contacts from SAP B1 to local database
     */
    public function syncAllContacts($fullSync = false)
    {
        $stats = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0,
            'batches_processed' => 0,
            'customers_synced' => 0,
            'suppliers_synced' => 0
        ];

        try {
            // Determine last sync time for incremental updates
            $lastSync = null;
            if (!$fullSync) {
                $latestSync = Contact::whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first();
                
                if ($latestSync) {
                    $lastSync = $latestSync->sap_last_sync->format('Y-m-d\TH:i:s');
                    Log::info("Performing incremental sync since: {$lastSync}");
                } else {
                    Log::info("No previous sync found. Performing full sync.");
                    $fullSync = true;
                }
            }

            // Build filter for business partners
            $filter = null;
            if (!$fullSync && $lastSync) {
                $filter = "UpdateDate ge '{$lastSync}'";
            }

            // Fields to select
            $select = "CardCode,CardName,CardType,GroupCode,Address,ZipCode,City,County,Country,Phone1,Phone2,EmailAddress,Currency,FederalTaxID,CreditLimit,PriceListNum,SalesPersonCode,Notes,CurrentAccountBalance,UpdateDate,CardForeignName";
            
            // First sync customers
            Log::info("بدء مزامنة العملاء...");
            $customerFilter = ($filter ? $filter . " and " : "") . "CardType eq 'cCustomer'";
            $customerStats = ['total' => 0, 'created' => 0, 'updated' => 0, 'failed' => 0, 'skipped' => 0];
            $this->syncContacts('customer', $customerFilter, $select, $customerStats);
            $stats['customers_synced'] = $customerStats['total'];
            $stats['batches_processed']++;
            
            // Merge customer stats
            $stats['total'] += $customerStats['total'];
            $stats['created'] += $customerStats['created'];
            $stats['updated'] += $customerStats['updated'];
            $stats['failed'] += $customerStats['failed'];
            $stats['skipped'] += $customerStats['skipped'];
            
            Log::info("انتهى مزامنة العملاء: {$customerStats['total']} عميل");
            
            // Then sync suppliers
            Log::info("بدء مزامنة الموردين...");
            $supplierFilter = ($filter ? $filter . " and " : "") . "CardType eq 'cSupplier'";
            $supplierStats = ['total' => 0, 'created' => 0, 'updated' => 0, 'failed' => 0, 'skipped' => 0];
            $this->syncContacts('supplier', $supplierFilter, $select, $supplierStats);
            $stats['suppliers_synced'] = $supplierStats['total'];
            $stats['batches_processed']++;
            
            // Merge supplier stats
            $stats['total'] += $supplierStats['total'];
            $stats['created'] += $supplierStats['created'];
            $stats['updated'] += $supplierStats['updated'];
            $stats['failed'] += $supplierStats['failed'];
            $stats['skipped'] += $supplierStats['skipped'];
            
            Log::info("انتهى مزامنة الموردين: {$supplierStats['total']} مورد");
            
            // Then sync lead types (if needed)
            // Note: SAP B1 doesn't support 'cBoth', only 'cCustomer', 'cSupplier', or 'cLid'
            // If you need to sync leads, uncomment the following:
            // $leadFilter = ($filter ? $filter . " and " : "") . "CardType eq 'cLid'";
            // $this->syncContacts('lead', $leadFilter, $select, $stats);

            Log::info("مجموع نتائج المزامنة: {$stats['total']} جهة اتصال ({$stats['customers_synced']} عميل، {$stats['suppliers_synced']} مورد)");
            return $stats;
        } catch (\Exception $e) {
            Log::error("SAP Sync All Contacts Exception: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Sync contacts of a specific type
     */
    protected function syncContacts($type, $filter, $select, &$stats)
    {
        // Get all business partners from SAP
        $partners = $this->getAllBusinessPartners($filter, $select);
        $stats['total'] += count($partners);
        Log::info("Retrieved " . count($partners) . " {$type} business partners from SAP.");

        // Process partners
        foreach ($partners as $partner) {
            try {
                $result = $this->processSapPartner($partner, $type);
                
                if ($result === 'created') {
                    $stats['created']++;
                } elseif ($result === 'updated') {
                    $stats['updated']++;
                } elseif ($result === 'skipped') {
                    $stats['skipped']++;
                }
            } catch (\Exception $e) {
                Log::error("Error processing partner {$partner['CardCode']}: " . $e->getMessage());
                $stats['failed']++;
            }
        }
    }

    /**
     * Process a single SAP business partner and create/update local contact
     */
    public function processSapPartner($partner, $type)
    {
        $cardCode = $partner['CardCode'];
        
        // Try to find existing contact by SAP card code
        $contact = Contact::where('card_code', $cardCode)->first();
        
        // If not found, try to find by contact_id
        if (!$contact) {
            $contact = Contact::where('contact_id', $cardCode)->first();
        }
        
        // Prepare data for create/update
        $data = [
            'card_code' => $cardCode,
            'card_name' => $partner['CardName'] ?? null,
            'name' => $partner['CardName'] ?? null,
            'card_type' => $partner['CardType'] ?? null,
            'group_code' => $partner['GroupCode'] ?? null,
            'address_line_1' => $partner['Address'] ?? null,
            'zip_code' => $partner['ZipCode'] ?? null,
            'city' => $partner['City'] ?? null,
            'state' => $partner['County'] ?? null,
            'country' => $partner['Country'] ?? null,
            'mobile' => $partner['Phone1'] ?? 'N/A',
            'email' => $partner['EmailAddress'] ?? null,
            'federal_tax_id' => $partner['FederalTaxID'] ?? null,
            'tax_number' => $partner['FederalTaxID'] ?? null,
            'balance' => $partner['CurrentAccountBalance'] ?? 0.00,
            'sap_price_list' => $partner['PriceListNum'] ?? null,
            'sap_sales_person_code' => $partner['SalesPersonCode'] ?? null,
            'sap_credit_limit' => $partner['CreditLimit'] ?? null,
            'sap_foreign_name' => $partner['CardForeignName'] ?? null,
            'sap_notes' => $partner['Notes'] ?? null,
            'sap_update_date' => !empty($partner['UpdateDate']) ? $partner['UpdateDate'] : null,
            'sap_last_sync' => now(),
            'sap_sync_status' => 'success',
            'type' => $type, // 'customer', 'supplier', or 'both'
            'contact_id' => $cardCode,
            'custom_field1' => null, // Map custom fields as needed
            'contact_status' => 'active',
            'business_id' => $this->getBusinessId(),
            'created_by' => $this->getUserId(),
        ];
        
        if ($contact) {
            // Update existing contact
            // Only update fields that exist in the contact model
            foreach ($data as $key => $value) {
                if (in_array($key, array_keys($contact->getAttributes()))) {
                    $contact->{$key} = $value;
                }
            }
            
            $contact->save();
            return 'updated';
        } else {
            // Create new contact
            $contact = new Contact();
            
            // Set all fields
            foreach ($data as $key => $value) {
                $contact->{$key} = $value;
            }
            
            // Generate contact_id if it doesn't exist
            if (empty($contact->contact_id)) {
                $contact->contact_id = $cardCode;
            }
            
            // Add default required fields
            $contact->business_id = $this->getBusinessId();
            
            $contact->save();
            return 'created';
        }
    }

    /**
     * Push a contact to SAP B1
     */
    public function pushContactToSap($contact)
    {
        try {
            $cardType = 'cCustomer';
            if ($contact->type == 'supplier') {
                $cardType = 'cSupplier';
            } elseif ($contact->type == 'lead') {
                $cardType = 'cLid';
            }
            // Note: SAP B1 doesn't support 'both' type, handle it as customer by default
            
            $bpData = [
                'CardCode' => $contact->card_code ?? $contact->contact_id,
                'CardName' => $contact->name,
                'CardType' => $cardType,
                'GroupCode' => $contact->group_code ?? null,
                'Address' => $contact->billing_address ?? null,
                'ZipCode' => $contact->billing_zip ?? null,
                'City' => $contact->billing_city ?? null,
                'County' => $contact->billing_state ?? null,
                'Country' => $contact->billing_country ?? null,
                'Phone1' => $contact->phone1 ?? $contact->billing_phone ?? null,
                'Phone2' => $contact->phone2 ?? null,
                'EmailAddress' => $contact->email ?? null,
                'FederalTaxID' => $contact->tax_number ?? $contact->federal_tax_id ?? null,
                'Currency' => $contact->currency ?? null,
                'CreditLimit' => $contact->sap_credit_limit ?? $contact->credit_limit ?? 0,
                'Notes' => $contact->sap_notes ?? null,
                'CardForeignName' => $contact->sap_foreign_name ?? null,
            ];
            
            // Check if the business partner already exists in SAP
            $existingBp = $this->getBusinessPartner($bpData['CardCode']);
            
            if ($existingBp) {
                // Update existing business partner
                $result = $this->updateBusinessPartner($bpData['CardCode'], $bpData);
                if ($result) {
                    // Update successful
                    $contact->sap_last_sync = now();
                    $contact->sap_sync_status = 'success';
                    $contact->save();
                    return true;
                }
            } else {
                // Create new business partner
                $result = $this->createBusinessPartner($bpData);
                if ($result) {
                    // Creation successful
                    $contact->card_code = $bpData['CardCode']; // Update contact with SAP card code
                    $contact->sap_last_sync = now();
                    $contact->sap_sync_status = 'success';
                    $contact->save();
                    return true;
                }
            }
            
            // If we reached here, there was an error
            $contact->sap_sync_status = 'failed';
            $contact->save();
            return false;
        } catch (\Exception $e) {
            Log::error("Error pushing contact {$contact->id} to SAP: " . $e->getMessage());
            $contact->sap_sync_status = 'failed';
            $contact->sap_sync_message = $e->getMessage();
            $contact->save();
            return false;
        }
    }

    /**
     * تشخيص مشكلة عدم جلب جميع العملاء
     */
    public function diagnoseCustomerFetchingIssue()
    {
        $diagnostics = [];
        
        try {
            // 1. الحصول على العدد الإجمالي
            $countResult = $this->get('BusinessPartners/$count', ['$filter' => "CardType eq 'cCustomer'"], 60000);
            $diagnostics['total_count'] = intval($countResult);
            
            // 2. جلب عينة من العملاء في نقاط مختلفة
            $samplePoints = [0, 500, 1000, 1500, 2000, 2300, 2400, 2500, 3000, 3500];
            $diagnostics['samples'] = [];
            
            foreach ($samplePoints as $skip) {
                try {
                    $result = $this->getBusinessPartners("CardType eq 'cCustomer'", "CardCode,CardName", 10, $skip, 30000);
                    $diagnostics['samples'][$skip] = [
                        'found' => !empty($result['value']),
                        'count' => count($result['value'] ?? []),
                        'first_card' => $result['value'][0]['CardCode'] ?? null,
                        'last_card' => $result['value'][count($result['value'])-1]['CardCode'] ?? null
                    ];
                } catch (\Exception $e) {
                    $diagnostics['samples'][$skip] = [
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            // 3. التحقق من قيود Service Layer
            $diagnostics['service_layer_info'] = [
                'max_page_size' => $this->getMaxPageSize(),
                'timeout_setting' => ini_get('default_socket_timeout'),
                'memory_limit' => ini_get('memory_limit')
            ];
            
        } catch (\Exception $e) {
            $diagnostics['error'] = $e->getMessage();
        }
        
        return $diagnostics;
    }

    /**
     * الحصول على أقصى حجم صفحة مسموح به في Service Layer
     */
    private function getMaxPageSize()
    {
        try {
            // محاولة جلب عدد كبير لمعرفة الحد الأقصى
            $result = $this->getBusinessPartners("CardType eq 'cCustomer'", "CardCode", 10000, 0, 30000);
            return count($result['value'] ?? []);
        } catch (\Exception $e) {
            // إذا فشل، نعود بقيمة افتراضية
            return 1000;
        }
    }

    /**
     * Get business ID - handles both web and console contexts
     */
    protected function getBusinessId()
    {
        // For web requests with session
        if (request()->hasSession() && request()->session()->has('user.business_id')) {
            return request()->session()->get('user.business_id');
        }
        
        // For console commands or API calls, you can:
        // 1. Set a default business ID
        // 2. Pass it as a parameter
        // 3. Get it from authenticated user
        
        // Option 1: Default business ID (change as needed)
        return 1; // or config('sap.default_business_id')
        
        // Option 2: Get from authenticated user if available
        // if (auth()->check()) {
        //     return auth()->user()->business_id;
        // }
    }
    
    /**
     * Get user ID - handles both web and console contexts
     */
    protected function getUserId()
    {
        // For web requests with session
        if (request()->hasSession() && request()->session()->has('user.id')) {
            return request()->session()->get('user.id');
        }
        
        // For console commands or API calls
        // Option 1: Default user ID
        return 1; // or config('sap.default_user_id')
        
        // Option 2: Get from authenticated user
        // if (auth()->check()) {
        //     return auth()->user()->id;
        // }
    }




  public function createContactInSapAndGetCardCode($contact)
{
    try {
        // تحديد نوع البطاقة بدقة
        $cardType = $this->getCardTypeForSap($contact->type);
        
        // محاولات متعددة للإنشاء
        $attempts = [
            // المحاولة الأولى: بدون CardCode
        
        
            $this->createWithoutCardCode($contact, $cardType),
            
            // المحاولة الثانية: مع CardCode يدوي
            $this->createWithManualCardCode($contact, $cardType),
            
            // المحاولة الثالثة: مع بادئة محددة
            $this->createWithPrefixedCardCode($contact, $cardType)
        ];
        
        // تجربة كل محاولة
        foreach ($attempts as $attempt) {
            try {
                $response = $this->post('BusinessPartners', $attempt['data']);
                
                // استخراج CardCode من الاستجابة
                $cardCode = $this->extractCardCodeFromResponse($response);
                
                if ($cardCode) {
                    // تحديث جهة الاتصال المحلية
                    $contact->card_code = $cardCode;
                    $contact->save();
                    
                    return [
                        'CardCode' => $cardCode,
                        'CardName' => $contact->name
                    ];
                }
            } catch (\Exception $e) {
                // تسجيل الخطأ واستمرار المحاولات
                Log::warning("محاولة فاشلة: " . $e->getMessage(), [
                    'attempt_data' => $attempt['data']
                ]);
            }
        }
        
        throw new \Exception('فشل إنشاء جهة الاتصال في SAP بعد جميع المحاولات');
        
    } catch (\Exception $e) {
        Log::error('فشل إنشاء جهة الاتصال في SAP: ' . $e->getMessage(), [
            'contact_id' => $contact->id,
            'contact_name' => $contact->name
        ]);
        
        throw $e;
    }
}

private function createWithoutCardCode($contact, $cardType)
{
    return [
        'method' => 'بدون CardCode',
        'data' => [
            'CardName' => $this->sanitizeForSap($contact->name),
            'CardType' => $cardType,
            'Phone1' => $this->formatPhoneForSap($contact->mobile),
            'GroupCode' => $this->getDefaultGroupCode($contact->type)
           // 'Notes' => $this->generateCreationNote($contact)
        ]
    ];
}

private function createWithManualCardCode($contact, $cardType)
{
    $prefix = $cardType == 'cSupplier' ? 'SUP' : 'IBG';
    $randomNumber = str_pad(rand(5000, 10000), 5, '0', STR_PAD_LEFT);
    $cardCode = $prefix . $randomNumber;
    
    return [
        'method' => 'CardCode يدوي',
        'data' => [
            'CardCode' => $cardCode,
            'CardName' => $this->sanitizeForSap($contact->name),
            'CardType' => $cardType,
            'Phone1' => $this->formatPhoneForSap($contact->mobile),
             'GroupCode' => $this->getDefaultGroupCode($contact->type)
           // 'Notes' => $this->generateCreationNote($contact)
        ]
    ];
}
private function createWithPrefixedCardCode($contact, $cardType)
{
    // استخدام معرف جهة الاتصال مع بادئة محددة
    $prefix = $cardType == 'cSupplier' ? 'SUP' : 'IBG';
   $formattedId = str_pad($contact->id, 5, '0', STR_PAD_LEFT);
$cardCode = $prefix . $formattedId;
    
    return [
        'method' => 'CardCode مع البادئة',
        'data' => [
            'CardCode' => $cardCode,
            'CardName' => $this->sanitizeForSap($contact->name),
            'CardType' => $cardType,
            'Phone1' => $this->formatPhoneForSap($contact->mobile),
             'GroupCode' => $this->getDefaultGroupCode($contact->type)
           // 'Notes' => $this->generateCreationNote($contact)
        ]
    ];
}


private function getCardTypeForSap($type)
{
    switch($type) {
        case 'supplier':
            return 'cSupplier';
        case 'customer':
        case 'both':
        default:
            return 'cCustomer';
    }
}

private function addOptionalFieldsToBusinessPartner($contact, &$bpData)
{
    $optionalFields = [
        'mobile' => 'Phone1',
        'alternate_number' => 'Phone2',
        'email' => 'EmailAddress',
        'address_line_1' => 'Address',
        'zip_code' => 'ZipCode',
        'city' => 'City',
        'state' => 'County',
        'country' => 'Country',
        'tax_number' => 'FederalTaxID'
    ];

    foreach ($optionalFields as $localField => $sapField) {
        if (!empty($contact->$localField)) {
            $bpData[$sapField] = $this->sanitizeForSap($contact->$localField);
        }
    }

    // إضافة GroupCode
    $bpData['GroupCode'] = $this->getDefaultGroupCode($contact->type);
    
    // إضافة ملاحظات
    $bpData['Notes'] = 'تم الإنشاء من نظام نقاط البيع في ' . now()->format('Y-m-d H:i:s');
}

private function extractCardCodeFromResponse($response)
{
    // استخراج CardCode من استجابة SAP
    if (isset($response['CardCode'])) {
        return $response['CardCode'];
    }
    
    // محاولة البحث باسم العميل إذا فشل الإرسال
    return $this->findCardCodeByName($response['CardName'] ?? null);
}


private function findCardCodeByName($contactName)
{
    if (empty($contactName)) return null;
    
    try {
        // تنظيف الاسم من الأحرف الخاصة
        $searchName = str_replace("'", "''", $contactName);
        
        // البحث بعدة طرق
        $searchFilters = [
            "CardName eq '" . $searchName . "'",
            "startswith(CardName, '" . substr($searchName, 0, 20) . "')",
            "contains(CardName, '" . $searchName . "')"
        ];
        
        foreach ($searchFilters as $filter) {
            Log::info('Searching SAP with filter', ['filter' => $filter]);
            
            $searchResult = $this->get('BusinessPartners', [
                '$filter' => $filter,
                '$select' => 'CardCode,CardName',
                '$orderby' => 'UpdateDate desc',
                '$top' => 5
            ], 30000);
            
            if (isset($searchResult['value']) && !empty($searchResult['value'])) {
                // البحث عن تطابق دقيق أولاً
                foreach ($searchResult['value'] as $item) {
                    if (trim(strtolower($item['CardName'])) === trim(strtolower($contactName))) {
                        Log::info('Exact match found', [
                            'CardCode' => $item['CardCode'],
                            'CardName' => $item['CardName']
                        ]);
                        return $item['CardCode'];
                    }
                }
                
                // إذا لم نجد تطابق دقيق، نأخذ الأول
                Log::info('Using first match', [
                    'CardCode' => $searchResult['value'][0]['CardCode'],
                    'CardName' => $searchResult['value'][0]['CardName']
                ]);
                return $searchResult['value'][0]['CardCode'];
            }
            
            // تأخير قصير بين المحاولات
            usleep(500000); // 0.5 ثانية
        }
        
        return null;
    } catch (\Exception $e) {
        Log::error('Failed to search CardCode by name: ' . $e->getMessage());
        return null;
    }
}

/**
 * دوال مساعدة
 */
private function sanitizeForSap($value)
{
    // إزالة الأحرف الخاصة التي قد تسبب مشاكل في SAP
    return preg_replace('/[^\p{L}\p{N}\s\-.,]/u', '', $value);
}

private function formatPhoneForSap($phone)
{
    // تنسيق رقم الهاتف حسب متطلبات SAP
    return preg_replace('/[^0-9+]/', '', $phone);
}



public function createContactWithSpecificCardCode($contact, $cardCode)
{
    try {
        // التحقق من أن CardCode غير موجود في SAP
        if ($this->checkCardCodeExists($cardCode)) {
            throw new \Exception("CardCode {$cardCode} already exists in SAP");
        }
        
        $cardType = $this->getCardTypeForSap($contact->type);
        
        // إعداد بيانات Business Partner الأساسية
        $bpData = [
            'CardCode' => $cardCode,
            'CardName' => $this->sanitizeForSap($contact->name ?: 'Contact ' . $cardCode),
            'CardType' => $cardType,
            'GroupCode' => intval($this->getDefaultGroupCode($contact->type))
        ];
        
        // إضافة رقم الهاتف
        if (!empty($contact->mobile)) {
            $bpData['Phone1'] = $this->formatPhoneForSap($contact->mobile);
        } elseif (!empty($contact->landline)) {
            $bpData['Phone1'] = $this->formatPhoneForSap($contact->landline);
        }
        
        // إضافة الحقول الاختيارية
        $this->addOptionalFieldsToBusinessPartner($contact, $bpData);
        
        // محاولات متعددة للعثور على Series الصحيح
        $seriesAttempts = $this->getSeriesAttempts($contact->type);
        
        foreach ($seriesAttempts as $attempt) {
            try {
                $attemptData = $bpData;
                if ($attempt['series'] !== null) {
                    $attemptData['Series'] = $attempt['series'];
                }
                
                Log::info("Attempting to create BP: {$attempt['description']}", [
                    'CardCode' => $cardCode,
                    'Series' => $attempt['series'] ?? 'not set',
                    'CardType' => $cardType
                ]);
                
                $response = $this->post('BusinessPartners', $attemptData);
                
                Log::info('BP created successfully', [
                    'CardCode' => $cardCode,
                    'method' => $attempt['description'],
                    'response' => $response
                ]);
                
                return [
                    'CardCode' => $cardCode,
                    'CardName' => $contact->name,
                    'success' => true
                ];
                
            } catch (\Exception $e) {
                Log::warning("Failed with {$attempt['description']}: " . $e->getMessage());
                
                // إذا كان الخطأ ليس عن Series، أعد طرح الاستثناء
                if (strpos($e->getMessage(), 'Series') === false && 
                    strpos($e->getMessage(), 'numbering series') === false) {
                    throw $e;
                }
            }
        }
        
        throw new \Exception('All attempts to create BP failed. Please check SAP numbering series configuration.');
        
    } catch (\Exception $e) {
        Log::error('Failed to create BP with specific CardCode', [
            'CardCode' => $cardCode,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        throw $e;
    }
}

private function getSeriesAttempts($contactType)
{
    $attempts = [];
    
    // جرب الحصول على Series ID من SAP
    $detectedSeriesId = $this->detectSeriesId($contactType);
    if ($detectedSeriesId !== null) {
        $attempts[] = [
            'series' => $detectedSeriesId,
            'description' => "Detected Series ID: {$detectedSeriesId}"
        ];
    }
    
    // القيم الشائعة
    $commonValues = [1, 0, 2, 3, 4, 5];
    foreach ($commonValues as $value) {
        $attempts[] = [
            'series' => $value,
            'description' => "Series = {$value}"
        ];
    }
    
    // بدون Series
    $attempts[] = [
        'series' => null,
        'description' => 'Without Series'
    ];
    
    return $attempts;
}

/**
 * محاولة اكتشاف Series ID
 */
private function detectSeriesId($contactType)
{
    try {
        // البحث عن Business Partner موجود بنفس البادئة
        $prefix = ($contactType == 'supplier') ? 'LV' : 'IBG';
        
        $response = $this->get('BusinessPartners', [
            '$filter' => "startswith(CardCode, '{$prefix}')",
            '$select' => 'CardCode,Series',
            '$top' => 1,
            '$orderby' => 'CreateDate desc'
        ]);
        
        if (isset($response['value']) && !empty($response['value']) && isset($response['value'][0]['Series'])) {
            $seriesId = intval($response['value'][0]['Series']);
            Log::info("Detected Series ID from existing BP", [
                'prefix' => $prefix,
                'seriesId' => $seriesId
            ]);
            return $seriesId;
        }
    } catch (\Exception $e) {
        Log::warning('Could not detect Series ID: ' . $e->getMessage());
    }
    
    return null;
}

/**
 * توليد CardCode حسب النوع مع البادئة الصحيحة
 */
public function generateCardCodeForType($type, $businessId)
{
    $prefix = ($type == 'supplier') ? 'LV' : 'IBG';
    
    // الحصول على آخر رقم مستخدم
    $lastNumber = $this->getLastUsedNumber($prefix);
    
    if ($lastNumber === null) {
        // إذا لم نجد، نبدأ من رقم افتراضي
        $lastNumber = ($type == 'supplier') ? 99 : 5076;
    }
    
    $nextNumber = $lastNumber + 1;
    
   $formattedNumber = str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
return $prefix . $formattedNumber;
}

/**
 * تحديث getDefaultGroupCode للتعامل مع الأنواع المختلفة
 */
private function getDefaultGroupCode($type)
{
    // تأكد من إرجاع رقم وليس نص
    if ($type == 'supplier') {
        return intval(config('sap.default_supplier_group', 101));
    }
    
    return intval(config('sap.default_customer_group', 100));
}

/**
 * طريقة تشخيصية لاختبار إنشاء BP
 */
public function testBusinessPartnerCreation($type = 'customer')
{
    try {
        // إنشاء بيانات اختبارية
        $testCardCode = ($type == 'supplier' ? 'LV' : 'IBG') . 'TEST' . rand(1000, 9999);
        
        $testData = [
            'CardCode' => $testCardCode,
            'CardName' => 'Test ' . ucfirst($type) . ' ' . date('Y-m-d H:i:s'),
            'CardType' => $type == 'supplier' ? 'cSupplier' : 'cCustomer',
            'GroupCode' => $type == 'supplier' ? 101 : 100
        ];
        
        Log::info('Testing BP creation', $testData);
        
        // محاولة إنشاء بدون Series
        try {
            $response = $this->post('BusinessPartners', $testData);
            Log::info('Test successful without Series', ['response' => $response]);
            
            // حذف BP الاختباري
            $this->delete("BusinessPartners('{$testCardCode}')");
            
            return ['success' => true, 'method' => 'without Series', 'response' => $response];
        } catch (\Exception $e) {
            Log::warning('Failed without Series: ' . $e->getMessage());
        }
        
        // محاولة مع Series مختلفة
        for ($i = 0; $i <= 5; $i++) {
            try {
                $testData['Series'] = $i;
                $response = $this->post('BusinessPartners', $testData);
                Log::info("Test successful with Series = {$i}", ['response' => $response]);
                
                // حذف BP الاختباري
                $this->delete("BusinessPartners('{$testCardCode}')");
                
                return ['success' => true, 'method' => "Series = {$i}", 'response' => $response];
            } catch (\Exception $e) {
                Log::warning("Failed with Series = {$i}: " . $e->getMessage());
            }
        }
        
        return ['success' => false, 'message' => 'All test attempts failed'];
        
    } catch (\Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}


/**
 * Helper method to get correct Series ID
 */
private function getCorrectSeriesId($cardType)
{
    // محاولة الحصول على Series ID من ذاكرة التخزين المؤقت
    $cacheKey = 'sap_series_id_' . $cardType;
    
    if (cache()->has($cacheKey)) {
        return cache($cacheKey);
    }
    
    // محاولة الحصول على Series ID
    $seriesId = $this->getBusinessPartnerSeriesId($cardType);
    
    if ($seriesId !== null) {
        // حفظ في الذاكرة المؤقتة لمدة ساعة
        cache([$cacheKey => $seriesId], 3600);
    }
    
    return $seriesId;
}

/**
 * Test method to debug series issues
 */
public function testSeriesConfiguration()
{
    $results = [
        'series_id' => $this->getBusinessPartnerSeriesId(),
        'series_info' => $this->getSeriesInfo(),
        'all_series' => $this->getAllNumberingSeries()
    ];
    
    Log::info('Series Configuration Test', $results);
    
    return $results;
}

/**
 * Get last used number for a prefix
 */
public function getLastUsedNumber($prefix)
{
    try {
        // البحث عن آخر CardCode بنفس البادئة
        $filter = "startswith(CardCode, '{$prefix}')";
        $result = $this->get('BusinessPartners', [
            '$filter' => $filter,
            '$select' => 'CardCode',
            '$orderby' => 'CardCode desc',
            '$top' => 1
        ], 30000);
        
        if (isset($result['value']) && !empty($result['value'])) {
            $lastCardCode = $result['value'][0]['CardCode'];
            // استخراج الرقم من CardCode
            $number = intval(substr($lastCardCode, strlen($prefix)));
            return $number;
        }
        
        return null;
    } catch (\Exception $e) {
        Log::error('Failed to get last used number from SAP: ' . $e->getMessage());
        return null;
    }
}

/**
 * Check if CardCode exists (محسّنة)
 */
public function checkCardCodeExists($cardCode)
{
    try {
        // محاولة جلب البطاقة مباشرة
        $result = $this->get("BusinessPartners('{$cardCode}')", [], 10000);
        return !empty($result);
    } catch (\Exception $e) {
        // إذا كان الخطأ 404، فالكود غير موجود
        if (strpos($e->getMessage(), '404') !== false || 
            strpos($e->getMessage(), 'not found') !== false) {
            return false;
        }
        
        // لأي خطأ آخر، نفترض أن الكود موجود للأمان
        Log::warning('Error checking CardCode existence', [
            'CardCode' => $cardCode,
            'error' => $e->getMessage()
        ]);
        return true;
    }
}

public function getBusinessPartnerSeriesId($cardType = 'cCustomer')
{
    try {
        // الحصول على جميع السلاسل المتاحة
        $response = $this->get('SeriesService_GetDocumentSeries', [
            'DocumentTypeParams' => [
                'Document' => '2' // 2 = Business Partners
            ]
        ]);
        
        if (isset($response['value'])) {
            foreach ($response['value'] as $series) {
                // البحث عن السلسلة الافتراضية أو التي تبدأ بـ IBG
                if (isset($series['IsDefault']) && $series['IsDefault'] == 'tYES') {
                    return $series['Series'];
                }
                if (isset($series['Name']) && strpos($series['Name'], 'IBG') !== false) {
                    return $series['Series'];
                }
            }
        }
        
        return null;
    } catch (\Exception $e) {
        Log::error('Failed to get series ID: ' . $e->getMessage());
        return null;
    }
}

/**
 * Alternative: Query the Series table directly
 */
public function getSeriesInfo()
{
    try {
        // محاولة الحصول على معلومات السلاسل
        $response = $this->get('view.svc/B1_SeriesView', [
            '$filter' => "ObjectCode eq 2", // 2 = Business Partners
            '$select' => 'Series,SeriesName,IsDefault'
        ]);
        
        Log::info('Series info from SAP', ['response' => $response]);
        
        return $response;
    } catch (\Exception $e) {
        Log::error('Failed to get series info: ' . $e->getMessage());
        return null;
    }
}

public function getAllNumberingSeries()
{
    try {
        // محاولة مختلفة للحصول على السلاسل
        $endpoints = [
            'NumerationSeries',
            'SeriesService_GetSeries',
            'SQLQueries("SELECT * FROM NNM1 WHERE ObjectCode = 2")/List'
        ];
        
        foreach ($endpoints as $endpoint) {
            try {
                $response = $this->get($endpoint);
                Log::info("Series from endpoint {$endpoint}", ['response' => $response]);
                return $response;
            } catch (\Exception $e) {
                Log::warning("Failed to get series from {$endpoint}: " . $e->getMessage());
                continue;
            }
        }
        
        return null;
    } catch (\Exception $e) {
        Log::error('Failed to get any series info: ' . $e->getMessage());
        return null;
    }
}


public function getIBGSeriesNumber()
{
    try {
        // محاولة 1: البحث في Business Partners الموجودين
        $response = $this->get('BusinessPartners', [
            '$filter' => "startswith(CardCode, 'IBG')",
            '$select' => 'CardCode,Series',
            '$top' => 1,
            '$orderby' => 'CreateDate desc'
        ]);
        
        if (isset($response['value']) && !empty($response['value'])) {
            $seriesNumber = $response['value'][0]['Series'] ?? null;
            if ($seriesNumber !== null) {
                Log::info('Found Series number from existing BP', ['Series' => $seriesNumber]);
                return intval($seriesNumber);
            }
        }
        
        // محاولة 2: query مباشر للسلاسل
        try {
            $sqlQuery = "SELECT Series, SeriesName FROM NNM1 WHERE SeriesName LIKE 'IBG%' AND ObjectCode = 2";
            $queryResponse = $this->get('SQLQueries', [
                'sqlQuery' => $sqlQuery
            ]);
            
            if (isset($queryResponse['value']) && !empty($queryResponse['value'])) {
                return intval($queryResponse['value'][0]['Series']);
            }
        } catch (\Exception $e) {
            Log::warning('SQL query for series failed: ' . $e->getMessage());
        }
        
        // القيم الافتراضية الشائعة
        return 1; // غالباً ما تكون 1 للسلسلة الأولى
        
    } catch (\Exception $e) {
        Log::error('Failed to get IBG series number: ' . $e->getMessage());
        return 1; // قيمة افتراضية
    }
}

/**
 * طريقة محدثة لإنشاء Contact مع CardCode محدد
 */
public function createContactWithSpecificCardCodeFixed($contact, $cardCode)
{
    try {
        // التحقق من أن CardCode غير موجود في SAP
        if ($this->checkCardCodeExists($cardCode)) {
            throw new \Exception("CardCode {$cardCode} already exists in SAP");
        }
        
        $cardType = $this->getCardTypeForSap($contact->type);
        
        // الحصول على رقم السلسلة الصحيح
        $seriesNumber = $this->getIBGSeriesNumber();
        
        // إعداد بيانات Business Partner
        $bpData = [
            'CardCode' => $cardCode,
            'CardName' => $this->sanitizeForSap($contact->name ?: 'Contact ' . $cardCode),
            'CardType' => $cardType,
            'Series' => $seriesNumber, // استخدام الرقم الصحيح
            'GroupCode' => intval($this->getDefaultGroupCode($contact->type)) // تأكد أنه رقم أيضاً
        ];
        
        // إضافة رقم الهاتف
        if (!empty($contact->mobile)) {
            $bpData['Phone1'] = $this->formatPhoneForSap($contact->mobile);
        } else {
            $bpData['Phone1'] = 'N/A';
        }
        
        // إضافة الحقول الاختيارية
        $this->addOptionalFieldsToBusinessPartner($contact, $bpData);
        
        Log::info('Creating BP with fixed Series number', [
            'CardCode' => $cardCode,
            'Series' => $seriesNumber,
            'data' => $bpData
        ]);
        
        $response = $this->post('BusinessPartners', $bpData);
        
        Log::info('BP created successfully', [
            'CardCode' => $cardCode,
            'response' => $response
        ]);
        
        return [
            'CardCode' => $cardCode,
            'CardName' => $contact->name,
            'success' => true
        ];
        
    } catch (\Exception $e) {
        Log::error('Failed to create BP', [
            'CardCode' => $cardCode,
            'error' => $e->getMessage()
        ]);
        
        // إذا فشل بسبب Series، جرب بدونه
        if (strpos($e->getMessage(), 'Series') !== false) {
            unset($bpData['Series']);
            
            try {
                Log::info('Retrying without Series', ['data' => $bpData]);
                $response = $this->post('BusinessPartners', $bpData);
                
                return [
                    'CardCode' => $cardCode,
                    'CardName' => $contact->name,
                    'success' => true
                ];
            } catch (\Exception $e2) {
                throw $e2;
            }
        }
        
        throw $e;
    }
}

    /**
     * Sync customers only from SAP B1 to local database
     */
    public function syncCustomersOnly($fullSync = false)
    {
        $stats = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0
        ];

        try {
            // Determine last sync time for incremental updates
            $lastSync = null;
            if (!$fullSync) {
                $latestSync = Contact::where('type', 'customer')
                    ->whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first();
                
                if ($latestSync) {
                    $lastSync = $latestSync->sap_last_sync->format('Y-m-d\TH:i:s');
                    Log::info("Performing incremental customer sync since: {$lastSync}");
                } else {
                    Log::info("No previous customer sync found. Performing full sync.");
                    $fullSync = true;
                }
            }

            // Build filter for customers
            $filter = "CardType eq 'cCustomer'";
            if (!$fullSync && $lastSync) {
                $filter = "UpdateDate ge '{$lastSync}' and " . $filter;
            }

            // Fields to select
            $select = "CardCode,CardName,CardType,GroupCode,Address,ZipCode,City,County,Country,Phone1,Phone2,EmailAddress,Currency,FederalTaxID,CreditLimit,PriceListNum,SalesPersonCode,Notes,CurrentAccountBalance,UpdateDate,CardForeignName";
            
            $this->syncContacts('customer', $filter, $select, $stats);

            return $stats;
        } catch (\Exception $e) {
            Log::error("SAP Sync Customers Only Exception: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Sync suppliers only from SAP B1 to local database
     */
    public function syncSuppliersOnly($fullSync = false)
    {
        $stats = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0
        ];

        try {
            // Determine last sync time for incremental updates
            $lastSync = null;
            if (!$fullSync) {
                $latestSync = Contact::where('type', 'supplier')
                    ->whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first();
                
                if ($latestSync) {
                    $lastSync = $latestSync->sap_last_sync->format('Y-m-d\TH:i:s');
                    Log::info("Performing incremental supplier sync since: {$lastSync}");
                } else {
                    Log::info("No previous supplier sync found. Performing full sync.");
                    $fullSync = true;
                }
            }

            // Build filter for suppliers
            $filter = "CardType eq 'cSupplier'";
            if (!$fullSync && $lastSync) {
                $filter = "UpdateDate ge '{$lastSync}' and " . $filter;
            }

            // Fields to select
            $select = "CardCode,CardName,CardType,GroupCode,Address,ZipCode,City,County,Country,Phone1,Phone2,EmailAddress,Currency,FederalTaxID,CreditLimit,PriceListNum,SalesPersonCode,Notes,CurrentAccountBalance,UpdateDate,CardForeignName";
            
            $this->syncContacts('supplier', $filter, $select, $stats);

            return $stats;
        } catch (\Exception $e) {
            Log::error("SAP Sync Suppliers Only Exception: " . $e->getMessage());
            throw $e;
        }
    }
}