<?php

namespace App\Services;

use App\Transaction;
use App\TransactionSellLine;
use App\Contact;
use App\Services\SapB1Service;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

/**
 * خدمة مزامنة الفواتير مع SAP B1
 * 
 * تم إزالة NumPerMsr نهائياً من جميع الوحدات:
 * - وحدات السيت (UoMEntry: 21) يتم إرسالها بدون UoMEntry وبدون NumPerMsr
 * - الوحدات الأخرى تُرسل مع UoMEntry فقط (بدون NumPerMsr)
 * 
 * هذا يحل مشكلة: "Specify Items per Unit greater than zero [INV1.NumPerMsr]"
 * 
 * @version 2.6
 * @updated 2025-01-27
 */
class SapInvoicesService
{
    protected $sapService;

    public function __construct(SapB1Service $sapService)
    {
        $this->sapService = $sapService;
    }

    /**
     * Push a transaction to SAP as an invoice or quotation
     *
     * @param Transaction $transaction
     * @return array|null
     */
    public function pushTransactionToSap(Transaction $transaction)
    {
        try {
            // Determine if this is a quotation, draft, or invoice
            $is_quotation = ($transaction->is_quotation == 1 || $transaction->sub_status == 'quotation');
            $is_draft = ($transaction->status == 'draft' && !$is_quotation);
            $is_final = ($transaction->status == 'final');
            
            // Prepare the document data with absolute minimum required fields
            $documentData = $this->prepareMinimalDocumentData($transaction);
            
            // Set the endpoint based on document type
            if ($is_quotation) {
                $endpoint = 'Quotations';
            } elseif ($is_draft) {
                // الدرافت يُرسل كـ Sales Order مع تحديد أنه درافت داخل SAP
                $endpoint = 'Orders';
                
                Log::info("Sending draft as Sales Order with draft status", [
                    'transaction_id' => $transaction->id,
                    'endpoint' => 'Orders',
                    'document_type' => 'draft',
                    'note' => 'Draft will be sent as sales order with draft indicators'
                ]);
            } else {
                $endpoint = 'Orders';
            }
            
            // Check if the transaction already has a SAP document ID (for updates)
            $documentId = $transaction->sap_document_id ?? null;
            
            // Special handling for quotation or draft to invoice conversion
            $is_conversion = !empty($transaction->sap_document_id) && 
                            !empty($transaction->sap_document_type) && 
                            ($transaction->sap_document_type == 'quotation' || $transaction->sap_document_type == 'draft_order') && 
                            $is_final;
            
            // If converting from quotation or draft to final sale, create new document instead of updating
            if ($is_conversion) {
                Log::info("Converting {$transaction->sap_document_type} to final sale - creating new SAP document", [
                    'transaction_id' => $transaction->id,
                    'old_sap_document_id' => $documentId,
                    'old_document_type' => $transaction->sap_document_type,
                    'new_document_type' => 'final order'
                ]);
                
                // Reset document ID to force creation of new document
                $documentId = null;
                $endpoint = 'Orders'; // Final sales go to Orders endpoint
            }
            

            if (!isset($documentData['DocDueDate'])) {
                // Set DocDueDate to today's date as requested
                $documentData['DocDueDate'] = date('Y-m-d');
                
                // Optionally, you might also want to set DocDate if not already set
                if (!isset($documentData['DocDate'])) {
                    $documentData['DocDate'] = date('Y-m-d');
                }
            }
            
            // Verify that NumPerMsr is not sent for any line
            $hasNumPerMsr = false;
            foreach ($documentData['DocumentLines'] as $line) {
                if (isset($line['NumPerMsr'])) {
                    $hasNumPerMsr = true;
                    break;
                }
            }
            
            Log::info("Final document verification before sending to SAP", [
                'total_lines' => count($documentData['DocumentLines']),
                'has_NumPerMsr' => $hasNumPerMsr,
                'confirmation' => $hasNumPerMsr ? 'ERROR: NumPerMsr found in document!' : 'CORRECT: No NumPerMsr in any line'
            ]);

            // Create or update the document in SAP
            if (empty($documentId)) {
                // Create new document
                Log::info("Attempting to create SAP document with minimal data", [
                    'transaction_id' => $transaction->id,
                    'document_type' => $is_draft ? 'draft (as sales order)' : ($is_quotation ? 'quotation' : 'final order'),
                    'endpoint' => $endpoint,
                    'is_draft' => $is_draft,
                    'is_quotation' => $is_quotation,
                    'draft_indicators' => $is_draft ? ['DocStatus' => 'bost_Open', 'Comments' => 'contains DRAFT DOCUMENT'] : 'None',
                    'data_summary' => [
                        'CardCode' => $documentData['CardCode'] ?? 'Not set',
                        'DocumentLines_count' => count($documentData['DocumentLines'] ?? []),
                        'DiscountPercent' => $documentData['DiscountPercent'] ?? 'None'
                    ]
                ]);
                
                // إضافة تسجيل خاص للوحدات بدون UoMEntry (وحدات السيت)
                foreach ($documentData['DocumentLines'] as $line) {
                    if (!isset($line['UoMEntry'])) {
                        Log::info('Document line without UoMEntry (Set unit)', [
                            'ItemCode' => $line['ItemCode'],
                            'Quantity' => $line['Quantity'],
                            'note' => 'Set unit sent without UoMEntry as requested'
                        ]);
                    } else {
                        Log::info('Document line with UoMEntry (non-Set unit)', [
                            'ItemCode' => $line['ItemCode'],
                            'UoMEntry' => $line['UoMEntry'],
                            'Quantity' => $line['Quantity'],
                            'note' => 'NumPerMsr is not sent for any unit type'
                        ]);
                    }
                }
                
                $response = $this->sapService->post($endpoint, $documentData);
                
                if (!empty($response) && isset($response['DocNum'])) {
                    // Update local transaction with SAP document ID and invoice number
                    DB::table('transactions')
                        ->where('id', $transaction->id)
                        ->update([
                            'sap_document_id' => $response['DocNum'],
                            'sap_document_type' => $is_quotation ? 'quotation' : ($is_draft ? 'draft_order' : 'order'),
                            'sap_sync_status' => 'success',
                            'sap_sync_date' => now(),
                            'invoice_no' => 'SAP-' . $response['DocNum']
                        ]);
                    
                    $documentTypeText = $is_draft ? "draft (sent as sales order)" : ($is_quotation ? "quotation" : "order");
                    Log::info("Successfully created SAP {$documentTypeText} #{$response['DocNum']} for transaction #{$transaction->id}", [
                        'transaction_id' => $transaction->id,
                        'sap_document_id' => $response['DocNum'],
                        'document_type' => $documentTypeText,
                        'endpoint' => $endpoint,
                        'is_draft' => $is_draft,
                        'is_quotation' => $is_quotation,
                        'database_type' => $is_draft ? 'draft_order' : ($is_quotation ? 'quotation' : 'order')
                    ]);
                    
                    return [
                        'success' => true,
                        'sap_document_id' => $response['DocNum'],
                        'invoice_no' => 'SAP-' . $response['DocNum'],
                        'message' => "Successfully created in SAP"
                    ];
                } else {
                    Log::error("Failed to create SAP document. Response: " . json_encode($response));
                }
            } else {
                // Update existing document - include discount and comments along with DocumentLines
                $updateData = [
                    'DocumentLines' => $documentData['DocumentLines']
                ];
                
                // إضافة معلومات الخصم في التحديث
                if (isset($documentData['DiscountPercent'])) {
                    $updateData['DiscountPercent'] = $documentData['DiscountPercent'];
                }
                
                // إضافة الملاحظات في التحديث
                if (isset($documentData['Comments'])) {
                    $updateData['Comments'] = $documentData['Comments'];
                }
                
                Log::info("Attempting to update SAP document with minimal data: " . json_encode($updateData));
                $response = $this->sapService->patch("$endpoint($documentId)", $updateData);
                
                if (!empty($response)) {
                    // Use direct update instead of model save to prevent potential issues
                    DB::table('transactions')
                        ->where('id', $transaction->id)
                        ->update([
                            'sap_sync_status' => 'success',
                            'sap_sync_date' => now()
                        ]);
                    
                    $documentTypeText = $is_draft ? "draft (sent as sales order)" : ($is_quotation ? "quotation" : "order");
                    Log::info("Successfully updated SAP {$documentTypeText} #$documentId for transaction #{$transaction->id}", [
                        'transaction_id' => $transaction->id,
                        'sap_document_id' => $documentId,
                        'document_type' => $documentTypeText,
                        'endpoint' => $endpoint,
                        'is_draft' => $is_draft,
                        'database_type' => $is_draft ? 'draft_order' : ($is_quotation ? 'quotation' : 'order')
                    ]);
                    
                    return [
                        'success' => true,
                        'sap_document_id' => $documentId,
                        'message' => "Successfully updated in SAP"
                    ];
                } else {
                    // Update failed, get error message from logs or response
                    $errorMessage = '';
                    $responseEmpty = empty($response);
                    
                    if (!$responseEmpty && isset($response['error'])) {
                        if (isset($response['error']['message']['value'])) {
                            $errorMessage = $response['error']['message']['value'];
                        }
                    }
                    
                    Log::warning("SAP document update failed, attempting to create new document", [
                        'transaction_id' => $transaction->id,
                        'document_id' => $documentId,
                        'response_empty' => $responseEmpty,
                        'error_message' => $errorMessage,
                        'endpoint' => $endpoint
                    ]);
                    
                    // Always try to create new document when update fails
                    // Either because response is empty/null or contains "does not exist" error
                    if ($responseEmpty || 
                        strpos($errorMessage, 'does not exist') !== false || 
                        strpos($errorMessage, 'Entity with value') !== false) {
                        
                        Log::info("Creating new SAP document instead of updating non-existent one", [
                            'transaction_id' => $transaction->id,
                            'old_document_id' => $documentId,
                            'reason' => $responseEmpty ? 'Empty response' : 'Document does not exist'
                        ]);
                        
                        // Try to create new document instead
                        $createResponse = $this->sapService->post($endpoint, $documentData);
                        
                        if (!empty($createResponse) && isset($createResponse['DocNum'])) {
                            // Update local transaction with new SAP document ID and invoice number
                            DB::table('transactions')
                                ->where('id', $transaction->id)
                                ->update([
                                    'sap_document_id' => $createResponse['DocNum'],
                                    'sap_document_type' => $is_quotation ? 'quotation' : ($is_draft ? 'draft_order' : 'order'),
                                    'sap_sync_status' => 'success',
                                    'sap_sync_date' => now(),
                                    'invoice_no' => 'SAP-' . $createResponse['DocNum']
                                ]);
                            
                            Log::info("Successfully created new SAP document after update failure", [
                                'transaction_id' => $transaction->id,
                                'old_document_id' => $documentId,
                                'new_document_id' => $createResponse['DocNum'],
                                'invoice_no' => 'SAP-' . $createResponse['DocNum']
                            ]);
                            
                            return [
                                'success' => true,
                                'sap_document_id' => $createResponse['DocNum'],
                                'invoice_no' => 'SAP-' . $createResponse['DocNum'],
                                'message' => "Successfully created new document in SAP after update failure"
                            ];
                        } else {
                            Log::error("Failed to create new SAP document after update failure", [
                                'transaction_id' => $transaction->id,
                                'create_response' => json_encode($createResponse)
                            ]);
                        }
                    } else {
                        Log::error("Failed to update SAP document with unhandled error", [
                            'transaction_id' => $transaction->id,
                            'response' => json_encode($response),
                            'error_message' => $errorMessage
                        ]);
                    }
                }
            }
            
            // If we reach here, there was an error
            // Check for specific error types in the response
            $errorMessage = "Failed to push to SAP";
            
            if (!empty($response) && isset($response['error'])) {
                if (isset($response['error']['message']['value'])) {
                    $errorMessage = $response['error']['message']['value'];
                    
                    // Check for currency exchange rate error
                    if (strpos($errorMessage, 'Update the exchange rate') !== false) {
                        // Try an even more minimal approach - just CardCode and DocumentLines
                        Log::warning("Exchange rate error detected, attempting super minimal document");
                        
                        $superMinimalData = [
                            'CardCode' => $documentData['CardCode'],
                            'DocumentLines' => $documentData['DocumentLines']
                        ];
                        
                        // إضافة المعلومات الأساسية للخصم والملاحظات في النهج المبسط
                        // if (isset($documentData['DocTotal'])) {
                        //     $superMinimalData['DocTotal'] = $documentData['DocTotal'];
                        // }
                        if (isset($documentData['DiscountPercent'])) {
                            $superMinimalData['DiscountPercent'] = $documentData['DiscountPercent'];
                        }
                        if (isset($documentData['Comments'])) {
                            $superMinimalData['Comments'] = $documentData['Comments'];
                        }
                        
                        Log::info("Attempting with super minimal data: " . json_encode($superMinimalData));
                        $minimalResponse = $this->sapService->post($endpoint, $superMinimalData);
                        
                        if (!empty($minimalResponse) && isset($minimalResponse['DocNum'])) {
                            // Update local transaction with SAP document ID and invoice number
                            DB::table('transactions')
                                ->where('id', $transaction->id)
                                ->update([
                                    'sap_document_id' => $minimalResponse['DocNum'],
                                    'sap_document_type' => $is_quotation ? 'quotation' : ($is_draft ? 'draft_order' : 'order'),
                                    'sap_sync_status' => 'success',
                                    'sap_sync_date' => now(),
                                    'invoice_no' => 'SAP-' . $minimalResponse['DocNum']
                                ]);
                            
                            Log::info("Successfully created SAP document with super minimal data, DocNum: {$minimalResponse['DocNum']}");
                            
                            return [
                                'success' => true,
                                'sap_document_id' => $minimalResponse['DocNum'],
                                'invoice_no' => 'SAP-' . $minimalResponse['DocNum'],
                                'message' => "Successfully created in SAP with minimal data"
                            ];
                        } else {
                            Log::error("Super minimal approach also failed. Response: " . json_encode($minimalResponse));
                        }
                    }
                    
                    // معالجة خاصة لخطأ NumPerMsr (لا يجب أن يحدث لأن NumPerMsr لا يُرسل)
                    if (strpos($errorMessage, 'NumPerMsr') !== false || strpos($errorMessage, 'Items per Unit') !== false) {
                        Log::error("NumPerMsr error detected for transaction #{$transaction->id}. This should not happen as NumPerMsr is not sent.", [
                            'transaction_id' => $transaction->id,
                            'error_message' => $errorMessage,
                            'document_lines' => $documentData['DocumentLines'] ?? 'Not available',
                            'suggestion' => 'NumPerMsr is not sent for any unit type. This error may indicate an issue with SAP configuration or data'
                        ]);
                    }
                    
                    // Log detailed error
                    Log::error("SAP Error for transaction #{$transaction->id}: " . $errorMessage . ' Full response: ' . json_encode($response));
                }
            }
            
            DB::table('transactions')
                ->where('id', $transaction->id)
                ->update(['sap_sync_status' => 'failed']);
            
            Log::error("Failed to push transaction #{$transaction->id} to SAP: " . json_encode($response ?? []));
            
            return [
                'success' => false,
                'message' => $errorMessage
            ];
            
        } catch (\Exception $e) {
            Log::error("Exception pushing transaction #{$transaction->id} to SAP: " . $e->getMessage());
            
            // Update transaction with error status using query builder
            if (!empty($transaction->id)) {
                DB::table('transactions')
                    ->where('id', $transaction->id)
                    ->update(['sap_sync_status' => 'failed']);
            }
            
            return [
                'success' => false,
                'message' => "Error: " . $e->getMessage()
            ];
        }
    }

    /**
     * Prepare minimal document data for SAP B1
     *
     * @param Transaction $transaction
     * @return array
     */
/**
 * Prepare minimal document data for SAP B1
 *
 * @param Transaction $transaction
 * @return array
 */
protected function prepareMinimalDocumentData(Transaction $transaction) {
    // استخراج بيانات العميل
    $contact = Contact::find($transaction->contact_id);
    $cardCode = $contact->card_code ?? null;
    
    // التحقق من وجود كود العميل
    if (empty($cardCode)) {
        throw new \Exception("Customer does not have a SAP customer code");
    }
    
    $documentLines = [];
    $sellLines = TransactionSellLine::where('transaction_id', $transaction->id)->get();
    
    // التحقق المسبق من وجود سطور بيع
    if ($sellLines->isEmpty()) {
        Log::warning("No sell lines found for transaction #{$transaction->id}");
        throw new \Exception("No sell lines found for transaction");
    }
    
    foreach ($sellLines as $line) {
        try {
            $product = \App\Product::find($line->product_id);
            
            if (!$product) {
                Log::warning("Product not found for sell line #{$line->id}");
                continue;
            }
            
            // تحديد كود المنتج
            $itemCode = $product->sap_item_code ?? $product->sku;
            
            // التحقق من وجود كود المنتج
            if (empty($itemCode)) {
                Log::warning("No item code for product #{$product->id}");
                continue;
            }
            
            // معالجة الوحدة الرئيسية
            $mainUnitEntry = null;
            if (!empty($product->unit_id)) {
                $mainUnit = \App\Unit::find($product->unit_id);
                
                if ($mainUnit) {
                    // معالجة حالة "درزن" بشكل خاص
                    if (strtolower($mainUnit->name) === 'درزن') {
                        $mainUnitEntry = 60;
                    } else {
                        // البحث عن UoMEntry للوحدات الرئيسية الأخرى
                        $mainUnitEntry = $this->getUoMEntryForUnit($mainUnit->short_name);
                    }
                }
            }
            
            // حساب الكمية
            $quantity = $line->quantity;
            
            // إنشاء بيانات السطر الأساسية
            $lineData = [
                'ItemCode' => $itemCode,
                'Quantity' => $quantity,
                // 'UnitPrice' => $line->unit_price_inc_tax,
                // 'LineTotal' => $line->unit_price_inc_tax * $quantity,
            ];
            
            // إضافة معلومات المستودع إذا كانت متوفرة
            if (!empty($line->warehouse_code)) {
                // تحويل صيغة المستودع من W05 إلى WH05 لـ SAP
                $sapWarehouseCode = $this->convertWarehouseCodeForSap($line->warehouse_code);
                $lineData['WarehouseCode'] = $sapWarehouseCode;
                
                Log::info("Adding warehouse information to SAP document line", [
                    'ItemCode' => $itemCode,
                    'original_warehouse_code' => $line->warehouse_code,
                    'sap_warehouse_code' => $sapWarehouseCode,
                    'warehouse_id' => $line->warehouse_id ?? 'Not set'
                ]);
            } else {
                // استخدام المستودع الافتراضي 01 لـ SAP
                $lineData['WarehouseCode'] = '01';
                
                Log::info("Using default warehouse for SAP document line", [
                    'ItemCode' => $itemCode,
                    'WarehouseCode' => '01',
                    'reason' => 'No warehouse_code specified in sell line'
                ]);
            }
            
            // معالجة الوحدات
            $shouldAddUoMEntry = false;
            $uomEntryValue = null;
            $isSetUnit = false; // متغير لتتبع وحدات السيت
            
            // أولاً: التحقق من الوحدة الفرعية (لها الأولوية)
            if (!empty($line->sub_unit_id)) {
                $subUnit = \App\Unit::find($line->sub_unit_id);
                
                if ($subUnit) {
                    // البحث عن UoMEntry للوحدة الفرعية
                    $subUnitEntry = $this->getUoMEntryForUnit($subUnit->short_name);
                    
                    if ($subUnitEntry !== null) {
                        $shouldAddUoMEntry = true;
                        $uomEntryValue = $subUnitEntry;
                        $isSetUnit = ($subUnitEntry == 21); // تحديد إذا كانت وحدة سيت
                    }
                }
            }
            // ثانياً: إذا لم توجد وحدة فرعية، استخدم الوحدة الرئيسية
            elseif ($mainUnitEntry !== null) {
                $shouldAddUoMEntry = true;
                $uomEntryValue = $mainUnitEntry;
                $isSetUnit = ($mainUnitEntry == 21); // تحديد إذا كانت وحدة سيت
            }
            
            // إضافة UoMEntry فقط إذا كانت هناك وحدة محددة
            if ($shouldAddUoMEntry && $uomEntryValue !== null) {
                // تخطي UoMEntry للوحدات من نوع سيت (21)
                if ($uomEntryValue == 21) {
                    Log::info("Skipping UoMEntry for Set unit as requested", [
                        'ItemCode' => $itemCode,
                        'UoMEntry_skipped' => $uomEntryValue,
                        'reason' => 'Set units should be sent without UoMEntry'
                    ]);
                } else {
                    $lineData['UoMEntry'] = $uomEntryValue;
                    
                    Log::info("Added UoMEntry for non-Set unit (without NumPerMsr)", [
                        'ItemCode' => $itemCode,
                        'UoMEntry' => $uomEntryValue,
                        'note' => 'NumPerMsr is not sent for any unit type as requested'
                    ]);
                }
            }
            // إذا لم تكن هناك وحدة محددة وكود المنتج يبدأ بـ S
            elseif (!$shouldAddUoMEntry && strpos($itemCode, 'S') === 0) {
                // تحقق ما إذا كان المنتج من نوع سيت قبل إضافة درزن
                // هنا نفترض أن المنتجات التي تبدأ بـ S وتحتوي على "سيت" في الاسم هي سيت
                // يمكن تحسين هذا المنطق حسب الحاجة
                $lineData['UoMEntry'] = 18; // درزن (للمنتجات التي تبدأ بـ S وليست سيت)
            }
            
            // تسجيل البيانات النهائية للسطر
            if ($isSetUnit) {
                Log::info("Final line data for Set unit (sent without UoMEntry)", [
                    'ItemCode' => $itemCode,
                    'Quantity' => $quantity,
                    'data_sent' => $lineData,
                    'note' => 'Set unit sent without UoMEntry. NumPerMsr is not sent for any unit type.'
                ]);
            } else {
                Log::info("Final line data for non-Set unit", [
                    'ItemCode' => $itemCode,
                    'Quantity' => $quantity,
                    'UoMEntry' => $lineData['UoMEntry'] ?? 'Not set',
                    'data_sent' => $lineData,
                    'note' => 'NumPerMsr is not sent for any unit type as requested'
                ]);
            }
            
            $documentLines[] = $lineData;

        } catch (\Exception $e) {
            // تسجيل الأخطاء
            Log::error("Error processing sell line for SAP sync", [
                'sell_line_id' => $line->id,
                'error_message' => $e->getMessage(),
                'error_trace' => $e->getTraceAsString()
            ]);
        }
    }
    
    // التحقق مرة أخرى من وجود سطور مستندات
    if (empty($documentLines)) {
        Log::error("No valid document lines after processing for transaction #{$transaction->id}");
        throw new \Exception("No valid document lines could be created");
    }
    
    // حساب القيمة الإجمالية للمستند قبل الخصم
    $totalBeforeDiscount = 0;
    foreach ($documentLines as $line) {
        if (isset($line['UnitPrice'])) {
            $totalBeforeDiscount += $line['Quantity'] * $line['UnitPrice'];
        }
    }
    
    // إنشاء بيانات المستند النهائية
    $documentData = [
        'CardCode' => $cardCode,
        'DocumentLines' => $documentLines,
        'DocDueDate' => date('Y-m-d'),
        'DocDate' => date('Y-m-d')
    ];
    
    // إضافة معاملات الدرافت - يُرسل كـ Sales Order مع تحديد نوعه
    if ($transaction->status == 'draft' && $transaction->is_quotation != 1) {
        // إضافة معاملات لتحديد أن هذا draft sales order
        $documentData['DocStatus'] = config('sap.draft_doc_status', 'bost_Open'); // حالة مفتوحة للدرافت
        $documentData['DocType'] = 'dDocument_Items'; // نوع المستند - عناصر
        $draftCommentPrefix = config('sap.draft_comment_prefix', 'DRAFT SALES ORDER');
        $documentData['Comments'] = (!empty($documentData['Comments']) ? $documentData['Comments'] . ' | ' : '') . $draftCommentPrefix;
        
        // إضافة حقول مخصصة لتمييز الدرافت إذا كانت متاحة
        // $documentData['U_IsDraft'] = 'Y'; // إذا كان لديك حقل مخصص
        // $documentData['U_DraftType'] = 'DRAFT_ORDER'; // نوع الدرافت
        
        Log::info("Adding draft indicators to sales order document", [
            'transaction_id' => $transaction->id,
            'DocStatus' => config('sap.draft_doc_status', 'bost_Open'),
            'DocType' => 'dDocument_Items',
            'draft_comment_prefix' => $draftCommentPrefix,
            'draft_comment_added' => true,
            'endpoint' => 'Orders',
            'note' => 'Draft sent as sales order with configurable draft indicators'
        ]);
    }
    
    // إضافة معلومات الخصم إذا كانت متوفرة
    if (!empty($transaction->discount_amount) && $transaction->discount_amount > 0) {
        if ($transaction->discount_type === 'percentage') {
            // خصم بالنسبة المئوية
            $discountAmount = ($transaction->discount_amount / 100) * $totalBeforeDiscount;
            $documentData['DiscountPercent'] = $transaction->discount_amount;
            // $documentData['DocTotal'] = $totalBeforeDiscount - $discountAmount;
            
            Log::info("Adding percentage discount to SAP document", [
                'transaction_id' => $transaction->id,
                'discount_percent' => $transaction->discount_amount,
                'total_before_discount' => $totalBeforeDiscount,
                'discount_amount' => $discountAmount,
                // 'final_total' => $documentData['DocTotal']
            ]);
        } else {
            // خصم ثابت - حساب النسبة المئوية
            $discountAmount = $transaction->discount_amount;
            
            // حساب النسبة المئوية للخصم الثابت
            if ($totalBeforeDiscount > 0) {
                $discountPercent = ($discountAmount / $totalBeforeDiscount) * 100;
                $documentData['DiscountPercent'] = round($discountPercent, 2);
            }
            // $documentData['DocTotal'] = $totalBeforeDiscount - $discountAmount;
            
            Log::info("Adding fixed discount converted to percentage to SAP document", [
                'transaction_id' => $transaction->id,
                'discount_amount' => $discountAmount,
                'total_before_discount' => $totalBeforeDiscount,
                'calculated_discount_percent' => $documentData['DiscountPercent'] ?? 0,
                // 'final_total' => $documentData['DocTotal']
            ]);
        }
    } else {
        // لا يوجد خصم
        // $documentData['DocTotal'] = $totalBeforeDiscount;
    }
    
    // إضافة الملاحظات إذا كانت متوفرة
    $comments = [];
    
    // إضافة ملاحظات البيع (additional_notes)
    if (!empty($transaction->additional_notes)) {
        $comments[] = 'ملاحظات البيع: ' . $transaction->additional_notes;
    }
    
    // إضافة ملاحظات الموظف (staff_note)
    if (!empty($transaction->staff_note)) {
        $comments[] = 'ملاحظات الموظف: ' . $transaction->staff_note;
    }
    
    // دمج الملاحظات وإضافتها للمستند
    if (!empty($comments)) {
        $allComments = implode(' | ', $comments);
        $documentData['Comments'] = $allComments;
        
        Log::info("Adding comments to SAP document", [
            'transaction_id' => $transaction->id,
            'comments' => $allComments
        ]);
    }
    
    Log::info("Prepared document data for SAP", [
        'transaction_id' => $transaction->id,
        'card_code' => $cardCode,
        'document_lines_count' => count($documentLines),
        'has_discount' => !empty($transaction->discount_amount),
        'discount_type' => $transaction->discount_type ?? 'none',
        'discount_amount' => $transaction->discount_amount ?? 0,
        'has_comments' => !empty($comments),
        // 'doc_total' => $documentData['DocTotal']
    ]);
    
    return $documentData;
}

/**
 * دالة ذكية للبحث عن UoMEntry للوحدة
 * محدثة لتتعامل بشكل أفضل مع "لك" و "0.25 كغم"
 *
 * @param string $unitShortName اسم الوحدة المختصر
 * @return int رقم UoMEntry (لن يرجع null أبداً)
 */
private function getUoMEntryForUnit($unitShortName) {
    // قاموس يحتوي على مطابقات الوحدات مع UoMEntry
    $uomEntries = [
        'Manual' => 1,
        'قطعة' => 2,
        'كارتون 120 ق' => 3,
        'لتر' => 4,
        'فل 0.25' => 11,
        'كغم' => 6,
        'لك' => 6,  // نفس كود الكيلوجرام
        'غم 50' => 7,
        '100 غم' => 8,
        'غم 125' => 9,
        'غم 250' => 10,
        '0.5 كيلو' => 10,
        '0.25 كغم' => 11,
        'دبة 5 لتر' => 12,
        'كغم 25' => 13,
        'كغم 100' => 14,
        'كغم 500' => 15,
        'كغم 1000' => 16,
        'درزن' => 17,
        'كارتون' => 18,
        'كارتون$10' => 19,
        'باكيت' => 20,
        'سيت 50ق' => 21,
        'برميل 200 لتر' => 22,
        'دبة 20 لتر' => 23,
        'برميل 30 لتر' => 24,
        'سيت' => 21,
        'كغم 250' => 26,
        'ربطة' => 27,
    ];
    
    // تنظيف اسم الوحدة
    $originalUnit = trim($unitShortName);
    $unitKey = mb_strtolower($originalUnit);
    
    // 1. محاولة المطابقة المباشرة
    foreach ($uomEntries as $key => $value) {
        if (mb_strtolower($key) === $unitKey) {
            Log::info("Direct match found: '{$originalUnit}' -> '{$key}' (UoMEntry: {$value})");
            return $value;
        }
    }
    
    // 2. معالجة حالات خاصة للوحدات الرئيسية
    // حالة "لك" - تعامل معها ككيلوجرام
    if ($unitKey === 'لك' || $unitKey === 'لـك') {
        Log::info("Special case 'لك' found: '{$originalUnit}' -> 'كغم' (UoMEntry: 6)");
        return 6;
    }
    
    // حالة "0.25 كغم" أو أي صيغة مشابهة
    if (preg_match('/^0\.25\s*(كغم|كيلو|كجم|كيلوغرام|كيلوجرام)$/u', $unitKey)) {
        Log::info("Special case '0.25 كغم' found: '{$originalUnit}' -> '0.25 كغم' (UoMEntry: 11)");
        return 11;
    }
    
    // 3. استخراج المكونات (رقم + وحدة)
    $number = null;
    $unit = null;
    $hasNumber = false;
    
    // نمط للبحث عن رقم في البداية أو النهاية
    if (preg_match('/^(\d+\.?\d*)\s*(.+)$/u', $unitKey, $matches)) {
        $number = $matches[1];
        $unit = trim($matches[2]);
        $hasNumber = true;
    } elseif (preg_match('/^(.+?)\s*(\d+\.?\d*)$/u', $unitKey, $matches)) {
        $unit = trim($matches[1]);
        $number = $matches[2];
        $hasNumber = true;
    } else {
        $unit = $unitKey;
    }
    
    // 4. قاموس المرادفات والتحويلات
    $unitSynonyms = [
        'غم' => ['جرام', 'غرام', 'جم', 'gm', 'gram'],
        'كغم' => ['كيلو', 'كيلوغرام', 'كيلوجرام', 'كجم', 'kg', 'كلغ', 'ك', 'لك', 'لـك'],
        'لتر' => ['ليتر', 'l', 'liter'],
        'قطعة' => ['قطع', 'حبة', 'عدد', 'piece', 'pcs'],
        'كارتون' => ['كرتون', 'كرتونة', 'صندوق', 'carton'],
        'درزن' => ['دزينة', 'دزن', 'dozen'],
        'باكيت' => ['باكت', 'بكت', 'بكيت', 'packet'],
        'سيت' => ['ست', 'طقم', 'set'],
        'فل' => ['فلة', 'fl'],
        'دبة' => ['جالون', 'دبه'],
        'برميل' => ['براميل', 'barrel'],
        'ربطة' => ['ربطه', 'حزمة', 'bundle']
    ];
    
    // تحويل الوحدة إلى الشكل القياسي
    $standardUnit = $unit;
    foreach ($unitSynonyms as $standard => $synonyms) {
        if ($unit === $standard || in_array($unit, $synonyms)) {
            $standardUnit = $standard;
            break;
        }
    }
    
    // 5. إعادة بناء مفتاح البحث بأشكال مختلفة
    $searchKeys = [];
    
    if ($hasNumber) {
        // جرب أشكال مختلفة للرقم والوحدة
        $searchKeys[] = $standardUnit . ' ' . $number;
        $searchKeys[] = $number . ' ' . $standardUnit;
        $searchKeys[] = $standardUnit . $number;
        $searchKeys[] = $number . $standardUnit;
        
        // حالات خاصة
        if ($standardUnit === 'غم') {
            $searchKeys[] = 'غم ' . $number;
            $searchKeys[] = $number . ' غم';
        }
        if ($standardUnit === 'كغم') {
            $searchKeys[] = 'كغم ' . $number;
            $searchKeys[] = $number . ' كغم';
            if ($number == '0.5') {
                $searchKeys[] = '0.5 كيلو';
            }
            if ($number == '0.25') {
                $searchKeys[] = '0.25 كغم';
            }
        }
        if ($standardUnit === 'فل' && $number == '0.25') {
            $searchKeys[] = 'فل 0.25';
        }
        if ($standardUnit === 'لتر') {
            $searchKeys[] = 'دبة ' . $number . ' لتر';
            if ($number == '200') {
                $searchKeys[] = 'برميل 200 لتر';
            }
        }
        if ($standardUnit === 'قطعة' && $number == '120') {
            $searchKeys[] = 'كارتون 120 ق';
        }
    } else {
        $searchKeys[] = $standardUnit;
    }
    
    // 6. البحث باستخدام المفاتيح المحولة
    foreach ($searchKeys as $searchKey) {
        foreach ($uomEntries as $key => $value) {
            if (mb_strtolower($key) === mb_strtolower($searchKey)) {
                Log::info("Smart match found: '{$originalUnit}' -> '{$key}' (UoMEntry: {$value})");
                return $value;
            }
        }
    }
    
    // 7. المطابقة الجزئية الذكية
    $bestMatch = null;
    $bestScore = 0;
    
    foreach ($uomEntries as $key => $value) {
        $keyLower = mb_strtolower($key);
        $score = 0;
        
        // حساب نقاط التشابه
        if ($hasNumber && strpos($keyLower, $number) !== false) {
            $score += 3;
        }
        if (strpos($keyLower, $standardUnit) !== false) {
            $score += 2;
        }
        
        // تحقق من الكلمات المشتركة
        $keyWords = preg_split('/\s+/', $keyLower);
        $unitWords = preg_split('/\s+/', $unitKey);
        $commonWords = array_intersect($keyWords, $unitWords);
        $score += count($commonWords);
        
        // تحديث أفضل مطابقة
        if ($score > $bestScore) {
            $bestScore = $score;
            $bestMatch = $value;
        }
    }
    
    if ($bestMatch !== null && $bestScore >= 2) {
        Log::info("Partial match found: '{$originalUnit}' -> UoMEntry: {$bestMatch} (Score: {$bestScore})");
        return $bestMatch;
    }
    
    // 8. القرار الذكي النهائي بناءً على نوع الوحدة
    $defaultMappings = [
        'غم' => 8,      // 100 غم كافتراضي للجرامات
        'كغم' => 6,     // كغم كافتراضي للكيلوجرامات
        'لتر' => 4,     // لتر
        'قطعة' => 2,    // قطعة
        'كارتون' => 18, // كارتون
        'درزن' => 17,   // درزن
        'باكيت' => 20,  // باكيت
        'سيت' => 25,    // سيت
        'فل' => 5,      // فل 0.25
        'دبة' => 12,    // دبة 5 لتر
        'برميل' => 22,  // برميل 200 لتر
        'ربطة' => 27    // ربطة
    ];
    
    // البحث عن وحدة افتراضية مناسبة
    foreach ($defaultMappings as $unitType => $defaultEntry) {
        if (strpos($unitKey, $unitType) !== false || strpos($standardUnit, $unitType) !== false) {
            Log::warning("Using default mapping: '{$originalUnit}' -> '{$unitType}' (UoMEntry: {$defaultEntry})");
            return $defaultEntry;
        }
    }
    
    // 9. الافتراضي النهائي - قطعة
    Log::warning("No match found for unit: '{$originalUnit}'. Using default 'قطعة' (UoMEntry: 2)");
    return 2; // قطعة كافتراضي نهائي
}

    /**
     * تحويل كود المستودع من صيغة النظام إلى صيغة SAP
     * مثال: W05 -> 05
     *
     * @param string $warehouseCode
     * @return string
     */
    private function convertWarehouseCodeForSap($warehouseCode)
    {
        // إذا كان الكود يبدأ بـ W متبوعاً برقم، أرسل الرقم فقط
        if (preg_match('/^W(\d+)$/', $warehouseCode, $matches)) {
            return $matches[1];
        }
        
        // إذا كان الكود يحتوي على صيغة أخرى، أعده كما هو
        return $warehouseCode;
    }

/* 
 * الدوال التالية محذوفة نهائياً لأن NumPerMsr لم يعد يُستخدم:
 * - getNumPerMsrForUoMEntry()
 * - extractNumPerMsrFromUnitName()
 * 
 * تم إزالة NumPerMsr من جميع المستندات المرسلة لـ SAP
 */
}