<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use App\Product;
use App\Variation;
use App\ProductVariation;
use App\SapWarehouseMapping;
use App\SapUomMapping;
use App\SapProductWarehouse;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Unit;
use App\VariationLocationDetails;  // أضف هذا السطر
use App\BusinessLocation;          // أضف هذا السطر

class SapItemsService extends SapB1Service
{
    /**
     * Get items/products from SAP B1
     */
    public function getItems($filter = null, $select = null, $top = 1000, $skip = 0, $timeout = 60000, $expand = null)
    {
        $queryParams = [];
        if ($filter) {
            $queryParams['$filter'] = $filter;
        }
        if ($select) {
            $queryParams['$select'] = $select;
        }
        if ($expand) {
            $queryParams['$expand'] = $expand;
        }
        if ($top > 0) {
            $queryParams['$top'] = $top;
        }
        if ($skip > 0) {
            $queryParams['$skip'] = $skip;
        }
        
        return $this->get('Items', $queryParams, $timeout);
    }

    /**
     * Get a single item by ItemCode
     */
    public function getItem($itemCode, $expand = null)
    {
        $queryParams = [];
        if ($expand) {
            $queryParams['$expand'] = $expand;
        }
        
        return $this->get("Items('" . addslashes($itemCode) . "')", $queryParams);
    }

    /**
     * Create a new item in SAP B1
     */
    public function createItem($itemData)
    {
        return $this->post('Items', $itemData);
    }

    /**
     * Update an existing item in SAP B1
     */
    public function updateItem($itemCode, $itemData)
    {
        return $this->patch("Items('" . addslashes($itemCode) . "')", $itemData);
    }

    /**
     * Create or update item in SAP B1
     */
    public function createOrUpdateItem($itemCode, $itemData)
    {
        try {
            // First try to get the item to see if it exists
            $existingItem = $this->getItem($itemCode);
            
            if ($existingItem) {
                // Item exists, update it
                return $this->updateItem($itemCode, $itemData);
            }
        } catch (\Exception $e) {
            // Item doesn't exist or error occurred, try to create
            try {
                return $this->createItem($itemData);
            } catch (\Exception $createError) {
                Log::error("Failed to create item {$itemCode}: " . $createError->getMessage());
                throw $createError;
            }
        }
        
        return null;
    }

    /**
     * Get all items using cursor-based pagination - محسّنة للبيانات الكبيرة
     */
    public function getAllItems($filter = null, $select = null, $expand = null)
    {
        // أولاً: جلب العدد الإجمالي من SAP
        try {
            $countParams = ['$count' => 'true', '$top' => 1];
            if ($filter) {
                $countParams['$filter'] = $filter;
            }
            
            Log::info("جلب العدد الإجمالي للمنتجات من SAP", ['filter' => $filter]);
            $countResult = $this->get('Items', $countParams, 60000);
            
            $totalAvailable = isset($countResult['@odata.count']) ? $countResult['@odata.count'] : 'unknown';
            Log::info("العدد الإجمالي للمنتجات المتاح في SAP: {$totalAvailable}");
            
        } catch (\Exception $e) {
            Log::warning("فشل في جلب العدد الإجمالي للمنتجات: " . $e->getMessage());
            $totalAvailable = 'unknown';
        }
        
        $allItems = [];
        $batchSize = 200; // حجم دفعة أصغر للبداية 
        $iteration = 0;
        $maxIterations = 20000; // زيادة كبيرة للحد الأقصى للمنتجات
        $consecutiveFailures = 0;
        $maxConsecutiveFailures = 3;
        $totalFetched = 0;
        $lastItemCode = '';
        
        Log::info("بدء جلب جميع المنتجات من SAP باستخدام cursor pagination", [
            'filter' => $filter,
            'select' => $select,
            'expand' => $expand,
            'batch_size' => $batchSize,
            'total_available_in_sap' => $totalAvailable
        ]);
        
        while ($iteration < $maxIterations) {
            $iteration++;
            
            try {
                // بناء فلتر مع cursor-based pagination
                $cursorFilter = $filter;
                if (!empty($lastItemCode)) {
                    $cursorCondition = "ItemCode gt '{$lastItemCode}'";
                    $cursorFilter = $cursorFilter ? "({$filter}) and {$cursorCondition}" : $cursorCondition;
                }
                
                $queryParams = [
                    '$top' => $batchSize,
                    '$orderby' => 'ItemCode asc',
                    '$count' => 'true' // للحصول على معلومات إضافية
                ];
                
                if ($cursorFilter) {
                    $queryParams['$filter'] = $cursorFilter;
                }
                
                if ($select) {
                    $queryParams['$select'] = $select;
                }
                
                if ($expand) {
                    $queryParams['$expand'] = $expand;
                }
                
                Log::info("جلب دفعة المنتجات {$iteration} بعد ItemCode: '{$lastItemCode}'", [
                    'filter' => $cursorFilter,
                    'batch_size' => $batchSize,
                    'total_fetched_so_far' => $totalFetched
                ]);
                
                // زيادة timeout إلى 4 دقائق للمنتجات لأنها تحتوي على بيانات أكثر
                $result = $this->get('Items', $queryParams, 240000);
                
                if (empty($result) || empty($result['value']) || count($result['value']) == 0) {
                    Log::info("لا توجد المزيد من المنتجات، انتهاء التصفح بعد {$totalFetched} منتج");
                    break;
                }

                $batchCount = count($result['value']);
                $allItems = array_merge($allItems, $result['value']);
                $totalFetched += $batchCount;
                $consecutiveFailures = 0; // إعادة تعيين عداد الفشل عند النجاح
                
                // تحديث آخر ItemCode
                $lastItemCode = $result['value'][$batchCount - 1]['ItemCode'];
                
                Log::info("تم جلب دفعة منتجات بنجاح", [
                    'batch_count' => $batchCount,
                    'iteration' => $iteration,
                    'total_fetched' => $totalFetched,
                    'last_item_code' => $lastItemCode,
                    'has_nextLink' => isset($result['odata.nextLink']),
                    'batch_vs_requested' => "{$batchCount}/{$batchSize}"
                ]);
                
                // التحقق من انتهاء البيانات يعتمد على nextLink وليس حجم الدفعة
                if (!isset($result['odata.nextLink'])) {
                    Log::info("انتهت البيانات - لا يوجد nextLink", [
                        'batch_count' => $batchCount,
                        'batch_size' => $batchSize,
                        'has_nextLink' => false
                    ]);
                    break;
                } else {
                    Log::info("يوجد المزيد من المنتجات - nextLink موجود", [
                        'batch_count' => $batchCount,
                        'nextLink' => substr($result['odata.nextLink'], -50) // آخر 50 حرف من الرابط
                    ]);
                }
                
                // تأخير قصير بين الدفعات لتجنب إرهاق الخادم
                if ($iteration % 5 == 0) {
                    Log::info("تأخير قصير بعد 5 دفعات...");
                    sleep(1);
                }
                
                // مراقبة الذاكرة بشكل أكثر تفصيلاً للمنتجات
                if ($iteration % 10 == 0) {
                    $memoryMB = memory_get_usage(true) / 1024 / 1024;
                    Log::info("معلومات الذاكرة: {$memoryMB} MB بعد {$totalFetched} منتج");
                    if ($memoryMB > 1024) { // أكثر من 1GB
                        Log::warning("استخدام ذاكرة عالي جداً، تقليل حجم الدفعة بقوة");
                        $batchSize = max(50, intval($batchSize * 0.5));
                    } elseif ($memoryMB > 512) { // أكثر من 512MB
                        Log::warning("استخدام ذاكرة عالي، تقليل حجم الدفعة");
                        $batchSize = max(100, intval($batchSize * 0.8));
                    }
                }
                
            } catch (\Exception $e) {
                $consecutiveFailures++;
                Log::error("خطأ في جلب دفعة المنتجات {$iteration} (بعد ItemCode: {$lastItemCode}): " . $e->getMessage());
                
                // إذا كان timeout أو خطأ في الذاكرة، نقلل حجم الدفعة
                if (strpos($e->getMessage(), 'timed out') !== false || 
                    strpos($e->getMessage(), 'timeout') !== false ||
                    strpos($e->getMessage(), 'Connection timed out') !== false ||
                    strpos($e->getMessage(), 'memory') !== false ||
                    strpos($e->getMessage(), 'Maximum execution time') !== false) {
                    
                    if ($batchSize > 50) {
                        $batchSize = max(50, intval($batchSize / 2));
                        Log::info("timeout أو مشكلة ذاكرة، تقليل حجم الدفعة إلى {$batchSize} وإعادة المحاولة");
                        // لا نغير lastItemCode، لإعادة المحاولة بنفس المكان
                        continue;
                    }
                }
                
                // للأخطاء الأخرى، نحاول الانتقال للدفعة التالية
                if ($consecutiveFailures < $maxConsecutiveFailures) {
                    Log::info("محاولة المتابعة بعد خطأ...");
                    
                    // محاولة تخطي هذا السجل المشكل
                    if (!empty($lastItemCode)) {
                        // الحصول على ItemCode التالي لتخطي المشكل
                        try {
                            $skipFilter = $filter ? "({$filter}) and ItemCode gt '{$lastItemCode}'" : "ItemCode gt '{$lastItemCode}'";
                            $skipResult = $this->get('Items', [
                                '$filter' => $skipFilter,
                                '$select' => 'ItemCode',
                                '$top' => 1,
                                '$orderby' => 'ItemCode asc'
                            ], 30000);
                            
                            if (!empty($skipResult['value'])) {
                                $lastItemCode = $skipResult['value'][0]['ItemCode'];
                                Log::info("تخطي إلى ItemCode التالي: {$lastItemCode}");
                            }
                        } catch (\Exception $skipError) {
                            Log::warning("فشل في العثور على ItemCode التالي: " . $skipError->getMessage());
                        }
                    }
                    
                    // تأخير أطول بعد الخطأ
                    sleep(5);
                } else {
                    Log::error("فشل متكرر ({$consecutiveFailures} مرات)، إيقاف العملية");
                    break;
                    
                    // إذا لم نحصل على أي بيانات، نرمي الخطأ
                    if ($totalFetched == 0) {
                        throw new \Exception("فشل في جلب أي منتجات من SAP: " . $e->getMessage());
                    }
                }
            }
        }

        // تحقق من الوصول للحد الأقصى للتكرارات
        if ($iteration >= $maxIterations) {
            Log::warning("تم الوصول للحد الأقصى للتكرارات ({$maxIterations})، قد يكون هناك المزيد من البيانات");
        }

        Log::info("انتهى جلب المنتجات: {$totalFetched} منتج بعد {$iteration} تكرار");
        return $allItems;
    }

    /**
     * Sync all products from SAP B1 to local database
     */
    public function syncAllProducts($fullSync = false)
    {
        $stats = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0,
            'warehouses_synced' => 0,
            'uom_groups_synced' => 0,
            'batches_processed' => 0,
            'processing_time' => 0
        ];
    
        $startTime = microtime(true);
        
        try {
            Log::info("بدء مزامنة المنتجات من SAP. مزامنة كاملة: " . ($fullSync ? 'نعم' : 'لا'));
            
            // Determine last sync time for incremental updates
            $lastSync = null;
            if (!$fullSync) {
                $latestSync = Product::whereNotNull('sap_last_sync')
                    ->orderBy('sap_last_sync', 'desc')
                    ->first();
                
                if ($latestSync) {
                    $carbon = Carbon::parse($latestSync->sap_last_sync);
                    $lastSync = $carbon->format('Y-m-d\TH:i:s');
                    Log::info("Performing incremental sync since: {$lastSync}");
                }
            }
    
            // Build filter for items
            $filter = null;
            if (!$fullSync && $lastSync) {
                $filter = "UpdateDate ge '{$lastSync}'";
            }
    
            // For full item data, we need to expand certain collections
            // This will include ItemPrices and ItemWarehouseInfoCollection in the response
            $expand = "ItemPrices,ItemWarehouseInfoCollection,ItemBarCodeCollection,ItemUnitOfMeasurementCollection";
    
            // Get total count of items for progress tracking
            $countResult = $this->getItems($filter, 'ItemCode', 1, 0, 30);
            $totalCount = $countResult['@odata.count'] ?? 0;
            Log::info("Total items to process: {$totalCount}");
    
            // Process items in batches
            $batchSize = 50; // Reduced batch size since we're expanding data
            $skip = 0;
            $hasMore = true;
            
            while ($hasMore) {
                DB::beginTransaction();
                try {
                    Log::info("Processing batch: skip={$skip}, size={$batchSize}");
                    
                    // Get batch of items from SAP with expanded data
                    $sapItems = $this->getItems($filter, null, $batchSize, $skip, 180, $expand);
                    
                    if (empty($sapItems) || empty($sapItems['value']) || count($sapItems['value']) == 0) {
                        Log::info("No more items to process, batch processing complete");
                        $hasMore = false;
                        DB::commit();
                        break;
                    }
                    
                    $batchCount = count($sapItems['value']);
                    $stats['batches_processed']++;
                    Log::info("Retrieved {$batchCount} items in batch {$stats['batches_processed']}");
                    
                    // Process each item in the batch
                    foreach ($sapItems['value'] as $sapItem) {
                        try {
                            $itemCode = $sapItem['ItemCode'];
                            
                            // The expanded data should already be included
                            // No need to fetch separately
                            
                            $result = $this->processSapItem($sapItem, $fullSync);
                            
                            if ($result === 'created') {
                                $stats['created']++;
                            } elseif ($result === 'updated') {
                                $stats['updated']++;
                            } elseif ($result === 'skipped') {
                                $stats['skipped']++;
                            }
                            
                            $stats['total']++;
                            
                        } catch (\Exception $e) {
                            Log::error("Error processing item {$sapItem['ItemCode']}: " . $e->getMessage());
                            $stats['failed']++;
                            $stats['total']++;
                        }
                    }
                    
                    // Commit transaction for this batch
                    DB::commit();
                    
                    // Update skip for next batch
                    $skip += $batchSize;
                    
                    // Check if we need to continue pagination
                    if (isset($sapItems['odata.nextLink'])) {
                        // Extract skip value from nextLink
                        if (preg_match('/\$skip=(\d+)/', $sapItems['odata.nextLink'], $matches)) {
                            $skip = (int)$matches[1];
                        }
                    } else if ($batchCount < $batchSize) {
                        // If we got fewer items than requested, we've reached the end
                        $hasMore = false;
                    }
                    
                    // Log progress
                    $progress = min(100, round(($stats['total'] / max(1, $totalCount)) * 100, 2));
                    Log::info("Sync progress: {$progress}% ({$stats['total']}/{$totalCount})");
                    
                } catch (\Exception $e) {
                    DB::rollBack();
                    Log::error("Error processing batch (skip: {$skip}): " . $e->getMessage());
                    
                    // If we hit a timeout or other error, we'll still try the next batch
                    $skip += $batchSize;
                    
                    // But if we've had multiple consecutive failures, stop
                    if ($stats['batches_processed'] > 0 && $stats['total'] == 0) {
                        $hasMore = false;
                        Log::error("Multiple failures with no successful retrievals, stopping");
                    }
                }
            }
    
            // Sync warehouses and UoM groups after all product batches
            try {
                DB::beginTransaction();
                
                // Sync warehouses
                Log::info("Syncing warehouses...");
                $stats['warehouses_synced'] = $this->syncWarehouses();
                
                // Sync UoM Groups
                Log::info("Syncing UoM groups...");
                $stats['uom_groups_synced'] = $this->syncUomGroups();
                
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error("Error syncing warehouses or UoM groups: " . $e->getMessage());
            }
            
            // حساب وقت المعالجة
            $endTime = microtime(true);
            $stats['processing_time'] = round($endTime - $startTime, 2);
            
            Log::info("انتهت مزامنة المنتجات بنجاح في {$stats['processing_time']} ثانية");
            Log::info("إحصائيات المزامنة: " . json_encode($stats));
            return $stats;
            
        } catch (\Exception $e) {
            if (DB::transactionLevel() > 0) {
                DB::rollBack();
            }
            Log::error("SAP Sync All Products Exception: " . $e->getMessage());
            Log::error("Stack trace: " . $e->getTraceAsString());
            throw $e;
        }
    }
    
    /**
     * Get warehouse data for a specific item
     */
    protected function getItemWarehouses($itemCode)
{
    try {
        // Get the full item data which includes warehouse info
        $item = $this->getItem($itemCode);
        
        if ($item && isset($item['ItemWarehouseInfoCollection'])) {
            return $item['ItemWarehouseInfoCollection'];
        }
    } catch (\Exception $e) {
        Log::debug("Could not get warehouse data for item {$itemCode}: " . $e->getMessage());
    }
    
    return [];
}
    
    /**
     * Get price data for a specific item
     */
    protected function getItemPrices($itemCode)
{
    try {
        // Get the full item data which includes price info
        $item = $this->getItem($itemCode);
        
        if ($item && isset($item['ItemPrices'])) {
            return $item['ItemPrices'];
        }
    } catch (\Exception $e) {
        Log::debug("Could not get price data for item {$itemCode}: " . $e->getMessage());
    }
    
    return [];
}

    /**
     * Calculate the total number of sale transactions for a product
     * 
     * @param int $productId
     * @return int Total count of sale transactions
     */
    protected function calculateProductSalesCount($productId)
    {
        try {
            // Count distinct sale transactions for this product
            // We're counting the number of transactions, not the total quantity
            $salesCount = DB::table('transaction_sell_lines')
                ->join('transactions', 'transaction_sell_lines.transaction_id', '=', 'transactions.id')
                ->where('transaction_sell_lines.product_id', $productId)
                ->where('transactions.status', 'final')
                ->where('transactions.type', 'sell')
                ->distinct('transaction_sell_lines.transaction_id')
                ->count('transaction_sell_lines.transaction_id');
            
            Log::info("Calculated sales count for product {$productId}: {$salesCount} transactions");
            
            return $salesCount;
            
        } catch (\Exception $e) {
            Log::error("Error calculating sales count for product {$productId}: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Update sales counts for all products
     * This can be called separately to refresh sales counts without full SAP sync
     * 
     * @return array Statistics about the update process
     */
    public function updateAllProductSalesCounts()
    {
        $stats = [
            'total' => 0,
            'updated' => 0,
            'failed' => 0
        ];
        
        try {
            Log::info("Starting bulk update of product sales counts");
            
            // Get all active products
            $products = Product::where('is_inactive', 0)->get();
            $stats['total'] = $products->count();
            
            foreach ($products as $product) {
                try {
                    $salesCount = $this->calculateProductSalesCount($product->id);
                    
                    // Update the custom field with sales count
                    $product->product_custom_field4 = $salesCount;
                    $product->save();
                    
                    $stats['updated']++;
                    
                    Log::info("Updated sales count for product {$product->id} (SKU: {$product->sku}): {$salesCount} transactions");
                    
                } catch (\Exception $e) {
                    $stats['failed']++;
                    Log::error("Failed to update sales count for product {$product->id}: " . $e->getMessage());
                }
            }
            
            Log::info("Completed bulk update of product sales counts. Stats: " . json_encode($stats));
            return $stats;
            
        } catch (\Exception $e) {
            Log::error("Error in updateAllProductSalesCounts: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Process a single SAP item and create/update local product
     */
    public function processSapItem($sapItem, $fullSync)
{
    $itemCode = $sapItem['ItemCode'];
    
    try {
        // Try to find existing product by SAP item code
        $product = Product::where('sap_item_code', $itemCode)->first();
        
        // If not found, try to find by SKU
        if (!$product) {
            $product = Product::where('sku', $itemCode)->first();
        }
        
        // Determine if this is a product we want to import
        $isImportable = $this->isItemImportable($sapItem);
        
        if (!$isImportable && !$product) {
            return 'skipped';
        }
        
        // Calculate total on-hand quantity from warehouse info
        $totalOnHand = 0;
        $warehouseData = [];
        
        if (!empty($sapItem['ItemWarehouseInfoCollection']) && is_array($sapItem['ItemWarehouseInfoCollection'])) {
            foreach ($sapItem['ItemWarehouseInfoCollection'] as $warehouseInfo) {
                $inStock = $warehouseInfo['InStock'] ?? 0;
                $totalOnHand += $inStock;
                
                // Store warehouse data for later processing
                $warehouseData[] = [
                    'warehouse_code' => $warehouseInfo['WarehouseCode'] ?? null,
                    'warehouse_name' => null, // Will be set from warehouse sync
                    'in_stock' => $inStock,
                    'committed' => $warehouseInfo['Committed'] ?? 0,
                    'ordered' => $warehouseInfo['Ordered'] ?? 0
                ];
            }
        }
        
        // Get price info from ItemPrices
        $sellPrice = null;
        $purchasePrice = $sapItem['MovingAveragePrice'] ?? 0;
        $priceList = null;
        $currency = null;
        
        if (!empty($sapItem['ItemPrices']) && is_array($sapItem['ItemPrices']) && count($sapItem['ItemPrices']) > 0) {
            // Get the first price list entry (usually the default)
            $priceData = $sapItem['ItemPrices'][0];
            $sellPrice = $priceData['Price'] ?? null;
            $priceList = $priceData['PriceList'] ?? null;
            $currency = $priceData['Currency'] ?? null;
        }
        
        // If no price from ItemPrices, try other price fields
        if (is_null($sellPrice) || $sellPrice == 0) {
            $sellPrice = $sapItem['AvgStdPrice'] ?? $purchasePrice;
        }
        
        // Get UoM information
        $inventoryUom = $sapItem['InventoryUOM'] ?? null;
        $uomGroupEntry = $sapItem['UoMGroupEntry'] ?? null;
        $saleunit = $sapItem['SalesUnit'] ?? null;
        $SItemPrices = $sapItem['ItemPrices'] ?? null;
        
        // Prepare SAP-specific data
        $sapData = [
            'sap_item_code' => $itemCode,
            'sap_foreign_name' => $sapItem['ForeignName'] ?? null,
            'sap_items_group_code' => $sapItem['ItemsGroupCode'] ?? null,
            'sap_inventory_item' => $sapItem['InventoryItem'] ?? null,
            'sap_purchase_item' => $sapItem['PurchaseItem'] ?? null,
            'sap_sales_item' => $sapItem['SalesItem'] ?? null,
            'sap_item_type' => $sapItem['ItemType'] ?? null,
            'sap_item_class' => $sapItem['ItemClass'] ?? null,
            'sap_inventory_uom' => $inventoryUom,
            'sale_unit' => $saleunit,
            'uom_prices' => $SItemPrices,
            'sap_uom_group_entry' => $uomGroupEntry,
            'sap_manufacturer' => $sapItem['Manufacturer'] ?? null,
            'sap_on_hand' => $totalOnHand,
            'sap_quantity_on_stock' => $sapItem['QuantityOnStock'] ?? null,
            'sap_quantity_ordered' => $sapItem['QuantityOrderedFromVendors'] ?? null,
            'sap_quantity_ordered_by_customers' => $sapItem['QuantityOrderedByCustomers'] ?? null,
            'sap_valid_from' => !empty($sapItem['ValidFrom']) ? $sapItem['ValidFrom'] : null,
            'sap_valid_to' => !empty($sapItem['ValidTo']) ? $sapItem['ValidTo'] : null,
            'sap_frozen' => $sapItem['Frozen'] ?? null,
            'sap_price' => $sellPrice,
            'sap_price_list' => $priceList,
            'sap_currency' => $currency,
            'sap_moving_average_price' => $purchasePrice,
            'sap_avg_std_price' => $sapItem['AvgStdPrice'] ?? null,
            'sap_default_warehouse' => $sapItem['DefaultWarehouse'] ?? null,
            'sap_manage_stock_by_warehouse' => $sapItem['ManageStockByWarehouse'] ?? null,
            'sap_material_type' => $sapItem['MaterialType'] ?? null,
            'sap_material_group' => $sapItem['MaterialGroup'] ?? null,
            'sap_update_date' => !empty($sapItem['UpdateDate']) ? $sapItem['UpdateDate'] : null,
            'sap_update_time' => $sapItem['UpdateTime'] ?? null,
            'sap_create_date' => !empty($sapItem['CreateDate']) ? $sapItem['CreateDate'] : null,
            'sap_create_time' => $sapItem['CreateTime'] ?? null,
            'sap_last_sync' => now(),
            'sap_sync_status' => 'success',
            'sap_sync_message' => null
        ];
        
        if ($product) {
            // Update existing product - FORCE UPDATE ALL FIELDS
            
            // Calculate and update sales count
            $salesCount = $this->calculateProductSalesCount($product->id);
            $sapData['product_custom_field4'] = $salesCount;
            Log::info("Updating sales count for existing product {$product->id} (SKU: {$itemCode}): {$salesCount} transactions");
            
            // Update SAP-specific fields
            foreach ($sapData as $key => $value) {
                $product->{$key} = $value;
            }
            
            // FORCE UPDATE core product fields with SAP data
            // Always update name with SAP ItemName
            $product->name = $sapItem['ItemName'] ?? $itemCode;
            
            // Always update type
            $product->type = $this->mapSapTypeToLocal($sapItem['ItemType'] ?? 'itItems');
            
            // Update quantity from SAP
            $product->quantity = $totalOnHand;
            
            // Force update price fields
            if (!is_null($sellPrice)) {
                $product->sale_price = $sellPrice;
                $product->default_sell_price = $sellPrice;
            }
            
            if ($purchasePrice > 0) {
                $product->purchase_price = $purchasePrice;
                $product->default_purchase_price = $purchasePrice;
            }
            
            // Update status fields
            $product->is_inactive = $sapItem['Frozen'] === 'tYES' ? 1 : 0;
            $product->not_for_selling = $sapItem['SalesItem'] === 'tNO' ? 1 : 0;
            
            // ALWAYS update SKU with SAP item code
            $product->sku = $itemCode;
            
            // Handle barcode if available
            if (!empty($sapItem['ItemBarCodeCollection']) && is_array($sapItem['ItemBarCodeCollection'])) {
                $firstBarcode = $sapItem['ItemBarCodeCollection'][0] ?? null;
                if ($firstBarcode && !empty($firstBarcode['Barcode'])) {
                    // If you have a barcode field, update it here
                    // $product->barcode = $firstBarcode['Barcode'];
                }
            }
            
            // Update sales count in custom field
            $product->product_custom_field4 = $salesCount;
            
            $product->save();
            
            // Force update variation prices and sub_sku
            $this->updateProductVariationPrices($product, $sellPrice, $purchasePrice);
            
            // Force update variation sub_sku to match
            foreach ($product->variations as $variation) {
                $variation->sub_sku = $itemCode;
                $variation->save();
            }
            
            // Process warehouse data
            $this->processWarehouseData($product, $warehouseData);
            
            // Update variation location details based on SAP warehouse data
            $this->updateVariationLocationDetails($product, $warehouseData);
            
            // Process UoM data if available
            if (!empty($sapItem['ItemUnitOfMeasurementCollection'])) {
                $this->processUomData($product, $sapItem['ItemUnitOfMeasurementCollection'], $uomGroupEntry);
            }
            
            return 'updated';
            
        } else {
            // Create new product
            $product = new Product();
            
            // For new products, sales count will be 0
            $sapData['product_custom_field4'] = 0;
            Log::info("Creating new product with SKU: {$itemCode}, initial sales count: 0");
            
            // Set SAP fields
            foreach ($sapData as $key => $value) {
                $product->{$key} = $value;
            }
            
            // Set local fields from SAP data
            $product->name = $sapItem['ItemName'] ?? $itemCode;
            $product->sku = $itemCode;
            
            // Handle barcode separately if available
            if (!empty($sapItem['ItemBarCodeCollection']) && is_array($sapItem['ItemBarCodeCollection'])) {
                $firstBarcode = $sapItem['ItemBarCodeCollection'][0] ?? null;
                if ($firstBarcode && !empty($firstBarcode['Barcode'])) {
                    // If you have a barcode field, set it here
                    // $product->barcode = $firstBarcode['Barcode'];
                }
            }
            
            $product->type = $this->mapSapTypeToLocal($sapItem['ItemType'] ?? 'itItems');
            $product->quantity = $totalOnHand;
            $product->enable_stock = 1;
            $product->is_inactive = $sapItem['Frozen'] === 'tYES' ? 1 : 0;
            $product->not_for_selling = $sapItem['SalesItem'] === 'tNO' ? 1 : 0;
            
            // Set price fields in products table
            if (!is_null($sellPrice) && $sellPrice > 0) {
                $product->sale_price = $sellPrice;
                $product->default_sell_price = $sellPrice;
            }
            
            if ($purchasePrice > 0) {
                $product->purchase_price = $purchasePrice;
                $product->default_purchase_price = $purchasePrice;
            }
            
            // Get required IDs
            $created_by = 1;
            $business_id = $this->getBusinessId();
            
            if (function_exists('creatorId')) {
                $created_by = creatorId();
            }
            
            $product->created_by = $created_by;
            $product->business_id = $business_id;
            
            // Set sales count for new product
            $product->product_custom_field4 = 0;
            
            $product->save();
            
            // Create variation with proper prices and sub_sku
            $this->createSingleProductVariation($product, $sellPrice, $purchasePrice);
            
            // Process warehouse data
            $this->processWarehouseData($product, $warehouseData);
            
            // Update variation location details
            $this->updateVariationLocationDetails($product, $warehouseData);
            
            // Process UoM data if available
            if (!empty($sapItem['ItemUnitOfMeasurementCollection'])) {
                $this->processUomData($product, $sapItem['ItemUnitOfMeasurementCollection'], $uomGroupEntry);
            }
            
            return 'created';
        }
        
    } catch (\Exception $e) {
        Log::error("Error in processSapItem for {$itemCode}: " . $e->getMessage());
        Log::error("Stack trace: " . $e->getTraceAsString());
        throw $e;
    }
}

    



protected function processUomData($product, $uomCollection, $uomGroupEntry)
{
    if (empty($uomCollection) || empty($uomGroupEntry)) {
        return;
    }
    
    try {
        // First, we need to get the UoM group details to understand the relationships
        $uomGroupDetails = $this->getUomGroupDetails($uomGroupEntry);
        $baseUomCode = $uomGroupDetails['base_uom'] ?? null;
        
        foreach ($uomCollection as $uomData) {
            $uomEntry = $uomData['UoMEntry'] ?? null;
            $uomType = $uomData['UoMType'] ?? null;
            
            if (empty($uomEntry)) {
                continue;
            }
            
            // Get the actual UoM name from SAP's UoM definition
            $uomDetails = $this->getUomDetails($uomGroupEntry, $uomEntry);
            $uomCode = $uomDetails['code'] ?? $uomEntry;
            $uomName = $uomDetails['name'] ?? $uomCode;
            $conversionRate = $uomDetails['conversion_rate'] ?? 1;
            
            // Create or update UoM mapping with full details
            $mapping = SapUomMapping::updateOrCreate(
                [
                    'uom_group_entry' => $uomGroupEntry,
                    'uom_code' => $uomCode
                ],
                [
                    'uom_name' => $uomName,
                    'uom_entry' => $uomEntry,
                    'uom_type' => $uomType,
                    'is_base_uom' => ($uomCode == $baseUomCode),
                    'conversion_rate' => $conversionRate,
                    'unit_id' => null // Will be mapped later
                ]
            );
            
            // Try to auto-map common units
            $this->attemptAutoMapping($mapping);
        }
        
        // Update product's default UoM reference
        if ($product->sap_inventory_uom && !$product->unit_id) {
            $this->mapProductDefaultUnit($product);
        }
        
    } catch (\Exception $e) {
        Log::error("Error processing UoM data for product {$product->id}: " . $e->getMessage());
    }
}


protected function getUomGroupDetails($uomGroupEntry)
{
    try {
        $response = $this->get("UnitOfMeasurementGroups({$uomGroupEntry})", [], 30);
        
        if ($response) {
            return [
                'code' => $response['Code'] ?? null,
                'name' => $response['Name'] ?? null,
                'base_uom' => $response['BaseUoM'] ?? null
            ];
        }
    } catch (\Exception $e) {
        Log::debug("Could not get UoM group details: " . $e->getMessage());
    }
    
    return [];
}

/**
 * Get specific UoM details from SAP
 */
protected function getUomDetails($uomGroupEntry, $uomEntry)
{
    try {
        // Try to get from the UoM definitions
        $response = $this->get("UnitOfMeasurementGroups({$uomGroupEntry})/UoMGroupDefinitionCollection", [], 30);
        
        if (isset($response['value'])) {
            foreach ($response['value'] as $uomDef) {
                if ($uomDef['AlternateUoM'] == $uomEntry || $uomDef['UoMEntry'] == $uomEntry) {
                    return [
                        'code' => $uomDef['AlternateUoM'] ?? $uomEntry,
                        'name' => $uomDef['AlternateUoM'] ?? $uomEntry,
                        'conversion_rate' => ($uomDef['BaseQuantity'] ?? 1) / ($uomDef['AlternateQuantity'] ?? 1)
                    ];
                }
            }
        }
    } catch (\Exception $e) {
        Log::debug("Could not get UoM details: " . $e->getMessage());
    }
    
    // Fallback to common UoM names
    $commonUoms = [
        '6' => ['code' => 'KG', 'name' => 'كغم'],
        '7' => ['code' => '50ML', 'name' => '50 مل'],
        '8' => ['code' => '75ML', 'name' => '75 مل'],
        '9' => ['code' => '100ML', 'name' => '100 مل'],
        '10' => ['code' => '500ML', 'name' => '500 مل'],
        '11' => ['code' => '250ML', 'name' => '250 مل'],
    ];
    
    return $commonUoms[$uomEntry] ?? ['code' => $uomEntry, 'name' => $uomEntry, 'conversion_rate' => 1];
}

/**
 * Attempt to automatically map common units
 */
protected function attemptAutoMapping($mapping)
{
    // Common unit mappings (SAP => Local)
    $autoMappings = [
        'KG' => ['كغم', 'كيلو', 'kg', 'kilogram'],
        'كغم' => ['كغم', 'كيلو', 'kg', 'kilogram'],
        'G' => ['غم', 'جرام', 'g', 'gram'],
        'غم' => ['غم', 'جرام', 'g', 'gram'],
        'L' => ['لتر', 'l', 'liter', 'litre'],
        'لتر' => ['لتر', 'l', 'liter', 'litre'],
        'ML' => ['مل', 'ml', 'milliliter'],
        'مل' => ['مل', 'ml', 'milliliter'],
        'EA' => ['قطعة', 'حبة', 'ea', 'each', 'piece'],
        'BOX' => ['صندوق', 'علبة', 'box', 'carton'],
        'PACK' => ['باكيت', 'رزمة', 'pack', 'package'],
    ];
    
    $sapCode = strtoupper($mapping->uom_code);
    
    if (isset($autoMappings[$sapCode])) {
        $localUnits = Unit::whereNull('deleted_at')->get();
        
        foreach ($localUnits as $unit) {
            foreach ($autoMappings[$sapCode] as $variant) {
                if (stripos($unit->short_name, $variant) !== false || 
                    stripos($unit->actual_name, $variant) !== false) {
                    
                    $mapping->unit_id = $unit->id;
                    $mapping->save();
                    
                    Log::info("Auto-mapped SAP UoM '{$mapping->uom_code}' to local unit '{$unit->short_name}'");
                    return;
                }
            }
        }
    }
}

/**
 * Map product's default unit based on SAP inventory UoM
 */
protected function mapProductDefaultUnit($product)
{
    if (!$product->sap_inventory_uom) {
        return;
    }
    
    // Find the mapping for this UoM
    $mapping = SapUomMapping::where('uom_code', $product->sap_inventory_uom)
        ->whereNotNull('unit_id')
        ->first();
    
    if ($mapping && $mapping->unit_id) {
        $product->unit_id = $mapping->unit_id;
        $product->save();
        
        Log::info("Mapped product {$product->id} to unit {$mapping->unit_id} based on SAP UoM");
    }
}
    
    /**
     * Update product variation prices
     */
  protected function updateProductVariationPrices($product, $sellPrice, $purchasePrice)
{
    $variations = $product->variations;
    
    foreach ($variations as $variation) {
        // FORCE UPDATE prices regardless of current values
        if (!is_null($sellPrice)) {
            $variation->default_sell_price = $sellPrice;
            $variation->sell_price_inc_tax = $sellPrice;
        }
        
        if (!is_null($purchasePrice)) {
            $variation->default_purchase_price = $purchasePrice;
            $variation->dpp_inc_tax = $purchasePrice;
        }
        
        $variation->save();
    }
}
    
    /**
     * Create a single product variation for a product with proper prices
     */
    protected function createSingleProductVariation($product, $sellPrice, $purchasePrice = 0)
{
    try {
        // Create product variation
        $productVariation = new ProductVariation();
        $productVariation->product_id = $product->id;
        $productVariation->name = 'DUMMY';
        $productVariation->is_dummy = 1;
        $productVariation->save();
        
        // Create variation with proper prices
        $variation = new Variation();
        $variation->product_id = $product->id;
        $variation->product_variation_id = $productVariation->id;
        $variation->name = 'DUMMY';
        
        // IMPORTANT: Use SAP item code as sub_sku
        $variation->sub_sku = $product->sap_item_code ?? $product->sku;
        
        $variation->default_purchase_price = $purchasePrice ?? 0;
        $variation->dpp_inc_tax = $purchasePrice ?? 0;
        $variation->profit_percent = 0;
        $variation->default_sell_price = $sellPrice ?? 0;
        $variation->sell_price_inc_tax = $sellPrice ?? 0;
        $variation->save();
        
        return $variation;
    } catch (\Exception $e) {
        Log::error("Error creating variation for product {$product->id}: " . $e->getMessage());
        throw $e;
    }
}
    
    /**
     * Process warehouse data for a product
     */

protected function processWarehouseData($product, $warehouseData)
{
    if (empty($product->sap_item_code) || empty($warehouseData)) {
        return;
    }
    
    // Track processed warehouses
    $processedWarehouses = [];
    
    foreach ($warehouseData as $warehouse) {
        $warehouseCode = $warehouse['warehouse_code'];
        
        if (empty($warehouseCode)) {
            continue;
        }
        
        try {
            // Save detailed warehouse info with proper quantities
            SapProductWarehouse::updateOrCreate(
                [
                    'sap_item_code' => $product->sap_item_code,
                    'warehouse_code' => $warehouseCode
                ],
                [
                    'in_stock' => $warehouse['in_stock'],
                    'committed' => $warehouse['committed'],
                    'ordered' => $warehouse['ordered'],
                    'last_sync' => now()
                ]
            );
            
            $processedWarehouses[] = $warehouseCode;
            
            // Ensure warehouse mapping exists
            $this->ensureWarehouseMapping($warehouseCode, $warehouse);
        } catch (\Exception $e) {
            Log::error("Error processing warehouse data for {$product->sap_item_code}, warehouse {$warehouseCode}: " . $e->getMessage());
        }
    }
    
    // Remove old warehouse entries that are no longer in SAP
    if (!empty($processedWarehouses)) {
        SapProductWarehouse::where('sap_item_code', $product->sap_item_code)
            ->whereNotIn('warehouse_code', $processedWarehouses)
            ->delete();
    }
}
    /**
     * Ensure warehouse mapping exists
     */
    protected function ensureWarehouseMapping($warehouseCode, $warehouseInfo = null)
    {
        try {
            return SapWarehouseMapping::firstOrCreate(
                ['warehouse_code' => $warehouseCode],
                [
                    'warehouse_name' => $warehouseInfo['warehouse_name'] ?? $warehouseCode,
                    'location_id' => null // Will be mapped manually later
                ]
            );
        } catch (\Exception $e) {
            Log::error("Error creating warehouse mapping for {$warehouseCode}: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Sync all warehouses from SAP
     */
    public function syncWarehouses()
    {
        try {
            $response = $this->get('Warehouses', [
                '$select' => 'WarehouseCode,WarehouseName'
            ], 120);
                
            if ($response && isset($response['value'])) {
                $warehouses = $response['value'];
                $count = 0;
                
                foreach ($warehouses as $warehouse) {
                    $warehouseCode = $warehouse['WarehouseCode'] ?? null;
                    $warehouseName = $warehouse['WarehouseName'] ?? $warehouseCode;
                    
                    if (empty($warehouseCode)) {
                        continue;
                    }
                    
                    try {
                        SapWarehouseMapping::updateOrCreate(
                            ['warehouse_code' => $warehouseCode],
                            ['warehouse_name' => $warehouseName]
                        );
                        $count++;
                    } catch (\Exception $e) {
                        Log::error("Error syncing warehouse {$warehouseCode}: " . $e->getMessage());
                    }
                }
                
                Log::info("Synced {$count} warehouses");
                return $count;
            }
            
            return 0;
        } catch (\Exception $e) {
            Log::error("SAP Sync Warehouses Exception: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Sync all UoM Groups from SAP
     */
    public function syncUomGroups() 
    {
        try {
            $response = $this->get('UnitOfMeasurementGroups', [], 120);
            
            if ($response && isset($response['value'])) {
                $uomGroups = $response['value'];
                $count = 0;
                
                foreach ($uomGroups as $uomGroup) {
                    $uomGroupEntry = $uomGroup['AbsEntry'] ?? null;
                    $uomGroupCode = $uomGroup['Code'] ?? null;
                    $uomGroupName = $uomGroup['Name'] ?? $uomGroupCode;
                    $baseUoM = $uomGroup['BaseUoM'] ?? null;
                    
                    if (empty($uomGroupEntry)) {
                        continue;
                    }
                    
                    // Create mapping for the base UoM
                    if (!empty($baseUoM)) {
                        try {
                            SapUomMapping::updateOrCreate(
                                [
                                    'uom_group_entry' => $uomGroupEntry,
                                    'uom_code' => $baseUoM
                                ],
                                [
                                    'uom_name' => $uomGroupName . ' (Base)',
                                    'unit_id' => null // Will be mapped manually later
                                ]
                            );
                            $count++;
                        } catch (\Exception $e) {
                            Log::error("Error syncing base UoM {$baseUoM}: " . $e->getMessage());
                        }
                    }
                    
                    // Try to get UoM definitions for this group
                    try {
                        // Method 1: Try expand parameter
                        $groupWithDetails = $this->get("UnitOfMeasurementGroups({$uomGroupEntry})?$expand=UoMGroupDefinitionCollection", [], 60);
                        
                        if (isset($groupWithDetails['UoMGroupDefinitionCollection']) && is_array($groupWithDetails['UoMGroupDefinitionCollection'])) {
                            foreach ($groupWithDetails['UoMGroupDefinitionCollection'] as $uomDef) {
                                $alternateUoM = $uomDef['AlternateUoM'] ?? null;
                                $alternateQty = $uomDef['AlternateQuantity'] ?? 1;
                                $baseQty = $uomDef['BaseQuantity'] ?? 1;
                                
                                if (!empty($alternateUoM)) {
                                    try {
                                        SapUomMapping::updateOrCreate(
                                            [
                                                'uom_group_entry' => $uomGroupEntry,
                                                'uom_code' => $alternateUoM
                                            ],
                                            [
                                                'uom_name' => $alternateUoM . " ({$alternateQty}:{$baseQty})",
                                                'unit_id' => null
                                            ]
                                        );
                                        $count++;
                                    } catch (\Exception $e) {
                                        Log::error("Error syncing UoM {$alternateUoM}: " . $e->getMessage());
                                    }
                                }
                            }
                        }
                    } catch (\Exception $e) {
                        Log::debug("Could not get UoM definitions for group {$uomGroupEntry}: " . $e->getMessage());
                        
                        // If we couldn't get definitions, at least create a mapping for the group itself
                        try {
                            SapUomMapping::updateOrCreate(
                                [
                                    'uom_group_entry' => $uomGroupEntry,
                                    'uom_code' => $uomGroupCode
                                ],
                                [
                                    'uom_name' => $uomGroupName,
                                    'unit_id' => null
                                ]
                            );
                            $count++;
                        } catch (\Exception $e) {
                            Log::error("Error syncing UoM group {$uomGroupCode}: " . $e->getMessage());
                        }
                    }
                }
                
                Log::info("Synced {$count} UoM mappings");
                return $count;
            }
            
            return 0;
        } catch (\Exception $e) {
            Log::error("SAP Sync UoM Groups Exception: " . $e->getMessage());
            return 0;
        }
    }
    /**
     * Check if an item needs to be updated
     */
    protected function needsUpdate($product, $data)
    {
        foreach ($data as $key => $value) {
            if (in_array($key, ['sap_last_sync', 'sap_sync_status', 'sap_sync_message'])) {
                continue;
            }
            
            // Convert both values to strings for comparison to handle null vs empty string issues
            $productValue = $product->{$key} ?? '';
            $dataValue = $value ?? '';
            
            if ((string)$productValue !== (string)$dataValue) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Map SAP item type to local product type
     */
    protected function mapSapTypeToLocal($sapItemType)
    {
        $mapping = [
            'itItems' => 'single',
            'itService' => 'service',
        ];
        
        return $mapping[$sapItemType] ?? 'single';
    }
    
    /**
     * Determine if an SAP item should be imported
     */
    protected function isItemImportable($sapItem)
    {
        // Only import items that are sales items, purchased items, or inventory items
        if (isset($sapItem['SalesItem']) && $sapItem['SalesItem'] === 'tNO' &&
            isset($sapItem['PurchaseItem']) && $sapItem['PurchaseItem'] === 'tNO' &&
            isset($sapItem['InventoryItem']) && $sapItem['InventoryItem'] === 'tNO') {
            return false;
        }
        
        return true;
    }
    
    /**
     * Get business ID helper
     */
    private function getBusinessId()
    {
        try {
            $firstProduct = Product::first();
            if ($firstProduct) {
                return $firstProduct->business_id;
            }
            
            $business = \App\Business::first();
            if ($business) {
                return $business->id;
            }
            
            return 1;
        } catch (\Exception $e) {
            Log::warning("Could not determine business_id: " . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Sync specific product by item code
     */
    public function syncSingleProduct($itemCode, $fullSync = true)
    {
        try {
            Log::info("Syncing single product: {$itemCode}");
            
            // Get item data from SAP
            $select = "ItemCode,ItemName,ForeignName,ItemsGroupCode,ItemType,ItemClass,InventoryItem,PurchaseItem,SalesItem,InventoryUOM,UoMGroupEntry,Manufacturer,QuantityOnStock,QuantityOrderedFromVendors,QuantityOrderedByCustomers,ValidFrom,ValidTo,Frozen,DefaultWarehouse,ManageStockByWarehouse,MovingAveragePrice,AvgStdPrice,UpdateDate,UpdateTime,CreateDate,CreateTime,MaterialType,MaterialGroup";
            
            $sapItem = $this->getItem($itemCode);
            
            if (!$sapItem) {
                throw new \Exception("Item {$itemCode} not found in SAP");
            }
            
            // Get additional data
            $warehouses = $this->getItemWarehouses($itemCode);
            $prices = $this->getItemPrices($itemCode);
            
            // Add the additional data to the item
            $sapItem['ItemWarehouseData'] = $warehouses;
            $sapItem['ItemPricesData'] = $prices;
            
            $result = $this->processSapItem($sapItem, $fullSync);
            
            Log::info("Single product sync result for {$itemCode}: {$result}");
            return $result;
            
        } catch (\Exception $e) {
            Log::error("Error syncing single product {$itemCode}: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Push local product to SAP
     */
    public function pushProductToSap($product)
    {
        try {
            Log::info("Pushing product to SAP: {$product->sku}");
            
            $itemData = [
                'ItemCode' => $product->sap_item_code ?? $product->sku,
                'ItemName' => $product->name,
                'ForeignName' => $product->sap_foreign_name ?? $product->name,
                'ItemType' => $this->mapLocalTypeToSap($product->type),
                'InventoryItem' => $product->enable_stock ? 'tYES' : 'tNO',
                'SalesItem' => $product->not_for_selling ? 'tNO' : 'tYES',
                'PurchaseItem' => 'tYES',
                'InventoryUOM' => $this->getProductUnitCode($product),
            ];
            
            // Add price if available
            if ($product->effective_selling_price > 0) {
                $itemData['DefaultPrice'] = $product->effective_selling_price;
            }
            
            $result = $this->createOrUpdateItem($product->sap_item_code ?? $product->sku, $itemData);
            
            if ($result) {
                // Update product with sync info
                $product->update([
                    'sap_item_code' => $product->sap_item_code ?? $product->sku,
                    'sap_sync_status' => 'success',
                    'sap_sync_message' => 'Pushed to SAP successfully',
                    'sap_last_sync' => now()
                ]);
            }
            
            return $result;
            
        } catch (\Exception $e) {
            // Update product with error info
            $product->update([
                'sap_sync_status' => 'failed',
                'sap_sync_message' => $e->getMessage()
            ]);
            
            Log::error("Error pushing product {$product->sku} to SAP: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Map local product type to SAP item type
     */
    protected function mapLocalTypeToSap($localType)
    {
        $mapping = [
            'single' => 'itItems',
            'service' => 'itService',
        ];
        
        return $mapping[$localType] ?? 'itItems';
    }
    
    /**
     * Get product unit code for SAP
     */
    protected function getProductUnitCode($product)
    {
        if ($product->unit && $product->unit->sapUomMapping) {
            return $product->unit->sapUomMapping->uom_code;
        }
        
        return 'EA'; // Default to Each
    }
    
    /**
     * Sync products that need sync (marked for sync or updated since last sync)
     */
    public function syncPendingProducts()
    {
        $stats = [
            'total' => 0,
            'synced' => 0,
            'failed' => 0
        ];
        
        try {
            // Get products that need sync
            $products = Product::where(function($query) {
                $query->where('sap_sync_status', 'pending')
                      ->orWhereNull('sap_sync_status')
                      ->orWhere(function($q) {
                          $q->whereNotNull('sap_last_sync')
                            ->whereColumn('updated_at', '>', 'sap_last_sync');
                      });
            })->limit(100)->get(); // Limit to avoid timeouts
            
            $stats['total'] = $products->count();
            
            foreach ($products as $product) {
                try {
                    if ($product->sap_item_code) {
                        // Sync from SAP
                        $this->syncSingleProduct($product->sap_item_code);
                    } else {
                        // Push to SAP
                        $this->pushProductToSap($product);
                    }
                    
                    $stats['synced']++;
                } catch (\Exception $e) {
                    $stats['failed']++;
                    Log::error("Failed to sync product {$product->sku}: " . $e->getMessage());
                }
            }
            
            return $stats;
            
        } catch (\Exception $e) {
            Log::error("Error in syncPendingProducts: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Get sync statistics
     */
    public function getSyncStats()
    {
        return [
            'total_products' => Product::count(),
            'synced_products' => Product::syncedWithSap()->count(),
            'not_synced_products' => Product::notSyncedWithSap()->count(),
            'failed_sync_products' => Product::withSyncErrors()->count(),
            'last_full_sync' => Product::whereNotNull('sap_last_sync')
                                     ->orderBy('sap_last_sync', 'desc')
                                     ->first()
                                     ->sap_last_sync ?? null,
            'products_needing_sync' => Product::where(function($query) {
                $query->where('sap_sync_status', 'pending')
                      ->orWhereNull('sap_sync_status')
                      ->orWhere(function($q) {
                          $q->whereNotNull('sap_last_sync')
                            ->whereColumn('updated_at', '>', 'sap_last_sync');
                      });
            })->count(),
            'total_warehouses' => SapWarehouseMapping::count(),
            'total_uom_mappings' => SapUomMapping::count(),
            'products_with_sales' => Product::where('product_custom_field4', '>', 0)->count(),
            'total_sale_transactions' => Product::sum('product_custom_field4'),
        ];
    }
    
    /**
     * Clean up old sync data
     */
    public function cleanupSyncData($daysOld = 30)
    {
        try {
            $cutoffDate = Carbon::now()->subDays($daysOld);
            
            // Clean up old warehouse data
            $deletedWarehouses = SapProductWarehouse::where('last_sync', '<', $cutoffDate)->delete();
            
            Log::info("Cleaned up {$deletedWarehouses} old warehouse records");
            
            return [
                'warehouse_records_deleted' => $deletedWarehouses
            ];
            
        } catch (\Exception $e) {
            Log::error("Error cleaning up sync data: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Validate SAP connection and permissions
     */
    public function validateSapConnection()
    {
        try {
            // Test basic connection with a simple query
            $result = $this->get('Items', [
                '$top' => 1,
                '$select' => 'ItemCode'
            ], 30);
            
            if (!$result || !isset($result['value'])) {
                throw new \Exception("Invalid response from SAP");
            }
            
            return [
                'status' => 'success',
                'message' => 'SAP connection is working',
                'test_time' => now()
            ];
            
        } catch (\Exception $e) {
            return [
                'status' => 'failed',
                'message' => $e->getMessage(),
                'test_time' => now()
            ];
        }
    }




    protected function updateVariationLocationDetails($product, $warehouseData)
{
    try {
        // Get all variations for this product
        $variations = $product->variations;
        
        if ($variations->isEmpty()) {
            Log::warning("No variations found for product {$product->id}");
            return;
        }
        
        // Get warehouse mappings to find location IDs
        $warehouseMappings = \App\SapWarehouseMapping::whereIn('warehouse_code', array_column($warehouseData, 'warehouse_code'))
            ->whereNotNull('location_id')
            ->get()
            ->keyBy('warehouse_code');
        
        foreach ($warehouseData as $warehouse) {
            $warehouseCode = $warehouse['warehouse_code'];
            $inStock = $warehouse['in_stock'] ?? 0;
            
            // Check if this warehouse is mapped to a location
            $mapping = $warehouseMappings->get($warehouseCode);
            
            if (!$mapping || !$mapping->location_id) {
                Log::debug("Warehouse {$warehouseCode} not mapped to any location, skipping variation_location_details update");
                continue;
            }
            
            $locationId = $mapping->location_id;
            
            // Update variation location details for each variation
            foreach ($variations as $variation) {
                // Find or create variation location details
                $vld = \App\VariationLocationDetails::firstOrCreate(
                    [
                        'variation_id' => $variation->id,
                        'product_id' => $product->id,
                        'location_id' => $locationId,
                        'product_variation_id' => $variation->product_variation_id
                    ],
                    [
                        'qty_available' => 0
                    ]
                );
                
                // Update quantity available based on SAP data
                // Only update if SAP has data (don't overwrite with 0 unless SAP explicitly says 0)
                if ($warehouse['in_stock'] !== null) {
                    $vld->qty_available = $inStock;
                    $vld->save();
                    
                    Log::info("Updated variation_location_details for product {$product->id}, variation {$variation->id}, location {$locationId}: qty = {$inStock}");
                }
            }
        }
        
        // Handle the case where a product has total stock but no warehouse-specific data
        if (empty($warehouseData) && $product->sap_on_hand > 0) {
            // Check if there's a default warehouse mapping
            $defaultWarehouse = $product->sap_default_warehouse;
            
            if ($defaultWarehouse) {
                $mapping = \App\SapWarehouseMapping::where('warehouse_code', $defaultWarehouse)
                    ->whereNotNull('location_id')
                    ->first();
                
                if ($mapping && $mapping->location_id) {
                    foreach ($variations as $variation) {
                        $vld = \App\VariationLocationDetails::firstOrCreate(
                            [
                                'variation_id' => $variation->id,
                                'product_id' => $product->id,
                                'location_id' => $mapping->location_id,
                                'product_variation_id' => $variation->product_variation_id
                            ],
                            [
                                'qty_available' => 0
                            ]
                        );
                        
                        $vld->qty_available = $product->sap_on_hand;
                        $vld->save();
                        
                        Log::info("Updated variation_location_details from total stock for product {$product->id}, location {$mapping->location_id}: qty = {$product->sap_on_hand}");
                    }
                }
            }
        }
        
    } catch (\Exception $e) {
        Log::error("Error updating variation location details for product {$product->id}: " . $e->getMessage());
        Log::error("Stack trace: " . $e->getTraceAsString());
    }
}


protected function getDefaultLocationId()
{
    // First, check if there's a configured default location for SAP sync
    $defaultLocation = BusinessLocation::where('is_default', 1)->first();
    
    if ($defaultLocation) {
        return $defaultLocation->id;
    }
    
    // Otherwise, get the first active location
    $firstLocation = \App\BusinessLocation::where('is_active', 1)->first();
    
    if ($firstLocation) {
        return $firstLocation->id;
    }
    
    // As a last resort, get any location
    $anyLocation = \App\BusinessLocation::first();
    
    return $anyLocation ? $anyLocation->id : null;
}

// أضف أيضاً دالة لمزامنة المخزون لمنتج واحد:

/**
 * Sync stock levels from SAP to variation_location_details for a single product
 */
public function syncProductStock($productId)
{
    try {
        $product = Product::find($productId);
        
        if (!$product || !$product->sap_item_code) {
            throw new \Exception("Product not found or not synced with SAP");
        }
        
        // Get warehouse data from SAP
        $warehouseData = \App\SapProductWarehouse::where('sap_item_code', $product->sap_item_code)
            ->get()
            ->map(function($warehouse) {
                return [
                    'warehouse_code' => $warehouse->warehouse_code,
                    'in_stock' => $warehouse->in_stock,
                    'committed' => $warehouse->committed,
                    'ordered' => $warehouse->ordered
                ];
            })
            ->toArray();
        
        // Update variation location details
        $this->updateVariationLocationDetails($product, $warehouseData);
        
        return [
            'success' => true,
            'message' => 'Stock synced successfully',
            'warehouses_updated' => count($warehouseData)
        ];
        
    } catch (\Exception $e) {
        Log::error("Error syncing stock for product {$productId}: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => $e->getMessage()
        ];
    }
}


// أضف دالة لمزامنة جميع المخزونات:

/**
 * Sync all product stocks from SAP warehouse data to variation_location_details
 */
public function syncAllProductStocks()
{
    $stats = [
        'total' => 0,
        'synced' => 0,
        'failed' => 0
    ];
    
    try {
        // Get all products with SAP item codes
        $products = Product::whereNotNull('sap_item_code')
            ->where('enable_stock', 1)
            ->get();
        
        $stats['total'] = $products->count();
        
        foreach ($products as $product) {
            try {
                $result = $this->syncProductStock($product->id);
                
                if ($result['success']) {
                    $stats['synced']++;
                } else {
                    $stats['failed']++;
                }
            } catch (\Exception $e) {
                $stats['failed']++;
                Log::error("Failed to sync stock for product {$product->id}: " . $e->getMessage());
            }
        }
        
        Log::info("Stock sync completed. Stats: " . json_encode($stats));
        return $stats;
        
    } catch (\Exception $e) {
        Log::error("Error in syncAllProductStocks: " . $e->getMessage());
        throw $e;
    }
}

}