<?php

namespace App;

use Illuminate\Database\Eloquent\Model;

class TransactionSellLine extends Model
{
    /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $fillable = [
        'transaction_id', 'product_id', 'variation_id', 'quantity', 'unit_price', 
        'unit_price_inc_tax', 'item_tax', 'tax_id', 'unit_price_before_discount',
        'line_discount_type', 'line_discount_amount', 'sell_line_note', 'sub_unit_id',
        'res_service_staff_id', 'secondary_unit_quantity', 'warehouse_code', 'warehouse_id',
        'parent_sell_line_id', 'children_type', 'so_line_id', 'lot_no_line_id'
    ];

    public function transaction()
    {
        return $this->belongsTo(\App\Transaction::class);
    }

    public function product()
    {
        return $this->belongsTo(\App\Product::class, 'product_id');
    }

    public function variations()
    {
        return $this->belongsTo(\App\Variation::class, 'variation_id');
    }

    public function modifiers()
    {
        return $this->hasMany(\App\TransactionSellLine::class, 'parent_sell_line_id')
            ->where('children_type', 'modifier');
    }

    public function sell_line_purchase_lines()
    {
        return $this->hasMany(\App\TransactionSellLinesPurchaseLines::class, 'sell_line_id');
    }

    /**
     * Get the quantity column.
     *
     * @param  string  $value
     * @return float $value
     */
    public function getQuantityAttribute($value)
    {
        return (float) $value;
    }

    public function lot_details()
    {
        return $this->belongsTo(\App\PurchaseLine::class, 'lot_no_line_id');
    }

    public function get_discount_amount()
    {
        $discount_amount = 0;
        if (! empty($this->line_discount_type) && ! empty($this->line_discount_amount)) {
            if ($this->line_discount_type == 'fixed') {
                $discount_amount = $this->line_discount_amount;
            } elseif ($this->line_discount_type == 'percentage') {
                $discount_amount = ($this->unit_price_before_discount * $this->line_discount_amount) / 100;
            }
        }

        return $discount_amount;
    }

    /**
     * Get the unit associated with the purchase line.
     */
    public function sub_unit()
    {
        return $this->belongsTo(\App\Unit::class, 'sub_unit_id');
    }

    public function order_statuses()
    {
        $statuses = [
            'received',
            'cooked',
            'served',
        ];
    }

    public function service_staff()
    {
        return $this->belongsTo(\App\User::class, 'res_service_staff_id');
    }

    /**
     * The warranties that belong to the sell lines.
     */
    public function warranties()
    {
        return $this->belongsToMany(\App\Warranty::class, 'sell_line_warranties', 'sell_line_id', 'warranty_id');
    }

    public function line_tax()
    {
        return $this->belongsTo(\App\TaxRate::class, 'tax_id');
    }

    public function so_line()
    {
        return $this->belongsTo(\App\TransactionSellLine::class, 'so_line_id');
    }

    /**
     * Get the warehouse code for this sell line
     *
     * @return string
     */
    public function getWarehouseCodeAttribute($value)
    {
        return $value ?: 'W01'; // Default warehouse if not set
    }

    /**
     * Set the warehouse code ensuring proper format
     *
     * @param string $value
     */
    public function setWarehouseCodeAttribute($value)
    {
        // Ensure warehouse code is in proper format (W01, W02, etc.)
        if (!empty($value)) {
            if (is_numeric($value)) {
                $value = 'W' . str_pad($value, 2, '0', STR_PAD_LEFT);
            }
            $this->attributes['warehouse_code'] = strtoupper($value);
        }
    }
}
