/**
 * POS API Module
 * Handles all AJAX requests, server communication, and data synchronization
 */

// API configuration
var APIConfig = {
    baseURL: '',
    timeout: 30000,
    retryAttempts: 3,
    retryDelay: 1000,
    cacheTimeout: 5 * 60 * 1000 // 5 minutes
};

// Request cache
var requestCache = {};
var pendingRequests = {};

// Initialize API module
$(document).ready(function() {
    setupAPIConfiguration();
    setupRequestInterceptors();
    setupResponseHandlers();
});

function setupAPIConfiguration() {
    // Set base URL from meta tag or default
    var baseURLMeta = $('meta[name="base-url"]').attr('content');
    if (baseURLMeta) {
        APIConfig.baseURL = baseURLMeta;
    }
    
    // Setup global AJAX defaults
    $.ajaxSetup({
        timeout: APIConfig.timeout,
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        }
    });
}

function setupRequestInterceptors() {
    // Add request interceptor for loading states
    $(document).ajaxSend(function(event, jqxhr, settings) {
        var requestId = generateRequestId(settings);
        pendingRequests[requestId] = true;
        
        // Show loading state for specific elements
        if (settings.showLoading) {
            POSUI.showLoading(settings.loadingElement, settings.loadingText);
        }
        
        // Add request to cache if needed
        if (settings.cache && settings.method === 'GET') {
            var cacheKey = generateCacheKey(settings);
            if (requestCache[cacheKey] && !isCacheExpired(cacheKey)) {
                // Return cached response
                jqxhr.abort();
                return;
            }
        }
    });
}

function setupResponseHandlers() {
    // Handle successful responses
    $(document).ajaxSuccess(function(event, jqxhr, settings, data) {
        var requestId = generateRequestId(settings);
        delete pendingRequests[requestId];
        
        // Hide loading state
        if (settings.showLoading) {
            POSUI.hideLoading(settings.loadingElement);
        }
        
        // Cache successful GET requests
        if (settings.cache && settings.method === 'GET' && data) {
            var cacheKey = generateCacheKey(settings);
            cacheResponse(cacheKey, data);
        }
        
        // Handle specific response types
        handleResponseSuccess(data, settings);
    });
    
    // Handle failed responses
    $(document).ajaxError(function(event, jqxhr, settings, error) {
        var requestId = generateRequestId(settings);
        delete pendingRequests[requestId];
        
        // Hide loading state
        if (settings.showLoading) {
            POSUI.hideLoading(settings.loadingElement);
        }
        
        // Handle specific error types
        handleResponseError(jqxhr, settings, error);
    });
}

// Product API functions
function searchProducts(query, options) {
    var params = {
        term: query,
        location_id: $('input#location_id').val(),
        not_for_selling: 0,
        include_warehouse_details: 1
    };
    
    // Add search fields
    var searchFields = [];
    $('.search_fields:checked').each(function(i) {
        searchFields[i] = $(this).val();
    });
    if (searchFields.length > 0) {
        params.search_fields = searchFields;
    }
    
    // Add price group
    if ($('#price_group').length > 0) {
        params.price_group = $('#price_group').val();
    }
    
    return makeRequest('/products/list', 'GET', params, {
        cache: true,
        cacheTimeout: 2 * 60 * 1000 // 2 minutes for product search
    });
}

function getProductDetails(productId) {
    return makeRequest('/products/get-product-details', 'GET', {
        product_id: productId,
        location_id: $('input#location_id').val()
    }, {
        cache: true,
        cacheTimeout: 10 * 60 * 1000 // 10 minutes for product details
    });
}

function getProductVariations(productId) {
    return makeRequest('/products/get-variations', 'GET', {
        product_id: productId
    }, {
        cache: true,
        cacheTimeout: 5 * 60 * 1000
    });
}

// Customer API functions
function searchCustomers(query) {
    return makeRequest('/contacts/customers', 'GET', {
        term: query,
        page: 1,
        limit: 20
    }, {
        cache: true,
        cacheTimeout: 5 * 60 * 1000
    });
}

function getCustomerDetails(customerId) {
    return makeRequest('/contacts/get', 'GET', {
        id: customerId
    }, {
        cache: true,
        cacheTimeout: 10 * 60 * 1000
    });
}

function getCustomerRewardPoints(customerId) {
    return makeRequest('/contacts/get-reward-points', 'GET', {
        contact_id: customerId
    });
}

// Stock API functions
function checkStockAvailability(productId, quantity, warehouseId) {
    return makeRequest('/products/check-stock', 'POST', {
        product_id: productId,
        quantity: quantity,
        warehouse_id: warehouseId
    });
}

function updateStockAfterSale(transactionId) {
    return makeRequest('/pos/update-stock-after-sale', 'POST', {
        transaction_id: transactionId
    });
}

function getWarehouseStock(productId, warehouseId) {
    return makeRequest('/products/warehouse-stock', 'GET', {
        product_id: productId,
        warehouse_id: warehouseId
    }, {
        cache: true,
        cacheTimeout: 2 * 60 * 1000
    });
}

// Sales API functions
function saveSale(saleData) {
    return makeRequest('/sells/store', 'POST', saleData, {
        showLoading: true,
        loadingElement: '#pos-finalize',
        loadingText: 'Saving Sale...'
    });
}

function saveDraft(saleData) {
    return makeRequest('/sells/store', 'POST', saleData, {
        showLoading: true,
        loadingElement: '#pos-draft',
        loadingText: 'Saving Draft...'
    });
}

function saveQuotation(saleData) {
    return makeRequest('/sells/store', 'POST', saleData, {
        showLoading: true,
        loadingElement: '#pos-quotation',
        loadingText: 'Saving Quotation...'
    });
}

function getSalesOrders(customerId) {
    return makeRequest('/sells/get-sales-orders', 'GET', {
        customer_id: customerId
    });
}

// Payment API functions
function getPaymentMethods() {
    return makeRequest('/payment-methods/list', 'GET', {}, {
        cache: true,
        cacheTimeout: 30 * 60 * 1000 // 30 minutes
    });
}

function processPayment(paymentData) {
    return makeRequest('/payments/process', 'POST', paymentData, {
        showLoading: true,
        loadingElement: '.payment-process-btn',
        loadingText: 'Processing Payment...'
    });
}

// Category and Brand API functions
function getCategories() {
    return makeRequest('/categories/list', 'GET', {}, {
        cache: true,
        cacheTimeout: 30 * 60 * 1000
    });
}

function getBrands() {
    return makeRequest('/brands/list', 'GET', {}, {
        cache: true,
        cacheTimeout: 30 * 60 * 1000
    });
}

function getProductsByCategory(categoryId) {
    return makeRequest('/products/by-category', 'GET', {
        category_id: categoryId,
        location_id: $('input#location_id').val()
    }, {
        cache: true,
        cacheTimeout: 5 * 60 * 1000
    });
}

function getProductsByBrand(brandId) {
    return makeRequest('/products/by-brand', 'GET', {
        brand_id: brandId,
        location_id: $('input#location_id').val()
    }, {
        cache: true,
        cacheTimeout: 5 * 60 * 1000
    });
}

// Utility API functions
function getLocationSettings() {
    return makeRequest('/business/get-location-settings', 'GET', {
        location_id: $('input#location_id').val()
    }, {
        cache: true,
        cacheTimeout: 60 * 60 * 1000 // 1 hour
    });
}

function getTaxRates() {
    return makeRequest('/tax-rates/list', 'GET', {}, {
        cache: true,
        cacheTimeout: 60 * 60 * 1000
    });
}

function getDiscountTypes() {
    return makeRequest('/discounts/types', 'GET', {}, {
        cache: true,
        cacheTimeout: 60 * 60 * 1000
    });
}

// Core request function
function makeRequest(url, method, data, options) {
    options = options || {};
    
    var requestConfig = {
        url: APIConfig.baseURL + url,
        method: method,
        data: data,
        dataType: 'json',
        timeout: options.timeout || APIConfig.timeout,
        showLoading: options.showLoading || false,
        loadingElement: options.loadingElement,
        loadingText: options.loadingText,
        cache: options.cache || false,
        cacheTimeout: options.cacheTimeout || APIConfig.cacheTimeout,
        retryAttempts: options.retryAttempts || APIConfig.retryAttempts
    };
    
    // Check cache first for GET requests
    if (requestConfig.cache && method === 'GET') {
        var cacheKey = generateCacheKey(requestConfig);
        var cachedResponse = getCachedResponse(cacheKey);
        if (cachedResponse) {
            return Promise.resolve(cachedResponse);
        }
    }
    
    return new Promise(function(resolve, reject) {
        var attempt = 0;
        
        function executeRequest() {
            attempt++;
            
            $.ajax(requestConfig)
                .done(function(response) {
                    resolve(response);
                })
                .fail(function(jqxhr, textStatus, errorThrown) {
                    // Retry logic
                    if (attempt < requestConfig.retryAttempts && shouldRetry(jqxhr.status)) {
                        setTimeout(executeRequest, APIConfig.retryDelay * attempt);
                    } else {
                        reject({
                            status: jqxhr.status,
                            statusText: jqxhr.statusText,
                            responseText: jqxhr.responseText,
                            error: errorThrown
                        });
                    }
                });
        }
        
        executeRequest();
    });
}

// Cache management functions
function generateCacheKey(config) {
    return config.method + ':' + config.url + ':' + JSON.stringify(config.data || {});
}

function cacheResponse(key, data) {
    requestCache[key] = {
        data: data,
        timestamp: Date.now(),
        timeout: APIConfig.cacheTimeout
    };
}

function getCachedResponse(key) {
    var cached = requestCache[key];
    if (cached && !isCacheExpired(key)) {
        return cached.data;
    }
    return null;
}

function isCacheExpired(key) {
    var cached = requestCache[key];
    if (!cached) return true;
    
    return (Date.now() - cached.timestamp) > cached.timeout;
}

function clearCache(pattern) {
    if (pattern) {
        Object.keys(requestCache).forEach(function(key) {
            if (key.includes(pattern)) {
                delete requestCache[key];
            }
        });
    } else {
        requestCache = {};
    }
}

// Utility functions
function generateRequestId(config) {
    return config.method + ':' + config.url + ':' + Date.now();
}

function shouldRetry(statusCode) {
    // Retry on server errors (5xx) and network errors
    return statusCode >= 500 || statusCode === 0;
}

function handleResponseSuccess(data, settings) {
    // Handle specific response patterns
    if (data && data.success === false) {
        POSUI.showNotification(data.msg || 'Operation failed', 'error');
    } else if (data && data.success === true) {
        // Show success message if provided
        if (data.msg) {
            POSUI.showNotification(data.msg, 'success');
        }
    }
}

function handleResponseError(jqxhr, settings, error) {
    var errorMessage = 'Request failed';
    
    if (jqxhr.status === 401) {
        errorMessage = 'Session expired. Please refresh the page.';
        // Redirect to login if needed
        setTimeout(function() {
            window.location.reload();
        }, 2000);
    } else if (jqxhr.status === 403) {
        errorMessage = 'Access denied. You don\'t have permission for this action.';
    } else if (jqxhr.status === 404) {
        errorMessage = 'Resource not found.';
    } else if (jqxhr.status === 422) {
        errorMessage = 'Validation failed. Please check your input.';
    } else if (jqxhr.status === 500) {
        errorMessage = 'Server error. Please try again later.';
    } else if (jqxhr.status === 0) {
        errorMessage = 'Network error. Please check your connection.';
    }
    
    POSUI.showNotification(errorMessage, 'error');
}

// Export API module
window.POSAPI = {
    // Configuration
    config: APIConfig,
    
    // Product functions
    searchProducts: searchProducts,
    getProductDetails: getProductDetails,
    getProductVariations: getProductVariations,
    
    // Customer functions
    searchCustomers: searchCustomers,
    getCustomerDetails: getCustomerDetails,
    getCustomerRewardPoints: getCustomerRewardPoints,
    
    // Stock functions
    checkStockAvailability: checkStockAvailability,
    updateStockAfterSale: updateStockAfterSale,
    getWarehouseStock: getWarehouseStock,
    
    // Sales functions
    saveSale: saveSale,
    saveDraft: saveDraft,
    saveQuotation: saveQuotation,
    getSalesOrders: getSalesOrders,
    
    // Payment functions
    getPaymentMethods: getPaymentMethods,
    processPayment: processPayment,
    
    // Category and Brand functions
    getCategories: getCategories,
    getBrands: getBrands,
    getProductsByCategory: getProductsByCategory,
    getProductsByBrand: getProductsByBrand,
    
    // Utility functions
    getLocationSettings: getLocationSettings,
    getTaxRates: getTaxRates,
    getDiscountTypes: getDiscountTypes,
    
    // Cache management
    clearCache: clearCache,
    getCachedResponse: getCachedResponse,
    
    // Core request function
    makeRequest: makeRequest
}; 