/**
 * POS Calculations Module
 * Handles all price, total, and calculation logic
 */

// Debounced total calculation to prevent excessive calls
function pos_total_row() {
    clearTimeout(totalCalculationTimeout);
    totalCalculationTimeout = setTimeout(function() {
        _calculateTotalRow();
    }, 100);
}

function _calculateTotalRow() {
    var total_quantity = 0;
    var price_total = get_subtotal();
    
    // Cache DOM elements for better performance
    var $posTable = $('table#pos_table tbody');
    var $shippingCharges = $('span#shipping_charges_amount');
    var $totalQuantity = $('span.total_quantity');
    var $priceTotal = $('span.price_total');
    
    $posTable.find('tr').each(function() {
        total_quantity = total_quantity + __read_number($(this).find('input.pos_quantity'));
    });

    // Update shipping charges
    $shippingCharges.text(
        __currency_trans_from_en(__read_number($('input#shipping_charges_modal')), false)
    );

    $totalQuantity.each(function() {
        $(this).html(__number_f(total_quantity));
    });

    $priceTotal.html(__currency_trans_from_en(price_total, false));
    calculate_billing_details(price_total);

    if (
        $('input[name="is_serial_no"]').length > 0 &&
        $('input[name="is_serial_no"]').val() == 1
    ) {
        update_serial_no();
    }
}

function get_subtotal() {
    var price_total = 0;

    $('table#pos_table tbody tr').each(function() {
        price_total = price_total + __read_number($(this).find('input.pos_line_total'));
    });

    // Go through the modifier prices
    $('input.modifiers_price').each(function() {
        var modifier_price = __read_number($(this));
        var modifier_quantity = $(this).closest('.product_modifier').find('.modifiers_quantity').val();
        var modifier_subtotal = modifier_price * modifier_quantity;
        price_total = price_total + modifier_subtotal;
    });

    return price_total;
}

function calculate_billing_details(price_total) {
    var discount = pos_discount(price_total);
    if ($('#reward_point_enabled').length) {
        total_customer_reward = $('#rp_redeemed_amount').val();
        discount = parseFloat(discount) + parseFloat(total_customer_reward);

        if ($('input[name="is_direct_sale"]').length <= 0) {
            $('span#total_discount').text(__currency_trans_from_en(discount, false));
        }
    }

    var order_tax = pos_order_tax(price_total, discount);

    // Add shipping charges
    var shipping_charges = __read_number($('input#shipping_charges'));

    var additional_expense = 0;
    // Calculate additional expenses
    if ($('input#additional_expense_value_1').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_1'));
    }
    if ($('input#additional_expense_value_2').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_2'))
    }
    if ($('input#additional_expense_value_3').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_3'))
    }
    if ($('input#additional_expense_value_4').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_4'))
    }

    // Add packaging charge
    var packing_charge = 0;
    if ($('#types_of_service_id').length > 0 && $('#types_of_service_id').val()) {
        packing_charge = __calculate_amount($('#packing_charge_type').val(), 
            __read_number($('input#packing_charge')), price_total);

        $('#packing_charge_text').text(__currency_trans_from_en(packing_charge, false));
    }

    var total_payable = price_total + order_tax - discount + shipping_charges + packing_charge + additional_expense;

    var rounding_multiple = $('#amount_rounding_method').val() ? parseFloat($('#amount_rounding_method').val()) : 0;
    var round_off_data = __round(total_payable, rounding_multiple);
    var total_payable_rounded = round_off_data.number;

    var round_off_amount = round_off_data.diff;
    if (round_off_amount != 0) {
        $('span#round_off_text').text(__currency_trans_from_en(round_off_amount, false));
    } else {
        $('span#round_off_text').text(0);
    }
    $('input#round_off_amount').val(round_off_amount);

    __write_number($('input#final_total_input'), total_payable_rounded);
    var curr_exchange_rate = 1;
    if ($('#exchange_rate').length > 0 && $('#exchange_rate').val()) {
        curr_exchange_rate = __read_number($('#exchange_rate'));
    }
    var shown_total = total_payable_rounded * curr_exchange_rate;
    $('span#total_payable').text(__currency_trans_from_en(shown_total, false));

    $('span.total_payable_span').text(__currency_trans_from_en(total_payable_rounded, true));

    // Check if edit form then don't update price
    if ($('form#edit_pos_sell_form').length == 0 && $('form#edit_sell_form').length == 0) {
        __write_number($('.payment-amount').first(), total_payable_rounded);
    }

    $(document).trigger('invoice_total_calculated');

    calculate_balance_due();
}

function pos_discount(total_amount) {
    var calculation_type = $('#discount_type').val();
    var calculation_amount = __read_number($('#discount_amount'));

    var discount = __calculate_amount(calculation_type, calculation_amount, total_amount);

    $('span#total_discount').text(__currency_trans_from_en(discount, false));

    return discount;
}

function pos_order_tax(price_total, discount) {
    var tax_rate_id = $('#tax_rate_id').val();
    var calculation_type = 'percentage';
    var calculation_amount = __read_number($('#tax_calculation_amount'));
    var total_amount = price_total - discount;

    if (tax_rate_id) {
        var order_tax = __calculate_amount(calculation_type, calculation_amount, total_amount);
    } else {
        var order_tax = 0;
    }

    $('span#order_tax').text(__currency_trans_from_en(order_tax, false));

    return order_tax;
}

function calculate_balance_due() {
    var total_payable = __read_number($('#final_total_input'));
    var total_paying = 0;
    $('#payment_rows_div')
        .find('.payment-amount')
        .each(function() {
            if (parseFloat($(this).val())) {
                total_paying += __read_number($(this));
            }
        });
    var bal_due = total_payable - total_paying;
    var change_return = 0;

    // Change return
    if (bal_due < 0 || Math.abs(bal_due) < 0.05) {
        __write_number($('input#change_return'), bal_due * -1);
        $('span.change_return_span').text(__currency_trans_from_en(bal_due * -1, true));
        change_return = bal_due * -1;
        bal_due = 0;
    } else {
        __write_number($('input#change_return'), 0);
        $('span.change_return_span').text(__currency_trans_from_en(0, true));
        change_return = 0;
    }

    if (change_return !== 0) {
        $('#change_return_payment_data').removeClass('hide');
    } else {
        $('#change_return_payment_data').addClass('hide');
    }

    __write_number($('input#total_paying_input'), total_paying);
    $('span.total_paying').text(__currency_trans_from_en(total_paying, true));

    __write_number($('input#in_balance_due'), bal_due);
    $('span.balance_due').text(__currency_trans_from_en(bal_due, true));

    __highlight(bal_due * -1, $('span.balance_due'));
    __highlight(change_return * -1, $('span.change_return_span'));
}

// Unified price calculation function
function calculatePrice(row, options = {}) {
    var unit_price = __read_number(row.find('input.pos_unit_price'));
    
    // Get unit_price from unit_price_inc_tax if not available
    if (!unit_price || unit_price <= 0) {
        var unit_price_inc_tax = __read_number(row.find('input.pos_unit_price_inc_tax'));
        var tax_rate = row.find('select.tax_id').find(':selected').data('rate') || 0;
        
        if (unit_price_inc_tax && unit_price_inc_tax > 0) {
            unit_price = tax_rate > 0 ? 
                __get_principle(unit_price_inc_tax, tax_rate) : 
                unit_price_inc_tax;
            
            __write_number(row.find('input.pos_unit_price'), unit_price);
        }
    }
    
    var row_discount_type = row.find('select.row_discount_type').val() || 'percentage';
    var row_discount_amount = __read_number(row.find('input.row_discount_amount')) || 0;
    var discounted_unit_price = unit_price;
    
    // Apply discount
    if (row_discount_amount) {
        if (row_discount_type == 'fixed') {
            discounted_unit_price = unit_price - row_discount_amount;
        } else {
            discounted_unit_price = __substract_percent(unit_price, row_discount_amount);
        }
    }
    
    // Calculate unit price from discounted price
    var this_unit_price = discounted_unit_price;
    if (row_discount_amount) {
        if (row_discount_type == 'fixed') {
            this_unit_price = discounted_unit_price + row_discount_amount;
        } else {
            this_unit_price = __get_principle(discounted_unit_price, row_discount_amount, true);
        }
    }
    
    return {
        unit_price: Math.max(0, unit_price),
        discounted_unit_price: Math.max(0, discounted_unit_price),
        original_unit_price: Math.max(0, this_unit_price)
    };
}

// Unified row calculation function
function calculateRowTotal(row) {
    var quantity = parseFloat(row.find('.pos_quantity').val()) || 0;
    var unitPrice = parseFloat(row.find('.pos_unit_price_inc_tax').val()) || 0;
    var discountPercent = parseFloat(row.find('td:eq(7) input').val()) || 0;
    
    // Apply discount
    var discountAmount = unitPrice * (discountPercent / 100);
    var priceAfterDiscount = unitPrice - discountAmount;
    
    // Calculate subtotal
    var subtotal = quantity * priceAfterDiscount;
    
    // Update line total fields
    row.find('.pos_line_total').val(formatNumber(subtotal, 2));
    row.find('.pos_line_total_text').text(__currency_trans_from_en(subtotal, true));
    
    // Update totals
    pos_total_row();
}

// Export calculations module
window.POSCalculations = {
    pos_total_row: pos_total_row,
    get_subtotal: get_subtotal,
    calculate_billing_details: calculate_billing_details,
    pos_discount: pos_discount,
    pos_order_tax: pos_order_tax,
    calculate_balance_due: calculate_balance_due,
    calculatePrice: calculatePrice,
    calculateRowTotal: calculateRowTotal
}; 