/**
 * POS Core Module
 * Handles initialization, global variables, and core functionality
 */

// Global variables
var global_brand_id = null;
var global_p_category_id = null;
var customer_set = false;
var pos_form_obj = null;
var totalCalculationTimeout;

// Core initialization - wait for jQuery
function initializeCore() {
    if (typeof $ === 'undefined') {
        console.log('⏳ Waiting for jQuery to be available...');
        setTimeout(initializeCore, 100);
        return;
    }
    
    initializePOS();
    setupGlobalEventHandlers();
    setupAjaxPrefilters();
}

// Start initialization when document is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeCore);
} else {
    initializeCore();
}

function initializePOS() {
    // Prevent enter key except in textarea
    $('form').on('keyup keypress', function(e) {
        var keyCode = e.keyCode || e.which;
        if (keyCode === 13 && e.target.tagName != 'TEXTAREA') {
            e.preventDefault();
            return false;
        }
    });

    // Initialize form objects
    if ($('form#edit_pos_sell_form').length > 0) {
        pos_total_row();
        pos_form_obj = $('form#edit_pos_sell_form');
    } else {
        pos_form_obj = $('form#add_pos_sell_form');
    }

    if ($('form#edit_pos_sell_form').length > 0 || $('form#add_pos_sell_form').length > 0) {
        // Initialize printer if function exists (will be loaded by print module)
        if (typeof initialize_printer === 'function') {
            initialize_printer();
        } else {
            console.log('⚠️ Printer initialization deferred - print module not loaded yet');
            // Try to initialize printer after a delay when modules are loaded
            setTimeout(function() {
                if (typeof initialize_printer === 'function') {
                    console.log('🖨️ Initializing printer after module load');
                    initialize_printer();
                }
            }, 2000);
        }
    }

    set_default_customer();
}

function setupGlobalEventHandlers() {
    // Location change handler
    $('select#select_location_id').change(function() {
        reset_pos_form();
        updateLocationSettings($(this));
        set_payment_type_dropdown();
        
        if ($('#types_of_service_id').length && $('#types_of_service_id').val()) {
            $('#types_of_service_id').change();
        }
    });
}

function setupAjaxPrefilters() {
    // Override AJAX to ensure status field is sent
    $.ajaxPrefilter(function(options, originalOptions, jqXHR) {
        if (options.url && (options.url.includes('/pos/create') || options.url.includes('/pos/update'))) {
            if (options.data && typeof options.data === 'string') {
                if (options.data.indexOf('status=') === -1) {
                    options.data += '&status=quotation';
                }
            } else if (options.data && typeof options.data === 'object') {
                if (!options.data.status) {
                    options.data.status = 'quotation';
                }
            }
        }
    });
}

function updateLocationSettings($locationSelect) {
    var default_price_group = $locationSelect.find(':selected').data('default_price_group');
    if (default_price_group) {
        if($("#price_group option[value='" + default_price_group + "']").length > 0) {
            $("#price_group").val(default_price_group);
            $("#price_group").change();
        }
    }

    // Set default invoice scheme
    if ($('#invoice_scheme_id').length) {
        var invoice_scheme_id;
        if ($('input[name="is_direct_sale"]').length > 0) {
            invoice_scheme_id = $locationSelect.find(':selected').data('default_sale_invoice_scheme_id');
        } else {
            invoice_scheme_id = $locationSelect.find(':selected').data('default_invoice_scheme_id');
        }
        $("#invoice_scheme_id").val(invoice_scheme_id).change();
    }

    // Set default invoice layout
    if ($('#invoice_layout_id').length) {
        let invoice_layout_id = $locationSelect.find(':selected').data('default_invoice_layout_id');
        $("#invoice_layout_id").val(invoice_layout_id).change();
    }
    
    // Set default price group
    if ($('#default_price_group').length) {
        var dpg = default_price_group ? default_price_group : 0;
        $('#default_price_group').val(dpg);
    }
}

// Export for use in other modules
window.POSCore = {
    initializePOS: initializePOS,
    updateLocationSettings: updateLocationSettings,
    global_brand_id: global_brand_id,
    global_p_category_id: global_p_category_id,
    customer_set: customer_set,
    pos_form_obj: pos_form_obj
}; 