/**
 * POS Events Module
 * Handles all event handlers, user interactions, and form submissions
 */

// Initialize event handlers
$(document).ready(function() {
    setupProductEvents();
    setupTableEvents();
    setupFormEvents();
    setupPaymentEvents();
    setupCustomerEvents();
    setupCategoryEvents();
});

function setupProductEvents() {
    // Product search autocomplete
    $('#search_product').autocomplete({
        delay: 1000,
        source: function(request, response) {
            var price_group = '';
            var search_fields = [];
            $('.search_fields:checked').each(function(i){
              search_fields[i] = $(this).val();
            });

            if ($('#price_group').length > 0) {
                price_group = $('#price_group').val();
            }
            $.getJSON(
                '/products/list',
                {
                    price_group: price_group,
                    location_id: $('input#location_id').val(),
                    term: request.term,
                    not_for_selling: 0,
                    search_fields: search_fields,
                    include_warehouse_details: 1
                },
                response
            );
        },
        response: function(event, ui) {
            if (ui.content.length === 0) {
                ui.content.push({
                    label: 'No products found',
                    value: '',
                    disabled: true
                });
            }
        },
        focus: function(event, ui) {
            if (ui.item.disabled) {
                return false;
            }
        },
        select: function(event, ui) {
            if (ui.item.disabled) {
                return false;
            }
            addProductToTable(ui.item);
            return false;
        }
    });

    // Product search keydown
    $('#search_product').keydown(function(e) {
        if (e.keyCode === 13) {
            e.preventDefault();
            var searchTerm = $(this).val();
            if (searchTerm.trim()) {
                // Trigger search
                $(this).autocomplete('search');
            }
        }
    });
}

function setupTableEvents() {
    // Quantity change
    $('table#pos_table tbody').on('change', 'input.pos_quantity', function() {
        var tr = $(this).closest('tr');
        var quantity = __read_number($(this));
        var unit_price_inc_tax = __read_number(tr.find('input.pos_unit_price_inc_tax'));
        var line_total = quantity * unit_price_inc_tax;
        
        __write_number(tr.find('input.pos_line_total'), line_total);
        tr.find('span.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        // Update modifier quantities
        tr.find('.modifier_qty_text').each(function(){
            $(this).text(quantity);
        });
        tr.find('.modifiers_quantity').each(function(){
            $(this).val(quantity);
        });
        
        POSCalculations.pos_total_row();
    });

    // Unit price change
    $('table#pos_table tbody').on('change', 'input.pos_unit_price', function() {
        var tr = $(this).closest('tr');
        var unit_price = __read_number($(this));
        var tax_rate = tr.find('select.tax_id option:selected').data('rate') || 0;
        var unit_price_inc_tax = tax_rate > 0 ? 
            __add_percent(unit_price, tax_rate) : unit_price;
        
        __write_number(tr.find('input.pos_unit_price_inc_tax'), unit_price_inc_tax);
        
        // Recalculate line total
        var quantity = __read_number(tr.find('input.pos_quantity'));
        var line_total = quantity * unit_price_inc_tax;
        __write_number(tr.find('input.pos_line_total'), line_total);
        tr.find('span.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        POSCalculations.pos_total_row();
    });

    // Tax change
    $('table#pos_table tbody').on('change', 'select.tax_id', function() {
        var tr = $(this).closest('tr');
        var unit_price = __read_number(tr.find('input.pos_unit_price'));
        var tax_rate = $(this).find(':selected').data('rate') || 0;
        var unit_price_inc_tax = tax_rate > 0 ? 
            __add_percent(unit_price, tax_rate) : unit_price;
        
        __write_number(tr.find('input.pos_unit_price_inc_tax'), unit_price_inc_tax);
        
        // Recalculate line total
        var quantity = __read_number(tr.find('input.pos_quantity'));
        var line_total = quantity * unit_price_inc_tax;
        __write_number(tr.find('input.pos_line_total'), line_total);
        tr.find('span.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        POSCalculations.pos_total_row();
    });

    // Remove row
    $('table#pos_table tbody').on('click', 'i.pos_remove_row', function() {
        $(this).closest('tr').remove();
        POSCalculations.pos_total_row();
        POSUtils.updateSerialNumbers();
    });
}

function setupFormEvents() {
    // Form submission
    $(document).on('click', 'button[type="submit"], #pos-finalize, #pos-draft, #pos-quotation, .pos-express-finalize', function() {
        var $form = $(this).closest('form');
        
        // Ensure status field is added
        if ($form.find('input[name="status"]').length === 0) {
            var statusValue = 'quotation';
            if ($(this).attr('id') === 'pos-finalize') {
                statusValue = 'final';
            } else if ($(this).attr('id') === 'pos-draft') {
                statusValue = 'draft';
            }
            $form.append('<input type="hidden" name="status" value="' + statusValue + '">');
        }
    });

    // AJAX send handler
    $(document).ajaxSend(function(event, jqxhr, settings) {
        // Add loading indicators
        if (settings.url && settings.url.includes('/pos/')) {
            $('body').addClass('loading');
        }
    });

    // AJAX complete handler
    $(document).ajaxComplete(function(event, jqxhr, settings) {
        if (settings.url && settings.url.includes('/pos/')) {
            $('body').removeClass('loading');
        }
    });
}

function setupPaymentEvents() {
    // Add payment row
    $('button#add-payment-row').click(function() {
        $.ajax({
            url: '/sells/get-payment-row',
            dataType: 'html',
            success: function(result) {
                $('#payment_rows_div').append(result);
                $('.payment-amount').last().focus();
            }
        });
    });

    // Remove payment row
    $(document).on('click', '.remove_payment_row', function() {
        $(this).closest('.payment_row').remove();
        POSCalculations.calculate_balance_due();
    });

    // Payment amount change
    $(document).on('change', '.payment-amount', function() {
        POSCalculations.calculate_balance_due();
    });
}

function setupCustomerEvents() {
    // Customer selection
    $('#customer_id').on('select2:select', function(e) {
        var data = e.params.data;
        
        if (data.pay_term_number) {
            $('input#pay_term_number').val(data.pay_term_number);
        } else {
            $('input#pay_term_number').val('');
        }

        if (data.pay_term_type) {
            $('#add_sell_form select[name="pay_term_type"]').val(data.pay_term_type);
            $('#edit_sell_form select[name="pay_term_type"]').val(data.pay_term_type);
        } else {
            $('#add_sell_form select[name="pay_term_type"]').val('');
            $('#edit_sell_form select[name="pay_term_type"]').val('');
        }
        
        update_shipping_address(data);
        $('#advance_balance_text').text(__currency_trans_from_en(data.balance), true);
        $('#advance_balance').val(data.balance);

        if (data.price_calculation_type == 'selling_price_group') {
            $('#price_group').val(data.selling_price_group_id);
            $('#price_group').change();
        }
        
        if ($('.contact_due_text').length) {
            get_contact_due(data.id);
        }
    });

    // Customer change
    $(document).on('change', 'select#customer_id', function(){
        var default_customer_id = $('#default_customer_id').val();
        if ($(this).val() == default_customer_id) {
            // Disable reward points for walkin customers
            if ($('#rp_redeemed_modal').length) {
                $('#rp_redeemed_modal').val('');
                $('#rp_redeemed_modal').change();
                $('#rp_redeemed_modal').attr('disabled', true);
                $('#available_rp').text('');
                updateRedeemedAmount();
                POSCalculations.pos_total_row();
            }
        } else {
            if ($('#rp_redeemed_modal').length) {
                $('#rp_redeemed_modal').removeAttr('disabled');
            }
            getCustomerRewardPoints();
        }

        get_sales_orders();
    });
}

function setupCategoryEvents() {
    // Category click
    $('.product_category').on('click', function(e) {
        e.preventDefault();
        var category_id = $(this).data('category_id');
        global_p_category_id = category_id;
        
        // Update active category
        $('.product_category').removeClass('active');
        $(this).addClass('active');
        
        // Load products for category
        loadProductsForCategory(category_id);
    });

    // Brand click
    $('.product_brand').on('click', function(e) {
        e.preventDefault();
        var brand_id = $(this).data('brand_id');
        global_brand_id = brand_id;
        
        // Update active brand
        $('.product_brand').removeClass('active');
        $(this).addClass('active');
        
        // Load products for brand
        loadProductsForBrand(brand_id);
    });
}

// Helper functions
function addProductToTable(product) {
    // Implementation for adding product to POS table
    // This would contain the logic to add a product row
}

function loadProductsForCategory(categoryId) {
    // Implementation for loading products by category
}

function loadProductsForBrand(brandId) {
    // Implementation for loading products by brand
}

function update_shipping_address(data) {
    // Implementation for updating shipping address
}

function get_contact_due(contactId) {
    // Implementation for getting contact due amount
}

function getCustomerRewardPoints() {
    // Implementation for getting customer reward points
}

function updateRedeemedAmount() {
    // Implementation for updating redeemed amount
}

function get_sales_orders() {
    // Implementation for getting sales orders
}

// Legacy function for setting default customer
function setDefaultCustomer() {
    if ($('#customer_id').length > 0 && !customer_set) {
        var defaultCustomer = $('#customer_id').find('option[value=""]');
        if (defaultCustomer.length > 0) {
            $('#customer_id').val('').change();
            customer_set = true;
        }
    }
}

// Export events module
window.POSEvents = {
    setupProductEvents: setupProductEvents,
    setupTableEvents: setupTableEvents,
    setupFormEvents: setupFormEvents,
    setupPaymentEvents: setupPaymentEvents,
    setupCustomerEvents: setupCustomerEvents,
    setupCategoryEvents: setupCategoryEvents,
    setDefaultCustomer: setDefaultCustomer
};

// Export legacy function for backward compatibility
window.set_default_customer = setDefaultCustomer; 