/**
 * POS Module Loader
 * Handles loading and initialization of all POS modules
 */

// Module loading order
var MODULE_LOAD_ORDER = [
    'pos-core.js',
    'pos-utils.js', 
    'pos-calculations.js',
    'pos-stock.js',
    'pos-ui.js',
    'pos-api.js',
    'pos-events.js',
    'pos-print.js',
    'pos-reports.js',
    'pos-settings.js',
    'pos-testing.js'
];

// Module dependencies
var MODULE_DEPENDENCIES = {
    'pos-core.js': [],
    'pos-utils.js': ['pos-core.js'],
    'pos-calculations.js': ['pos-core.js', 'pos-utils.js'],
    'pos-stock.js': ['pos-core.js', 'pos-utils.js'],
    'pos-ui.js': ['pos-core.js', 'pos-utils.js'],
    'pos-api.js': ['pos-core.js', 'pos-utils.js', 'pos-ui.js'],
    'pos-events.js': ['pos-core.js', 'pos-utils.js', 'pos-calculations.js', 'pos-stock.js', 'pos-ui.js', 'pos-api.js'],
    'pos-print.js': ['pos-core.js', 'pos-utils.js', 'pos-ui.js', 'pos-api.js'],
    'pos-reports.js': ['pos-core.js', 'pos-utils.js', 'pos-ui.js', 'pos-api.js'],
    'pos-settings.js': ['pos-core.js', 'pos-utils.js', 'pos-ui.js'],
    'pos-testing.js': ['pos-core.js', 'pos-utils.js', 'pos-calculations.js', 'pos-stock.js', 'pos-ui.js', 'pos-api.js', 'pos-events.js', 'pos-print.js', 'pos-reports.js', 'pos-settings.js']
};

// Loaded modules tracking
var loadedModules = {};
var moduleLoadPromises = {};

/**
 * Load a single module
 */
function loadModule(moduleName) {
    return new Promise(function(resolve, reject) {
        // Check if already loaded
        if (loadedModules[moduleName]) {
            resolve(loadedModules[moduleName]);
            return;
        }

        // Check if already loading
        if (moduleLoadPromises[moduleName]) {
            moduleLoadPromises[moduleName].then(resolve).catch(reject);
            return;
        }

        // Create script element
        var script = document.createElement('script');
        script.src = '/js/pos-modules/' + moduleName;
        script.async = false;
        script.type = 'text/javascript';

        // Handle load success
        script.onload = function() {
            loadedModules[moduleName] = true;
            console.log('✅ Loaded module: ' + moduleName);
            resolve(true);
        };

        // Handle load error
        script.onerror = function() {
            console.error('❌ Failed to load module: ' + moduleName);
            reject(new Error('Failed to load module: ' + moduleName));
        };

        // Store promise
        moduleLoadPromises[moduleName] = new Promise(function(resolvePromise, rejectPromise) {
            script.onload = function() {
                loadedModules[moduleName] = true;
                console.log('✅ Loaded module: ' + moduleName);
                resolvePromise(true);
            };
            script.onerror = function() {
                console.error('❌ Failed to load module: ' + moduleName);
                rejectPromise(new Error('Failed to load module: ' + moduleName));
            };
        });

        // Append script to head
        document.head.appendChild(script);
    });
}

/**
 * Load module dependencies
 */
function loadModuleDependencies(moduleName) {
    var dependencies = MODULE_DEPENDENCIES[moduleName] || [];
    var dependencyPromises = dependencies.map(function(dep) {
        return loadModule(dep);
    });
    
    return Promise.all(dependencyPromises);
}

/**
 * Load all modules in correct order
 */
function loadAllModules() {
    console.log('🚀 Starting POS module loading...');
    
    var loadPromises = MODULE_LOAD_ORDER.map(function(moduleName) {
        return loadModuleDependencies(moduleName).then(function() {
            return loadModule(moduleName);
        });
    });

    return Promise.all(loadPromises).then(function() {
        console.log('✅ All POS modules loaded successfully');
        initializePOSModules();
    }).catch(function(error) {
        console.error('❌ Error loading POS modules:', error);
        handleModuleLoadError(error);
    });
}

/**
 * Initialize all loaded modules
 */
function initializePOSModules() {
    try {
        // Initialize core module
        if (window.POSCore && window.POSCore.initializePOS) {
            window.POSCore.initializePOS();
        }

        // Initialize events module
        if (window.POSEvents) {
            // Events are auto-initialized via document.ready
            console.log('✅ POS events initialized');
        }

        // Initialize calculations module
        if (window.POSCalculations) {
            console.log('✅ POS calculations initialized');
        }

        // Initialize stock module
        if (window.POSStock) {
            console.log('✅ POS stock management initialized');
        }

        // Initialize UI module
        if (window.POSUI) {
            console.log('✅ POS UI management initialized');
        }

        // Initialize API module
        if (window.POSAPI) {
            console.log('✅ POS API management initialized');
        }

        // Initialize Print module
        if (window.POSPrint) {
            console.log('✅ POS Print management initialized');
        }

        // Initialize Reports module
        if (window.POSReports) {
            console.log('✅ POS Reports management initialized');
        }

        // Initialize Settings module
        if (window.POSSettings) {
            console.log('✅ POS Settings management initialized');
        }

        // Initialize testing module (development only)
        if (window.POSTest && (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1')) {
            console.log('✅ POS Testing module initialized');
        }

        // Initialize utilities module
        if (window.POSUtils) {
            console.log('✅ POS utilities initialized');
        }

        // Trigger module ready event
        $(document).trigger('pos-modules-ready');
        
        console.log('🎉 POS system fully initialized');
        
    } catch (error) {
        console.error('❌ Error initializing POS modules:', error);
        handleModuleInitError(error);
    }
}

/**
 * Handle module loading errors
 */
function handleModuleLoadError(error) {
    console.error('Module load error:', error);
    
    // Show user-friendly error message
    if (typeof toastr !== 'undefined') {
        toastr.error('Failed to load POS system. Please refresh the page.');
    } else {
        alert('Failed to load POS system. Please refresh the page.');
    }
    
    // Try to load fallback
    loadFallbackPOS();
}

/**
 * Handle module initialization errors
 */
function handleModuleInitError(error) {
    console.error('Module init error:', error);
    
    // Show user-friendly error message
    if (typeof toastr !== 'undefined') {
        toastr.error('POS system initialization failed. Some features may not work.');
    }
}

/**
 * Load fallback POS system
 */
function loadFallbackPOS() {
    console.log('🔄 Loading fallback POS system...');
    
    // Load the original pos.js file as fallback
    var script = document.createElement('script');
    script.src = '/js/pos.js';
    script.async = false;
    script.type = 'text/javascript';
    
    script.onload = function() {
        console.log('✅ Fallback POS system loaded');
    };
    
    script.onerror = function() {
        console.error('❌ Failed to load fallback POS system');
    };
    
    document.head.appendChild(script);
}

/**
 * Check if all modules are loaded
 */
function areModulesLoaded() {
    return MODULE_LOAD_ORDER.every(function(moduleName) {
        return loadedModules[moduleName] === true;
    });
}

/**
 * Get module loading status
 */
function getModuleStatus() {
    var status = {};
    MODULE_LOAD_ORDER.forEach(function(moduleName) {
        status[moduleName] = loadedModules[moduleName] || false;
    });
    return status;
}

/**
 * Reload a specific module
 */
function reloadModule(moduleName) {
    return new Promise(function(resolve, reject) {
        // Remove from loaded modules
        delete loadedModules[moduleName];
        delete moduleLoadPromises[moduleName];
        
        // Reload module
        loadModule(moduleName).then(resolve).catch(reject);
    });
}

/**
 * Initialize POS system
 */
function initializePOS() {
    // Check if jQuery is available
    if (typeof $ === 'undefined') {
        console.error('❌ jQuery is required but not loaded');
        return;
    }
    
    // Check if we're on a POS page
    if ($('#pos_table').length === 0 && $('form#add_pos_sell_form').length === 0) {
        console.log('ℹ️ Not on POS page, skipping module loading');
        return;
    }
    
    // Load all modules
    loadAllModules();
}

// Auto-initialize when DOM is ready
$(document).ready(function() {
    // Small delay to ensure all scripts are loaded
    setTimeout(function() {
        initializePOS();
    }, 100);
});

// Export loader functions
window.POSLoader = {
    loadAllModules: loadAllModules,
    loadModule: loadModule,
    areModulesLoaded: areModulesLoaded,
    getModuleStatus: getModuleStatus,
    reloadModule: reloadModule,
    initializePOS: initializePOS
}; 