/**
 * POS Print Module
 * Handles receipt printing, invoice generation, and print management
 */

// Print configuration
var PrintConfig = {
    defaultPrinter: '',
    printFormat: 'receipt',
    paperSize: '80mm',
    fontSize: '12px',
    autoPrint: false,
    printPreview: true
};

// Initialize print module
$(document).ready(function() {
    setupPrintHandlers();
    loadPrintSettings();
});

function setupPrintHandlers() {
    // Print receipt button
    $(document).on('click', '.print-receipt', function(e) {
        e.preventDefault();
        var transactionId = $(this).data('transaction-id');
        printReceipt(transactionId);
    });
    
    // Print invoice button
    $(document).on('click', '.print-invoice', function(e) {
        e.preventDefault();
        var transactionId = $(this).data('transaction-id');
        printInvoice(transactionId);
    });
    
    // Print quote button
    $(document).on('click', '.print-quote', function(e) {
        e.preventDefault();
        var transactionId = $(this).data('transaction-id');
        printQuote(transactionId);
    });
    
    // Print settings
    $(document).on('click', '.print-settings', function(e) {
        e.preventDefault();
        showPrintSettings();
    });
}

function loadPrintSettings() {
    // Load saved print settings from localStorage
    var savedSettings = localStorage.getItem('pos_print_settings');
    if (savedSettings) {
        try {
            var settings = JSON.parse(savedSettings);
            Object.assign(PrintConfig, settings);
        } catch (e) {
            console.error('Failed to load print settings:', e);
        }
    }
}

function savePrintSettings() {
    try {
        localStorage.setItem('pos_print_settings', JSON.stringify(PrintConfig));
    } catch (e) {
        console.error('Failed to save print settings:', e);
    }
}

// Print receipt function
function printReceipt(transactionId) {
    if (!transactionId) {
        POSUI.showNotification('Transaction ID is required', 'error');
        return;
    }
    
    POSUI.showLoading('.print-receipt', 'Generating Receipt...');
    
    // Get receipt data
    $.ajax({
        url: '/sells/get-receipt',
        method: 'GET',
        data: {
            transaction_id: transactionId,
            format: PrintConfig.printFormat
        },
        success: function(response) {
            POSUI.hideLoading('.print-receipt');
            
            if (response.success) {
                generatePrintContent(response.data, 'receipt');
            } else {
                POSUI.showNotification(response.msg || 'Failed to generate receipt', 'error');
            }
        },
        error: function() {
            POSUI.hideLoading('.print-receipt');
            POSUI.showNotification('Failed to generate receipt', 'error');
        }
    });
}

// Print invoice function
function printInvoice(transactionId) {
    if (!transactionId) {
        POSUI.showNotification('Transaction ID is required', 'error');
        return;
    }
    
    POSUI.showLoading('.print-invoice', 'Generating Invoice...');
    
    // Get invoice data
    $.ajax({
        url: '/sells/get-invoice',
        method: 'GET',
        data: {
            transaction_id: transactionId,
            format: 'pdf'
        },
        success: function(response) {
            POSUI.hideLoading('.print-invoice');
            
            if (response.success) {
                if (response.data.pdf_url) {
                    // Open PDF in new window
                    window.open(response.data.pdf_url, '_blank');
                } else {
                    generatePrintContent(response.data, 'invoice');
                }
            } else {
                POSUI.showNotification(response.msg || 'Failed to generate invoice', 'error');
            }
        },
        error: function() {
            POSUI.hideLoading('.print-invoice');
            POSUI.showNotification('Failed to generate invoice', 'error');
        }
    });
}

// Print quote function
function printQuote(transactionId) {
    if (!transactionId) {
        POSUI.showNotification('Transaction ID is required', 'error');
        return;
    }
    
    POSUI.showLoading('.print-quote', 'Generating Quote...');
    
    // Get quote data
    $.ajax({
        url: '/sells/get-quote',
        method: 'GET',
        data: {
            transaction_id: transactionId,
            format: 'pdf'
        },
        success: function(response) {
            POSUI.hideLoading('.print-quote');
            
            if (response.success) {
                if (response.data.pdf_url) {
                    // Open PDF in new window
                    window.open(response.data.pdf_url, '_blank');
                } else {
                    generatePrintContent(response.data, 'quote');
                }
            } else {
                POSUI.showNotification(response.msg || 'Failed to generate quote', 'error');
            }
        },
        error: function() {
            POSUI.hideLoading('.print-quote');
            POSUI.showNotification('Failed to generate quote', 'error');
        }
    });
}

// Generate print content
function generatePrintContent(data, type) {
    var printWindow = window.open('', '_blank', 'width=800,height=600');
    var printContent = '';
    
    switch (type) {
        case 'receipt':
            printContent = generateReceiptHTML(data);
            break;
        case 'invoice':
            printContent = generateInvoiceHTML(data);
            break;
        case 'quote':
            printContent = generateQuoteHTML(data);
            break;
        default:
            printContent = generateReceiptHTML(data);
    }
    
    printWindow.document.write(printContent);
    printWindow.document.close();
    
    // Auto print if enabled
    if (PrintConfig.autoPrint) {
        setTimeout(function() {
            printWindow.print();
        }, 500);
    }
}

// Generate receipt HTML
function generateReceiptHTML(data) {
    var html = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Receipt - ${data.invoice_no}</title>
            <style>
                body {
                    font-family: 'Courier New', monospace;
                    font-size: ${PrintConfig.fontSize};
                    margin: 0;
                    padding: 10px;
                    width: ${PrintConfig.paperSize};
                }
                .header {
                    text-align: center;
                    border-bottom: 1px dashed #000;
                    padding-bottom: 10px;
                    margin-bottom: 10px;
                }
                .business-name {
                    font-size: 16px;
                    font-weight: bold;
                    margin-bottom: 5px;
                }
                .business-info {
                    font-size: 12px;
                    margin-bottom: 5px;
                }
                .receipt-details {
                    margin-bottom: 15px;
                }
                .receipt-row {
                    display: flex;
                    justify-content: space-between;
                    margin-bottom: 3px;
                }
                .items-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-bottom: 15px;
                }
                .items-table th,
                .items-table td {
                    text-align: left;
                    padding: 2px 0;
                    border-bottom: 1px dotted #ccc;
                }
                .total-section {
                    border-top: 1px dashed #000;
                    padding-top: 10px;
                    margin-top: 10px;
                }
                .total-row {
                    display: flex;
                    justify-content: space-between;
                    margin-bottom: 5px;
                }
                .total-row.final {
                    font-weight: bold;
                    font-size: 14px;
                }
                .footer {
                    text-align: center;
                    margin-top: 20px;
                    font-size: 10px;
                }
                @media print {
                    body { margin: 0; }
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="business-name">${data.business_name || 'Business Name'}</div>
                <div class="business-info">${data.business_address || ''}</div>
                <div class="business-info">${data.business_phone || ''}</div>
                <div class="business-info">${data.business_email || ''}</div>
            </div>
            
            <div class="receipt-details">
                <div class="receipt-row">
                    <span>Receipt No:</span>
                    <span>${data.invoice_no}</span>
                </div>
                <div class="receipt-row">
                    <span>Date:</span>
                    <span>${data.transaction_date}</span>
                </div>
                <div class="receipt-row">
                    <span>Customer:</span>
                    <span>${data.customer_name || 'Walk-in Customer'}</span>
                </div>
                <div class="receipt-row">
                    <span>Cashier:</span>
                    <span>${data.created_by}</span>
                </div>
            </div>
            
            <table class="items-table">
                <thead>
                    <tr>
                        <th>Item</th>
                        <th>Qty</th>
                        <th>Price</th>
                        <th>Total</th>
                    </tr>
                </thead>
                <tbody>`;
    
    // Add items
    if (data.sell_lines && data.sell_lines.length > 0) {
        data.sell_lines.forEach(function(item) {
            html += `
                <tr>
                    <td>${item.product_name}</td>
                    <td>${item.quantity}</td>
                    <td>${formatCurrency(item.unit_price_inc_tax)}</td>
                    <td>${formatCurrency(item.line_total)}</td>
                </tr>`;
        });
    }
    
    html += `
                </tbody>
            </table>
            
            <div class="total-section">
                <div class="total-row">
                    <span>Subtotal:</span>
                    <span>${formatCurrency(data.total_before_tax)}</span>
                </div>`;
    
    // Add tax if applicable
    if (data.tax_amount > 0) {
        html += `
                <div class="total-row">
                    <span>Tax:</span>
                    <span>${formatCurrency(data.tax_amount)}</span>
                </div>`;
    }
    
    // Add discount if applicable
    if (data.discount_amount > 0) {
        html += `
                <div class="total-row">
                    <span>Discount:</span>
                    <span>-${formatCurrency(data.discount_amount)}</span>
                </div>`;
    }
    
    html += `
                <div class="total-row final">
                    <span>Total:</span>
                    <span>${formatCurrency(data.final_total)}</span>
                </div>
                <div class="total-row">
                    <span>Paid:</span>
                    <span>${formatCurrency(data.total_paid)}</span>
                </div>
                <div class="total-row">
                    <span>Change:</span>
                    <span>${formatCurrency(data.total_paid - data.final_total)}</span>
                </div>
            </div>
            
            <div class="footer">
                <div>Thank you for your purchase!</div>
                <div>Please visit again</div>
                <div class="no-print">
                    <button onclick="window.print()">Print</button>
                    <button onclick="window.close()">Close</button>
                </div>
            </div>
        </body>
        </html>`;
    
    return html;
}

// Generate invoice HTML
function generateInvoiceHTML(data) {
    // Similar to receipt but with more formal layout
    var html = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Invoice - ${data.invoice_no}</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    font-size: 12px;
                    margin: 20px;
                    line-height: 1.4;
                }
                .invoice-header {
                    display: flex;
                    justify-content: space-between;
                    margin-bottom: 30px;
                }
                .business-info {
                    flex: 1;
                }
                .invoice-info {
                    flex: 1;
                    text-align: right;
                }
                .customer-info {
                    margin-bottom: 30px;
                }
                .items-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-bottom: 30px;
                }
                .items-table th,
                .items-table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                }
                .items-table th {
                    background-color: #f5f5f5;
                }
                .total-section {
                    text-align: right;
                }
                .total-row {
                    margin-bottom: 5px;
                }
                .total-row.final {
                    font-weight: bold;
                    font-size: 16px;
                    border-top: 2px solid #000;
                    padding-top: 10px;
                }
                @media print {
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="invoice-header">
                <div class="business-info">
                    <h2>${data.business_name || 'Business Name'}</h2>
                    <p>${data.business_address || ''}</p>
                    <p>Phone: ${data.business_phone || ''}</p>
                    <p>Email: ${data.business_email || ''}</p>
                </div>
                <div class="invoice-info">
                    <h1>INVOICE</h1>
                    <p><strong>Invoice No:</strong> ${data.invoice_no}</p>
                    <p><strong>Date:</strong> ${data.transaction_date}</p>
                    <p><strong>Due Date:</strong> ${data.due_date || data.transaction_date}</p>
                </div>
            </div>
            
            <div class="customer-info">
                <h3>Bill To:</h3>
                <p><strong>${data.customer_name || 'Walk-in Customer'}</strong></p>
                <p>${data.customer_address || ''}</p>
                <p>Phone: ${data.customer_phone || ''}</p>
                <p>Email: ${data.customer_email || ''}</p>
            </div>
            
            <table class="items-table">
                <thead>
                    <tr>
                        <th>Item</th>
                        <th>Description</th>
                        <th>Qty</th>
                        <th>Unit Price</th>
                        <th>Tax</th>
                        <th>Total</th>
                    </tr>
                </thead>
                <tbody>`;
    
    // Add items
    if (data.sell_lines && data.sell_lines.length > 0) {
        data.sell_lines.forEach(function(item) {
            html += `
                <tr>
                    <td>${item.product_name}</td>
                    <td>${item.product_description || ''}</td>
                    <td>${item.quantity}</td>
                    <td>${formatCurrency(item.unit_price)}</td>
                    <td>${formatCurrency(item.tax_amount)}</td>
                    <td>${formatCurrency(item.line_total)}</td>
                </tr>`;
        });
    }
    
    html += `
                </tbody>
            </table>
            
            <div class="total-section">
                <div class="total-row">
                    <span>Subtotal: ${formatCurrency(data.total_before_tax)}</span>
                </div>`;
    
    if (data.tax_amount > 0) {
        html += `
                <div class="total-row">
                    <span>Tax: ${formatCurrency(data.tax_amount)}</span>
                </div>`;
    }
    
    if (data.discount_amount > 0) {
        html += `
                <div class="total-row">
                    <span>Discount: -${formatCurrency(data.discount_amount)}</span>
                </div>`;
    }
    
    html += `
                <div class="total-row final">
                    <span>Total Amount: ${formatCurrency(data.final_total)}</span>
                </div>
                <div class="total-row">
                    <span>Amount Paid: ${formatCurrency(data.total_paid)}</span>
                </div>
                <div class="total-row">
                    <span>Balance Due: ${formatCurrency(data.final_total - data.total_paid)}</span>
                </div>
            </div>
            
            <div class="no-print" style="margin-top: 30px; text-align: center;">
                <button onclick="window.print()">Print Invoice</button>
                <button onclick="window.close()">Close</button>
            </div>
        </body>
        </html>`;
    
    return html;
}

// Generate quote HTML
function generateQuoteHTML(data) {
    // Similar to invoice but marked as quote
    var html = generateInvoiceHTML(data);
    html = html.replace('<h1>INVOICE</h1>', '<h1>QUOTE</h1>');
    html = html.replace('Invoice No:', 'Quote No:');
    html = html.replace('Due Date:', 'Valid Until:');
    html = html.replace('Bill To:', 'Quote For:');
    html = html.replace('Total Amount:', 'Quote Total:');
    html = html.replace('Amount Paid:', 'Deposit Required:');
    html = html.replace('Balance Due:', 'Remaining Amount:');
    
    return html;
}

// Show print settings modal
function showPrintSettings() {
    var modalHtml = `
        <div class="modal fade" id="print-settings-modal" tabindex="-1" role="dialog">
            <div class="modal-dialog" role="document">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Print Settings</h5>
                        <button type="button" class="close" data-dismiss="modal">
                            <span>&times;</span>
                        </button>
                    </div>
                    <div class="modal-body">
                        <form id="print-settings-form">
                            <div class="form-group">
                                <label>Default Printer</label>
                                <select class="form-control" id="default-printer">
                                    <option value="">Auto-detect</option>
                                    <option value="thermal">Thermal Printer</option>
                                    <option value="inkjet">Inkjet Printer</option>
                                    <option value="laser">Laser Printer</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Print Format</label>
                                <select class="form-control" id="print-format">
                                    <option value="receipt">Receipt</option>
                                    <option value="invoice">Invoice</option>
                                    <option value="quote">Quote</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Paper Size</label>
                                <select class="form-control" id="paper-size">
                                    <option value="80mm">80mm (Thermal)</option>
                                    <option value="A4">A4</option>
                                    <option value="Letter">Letter</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Font Size</label>
                                <select class="form-control" id="font-size">
                                    <option value="10px">Small</option>
                                    <option value="12px">Medium</option>
                                    <option value="14px">Large</option>
                                </select>
                            </div>
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" id="auto-print">
                                <label class="form-check-label" for="auto-print">
                                    Auto-print when generated
                                </label>
                            </div>
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" id="print-preview">
                                <label class="form-check-label" for="print-preview">
                                    Show print preview
                                </label>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary" onclick="savePrintSettingsFromForm()">Save Settings</button>
                    </div>
                </div>
            </div>
        </div>`;
    
    // Remove existing modal if any
    $('#print-settings-modal').remove();
    
    // Add modal to body
    $('body').append(modalHtml);
    
    // Populate form with current settings
    $('#default-printer').val(PrintConfig.defaultPrinter);
    $('#print-format').val(PrintConfig.printFormat);
    $('#paper-size').val(PrintConfig.paperSize);
    $('#font-size').val(PrintConfig.fontSize);
    $('#auto-print').prop('checked', PrintConfig.autoPrint);
    $('#print-preview').prop('checked', PrintConfig.printPreview);
    
    // Show modal
    $('#print-settings-modal').modal('show');
}

// Save print settings from form
function savePrintSettingsFromForm() {
    PrintConfig.defaultPrinter = $('#default-printer').val();
    PrintConfig.printFormat = $('#print-format').val();
    PrintConfig.paperSize = $('#paper-size').val();
    PrintConfig.fontSize = $('#font-size').val();
    PrintConfig.autoPrint = $('#auto-print').is(':checked');
    PrintConfig.printPreview = $('#print-preview').is(':checked');
    
    savePrintSettings();
    
    $('#print-settings-modal').modal('hide');
    POSUI.showNotification('Print settings saved successfully', 'success');
}

// Utility functions
function formatCurrency(amount) {
    if (typeof amount !== 'number') {
        amount = parseFloat(amount) || 0;
    }
    return '$' + amount.toFixed(2);
}

// Get available printers
function getAvailablePrinters() {
    return new Promise(function(resolve, reject) {
        if (navigator.mediaDevices && navigator.mediaDevices.enumerateDevices) {
            navigator.mediaDevices.enumerateDevices()
                .then(function(devices) {
                    var printers = devices.filter(function(device) {
                        return device.kind === 'printer';
                    });
                    resolve(printers);
                })
                .catch(reject);
        } else {
            resolve([]);
        }
    });
}

// Legacy printer initialization function
function initialize_printer() {
    console.log('🖨️ Initializing printer system...');
    
    // Load print settings
    loadPrintSettings();
    
    // Setup print event handlers
    setupPrintHandlers();
    
    console.log('✅ Printer system initialized');
}

// Export print module
window.POSPrint = {
    // Configuration
    config: PrintConfig,
    
    // Main functions
    printReceipt: printReceipt,
    printInvoice: printInvoice,
    printQuote: printQuote,
    
    // Settings
    showPrintSettings: showPrintSettings,
    savePrintSettings: savePrintSettings,
    loadPrintSettings: loadPrintSettings,
    
    // Utilities
    getAvailablePrinters: getAvailablePrinters,
    formatCurrency: formatCurrency,
    
    // Legacy compatibility
    initialize_printer: initialize_printer
};

// Export legacy function for backward compatibility
window.initialize_printer = initialize_printer; 