/**
 * POS Reports Module
 * Handles sales analytics, reporting, and data visualization
 */

// Reports configuration
var ReportsConfig = {
    defaultDateRange: 'today',
    refreshInterval: 300000, // 5 minutes
    autoRefresh: false,
    chartType: 'line',
    currency: '$',
    locale: 'en-US'
};

// Report data cache
var reportCache = {};
var refreshTimer = null;

// Initialize reports module
$(document).ready(function() {
    setupReportHandlers();
    loadReportSettings();
    initializeCharts();
});

function setupReportHandlers() {
    // Report navigation
    $(document).on('click', '.report-nav-item', function(e) {
        e.preventDefault();
        var reportType = $(this).data('report-type');
        loadReport(reportType);
    });
    
    // Date range changes
    $(document).on('change', '#report-date-range', function() {
        var dateRange = $(this).val();
        updateDateRange(dateRange);
    });
    
    // Export buttons
    $(document).on('click', '.export-report', function(e) {
        e.preventDefault();
        var reportType = $(this).data('report-type');
        var format = $(this).data('format') || 'pdf';
        exportReport(reportType, format);
    });
    
    // Refresh button
    $(document).on('click', '.refresh-report', function(e) {
        e.preventDefault();
        refreshCurrentReport();
    });
    
    // Auto refresh toggle
    $(document).on('change', '#auto-refresh-toggle', function() {
        ReportsConfig.autoRefresh = $(this).is(':checked');
        toggleAutoRefresh();
    });
}

function loadReportSettings() {
    // Load saved settings from localStorage
    var savedSettings = localStorage.getItem('pos_reports_settings');
    if (savedSettings) {
        try {
            var settings = JSON.parse(savedSettings);
            Object.assign(ReportsConfig, settings);
        } catch (e) {
            console.error('Failed to load report settings:', e);
        }
    }
}

function saveReportSettings() {
    try {
        localStorage.setItem('pos_reports_settings', JSON.stringify(ReportsConfig));
    } catch (e) {
        console.error('Failed to save report settings:', e);
    }
}

// Load specific report
function loadReport(reportType) {
    if (!reportType) {
        reportType = 'sales-summary';
    }
    
    // Update active navigation
    $('.report-nav-item').removeClass('active');
    $('.report-nav-item[data-report-type="' + reportType + '"]').addClass('active');
    
    // Show loading
    POSUI.showLoading('#report-container', 'Loading Report...');
    
    // Get report data
    $.ajax({
        url: '/reports/' + reportType,
        method: 'GET',
        data: {
            date_range: ReportsConfig.defaultDateRange,
            location_id: $('input#location_id').val()
        },
        success: function(response) {
            POSUI.hideLoading('#report-container');
            
            if (response.success) {
                displayReport(reportType, response.data);
                cacheReportData(reportType, response.data);
            } else {
                POSUI.showNotification(response.msg || 'Failed to load report', 'error');
            }
        },
        error: function() {
            POSUI.hideLoading('#report-container');
            POSUI.showNotification('Failed to load report', 'error');
        }
    });
}

// Display report content
function displayReport(reportType, data) {
    var container = $('#report-container');
    
    switch (reportType) {
        case 'sales-summary':
            displaySalesSummary(container, data);
            break;
        case 'product-sales':
            displayProductSales(container, data);
            break;
        case 'customer-sales':
            displayCustomerSales(container, data);
            break;
        case 'payment-methods':
            displayPaymentMethods(container, data);
            break;
        case 'tax-summary':
            displayTaxSummary(container, data);
            break;
        case 'profit-loss':
            displayProfitLoss(container, data);
            break;
        default:
            displaySalesSummary(container, data);
    }
}

// Sales Summary Report
function displaySalesSummary(container, data) {
    var html = `
        <div class="row">
            <div class="col-md-3">
                <div class="info-box">
                    <span class="info-box-icon bg-success"><i class="fa fa-dollar"></i></span>
                    <div class="info-box-content">
                        <span class="info-box-text">Total Sales</span>
                        <span class="info-box-number">${formatCurrency(data.total_sales)}</span>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="info-box">
                    <span class="info-box-icon bg-info"><i class="fa fa-shopping-cart"></i></span>
                    <div class="info-box-content">
                        <span class="info-box-text">Total Orders</span>
                        <span class="info-box-number">${data.total_orders}</span>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="info-box">
                    <span class="info-box-icon bg-warning"><i class="fa fa-users"></i></span>
                    <div class="info-box-content">
                        <span class="info-box-text">Customers</span>
                        <span class="info-box-number">${data.total_customers}</span>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="info-box">
                    <span class="info-box-icon bg-danger"><i class="fa fa-percent"></i></span>
                    <div class="info-box-content">
                        <span class="info-box-text">Average Order</span>
                        <span class="info-box-number">${formatCurrency(data.average_order)}</span>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="row">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Sales Trend</h3>
                    </div>
                    <div class="card-body">
                        <canvas id="sales-chart" width="400" height="200"></canvas>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Top Products</h3>
                    </div>
                    <div class="card-body">
                        <div id="top-products-list"></div>
                    </div>
                </div>
            </div>
        </div>`;
    
    container.html(html);
    
    // Initialize charts
    if (data.sales_trend) {
        createSalesChart(data.sales_trend);
    }
    
    if (data.top_products) {
        displayTopProducts(data.top_products);
    }
}

// Product Sales Report
function displayProductSales(container, data) {
    var html = `
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Product Sales Report</h3>
                <div class="card-tools">
                    <button class="btn btn-sm btn-primary export-report" data-report-type="product-sales" data-format="excel">
                        <i class="fa fa-download"></i> Export Excel
                    </button>
                </div>
            </div>
            <div class="card-body">
                <table class="table table-bordered table-striped" id="product-sales-table">
                    <thead>
                        <tr>
                            <th>Product</th>
                            <th>Category</th>
                            <th>Quantity Sold</th>
                            <th>Revenue</th>
                            <th>Cost</th>
                            <th>Profit</th>
                            <th>Profit Margin</th>
                        </tr>
                    </thead>
                    <tbody>`;
    
    if (data.products && data.products.length > 0) {
        data.products.forEach(function(product) {
            var profitMargin = product.revenue > 0 ? ((product.profit / product.revenue) * 100).toFixed(2) : 0;
            html += `
                <tr>
                    <td>${product.name}</td>
                    <td>${product.category}</td>
                    <td>${product.quantity_sold}</td>
                    <td>${formatCurrency(product.revenue)}</td>
                    <td>${formatCurrency(product.cost)}</td>
                    <td>${formatCurrency(product.profit)}</td>
                    <td>${profitMargin}%</td>
                </tr>`;
        });
    }
    
    html += `
                    </tbody>
                </table>
            </div>
        </div>`;
    
    container.html(html);
    
    // Initialize DataTable
    if ($.fn.DataTable) {
        $('#product-sales-table').DataTable({
            responsive: true,
            order: [[3, 'desc']], // Sort by revenue
            pageLength: 25
        });
    }
}

// Customer Sales Report
function displayCustomerSales(container, data) {
    var html = `
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Customer Sales Report</h3>
                <div class="card-tools">
                    <button class="btn btn-sm btn-primary export-report" data-report-type="customer-sales" data-format="excel">
                        <i class="fa fa-download"></i> Export Excel
                    </button>
                </div>
            </div>
            <div class="card-body">
                <table class="table table-bordered table-striped" id="customer-sales-table">
                    <thead>
                        <tr>
                            <th>Customer</th>
                            <th>Email</th>
                            <th>Phone</th>
                            <th>Orders</th>
                            <th>Total Spent</th>
                            <th>Average Order</th>
                            <th>Last Order</th>
                        </tr>
                    </thead>
                    <tbody>`;
    
    if (data.customers && data.customers.length > 0) {
        data.customers.forEach(function(customer) {
            html += `
                <tr>
                    <td>${customer.name}</td>
                    <td>${customer.email || '-'}</td>
                    <td>${customer.phone || '-'}</td>
                    <td>${customer.orders_count}</td>
                    <td>${formatCurrency(customer.total_spent)}</td>
                    <td>${formatCurrency(customer.average_order)}</td>
                    <td>${customer.last_order_date || '-'}</td>
                </tr>`;
        });
    }
    
    html += `
                    </tbody>
                </table>
            </div>
        </div>`;
    
    container.html(html);
    
    // Initialize DataTable
    if ($.fn.DataTable) {
        $('#customer-sales-table').DataTable({
            responsive: true,
            order: [[4, 'desc']], // Sort by total spent
            pageLength: 25
        });
    }
}

// Payment Methods Report
function displayPaymentMethods(container, data) {
    var html = `
        <div class="row">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Payment Methods Distribution</h3>
                    </div>
                    <div class="card-body">
                        <canvas id="payment-methods-chart" width="400" height="300"></canvas>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Payment Methods Summary</h3>
                    </div>
                    <div class="card-body">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th>Payment Method</th>
                                    <th>Count</th>
                                    <th>Amount</th>
                                    <th>Percentage</th>
                                </tr>
                            </thead>
                            <tbody>`;
    
    if (data.payment_methods && data.payment_methods.length > 0) {
        data.payment_methods.forEach(function(method) {
            var percentage = data.total_amount > 0 ? ((method.amount / data.total_amount) * 100).toFixed(2) : 0;
            html += `
                <tr>
                    <td>${method.name}</td>
                    <td>${method.count}</td>
                    <td>${formatCurrency(method.amount)}</td>
                    <td>${percentage}%</td>
                </tr>`;
        });
    }
    
    html += `
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>`;
    
    container.html(html);
    
    // Create pie chart
    if (data.payment_methods) {
        createPaymentMethodsChart(data.payment_methods);
    }
}

// Tax Summary Report
function displayTaxSummary(container, data) {
    var html = `
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Tax Summary Report</h3>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th>Tax Rate</th>
                                    <th>Taxable Amount</th>
                                    <th>Tax Collected</th>
                                </tr>
                            </thead>
                            <tbody>`;
    
    if (data.tax_rates && data.tax_rates.length > 0) {
        data.tax_rates.forEach(function(tax) {
            html += `
                <tr>
                    <td>${tax.rate}%</td>
                    <td>${formatCurrency(tax.taxable_amount)}</td>
                    <td>${formatCurrency(tax.tax_collected)}</td>
                </tr>`;
        });
    }
    
    html += `
                            </tbody>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <div class="info-box">
                            <span class="info-box-icon bg-info"><i class="fa fa-percent"></i></span>
                            <div class="info-box-content">
                                <span class="info-box-text">Total Tax Collected</span>
                                <span class="info-box-number">${formatCurrency(data.total_tax_collected)}</span>
                            </div>
                        </div>
                        <div class="info-box">
                            <span class="info-box-icon bg-success"><i class="fa fa-dollar"></i></span>
                            <div class="info-box-content">
                                <span class="info-box-text">Total Taxable Sales</span>
                                <span class="info-box-number">${formatCurrency(data.total_taxable_sales)}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>`;
    
    container.html(html);
}

// Profit Loss Report
function displayProfitLoss(container, data) {
    var html = `
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Profit & Loss Report</h3>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h4>Revenue</h4>
                        <table class="table table-bordered">
                            <tr>
                                <td>Gross Sales</td>
                                <td class="text-right">${formatCurrency(data.gross_sales)}</td>
                            </tr>
                            <tr>
                                <td>Less: Returns</td>
                                <td class="text-right">-${formatCurrency(data.returns)}</td>
                            </tr>
                            <tr>
                                <td>Less: Discounts</td>
                                <td class="text-right">-${formatCurrency(data.discounts)}</td>
                            </tr>
                            <tr class="table-active">
                                <td><strong>Net Sales</strong></td>
                                <td class="text-right"><strong>${formatCurrency(data.net_sales)}</strong></td>
                            </tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <h4>Cost of Goods Sold</h4>
                        <table class="table table-bordered">
                            <tr>
                                <td>Cost of Products Sold</td>
                                <td class="text-right">${formatCurrency(data.cost_of_goods)}</td>
                            </tr>
                            <tr class="table-active">
                                <td><strong>Gross Profit</strong></td>
                                <td class="text-right"><strong>${formatCurrency(data.gross_profit)}</strong></td>
                            </tr>
                        </table>
                        
                        <h4>Operating Expenses</h4>
                        <table class="table table-bordered">
                            <tr>
                                <td>Total Operating Expenses</td>
                                <td class="text-right">${formatCurrency(data.operating_expenses)}</td>
                            </tr>
                            <tr class="table-active">
                                <td><strong>Net Profit</strong></td>
                                <td class="text-right"><strong>${formatCurrency(data.net_profit)}</strong></td>
                            </tr>
                        </table>
                    </div>
                </div>
                
                <div class="row mt-4">
                    <div class="col-md-12">
                        <div class="alert alert-info">
                            <strong>Profit Margin:</strong> ${data.profit_margin}% | 
                            <strong>Gross Profit Margin:</strong> ${data.gross_profit_margin}%
                        </div>
                    </div>
                </div>
            </div>
        </div>`;
    
    container.html(html);
}

// Chart creation functions
function createSalesChart(data) {
    var ctx = document.getElementById('sales-chart').getContext('2d');
    
    if (window.salesChart) {
        window.salesChart.destroy();
    }
    
    window.salesChart = new Chart(ctx, {
        type: ReportsConfig.chartType,
        data: {
            labels: data.labels,
            datasets: [{
                label: 'Sales',
                data: data.values,
                borderColor: 'rgb(75, 192, 192)',
                backgroundColor: 'rgba(75, 192, 192, 0.2)',
                tension: 0.1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return formatCurrency(value);
                        }
                    }
                }
            },
            plugins: {
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return 'Sales: ' + formatCurrency(context.parsed.y);
                        }
                    }
                }
            }
        }
    });
}

function createPaymentMethodsChart(data) {
    var ctx = document.getElementById('payment-methods-chart').getContext('2d');
    
    if (window.paymentMethodsChart) {
        window.paymentMethodsChart.destroy();
    }
    
    var labels = data.map(function(item) { return item.name; });
    var values = data.map(function(item) { return item.amount; });
    var colors = generateColors(data.length);
    
    window.paymentMethodsChart = new Chart(ctx, {
        type: 'pie',
        data: {
            labels: labels,
            datasets: [{
                data: values,
                backgroundColor: colors,
                borderWidth: 2,
                borderColor: '#fff'
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom'
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            var total = context.dataset.data.reduce(function(a, b) { return a + b; }, 0);
                            var percentage = ((context.parsed / total) * 100).toFixed(2);
                            return context.label + ': ' + formatCurrency(context.parsed) + ' (' + percentage + '%)';
                        }
                    }
                }
            }
        }
    });
}

// Utility functions
function displayTopProducts(products) {
    var container = $('#top-products-list');
    var html = '';
    
    products.forEach(function(product, index) {
        html += `
            <div class="d-flex justify-content-between align-items-center mb-2">
                <div>
                    <strong>${index + 1}. ${product.name}</strong>
                    <br><small class="text-muted">${product.category}</small>
                </div>
                <div class="text-right">
                    <div>${product.quantity_sold} sold</div>
                    <div class="text-success">${formatCurrency(product.revenue)}</div>
                </div>
            </div>`;
    });
    
    container.html(html);
}

function generateColors(count) {
    var colors = [
        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF',
        '#FF9F40', '#FF6384', '#C9CBCF', '#4BC0C0', '#FF6384'
    ];
    
    var result = [];
    for (var i = 0; i < count; i++) {
        result.push(colors[i % colors.length]);
    }
    return result;
}

function formatCurrency(amount) {
    if (typeof amount !== 'number') {
        amount = parseFloat(amount) || 0;
    }
    return ReportsConfig.currency + amount.toFixed(2);
}

function updateDateRange(dateRange) {
    ReportsConfig.defaultDateRange = dateRange;
    saveReportSettings();
    refreshCurrentReport();
}

function refreshCurrentReport() {
    var activeReport = $('.report-nav-item.active').data('report-type');
    if (activeReport) {
        loadReport(activeReport);
    }
}

function toggleAutoRefresh() {
    if (ReportsConfig.autoRefresh) {
        refreshTimer = setInterval(function() {
            refreshCurrentReport();
        }, ReportsConfig.refreshInterval);
    } else {
        if (refreshTimer) {
            clearInterval(refreshTimer);
            refreshTimer = null;
        }
    }
    saveReportSettings();
}

function cacheReportData(reportType, data) {
    reportCache[reportType] = {
        data: data,
        timestamp: Date.now()
    };
}

function exportReport(reportType, format) {
    var dateRange = ReportsConfig.defaultDateRange;
    var locationId = $('input#location_id').val();
    
    var url = '/reports/export/' + reportType + '?format=' + format + '&date_range=' + dateRange + '&location_id=' + locationId;
    
    // Create temporary link and trigger download
    var link = document.createElement('a');
    link.href = url;
    link.download = reportType + '_' + dateRange + '.' + format;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    POSUI.showNotification('Report export started', 'success');
}

// Load Chart.js dynamically
function loadChartJS() {
    return new Promise(function(resolve, reject) {
        if (typeof Chart !== 'undefined') {
            resolve();
            return;
        }
        
        var script = document.createElement('script');
        script.src = 'https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js';
        script.onload = function() {
            console.log('✅ Chart.js loaded dynamically');
            resolve();
        };
        script.onerror = function() {
            console.error('❌ Failed to load Chart.js');
            reject();
        };
        document.head.appendChild(script);
    });
}

function initializeCharts() {
    // Initialize Chart.js if available
    if (typeof Chart !== 'undefined') {
        console.log('✅ Charts initialized');
    } else {
        console.warn('⚠️ Chart.js not loaded - charts will not be available');
        // Try to load Chart.js dynamically
        loadChartJS().then(function() {
            console.log('✅ Chart.js loaded dynamically');
        }).catch(function() {
            console.warn('❌ Failed to load Chart.js dynamically');
        });
    }
}

// Export reports module
window.POSReports = {
    // Configuration
    config: ReportsConfig,
    
    // Main functions
    loadReport: loadReport,
    refreshCurrentReport: refreshCurrentReport,
    exportReport: exportReport,
    
    // Settings
    saveReportSettings: saveReportSettings,
    loadReportSettings: loadReportSettings,
    
    // Utilities
    formatCurrency: formatCurrency,
    updateDateRange: updateDateRange,
    toggleAutoRefresh: toggleAutoRefresh
}; 