/**
 * POS Settings Module
 * Handles configuration management, system preferences, and settings persistence
 */

// Settings configuration
var SettingsConfig = {
    theme: 'default',
    language: 'en',
    currency: '$',
    timezone: 'UTC',
    dateFormat: 'Y-m-d',
    timeFormat: 'H:i:s',
    decimalPlaces: 2,
    autoSave: true,
    notifications: true,
    soundEnabled: true,
    keyboardShortcuts: true,
    debugMode: false
};

// Settings categories
var SettingsCategories = {
    general: {
        name: 'General',
        icon: 'fa fa-cog',
        settings: ['theme', 'language', 'timezone', 'dateFormat', 'timeFormat']
    },
    business: {
        name: 'Business',
        icon: 'fa fa-building',
        settings: ['currency', 'decimalPlaces', 'businessName', 'businessAddress']
    },
    pos: {
        name: 'POS',
        icon: 'fa fa-cash-register',
        settings: ['autoSave', 'defaultCustomer', 'defaultPaymentMethod', 'receiptFooter']
    },
    notifications: {
        name: 'Notifications',
        icon: 'fa fa-bell',
        settings: ['notifications', 'soundEnabled', 'emailNotifications', 'smsNotifications']
    },
    advanced: {
        name: 'Advanced',
        icon: 'fa fa-cogs',
        settings: ['debugMode', 'keyboardShortcuts', 'autoBackup', 'performanceMode']
    }
};

// Initialize settings module
$(document).ready(function() {
    setupSettingsHandlers();
    loadSettings();
    applySettings();
});

function setupSettingsHandlers() {
    // Settings modal trigger
    $(document).on('click', '.open-settings', function(e) {
        e.preventDefault();
        showSettingsModal();
    });
    
    // Settings form submission
    $(document).on('submit', '#settings-form', function(e) {
        e.preventDefault();
        saveSettingsFromForm();
    });
    
    // Settings category navigation
    $(document).on('click', '.settings-category', function(e) {
        e.preventDefault();
        var category = $(this).data('category');
        showSettingsCategory(category);
    });
    
    // Reset settings
    $(document).on('click', '.reset-settings', function(e) {
        e.preventDefault();
        resetSettings();
    });
    
    // Export settings
    $(document).on('click', '.export-settings', function(e) {
        e.preventDefault();
        exportSettings();
    });
    
    // Import settings
    $(document).on('click', '.import-settings', function(e) {
        e.preventDefault();
        importSettings();
    });
}

function loadSettings() {
    // Load from localStorage
    var savedSettings = localStorage.getItem('pos_settings');
    if (savedSettings) {
        try {
            var settings = JSON.parse(savedSettings);
            Object.assign(SettingsConfig, settings);
        } catch (e) {
            console.error('Failed to load settings:', e);
        }
    }
    
    // Load from server if available
    $.ajax({
        url: '/settings/get',
        method: 'GET',
        success: function(response) {
            if (response.success && response.data) {
                Object.assign(SettingsConfig, response.data);
                saveSettings();
            }
        },
        error: function() {
            console.warn('Failed to load settings from server, using local settings');
        }
    });
}

function saveSettings() {
    try {
        localStorage.setItem('pos_settings', JSON.stringify(SettingsConfig));
        
        // Save to server
        $.ajax({
            url: '/settings/save',
            method: 'POST',
            data: SettingsConfig,
            success: function(response) {
                if (response.success) {
                    console.log('Settings saved successfully');
                }
            },
            error: function() {
                console.warn('Failed to save settings to server');
            }
        });
    } catch (e) {
        console.error('Failed to save settings:', e);
    }
}

function applySettings() {
    // Apply theme
    applyTheme(SettingsConfig.theme);
    
    // Apply language
    applyLanguage(SettingsConfig.language);
    
    // Apply currency format
    applyCurrencyFormat(SettingsConfig.currency, SettingsConfig.decimalPlaces);
    
    // Apply date/time format
    applyDateTimeFormat(SettingsConfig.dateFormat, SettingsConfig.timeFormat);
    
    // Apply notifications
    applyNotificationSettings(SettingsConfig.notifications, SettingsConfig.soundEnabled);
    
    // Apply debug mode
    applyDebugMode(SettingsConfig.debugMode);
    
    // Apply keyboard shortcuts
    applyKeyboardShortcuts(SettingsConfig.keyboardShortcuts);
}

// Show settings modal
function showSettingsModal() {
    var modalHtml = `
        <div class="modal fade" id="settings-modal" tabindex="-1" role="dialog">
            <div class="modal-dialog modal-lg" role="document">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">
                            <i class="fa fa-cog"></i> POS Settings
                        </h5>
                        <button type="button" class="close" data-dismiss="modal">
                            <span>&times;</span>
                        </button>
                    </div>
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="nav flex-column nav-pills" id="settings-tabs" role="tablist">
                                    <a class="nav-link active settings-category" data-category="general" href="#">
                                        <i class="fa fa-cog"></i> General
                                    </a>
                                    <a class="nav-link settings-category" data-category="business" href="#">
                                        <i class="fa fa-building"></i> Business
                                    </a>
                                    <a class="nav-link settings-category" data-category="pos" href="#">
                                        <i class="fa fa-cash-register"></i> POS
                                    </a>
                                    <a class="nav-link settings-category" data-category="notifications" href="#">
                                        <i class="fa fa-bell"></i> Notifications
                                    </a>
                                    <a class="nav-link settings-category" data-category="advanced" href="#">
                                        <i class="fa fa-cogs"></i> Advanced
                                    </a>
                                </div>
                            </div>
                            <div class="col-md-9">
                                <form id="settings-form">
                                    <div id="settings-content">
                                        <!-- Settings content will be loaded here -->
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-warning reset-settings">Reset</button>
                        <button type="button" class="btn btn-info export-settings">Export</button>
                        <button type="button" class="btn btn-info import-settings">Import</button>
                        <button type="submit" form="settings-form" class="btn btn-primary">Save Settings</button>
                    </div>
                </div>
            </div>
        </div>`;
    
    // Remove existing modal if any
    $('#settings-modal').remove();
    
    // Add modal to body
    $('body').append(modalHtml);
    
    // Show general settings by default
    showSettingsCategory('general');
    
    // Show modal
    $('#settings-modal').modal('show');
}

// Show specific settings category
function showSettingsCategory(category) {
    // Update active tab
    $('.settings-category').removeClass('active');
    $('.settings-category[data-category="' + category + '"]').addClass('active');
    
    var content = '';
    
    switch (category) {
        case 'general':
            content = generateGeneralSettings();
            break;
        case 'business':
            content = generateBusinessSettings();
            break;
        case 'pos':
            content = generatePOSSettings();
            break;
        case 'notifications':
            content = generateNotificationSettings();
            break;
        case 'advanced':
            content = generateAdvancedSettings();
            break;
        default:
            content = generateGeneralSettings();
    }
    
    $('#settings-content').html(content);
    populateSettingsForm();
}

// Generate settings forms
function generateGeneralSettings() {
    return `
        <h4>General Settings</h4>
        <div class="form-group">
            <label for="theme">Theme</label>
            <select class="form-control" id="theme" name="theme">
                <option value="default">Default</option>
                <option value="dark">Dark</option>
                <option value="light">Light</option>
                <option value="blue">Blue</option>
                <option value="green">Green</option>
            </select>
        </div>
        <div class="form-group">
            <label for="language">Language</label>
            <select class="form-control" id="language" name="language">
                <option value="en">English</option>
                <option value="es">Spanish</option>
                <option value="fr">French</option>
                <option value="de">German</option>
                <option value="ar">Arabic</option>
            </select>
        </div>
        <div class="form-group">
            <label for="timezone">Timezone</label>
            <select class="form-control" id="timezone" name="timezone">
                <option value="UTC">UTC</option>
                <option value="America/New_York">Eastern Time</option>
                <option value="America/Chicago">Central Time</option>
                <option value="America/Denver">Mountain Time</option>
                <option value="America/Los_Angeles">Pacific Time</option>
            </select>
        </div>
        <div class="form-group">
            <label for="dateFormat">Date Format</label>
            <select class="form-control" id="dateFormat" name="dateFormat">
                <option value="Y-m-d">YYYY-MM-DD</option>
                <option value="m/d/Y">MM/DD/YYYY</option>
                <option value="d/m/Y">DD/MM/YYYY</option>
                <option value="Y-m-d H:i">YYYY-MM-DD HH:MM</option>
            </select>
        </div>
        <div class="form-group">
            <label for="timeFormat">Time Format</label>
            <select class="form-control" id="timeFormat" name="timeFormat">
                <option value="H:i:s">24 Hour</option>
                <option value="h:i:s A">12 Hour</option>
            </select>
        </div>`;
}

function generateBusinessSettings() {
    return `
        <h4>Business Settings</h4>
        <div class="form-group">
            <label for="currency">Currency</label>
            <select class="form-control" id="currency" name="currency">
                <option value="$">USD ($)</option>
                <option value="€">EUR (€)</option>
                <option value="£">GBP (£)</option>
                <option value="¥">JPY (¥)</option>
                <option value="₹">INR (₹)</option>
            </select>
        </div>
        <div class="form-group">
            <label for="decimalPlaces">Decimal Places</label>
            <select class="form-control" id="decimalPlaces" name="decimalPlaces">
                <option value="0">0</option>
                <option value="2">2</option>
                <option value="3">3</option>
                <option value="4">4</option>
            </select>
        </div>
        <div class="form-group">
            <label for="businessName">Business Name</label>
            <input type="text" class="form-control" id="businessName" name="businessName" placeholder="Enter business name">
        </div>
        <div class="form-group">
            <label for="businessAddress">Business Address</label>
            <textarea class="form-control" id="businessAddress" name="businessAddress" rows="3" placeholder="Enter business address"></textarea>
        </div>`;
}

function generatePOSSettings() {
    return `
        <h4>POS Settings</h4>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="autoSave" name="autoSave">
                <label class="custom-control-label" for="autoSave">Auto-save transactions</label>
            </div>
        </div>
        <div class="form-group">
            <label for="defaultCustomer">Default Customer</label>
            <select class="form-control" id="defaultCustomer" name="defaultCustomer">
                <option value="">Select default customer</option>
                <option value="walkin">Walk-in Customer</option>
            </select>
        </div>
        <div class="form-group">
            <label for="defaultPaymentMethod">Default Payment Method</label>
            <select class="form-control" id="defaultPaymentMethod" name="defaultPaymentMethod">
                <option value="cash">Cash</option>
                <option value="card">Card</option>
                <option value="bank_transfer">Bank Transfer</option>
            </select>
        </div>
        <div class="form-group">
            <label for="receiptFooter">Receipt Footer</label>
            <textarea class="form-control" id="receiptFooter" name="receiptFooter" rows="3" placeholder="Enter receipt footer text"></textarea>
        </div>`;
}

function generateNotificationSettings() {
    return `
        <h4>Notification Settings</h4>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="notifications" name="notifications">
                <label class="custom-control-label" for="notifications">Enable notifications</label>
            </div>
        </div>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="soundEnabled" name="soundEnabled">
                <label class="custom-control-label" for="soundEnabled">Enable sound notifications</label>
            </div>
        </div>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="emailNotifications" name="emailNotifications">
                <label class="custom-control-label" for="emailNotifications">Email notifications</label>
            </div>
        </div>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="smsNotifications" name="smsNotifications">
                <label class="custom-control-label" for="smsNotifications">SMS notifications</label>
            </div>
        </div>`;
}

function generateAdvancedSettings() {
    return `
        <h4>Advanced Settings</h4>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="debugMode" name="debugMode">
                <label class="custom-control-label" for="debugMode">Debug mode</label>
            </div>
        </div>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="keyboardShortcuts" name="keyboardShortcuts">
                <label class="custom-control-label" for="keyboardShortcuts">Keyboard shortcuts</label>
            </div>
        </div>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="autoBackup" name="autoBackup">
                <label class="custom-control-label" for="autoBackup">Auto backup</label>
            </div>
        </div>
        <div class="form-group">
            <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="performanceMode" name="performanceMode">
                <label class="custom-control-label" for="performanceMode">Performance mode</label>
            </div>
        </div>`;
}

// Populate form with current settings
function populateSettingsForm() {
    Object.keys(SettingsConfig).forEach(function(key) {
        var element = $('#' + key);
        if (element.length) {
            if (element.attr('type') === 'checkbox') {
                element.prop('checked', SettingsConfig[key]);
            } else {
                element.val(SettingsConfig[key]);
            }
        }
    });
}

// Save settings from form
function saveSettingsFromForm() {
    var formData = {};
    
    $('#settings-form').serializeArray().forEach(function(item) {
        if (item.name) {
            formData[item.name] = item.value;
        }
    });
    
    // Handle checkboxes
    $('#settings-form input[type="checkbox"]').each(function() {
        formData[$(this).attr('name')] = $(this).is(':checked');
    });
    
    // Update settings
    Object.assign(SettingsConfig, formData);
    
    // Save and apply
    saveSettings();
    applySettings();
    
    // Close modal
    $('#settings-modal').modal('hide');
    
    POSUI.showNotification('Settings saved successfully', 'success');
}

// Reset settings to defaults
function resetSettings() {
    if (confirm('Are you sure you want to reset all settings to defaults? This action cannot be undone.')) {
        SettingsConfig = {
            theme: 'default',
            language: 'en',
            currency: '$',
            timezone: 'UTC',
            dateFormat: 'Y-m-d',
            timeFormat: 'H:i:s',
            decimalPlaces: 2,
            autoSave: true,
            notifications: true,
            soundEnabled: true,
            keyboardShortcuts: true,
            debugMode: false
        };
        
        saveSettings();
        applySettings();
        
        POSUI.showNotification('Settings reset to defaults', 'success');
    }
}

// Export settings
function exportSettings() {
    var settingsData = {
        settings: SettingsConfig,
        exportDate: new Date().toISOString(),
        version: '1.0'
    };
    
    var dataStr = JSON.stringify(settingsData, null, 2);
    var dataBlob = new Blob([dataStr], {type: 'application/json'});
    
    var link = document.createElement('a');
    link.href = URL.createObjectURL(dataBlob);
    link.download = 'pos_settings_' + new Date().toISOString().split('T')[0] + '.json';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    POSUI.showNotification('Settings exported successfully', 'success');
}

// Import settings
function importSettings() {
    var input = document.createElement('input');
    input.type = 'file';
    input.accept = '.json';
    
    input.onchange = function(e) {
        var file = e.target.files[0];
        if (file) {
            var reader = new FileReader();
            reader.onload = function(e) {
                try {
                    var importedData = JSON.parse(e.target.result);
                    if (importedData.settings) {
                        Object.assign(SettingsConfig, importedData.settings);
                        saveSettings();
                        applySettings();
                        POSUI.showNotification('Settings imported successfully', 'success');
                    } else {
                        POSUI.showNotification('Invalid settings file', 'error');
                    }
                } catch (error) {
                    POSUI.showNotification('Failed to import settings', 'error');
                }
            };
            reader.readAsText(file);
        }
    };
    
    input.click();
}

// Apply settings functions
function applyTheme(theme) {
    // Remove existing theme classes
    $('body').removeClass('theme-default theme-dark theme-light theme-blue theme-green');
    
    // Add new theme class
    $('body').addClass('theme-' + theme);
    
    // Update CSS variables if needed
    updateThemeColors(theme);
}

function applyLanguage(language) {
    // Load language file
    $.getScript('/js/lang/' + language + '.js', function() {
        // Update page text
        updatePageLanguage();
    }).fail(function() {
        console.warn('Failed to load language file:', language);
    });
}

function applyCurrencyFormat(currency, decimalPlaces) {
    // Update currency symbol globally
    window.CURRENCY_SYMBOL = currency;
    window.DECIMAL_PLACES = decimalPlaces;
    
    // Update displayed amounts
    $('.currency-amount').each(function() {
        var amount = parseFloat($(this).data('amount')) || 0;
        $(this).text(formatCurrency(amount, currency, decimalPlaces));
    });
}

function applyDateTimeFormat(dateFormat, timeFormat) {
    // Update date/time format globally
    window.DATE_FORMAT = dateFormat;
    window.TIME_FORMAT = timeFormat;
    
    // Update displayed dates/times
    $('.formatted-date').each(function() {
        var date = $(this).data('date');
        if (date) {
            $(this).text(formatDate(date, dateFormat));
        }
    });
    
    $('.formatted-time').each(function() {
        var time = $(this).data('time');
        if (time) {
            $(this).text(formatTime(time, timeFormat));
        }
    });
}

function applyNotificationSettings(notifications, soundEnabled) {
    if (notifications) {
        // Enable notifications
        if ('Notification' in window && Notification.permission === 'granted') {
            console.log('Notifications enabled');
        } else if ('Notification' in window && Notification.permission !== 'denied') {
            Notification.requestPermission();
        }
    }
    
    if (soundEnabled) {
        // Enable sound notifications
        window.SOUND_ENABLED = true;
    } else {
        window.SOUND_ENABLED = false;
    }
}

function applyDebugMode(debugMode) {
    if (debugMode) {
        window.POS_DEBUG = true;
        console.log('Debug mode enabled');
    } else {
        window.POS_DEBUG = false;
    }
}

function applyKeyboardShortcuts(enabled) {
    if (enabled) {
        // Enable keyboard shortcuts
        setupKeyboardShortcuts();
    } else {
        // Disable keyboard shortcuts
        $(document).off('keydown.pos-shortcuts');
    }
}

// Utility functions
function updateThemeColors(theme) {
    var colors = {
        default: { primary: '#007bff', secondary: '#6c757d' },
        dark: { primary: '#343a40', secondary: '#495057' },
        light: { primary: '#f8f9fa', secondary: '#e9ecef' },
        blue: { primary: '#007bff', secondary: '#0056b3' },
        green: { primary: '#28a745', secondary: '#1e7e34' }
    };
    
    var themeColors = colors[theme] || colors.default;
    
    // Update CSS custom properties
    document.documentElement.style.setProperty('--primary-color', themeColors.primary);
    document.documentElement.style.setProperty('--secondary-color', themeColors.secondary);
}

function updatePageLanguage() {
    // Update page text based on loaded language
    $('[data-lang]').each(function() {
        var key = $(this).data('lang');
        var translation = window.LANG && window.LANG[key];
        if (translation) {
            $(this).text(translation);
        }
    });
}

function formatCurrency(amount, currency, decimalPlaces) {
    if (typeof amount !== 'number') {
        amount = parseFloat(amount) || 0;
    }
    return currency + amount.toFixed(decimalPlaces);
}

function formatDate(date, format) {
    // Simple date formatting
    var d = new Date(date);
    return d.toLocaleDateString();
}

function formatTime(time, format) {
    // Simple time formatting
    var d = new Date('2000-01-01 ' + time);
    return d.toLocaleTimeString();
}

function setupKeyboardShortcuts() {
    $(document).off('keydown.pos-shortcuts').on('keydown.pos-shortcuts', function(e) {
        // Only handle shortcuts when not typing in input fields
        if ($(e.target).is('input, textarea, select')) {
            return;
        }
        
        // Ctrl/Cmd + , : Open settings
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 188) {
            e.preventDefault();
            showSettingsModal();
        }
    });
}

// Export settings module
window.POSSettings = {
    // Configuration
    config: SettingsConfig,
    categories: SettingsCategories,
    
    // Main functions
    showSettingsModal: showSettingsModal,
    loadSettings: loadSettings,
    saveSettings: saveSettings,
    applySettings: applySettings,
    
    // Utility functions
    formatCurrency: formatCurrency,
    formatDate: formatDate,
    formatTime: formatTime,
    
    // Advanced functions
    exportSettings: exportSettings,
    importSettings: importSettings,
    resetSettings: resetSettings
}; 