/**
 * POS Stock Management Module
 * Handles stock information, warehouse management, and inventory operations
 */

// Unified stock information update function
function updateStockInfo(row, product, options = {}) {
    var stockCell = row.find('td:eq(10) .stock-info');
    if (!stockCell.length) {
        stockCell = row.find('td:eq(10)').html('<span class="stock-info text-center">-</span>').find('.stock-info');
    }
    
    var warehouseSelect = row.find('.warehouse-selector');
    var quantityInput = row.find('.pos_quantity');
    var selectedWarehouse = warehouseSelect.val();
    
    function updateStockDisplay() {
        var stockInfo = '';
        var stockClass = 'stock-available';
        var availableQty = 0;
        
        if (product.enable_stock == 1) {
            if (selectedWarehouse && product.warehouse_details) {
                // Find specific warehouse
                var warehouse = product.warehouse_details.find(wh => wh.id == selectedWarehouse);
                if (warehouse) {
                    availableQty = warehouse.available || warehouse.quantity || 0;
                    stockInfo = POSUtils.formatNumber(availableQty, 0);
                    
                    if (availableQty <= 0) {
                        stockClass = 'stock-error';
                        stockInfo += ' <small>(Out)</small>';
                    } else if (availableQty <= 5) {
                        stockClass = 'stock-warning';
                        stockInfo += ' <small>(Low)</small>';
                    } else {
                        stockClass = 'stock-available';
                    }
                } else {
                    stockInfo = '0 <small>(N/A)</small>';
                    stockClass = 'stock-error';
                }
            } else {
                // Show total stock
                availableQty = product.total_stock || 0;
                stockInfo = POSUtils.formatNumber(availableQty, 0);
                
                if (availableQty <= 0) {
                    stockClass = 'stock-error';
                    stockInfo += ' <small>(Total - Out)</small>';
                } else if (availableQty <= 10) {
                    stockClass = 'stock-warning';
                    stockInfo += ' <small>(Total - Low)</small>';
                } else {
                    stockClass = 'stock-available';
                    stockInfo += ' <small>(Total)</small>';
                }
            }
            
            // Update quantity constraints
            quantityInput.attr('data-qty_available', availableQty);
            quantityInput.attr('data-rule-max-value', availableQty);
        } else {
            stockInfo = '<span class="text-muted">N/A</span>';
        }
        
        stockCell.html(stockInfo)
                .removeClass('stock-available stock-warning stock-error')
                .addClass(stockClass);
    }
    
    // Update display when warehouse changes
    warehouseSelect.off('change.stock').on('change.stock', updateStockDisplay);
    
    // Initial display update
    updateStockDisplay();
}

function setupWarehouseChangeHandler(row, product) {
    var warehouseSelect = row.find('.warehouse-selector');
    var quantityInput = row.find('.pos_quantity');
    
    warehouseSelect.on('change', function() {
        var selectedWarehouse = $(this).val();
        
        // Update stock information
        updateStockInfo(row, product);
        
        // Show warning if requested quantity exceeds available
        var requestedQty = parseFloat(quantityInput.val()) || 0;
        var availableQty = parseFloat(quantityInput.attr('data-qty_available')) || 0;
        
        if (requestedQty > availableQty && availableQty > 0) {
            toastr.warning(`Requested quantity (${requestedQty}) exceeds available stock (${availableQty}) in selected warehouse`);
            
            // Set quantity to maximum available
            if (!$('input#is_overselling_allowed').length) {
                quantityInput.val(availableQty).trigger('change');
            }
        }
    });
}

function updateStockInfoForWarehouse(row, warehouseCode) {
    var productWarehouses = row.find('.product_warehouses').val();
    var stockCell = row.find('.stock-info');
    
    if (productWarehouses) {
        try {
            var warehouses = JSON.parse(productWarehouses);
            var warehouse = warehouses.find(w => w.code === warehouseCode);
            
            if (warehouse) {
                var qty = warehouse.available || warehouse.quantity || 0;
                var stockClass = qty > 10 ? 'stock-available' : (qty > 0 ? 'stock-warning' : 'stock-error');
                
                stockCell.html(POSUtils.formatNumber(qty, 0))
                        .removeClass('stock-available stock-warning stock-error')
                        .addClass(stockClass);
                
                // Update quantity constraints
                var qtyInput = row.find('.pos_quantity');
                qtyInput.attr('data-qty_available', qty);
                qtyInput.attr('data-rule-max-value', qty);
                qtyInput.attr('data-msg-max-value', 'Max available in ' + warehouseCode + ': ' + qty);
            } else {
                stockCell.html('0').addClass('stock-error');
            }
        } catch (e) {
            POSUtils.handleError(e, 'updateStockInfoForWarehouse');
        }
    }
}

function updateStockDisplay() {
    $('#pos_table tbody tr').each(function() {
        var row = $(this);
        var productId = row.find('.product_id').val();
        
        if (productId) {
            // Get product data from cache or make AJAX call
            var product = window.productDataStore ? window.productDataStore[productId] : null;
            if (product) {
                updateStockInfo(row, product);
            }
        }
    });
}

function checkStockAvailability(productId, quantity, warehouseId) {
    return new Promise(function(resolve, reject) {
        $.ajax({
            url: '/products/check-stock',
            method: 'POST',
            data: {
                product_id: productId,
                quantity: quantity,
                warehouse_id: warehouseId
            },
            success: function(response) {
                resolve(response);
            },
            error: function(xhr, status, error) {
                reject(error);
            }
        });
    });
}

function updateStockAfterSale(transactionId) {
    $.ajax({
        url: '/pos/update-stock-after-sale',
        method: 'POST',
        data: {
            transaction_id: transactionId
        },
        success: function(response) {
            if (response.success) {
                updateStockDisplay();
                toastr.success('Stock updated successfully');
            } else {
                toastr.error('Failed to update stock');
            }
        },
        error: function() {
            toastr.error('Error updating stock');
        }
    });
}

// Export stock module
window.POSStock = {
    updateStockInfo: updateStockInfo,
    setupWarehouseChangeHandler: setupWarehouseChangeHandler,
    updateStockInfoForWarehouse: updateStockInfoForWarehouse,
    updateStockDisplay: updateStockDisplay,
    checkStockAvailability: checkStockAvailability,
    updateStockAfterSale: updateStockAfterSale
}; 