/**
 * POS Testing Module
 * Provides testing utilities and test cases for the modular POS system
 */

// Test configuration
var TestConfig = {
    enabled: false,
    verbose: false,
    timeout: 5000,
    autoRun: false
};

// Test results storage
var testResults = {
    passed: 0,
    failed: 0,
    total: 0,
    details: []
};

// Initialize testing module
$(document).ready(function() {
    if (TestConfig.enabled) {
        setupTestEnvironment();
        if (TestConfig.autoRun) {
            runAllTests();
        }
    }
});

function setupTestEnvironment() {
    // Create test container
    if ($('#pos-test-container').length === 0) {
        $('body').append('<div id="pos-test-container" style="display:none;"></div>');
    }
    
    // Setup test utilities
    window.POSTest = {
        runAll: runAllTests,
        runModule: runModuleTests,
        runFunction: runFunctionTest,
        getResults: getTestResults,
        clearResults: clearTestResults,
        enable: enableTesting,
        disable: disableTesting
    };
    
    console.log('🧪 POS Testing module initialized');
}

// Test runner functions
function runAllTests() {
    console.log('🚀 Running all POS tests...');
    clearTestResults();
    
    var testSuites = [
        runCoreTests,
        runUtilsTests,
        runCalculationsTests,
        runStockTests,
        runUITests,
        runAPITests,
        runEventsTests
    ];
    
    var promises = testSuites.map(function(testSuite) {
        return new Promise(function(resolve) {
            try {
                testSuite();
                resolve();
            } catch (error) {
                logTestResult('Test Suite', false, error.message);
                resolve();
            }
        });
    });
    
    Promise.all(promises).then(function() {
        displayTestResults();
    });
}

function runModuleTests(moduleName) {
    console.log('🧪 Running tests for module:', moduleName);
    
    switch (moduleName) {
        case 'core':
            runCoreTests();
            break;
        case 'utils':
            runUtilsTests();
            break;
        case 'calculations':
            runCalculationsTests();
            break;
        case 'stock':
            runStockTests();
            break;
        case 'ui':
            runUITests();
            break;
        case 'api':
            runAPITests();
            break;
        case 'events':
            runEventsTests();
            break;
        default:
            console.error('Unknown module:', moduleName);
    }
}

function runFunctionTest(functionName, testFunction) {
    try {
        testFunction();
        logTestResult(functionName, true);
    } catch (error) {
        logTestResult(functionName, false, error.message);
    }
}

// Core module tests
function runCoreTests() {
    console.log('Testing Core Module...');
    
    // Test global variables
    runFunctionTest('Global Variables', function() {
        assert(typeof pos_total_row !== 'undefined', 'pos_total_row should be defined');
        assert(typeof __read_number !== 'undefined', '__read_number should be defined');
        assert(typeof __write_number !== 'undefined', '__write_number should be defined');
    });
    
    // Test initialization
    runFunctionTest('Initialization', function() {
        assert(typeof POSCore !== 'undefined', 'POSCore should be defined');
        assert(typeof POSCore.initializePOS === 'function', 'initializePOS should be a function');
    });
    
    // Test AJAX setup
    runFunctionTest('AJAX Setup', function() {
        assert($.ajaxSettings.timeout > 0, 'AJAX timeout should be set');
        assert($.ajaxSettings.headers, 'AJAX headers should be set');
    });
}

// Utils module tests
function runUtilsTests() {
    console.log('Testing Utils Module...');
    
    // Test similarity calculation
    runFunctionTest('Similarity Calculation', function() {
        var similarity = POSUtils.calculateSimilarity('laptop', 'laptops');
        assert(similarity > 0.5, 'Similarity should be greater than 0.5');
        
        var exactMatch = POSUtils.calculateSimilarity('test', 'test');
        assert(exactMatch === 1, 'Exact match should return 1');
    });
    
    // Test number formatting
    runFunctionTest('Number Formatting', function() {
        var formatted = POSUtils.formatNumber(1234.56, 2);
        assert(formatted.includes('1,234'), 'Number should be formatted with commas');
        
        var zeroFormatted = POSUtils.formatNumber(0, 2);
        assert(zeroFormatted === '0.00', 'Zero should be formatted correctly');
    });
    
    // Test debounce function
    runFunctionTest('Debounce Function', function() {
        var callCount = 0;
        var debouncedFn = POSUtils.debounce(function() {
            callCount++;
        }, 100);
        
        // Call multiple times quickly
        debouncedFn();
        debouncedFn();
        debouncedFn();
        
        assert(callCount === 0, 'Debounced function should not execute immediately');
    });
    
    // Test error handling
    runFunctionTest('Error Handling', function() {
        var errorLogged = false;
        var originalConsole = console.error;
        console.error = function() {
            errorLogged = true;
        };
        
        POSUtils.handleError(new Error('Test error'), 'testFunction');
        
        console.error = originalConsole;
        assert(errorLogged, 'Error should be logged');
    });
}

// Calculations module tests
function runCalculationsTests() {
    console.log('Testing Calculations Module...');
    
    // Test price calculation
    runFunctionTest('Price Calculation', function() {
        var price = POSCalculations.calculatePrice(100, 10, 5);
        assert(price > 100, 'Price with tax should be greater than base price');
        
        var priceNoTax = POSCalculations.calculatePrice(100, 0, 5);
        assert(priceNoTax < 100, 'Price with discount should be less than base price');
    });
    
    // Test total calculation
    runFunctionTest('Total Calculation', function() {
        // Mock table structure
        var mockTable = createMockTable();
        
        var total = POSCalculations.calculateTableTotal(mockTable);
        assert(total > 0, 'Total should be greater than 0');
    });
    
    // Test discount calculation
    runFunctionTest('Discount Calculation', function() {
        var discount = POSCalculations.calculateDiscount(100, 10);
        assert(discount === 10, '10% discount on 100 should be 10');
        
        var zeroDiscount = POSCalculations.calculateDiscount(100, 0);
        assert(zeroDiscount === 0, '0% discount should be 0');
    });
}

// Stock module tests
function runStockTests() {
    console.log('Testing Stock Module...');
    
    // Test stock info update
    runFunctionTest('Stock Info Update', function() {
        var mockRow = createMockTableRow();
        var mockProduct = {
            id: 1,
            name: 'Test Product',
            enable_stock: 1,
            total_stock: 50
        };
        
        POSStock.updateStockInfo(mockRow, mockProduct);
        
        var stockCell = mockRow.find('.stock-info');
        assert(stockCell.length > 0, 'Stock info should be displayed');
    });
    
    // Test stock availability check
    runFunctionTest('Stock Availability Check', function() {
        // Mock API response
        var mockResponse = { available: true, quantity: 10 };
        
        // Test with available stock
        assert(mockResponse.available === true, 'Stock should be available');
        assert(mockResponse.quantity > 0, 'Quantity should be greater than 0');
    });
}

// UI module tests
function runUITests() {
    console.log('Testing UI Module...');
    
    // Test notification system
    runFunctionTest('Notification System', function() {
        assert(typeof POSUI.showNotification === 'function', 'showNotification should be a function');
        
        // Test notification types
        var types = ['success', 'error', 'warning', 'info'];
        types.forEach(function(type) {
            assert(typeof POSUI.showNotification === 'function', 'showNotification should handle ' + type);
        });
    });
    
    // Test loading states
    runFunctionTest('Loading States', function() {
        var testElement = $('<button id="test-loading">Test</button>');
        $('body').append(testElement);
        
        POSUI.showLoading('#test-loading', 'Loading...');
        assert($('#test-loading').prop('disabled'), 'Element should be disabled when loading');
        
        POSUI.hideLoading('#test-loading');
        assert(!$('#test-loading').prop('disabled'), 'Element should be enabled after loading');
        
        testElement.remove();
    });
    
    // Test modal functions
    runFunctionTest('Modal Functions', function() {
        assert(typeof POSUI.openModal === 'function', 'openModal should be a function');
        assert(typeof POSUI.closeModal === 'function', 'closeModal should be a function');
    });
}

// API module tests
function runAPITests() {
    console.log('Testing API Module...');
    
    // Test API configuration
    runFunctionTest('API Configuration', function() {
        assert(typeof POSAPI.config === 'object', 'API config should be an object');
        assert(POSAPI.config.timeout > 0, 'API timeout should be positive');
        assert(POSAPI.config.retryAttempts > 0, 'Retry attempts should be positive');
    });
    
    // Test cache functions
    runFunctionTest('Cache Functions', function() {
        assert(typeof POSAPI.clearCache === 'function', 'clearCache should be a function');
        assert(typeof POSAPI.getCachedResponse === 'function', 'getCachedResponse should be a function');
    });
    
    // Test request function
    runFunctionTest('Request Function', function() {
        assert(typeof POSAPI.makeRequest === 'function', 'makeRequest should be a function');
    });
}

// Events module tests
function runEventsTests() {
    console.log('Testing Events Module...');
    
    // Test event setup functions
    runFunctionTest('Event Setup Functions', function() {
        assert(typeof POSEvents.setupProductEvents === 'function', 'setupProductEvents should be a function');
        assert(typeof POSEvents.setupTableEvents === 'function', 'setupTableEvents should be a function');
        assert(typeof POSEvents.setupFormEvents === 'function', 'setupFormEvents should be a function');
    });
}

// Test utilities
function assert(condition, message) {
    if (!condition) {
        throw new Error(message || 'Assertion failed');
    }
}

function logTestResult(testName, passed, errorMessage) {
    testResults.total++;
    
    if (passed) {
        testResults.passed++;
        if (TestConfig.verbose) {
            console.log('✅ PASS:', testName);
        }
    } else {
        testResults.failed++;
        console.error('❌ FAIL:', testName, errorMessage);
    }
    
    testResults.details.push({
        name: testName,
        passed: passed,
        error: errorMessage,
        timestamp: new Date()
    });
}

function getTestResults() {
    return {
        summary: {
            total: testResults.total,
            passed: testResults.passed,
            failed: testResults.failed,
            successRate: testResults.total > 0 ? (testResults.passed / testResults.total * 100).toFixed(2) : 0
        },
        details: testResults.details
    };
}

function clearTestResults() {
    testResults = {
        passed: 0,
        failed: 0,
        total: 0,
        details: []
    };
}

function displayTestResults() {
    var results = getTestResults();
    
    console.log('📊 Test Results:');
    console.log('Total Tests:', results.summary.total);
    console.log('Passed:', results.summary.passed);
    console.log('Failed:', results.summary.failed);
    console.log('Success Rate:', results.summary.successRate + '%');
    
    if (results.summary.failed > 0) {
        console.log('❌ Failed Tests:');
        results.details.filter(function(test) {
            return !test.passed;
        }).forEach(function(test) {
            console.log('  -', test.name + ':', test.error);
        });
    }
    
    // Display in UI if available
    if (typeof POSUI !== 'undefined') {
        var message = 'Tests completed: ' + results.summary.passed + '/' + results.summary.total + ' passed';
        var type = results.summary.failed > 0 ? 'warning' : 'success';
        POSUI.showNotification(message, type);
    }
}

// Mock data creation
function createMockTable() {
    var table = $('<table id="pos_table"><tbody></tbody></table>');
    
    // Add mock rows
    for (var i = 0; i < 3; i++) {
        var row = createMockTableRow();
        table.find('tbody').append(row);
    }
    
    return table;
}

function createMockTableRow() {
    var row = $('<tr></tr>');
    
    // Add mock cells
    row.append('<td><input type="hidden" class="product_id" value="1"></td>');
    row.append('<td><input type="text" class="pos_quantity" value="2"></td>');
    row.append('<td><input type="text" class="pos_unit_price" value="100"></td>');
    row.append('<td><input type="text" class="pos_unit_price_inc_tax" value="110"></td>');
    row.append('<td><input type="text" class="pos_line_total" value="220"></td>');
    row.append('<td><span class="pos_line_total_text">$220.00</span></td>');
    row.append('<td><select class="tax_id"><option data-rate="10">10%</option></select></td>');
    row.append('<td><span class="stock-info">50</span></td>');
    
    return row;
}

// Test control functions
function enableTesting() {
    TestConfig.enabled = true;
    console.log('🧪 Testing enabled');
}

function disableTesting() {
    TestConfig.enabled = false;
    console.log('🧪 Testing disabled');
}

// Export testing module
window.POSTest = {
    runAll: runAllTests,
    runModule: runModuleTests,
    runFunction: runFunctionTest,
    getResults: getTestResults,
    clearResults: clearTestResults,
    enable: enableTesting,
    disable: disableTesting,
    config: TestConfig
};

// Auto-enable testing in development
if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
    enableTesting();
    TestConfig.verbose = true;
} 