/**
 * POS UI Module
 * Handles user interface interactions, visual feedback, and UI state management
 */

// UI state management
var UIState = {
    isModalOpen: false,
    activeTab: 'products',
    searchFocused: false,
    keyboardMode: false,
    loadingStates: {}
};

// Initialize UI components
$(document).ready(function() {
    setupUIComponents();
    setupKeyboardShortcuts();
    setupVisualFeedback();
    setupResponsiveUI();
});

function setupUIComponents() {
    // Setup modals
    setupModalHandlers();
    
    // Setup tabs
    setupTabHandlers();
    
    // Setup search focus
    setupSearchFocus();
    
    // Setup loading states
    setupLoadingStates();
    
    // Setup notifications
    setupNotifications();
}

function setupModalHandlers() {
    // Modal open handlers
    $(document).on('click', '[data-toggle="modal"]', function(e) {
        e.preventDefault();
        var target = $(this).data('target');
        openModal(target);
    });
    
    // Modal close handlers
    $(document).on('click', '.modal-close, .modal-backdrop', function() {
        closeModal();
    });
    
    // ESC key to close modal
    $(document).on('keydown', function(e) {
        if (e.keyCode === 27 && UIState.isModalOpen) {
            closeModal();
        }
    });
}

function setupTabHandlers() {
    $('.nav-tabs a').on('click', function(e) {
        e.preventDefault();
        var tabId = $(this).attr('href');
        switchTab(tabId);
    });
}

function setupSearchFocus() {
    $('#search_product').on('focus', function() {
        UIState.searchFocused = true;
        $(this).addClass('search-focused');
        showSearchSuggestions();
    }).on('blur', function() {
        UIState.searchFocused = false;
        $(this).removeClass('search-focused');
        hideSearchSuggestions();
    });
}

function setupLoadingStates() {
    // Add loading spinner to buttons
    $(document).on('click', '.btn-loading', function() {
        var $btn = $(this);
        var originalText = $btn.text();
        
        $btn.prop('disabled', true)
            .html('<i class="fa fa-spinner fa-spin"></i> Loading...');
        
        // Store original text for restoration
        $btn.data('original-text', originalText);
        
        // Auto-restore after 5 seconds (fallback)
        setTimeout(function() {
            if ($btn.prop('disabled')) {
                restoreButton($btn);
            }
        }, 5000);
    });
}

function setupNotifications() {
    // Setup toastr configuration
    if (typeof toastr !== 'undefined') {
        toastr.options = {
            closeButton: true,
            progressBar: true,
            positionClass: 'toast-top-right',
            timeOut: 4000,
            extendedTimeOut: 1000
        };
    }
}

function setupKeyboardShortcuts() {
    $(document).on('keydown', function(e) {
        // Only handle shortcuts when not typing in input fields
        if ($(e.target).is('input, textarea, select')) {
            return;
        }
        
        // Ctrl/Cmd + Enter: Finalize sale
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 13) {
            e.preventDefault();
            $('#pos-finalize').click();
        }
        
        // Ctrl/Cmd + S: Save draft
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 83) {
            e.preventDefault();
            $('#pos-draft').click();
        }
        
        // Ctrl/Cmd + F: Focus search
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 70) {
            e.preventDefault();
            $('#search_product').focus();
        }
        
        // F1: Help
        if (e.keyCode === 112) {
            e.preventDefault();
            showHelp();
        }
        
        // F2: Quick product add
        if (e.keyCode === 113) {
            e.preventDefault();
            showQuickProductModal();
        }
        
        // F3: Customer selection
        if (e.keyCode === 114) {
            e.preventDefault();
            $('#customer_id').select2('open');
        }
        
        // F4: Payment modal
        if (e.keyCode === 115) {
            e.preventDefault();
            showPaymentModal();
        }
        
        // F5: Refresh
        if (e.keyCode === 116) {
            e.preventDefault();
            refreshPOS();
        }
    });
}

function setupVisualFeedback() {
    // Success feedback
    $(document).on('click', '.btn-success', function() {
        showSuccessFeedback($(this));
    });
    
    // Error feedback
    $(document).on('click', '.btn-danger', function() {
        showErrorFeedback($(this));
    });
    
    // Warning feedback
    $(document).on('click', '.btn-warning', function() {
        showWarningFeedback($(this));
    });
}

function setupResponsiveUI() {
    // Handle window resize
    $(window).on('resize', debounce(function() {
        adjustLayoutForScreenSize();
    }, 250));
    
    // Initial layout adjustment
    adjustLayoutForScreenSize();
}

// Modal functions
function openModal(modalId) {
    var $modal = $(modalId);
    if ($modal.length) {
        $modal.modal('show');
        UIState.isModalOpen = true;
        
        // Focus first input in modal
        setTimeout(function() {
            $modal.find('input:first').focus();
        }, 500);
    }
}

function closeModal() {
    $('.modal').modal('hide');
    UIState.isModalOpen = false;
}

// Tab functions
function switchTab(tabId) {
    $('.nav-tabs a[href="' + tabId + '"]').tab('show');
    UIState.activeTab = tabId.replace('#', '');
    
    // Update active state
    $('.nav-tabs a').removeClass('active');
    $('.nav-tabs a[href="' + tabId + '"]').addClass('active');
}

// Search functions
function showSearchSuggestions() {
    if (UIState.searchFocused) {
        $('#search_suggestions').show();
    }
}

function hideSearchSuggestions() {
    $('#search_suggestions').hide();
}

// Loading state functions
function showLoading(element, text) {
    var $element = $(element);
    var originalContent = $element.html();
    
    $element.prop('disabled', true)
            .html('<i class="fa fa-spinner fa-spin"></i> ' + (text || 'Loading...'))
            .data('original-content', originalContent);
    
    UIState.loadingStates[element] = true;
}

function hideLoading(element) {
    var $element = $(element);
    var originalContent = $element.data('original-content');
    
    if (originalContent) {
        $element.prop('disabled', false)
                .html(originalContent)
                .removeData('original-content');
    }
    
    UIState.loadingStates[element] = false;
}

function restoreButton($btn) {
    var originalText = $btn.data('original-text');
    if (originalText) {
        $btn.prop('disabled', false).text(originalText);
    }
}

// Visual feedback functions
function showSuccessFeedback($element) {
    $element.addClass('btn-success-feedback');
    setTimeout(function() {
        $element.removeClass('btn-success-feedback');
    }, 1000);
}

function showErrorFeedback($element) {
    $element.addClass('btn-error-feedback');
    setTimeout(function() {
        $element.removeClass('btn-error-feedback');
    }, 1000);
}

function showWarningFeedback($element) {
    $element.addClass('btn-warning-feedback');
    setTimeout(function() {
        $element.removeClass('btn-warning-feedback');
    }, 1000);
}

// Keyboard shortcut functions
function showHelp() {
    openModal('#help-modal');
}

function showQuickProductModal() {
    openModal('#quick-product-modal');
}

function showPaymentModal() {
    openModal('#payment-modal');
}

function refreshPOS() {
    if (confirm('Are you sure you want to refresh? Unsaved changes will be lost.')) {
        location.reload();
    }
}

// Layout functions
function adjustLayoutForScreenSize() {
    var windowWidth = $(window).width();
    
    if (windowWidth < 768) {
        // Mobile layout
        $('body').addClass('mobile-layout');
        $('.pos-sidebar').addClass('collapsed');
    } else if (windowWidth < 1024) {
        // Tablet layout
        $('body').addClass('tablet-layout');
    } else {
        // Desktop layout
        $('body').removeClass('mobile-layout tablet-layout');
    }
}

// Notification functions
function showNotification(message, type, options) {
    if (typeof toastr !== 'undefined') {
        switch (type) {
            case 'success':
                toastr.success(message, options);
                break;
            case 'error':
                toastr.error(message, options);
                break;
            case 'warning':
                toastr.warning(message, options);
                break;
            case 'info':
                toastr.info(message, options);
                break;
            default:
                toastr.info(message, options);
        }
    } else {
        // Fallback to alert
        alert(message);
    }
}

// UI utility functions
function toggleElement(element, show) {
    if (show === undefined) {
        show = !$(element).is(':visible');
    }
    
    if (show) {
        $(element).show();
    } else {
        $(element).hide();
    }
}

function updateElementText(element, text) {
    $(element).text(text);
}

function updateElementHTML(element, html) {
    $(element).html(html);
}

function addElementClass(element, className) {
    $(element).addClass(className);
}

function removeElementClass(element, className) {
    $(element).removeClass(className);
}

function setElementAttribute(element, attribute, value) {
    $(element).attr(attribute, value);
}

// Legacy UI functions
function resetForm() {
    // Reset POS form to initial state
    if (pos_form_obj) {
        pos_form_obj[0].reset();
        $('#pos_table tbody').empty();
        if (window.POSCalculations && window.POSCalculations.pos_total_row) {
            window.POSCalculations.pos_total_row();
        }
        console.log('✅ POS form reset');
    }
}

function setPaymentTypeDropdown() {
    // Set payment type dropdown based on location
    var locationId = $('select#select_location_id').val();
    if (locationId) {
        $.ajax({
            url: '/business/get-payment-types',
            data: { location_id: locationId },
            success: function(data) {
                var $paymentTypeSelect = $('select[name="payment_type"]');
                $paymentTypeSelect.empty();
                $paymentTypeSelect.append('<option value="">Select Payment Type</option>');
                
                data.forEach(function(type) {
                    $paymentTypeSelect.append('<option value="' + type.id + '">' + type.name + '</option>');
                });
            }
        });
    }
}

// Export UI module
window.POSUI = {
    // State management
    getState: function() { return UIState; },
    setState: function(key, value) { UIState[key] = value; },
    
    // Modal functions
    openModal: openModal,
    closeModal: closeModal,
    
    // Tab functions
    switchTab: switchTab,
    
    // Loading functions
    showLoading: showLoading,
    hideLoading: hideLoading,
    
    // Feedback functions
    showSuccessFeedback: showSuccessFeedback,
    showErrorFeedback: showErrorFeedback,
    showWarningFeedback: showWarningFeedback,
    
    // Notification functions
    showNotification: showNotification,
    
    // Utility functions
    toggleElement: toggleElement,
    updateElementText: updateElementText,
    updateElementHTML: updateElementHTML,
    addElementClass: addElementClass,
    removeElementClass: removeElementClass,
    setElementAttribute: setElementAttribute,
    
    // Legacy functions
    resetForm: resetForm,
    setPaymentTypeDropdown: setPaymentTypeDropdown
};

// Export legacy functions for backward compatibility
window.reset_pos_form = resetForm;
window.set_payment_type_dropdown = setPaymentTypeDropdown; 