/**
 * POS Utilities Module
 * Contains helper functions, utilities, and common operations
 */

// Unified similarity calculation function (Levenshtein Distance)
function calculateSimilarity(str1, str2) {
    if (!str1 || !str2) return 0;
    
    str1 = str1.toLowerCase().trim();
    str2 = str2.toLowerCase().trim();
    
    // Exact match
    if (str1 === str2) return 100;
    
    // Contains match
    if (str1.includes(str2) || str2.includes(str1)) return 90;
    
    // Clean comparison
    var clean1 = str1.replace(/[\s\-_\.]/g, '');
    var clean2 = str2.replace(/[\s\-_\.]/g, '');
    
    if (clean1 === clean2) return 85;
    if (clean1.includes(clean2) || clean2.includes(clean1)) return 80;
    
    // Levenshtein distance calculation
    var matrix = [];
    for (var i = 0; i <= str2.length; i++) {
        matrix[i] = [i];
    }
    
    for (var j = 0; j <= str1.length; j++) {
        matrix[0][j] = j;
    }
    
    for (var i = 1; i <= str2.length; i++) {
        for (var j = 1; j <= str1.length; j++) {
            if (str2.charAt(i - 1) === str1.charAt(j - 1)) {
                matrix[i][j] = matrix[i - 1][j - 1];
            } else {
                matrix[i][j] = Math.min(
                    matrix[i - 1][j - 1] + 1,
                    matrix[i][j - 1] + 1,
                    matrix[i - 1][j] + 1
                );
            }
        }
    }
    
    var distance = matrix[str2.length][str1.length];
    var maxLength = Math.max(str1.length, str2.length);
    
    return Math.round(((maxLength - distance) / maxLength) * 100);
}

// Format number helper
function formatNumber(number, decimals = 2) {
    return parseFloat(number || 0).toFixed(decimals);
}

// Format currency helper
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: 'USD',
        minimumFractionDigits: 2
    }).format(amount);
}

// Auto-resize input based on content
function autoResizeInput(input) {
    var value = input.val();
    var length = value.length;
    var minWidth = 60;
    var maxWidth = 200;
    var charWidth = 8; // Approximate character width in pixels
    
    var newWidth = Math.max(minWidth, Math.min(maxWidth, length * charWidth + 20));
    input.css('width', newWidth + 'px');
}

// Copy selected cells functionality
function copySelectedCells() {
    var selectedCells = $('#pos_table').find('.selected');
    if (selectedCells.length === 0) return;
    
    var copyData = [];
    selectedCells.each(function() {
        var cellValue = $(this).is('input, select') ? $(this).val() : $(this).text();
        copyData.push(cellValue);
    });
    
    // Copy to clipboard (modern browsers)
    if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(copyData.join('\t'));
    } else {
        // Fallback for older browsers
        var textArea = document.createElement('textarea');
        textArea.value = copyData.join('\t');
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
    }
    
    toastr.success('Copied ' + copyData.length + ' cells');
}

// Enhanced paste functionality for Excel-like data
function handleExcelPaste(pastedData, startRow) {
    var rows = pastedData.split('\n').filter(row => row.trim());
    var startRowIndex = startRow.index();
    
    $('.paste-indicator').show();
    
    rows.forEach(function(rowData, rowOffset) {
        var cells = rowData.split('\t');
        var targetRowIndex = startRowIndex + rowOffset;
        var targetRow = $('#pos_table tbody tr').eq(targetRowIndex);
        
        // Create new row if needed
        if (targetRow.length === 0 || targetRow.hasClass('empty-row')) {
            if (targetRow.hasClass('empty-row')) {
                targetRow.removeClass('empty-row');
            } else {
                addEmptyProductRow();
                targetRow = $('#pos_table tbody tr').eq(targetRowIndex);
            }
        }
        
        // Process each cell in the row
        cells.forEach(function(cellData, cellOffset) {
            cellData = cellData.trim();
            if (!cellData) return;
            
            switch(cellOffset) {
                case 0: // Product name/search
                    if (cellData && !targetRow.find('.product_id').val()) {
                        targetRow.find('.product-search-input').val(cellData);
                        setTimeout(function() {
                            targetRow.find('.product-search-input').trigger('input');
                        }, rowOffset * 100);
                    }
                    break;
                    
                case 1: // SKU (read-only, skip)
                    break;
                    
                case 2: // Quantity
                    if (!isNaN(cellData) && parseFloat(cellData) > 0) {
                        targetRow.find('td:eq(3) input').val(cellData);
                    }
                    break;
                    
                case 3: // Unit (handle later when product is loaded)
                    break;
                    
                case 4: // Price USD
                    if (!isNaN(cellData) && parseFloat(cellData) > 0) {
                        targetRow.find('td:eq(5) input').val(cellData).trigger('input');
                    }
                    break;
                    
                case 5: // Price IQD
                    if (!isNaN(cellData) && parseFloat(cellData) > 0) {
                        targetRow.find('td:eq(6) input').val(cellData).trigger('input');
                    }
                    break;
                    
                case 6: // Discount %
                    if (!isNaN(cellData) && parseFloat(cellData) >= 0) {
                        targetRow.find('td:eq(7) input').val(cellData).trigger('input');
                    }
                    break;
            }
        });
    });
    
    setTimeout(function() {
        $('.paste-indicator').hide();
        toastr.success('Pasted ' + rows.length + ' rows');
    }, 1000);
}

// Unified serial numbers update
function updateSerialNumbers() {
    $('#pos_table tbody tr').each(function(index) {
        $(this).find('.serial-number').text(index + 1);
        $(this).attr('data-row_index', index);
        
        // Update input names
        var inputs = $(this).find('input, select');
        inputs.each(function() {
            var name = $(this).attr('name');
            if (name && name.includes('[') && name.includes(']')) {
                var newName = name.replace(/\[\d+\]/, '[' + index + ']');
                $(this).attr('name', newName);
            }
        });
    });
}

// Memory optimization function
function optimizeMemoryUsage() {
    // Clean old data
    if (window.productDataStore) {
        var storeSize = Object.keys(window.productDataStore).length;
        if (storeSize > 100) {
            // Keep only last 50 products
            var keys = Object.keys(window.productDataStore);
            var keysToRemove = keys.slice(0, keys.length - 50);
            keysToRemove.forEach(key => {
                delete window.productDataStore[key];
            });
        }
    }
    
    // Clean unused event handlers
    $('*').off('.unused-events');
}

// Validation utilities
function validateInput(value, type) {
    switch(type) {
        case 'number':
            return !isNaN(parseFloat(value)) && isFinite(value);
        case 'email':
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(value);
        case 'phone':
            var phoneRegex = /^[\+]?[1-9][\d]{0,15}$/;
            return phoneRegex.test(value.replace(/[\s\-\(\)]/g, ''));
        default:
            return value && value.trim().length > 0;
    }
}

// Error handling utility
function handleError(error, context) {
    console.error('POS Error in ' + context + ':', error);
    
    // Show user-friendly error message
    var message = 'An error occurred. Please try again.';
    if (error.message) {
        message = error.message;
    }
    
    toastr.error(message);
    
    // Log to server if needed
    if (window.logErrorToServer) {
        window.logErrorToServer(error, context);
    }
}

// Debounce utility
function debounce(func, wait, immediate) {
    var timeout;
    return function executedFunction() {
        var context = this;
        var args = arguments;
        var later = function() {
            timeout = null;
            if (!immediate) func.apply(context, args);
        };
        var callNow = immediate && !timeout;
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
        if (callNow) func.apply(context, args);
    };
}

// Throttle utility
function throttle(func, limit) {
    var inThrottle;
    return function() {
        var args = arguments;
        var context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// Export utilities module
window.POSUtils = {
    calculateSimilarity: calculateSimilarity,
    formatNumber: formatNumber,
    formatCurrency: formatCurrency,
    autoResizeInput: autoResizeInput,
    copySelectedCells: copySelectedCells,
    handleExcelPaste: handleExcelPaste,
    updateSerialNumbers: updateSerialNumbers,
    optimizeMemoryUsage: optimizeMemoryUsage,
    validateInput: validateInput,
    handleError: handleError,
    debounce: debounce,
    throttle: throttle
}; 