/**
 * POS System JavaScript - Optimized and Consolidated Version
 * 
 * This file has been optimized to:
 * - Consolidate duplicated functions and event handlers
 * - Centralize status management (quotation/draft/final)
 * - Improve form validation with centralized validators
 * - Reduce code duplication and improve maintainability
 * - Maintain backward compatibility with existing code
 * 
 * Key improvements:
 * - StatusManager: Centralized status handling
 * - FormValidator: Centralized form validation
 * - AjaxHandler: Centralized AJAX request handling
 * - ButtonHandlers: Consolidated button event handlers
 * - UtilityFunctions: Centralized utility functions
 * 
 
 */

var global_brand_id = null;
var global_p_category_id = null;

// Centralized IQD Exchange Rate Configuration
// To change the exchange rate, simply update the value below
// You can also set window.customIQDExchangeRate to override this value dynamically
var IQD_EXCHANGE_RATE = 1410; // USD to IQD exchange rate

// Helper function to get IQD exchange rate
function getIQDExchangeRate() {
    // Check if there's a custom rate set in the system
    if (window.customIQDExchangeRate) {
        return window.customIQDExchangeRate;
    }
    // Check if there's a rate in localStorage (for persistence)
    var storedRate = localStorage.getItem('iqd_exchange_rate');
    if (storedRate) {
        return parseFloat(storedRate);
    }
    return IQD_EXCHANGE_RATE;
}

// Function to update exchange rate
function updateIQDExchangeRate(newRate) {
    if (newRate && newRate > 0) {
        window.customIQDExchangeRate = newRate;
        localStorage.setItem('iqd_exchange_rate', newRate);
        // Trigger event for any listeners
        $(document).trigger('iqd-exchange-rate-changed', newRate);
        return true;
    }
    return false;
}

// Helper function to convert USD to IQD
function convertUSDToIQD(usdAmount) {
    return usdAmount * getIQDExchangeRate();
}

// Helper function to convert IQD to USD
function convertIQDToUSD(iqdAmount) {
    return iqdAmount / getIQDExchangeRate();
}

// Debug function to check current exchange rate
function checkIQDRate() {
    console.log('Current IQD Exchange Rate Configuration:');
    console.log('Default Rate:', IQD_EXCHANGE_RATE);
    console.log('Custom Rate:', window.customIQDExchangeRate || 'Not set');
    console.log('Stored Rate:', localStorage.getItem('iqd_exchange_rate') || 'Not set');
    console.log('Active Rate:', getIQDExchangeRate());
    console.log('---');
    console.log('Example: $100 USD = ' + convertUSDToIQD(100) + ' IQD');
    console.log('Example: 141,000 IQD = $' + convertIQDToUSD(141000).toFixed(2) + ' USD');
}

// Centralized status management system
var StatusManager = {
    defaultStatus: 'quotation',
    
    setFormStatus: function(status) {
        var forms = ['#add_pos_sell_form', '#edit_pos_sell_form'];
        forms.forEach(function(formSelector) {
            var form = $(formSelector);
            if (form.length > 0) {
                form.find('input[name="status"], select[name="status"]').remove();
                form.append('<input type="hidden" name="status" value="' + status + '">');
            }
        });
    },
    
    getStatusFromButton: function(buttonId, buttonClass) {
        if (buttonId === 'pos-finalize' || $(buttonClass).hasClass('pos-express-finalize')) {
            return 'final';
        } else if (buttonId === 'pos-draft') {
            return 'draft';
        } else if (buttonId === 'pos-quotation') {
            return 'quotation';
        }
        return this.defaultStatus;
    },
    
    initialize: function() {
        this.setFormStatus(this.defaultStatus);
    }
};

// Centralized AJAX handling
var AjaxHandler = {
    setupPrefilter: function() {
        $.ajaxPrefilter(function(options, originalOptions, jqXHR) {
            if (options.url && (options.url.includes('/pos/create') || options.url.includes('/pos/update'))) {
                if (options.data && typeof options.data === 'string') {
                    if (options.data.indexOf('status=') === -1) {
                        options.data += '&status=' + StatusManager.defaultStatus;
                    }
                } else if (options.data && typeof options.data === 'object') {
                    if (!options.data.status) {
                        options.data.status = StatusManager.defaultStatus;
                    }
                }
            }
        });
    }
};

$(document).ready(function() {
    // Initialize centralized systems
    StatusManager.initialize();
    AjaxHandler.setupPrefilter();
    
    // Ensure applyUnitFilterToRow is available globally
    if (typeof window.applyUnitFilterToRow !== 'function' && typeof applyUnitFilterToRow === 'function') {
        window.applyUnitFilterToRow = applyUnitFilterToRow;
    }
    
    customer_set = false;
    //Prevent enter key function except texarea
    $('form').on('keyup keypress', function(e) {
        var keyCode = e.keyCode || e.which;
        if (keyCode === 13 && e.target.tagName != 'TEXTAREA') {
            e.preventDefault();
            return false;
        }
    });

    //For edit pos form
    if ($('form#edit_pos_sell_form').length > 0) {
        pos_total_row();
        pos_form_obj = $('form#edit_pos_sell_form');
    } else {
        pos_form_obj = $('form#add_pos_sell_form');
    }
    if ($('form#edit_pos_sell_form').length > 0 || $('form#add_pos_sell_form').length > 0) {
        initialize_printer();
    }

    $('select#select_location_id').change(function() {
        reset_pos_form();

        var default_price_group = $(this).find(':selected').data('default_price_group')
        if (default_price_group) {
            if($("#price_group option[value='" + default_price_group + "']").length > 0) {
                $("#price_group").val(default_price_group);
                $("#price_group").change();
            }
        }

        //Set default invoice scheme for location
        if ($('#invoice_scheme_id').length) {
            if($('input[name="is_direct_sale"]').length > 0){
                //default scheme for sale screen
                var invoice_scheme_id = $(this).find(':selected').data('default_sale_invoice_scheme_id');
            } else {
                var invoice_scheme_id =  $(this).find(':selected').data('default_invoice_scheme_id');
            }
            
            $("#invoice_scheme_id").val(invoice_scheme_id).change();
        }

        //Set default invoice layout for location
        if ($('#invoice_layout_id').length) {
            let invoice_layout_id = $(this).find(':selected').data('default_invoice_layout_id');
            $("#invoice_layout_id").val(invoice_layout_id).change();
        }
        
        //Set default price group
        if ($('#default_price_group').length) {
            var dpg = default_price_group ?
            default_price_group : 0;
            $('#default_price_group').val(dpg);
        }

        set_payment_type_dropdown();

        if ($('#types_of_service_id').length && $('#types_of_service_id').val()) {
            $('#types_of_service_id').change();
        }
    });

    //get customer
    $('select#customer_id').select2({
        ajax: {
            url: '/contacts/customers',
            dataType: 'json',
            delay: 250,
            data: function(params) {
                return {
                    q: params.term, // search term
                    page: params.page,
                };
            },
            processResults: function(data) {
                return {
                    results: data,
                };
            },
        },
        templateResult: function (data) { 
            var template = '';
            if (data.supplier_business_name) {
                template += data.supplier_business_name + "<br>";
            }
            template += data.text + "<br>" + LANG.mobile + ": " + data.mobile;

            if (typeof(data.total_rp) != "undefined") {
                var rp = data.total_rp ? data.total_rp : 0;
                template += "<br><i class='fa fa-gift text-success'></i> " + rp;
            }

            return  template;
        },
        minimumInputLength: 1,
        language: {
            inputTooShort: function (args) {
                return LANG.please_enter + args.minimum + LANG.or_more_characters;
            },
            noResults: function() {
                var name = $('#customer_id')
                    .data('select2')
                    .dropdown.$search.val();
                return (
                    '<button type="button" data-name="' +
                    name +
                    '" class="btn btn-link add_new_customer"><i class="fa fa-plus-circle fa-lg" aria-hidden="true"></i>&nbsp; ' +
                    __translate('add_name_as_new_customer', { name: name }) +
                    '</button>'
                );
            },
        },
        escapeMarkup: function(markup) {
            return markup;
        },
    });
    $('#customer_id').on('select2:select', function(e) {
        var data = e.params.data;
        if (data.pay_term_number) {
            $('input#pay_term_number').val(data.pay_term_number);
        } else {
            $('input#pay_term_number').val('');
        }

        if (data.pay_term_type) {
            $('#add_sell_form select[name="pay_term_type"]').val(data.pay_term_type);
            $('#edit_sell_form select[name="pay_term_type"]').val(data.pay_term_type);
        } else {
            $('#add_sell_form select[name="pay_term_type"]').val('');
            $('#edit_sell_form select[name="pay_term_type"]').val('');
        }
        
        update_shipping_address(data);
        $('#advance_balance_text').text(__currency_trans_from_en(data.balance), true);
        $('#advance_balance').val(data.balance);

        if (data.price_calculation_type == 'selling_price_group') {
            $('#price_group').val(data.selling_price_group_id);
            $('#price_group').change();
        }
        //  else {
        //     $('#price_group').val(0);
        //     $('#price_group').change();
        // }
        if ($('.contact_due_text').length) {
            get_contact_due(data.id);
        }
    });

    set_default_customer();

    // متغيرات تحكم في جدول البحث
    var searchTableVisible = false;
var keepSearchOpen = false;
var lastSearchTerm = '';
// إضافة متغيرات جديدة للاحتفاظ بحالة البحث
var lastSearchResults = [];
var shouldMaintainSearchResults = false;

// دالة لتحديث الـ autocomplete مع نتائج محددة
function updateAutocompleteWithResults(results) {
    console.log('🔄 تحديث الـ autocomplete مع النتائج:', results.length);
    
    if (searchTableVisible && $('#search_product').data('ui-autocomplete')) {
        // إغلاق الـ autocomplete الحالي
        $('#search_product').autocomplete('close');
        
        // انتظار قصير ثم إعادة فتحه مع النتائج الجديدة
        setTimeout(function() {
            // محاكاة استجابة البحث مع النتائج المفلترة
            var autocomplete = $('#search_product').data('ui-autocomplete');
            if (autocomplete) {
                // تطبيق النتائج مباشرة على الـ autocomplete
                autocomplete._response(results);
                
                // فتح الجدول إذا كانت هناك نتائج
                if (results.length > 0) {
                    searchTableVisible = true;
                    keepSearchOpen = true;
                }
            }
        }, 100);
    } else if (shouldMaintainSearchResults && lastSearchTerm.length >= 2) {
        // إذا لم يكن الجدول مفتوحاً، فتحه مع النتائج الجديدة
        console.log('🔓 فتح الجدول مع النتائج المفلترة');
        $('#search_product').autocomplete('search', lastSearchTerm);
    }
}

$('#search_product')
.autocomplete({
    delay: 1000,
    source: function(request, response) {
        var price_group = '';
        var search_fields = [];
        $('.search_fields:checked').each(function(i){
          search_fields[i] = $(this).val();
        });

        if ($('#price_group').length > 0) {
            price_group = $('#price_group').val();
        }
        
        // التحقق من أن هذا بحث جديد وليس إعادة عرض للنتائج السابقة
        if (request.term !== lastSearchTerm || !shouldMaintainSearchResults) {
            // حفظ آخر مصطلح بحث
            lastSearchTerm = request.term;
            console.log('🔍 البحث عن:', request.term);
            
            $.getJSON(
                '/products/list',
                {
                    price_group: price_group,
                    location_id: $('input#location_id').val(),
                    term: request.term,
                    not_for_selling: 0,
                    search_fields: search_fields,
                    include_warehouse_details: 1
                },
                function(data) {
                    console.log('✅ نتائج البحث:', data.length, 'منتج');
                    // حفظ آخر نتائج بحث الأصلية
                    lastSearchResults = data;
                    shouldMaintainSearchResults = true;
                    
                    // تطبيق الفلاتر النشطة على النتائج
                    var filteredData = data;
                    var hasActiveFilters = (window.activeFilters && window.activeFilters.length > 0) || 
                                         window.activeUnitFilter !== null || 
                                         window.fullPlasticFilterActive !== null;
                    
                    if (hasActiveFilters && typeof applyAllFiltersToProducts === 'function') {
                        console.log('🔄 تطبيق الفلاتر النشطة على نتائج البحث الجديدة');
                        filteredData = applyAllFiltersToProducts([...data]);
                        console.log('📋 النتائج بعد الفلترة:', filteredData.length, 'منتج');
                    }
                    
                    response(filteredData);
                }
            );
        } else {
            // إعادة عرض النتائج المحفوظة مع تطبيق الفلاتر النشطة
            console.log('📋 إعادة عرض النتائج المحفوظة:', lastSearchResults.length, 'منتج');
            var filteredResults = lastSearchResults;
            
            var hasActiveFilters = (window.activeFilters && window.activeFilters.length > 0) || 
                                 window.activeUnitFilter !== null || 
                                 window.fullPlasticFilterActive !== null;
            
            if (hasActiveFilters && typeof applyAllFiltersToProducts === 'function') {
                console.log('🔄 تطبيق الفلاتر النشطة على النتائج المحفوظة');
                filteredResults = applyAllFiltersToProducts([...lastSearchResults]);
                console.log('📋 النتائج المحفوظة بعد الفلترة:', filteredResults.length, 'منتج');
            }
            
            response(filteredResults);
        }
    },
    minLength: 2,
    open: function(event, ui) {
        searchTableVisible = true;
            keepSearchOpen = true;
            // التركيز على الجدول بدلاً من حقل البحث
            setTimeout(function() {
                var $table = $('.ui-autocomplete .product-search-table');
                var $container = $('.ui-autocomplete .product-search-container');
                if ($table.length && $container.length) {
                                            // إضافة حقل البحث الإضافي في أعلى الجدول
                    if (!$container.find('.secondary-search-container').length) {
                        console.log('📝 إضافة حقل البحث الثانوي');
                        var $secondarySearchContainer = $('<div class="secondary-search-container" style="padding: 10px; border-bottom: 1px solid #ddd; background: #f9f9f9;">');
                        var $secondarySearch = $('<input type="text" class="secondary-search-input form-control" placeholder="البحث في النتائج..." style="margin-bottom: 5px;">');
                        var $closeBtn = $('<button type="button" class="btn btn-sm btn-danger pull-right" style="margin-left: 5px;">إغلاق</button>');
                        
                        $secondarySearchContainer.append($closeBtn);
                        $secondarySearchContainer.append($secondarySearch);
                        $container.prepend($secondarySearchContainer);
                        console.log('✅ تم إضافة حقل البحث الثانوي بنجاح');
                        
                        // ربط حقل البحث الثانوي بالرئيسي
                        $secondarySearch.on('input', function() {
                            var searchValue = $(this).val();
                            console.log('🔄 البحث من الحقل الثانوي:', searchValue, 'آخر مصطلح:', lastSearchTerm);
                            $('#search_product').val(searchValue);
                            if (searchValue.length >= 2 && searchValue !== lastSearchTerm) {
                                console.log('🔍 تنفيذ بحث جديد:', searchValue);
                                // مسح النتائج السابقة عند البحث عن مصطلح جديد
                                shouldMaintainSearchResults = false;
                                lastSearchResults = [];
                                lastSearchTerm = searchValue;
                                $('#search_product').autocomplete('search', searchValue);
                            } else if (searchValue.length < 2) {
                                console.log('❌ إغلاق الجدول - النص قصير جداً');
                                // إغلاق الجدول إذا كان النص أقل من حرفين
                                $('#search_product').autocomplete('close');
                                searchTableVisible = false;
                                keepSearchOpen = false;
                                // مسح النتائج المحفوظة عند مسح البحث
                                shouldMaintainSearchResults = false;
                                lastSearchResults = [];
                                lastSearchTerm = '';
                            } else {
                                console.log('⏭️ تجاهل البحث - نفس المصطلح');
                            }
                        });
                        
                        // إغلاق الجدول عند النقر على زر الإغلاق
                        $closeBtn.on('click', function() {
                            $('#search_product').autocomplete('close');
                            searchTableVisible = false;
                            keepSearchOpen = false;
                            // عدم مسح النتائج المحفوظة عند إغلاق الجدول يدوياً
                            console.log('🔒 إغلاق الجدول يدوياً مع الاحتفاظ بالنتائج');
                        });
                        
                        // نسخ القيمة الحالية
                        $secondarySearch.val($('#search_product').val());
                    }
                    
                    // التركيز على أول صف في الجدول
                    var $firstRow = $table.find('tbody tr').first();
                    if ($firstRow.length) {
                        $firstRow.addClass('ui-state-focus highlighted');
                        $table.focus();
                    }
                }
            }, 100);
        },
        close: function(event, ui) {
            searchTableVisible = false;
            keepSearchOpen = false;
            // عدم مسح shouldMaintainSearchResults هنا للحفاظ على النتائج
            console.log('🔒 إغلاق الـ autocomplete مع الاحتفاظ بالنتائج');
        },
        response: function(event, ui) {
            if (ui.content.length == 1) {
                ui.item = ui.content[0];

                var is_overselling_allowed = false;
                if($('input#is_overselling_allowed').length) {
                    is_overselling_allowed = true;
                }
                var for_so = false;
                if ($('#sale_type').length && $('#sale_type').val() == 'sales_order') {
                    for_so = true;
                }

                if ((ui.item.enable_stock == 1 && ui.item.qty_available > 0) || 
                        (ui.item.enable_stock == 0) || is_overselling_allowed || for_so) {
                    $(this)
                        .data('ui-autocomplete')
                        ._trigger('select', 'autocompleteselect', ui);
                    $(this).autocomplete('close');
                }
            } else if (ui.content.length == 0) {
                toastr.error(LANG.no_products_found);
                $('input#search_product').select();
            }
        },
        focus: function(event, ui) {
            if (ui.item.qty_available <= 0) {
                return false;
            }
        },
        select: function(event, ui) {
            var searched_term = $(this).val();
            var is_overselling_allowed = false;
            if($('input#is_overselling_allowed').length) {
                is_overselling_allowed = true;
            }
            var for_so = false;
            if ($('#sale_type').length && $('#sale_type').val() == 'sales_order') {
                for_so = true;
            }

            var is_draft=false;
            if($('#status') && ($('#status').val()=='quotation' || 
            $('#status').val()=='draft')) {
                var is_draft=true;
            }

            if (ui.item.enable_stock != 1 || ui.item.qty_available > 0 || is_overselling_allowed || for_so || is_draft) {
                $(this).val(null);
                
                // إغلاق جدول البحث عند اختيار منتج مع الاحتفاظ بالنتائج
                searchTableVisible = false;
                keepSearchOpen = false;
                console.log('✅ تم اختيار منتج مع الاحتفاظ بالنتائج المحفوظة');

                console.log('🎯 Product selected from autocomplete:', ui.item.name, 'Base unit:', ui.item.product_custom_field2);

                var purchase_line_id = ui.item.purchase_line_id && searched_term == ui.item.lot_number ? ui.item.purchase_line_id : null;
                pos_product_row(ui.item.variation_id, purchase_line_id);
            } else {
                alert(LANG.out_of_stock);
            }
        },
    })
    .autocomplete('instance')._renderMenu = function(ul, items) {
        var $div = $('<div class="product-search-container">');
        
        var $table = $('<table class="product-search-table table table-bordered table-striped" tabindex="0">');
        var $thead = $('<thead>').appendTo($table);
        var $tbody = $('<tbody>').appendTo($table);
        
        $thead.append('<tr>' +
                     '<th>Product</th>' +
                     '<th>Item Group</th>' +
                     '<th>Foreign Name</th>' +
                     '<th>Code</th>' +
                     '<th>Price (USD)</th>' +
                     '<th>Price (IQD)</th>' +
                     '<th>Discount</th>' +
                     '<th>Price After Discount</th>' +
                     '<th>UOM</th>' +
                     '<th>Stock</th>' +
                     '<th>Warehouses</th>' +
                     '</tr>');
        
        var self = this;
        $.each(items, function(index, item) {
            var $tr = $('<tr>');
            
            var is_overselling_allowed = $('input#is_overselling_allowed').length > 0;
            var for_so = $('#sale_type').length && $('#sale_type').val() == 'sales_order';
            var is_draft = $('#status').length && ($('#status').val()=='quotation' || $('#status').val()=='draft');
            
            if (item.enable_stock == 1 && item.qty_available <= 0 && !is_overselling_allowed && !for_so && !is_draft) {
                $tr.addClass('out-of-stock');
            }
            
            var product_name = item.name;
            if (item.type == 'variable') {
                product_name += ' - ' + item.variation;
            }
            $tr.append('<td>' + product_name + '</td>');
            $tr.append('<td>' + product_name + '</td>');
            
            $tr.append('<td>' + (item.ForeignName || '-') + '</td>');
            
            $tr.append('<td>' + (item.sub_sku || '-') + '</td>');
            
            var selling_price_usd = item.variation_group_price || item.selling_price || 0;
            $tr.append('<td>' + __currency_trans_from_en(selling_price_usd, false, false, __currency_precision, true) + '</td>');
            
            var exchange_rate = getIQDExchangeRate();
var selling_price_iqd = selling_price_usd * exchange_rate;
            $tr.append('<td>' + __currency_trans_from_en(selling_price_iqd, false, false, 0, true) + '</td>');
            
            var discount_percent = item.discount_percent || 0;
            $tr.append('<td>' + discount_percent + '%</td>');
            
            var price_after_discount = selling_price_iqd - (selling_price_iqd * discount_percent / 100);
            $tr.append('<td>' + __currency_trans_from_en(price_after_discount, false, false, 0, true) + '</td>');
            
            $tr.append('<td>' + (item.unit || '-') + '</td>');
            
            var stock_info = '';
            if (item.enable_stock == 1) {
                stock_info = __currency_trans_from_en(item.qty_available, false, false, __currency_precision, true);
                if (item.qty_available <= 0 && !is_overselling_allowed && !for_so && !is_draft) {
                    stock_info += ' (Out of Stock)';
                }
            }
            $tr.append('<td>' + stock_info + '</td>');
            
            var warehouse_info = '';
            if (item.warehouses && item.warehouses.length > 0) {
                var warehouses_list = [];
                $.each(item.warehouses, function(i, wh) {

                    
                    warehouses_list.push(wh.name + ': ' + wh.qty);
                });
                warehouse_info = warehouses_list.join(' | ');
            } else {
                warehouse_info = '-';
            }
            $tr.append('<td>' + warehouse_info + '</td>');
            
            // هام: إضافة بيانات العنصر للصف للسماح بالنقر والاختيار
            $tr.data('ui-autocomplete-item', item);
            
            // إضافة class للعناصر التي يمكن اختيارها
            $tr.addClass('ui-menu-item');
            
            // إضافة wrapper للتوافق مع jQuery UI Autocomplete
            var $wrapper = $('<div class="ui-menu-item-wrapper"></div>');
            $wrapper.data('ui-autocomplete-item', item);
            $tr.wrapInner($wrapper);
            
            $tbody.append($tr);
        });
        
        // جعل الجدول قابل للتركيز عليه
        $table.attr('tabindex', '0');
        
        $div.append($table);
        $(ul).html($div);
        
        // إضافة دعم النقر بالماوس
        $table.find('tr').on('click', function(e) {
            e.preventDefault();
            var item = $(this).data('ui-autocomplete-item');
            if (item) {
                self._trigger('select', 'autocompleteselect', {item: item});
                self.element.autocomplete('close');
            }
        });
        
        // إضافة معالج للتنقل بالكيبورد على الجدول
        $table.on('keydown', function(e) {
            var $rows = $(this).find('tbody tr');
            var $current = $rows.filter('.ui-state-focus');
            var currentIndex = $rows.index($current);
            
            switch(e.which) {
                case 38: // Up arrow
                    e.preventDefault();
                    if (currentIndex > 0) {
                        $rows.removeClass('ui-state-focus highlighted');
                        $rows.eq(currentIndex - 1).addClass('ui-state-focus highlighted');
                    }
                    break;
                case 40: // Down arrow
                    e.preventDefault();
                    if (currentIndex < $rows.length - 1) {
                        $rows.removeClass('ui-state-focus highlighted');
                        $rows.eq(currentIndex + 1).addClass('ui-state-focus highlighted');
                    }
                    break;
                case 13: // Enter
                    e.preventDefault();
                    if ($current.length) {
                        $current.click();
                    }
                    break;
                case 27: // Escape
                    e.preventDefault();
                    self.element.autocomplete('close');
                    searchTableVisible = false;
                    keepSearchOpen = false;
                    // عدم مسح النتائج المحفوظة عند الضغط على Escape من الجدول
                    console.log('🔒 إغلاق الجدول بـ Escape من الجدول مع الاحتفاظ بالنتائج');
                    break;
            }
        });
        
        return $div;
    };

// إضافة CSS للتحسينات الجديدة
$('<style>')
    .prop('type', 'text/css')
    .html(`
        .product-search-table:focus {
            outline: 2px solid #007bff !important;
            outline-offset: -2px;
        }
        
        .product-search-table tbody tr.highlighted {
            background-color: #e3f2fd !important;
            border-color: #2196f3 !important;
        }
        
        .product-search-table tbody tr.highlighted td {
            background-color: #e3f2fd !important;
            border-color: #2196f3 !important;
        }
        
        .secondary-search-container {
            border-radius: 5px 5px 0 0;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .secondary-search-input {
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 8px;
            font-size: 14px;
        }
        
        .secondary-search-input:focus {
            border-color: #007bff;
            box-shadow: 0 0 0 0.2rem rgba(0,123,255,.25);
            outline: 0;
        }
        
        .ui-autocomplete {
            max-height: 500px;
            overflow-y: auto;
            z-index: 9999 !important;
        }
        
        .product-search-table {
            margin-bottom: 0 !important;
            font-size: 13px;
        }
        
        .product-search-table th,
        .product-search-table td {
            padding: 8px 6px !important;
            vertical-align: middle !important;
        }
        
        .product-search-table tbody tr {
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .product-search-table tbody tr:hover {
            background-color: #f8f9fa !important;
        }
    `)
    .appendTo('head');

// تحسين دعم التنقل بالأسهم للجدول
$('#search_product').autocomplete('instance')._move = function(direction, event) {
    console.log('⌨️ التنقل بالأسهم:', direction);
    
    // إذا كان الجدول مرئي، تنقل داخله بدلاً من البحث الجديد
    if (this.menu.element.is(':visible')) {
        var $table = this.menu.element.find('.product-search-table');
        var $rows = $table.find('tbody tr');
        
        console.log('📊 الجدول مرئي، عدد الصفوف:', $rows.length);
        
        if ($rows.length > 0) {
            var $current = $rows.filter('.ui-state-focus');
            var $next;
            
            if ($current.length === 0) {
                // لا يوجد صف محدد حالياً
                $next = direction === 'next' ? $rows.first() : $rows.last();
                console.log('🎯 لا يوجد صف محدد، اختيار:', direction === 'next' ? 'الأول' : 'الأخير');
            } else {
                // يوجد صف محدد
                var currentIndex = $rows.index($current);
                var nextIndex = direction === 'next' ? currentIndex + 1 : currentIndex - 1;
                console.log('📍 الصف الحالي:', currentIndex, 'الصف التالي:', nextIndex);
                
                if (nextIndex >= 0 && nextIndex < $rows.length) {
                    $next = $rows.eq(nextIndex);
                }
            }
            
            if ($next && $next.length) {
                // إزالة التحديد من الصف الحالي
                $rows.removeClass('ui-state-focus highlighted');
                // إضافة التحديد للصف الجديد
                $next.addClass('ui-state-focus highlighted');
                console.log('✅ تم تحديد صف جديد');
                
                // تحديث القيمة المحددة
                var item = $next.data('ui-autocomplete-item');
                if (item) {
                    this._trigger('focus', event, {item: item});
                }
                
                // التمرير للصف المحدد إذا لزم الأمر
                var containerHeight = this.menu.element.height();
                var rowTop = $next.position().top;
                var rowHeight = $next.outerHeight();
                
                if (rowTop < 0) {
                    this.menu.element.scrollTop(this.menu.element.scrollTop() + rowTop);
                } else if (rowTop + rowHeight > containerHeight) {
                    this.menu.element.scrollTop(this.menu.element.scrollTop() + rowTop + rowHeight - containerHeight);
                }
            }
            console.log('🚫 منع تنفيذ البحث الجديد');
            return; // منع تنفيذ الكود التالي
        }
    }
    
    console.log('⏹️ الجدول غير مرئي أو فارغ - لا فعل شيء');
    // إذا لم يكن الجدول مرئي أو لا يحتوي على بيانات، فقط عودة بدون فعل أي شيء
    // لا نريد استدعاء this.search(null, event) لأنه يسبب إظهار جميع المنتجات
    return;
};

// منع إغلاق الـ autocomplete عند فقدان التركيز إلا إذا كان مقصوداً
$('#search_product').on('blur', function(e) {
    if (keepSearchOpen && searchTableVisible) {
        // منع إغلاق الـ autocomplete
        setTimeout(function() {
            if (keepSearchOpen) {
                var $table = $('.ui-autocomplete .product-search-table');
                if ($table.length && !$table.is(':focus') && !$('.secondary-search-input').is(':focus')) {
                    $table.focus();
                }
            }
        }, 10);
    }
});

// معالج لإعادة فتح الجدول مع النتائج المحفوظة عند التركيز
$('#search_product').on('focus', function(e) {
    var currentValue = $(this).val().trim();
    // إذا كان هناك نتائج محفوظة ولا يوجد جدول مفتوح حالياً
    if (shouldMaintainSearchResults && lastSearchResults.length > 0 && !searchTableVisible && currentValue.length >= 2) {
        console.log('🔓 إعادة فتح الجدول مع النتائج المحفوظة عند التركيز');
        $(this).autocomplete('search', currentValue);
    }
});

// معالج للنقر خارج جدول البحث لإغلاقه
$(document).on('click', function(e) {
    if (searchTableVisible && keepSearchOpen) {
        var $target = $(e.target);
        // التحقق من عدم النقر داخل جدول البحث أو عناصره
        if (!$target.closest('.ui-autocomplete, #search_product, .secondary-search-input, .secondary-search-container').length) {
            $('#search_product').autocomplete('close');
            searchTableVisible = false;
            keepSearchOpen = false;
            // عدم مسح النتائج المحفوظة عند النقر خارج الجدول
            console.log('🔒 إغلاق الجدول بالنقر خارجه مع الاحتفاظ بالنتائج');
        }
    }
});

// معالج لمفتاح Escape لإغلاق جدول البحث
$('#search_product').on('keydown', function(e) {
    if (e.which === 27 && searchTableVisible) { // Escape key
        e.preventDefault();
        $(this).autocomplete('close');
        searchTableVisible = false;
        keepSearchOpen = false;
        // عدم مسح النتائج المحفوظة عند الضغط على Escape
        console.log('🔒 إغلاق الجدول بـ Escape مع الاحتفاظ بالنتائج');
    } else if ((e.which === 38 || e.which === 40) && searchTableVisible) {
        // منع السلوك الافتراضي للأسهم عندما يكون الجدول مرئي
        // سيتم التعامل معها في _move function
        return;
    }
});

// معالج للتأكد من تحديث حقل البحث الثانوي عند تغيير الرئيسي
$('#search_product').on('input', function() {
    var searchValue = $(this).val();
    
    if (searchTableVisible && $('.secondary-search-input').length) {
        $('.secondary-search-input').val(searchValue);
    }
    
    // مسح النتائج المحفوظة عند تغيير النص في حقل البحث الرئيسي
    if (searchValue !== lastSearchTerm) {
        console.log('🔄 تغيير في حقل البحث الرئيسي - مسح النتائج المحفوظة');
        shouldMaintainSearchResults = false;
        lastSearchResults = [];
        if (searchValue.length < 2) {
            lastSearchTerm = '';
        }
    }
});

// معالج للتأكد من العودة للجدول بعد استخدام البحث الثانوي
$(document).on('keydown', '.secondary-search-input', function(e) {
    if (e.which === 13) { // Enter key
        e.preventDefault();
        var $table = $('.ui-autocomplete .product-search-table');
        if ($table.length) {
            $table.focus();
            var $firstRow = $table.find('tbody tr').first();
            if ($firstRow.length) {
                $table.find('tbody tr').removeClass('ui-state-focus highlighted');
                $firstRow.addClass('ui-state-focus highlighted');
            }
        }
    } else if (e.which === 27) { // Escape key
        e.preventDefault();
        $('#search_product').autocomplete('close');
        searchTableVisible = false;
        keepSearchOpen = false;
        // عدم مسح النتائج المحفوظة عند الضغط على Escape من الحقل الثانوي
        console.log('🔒 إغلاق الجدول بـ Escape من الحقل الثانوي مع الاحتفاظ بالنتائج');
    } else if (e.which === 40) { // Down arrow - go to table
        e.preventDefault();
        var $table = $('.ui-autocomplete .product-search-table');
        if ($table.length) {
            $table.focus();
            var $firstRow = $table.find('tbody tr').first();
            if ($firstRow.length) {
                $table.find('tbody tr').removeClass('ui-state-focus highlighted');
                $firstRow.addClass('ui-state-focus highlighted');
            }
        }
    }
});

// معالج للتنقل من الجدول إلى حقل البحث الثانوي
$(document).on('keydown', '.product-search-table', function(e) {
    if (e.which === 38) { // Up arrow
        var $rows = $(this).find('tbody tr');
        var $current = $rows.filter('.ui-state-focus');
        var currentIndex = $rows.index($current);
        
        if (currentIndex === 0) {
            // إذا كنا في أول صف، انتقل لحقل البحث الثانوي
            e.preventDefault();
            $('.secondary-search-input').focus();
        }
    }
});

    //Update line total and check for quantity not greater than max quantity
    $('table#pos_table tbody').on('change', 'input.pos_quantity', function() {
        // comment line becouse it validate form at increment and decrement item
        // if (sell_form_validator) {
        //     sell_form.valid();
        // }
        if (pos_form_validator) {
            pos_form_validator.element($(this));
        }
        // var max_qty = parseFloat($(this).data('rule-max'));
        var entered_qty = __read_number($(this));

        var tr = $(this).parents('tr');

        var unit_price_inc_tax = __read_number(tr.find('input.pos_unit_price_inc_tax'));
        var line_total = entered_qty * unit_price_inc_tax;

        __write_number(tr.find('input.pos_line_total'), line_total, false);
        tr.find('span.pos_line_total_text').text(__currency_trans_from_en(line_total, true));

        //Change modifier quantity
        tr.find('.modifier_qty_text').each( function(){
            $(this).text(__currency_trans_from_en(entered_qty, false));
        });
        tr.find('.modifiers_quantity').each( function(){
            $(this).val(entered_qty);
        });

        pos_total_row();

       // adjustComboQty(tr);
    });

    //If change in unit price update price including tax and line total
    $('table#pos_table tbody').on('change', 'input.pos_unit_price', function() {
        var tr = $(this).parents('tr');
        
        // إذا كان نظام التسعير المتقدم نشطاً، لا تتداخل معه
        if (tr.hasClass('advanced-pricing-active')) {
            console.log('🛡️ Unit price handler blocked - Advanced pricing is active');
            return;
        }
        
        var unit_price = __read_number($(this));

        //calculate discounted unit price
        var discounted_unit_price = calculate_discounted_unit_price(tr);

        var tax_rate = tr
            .find('select.tax_id')
            .find(':selected')
            .data('rate');
        var quantity = __read_number(tr.find('input.pos_quantity'));

        var unit_price_inc_tax = __add_percent(discounted_unit_price, tax_rate);
        var line_total = quantity * unit_price_inc_tax;

        __write_number(tr.find('input.pos_unit_price_inc_tax'), unit_price_inc_tax);
        __write_number(tr.find('input.pos_line_total'), line_total);
        tr.find('span.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        pos_each_row(tr);
        pos_total_row();
        round_row_to_iraqi_dinnar(tr);
    });

    //If change in tax rate then update unit price according to it.
    $('table#pos_table tbody').on('change', 'select.tax_id', function() {
        var tr = $(this).parents('tr');

        var tax_rate = tr
            .find('select.tax_id')
            .find(':selected')
            .data('rate');
        var unit_price_inc_tax = __read_number(tr.find('input.pos_unit_price_inc_tax'));

        var discounted_unit_price = __get_principle(unit_price_inc_tax, tax_rate);
        var unit_price = get_unit_price_from_discounted_unit_price(tr, discounted_unit_price);
        __write_number(tr.find('input.pos_unit_price'), unit_price);
        pos_each_row(tr);
    });

    //If change in unit price including tax, update unit price
    $('table#pos_table tbody').on('change', 'input.pos_unit_price_inc_tax', function() {
        var tr = $(this).parents('tr');
        
        // إذا كان نظام التسعير المتقدم نشطاً، لا تتداخل معه
        if (tr.hasClass('advanced-pricing-active')) {
            console.log('🛡️ Unit price inc tax handler blocked - Advanced pricing is active');
            return;
        }
        
        var unit_price_inc_tax = __read_number($(this));

        if (iraqi_selling_price_adjustment) {
            unit_price_inc_tax = round_to_iraqi_dinnar(unit_price_inc_tax);
            __write_number($(this), unit_price_inc_tax);
        }

        var tax_rate = tr
            .find('select.tax_id')
            .find(':selected')
            .data('rate');
        var quantity = __read_number(tr.find('input.pos_quantity'));

        var line_total = quantity * unit_price_inc_tax;
        var discounted_unit_price = __get_principle(unit_price_inc_tax, tax_rate);
        var unit_price = get_unit_price_from_discounted_unit_price(tr, discounted_unit_price);

        __write_number(tr.find('input.pos_unit_price'), unit_price);
        __write_number(tr.find('input.pos_line_total'), line_total, false);
        tr.find('span.pos_line_total_text').text(__currency_trans_from_en(line_total, true));

        pos_each_row(tr);
        pos_total_row();
    });

    //Change max quantity rule if lot number changes
    $('table#pos_table tbody').on('change', 'select.lot_number', function() {
        var qty_element = $(this)
            .closest('tr')
            .find('input.pos_quantity');

        var tr = $(this).closest('tr');
        var multiplier = 1;
        var unit_name = '';
        var sub_unit_length = tr.find('select.sub_unit').length;
        if (sub_unit_length > 0) {
            var select = tr.find('select.sub_unit');
            multiplier = parseFloat(select.find(':selected').data('multiplier'));
            unit_name = select.find(':selected').data('unit_name');
        }
        var allow_overselling = qty_element.data('allow-overselling');
        if ($(this).val() && !allow_overselling) {
            var lot_qty = $('option:selected', $(this)).data('qty_available');
            var max_err_msg = $('option:selected', $(this)).data('msg-max');

            if (sub_unit_length > 0) {
                lot_qty = lot_qty / multiplier;
                var lot_qty_formated = __number_f(lot_qty, false);
                max_err_msg = __translate('lot_max_qty_error', {
                    max_val: lot_qty_formated,
                    unit_name: unit_name,
                });
            }

            qty_element.attr('data-rule-max-value', lot_qty);
            qty_element.attr('data-msg-max-value', max_err_msg);

            qty_element.rules('add', {
                'max-value': lot_qty,
                messages: {
                    'max-value': max_err_msg,
                },
            });
        } else {
            var default_qty = qty_element.data('qty_available');
            var default_err_msg = qty_element.data('msg_max_default');
            if (sub_unit_length > 0) {
               // default_qty = default_qty / multiplier;
                var lot_qty_formated = __number_f(default_qty, false);
                default_err_msg = __translate('pos_max_qty_error', {
                    max_val: lot_qty_formated,
                    unit_name: unit_name,
                });
            }

            qty_element.attr('data-rule-max-value', default_qty);
            qty_element.attr('data-msg-max-value', default_err_msg);

            qty_element.rules('add', {
                'max-value': default_qty,
                messages: {
                    'max-value': default_err_msg,
                },
            });
        }
        qty_element.trigger('change');
    });

    //Change in row discount type or discount amount
    $('table#pos_table tbody').on(
        'change',
        'select.row_discount_type, input.row_discount_amount',
        function() {
            var tr = $(this).parents('tr');

            //calculate discounted unit price
            var discounted_unit_price = calculate_discounted_unit_price(tr);

            var tax_rate = tr
                .find('select.tax_id')
                .find(':selected')
                .data('rate');
            var quantity = __read_number(tr.find('input.pos_quantity'));

            var unit_price_inc_tax = __add_percent(discounted_unit_price, tax_rate);
            var line_total = quantity * unit_price_inc_tax;

            __write_number(tr.find('input.pos_unit_price_inc_tax'), unit_price_inc_tax);
            __write_number(tr.find('input.pos_line_total'), line_total, false);
            tr.find('span.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
            pos_each_row(tr);
            pos_total_row();
            round_row_to_iraqi_dinnar(tr);
        }
    );

    //Remove row on click on remove row
    $('table#pos_table tbody').on('click', 'i.pos_remove_row', function() {
        $(this)
            .parents('tr')
            .remove();
        pos_total_row();
    });

    // Centralized button handlers
    var ButtonHandlers = {
        setupActionButtons: function() {
            // Cancel button
            $('button#pos-cancel').click(function() {
                swal({
                    title: LANG.sure,
                    icon: 'warning',
                    buttons: true,
                    dangerMode: true,
                }).then(confirm => {
                    if (confirm) {
                        reset_pos_form();
                    }
                });
            });

            // Draft button
            $('button#pos-draft').click(function() {
                removeEmptyRowsBeforeSubmit();
                if ($('table#pos_table tbody').find('.product_row').length <= 0) {
                    toastr.warning(LANG.no_products_added);
                    return false;
                }

                var is_valid = FormValidator.validatePosForm();
                if (is_valid != true) {
                    return;
                }

                var data = pos_form_obj.serialize();
                data = data + '&status=draft';
                var url = pos_form_obj.attr('action');

                disable_pos_form_actions();
                $.ajax({
                    method: 'POST',
                    url: url,
                    data: data,
                    dataType: 'json',
                    success: function(result) {
                        enable_pos_form_actions();
                        if (result.success == 1) {
                            reset_pos_form();
                            toastr.success(result.msg);
                        } else {
                            toastr.error(result.msg);
                        }
                    },
                });
            });

            // Quotation button
            $('button#pos-quotation').click(function() {
                removeEmptyRowsBeforeSubmit();

                if ($('table#pos_table tbody').find('.product_row').length <= 0) {
                    toastr.warning(LANG.no_products_added);
                    return false;
                }

                var is_valid = FormValidator.validatePosForm();
                if (is_valid != true) {
                    return;
                }

                var data = pos_form_obj.serialize();
                data = data + '&status=quotation';
                var url = pos_form_obj.attr('action');

                disable_pos_form_actions();
                $.ajax({
                    method: 'POST',
                    url: url,
                    data: data,
                    dataType: 'json',
                    success: function(result) {
                        enable_pos_form_actions();
                        if (result.success == 1) {
                            reset_pos_form();
                            toastr.success(result.msg);

                            if (result.receipt.is_enabled) {
                                pos_print(result.receipt);
                            }
                        } else {
                            toastr.error(result.msg);
                        }
                    },
                });
            });

            // Finalize button
            $('button#pos-finalize').click(function() {
                removeEmptyRowsBeforeSubmit();
                if ($('table#pos_table tbody').find('.product_row').length <= 0) {
                    toastr.warning(LANG.no_products_added);
                    return false;
                }

                if ($('#reward_point_enabled').length) {
                    var validate_rp = isValidatRewardPoint();
                    if (!validate_rp['is_valid']) {
                        toastr.error(validate_rp['msg']);
                        return false;
                    }
                }

                $('#modal_payment').modal('show');
            });
        }
    };

    // Initialize button handlers
    ButtonHandlers.setupActionButtons();

    $('#modal_payment').one('shown.bs.modal', function() {
        $('#modal_payment')
            .find('input')
            .filter(':visible:first')
            .focus()
            .select();
        if ($('form#edit_pos_sell_form').length == 0) {
            $(this).find('#method_0').change();
        }
    });

            // Express finalize buttons
            $('button.pos-express-finalize').click(function() {
                removeEmptyRowsBeforeSubmit();
                if ($('table#pos_table tbody').find('.product_row').length <= 0) {
                    toastr.warning(LANG.no_products_added);
                    return false;
                }

                if ($('#reward_point_enabled').length) {
                    var validate_rp = isValidatRewardPoint();
                    if (!validate_rp['is_valid']) {
                        toastr.error(validate_rp['msg']);
                        return false;
                    }
                }

                var pay_method = $(this).data('pay_method');

                if (pay_method == 'credit_sale') {
                    $('#is_credit_sale').val(1);
                    pos_form_obj.submit();
                    return true;
                } else {
                    if ($('#is_credit_sale').length) {
                        $('#is_credit_sale').val(0);
                    }
                }

                var total_payable = __read_number($('input#final_total_input'));
                var total_paying = __read_number($('input#total_paying_input'));
                if (total_payable > total_paying) {
                    var bal_due = total_payable - total_paying;
                    var first_row = $('#payment_rows_div').find('.payment-amount').first();
                    var first_row_val = __read_number(first_row);
                    first_row_val = first_row_val + bal_due;
                    __write_number(first_row, first_row_val);
                    first_row.trigger('change');
                }

                var payment_method_dropdown = $('#payment_rows_div').find('.payment_types_dropdown').first();
                payment_method_dropdown.val(pay_method);
                payment_method_dropdown.change();
                
                if (pay_method == 'card') {
                    $('div#card_details_modal').modal('show');
                } else if (pay_method == 'suspend') {
                    $('div#confirmSuspendModal').modal('show');
                } else {
                    pos_form_obj.submit();
                }
            });

    $('div#card_details_modal').on('shown.bs.modal', function(e) {
        $('input#card_number').focus();
    });

    $('div#confirmSuspendModal').on('shown.bs.modal', function(e) {
        $(this)
            .find('textarea')
            .focus();
    });

    //on save card details
    $('button#pos-save-card').click(function() {
        $('input#card_number_0').val($('#card_number').val());
        $('input#card_holder_name_0').val($('#card_holder_name').val());
        $('input#card_transaction_number_0').val($('#card_transaction_number').val());
        $('select#card_type_0').val($('#card_type').val());
        $('input#card_month_0').val($('#card_month').val());
        $('input#card_year_0').val($('#card_year').val());
        $('input#card_security_0').val($('#card_security').val());

        $('div#card_details_modal').modal('hide');
        pos_form_obj.submit();
    });

    $('button#pos-suspend').click(function() {
        $('input#is_suspend').val(1);
        $('div#confirmSuspendModal').modal('hide');
        pos_form_obj.submit();
        $('input#is_suspend').val(0);
    });

    //fix select2 input issue on modal
    $('#modal_payment')
        .find('.select2')
        .each(function() {
            $(this).select2({
                dropdownParent: $('#modal_payment'),
            });
        });


        // Centralized submit handlers
        $(document).on('click', 'button[type="submit"], #pos-finalize, #pos-draft, #pos-quotation, .pos-express-finalize', function() {
            var buttonId = $(this).attr('id');
            var buttonClass = $(this).attr('class');
            var statusValue = StatusManager.getStatusFromButton(buttonId, buttonClass);
            
            StatusManager.setFormStatus(statusValue);
            console.log('Status field added with value:', statusValue);
        });

        // Setup AJAX send handler
        $(document).ajaxSend(function(event, jqxhr, settings) {
            if (settings.url && settings.url.includes('/pos/create')) {
                if (settings.data && settings.data.indexOf('status=') === -1) {
                    settings.data += '&status=' + StatusManager.defaultStatus;
                }
            }
        });



    $('button#add-payment-row').click(function() {
        var row_index = $('#payment_row_index').val();
        var location_id = $('input#location_id').val();
        $.ajax({
            method: 'POST',
            url: '/sells/pos/get_payment_row',
            data: { row_index: row_index, location_id: location_id },
            dataType: 'html',
            success: function(result) {
                if (result) {
                    var appended = $('#payment_rows_div').append(result);

                    var total_payable = __read_number($('input#final_total_input'));
                    var total_paying = __read_number($('input#total_paying_input'));
                    var b_due = total_payable - total_paying;
                    $(appended)
                        .find('input.payment-amount')
                        .focus();
                    $(appended)
                        .find('input.payment-amount')
                        .last()
                        .val(__currency_trans_from_en(b_due, false))
                        .change()
                        .select();
                    __select2($(appended).find('.select2'));
                    $(appended).find('#method_' + row_index).change();
                    $('#payment_row_index').val(parseInt(row_index) + 1);
                }
            },
        });
    });

    $(document).on('click', '.remove_payment_row', function() {
        swal({
            title: LANG.sure,
            icon: 'warning',
            buttons: true,
            dangerMode: true,
        }).then(willDelete => {
            if (willDelete) {
                $(this)
                    .closest('.payment_row')
                    .remove();
                calculate_balance_due();
            }
        });
    });

    pos_form_validator = pos_form_obj.validate({
        submitHandler: function(form) {
            // var total_payble = __read_number($('input#final_total_input'));
            // var total_paying = __read_number($('input#total_paying_input'));
            var cnf = true;

            //Ignore if the difference is less than 0.5
            if ($('input#in_balance_due').val() >= 0.5) {
                cnf = confirm(LANG.paid_amount_is_less_than_payable);
                // if( total_payble > total_paying ){
                // 	cnf = confirm( LANG.paid_amount_is_less_than_payable );
                // } else if(total_payble < total_paying) {
                // 	alert( LANG.paid_amount_is_more_than_payable );
                // 	cnf = false;
                // }
            }

            var total_advance_payments = 0;
            $('#payment_rows_div').find('select.payment_types_dropdown').each( function(){
                if ($(this).val() == 'advance') {
                    total_advance_payments++
                };
            });

            if (total_advance_payments > 1) {
                alert(LANG.advance_payment_cannot_be_more_than_once);
                return false;
            }

            var is_msp_valid = FormValidator.validateMinimumSellingPrice();
            if (!is_msp_valid) {
                return false;
            }

            if (cnf) {
                disable_pos_form_actions();

                var data = $(form).serialize();
                data = data + '&status=final';
                var url = $(form).attr('action');
                $.ajax({
                    method: 'POST',
                    url: url,
                    data: data,
                    dataType: 'json',
                    success: function(result) {
                        if (result.success == 1) {
                            // Check for SAP sync errors
                            if (result.sap_error) {
                                toastr.warning('تحذير SAP: ' + result.sap_error);
                            }
                            
                            if (result.whatsapp_link) {
                                window.open(result.whatsapp_link);
                            }
                            $('#modal_payment').modal('hide');
                            toastr.success(result.msg);

                            reset_pos_form();

                            //Check if enabled or not
                            if (result.receipt.is_enabled) {
                                pos_print(result.receipt);
                            }
                        } else {
                            toastr.error(result.msg);
                        }

                        enable_pos_form_actions();
                    },
                });
            }
            return false;
        },
    });

    $(document).on('change', '.payment-amount', function() {
        calculate_balance_due();
    });

    //Update discount
    $('button#posEditDiscountModalUpdate').click(function() {

        //if discount amount is not valid return false
        if (!$("#discount_amount_modal").valid()) {
            return false;
        }
        //Close modal
        $('div#posEditDiscountModal').modal('hide');

        //Update values
        $('input#discount_type').val($('select#discount_type_modal').val());
        __write_number($('input#discount_amount'), __read_number($('input#discount_amount_modal')));

        if ($('#reward_point_enabled').length) {
            var reward_validation = isValidatRewardPoint();
            if (!reward_validation['is_valid']) {
                toastr.error(reward_validation['msg']);
                $('#rp_redeemed_modal').val(0);
                $('#rp_redeemed_modal').change();
            }
            updateRedeemedAmount();
        }

        pos_total_row();
    });

    //Shipping
    $('button#posShippingModalUpdate').click(function() {
        //Close modal
        $('div#posShippingModal').modal('hide');

        //update shipping details
        $('input#shipping_details').val($('#shipping_details_modal').val());

        $('input#shipping_address').val($('#shipping_address_modal').val());
        $('input#shipping_status').val($('#shipping_status_modal').val());
        $('input#delivered_to').val($('#delivered_to_modal').val());
        $('input#delivery_person').val($('#delivery_person_modal').val());

        //Update shipping charges
        __write_number(
            $('input#shipping_charges'),
            __read_number($('input#shipping_charges_modal'))
        );

        //$('input#shipping_charges').val(__read_number($('input#shipping_charges_modal')));

        pos_total_row();
    });

    $('#posShippingModal').on('shown.bs.modal', function() {
        $('#posShippingModal')
            .find('#shipping_details_modal')
            .filter(':visible:first')
            .focus()
            .select();
            // $('.select2-selection__rendered').css('padding-right', '150px');
    });

    $(document).on('shown.bs.modal', '.row_edit_product_price_model', function() {
        $('.row_edit_product_price_model')
            .find('input')
            .filter(':visible:first')
            .focus()
            .select();
    });

    //Update Order tax
    $('button#posEditOrderTaxModalUpdate').click(function() {
        //Close modal
        $('div#posEditOrderTaxModal').modal('hide');

        var tax_obj = $('select#order_tax_modal');
        var tax_id = tax_obj.val();
        var tax_rate = tax_obj.find(':selected').data('rate');

        $('input#tax_rate_id').val(tax_id);

        __write_number($('input#tax_calculation_amount'), tax_rate);
        pos_total_row();
    });

    $(document).on('click', '.add_new_customer', function() {
        $('#customer_id').select2('close');
        var name = $(this).data('name');
        $('.contact_modal')
            .find('input#name')
            .val(name);
        $('.contact_modal')
            .find('select#contact_type')
            .val('customer')
            .closest('div.contact_type_div')
            .addClass('hide');
        $('.contact_modal').modal('show');
    });
    $('form#quick_add_contact')
        .submit(function(e) {
            e.preventDefault();
        })
        .validate({
            rules: {
                contact_id: {
                    remote: {
                        url: '/contacts/check-contacts-id',
                        type: 'post',
                        data: {
                            contact_id: function() {
                                return $('#contact_id').val();
                            },
                            hidden_id: function() {
                                if ($('#hidden_id').length) {
                                    return $('#hidden_id').val();
                                } else {
                                    return '';
                                }
                            },
                        },
                    },
                },
            },
            messages: {
                contact_id: {
                    remote: LANG.contact_id_already_exists,
                },
            },
            submitHandler: function(form) {
                $.ajax({
                    method: 'POST',
                    url: base_path + '/check-mobile',
                    dataType: 'json',
                    data: {
                        contact_id: function() {
                            return $('#hidden_id').val();
                        },
                        mobile_number: function() {
                            return $('#mobile').val();
                        },
                    },
                    beforeSend: function(xhr) {
                        __disable_submit_button($(form).find('button[type="submit"]'));
                    },
                    success: function(result) {
                        if (result.is_mobile_exists == true) {
                            swal({
                                title: LANG.sure,
                                text: result.msg,
                                icon: 'warning',
                                buttons: true,
                                dangerMode: true,
                            }).then(willContinue => {
                                if (willContinue) {
                                    submitQuickContactForm(form);
                                } else {
                                    $('#mobile').select();
                                }
                            });
                            
                        } else {
                            submitQuickContactForm(form);
                        }
                    },
                });
            },
        });
    $('.contact_modal').on('hidden.bs.modal', function() {
        $('form#quick_add_contact')
            .find('button[type="submit"]')
            .removeAttr('disabled');
        $('form#quick_add_contact')[0].reset();
    });

    //Updates for add sell
    $('select#discount_type, input#discount_amount, input#shipping_charges, \
        input#rp_redeemed_amount').change(function() {
        pos_total_row();
    });
    $('select#tax_rate_id').change(function() {
        var tax_rate = $(this)
            .find(':selected')
            .data('rate');
        __write_number($('input#tax_calculation_amount'), tax_rate);
        pos_total_row();
    });
    //Datetime picker
    $('#transaction_date').datetimepicker({
        format: moment_date_format + ' ' + moment_time_format,
        ignoreReadonly: true,
    });

    //Direct sell submit
    sell_form = $('form#add_sell_form');
    if ($('form#edit_sell_form').length) {
        sell_form = $('form#edit_sell_form');
        pos_total_row();
    }
    sell_form_validator = sell_form.validate();

    $('button#submit-sell, button#save-and-print').click(function(e) {
        //Check if product is present or not.
        if ($('table#pos_table tbody').find('.product_row').length <= 0) {
            toastr.warning(LANG.no_products_added);
            return false;
        }

        var is_msp_valid = true;
        //Validate minimum selling price if hidden
        $('.pos_unit_price_inc_tax').each( function(){
            if (!$(this).is(":visible") && $(this).data('rule-min-value')) {
                var val = __read_number($(this));
                var error_msg_td = $(this).closest('tr').find('.pos_line_total_text').closest('td');
                if (val > $(this).data('rule-min-value')) {
                    is_msp_valid = false;
                    error_msg_td.append( '<label class="error">' + $(this).data('msg-min-value') + '</label>');
                } else {
                    error_msg_td.find('label.error').remove();
                }
            }
        });

        if (!is_msp_valid) {
            return false;
        }

        if ($(this).attr('id') == 'save-and-print') {
            $('#is_save_and_print').val(1);           
        } else {
            $('#is_save_and_print').val(0);
        }

        if ($('#reward_point_enabled').length) {
            var validate_rp = isValidatRewardPoint();
            if (!validate_rp['is_valid']) {
                toastr.error(validate_rp['msg']);
                return false;
            }
        }

        if ($('.enable_cash_denomination_for_payment_methods').length) {
            var payment_row = $('.enable_cash_denomination_for_payment_methods').closest('.payment_row');
            var is_valid = true;
            var payment_type = payment_row.find('.payment_types_dropdown').val();
            var denomination_for_payment_types = JSON.parse($('.enable_cash_denomination_for_payment_methods').val());
            if (denomination_for_payment_types.includes(payment_type) && payment_row.find('.is_strict').length && payment_row.find('.is_strict').val() === '1' ) {
                var payment_amount = __read_number(payment_row.find('.payment-amount'));
                var total_denomination = payment_row.find('input.denomination_total_amount').val();
                if (payment_amount != total_denomination ) {
                    is_valid = false;
                }
            }

            if (!is_valid) {
                payment_row.find('.cash_denomination_error').removeClass('hide');
                toastr.error(payment_row.find('.cash_denomination_error').text());
                e.preventDefault();
                return false;
            } else {
                payment_row.find('.cash_denomination_error').addClass('hide');
            }
        }

        if (sell_form.valid()) {
            window.onbeforeunload = null;
            $(this).attr('disabled', true);
            sell_form.submit();
        }
    });

    //REPAIR MODULE:check if repair module field is present send data to filter product
    var is_enabled_stock = null;
    if ($("#is_enabled_stock").length) {
        is_enabled_stock = $("#is_enabled_stock").val();
    }

    var device_model_id = null;
    if ($("#repair_model_id").length) {
        device_model_id = $("#repair_model_id").val();
    }

    //Show product list.
    get_product_suggestion_list(
        global_p_category_id,
        global_brand_id,
        $('input#location_id').val(),
        null,
        is_enabled_stock,
        device_model_id
    );
    
    $('select#select_location_id').on('change', function(e) {
        $('input#suggestion_page').val(1);
        var location_id = $('input#location_id').val();
        if (location_id != '' || location_id != undefined) {
            get_product_suggestion_list(
                global_p_category_id,
                global_brand_id,
                $('input#location_id').val(),
                null
            );
        }
        get_featured_products();
    });

// on click sub category in category drawer
    $('.product_category').on('click', function(e) {
        global_p_category_id = $(this).data('value');
        $('input#suggestion_page').val(1);
        var location_id = $('input#location_id').val();
        if (location_id != '' || location_id != undefined) {
            get_product_suggestion_list(
                global_p_category_id,
                global_brand_id,
                $('input#location_id').val(),
                null
            );
        }
        get_featured_products();
        $('.overlay-category').trigger('click');
    });

    //  function for show sub category 
    $('.main-category').on('click', function(){

        global_p_category_id = $(this).data('value');
        parent = $(this).data('parent');

        if (parent == 0) {
            get_product_suggestion_list(
                global_p_category_id,
                global_brand_id,
                $('input#location_id').val(),
                null
            );
            get_featured_products();
            $('.overlay-category').trigger('click');
        }
        else {
            var main_category = $(this).data('value');

            $('.main-category-div').hide();
            $('.'+ main_category).fadeIn();
            $('.category_heading').text('Sub Category ' + $(this).data('name'));
            $('.category-back').fadeIn();
        }
    })

    // function for back button in category 
    $('.category-back').on('click', function(){
        $('.main-category-div').fadeIn();
        $('.main-category-all').fadeIn();
        $('.all-sub-category').hide();
        $('.category-back').hide();
        $('.category_heading').text('Category');
    });

    // on click brand in brand drawer 
    $('.product_brand').on('click', function(e) {
        global_brand_id = $(this).data('value');
        $('input#suggestion_page').val(1);
        var location_id = $('input#location_id').val();

        if (location_id != '' || location_id != undefined) {
            get_product_suggestion_list(
                global_p_category_id,
                global_brand_id,
                $('input#location_id').val(),
                null
            );
        }
        get_featured_products();
        $('.overlay-brand').trigger('click');
    });

    // close side bar 

    $('.close-side-bar-category').on('click', function() {
        $('.overlay-category').trigger('click');
    });

    $('.close-side-bar-brand').on('click', function() {
        $('.overlay-brand').trigger('click');
    });


    

    $(document).on('click', 'div.product_box', function() {
        //Check if location is not set then show error message.
        if ($('input#location_id').val() == '') {
            toastr.warning(LANG.select_location);
        } else {
            pos_product_row($(this).data('variation_id'));
        }
    });

    $(document).on('shown.bs.modal', '.row_description_modal', function() {
        $(this)
            .find('textarea')
            .first()
            .focus();
    });

    //Press enter on search product to jump into last quantty and vice-versa
    $('#search_product').keydown(function(e) {
        var key = e.which;
        if (key == 9) {
            // the tab key code
            e.preventDefault();
            if ($('#pos_table tbody tr').length > 0) {
                $('#pos_table tbody tr:last')
                    .find('input.pos_quantity')
                    .focus()
                    .select();
            }
        }
    });
    $('#pos_table').on('keypress', 'input.pos_quantity', function(e) {
        var key = e.which;
        if (key == 13) {
            // the enter key code
            $('#search_product').focus();
        }
    });

    $('#exchange_rate').change(function() {
        var curr_exchange_rate = 1;
        if ($(this).val()) {
            curr_exchange_rate = __read_number($(this));
        }
        var total_payable = __read_number($('input#final_total_input'));
        var shown_total = total_payable * curr_exchange_rate;
        $('span#total_payable').text(__currency_trans_from_en(shown_total, false));
    });

    $('select#price_group').change(function() {
        $('input#hidden_price_group').val($(this).val());
    });

    //Quick add product
    $(document).on('click', 'button.pos_add_quick_product', function() {
        var url = $(this).data('href');
        var container = $(this).data('container');
        $.ajax({
            url: url + '?product_for=pos',
            dataType: 'html',
            success: function(result) {
                $(container)
                    .html(result)
                    .modal('show');
                $('.os_exp_date').datepicker({
                    autoclose: true,
                    format: 'dd-mm-yyyy',
                    clearBtn: true,
                });
            },
        });
    });

    $(document).on('change', 'form#quick_add_product_form input#single_dpp', function() {
        var unit_price = __read_number($(this));
        $('table#quick_product_opening_stock_table tbody tr').each(function() {
            var input = $(this).find('input.unit_price');
            __write_number(input, unit_price);
            input.change();
        });
    });

    $(document).on('quickProductAdded', function(e) {
        //Check if location is not set then show error message.
        if ($('input#location_id').val() == '') {
            toastr.warning(LANG.select_location);
        } else {
            pos_product_row(e.variation.id);
        }
    });

    $('div.view_modal').on('show.bs.modal', function() {
        __currency_convert_recursively($(this));
    });

    $('table#pos_table').on('change', 'select.sub_unit', function() {
        var tr = $(this).closest('tr');
        
        // إذا كان نظام التسعير المتقدم نشطاً، لا تتداخل معه
        if (tr.hasClass('advanced-pricing-active')) {
            console.log('🛡️ Dropdown handler blocked - Advanced pricing is active');
            return;
        }
        
        var base_unit_selling_price = tr.find('input.hidden_base_unit_sell_price').val();

        var selected_option = $(this).find(':selected');
        var unitText = selected_option.text().trim();

        var multiplier = parseFloat(selected_option.data('multiplier'));

        var allow_decimal = parseInt(selected_option.data('allow_decimal'));

        tr.find('input.base_unit_multiplier').val(multiplier);

        // تعامل خاص مع وحدة السيت
        var isSetUnit = unitText.toLowerCase().includes('سيت') || unitText.toLowerCase().includes('set');
        var unit_sp;
        
        if (isSetUnit) {
            // للسيت: استخدم السعر الأساسي مباشرة بدون ضرب في المضاعف
            unit_sp = parseFloat(base_unit_selling_price);
            console.log('🎯 Set unit in dropdown - using base price directly');
        } else {
            // للوحدات الأخرى: استخدم المضاعف كالمعتاد
            unit_sp = base_unit_selling_price * multiplier;
        }

        var sp_element = tr.find('input.pos_unit_price');
        __write_number(sp_element, unit_sp);

        sp_element.change();
        
        // تسجيل لأغراض التصحيح
        console.log('📦 Dropdown unit changed:', {
            unit: unitText,
            multiplier: multiplier,
            isSetUnit: isSetUnit,
            basePrice: base_unit_selling_price,
            finalPrice: unit_sp
        });

        var qty_element = tr.find('input.pos_quantity');
        var base_max_avlbl = qty_element.data('qty_available');
        var error_msg_line = 'pos_max_qty_error';

        if (tr.find('select.lot_number').length > 0) {
            var lot_select = tr.find('select.lot_number');
            if (lot_select.val()) {
                base_max_avlbl = lot_select.find(':selected').data('qty_available');
                error_msg_line = 'lot_max_qty_error';
            }
        }

        qty_element.attr('data-decimal', allow_decimal);
        var abs_digit = true;
        if (allow_decimal) {
            abs_digit = false;
        }
        qty_element.rules('add', {
            abs_digit: abs_digit,
        });

        if (base_max_avlbl) {
            var max_avlbl = parseFloat(base_max_avlbl) / multiplier;
            var formated_max_avlbl = __number_f(max_avlbl);
            var unit_name = selected_option.data('unit_name');
            var max_err_msg = __translate(error_msg_line, {
                max_val: formated_max_avlbl,
                unit_name: unit_name,
            });
            qty_element.attr('data-rule-max-value', max_avlbl);
            qty_element.attr('data-msg-max-value', max_err_msg);
            qty_element.rules('add', {
                'max-value': max_avlbl,
                messages: {
                    'max-value': max_err_msg,
                },
            });
            qty_element.trigger('change');
        }
     //   adjustComboQty(tr);
    });

    //Confirmation before page load.
    window.onbeforeunload = function() {
        if($('form#edit_pos_sell_form').length == 0){
            if($('table#pos_table tbody tr').length > 0) {
                return LANG.sure;
            } else {
                return null;
            }
        }
    }
    $(window).resize(function() {
        var win_height = $(window).height();
        div_height = __calculate_amount('percentage', 63, win_height);
        // $('div.pos_product_div').css('min-height', div_height + 'px');
        // $('div.pos_product_div').css('max-height', div_height + 'px');
    });

    //Used for weighing scale barcode
    $('#weighing_scale_modal').on('shown.bs.modal', function (e) {

        //Attach the scan event
        onScan.attachTo(document, {
            suffixKeyCodes: [13], // enter-key expected at the end of a scan
            reactToPaste: true, // Compatibility to built-in scanners in paste-mode (as opposed to keyboard-mode)
            onScan: function(sCode, iQty) {
                console.log('Scanned: ' + iQty + 'x ' + sCode); 
                $('input#weighing_scale_barcode').val(sCode);
                $('button#weighing_scale_submit').trigger('click');
            },
            onScanError: function(oDebug) {
                console.log(oDebug); 
            },
            minLength: 2
            // onKeyDetect: function(iKeyCode){ // output all potentially relevant key events - great for debugging!
            //     console.log('Pressed: ' + iKeyCode);
            // }
        });

        $('input#weighing_scale_barcode').focus();
    });

    $('#weighing_scale_modal').on('hide.bs.modal', function (e) {
        //Detach from the document once modal is closed.
        onScan.detachFrom(document);
    });

    $('button#weighing_scale_submit').click(function(){

        var price_group = '';
        if ($('#price_group').length > 0) {
            price_group = $('#price_group').val();
        }

        if($('#weighing_scale_barcode').val().length > 0){
            pos_product_row(null, null, $('#weighing_scale_barcode').val());
            $('#weighing_scale_modal').modal('hide');
            $('input#weighing_scale_barcode').val('');
        } else{
            $('input#weighing_scale_barcode').focus();
        }
    });

    $('#show_featured_products').click( function(){
        if (!$('#featured_products_box').is(':visible')) {
            $('#featured_products_box').fadeIn();
        } else {
            $('#featured_products_box').fadeOut();
        }
    });
    validate_discount_field();
    set_payment_type_dropdown();
    if ($('#__is_mobile').length) {
        $('.pos_form_totals').css('margin-bottom', $('.pos-form-actions').height() - 30);
    }

    setInterval(function () {
        if ($('span.curr_datetime').length) {
            $('span.curr_datetime').html(__current_datetime());
        }
    }, 60000);

    set_search_fields();
});

function set_payment_type_dropdown() {
    var payment_settings = $('#location_id').data('default_payment_accounts');
    payment_settings = payment_settings ? payment_settings : [];
    enabled_payment_types = [];
    for (var key in payment_settings) {
        if (payment_settings[key] && payment_settings[key]['is_enabled']) {
            enabled_payment_types.push(key);
        }
    }
    if (enabled_payment_types.length) {
        $(".payment_types_dropdown > option").each(function() {
            //skip if advance
            if ($(this).val() && $(this).val() != 'advance') {
                if (enabled_payment_types.indexOf($(this).val()) != -1) {
                    $(this).removeClass('hide');
                } else {
                    $(this).addClass('hide');
                }
            }
        });
    }
}

function get_featured_products() {
    var location_id = $('#location_id').val();
    if (location_id && $('#featured_products_box').length > 0) {
        $.ajax({
            method: 'GET',
            url: '/sells/pos/get-featured-products/' + location_id,
            dataType: 'html',
            success: function(result) {
                if (result) {
                    $('#feature_product_div').removeClass('hide');
                    $('#featured_products_box').html(result);
                } else {
                    $('#feature_product_div').addClass('hide');
                    $('#featured_products_box').html('');
                }
            },
        });
    } else {
        $('#feature_product_div').addClass('hide');
        $('#featured_products_box').html('');
    }
}

function get_product_suggestion_list(category_id, brand_id, location_id, url = null, is_enabled_stock = null, repair_model_id = null) {
    if($('div#product_list_body').length == 0) {
        return false;
    }

    if (url == null) {
        url = '/sells/pos/get-product-suggestion';
    }
    $('#suggestion_page_loader').fadeIn(700);
    var page = $('input#suggestion_page').val();
    if (page == 1) {
        $('div#product_list_body').html('');
    }
    if ($('div#product_list_body').find('input#no_products_found').length > 0) {
        $('#suggestion_page_loader').fadeOut(700);
        return false;
    }
    $.ajax({
        method: 'GET',
        url: url,
        data: {
            category_id: category_id,
            brand_id: brand_id,
            location_id: location_id,
            page: page,
            is_enabled_stock: is_enabled_stock,
            repair_model_id: repair_model_id
        },
        dataType: 'html',
        success: function(result) {
            $('div#product_list_body').append(result);
            $('#suggestion_page_loader').fadeOut(700);
        },
    });
}

//Get recent transactions
function get_recent_transactions(status, element_obj) {
    if (element_obj.length == 0) {
        return false;
    }
    var transaction_sub_type = $("#transaction_sub_type").val();
    $.ajax({
        method: 'GET',
        url: '/sells/pos/get-recent-transactions',
        data: { status: status , transaction_sub_type: transaction_sub_type},
        dataType: 'html',
        success: function(result) {
            element_obj.html(result);
            __currency_convert_recursively(element_obj);
        },
    });
}

//variation_id is null when weighing_scale_barcode is used.
// تعديل دالة pos_product_row لتكون أكثر تفصيلاً
function pos_product_row(variation_id = null, purchase_line_id = null, weighing_scale_barcode = null, quantity = 1) {
    //Get item addition method
    var item_addtn_method = 0;
    var add_via_ajax = true;

    if (variation_id != null && $('#item_addition_method').length) {
        item_addtn_method = $('#item_addition_method').val();
    }

    if (item_addtn_method == 0) {
        add_via_ajax = true;
    } else {
        var is_added = false;

        //Search for variation id in each row of pos table
        $('#pos_table tbody')
            .find('tr')
            .each(function() {
                var row_v_id = $(this)
                    .find('.row_variation_id')
                    .val();
                var enable_sr_no = $(this)
                    .find('.enable_sr_no')
                    .val();
                var modifiers_exist = false;
                if ($(this).find('input.modifiers_exist').length > 0) {
                    modifiers_exist = true;
                }

                if (
                    row_v_id == variation_id &&
                    enable_sr_no !== '1' &&
                    !modifiers_exist &&
                    !is_added
                ) {
                    add_via_ajax = false;
                    is_added = true;

                    //Increment product quantity
                    qty_element = $(this).find('.pos_quantity');
                    var qty = __read_number(qty_element);
                    __write_number(qty_element, qty + 1);
                    qty_element.change();

                    round_row_to_iraqi_dinnar($(this));

                    $('input#search_product')
                        .focus()
                        .select();
                }
            });
    }

    if (add_via_ajax) {
        var product_row = $('input#product_row_count').val();
        var location_id = $('input#location_id').val();
        var customer_id = $('select#customer_id').val();
        var is_direct_sell = false;
        if (
            $('input[name="is_direct_sale"]').length > 0 &&
            $('input[name="is_direct_sale"]').val() == 1
        ) {
            is_direct_sell = true;
        }

        var disable_qty_alert = false;

        if ($('#disable_qty_alert').length) {
            disable_qty_alert = true;
        }

        var is_sales_order = $('#sale_type').length && $('#sale_type').val() == 'sales_order' ? true : false;

        var price_group = '';
        if ($('#price_group').length > 0) {
            price_group = parseInt($('#price_group').val());
        }

        //If default price group present
        if ($('#default_price_group').length > 0 && 
            price_group === '') {
            price_group = $('#default_price_group').val();
        }

        //If types of service selected give more priority
        if ($('#types_of_service_price_group').length > 0 && 
            $('#types_of_service_price_group').val()) {
            price_group = $('#types_of_service_price_group').val();
        }

        var is_draft=false;
        if($('#status') && ($('#status').val()=='quotation' || 
        $('#status').val()=='draft')) {
            is_draft=true;
        }

        var is_serial_no = false;

        if (
            $('input[name="is_serial_no"]').length > 0 &&
            $('input[name="is_serial_no"]').val() == 1
        ) {
            is_serial_no = true;
        }
        
        $.ajax({
            method: 'GET',
            url: '/sells/pos/get_product_row/' + variation_id + '/' + location_id,
            async: false,
            data: {
                product_row: product_row,
                customer_id: customer_id,
                is_direct_sell: is_direct_sell,
                is_serial_no: is_serial_no,
                price_group: price_group,
                purchase_line_id: purchase_line_id,
                weighing_scale_barcode: weighing_scale_barcode,
                quantity: quantity,
                is_sales_order: is_sales_order,
                disable_qty_alert: disable_qty_alert,
                is_draft: is_draft
            },
            dataType: 'json',
            success: function(result) {
                if (result.success) {
                    $('table#pos_table tbody')
                        .append(result.html_content)
                        .find('input.pos_quantity');
                    
                    //increment row count
                    $('input#product_row_count').val(parseInt(product_row) + 1);
                    
                    var this_row = $('table#pos_table tbody')
                        .find('tr')
                        .last();
                    
                    // التأكد من وجود جميع الحقول المطلوبة
                    var rowIndex = parseInt(product_row);
                    
                    // التأكد من وجود حقل unit_price
                    if (this_row.find('input.pos_unit_price').length === 0 || 
                        !this_row.find('input.pos_unit_price').attr('name')) {
                        
                        var unit_price_val = 0;
                        if (this_row.find('input.pos_unit_price_inc_tax').length > 0) {
                            var unit_price_inc_tax = __read_number(this_row.find('input.pos_unit_price_inc_tax'));
                            var tax_rate = this_row.find('select.tax_id').find(':selected').data('rate') || 0;
                            unit_price_val = tax_rate > 0 ? 
                                __get_principle(unit_price_inc_tax, tax_rate) : 
                                unit_price_inc_tax;
                        }
                        
                        if (this_row.find('input[name="products[' + rowIndex + '][unit_price]"]').length === 0) {
                            this_row.append('<input type="hidden" class="pos_unit_price" name="products[' + rowIndex + '][unit_price]" value="' + unit_price_val + '">');
                        }
                    }
                    
                    // التأكد من وجود حقل unit_price_inc_tax
                    if (this_row.find('input.pos_unit_price_inc_tax').length === 0 ||
                        !this_row.find('input.pos_unit_price_inc_tax').attr('name')) {
                        
                        var unit_price_inc_tax_val = 0;
                        if (this_row.find('input.pos_unit_price').length > 0) {
                            var unit_price = __read_number(this_row.find('input.pos_unit_price'));
                            var tax_rate = this_row.find('select.tax_id').find(':selected').data('rate') || 0;
                            unit_price_inc_tax_val = __add_percent(unit_price, tax_rate);
                        }
                        
                        if (this_row.find('input[name="products[' + rowIndex + '][unit_price_inc_tax]"]').length === 0) {
                            this_row.append('<input type="hidden" class="pos_unit_price_inc_tax" name="products[' + rowIndex + '][unit_price_inc_tax]" value="' + unit_price_inc_tax_val + '">');
                        }
                    }
                    
                    // التأكد من وجود حقل line_total
                    if (this_row.find('input.pos_line_total').length === 0 ||
                        !this_row.find('input.pos_line_total').attr('name')) {
                        
                        var line_total_val = 0;
                        var qty = __read_number(this_row.find('input.pos_quantity')) || quantity;
                        var unit_price_inc_tax = __read_number(this_row.find('input.pos_unit_price_inc_tax')) || 0;
                        line_total_val = qty * unit_price_inc_tax;
                        
                        if (this_row.find('input[name="products[' + rowIndex + '][line_total]"]').length === 0) {
                            this_row.append('<input type="hidden" class="pos_line_total" name="products[' + rowIndex + '][line_total]" value="' + line_total_val + '">');
                        }
                    }
                    
                    // التأكد من وجود حقل quantity
                    if (this_row.find('input.pos_quantity').length === 0 ||
                        !this_row.find('input.pos_quantity').attr('name')) {
                        
                        var qty_val = quantity;
                        if (this_row.find('input[name="products[' + rowIndex + '][quantity]"]').length === 0) {
                            this_row.append('<input type="hidden" class="pos_quantity" name="products[' + rowIndex + '][quantity]" value="' + qty_val + '">');
                        }
                    }
                    
                    pos_each_row(this_row);

                    //For initial discount if present
                    var line_total = __read_number(this_row.find('input.pos_line_total'));
                    this_row.find('span.pos_line_total_text').text(line_total);

                    pos_total_row();

                    //Check if multipler is present then multiply it when a new row is added.
                    if(__getUnitMultiplier(this_row) > 1){
                        this_row.find('select.sub_unit').trigger('change');
                    }

                    if (result.enable_sr_no == '1') {
                        var new_row = $('table#pos_table tbody')
                            .find('tr')
                            .last();
                        new_row.find('.row_edit_product_price_model').modal('show');
                    }

                    round_row_to_iraqi_dinnar(this_row);
                    __currency_convert_recursively(this_row);

                    // جلب بيانات المنتج الكاملة وتطبيق التسعير المتقدم
                    setTimeout(function() {
                        var variationId = this_row.find('.variation_id').val();
                        var productId = this_row.find('.product_id').val();
                        
                        if (variationId || productId) {
                            console.log('🔍 Fetching complete product data for advanced pricing');
                            
                            // جلب بيانات المنتج الكاملة من الخادم
                            $.ajax({
                                url: base_path + '/products/list',
                                method: 'GET',
                                dataType: 'json',
                                data: {
                                    variation_id: variationId,
                                    product_id: productId,
                                    with_sub_units: true,
                                    include_unit_details: true,
                                    load_sub_units: true
                                },
                                success: function(products) {
                                    if (products && products.length > 0) {
                                        var product = products[0];
                                        
                                        // حفظ بيانات المنتج في الصف
                                        this_row.data('product-data', product);
                                        
                                        console.log('💾 Complete product data saved:', {
                                            name: product.name,
                                            sku: product.sku,
                                            base_unit: product.product_custom_field2,
                                            has_price_array: !!product.product_custom_field3
                                        });
                                        
                                        // تطبيق التسعير المباشر للكارتون من المصفوفة أولاً
                                        var directPricingApplied = false;
                                        if (typeof forceCartonPricingFromArray === 'function') {
                                            directPricingApplied = forceCartonPricingFromArray(this_row);
                                        }
                                        
                                        // إذا لم يتم تطبيق التسعير المباشر، استخدم النظام المتقدم
                                        if (!directPricingApplied && typeof applyAdvancedPricingToRow === 'function') {
                                            console.log('🎯 Applying advanced pricing with complete data');
                                            applyAdvancedPricingToRow(this_row);
                                            
                                            // إعادة حساب الإجماليات بعد تطبيق السعر الجديد
                                            setTimeout(function() {
                                                pos_each_row(this_row);
                                                pos_total_row();
                                                console.log('✅ Advanced pricing applied and totals recalculated');
                                            }, 100);
                                        }
                                    }
                                },
                                error: function() {
                                    console.log('⚠️ Could not fetch complete product data - applying basic advanced pricing');
                                    
                                    // في حالة فشل جلب البيانات، طبق التسعير المتقدم على أي حال
                                    if (typeof applyAdvancedPricingToRow === 'function') {
                                        applyAdvancedPricingToRow(this_row);
                                    }
                                }
                            });
                        } else {
                            console.log('⚠️ No variation_id or product_id found in row');
                        }
                    }, 200);

                    $('input#search_product')
                        .focus()
                        .select();

                    //Used in restaurant module
                    if (result.html_modifier) {
                        $('table#pos_table tbody')
                            .find('tr')
                            .last()
                            .find('td:first')
                            .append(result.html_modifier);
                    }

                    //scroll bottom of items list
                    $(".pos_product_div").animate({ scrollTop: $('.pos_product_div').prop("scrollHeight")}, 1000);
                } else {
                    toastr.error(result.msg);
                    $('input#search_product')
                        .focus()
                        .select();
                }
            },
        });
    }
}


// دالة لتحسين الصف بإضافة الحقول الجديدة
function enhanceProductRow(row, result) {
    // تنظيف الصف أولاً
    var cells = row.find('td');
    var original_content = {};
    
    // حفظ المحتوى الأصلي
    cells.each(function(index, cell) {
        original_content[index] = $(cell).html();
    });
    
    // إعادة بناء الصف بالترتيب الصحيح
    row.empty();
    
    // 1. رقم تسلسلي
    row.append('<td class="serial_no text-center">' + ($('#pos_table tbody tr').length) + '</td>');
    
    // 2. اسم المنتج
    row.append('<td class="product_name">' + original_content[0] + '</td>');
    
    // 3. SKU
    var sku = row.find('.label:contains(":")').text().split(':')[1] || '-';
    if (!sku || sku === '-') {
        // استخراج SKU من النص
        var productInfo = $(original_content[0]).find('span.label').text();
        if (productInfo.includes(':')) {
            sku = productInfo.split(':')[1].trim();
        }
    }
    row.append('<td class="text-center">' + sku + '</td>');
    
    // 4. وحدة القياس
    var unit_html = '';
    var unit_selector = $(original_content[0]).find('select.sub_unit');
    if (unit_selector.length > 0) {
        unit_html = unit_selector[0].outerHTML;
    } else {
        var unit_text = $(original_content[0]).find('.input-group-addon').text() || 'EA';
        unit_html = '<span class="unit-display">' + unit_text + '</span>';
    }
    row.append('<td class="text-center">' + unit_html + '</td>');
    
    // 5. الكمية
    row.append('<td class="text-center">' + original_content[1] + '</td>');
    
    // 6. السعر USD
    var price_inc_tax = __read_number($(original_content[2]).find('input.pos_unit_price_inc_tax'));
    var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
    var price_usd = price_inc_tax / exchange_rate;
    row.append('<td class="text-center">' +
        '<input type="text" class="form-control pos_unit_price_usd" value="' + __number_f(price_usd, false, false, 2) + '">' +
        '</td>');
    
    // 7. السعر IQD
    var price_iqd = convertUSDToIQD(price_usd);
    row.append('<td class="text-center">' +
        '<input type="text" class="form-control pos_unit_price_iqd" value="' + __number_f(price_iqd, false, false, 0) + '">' +
        '</td>');
    
    // 8. نسبة الخصم
    var discount_html = '<input type="text" class="form-control discount_percent" value="0">';
    row.append('<td class="text-center">' + discount_html + '</td>');
    
    // 9. السعر شامل الضريبة
    row.append('<td class="text-center">' + original_content[2] + '</td>');
    
    // 10. الإجمالي الفرعي
    row.append('<td class="text-right">' + original_content[3] + '</td>');
    
    // 11. المستودع
    row.append('<td class="text-center">' +
        '<select class="form-control warehouse-selector">' +
        '<option value="">Default</option>' +
        '</select>' +
        '</td>');
    
    // 12. معلومات المخزون
    var qty_available = $(original_content[1]).find('input.pos_quantity').data('qty_available');
    var stock_info = qty_available ? __number_f(qty_available) : 'N/A';
    row.append('<td class="stock-info-cell text-center">' + stock_info + '</td>');
    
    // 13. زر الحذف

    
    var delete_btn = $(original_content[cells.length - 1]).find('.pos_remove_row');
    if (delete_btn.length === 0) {
        delete_btn = '<i class="fa fa-times pos_remove_row"></i>';
    } else {
        delete_btn = delete_btn[0].outerHTML;
    }
    row.append('<td class="text-center">' + delete_btn + '</td>');
    
    // إعادة تنشيط select2
    row.find('select.select2').select2({
        width: '100%',
        theme: 'default'
    });
    
    // إضافة معالجات الأحداث
    attachRowEventHandlers(row);
}

// تحسين دالة معالجات الأحداث
function attachRowEventHandlers(row) {
    // معالج تغيير السعر USD
    row.find('.pos_unit_price_usd').on('change', function() {
        var usd_price = __read_number($(this));
        var iqd_price = convertUSDToIQD(usd_price);
        var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
        var local_price = usd_price * exchange_rate;
        
        row.find('.pos_unit_price_iqd').val(__number_f(iqd_price, false, false, 0));
        __write_number(row.find('input.pos_unit_price_inc_tax'), local_price);
        
        pos_each_row(row);
        pos_total_row();
    });
    
    // معالج تغيير السعر IQD
    row.find('.pos_unit_price_iqd').on('change', function() {
        var iqd_price = __read_number($(this));
        var usd_price = convertIQDToUSD(iqd_price);
        var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
        var local_price = usd_price * exchange_rate;
        
        row.find('.pos_unit_price_usd').val(__number_f(usd_price, false, false, 2));
        __write_number(row.find('input.pos_unit_price_inc_tax'), local_price);
        
        pos_each_row(row);
        pos_total_row();
    });
    
    // معالج تغيير نسبة الخصم
    row.find('.discount_percent').on('change', function() {
        var discount_percent = __read_number($(this));
        
        // تحديث نوع الخصم ليكون نسبة مئوية
        row.find('select.row_discount_type').val('percentage');
        __write_number(row.find('input.row_discount_amount'), discount_percent);
        
        row.find('select.row_discount_type').trigger('change');
    });
}

// تحديث الأرقام التسلسلية عند حذف صف
$(document).on('click', '.pos_remove_row', function() {
    var tr = $(this).closest('tr');
    tr.remove();
    
    // إعادة ترقيم الصفوف
    $('#pos_table tbody tr').each(function(index) {
        $(this).find('td.serial_no').text(index + 1);
    });
    
    pos_total_row();
});

//Update values for each row
function pos_each_row(row_obj) {
    var unit_price = __read_number(row_obj.find('input.pos_unit_price'));

    var discounted_unit_price = calculate_discounted_unit_price(row_obj);
    var tax_rate = row_obj
        .find('select.tax_id')
        .find(':selected')
        .data('rate');

    var unit_price_inc_tax =
        discounted_unit_price + __calculate_amount('percentage', tax_rate, discounted_unit_price);
    __write_number(row_obj.find('input.pos_unit_price_inc_tax'), unit_price_inc_tax);

    var discount = __read_number(row_obj.find('input.row_discount_amount'));

    if (discount > 0) {
        var qty = __read_number(row_obj.find('input.pos_quantity'));
        var line_total = qty * unit_price_inc_tax;
        __write_number(row_obj.find('input.pos_line_total'), line_total);
    }

    //var unit_price_inc_tax = __read_number(row_obj.find('input.pos_unit_price_inc_tax'));

    __write_number(row_obj.find('input.item_tax'), unit_price_inc_tax - discounted_unit_price);
}

function pos_total_row() {
    var total_quantity = 0;
    var price_total = get_subtotal();
    $('table#pos_table tbody tr').each(function() {
        total_quantity = total_quantity + __read_number($(this).find('input.pos_quantity'));
    });

    //updating shipping charges
    $('span#shipping_charges_amount').text(
        __currency_trans_from_en(__read_number($('input#shipping_charges_modal')), false)
    );

    $('span.total_quantity').each(function() {
        $(this).html(__number_f(total_quantity));
    });

    //$('span.unit_price_total').html(unit_price_total);
    $('span.price_total').html(__currency_trans_from_en(price_total, false));
    calculate_billing_details(price_total);

    if (
        $('input[name="is_serial_no"]').length > 0 &&
        $('input[name="is_serial_no"]').val() == 1
    ) {
        update_serial_no();
    }
}

function get_subtotal() {
    var price_total = 0;

    $('table#pos_table tbody tr').each(function() {
        price_total = price_total + __read_number($(this).find('input.pos_line_total'));
    });

    //Go through the modifier prices.
    $('input.modifiers_price').each(function() {
        var modifier_price = __read_number($(this));
        var modifier_quantity = $(this).closest('.product_modifier').find('.modifiers_quantity').val();
        var modifier_subtotal = modifier_price * modifier_quantity;
        price_total = price_total + modifier_subtotal;
    });

    return price_total;
}

function calculate_billing_details(price_total) {
    var discount = pos_discount(price_total);
    if ($('#reward_point_enabled').length) {
        total_customer_reward = $('#rp_redeemed_amount').val();
        discount = parseFloat(discount) + parseFloat(total_customer_reward);

        if ($('input[name="is_direct_sale"]').length <= 0) {
            $('span#total_discount').text(__currency_trans_from_en(discount, false));
        }
    }

    var order_tax = pos_order_tax(price_total, discount);

    //Add shipping charges.
    var shipping_charges = __read_number($('input#shipping_charges'));

    var additional_expense = 0;
    //calculate additional expenses
    if ($('input#additional_expense_value_1').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_1'));
    }
    if ($('input#additional_expense_value_2').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_2'))
    }
    if ($('input#additional_expense_value_3').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_3'))
    }
    if ($('input#additional_expense_value_4').length > 0) {
        additional_expense += __read_number($('input#additional_expense_value_4'))
    }

    //Add packaging charge
    var packing_charge = 0;
    if ($('#types_of_service_id').length > 0 && 
            $('#types_of_service_id').val()) {
        packing_charge = __calculate_amount($('#packing_charge_type').val(), 
            __read_number($('input#packing_charge')), price_total);

        $('#packing_charge_text').text(__currency_trans_from_en(packing_charge, false));
    }

    var total_payable = price_total + order_tax - discount + shipping_charges + packing_charge + additional_expense;

    var rounding_multiple = $('#amount_rounding_method').val() ? parseFloat($('#amount_rounding_method').val()) : 0;
    var round_off_data = __round(total_payable, rounding_multiple);
    var total_payable_rounded = round_off_data.number;

    var round_off_amount = round_off_data.diff;
    if (round_off_amount != 0) {
        $('span#round_off_text').text(__currency_trans_from_en(round_off_amount, false));
    } else {
        $('span#round_off_text').text(0);
    }
    $('input#round_off_amount').val(round_off_amount);

    __write_number($('input#final_total_input'), total_payable_rounded);
    var curr_exchange_rate = 1;
    if ($('#exchange_rate').length > 0 && $('#exchange_rate').val()) {
        curr_exchange_rate = __read_number($('#exchange_rate'));
    }
    var shown_total = total_payable_rounded * curr_exchange_rate;
    $('span#total_payable').text(__currency_trans_from_en(shown_total, false));

    $('span.total_payable_span').text(__currency_trans_from_en(total_payable_rounded, true));

    //Check if edit form then don't update price.
    if ($('form#edit_pos_sell_form').length == 0 && $('form#edit_sell_form').length == 0) {
        __write_number($('.payment-amount').first(), total_payable_rounded);
    }

    $(document).trigger('invoice_total_calculated');

    calculate_balance_due();
}

function pos_discount(total_amount) {
    var calculation_type = $('#discount_type').val();
    var calculation_amount = __read_number($('#discount_amount'));

    var discount = __calculate_amount(calculation_type, calculation_amount, total_amount);

    $('span#total_discount').text(__currency_trans_from_en(discount, false));

    return discount;
}

function pos_order_tax(price_total, discount) {
    var tax_rate_id = $('#tax_rate_id').val();
    var calculation_type = 'percentage';
    var calculation_amount = __read_number($('#tax_calculation_amount'));
    var total_amount = price_total - discount;

    if (tax_rate_id) {
        var order_tax = __calculate_amount(calculation_type, calculation_amount, total_amount);
    } else {
        var order_tax = 0;
    }

    $('span#order_tax').text(__currency_trans_from_en(order_tax, false));

    return order_tax;
}

function calculate_balance_due() {
    var total_payable = __read_number($('#final_total_input'));
    var total_paying = 0;
    $('#payment_rows_div')
        .find('.payment-amount')
        .each(function() {
            if (parseFloat($(this).val())) {
                total_paying += __read_number($(this));
            }
        });
    var bal_due = total_payable - total_paying;
    var change_return = 0;

    //change_return
    if (bal_due < 0 || Math.abs(bal_due) < 0.05) {
        __write_number($('input#change_return'), bal_due * -1);
        $('span.change_return_span').text(__currency_trans_from_en(bal_due * -1, true));
        change_return = bal_due * -1;
        bal_due = 0;
    } else {
        __write_number($('input#change_return'), 0);
        $('span.change_return_span').text(__currency_trans_from_en(0, true));
        change_return = 0;
        
    }

    if (change_return !== 0) {
        $('#change_return_payment_data').removeClass('hide');
    } else {
        $('#change_return_payment_data').addClass('hide');
    }

    __write_number($('input#total_paying_input'), total_paying);
    $('span.total_paying').text(__currency_trans_from_en(total_paying, true));

    __write_number($('input#in_balance_due'), bal_due);
    $('span.balance_due').text(__currency_trans_from_en(bal_due, true));

    __highlight(bal_due * -1, $('span.balance_due'));
    __highlight(change_return * -1, $('span.change_return_span'));
}

// Centralized form validation
var FormValidator = {
    validatePosForm: function() {
        var flag = true;
        $('span.error').remove();

        if ($('select#customer_id').val() == null) {
            flag = false;
            var error = '<span class="error">' + LANG.required + '</span>';
            $(error).insertAfter($('select#customer_id').parent('div'));
        }

        if ($('tr.product_row').length == 0) {
            flag = false;
            var error = '<span class="error">' + LANG.no_products + '</span>';
            $(error).insertAfter($('input#search_product').parent('div'));
        }

        return flag;
    },
    
    validateMinimumSellingPrice: function() {
        var is_msp_valid = true;
        $('.pos_unit_price_inc_tax').each(function() {
            if (!$(this).is(":visible") && $(this).data('rule-min-value')) {
                var val = __read_number($(this));
                var error_msg_td = $(this).closest('tr').find('.pos_line_total_text').closest('td');
                if (val > $(this).data('rule-min-value')) {
                    is_msp_valid = false;
                    error_msg_td.append('<label class="error">' + $(this).data('msg-min-value') + '</label>');
                } else {
                    error_msg_td.find('label.error').remove();
                }
            }
        });
        return is_msp_valid;
    }
};

// Backward compatibility
function isValidPosForm() {
    return FormValidator.validatePosForm();
}

function reset_pos_form(){

	//If on edit page then redirect to Add POS page
	if($('form#edit_pos_sell_form').length > 0){
		setTimeout(function() {
			window.location = $("input#pos_redirect_url").val();
		}, 4000);
		return true;
	}
	
    //reset all repair defects tags
    if ($("#repair_defects").length > 0) {
        tagify_repair_defects.removeAllTags();
    }

	if(pos_form_obj[0]){
		pos_form_obj[0].reset();
	}
	if(sell_form[0]){
		sell_form[0].reset();
	}
	set_default_customer();
	set_location();

	$('tr.product_row').remove();
	$('span.total_quantity, span.price_total, span#total_discount, span#order_tax, span#total_payable, span#shipping_charges_amount').text(0);
	$('span.total_payable_span', 'span.total_paying', 'span.balance_due').text(0);

	$('#modal_payment').find('.remove_payment_row').each( function(){
		$(this).closest('.payment_row').remove();
	});

    if ($('#is_credit_sale').length) {
        $('#is_credit_sale').val(0);
    }

	//Reset discount
	__write_number($('input#discount_amount'), $('input#discount_amount').data('default'));
	$('input#discount_type').val($('input#discount_type').data('default'));

	//Reset tax rate
	$('input#tax_rate_id').val($('input#tax_rate_id').data('default'));
	__write_number($('input#tax_calculation_amount'), $('input#tax_calculation_amount').data('default'));

	$('select.payment_types_dropdown').val('cash').trigger('change');
	$('#price_group').trigger('change');

	//Reset shipping
	__write_number($('input#shipping_charges'), $('input#shipping_charges').data('default'));
	$('input#shipping_details').val($('input#shipping_details').data('default'));
    $('input#shipping_address, input#shipping_status, input#delivered_to').val('');
	if($('input#is_recurring').length > 0){
		$('input#is_recurring').iCheck('update');
	};
    if($('input#is_kitchen_order').length > 0){
		$('input#is_kitchen_order').iCheck('update');
	};
    if($('#invoice_layout_id').length > 0){
        $('#invoice_layout_id').trigger('change');
    };
    $('span#round_off_text').text(0);

    //repair module extra  fields reset
    if ($('#repair_device_id').length > 0) {
        $('#repair_device_id').val('').trigger('change');
    }

    //Status is hidden in sales order
    if ($('#status').length > 0 && $('#status').is(":visible")) {
        $('#status').val('').trigger('change');
    }
    if ($('#transaction_date').length > 0) {
        $('#transaction_date').data("DateTimePicker").date(moment());
    }
    if ($('.paid_on').length > 0) {
        $('.paid_on').data("DateTimePicker").date(moment());
    }
    if ($('#commission_agent').length > 0) {
        $('#commission_agent').val('').trigger('change');
    } 

    //reset contact due
    $('.contact_due_text').find('span').text('');
    $('.contact_due_text').addClass('hide');

    $(document).trigger('sell_form_reset');
}

function set_default_customer() {
    var default_customer_id = $('#default_customer_id').val();
    var default_customer_name = $('#default_customer_name').val();
    var default_customer_balance = $('#default_customer_balance').val();
    var default_customer_address = $('#default_customer_address').val();
    var exists = default_customer_id ? $('select#customer_id option[value=' + default_customer_id + ']').length : 0;
    if (exists == 0 && default_customer_id) {
        $('select#customer_id').append(
            $('<option>', { value: default_customer_id, text: default_customer_name })
        );
    }
    $('#advance_balance_text').text(__currency_trans_from_en(default_customer_balance), true);
    $('#advance_balance').val(default_customer_balance);
    $('#shipping_address_modal').val(default_customer_address);
    if (default_customer_address) {
        $('#shipping_address').val(default_customer_address);
    }
    $('select#customer_id')
        .val(default_customer_id)
        .trigger('change');

    if ($('#default_selling_price_group').length) {
        $('#price_group').val($('#default_selling_price_group').val());
        $('#price_group').change();
    }

    //initialize tags input (tagify)
    if ($("textarea#repair_defects").length > 0 && !customer_set) {
        let suggestions = [];
        if ($("input#pos_repair_defects_suggestion").length > 0 && $("input#pos_repair_defects_suggestion").val().length > 2) {
            suggestions = JSON.parse($("input#pos_repair_defects_suggestion").val());    
        }
        let repair_defects = document.querySelector('textarea#repair_defects');
        tagify_repair_defects = new Tagify(repair_defects, {
                  whitelist: suggestions,
                  maxTags: 100,
                  dropdown: {
                    maxItems: 100,           // <- mixumum allowed rendered suggestions
                    classname: "tags-look", // <- custom classname for this dropdown, so it could be targeted
                    enabled: 0,             // <- show suggestions on focus
                    closeOnSelect: false    // <- do not hide the suggestions dropdown once an item has been selected
                  }
                });
    }

    customer_set = true;
}

//Set the location and initialize printer
function set_location() {
    if ($('select#select_location_id').length == 1) {
        $('input#location_id').val($('select#select_location_id').val());
        $('input#location_id').data(
            'receipt_printer_type',
            $('select#select_location_id')
                .find(':selected')
                .data('receipt_printer_type')
        );
        $('input#location_id').data(
            'default_payment_accounts',
            $('select#select_location_id')
                .find(':selected')
                .data('default_payment_accounts')
        );

        $('input#location_id').attr(
            'data-default_price_group',
            $('select#select_location_id')
                .find(':selected')
                .data('default_price_group')
        );
    }

    if ($('input#location_id').val()) {
        $('input#search_product')
            .prop('disabled', false)
            .focus();
    } else {
        $('input#search_product').prop('disabled', true);
    }

    initialize_printer();
}

function initialize_printer() {
    if ($('input#location_id').data('receipt_printer_type') == 'printer') {
        initializeSocket();
    }
}

$('body').on('click', 'label', function(e) {
    var field_id = $(this).attr('for');
    if (field_id) {
        if ($('#' + field_id).hasClass('select2')) {
            $('#' + field_id).select2('open');
            return false;
        }
    }
});

$('body').on('focus', 'select', function(e) {
    var field_id = $(this).attr('id');
    if (field_id) {
        if ($('#' + field_id).hasClass('select2')) {
            $('#' + field_id).select2('open');
            return false;
        }
    }
});

function round_row_to_iraqi_dinnar(row) {
    if (iraqi_selling_price_adjustment) {
        var element = row.find('input.pos_unit_price_inc_tax');
        var unit_price = round_to_iraqi_dinnar(__read_number(element));
        __write_number(element, unit_price);
        element.change();
    }
}

function pos_print(receipt) {
    //If printer type then connect with websocket
    if (receipt.print_type == 'printer') {
        var content = receipt;
        content.type = 'print-receipt';

        //Check if ready or not, then print.
        if (socket != null && socket.readyState == 1) {
            socket.send(JSON.stringify(content));
        } else {
            initializeSocket();
            setTimeout(function() {
                socket.send(JSON.stringify(content));
            }, 700);
        }

    } else if (receipt.html_content != '') {
        var title = document.title;
        if (typeof receipt.print_title != 'undefined') {
            document.title = receipt.print_title;
        }

        //If printer type browser then print content
        $('#receipt_section').html(receipt.html_content);
        __currency_convert_recursively($('#receipt_section'));
        __print_receipt('receipt_section');

        setTimeout(function() {
            document.title = title;
        }, 1200);
    }
}
function calculate_discounted_unit_price(row) {
    var unit_price = __read_number(row.find('input.pos_unit_price'));
    
    // إذا كانت قيمة unit_price صفر أو غير موجودة، حاول الحصول عليها من unit_price_inc_tax
    if (!unit_price || unit_price <= 0) {
        var unit_price_inc_tax = __read_number(row.find('input.pos_unit_price_inc_tax'));
        var tax_rate = row.find('select.tax_id').find(':selected').data('rate') || 0;
        
        if (unit_price_inc_tax && unit_price_inc_tax > 0) {
            // حساب unit_price من unit_price_inc_tax عن طريق إزالة الضريبة
            unit_price = tax_rate > 0 ? 
                __get_principle(unit_price_inc_tax, tax_rate) : 
                unit_price_inc_tax;
            
            // تحديث حقل unit_price
            __write_number(row.find('input.pos_unit_price'), unit_price);
        }
    }
    
    var row_discount_type = row.find('select.row_discount_type').val() || 'percentage';
    var row_discount_amount = __read_number(row.find('input.row_discount_amount')) || 0;
    var discounted_unit_price = unit_price;
    
    if (row_discount_amount) {
        if (row_discount_type == 'fixed') {
            discounted_unit_price = unit_price - row_discount_amount;
        } else {
            discounted_unit_price = __substract_percent(unit_price, row_discount_amount);
        }
    }
    
    // التأكد من عدم إرجاع قيم سالبة
    return Math.max(0, discounted_unit_price);
}

function get_unit_price_from_discounted_unit_price(row, discounted_unit_price) {
    var this_unit_price = discounted_unit_price;
    var row_discount_type = row.find('select.row_discount_type').val();
    var row_discount_amount = __read_number(row.find('input.row_discount_amount'));
    if (row_discount_amount) {
        if (row_discount_type == 'fixed') {
            this_unit_price = discounted_unit_price + row_discount_amount;
        } else {
            this_unit_price = __get_principle(discounted_unit_price, row_discount_amount, true);
        }
    }

    return this_unit_price;
}

//Update quantity if line subtotal changes
$('table#pos_table tbody').on('change', 'input.pos_line_total', function() {

    var subtotal = __read_number($(this));
    var tr = $(this).parents('tr');
    var quantity_element = tr.find('input.pos_quantity');
    var unit_price_inc_tax = __read_number(tr.find('input.pos_unit_price_inc_tax'));
    var quantity = subtotal / unit_price_inc_tax;
    __write_number(quantity_element, quantity);

    __write_number($(this), subtotal, false);


    if (sell_form_validator) {
        sell_form_validator.element(quantity_element);
    }
    if (pos_form_validator) {
        pos_form_validator.element(quantity_element);
    }
    tr.find('span.pos_line_total_text').text(__currency_trans_from_en(subtotal, true));

    pos_total_row();
});

$('div#product_list_body').on('scroll', function() {


    if ($(this).scrollTop() + $(this).innerHeight() >= $(this)[0].scrollHeight) {
        var page = parseInt($('#suggestion_page').val());
        page += 1;
        $('#suggestion_page').val(page);
        var location_id = $('input#location_id').val();
        var category_id = global_p_category_id;
        var brand_id = global_brand_id;

        var is_enabled_stock = null;
        if ($("#is_enabled_stock").length) {
            is_enabled_stock = $("#is_enabled_stock").val();
        }

        var device_model_id = null;
        if ($("#repair_model_id").length) {
            device_model_id = $("#repair_model_id").val();
        }

        get_product_suggestion_list(category_id, brand_id, location_id, null, is_enabled_stock, device_model_id);
    }
});

$(document).on('ifChecked', '#is_recurring', function() {
    $('#recurringInvoiceModal').modal('show');
});

$(document).on('shown.bs.modal', '#recurringInvoiceModal', function() {
    $('input#recur_interval').focus();
});

$(document).on('click', '#select_all_service_staff', function() {
    var val = $('#res_waiter_id').val();
    $('#pos_table tbody')
        .find('select.order_line_service_staff')
        .each(function() {
            $(this)
                .val(val)
                .change();
        });
});

$(document).on('click', '.print-invoice-link', function(e) {
    e.preventDefault();
    $.ajax({
        url: $(this).attr('href') + "?check_location=true",
        dataType: 'json',
        success: function(result) {
            if (result.success == 1) {
                //Check if enabled or not
                if (result.receipt.is_enabled) {
                    pos_print(result.receipt);
                }
            } else {
                // Show detailed error message
                if (result.sap_error) {
                    toastr.error(result.msg + '<br><small>' + result.sap_error + '</small>', 'خطأ SAP', {
                        timeOut: 10000,
                        closeButton: true,
                        progressBar: true,
                        escapeHtml: false
                    });
                } else {
                    toastr.error(result.msg);
                }
            }
        },
        error: function(xhr) {
            toastr.error('حدث خطأ في الاتصال');
        }
    });
});

function getCustomerRewardPoints() {
    if ($('#reward_point_enabled').length <= 0) {
        return false;
    }
    var is_edit = $('form#edit_sell_form').length || 
    $('form#edit_pos_sell_form').length ? true : false;
    if (is_edit && !customer_set) {
        return false;
    }

    var customer_id = $('#customer_id').val();

    $.ajax({
        method: 'POST',
        url: '/sells/pos/get-reward-details',
        data: { 
            customer_id: customer_id
        },
        dataType: 'json',
        success: function(result) {
            $('#available_rp').text(result.points);
            $('#rp_redeemed_modal').data('max_points', result.points);
            updateRedeemedAmount();
            $('#rp_redeemed_amount').change()
        },
    });
}

function updateRedeemedAmount(argument) {
    var points = $('#rp_redeemed_modal').val().trim();
    points = points == '' ? 0 : parseInt(points);
    var amount_per_unit_point = parseFloat($('#rp_redeemed_modal').data('amount_per_unit_point'));
    var redeemed_amount = points * amount_per_unit_point;
    $('#rp_redeemed_amount_text').text(__currency_trans_from_en(redeemed_amount, true));
    $('#rp_redeemed').val(points);
    $('#rp_redeemed_amount').val(redeemed_amount);
}

$(document).on('change', 'select#customer_id', function(){
    var default_customer_id = $('#default_customer_id').val();
    if ($(this).val() == default_customer_id) {
        //Disable reward points for walkin customers
        if ($('#rp_redeemed_modal').length) {
            $('#rp_redeemed_modal').val('');
            $('#rp_redeemed_modal').change();
            $('#rp_redeemed_modal').attr('disabled', true);
            $('#available_rp').text('');
            updateRedeemedAmount();
            pos_total_row();
        }
    } else {
        if ($('#rp_redeemed_modal').length) {
            $('#rp_redeemed_modal').removeAttr('disabled');
        }
        getCustomerRewardPoints();
    }

    get_sales_orders();
});

$(document).on('change', '#rp_redeemed_modal', function(){
    var points = $(this).val().trim();
    points = points == '' ? 0 : parseInt(points);
    var amount_per_unit_point = parseFloat($(this).data('amount_per_unit_point'));
    var redeemed_amount = points * amount_per_unit_point;
    $('#rp_redeemed_amount_text').text(__currency_trans_from_en(redeemed_amount, true));
    var reward_validation = isValidatRewardPoint();
    if (!reward_validation['is_valid']) {
        toastr.error(reward_validation['msg']);
        $('#rp_redeemed_modal').select();
    }
});

$(document).on('change', '.direct_sell_rp_input', function(){
    updateRedeemedAmount();
    pos_total_row();
});

function isValidatRewardPoint() {
    var element = $('#rp_redeemed_modal');
    var points = element.val().trim();
    points = points == '' ? 0 : parseInt(points);

    var max_points = parseInt(element.data('max_points'));
    var is_valid = true;
    var msg = '';

    if (points == 0) {
        return {
            is_valid: is_valid,
            msg: msg
        }
    }

    var rp_name = $('input#rp_name').val();
    if (points > max_points) {
        is_valid = false;
        msg = __translate('max_rp_reached_error', {max_points: max_points, rp_name: rp_name});
    }

    var min_order_total_required = parseFloat(element.data('min_order_total'));

    var order_total = __read_number($('#final_total_input'));

    if (order_total < min_order_total_required) {
        is_valid = false;
        msg = __translate('min_order_total_error', {min_order: __currency_trans_from_en(min_order_total_required, true), rp_name: rp_name});
    }

    var output = {
        is_valid: is_valid,
        msg: msg,
    }

    return output;
}

function adjustComboQty(tr){
    if(tr.find('input.product_type').val() == 'combo'){
        var qty = __read_number(tr.find('input.pos_quantity'));
        var multiplier = __getUnitMultiplier(tr);

        tr.find('input.combo_product_qty').each(function(){
            $(this).val($(this).data('unit_quantity') * qty * multiplier);
        });
    }
}

$(document).on('change', '#types_of_service_id', function(){
    var types_of_service_id = $(this).val();
    var location_id = $('#location_id').val();

    if(types_of_service_id) {
        $.ajax({
            method: 'POST',
            url: '/sells/pos/get-types-of-service-details',
            data: { 
                types_of_service_id: types_of_service_id,
                location_id: location_id
            },
            dataType: 'json',
            success: function(result) {
                //reset form if price group is changed
                var prev_price_group = $('#types_of_service_price_group').val();
                if(result.price_group_id) {
                    $('#types_of_service_price_group').val(result.price_group_id);
                    $('#price_group_text').removeClass('hide');
                    $('#price_group_text span').text(result.price_group_name);
                } else {
                    $('#types_of_service_price_group').val('');
                    $('#price_group_text').addClass('hide');
                    $('#price_group_text span').text('');
                }
                $('#types_of_service_id').val(types_of_service_id);
                $('.types_of_service_modal').html(result.modal_html);
                
                if (prev_price_group != result.price_group_id) {
                    if ($('form#edit_pos_sell_form').length > 0) {
                        $('table#pos_table tbody').html('');
                        pos_total_row();
                    } else {
                        reset_pos_form();
                    }
                } else {
                    pos_total_row();
                }

                $('.types_of_service_modal').modal('show');
            },
        });
    } else {
        $('.types_of_service_modal').html('');
        $('#types_of_service_price_group').val('');
        $('#price_group_text').addClass('hide');
        $('#price_group_text span').text('');
        $('#packing_charge_text').text('');
        if ($('form#edit_pos_sell_form').length > 0) {
            $('table#pos_table tbody').html('');
            pos_total_row();
        } else {
            reset_pos_form();
        }
    }
});

$(document).on('change', 'input#packing_charge, #additional_expense_value_1, #additional_expense_value_2, \
        #additional_expense_value_3, #additional_expense_value_4', function() {
    pos_total_row();
});

$(document).on('click', '.service_modal_btn', function(e) {
    if ($('#types_of_service_id').val()) {
        $('.types_of_service_modal').modal('show');
    }
});

$(document).on('change', '.payment_types_dropdown', function(e) {
    var default_accounts = $('select#select_location_id').length ? 
                $('select#select_location_id')
                .find(':selected')
                .data('default_payment_accounts') : $('#location_id').data('default_payment_accounts');
    var payment_type = $(this).val();
    var payment_row = $(this).closest('.payment_row');
    if (payment_type && payment_type != 'advance') {
        var default_account = default_accounts && default_accounts[payment_type]['account'] ? 
            default_accounts[payment_type]['account'] : '';
        var row_index = payment_row.find('.payment_row_index').val();

        var account_dropdown = payment_row.find('select#account_' + row_index);
        if (account_dropdown.length && default_accounts) {
            account_dropdown.val(default_account);
            account_dropdown.change();
        }
    }

    //Validate max amount and disable account if advance 
    amount_element = payment_row.find('.payment-amount');
    account_dropdown = payment_row.find('.account-dropdown');
    if (payment_type == 'advance') {
        max_value = $('#advance_balance').val();
        msg = $('#advance_balance').data('error-msg');
        amount_element.rules('add', {
            'max-value': max_value,
            messages: {
                'max-value': msg,
            },
        });
        if (account_dropdown) {
            account_dropdown.prop('disabled', true);
            account_dropdown.closest('.form-group').addClass('hide');
        }
    } else {
        amount_element.rules("remove", "max-value");
        if (account_dropdown) {
            account_dropdown.prop('disabled', false); 
            account_dropdown.closest('.form-group').removeClass('hide');
        }    
    }
});

$(document).on('show.bs.modal', '#recent_transactions_modal', function () {
    get_recent_transactions('final', $('div#tab_final'));
});
$(document).on('shown.bs.tab', 'a[href="#tab_quotation"]', function () {
    get_recent_transactions('quotation', $('div#tab_quotation'));
});
$(document).on('shown.bs.tab', 'a[href="#tab_draft"]', function () {
    get_recent_transactions('draft', $('div#tab_draft'));
});

function disable_pos_form_actions(){
    if (!window.navigator.onLine) {
        return false;
    }

    $('div.pos-processing').show();
    $('#pos-save').attr('disabled', 'true');
    $('div.pos-form-actions').find('button').attr('disabled', 'true');
}

function enable_pos_form_actions(){
    $('div.pos-processing').hide();
    $('#pos-save').removeAttr('disabled');
    $('div.pos-form-actions').find('button').removeAttr('disabled');
}

$(document).on('change', '#recur_interval_type', function() {
    if ($(this).val() == 'months') {
        $('.subscription_repeat_on_div').removeClass('hide');
    } else {
        $('.subscription_repeat_on_div').addClass('hide');
    }
});

function validate_discount_field() {
    discount_element = $('#discount_amount_modal');
    discount_type_element = $('#discount_type_modal');

    if ($('#add_sell_form').length || $('#edit_sell_form').length) {
        discount_element = $('#discount_amount');
        discount_type_element = $('#discount_type');
    }
    var max_value = parseFloat(discount_element.data('max-discount'));
    if (discount_element.val() != '' && !isNaN(max_value)) {
        if (discount_type_element.val() == 'fixed') {
            var subtotal = get_subtotal();
            //get max discount amount
            max_value = __calculate_amount('percentage', max_value, subtotal)
        }

        discount_element.rules('add', {
            'max-value': max_value,
            messages: {
                'max-value': discount_element.data('max-discount-error_msg'),
            },
        });
    } else {
        discount_element.rules("remove", "max-value");      
    }
    discount_element.trigger('change');
}

$(document).on('change', '#discount_type_modal, #discount_type', function() {
    validate_discount_field();
});

function update_shipping_address(data) {
    if ($('#shipping_address_div').length) {
        var shipping_address = '';
        if (data.supplier_business_name) {
            shipping_address += data.supplier_business_name;
        }
        if (data.name) {
            shipping_address += ',<br>' + data.name;
        }
        if (data.text) {
            shipping_address += ',<br>' + data.text;
        }
        shipping_address += ',<br>' + data.shipping_address ;
        $('#shipping_address_div').html(shipping_address);
    }
    if ($('#billing_address_div').length) {
        var address = [];
        if (data.supplier_business_name) {
            address.push(data.supplier_business_name);
        }
        if (data.name) {
            address.push('<br>' + data.name);
        }
        if (data.text) {
            address.push('<br>' + data.text);
        }
        if (data.address_line_1) {
            address.push('<br>' + data.address_line_1);
        }
        if (data.address_line_2) {
            address.push('<br>' + data.address_line_2);
        }
        if (data.city) {
            address.push('<br>' + data.city);
        }
        if (data.state) {
            address.push(data.state);
        }
        if (data.country) {
            address.push(data.country);
        }
        if (data.zip_code) {
            address.push('<br>' + data.zip_code);
        }
        var billing_address = address.join(', ');
        $('#billing_address_div').html(billing_address);
    }

    if ($('#shipping_custom_field_1').length) {
        let shipping_custom_field_1 = data.shipping_custom_field_details != null ? data.shipping_custom_field_details.shipping_custom_field_1 : '';
        $('#shipping_custom_field_1').val(shipping_custom_field_1);
    }

    if ($('#shipping_custom_field_2').length) {
        let shipping_custom_field_2 = data.shipping_custom_field_details != null ? data.shipping_custom_field_details.shipping_custom_field_2 : '';
        $('#shipping_custom_field_2').val(shipping_custom_field_2);
    }

    if ($('#shipping_custom_field_3').length) {
        let shipping_custom_field_3 = data.shipping_custom_field_details != null ? data.shipping_custom_field_details.shipping_custom_field_3 : '';
        $('#shipping_custom_field_3').val(shipping_custom_field_3);
    }

    if ($('#shipping_custom_field_4').length) {
        let shipping_custom_field_4 = data.shipping_custom_field_details != null ? data.shipping_custom_field_details.shipping_custom_field_4 : '';
        $('#shipping_custom_field_4').val(shipping_custom_field_4);
    }

    if ($('#shipping_custom_field_5').length) {
        let shipping_custom_field_5 = data.shipping_custom_field_details != null ? data.shipping_custom_field_details.shipping_custom_field_5 : '';
        $('#shipping_custom_field_5').val(shipping_custom_field_5);
    }
    
    //update export fields
    if (data.is_export) {
        $('#is_export').prop('checked', true);
        $('div.export_div').show();
        if ($('#export_custom_field_1').length) {
            $('#export_custom_field_1').val(data.export_custom_field_1);
        }
        if ($('#export_custom_field_2').length) {
            $('#export_custom_field_2').val(data.export_custom_field_2);
        }
        if ($('#export_custom_field_3').length) {
            $('#export_custom_field_3').val(data.export_custom_field_3);
        }
        if ($('#export_custom_field_4').length) {
            $('#export_custom_field_4').val(data.export_custom_field_4);
        }
        if ($('#export_custom_field_5').length) {
            $('#export_custom_field_5').val(data.export_custom_field_5);
        }
        if ($('#export_custom_field_6').length) {
            $('#export_custom_field_6').val(data.export_custom_field_6);
        }
    } else {
        $('#export_custom_field_1, #export_custom_field_2, #export_custom_field_3, #export_custom_field_4, #export_custom_field_5, #export_custom_field_6').val('');
        $('#is_export').prop('checked', false);
        $('div.export_div').hide();
    }
    
    $('#shipping_address_modal').val(data.shipping_address);
    $('#shipping_address').val(data.shipping_address);
}

function get_sales_orders() {
    if ($('#sales_order_ids').length) {
        if ($('#sales_order_ids').hasClass('not_loaded')) {
            $('#sales_order_ids').removeClass('not_loaded');
            return false;
        }
        var customer_id = $('select#customer_id').val();
        var location_id = $('input#location_id').val();
        $.ajax({
            url: '/get-sales-orders/' + customer_id + '?location_id=' + location_id,
            dataType: 'json',
            success: function(data) {
                $('#sales_order_ids').select2('destroy').empty().select2({data: data});
                $('table#pos_table tbody').find('tr').each( function(){
                    if (typeof($(this).data('so_id')) !== 'undefined') {
                        $(this).remove();
                    }
                });
                pos_total_row();
            },
        });
    }
}

$("#sales_order_ids").on("select2:select", function (e) {
    var sales_order_id = e.params.data.id;
    var product_row = $('input#product_row_count').val();
    var location_id = $('input#location_id').val();
    $.ajax({
        method: 'GET',
        url: '/get-sales-order-lines',
        async: false,
        data: {
            product_row: product_row,
            sales_order_id: sales_order_id
        },
        dataType: 'json',
        success: function(result) {
            if (result.html) {
                var html = result.html;
                $(html).find('tr').each(function(){
                    $('table#pos_table tbody')
                    .append($(this))
                    .find('input.pos_quantity');
                    
                    var this_row = $('table#pos_table tbody')
                        .find('tr')
                        .last();
                    pos_each_row(this_row);

                    product_row = parseInt(product_row) + 1;

                    //For initial discount if present
                    var line_total = __read_number(this_row.find('input.pos_line_total'));
                    this_row.find('span.pos_line_total_text').text(line_total);

                    //Check if multipler is present then multiply it when a new row is added.
                    if(__getUnitMultiplier(this_row) > 1){
                        this_row.find('select.sub_unit').trigger('change');
                    }

                    round_row_to_iraqi_dinnar(this_row);
                    __currency_convert_recursively(this_row);
                });

                set_so_values(result.sales_order);

                //increment row count
                $('input#product_row_count').val(product_row);
                
                pos_total_row();
            
            } else {
                toastr.error(result.msg);
                $('input#search_product')
                    .focus()
                    .select();
            }
        },
    });
});

function set_so_values(so) {
    $('textarea[name="sale_note"]').val(so.additional_notes);
    if ($('#shipping_details').is(':visible')) {
        $('#shipping_details').val(so.shipping_details);
    }
    $('#shipping_address').val(so.shipping_address);
    $('#delivered_to').val(so.delivered_to);
    $('#shipping_charges').val( __number_f(so.shipping_charges));
    $('#shipping_status').val(so.shipping_status);
    if ($('#shipping_custom_field_1').length) {
        $('#shipping_custom_field_1').val(so.shipping_custom_field_1);
    }
    if ($('#shipping_custom_field_2').length) {
        $('#shipping_custom_field_2').val(so.shipping_custom_field_2);
    }
    if ($('#shipping_custom_field_3').length) {
        $('#shipping_custom_field_3').val(so.shipping_custom_field_3);
    }
    if ($('#shipping_custom_field_4').length) {
        $('#shipping_custom_field_4').val(so.shipping_custom_field_4);
    }
    if ($('#shipping_custom_field_5').length) {
        $('#shipping_custom_field_5').val(so.shipping_custom_field_5);
    }
}

$("#sales_order_ids").on("select2:unselect", function (e) {
    var sales_order_id = e.params.data.id;
    $('table#pos_table tbody').find('tr').each( function(){
        if (typeof($(this).data('so_id')) !== 'undefined' 
            && $(this).data('so_id') == sales_order_id) {
            $(this).remove();
        pos_total_row();
        }
    });
});

$(document).on('click', '#add_expense', function(){
    $.ajax({
        url: '/expenses/create',
        data: { 
            location_id: $('#select_location_id').val()
        },
        dataType: 'html',
        success: function(result) {
            $('#expense_modal').html(result);
            $('#expense_modal').modal('show');
        },
    });
});

$(document).on('shown.bs.modal', '#expense_modal', function(){
    $('#expense_transaction_date').datetimepicker({
        format: moment_date_format + ' ' + moment_time_format,
        ignoreReadonly: true,
    });
    $('#expense_modal .paid_on').datetimepicker({
        format: moment_date_format + ' ' + moment_time_format,
        ignoreReadonly: true,
    });
    $(this).find('.select2').select2();
    $('#add_expense_modal_form').validate();
});

$(document).on('hidden.bs.modal', '#expense_modal', function(){
    $(this).html('');
});

$(document).on('submit', 'form#add_expense_modal_form', function(e) {
    e.preventDefault();
    var data = $(this).serialize();

    $.ajax({
        method: 'POST',
        url: $(this).attr('action'),
        dataType: 'json',
        data: data,
        success: function(result) {
            if (result.success == true) {
                $('#expense_modal').modal('hide');
                toastr.success(result.msg);
            } else {
                toastr.error(result.msg);
            }
        },
    });
});

function get_contact_due(id) {
    $.ajax({
        method: 'get',
        url: /get-contact-due/ + id,
        dataType: 'text',
        success: function(result) {
            if (result != '') {
                $('.contact_due_text').find('span').text(result);
                $('.contact_due_text').removeClass('hide');
            } else {
                $('.contact_due_text').find('span').text('');
                $('.contact_due_text').addClass('hide');
            }
        },
    });
}

function submitQuickContactForm(form) {
    var data = $(form).serialize();
    $.ajax({
        method: 'POST',
        url: $(form).attr('action'),
        dataType: 'json',
        data: data,
        beforeSend: function(xhr) {
            __disable_submit_button($(form).find('button[type="submit"]'));
        },
        success: function(result) {
            if (result.success == true) {
                var name = result.data.name;

                if (result.data.supplier_business_name) {
                    name += result.data.supplier_business_name;
                }
                
                $('select#customer_id').append(
                    $('<option>', { value: result.data.id, text: name })
                );
                $('select#customer_id')
                    .val(result.data.id)
                    .trigger('change');
                $('div.contact_modal').modal('hide');
                update_shipping_address(result.data)
                toastr.success(result.msg);
            } else {
                toastr.error(result.msg);
            }
        },
    });
}

$(document).on('click', '#send_for_sell_return', function(e) {
    var invoice_no = $('#send_for_sell_return_invoice_no').val();

    if (invoice_no) {
        $.ajax({
            method: 'get',
            url: /validate-invoice-to-return/ + encodeURI(invoice_no),
            dataType: 'json',
            success: function(result) {
                if (result.success == true) {
                    window.location = result.redirect_url ;
                } else {
                    toastr.error(result.msg);
                }
            },
        });
    }
})

    $(document).on('click', '#send_for_sercice_staff_replacement', function (e) {
        var invoice_no = $('#send_for_sell_service_staff_invoice_no').val();

        if (invoice_no) {
            $.ajax({
                method: 'get',
                url: /validate-invoice-to-service-staff-replacement/ + encodeURI(invoice_no),
                dataType: 'json',
                success: function (result) {
                    if (result.success == true) {
                        $('#service_staff_replacement').popover('hide');
                        $('#service_staff_modal').html(result.msg);
                        $('#service_staff_modal').modal('show');
                       
                    } else {
                        toastr.error(result.msg);
                    }
                },
            });
        }
    });

    $(document).on('shown.bs.modal', '#service_staff_modal', function () {
        $('#change_service_staff').validate();
    });


    $(document).on('submit', 'form#change_service_staff', function (e) {
        e.preventDefault();
        var data = $(this).serialize();

        $.ajax({
            method: 'POST',
            url: $(this).attr('action'),
            dataType: 'json',
            data: data,
            success: function (result) {
                if (result.success == true) {
                    $('#service_staff_modal').modal('hide');
                    toastr.success(result.msg);
                } else {
                    toastr.error(result.msg);
                }
            },
        });
    });

$(document).on('ifChanged', 'input[name="search_fields[]"]', function(event) {
    var search_fields = [];
    $('input[name="search_fields[]"]:checked').each(function() {
       search_fields.push($(this).val());
    });

    localStorage.setItem('pos_search_fields', search_fields);
});

function set_search_fields() {
    if ($('input[name="search_fields[]"]').length == 0) {
        return false;
    }

    var pos_search_fields = localStorage.getItem('pos_search_fields');

    if (pos_search_fields === null) {
        pos_search_fields = ['name', 'sku', 'lot'];
    }

    $('input[name="search_fields[]"]').each(function() {
        if (pos_search_fields.indexOf($(this).val()) >= 0) {
            $(this).iCheck('check');
        } else {
            $(this).iCheck('uncheck');
        }
    });
}

$(document).on('click', '#show_service_staff_availability', function(){
    loadServiceStaffAvailability();
})
$(document).on('click', '#refresh_service_staff_availability_status', function(){
    loadServiceStaffAvailability(false);
})
$(document).on('click', 'button.pause_resume_timer', function(e){
    $('.view_modal').find('.overlay').removeClass('hide');
    $.ajax({
        method: 'get',
        url: $(this).attr('data-href'),
        dataType: 'json',
        success: function(result) {
            loadServiceStaffAvailability(false);
        },
    });
})

$(document).on('click', '.mark_as_available', function(e){
    e.preventDefault()
    $('.view_modal').find('.overlay').removeClass('hide');
    $.ajax({
        method: 'get',
        url: $(this).attr('href'),
        dataType: 'json',
        success: function(result) {
            loadServiceStaffAvailability(false);
        },
    });
})
var service_staff_availability_interval = null;

function loadServiceStaffAvailability(show = true) {
    var location_id = $('[name="location_id"]').val();
    $.ajax({
        method: 'get',
        url: $('#show_service_staff_availability').attr('data-href'),
        dataType: 'html',
        data: {location_id: location_id},
        success: function(result) {
            $('.view_modal').html(result);
            if (show) {
                $('.view_modal').modal('show')

                //auto refresh service staff availabilty if modal is open
                service_staff_availability_interval = setInterval(function () {
                    loadServiceStaffAvailability(false);
                }, 60000);
            }
        },
    });
}

$(document).on('hidden.bs.modal', '.view_modal', function(){
    if (service_staff_availability_interval !== null) {
        clearInterval(service_staff_availability_interval);
    }
    service_staff_availability_interval = null;
});


$(document).on('change', '#res_waiter_id', function(e){
    var is_enable = $(this).find('option:selected').data('is_enable');

    if(is_enable){
        swal({
            text: LANG.enter_pin_here,
            buttons: true,
            dangerMode: true,
            content: {
                element: "input",
                attributes: {
                    placeholder: LANG.enter_pin_here,
                    type: "password",
                },
            },
        })
        .then((inputValue) => {
            if (inputValue !== null) {
                    $.ajax({
                        method: 'get',
                        url: '/modules/data/check-staff-pin',
                        dataType: 'json',
                        data: {
                        service_staff_pin: inputValue,
                        user_id : $("#res_waiter_id").val(),
                        },
                        success: (result) => {

                            if (result == false) {
                                toastr.error(LANG.authentication_failed);
                                $("#res_waiter_id").val('');
                            } else {
                                // AJAX request succeeded, resolve
                                toastr.success(LANG.authentication_successfull);
                            }
                        },
                    });
            } else {
                // Handle the "Cancel" action
                $("#res_waiter_id").val('');
            }
        });
        
    }
})

// update serial number of product item
function update_serial_no(){
    $('.product_row').each(function (index) {
        // Add the serial number to the first <td> of each row (index + 1 to start from 1)
        if ($(this).find('td:first').hasClass('serial_no')) {
            $(this).find('td:first').text(index + 1);
        }
    });
}




// Enhanced POS Table Functions - Excel-like Interface

// Global variables for Excel-like functionality
var clipboardData = null;
var searchTimeout = null;
var isProcessingPaste = false;

$(document).ready(function() {

    // Initialize Excel-like table on page load
    initializeExcelLikePOSTable();
    
    // Add empty row initially
    addEmptyProductRow();
    
    // Add keyboard shortcuts help
    addKeyboardShortcutsHelp();
    
    // Handle paste events globally
    $(document).on('paste', handleGlobalPaste);
    
    // Handle keyboard navigation
    $(document).on('keydown', 'table#pos_table input, table#pos_table select', handleKeyboardNavigation);
    
    // Handle cell editing
    $(document).on('focus', 'table#pos_table .editable-cell', handleCellFocus);
    $(document).on('blur', 'table#pos_table .editable-cell', handleCellBlur);
    
    // Handle product search in empty rows
    $(document).on('input', '.product-search-input', handleProductSearch);
    
    // Handle quantity batch input
    $(document).on('input', '.quantity-batch-input', handleQuantityBatchInput);
    
    // Handle click on product rows to edit
    $(document).on('click', '#pos_table tbody tr.product_row td', function(e) {
        handleProductRowClick($(this), e);
    });
    
    // Add tooltip to product name column
    $(document).on('mouseenter', '#pos_table tbody tr.product_row td:nth-child(2)', function() {
        if (!$(this).find('.product-edit-search').length) {
            $(this).attr('title', 'Click to change product');
        }
    });

        fixUnitSubmissionIssue();

});

// Initialize Excel-like POS table
function initializeExcelLikePOSTable() {
    // Update table headers to match Excel style
    updateTableHeaders();
    
    // Apply Excel-like styling
    applyExcelStyling();
    
    // Set up event listeners
    setupExcelLikeEventListeners();

    $(document).on('focusout', '.unit-input', function() {
    var row = $(this).closest('tr');
    persistUnitValue(row);
});

// معالج للحفاظ على القيم عند إضافة صف جديد
$(document).on('row-added', function(e, newRow) {
    // التأكد من عدم فقدان قيم الصفوف السابقة
    $('#pos_table tbody tr').each(function() {
        restoreUnitValue($(this));
    });
});
}

// Update table headers for Excel-like layout
function updateTableHeaders() {
    var headerHTML = `
        <tr>
            <th width="35%" class="excel-header">NAME</th>
            <th width="10%" class="excel-header">WH</th>
            
            <th width="8%" class="excel-header">Unit</th>
            <th width="4%" class="excel-header">Qty</th>
            <th width="10%" class="excel-header">USD</th>
            <th width="10%" class="excel-header">IQD</th>
            <th width="8%" class="excel-header">Dis %</th>
            <th width="10%" class="excel-header">Price </th>
            <th width="8%" class="excel-header">Subtotal</th>
            <th width="2%" class="excel-header">Stock</th>
        </tr>
    `;
    
    $('#pos_table thead').html(headerHTML);
}

// Apply Excel-like styling
function applyExcelStyling() {
    var styles = `
        <style>
        /* Excel-like POS Table Styling */
        #pos_table {
            border-collapse: separate;
            border-spacing: 0;
            border: 2px solid #d1d5db;
            border-radius: 8px;
            overflow: hidden;
            background: white;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        #pos_table th.excel-header {
            background: linear-gradient(180deg, #f8f9fa 0%, #e9ecef 100%);
            border: 1px solid #d1d5db;
            border-top: none;
            padding: 8px 6px;
            text-align: center;
            font-weight: 600;
            font-size: 12px;
            color: #374151;
            height: 32px;
            white-space: nowrap;
            text-transform: uppercase;
            letter-spacing: 0.3px;
        }
        
        #pos_table td {
            border: 1px solid #e5e7eb;
            padding: 0;
            height: 28px;
            vertical-align: middle;
            background: white;
            position: relative;
        }
        
        #pos_table tr:nth-child(even) td {
            background: #f9fafb;
        }
        
        #pos_table tr:hover td {
            background: #e0f2fe !important;
        }
        
        #pos_table tr.empty-row td {
            background: #fffbeb;
            border-style: dashed;
        }
        
        /* Clickable product name column */
        #pos_table tr.product_row td:nth-child(2) {
            cursor: pointer;
            position: relative;
        }
        
        #pos_table tr.product_row td:nth-child(2):hover {
            background: #dbeafe !important;
        }
        
        #pos_table tr.product_row td:nth-child(2):hover .product-info {
            text-decoration: underline;
            color: #2563eb;
        }
        
        #pos_table tr.product_row td:nth-child(2) .fa-edit {
            transition: opacity 0.2s;
        }
        
        #pos_table tr.product_row td:nth-child(2):hover .fa-edit {
            opacity: 1 !important;
        }
        
        #pos_table tr.empty-row:hover td {
            background: #fef3c7 !important;
        }
        
        /* Excel-like input styling */
        .excel-input {
            width: 100%;
            height: 32px;
            border: none;
            outline: none;
            padding: 2px 6px;
            font-size: 18px;
            background: transparent;
            text-align: center;
          line-height: 2.7;
        }
        
        .excel-input:focus {
            background: white;
            border: 2px solid #2563eb;
            z-index: 10;
            position: relative;
            box-shadow: 0 0 0 2px rgba(37, 99, 235, 0.1);
        }
        
        .excel-input.text-input {
            text-align: left;
            padding-left: 8px;
        }
        
        .excel-input.number-input {
            text-align: right;
            padding-right: 8px;
        }
        
        /* Product search input */
        .product-search-input {
            width: 100%;
            height: 28px;
            border: none;
            outline: none;
            padding: 2px 8px;
            font-size: 12px;
            background: transparent;
            text-align: left;
        }
        
        .product-search-input:focus {
            background: white;
            border: 2px solid #2563eb;
            box-shadow: 0 0 0 2px rgba(37, 99, 235, 0.1);
        }
        
        .product-search-input::placeholder {
            color: #9ca3af;
            font-style: italic;
        }
        
        /* Unit selector styling */
        .unit-selector {
            width: 100%;
            height: 28px;
            border: none;
            outline: none;
            padding: 0 6px;
            font-size: 12px;
            background: transparent;
            cursor: pointer;
        }
        
        .unit-selector:focus {
            background: white;
            border: 2px solid #2563eb;
            box-shadow: 0 0 0 2px rgba(37, 99, 235, 0.1);
        }
        
        /* Serial number styling */
        .serial-number {
            color: #6b7280;
            font-weight: 500;
            text-align: center;
            padding: 6px;
            font-size: 12px;
        }
        
        /* Action buttons */
        .excel-action-btn {
            background: none;
            border: none;
            color: #dc2626;
            cursor: pointer;
            padding: 4px;
            font-size: 14px;
            border-radius: 4px;
            transition: all 0.2s ease;
        }
        
        .excel-action-btn:hover {
            background: #dc2626;
            color: white;
            transform: scale(1.1);
        }
        
        /* Quantity batch input */
        .quantity-batch-input {
            width: 100%;
            height: 28px;
            border: none;
            outline: none;
            padding: 2px 8px;
            font-size: 12px;
            background: transparent;
            text-align: center;
        }
        
        .quantity-batch-input:focus {
            background: white;
            border: 2px solid #2563eb;
            box-shadow: 0 0 0 2px rgba(37, 99, 235, 0.1);
        }
        
        /* Empty row indicator */
        .empty-row-indicator {
            position: absolute;
            left: 4px;
            top: 50%;
            transform: translateY(-50%);
            color: #9ca3af;
            font-size: 10px;
            pointer-events: none;
        }
        
        /* Loading indicator */
        .cell-loading {
            position: relative;
        }
        
        .cell-loading::after {
            content: '';
            position: absolute;
            right: 4px;
            top: 50%;
            transform: translateY(-50%);
            width: 12px;
            height: 12px;
            border: 2px solid #d1d5db;
            border-top-color: #2563eb;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }
        
        @keyframes spin {
            to {
                transform: translateY(-50%) rotate(360deg);
            }
        }
        
        /* Paste indicator */
        .paste-indicator {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 12px 24px;
            border-radius: 8px;
            z-index: 9999;
            font-size: 14px;
            display: none;
        }
        
        /* Table container */
        .pos_product_div {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        </style>
    `;
    
    $('head').append(styles);
    $('body').append('<div class="paste-indicator">Processing paste...</div>');
}

// Setup Excel-like event listeners
function setupExcelLikeEventListeners() {
    // Remove existing row when last row is modified
    $(document).on('input change', '#pos_table tbody tr:last-child input, #pos_table tbody tr:last-child select', function() {
        var row = $(this).closest('tr');
        if (row.hasClass('empty-row')) {
            row.removeClass('empty-row');
            addEmptyProductRow();
        }
    });
    
    // Auto-resize text inputs based on content
    $(document).on('input', '.excel-input', function() {
        autoResizeInput($(this));
    });
    
    // Handle copy functionality
    $(document).on('keydown', '#pos_table', function(e) {
        if (e.ctrlKey && e.key === 'c') {
            copySelectedCells();
        }
    });
}
function addEmptyProductRow() {
    if ($('#pos_table tbody tr.empty-row').length > 0) {
        console.log('Empty row already exists, skipping...');
        return;
    }
    
    // تنظيف العلامات القديمة عند إضافة صف جديد
    $('.advanced-pricing-active').removeClass('advanced-pricing-active');
    $('.unit-input').removeData('advanced-pricing-processing');
    
    console.log('Adding new empty row...');

    var rowCount = $('#pos_table tbody tr').length;
    var rowIndex = rowCount;
    
    var emptyRowHTML = `
        <tr class="empty-row product_row" data-row_index="${rowIndex}">
            <td><input type="checkbox" class="row-checkbox"> ${rowIndex + 1}</td>
            <td class="serial-num">
                <input type="text" class="product-search-input form-control" placeholder="Search products in all warehouses..." data-row="${rowIndex}">
                
                <!-- الحقول الأساسية المطلوبة -->
                <input type="hidden" class="product_id" name="products[${rowIndex}][product_id]" value="">
                <input type="hidden" class="variation_id row_variation_id" name="products[${rowIndex}][variation_id]" value="">
                <input type="hidden" class="product_name" name="products[${rowIndex}][product_name]" value="">
                <input type="hidden" class="product_type" name="products[${rowIndex}][product_type]" value="single">
                <input type="hidden" class="enable_sr_no" value="0">
                <input type="hidden" class="modifiers_exist" value="0">
                <input type="hidden" name="products[${rowIndex}][transaction_sell_lines_id]" value="">
                <input type="hidden" name="products[${rowIndex}][lot_no_line_id]" value="">
                <input type="hidden" class="product_unit_id" name="products[${rowIndex}][product_unit_id]" value="">
                <input type="hidden" class="base_unit_multiplier" name="products[${rowIndex}][base_unit_multiplier]" value="1">
                <input type="hidden" class="hidden_base_unit_sell_price" value="0">
            </td>
            
            <td>
                <!-- تحويل حقل المستودع إلى حقل كتابة -->
                <input type="text" 
                       class="form-control warehouse-input excel-input" 
                       name="products[${rowIndex}][warehouse_code]"
                       placeholder="W01"
                       value=""
                       style="text-align: center;">
                <input type="hidden" 
                       class="warehouse_id" 
                       name="products[${rowIndex}][warehouse_id]" 
                       value="">
            </td>
            
            <td>
                <!-- تحويل حقل الوحدة إلى حقل كتابة -->
                <input type="text" 
                       class="form-control unit-input excel-input" 
                       name="products[${rowIndex}][unit_name]"
                       placeholder="EA"
                       value="EA"
                       style="text-align: center;">
                <input type="hidden" 
                       class="sub_unit_id" 
                       name="products[${rowIndex}][sub_unit_id]" 
                       value="">
                <input type="hidden" 
                       class="unit_multiplier" 
                       name="products[${rowIndex}][unit_multiplier]" 
                       value="1">
            </td>
            
            <td>
                <input type="text" 
                       class="form-control pos_quantity quantity-batch-input" 
                       name="products[${rowIndex}][quantity]"
                       placeholder="Qty" 
                       data-row="${rowIndex}"
                       value="1"
                       data-rule-required="true"
                       data-msg-required="Please enter quantity"
                       data-qty_available="0"
                       data-rule-max-value="999999"
                       data-msg-max-value="Max: 999999">
            </td>
            
            <td><input type="text" class="form-control excel-input number-input price-input" placeholder="0.00"></td>
                       <td><input type="text" class="form-control excel-input number-input" readonly placeholder="0" style="background-color: #f8f9fa; cursor: not-allowed;"></td>

            
            <td>
                <input type="text" class="form-control excel-input number-input discount_percent" value="0">
                <select class="row_discount_type hide" name="products[${rowIndex}][line_discount_type]">
                    <option value="percentage" selected>Percentage</option>
                    <option value="fixed">Fixed</option>
                </select>
                <input type="hidden" class="row_discount_amount" name="products[${rowIndex}][line_discount_amount]" value="0">
            </td>
            
            <td>
                <input type="text" 
                       class="form-control pos_unit_price_inc_tax excel-input number-input" 
                       name="products[${rowIndex}][unit_price_inc_tax]"
                       readonly 
                       placeholder="0.00"
                       value="0">
                <input type="hidden" 
                       class="pos_unit_price" 
                       name="products[${rowIndex}][unit_price]"
                       value="0">
                <input type="hidden" 
                       class="item_tax" 
                       name="products[${rowIndex}][item_tax]" 
                       value="0">
            </td>
            
            <td class="text-right">
                <input type="hidden" 
                       class="pos_line_total" 
                       name="products[${rowIndex}][line_total]" 
                       value="0">
                <span class="pos_line_total_text">0.00</span>
            </td>
            
            <td><span class="stock-info text-center">-</span></td>
            
            <td>
                <button type="button" class="excel-action-btn remove-row" title="Remove">
                    <i class="fa fa-times"></i>
                </button>
            </td>
            
            <!-- حقل الضريبة المخفي -->
            <td class="hide">
                <select class="tax_id" name="products[${rowIndex}][tax_id]">
                    <option value="" data-rate="0" selected>No Tax</option>
                </select>
            </td>
        </tr>
    `;
    
    $('#pos_table tbody').append(emptyRowHTML);
    updateSerialNumbers();
    
    // إضافة event listeners للصف الجديد
    var newRow = $('#pos_table tbody tr').last();
    attachRowEventListeners(newRow);
    attachUnitWarehouseEventListeners(newRow);
}

// دالة لإضافة معالجات الأحداث للوحدة والمستودع
function attachUnitWarehouseEventListeners(row) {
    // معالج حدث لحقل المستودع
    row.find('.warehouse-input').on('blur change', function() {
        var warehouseCode = $(this).val().trim().toUpperCase();
        var warehouseIdField = row.find('.warehouse_id');
        
        if (warehouseCode) {
            // تحديث الكود بصيغة موحدة (مثل W01, W02, إلخ)
            if (/^\d+$/.test(warehouseCode)) {
                // إذا كان رقم فقط، أضف W في البداية
                warehouseCode = 'W' + warehouseCode.padStart(2, '0');
                $(this).val(warehouseCode);
            }
            
            // حفظ معرف المستودع (يمكن أن يكون نفس الكود أو معرف من قاعدة البيانات)
            warehouseIdField.val(warehouseCode);
            
            // التحقق من صحة المستودع (اختياري)
            validateWarehouse(warehouseCode, row);
        }
    });
    
    // معالج حدث لحقل الوحدة مع التثبيت
    row.find('.unit-input').on('blur change', function() {
        var unitName = $(this).val().trim().toUpperCase();
        var unitIdField = row.find('.sub_unit_id');
        var multiplierField = row.find('.unit_multiplier');
        
        if (unitName) {
            $(this).val(unitName);
            
            // تحديث معرف الوحدة ومضاعف الوحدة
            updateUnitDetails(unitName, row);
            
            // حفظ القيمة
            persistUnitValue(row);
        }
    });
    
    // معالج فقدان التركيز للوحدة - تثبيت القيمة
    row.find('.unit-input').on('blur', function() {
        persistUnitValue(row);
    });
    
    // معالج الحصول على التركيز للوحدة - استعادة القيمة
    row.find('.unit-input').on('focus', function() {
        restoreUnitValue(row);
        $(this).select(); // تحديد النص بالكامل
    });
    
    // معالج الضغط على Enter للانتقال للحقل التالي
    row.find('.warehouse-input, .unit-input').on('keypress', function(e) {
        if (e.which === 13) { // Enter key
            e.preventDefault();
            
            // حفظ قيمة الوحدة إذا كان الحقل الحالي هو حقل الوحدة
            if ($(this).hasClass('unit-input')) {
                persistUnitValue(row);
            }
            
            // الانتقال للحقل التالي
            var nextInput = $(this).closest('td').next('td').find('input:visible').first();
            if (nextInput.length) {
                nextInput.focus().select();
            }
        }
    });
    
    // معالج لحفظ القيمة عند استخدام Tab
    row.find('.unit-input').on('keydown', function(e) {
        if (e.which === 9) { // Tab key
            persistUnitValue(row);
        }
    });
    
    // استعادة القيمة المحفوظة إذا وجدت عند التهيئة
    setTimeout(function() {
        restoreUnitValue(row);
    }, 100);
}

// دالة للتحقق من صحة المستودع
function validateWarehouse(warehouseCode, row) {
    // يمكنك إضافة AJAX call هنا للتحقق من وجود المستودع
    // مثال بسيط للتحقق من النطاق
    var warehouseNumber = parseInt(warehouseCode.replace('W', ''));
    
    if (warehouseNumber >= 1 && warehouseNumber <= 18) {
        row.find('.warehouse-input').removeClass('error-input');
        updateStockInfoForWarehouse(row, warehouseCode);
    } else {
        row.find('.warehouse-input').addClass('error-input');
        toastr.warning('Invalid warehouse code. Use W01-W18');
    }
}

// دالة لتحديث تفاصيل الوحدة
function updateUnitDetails(unitName, row) {
    // قائمة الوحدات الشائعة ومضاعفاتها
    var unitMappings = {
        'EA': { id: 1, multiplier: 1, allow_decimal: true },
        'PCS': { id: 2, multiplier: 1, allow_decimal: true },
        'BOX': { id: 3, multiplier: 12, allow_decimal: false },
        'CTN': { id: 4, multiplier: 24, allow_decimal: false },
        'DZ': { id: 5, multiplier: 12, allow_decimal: false },
        'PACK': { id: 6, multiplier: 6, allow_decimal: false },
        'KG': { id: 7, multiplier: 1, allow_decimal: true },
        'GM': { id: 8, multiplier: 0.001, allow_decimal: true },
        'LTR': { id: 9, multiplier: 1, allow_decimal: true },
        'ML': { id: 10, multiplier: 0.001, allow_decimal: true }
    };
    
    var unitDetails = unitMappings[unitName] || { id: '', multiplier: 1, allow_decimal: true };
    
    // تحديث الحقول المخفية
    row.find('.sub_unit_id').val(unitDetails.id);
    row.find('.unit_multiplier').val(unitDetails.multiplier);
    row.find('.base_unit_multiplier').val(unitDetails.multiplier);
    
    // تحديث قواعد التحقق للكمية
    var qtyElement = row.find('.pos_quantity');
    qtyElement.attr('data-decimal', unitDetails.allow_decimal ? 1 : 0);
    
    if (qtyElement.rules && typeof qtyElement.rules === 'function') {
        qtyElement.rules('add', {
            abs_digit: !unitDetails.allow_decimal
        });
    }
    
    // إعادة حساب السعر بناءً على المضاعف
   
}

function updatePriceBasedOnUnitWithAddition(row, multiplier) {
    var basePrice = parseFloat(row.find('.hidden_base_unit_sell_price').val()) || 0;
    var newPrice = basePrice * multiplier;
    
    // إضافة المبلغ الإضافي بناءً على المضاعف
    var additionalAmount = 0;
    
    if (multiplier === 0.5) {
        additionalAmount = 0; // إضافة دولار واحد
    } else if (multiplier === 0.25) {
        additionalAmount = 0; // إضافة دولارين
    } else if (multiplier === 0.1) {
        additionalAmount = 0; // إضافة دولار واحد
    }
    
    // تطبيق الإضافة على السعر النهائي
    newPrice = newPrice + additionalAmount;
    
    // تحديث حقول السعر
    __write_number(row.find('.pos_unit_price'), newPrice);
    row.find('.pos_unit_price').trigger('change');
    
    // تحديث السعر بالدولار في العرض
    row.find('td:eq(5) input').val(formatNumber(newPrice, 2));
    
    // تحديث السعر بالدينار العراقي
    var iqrPrice = convertUSDToIQD(newPrice);
    row.find('td:eq(6) input').val(formatNumber(iqrPrice, 0));
    
    console.log('Price calculation with addition:', {
        basePrice: basePrice,
        multiplier: multiplier,
        priceAfterMultiplier: basePrice * multiplier,
        additionalAmount: additionalAmount,
        finalPrice: newPrice
    });
}


// دالة لتحميل المستودعات للصف
function loadWarehousesForRow(row, rowIndex) {
    loadWarehousesDropdown(function(warehouses) {
        var warehouseOptions = '<option value="">All Warehouses</option>';
        
        // تجميع المستودعات حسب النوع
        var sapWarehouses = [];
        var localLocations = [];
        
        warehouses.forEach(function(warehouse) {
            if (warehouse.type === 'sap_warehouse') {
                sapWarehouses.push(warehouse);
            } else {
                localLocations.push(warehouse);
            }
        });
        
        // إضافة المستودعات SAP
        if (sapWarehouses.length > 0) {
            warehouseOptions += '<optgroup label="SAP Warehouses">';
            sapWarehouses.forEach(function(warehouse) {
                warehouseOptions += `<option value="${warehouse.id}" data-type="sap">
                                       ${warehouse.code} - ${warehouse.name}
                                    </option>`;
            });
            warehouseOptions += '</optgroup>';
        }
        
        // إضافة المواقع المحلية
        if (localLocations.length > 0) {
            warehouseOptions += '<optgroup label="Local Locations">';
            localLocations.forEach(function(location) {
                warehouseOptions += `<option value="${location.id}" data-type="local">
                                       ${location.code} - ${location.name}
                                    </option>`;
            });
            warehouseOptions += '</optgroup>';
        }
        
        // إضافة المستودعات الافتراضية W01-W18 إذا لم تكن موجودة
        var hasDefaultWarehouses = false;
        for (let i = 1; i <= 18; i++) {
            const warehouseCode = 'W' + i.toString().padStart(2, '0');
            if (!warehouses.some(w => w.code === warehouseCode)) {
                if (!hasDefaultWarehouses) {
                    warehouseOptions += '<optgroup label="Default Warehouses">';
                    hasDefaultWarehouses = true;
                }
                warehouseOptions += `<option value="${warehouseCode}" data-type="default">
                                       ${warehouseCode}
                                    </option>`;
            }
        }
        if (hasDefaultWarehouses) {
            warehouseOptions += '</optgroup>';
        }
        
        // تحديث dropdown
        row.find('.warehouse-selector').html(warehouseOptions);
        
        console.log('Loaded ' + warehouses.length + ' warehouses for row ' + rowIndex);
    });
}

function attachRowEventListeners(row) {
    if (!row || row.length === 0) return;
    
    var rowIndex = row.data('row_index') || row.index();
    
    // حدث تغيير الكمية
    row.find('.pos_quantity').off('input change').on('input change', function() {
        if (pos_form_validator) {
            pos_form_validator.element($(this));
        }
        
        var quantity = __read_number($(this)) || 0;
        var unit_price_inc_tax = __read_number(row.find('.pos_unit_price_inc_tax'));
        var line_total = quantity * unit_price_inc_tax;
        
        __write_number(row.find('.pos_line_total'), line_total, false);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        pos_total_row();
       // adjustComboQty(row);
    });
    
    // حدث تغيير السعر شامل الضريبة
    row.find('.pos_unit_price_inc_tax').off('input change').on('input change', function() {
        var unit_price_inc_tax = __read_number($(this));
        var tax_rate = row.find('.tax_id').find(':selected').data('rate') || 0;
        var quantity = __read_number(row.find('.pos_quantity'));
        
        var line_total = quantity * unit_price_inc_tax;
        var discounted_unit_price = __get_principle(unit_price_inc_tax, tax_rate);
        var unit_price = get_unit_price_from_discounted_unit_price(row, discounted_unit_price);
        
        __write_number(row.find('.pos_unit_price'), unit_price);
        __write_number(row.find('.pos_line_total'), line_total, false);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        pos_each_row(row);
        pos_total_row();
    });
    
    // حدث تغيير الخصم
    row.find('.discount_percent').off('change input').on('change input', function() {
        var discountPercent = parseFloat($(this).val()) || 0;
        
        // تحديث نوع ومقدار الخصم
        row.find('.row_discount_type').val('percentage');
        __write_number(row.find('.row_discount_amount'), discountPercent);
        
        // إعادة حساب السعر
        var discounted_unit_price = calculate_discounted_unit_price(row);
        var tax_rate = row.find('.tax_id').find(':selected').data('rate') || 0;
        var quantity = __read_number(row.find('.pos_quantity'));
        
        var unit_price_inc_tax = __add_percent(discounted_unit_price, tax_rate);
        var line_total = quantity * unit_price_inc_tax;
        
        __write_number(row.find('.pos_unit_price_inc_tax'), unit_price_inc_tax);
        __write_number(row.find('.pos_line_total'), line_total, false);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        pos_each_row(row);
        pos_total_row();
    });
    
    // حدث زر الحذف
    row.find('.remove-row').off('click').on('click', function() {
        if (confirm('Are you sure you want to remove this item?')) {
            row.remove();
            updateSerialNumbers();
            pos_total_row();
        }
    });
    
    // حدث البحث عن المنتج
    row.find('.product-search-input').off('input').on('input', function() {
        handleProductSearch.call(this);
    });
}

var additionalStyles = `
<style>
.stock-available {
    color: #059669;
    font-weight: 500;
}

.stock-warning {
    color: #d97706;
    font-weight: 500;
}

.stock-error {
    color: #dc2626;
    font-weight: 500;
}

.stock-info small {
    font-size: 10px;
    opacity: 0.8;
}

.product-search-input:focus::placeholder {
    color: #60a5fa;
}

.warehouse-selector {
    font-size: 11px;
    height: 28px;
}

.warehouse-selector:focus {
    box-shadow: 0 0 0 2px rgba(59, 130, 246, 0.1);
}

/* تحسين مظهر جدول البحث */
.product-search-table {
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    border-radius: 8px;
    overflow: hidden;
}

.product-search-table thead th {
    border: none !important;
    text-transform: uppercase;
    font-size: 11px;
    letter-spacing: 0.5px;
}

.product-search-table tbody tr {
    transition: all 0.2s ease;
}
</style>
`;

$(document).ready(function() {
    $('head').append(additionalStyles);
});

// Handle product search in empty rows
function handleProductSearch(e) {
    var input = $(this);
    var searchTerm = input.val().trim();
    var row = input.closest('tr');
    var rowIndex = row.data('row_index');
    
    // Clear previous timeout
    if (searchTimeout) {
        clearTimeout(searchTimeout);
    }
    
    // Add loading indicator
    input.addClass('cell-loading');
    
    // Debounce search
    searchTimeout = setTimeout(function() {
        if (searchTerm.length >= 2) {
            searchProducts(searchTerm, row, rowIndex);
        } else {
            input.removeClass('cell-loading');
            clearRowData(row);
        }
    }, 500);
}
function searchProducts(searchTerm, row, rowIndex) {
    var price_group = $('#price_group').val() || '';
    
    $.ajax({
        url: base_path + '/products/list',
        method: 'GET',
        dataType: 'json',
        data: {
            price_group: price_group,
            term: searchTerm,
            not_for_selling: 0,
            limit: 20,
            search_all_locations: true,
            include_all_warehouses: true,
            with_warehouse_stock: true,
            // إضافة المعاملات الجديدة لجلب الوحدات الفرعية
            with_sub_units: true,
            include_unit_details: true,
            load_sub_units: true
        },
        success: function(products) {
            var input = row.find('.product-search-input');
            input.removeClass('cell-loading');
            
            if (!products || !Array.isArray(products)) {
                console.warn('Invalid products data received');
                products = [];
            }
            
            // معالجة البيانات لاستخراج الوحدات الفرعية
            products.forEach(function(product) {
                try {
                    processProductUnitsData(product);
                    processProductWarehouseData(product);
                } catch (e) {
                    console.error('Error processing product data:', e);
                }
            });
            
            if (products.length === 1) {
                populateRowWithProduct(row, products[0], rowIndex);
            } else if (products.length > 1) {
                showProductDropdown(input, products, row, rowIndex);
            } else {
                toastr.warning('No products found for: ' + searchTerm);
                clearRowData(row);
            }
            
        
        // عرض أسعار وحدات القياس
        
        },
        error: function(jqXHR, textStatus, errorThrown) {
            console.error('Search error:', textStatus, errorThrown);
            row.find('.product-search-input').removeClass('cell-loading');
            toastr.error('Error searching products');
        }
    });
    console.log('All Product Data:', products);

}

// 2. دالة معالجة بيانات الوحدات الفرعية المحسنة
function processProductUnitsData(product) {
    var units = [];
    var hasSubUnits = false;
    
    console.log('Processing units for product:', product.name);
    console.log('Product data:', product);
    
    // 1. البحث في product.units أو product.sub_units
    var sourceUnits = product.units || product.sub_units || product.processed_units || [];
    
    if (sourceUnits && Array.isArray(sourceUnits) && sourceUnits.length > 0) {
        console.log('Found units in product data:', sourceUnits);
        units = sourceUnits.map(function(unit) {
            return {
                id: unit.id || unit.unit_id,
                unit_name: unit.unit_name || unit.name || unit.actual_name,
                multiplier: parseFloat(unit.multiplier || unit.base_unit_multiplier) || 1,
                is_base_unit: unit.is_base_unit || (unit.multiplier == 1 ? 1 : 0),
                allow_decimal: unit.allow_decimal !== undefined ? unit.allow_decimal : 1,
                base_unit_selling_price: parseFloat(product.selling_price || product.default_sell_price || 0)
            };
        });
        hasSubUnits = units.length > 1;
    }
    
    // 2. إذا لم نجد وحدات، أنشئ الوحدة الأساسية
    if (units.length === 0) {
        console.log('No units found, creating base unit');
        units.push({
            id: product.unit_id || '',
            unit_name: product.unit || 'EA',
            multiplier: 1,
            is_base_unit: 1,
            allow_decimal: product.unit_allow_decimal || 1,
            base_unit_selling_price: parseFloat(product.selling_price || product.default_sell_price || 0)
        });
    }
    
    // 3. البحث في product_locations للوحدات الفرعية (كبديل)
    if (!hasSubUnits && product.product_locations && Array.isArray(product.product_locations)) {
        console.log('Checking product_locations for units');
        var location = product.product_locations[0];
        
        if (location && location.sub_units && Array.isArray(location.sub_units) && location.sub_units.length > 0) {
            console.log('Found sub units in location:', location.sub_units);
            location.sub_units.forEach(function(subUnit) {
                // تجنب التكرار
                var exists = units.some(function(u) { return u.id == subUnit.id; });
                if (!exists) {
                    units.push({
                        id: subUnit.id,
                        unit_name: subUnit.unit_name || subUnit.name || subUnit.actual_name,
                        multiplier: parseFloat(subUnit.multiplier || subUnit.base_unit_multiplier) || 1,
                        is_base_unit: 0,
                        allow_decimal: subUnit.allow_decimal || 0,
                        base_unit_selling_price: parseFloat(product.selling_price || product.default_sell_price || 0)
                    });
                }
            });
            hasSubUnits = units.length > 1;
        }
    }
    
    // 4. إزالة الوحدات المكررة
    var uniqueUnits = [];
    var seenUnits = new Set();
    
    units.forEach(function(unit) {
        var key = unit.unit_name + '_' + unit.multiplier;
        if (!seenUnits.has(key)) {
            seenUnits.add(key);
            uniqueUnits.push(unit);
        }
    });
    
    // حفظ النتائج في المنتج
    product.processed_units = uniqueUnits;
    product.has_sub_units = hasSubUnits;
    product.total_units_count = uniqueUnits.length;
    
    console.log('Final processed units for', product.name, ':', uniqueUnits);
    console.log('Has sub units:', hasSubUnits);
    
    return {
        units: uniqueUnits,
        hasSubUnits: hasSubUnits
    };
}


// تحديث دالة processProductWarehouseData لمعالجة البيانات بشكل صحيح
function processProductWarehouseData(product) {
    var totalStock = 0;
    var warehouseDetails = [];
    var processedWarehouses = new Set(); // لتجنب التكرار
    
    // 1. معالجة بيانات المستودعات من SAP
    if (product.warehouses && Array.isArray(product.warehouses)) {
        product.warehouses.forEach(function(warehouse) {
            var warehouseId = warehouse.warehouse_code || warehouse.warehouse_id || warehouse.id;
            
            if (!processedWarehouses.has(warehouseId)) {
                processedWarehouses.add(warehouseId);
                
                var qty = parseFloat(warehouse.quantity || warehouse.qty || warehouse.in_stock || 0);
                totalStock += qty;
                
                warehouseDetails.push({
                    id: warehouseId,
                    code: warehouse.warehouse_code || warehouse.code || warehouseId,
                    name: warehouse.warehouse_name || warehouse.name || 'Warehouse ' + warehouseId,
                    quantity: qty,
                    committed: parseFloat(warehouse.committed || 0),
                    ordered: parseFloat(warehouse.ordered || 0),
                    available: qty - parseFloat(warehouse.committed || 0),
                    type: 'sap'
                });
            }
        });
    }
    
    // 2. معالجة بيانات المواقع من جميع المواقع
    if (product.all_location_details && Array.isArray(product.all_location_details)) {
        product.all_location_details.forEach(function(location) {
            var locationId = 'LOC_' + location.location_id;
            
            if (!processedWarehouses.has(locationId)) {
                processedWarehouses.add(locationId);
                
                var qty = parseFloat(location.qty_available || 0);
                totalStock += qty;
                
                warehouseDetails.push({
                    id: location.location_id,
                    code: location.location_code || locationId,
                    name: location.location_name || 'Location ' + location.location_id,
                    quantity: qty,
                    committed: parseFloat(location.qty_committed || 0),
                    ordered: 0,
                    available: qty - parseFloat(location.qty_committed || 0),
                    type: 'local'
                });
            }
        });
    }
    
    // 3. معالجة variation_location_details كخيار احتياطي
    if (warehouseDetails.length === 0 && product.variation_location_details) {
        product.variation_location_details.forEach(function(vld) {
            var locationId = 'VLD_' + vld.location_id;
            
            if (!processedWarehouses.has(locationId)) {
                processedWarehouses.add(locationId);
                
                var qty = parseFloat(vld.qty_available || 0);
                totalStock += qty;
                
                warehouseDetails.push({
                    id: vld.location_id,
                    code: vld.location_code || locationId,
                    name: vld.location_name || 'Location ' + vld.location_id,
                    quantity: qty,
                    committed: 0,
                    ordered: 0,
                    available: qty,
                    type: 'variation'
                });
            }
        });
    }
    
    // 4. إذا لم تكن هناك بيانات مفصلة، استخدم الكمية الإجمالية
    if (warehouseDetails.length === 0 && product.qty_available) {
        totalStock = parseFloat(product.qty_available || 0);
        warehouseDetails.push({
            id: 'all',
            code: 'ALL',
            name: 'All Locations',
            quantity: totalStock,
            committed: 0,
            ordered: 0,
            available: totalStock,
            type: 'default'
        });
    }
    
    // ترتيب المستودعات حسب الكمية المتاحة (الأكثر أولاً)
    warehouseDetails.sort((a, b) => b.available - a.available);
    
    // تحديث بيانات المنتج
    product.total_stock = totalStock;
    product.warehouse_details = warehouseDetails;
    product.has_multiple_locations = warehouseDetails.length > 1;
    
    console.log('Processed warehouse data for product:', {
        product_name: product.name,
        total_stock: totalStock,
        warehouses_count: warehouseDetails.length,
        warehouses: warehouseDetails
    });
}

// دالة مساعدة لجلب تفاصيل المستودعات لمنتج محدد
function fetchProductWarehouseDetails(variationId, callback) {
    $.ajax({
        url: '/products/warehouse-details/' + variationId,
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                callback(response.data);
            } else {
                console.error('Failed to fetch warehouse details:', response.msg);
                callback(null);
            }
        },
        error: function(xhr, status, error) {
            console.error('Error fetching warehouse details:', error);
            callback(null);
        }
    });
}

// دالة لتحميل قائمة المستودعات للاختيار
function loadWarehousesDropdown(callback) {
    // استخدام المسار الكامل مع معالجة الأخطاء
    var url = (typeof base_path !== 'undefined' ? base_path : '') + '/warehouses/all';
    
    $.ajax({
        url: url,
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response && response.success && response.warehouses) {
                callback(response.warehouses);
            } else {
                console.warn('Invalid warehouses response');
                callback(getDefaultWarehouses());
            }
        },
        error: function(xhr, status, error) {
            console.error('Error loading warehouses:', error);
            // استخدام مستودعات افتراضية في حالة الفشل
            callback(getDefaultWarehouses());
        }
    });
}

// دالة للحصول على المستودعات الافتراضية
function getDefaultWarehouses() {
    var warehouses = [];
    for (let i = 1; i <= 18; i++) {
        const warehouseCode = 'W' + i.toString().padStart(2, '0');
        warehouses.push({
            id: warehouseCode,
            code: warehouseCode,
            name: 'Warehouse ' + i.toString().padStart(2, '0'),
            type: 'default'
        });
    }
    return warehouses;
}
// Show product dropdown for selection
// function showProductDropdown(input, products, row, rowIndex) {
//     $('.product-dropdown').remove();

//     var dropdown = $('<div class="product-dropdown product-search-container"></div>');
//     dropdown.css({
//         position: 'fixed',
//         top: input.offset().top + input.outerHeight(),
//         left: '10px',
//         right: '10px',
//         width: 'auto',
//         maxHeight: '600px',
//         overflowY: 'auto',
//         overflowX: 'auto',
//         background: 'white',
//         border: '2px solid #d1d5db',
//         borderRadius: '8px',
//         boxShadow: '0 10px 30px rgba(0, 0, 0, 0.2)',
//         zIndex: 1000
//     });

//     var $table = $('<table class="product-search-table table table-bordered table-striped">');
//     $table.css({
//         'table-layout': 'auto',
//         'width': '100%',
//         'min-width': '1500px',
//         'border-collapse': 'collapse',
//         'margin': '0',
//         'font-size': '12px'
//     });
    
//     var $thead = $('<thead>').appendTo($table);
//     var $tbody = $('<tbody>').appendTo($table);

//     // رؤوس الأعمدة المحدثة
//     var headerRow = $('<tr>');
//     headerRow.append('<th style="width: 20%; padding: 12px; background: #1a365d; color: white; font-weight: 600;">Product</th>');
   
//     headerRow.append('<th style="width: 10%; padding: 12px; background: #1a365d; color: white; font-weight: 600;">Category</th>');
//     headerRow.append('<th style="width: 12%; padding: 12px; background: #1a365d; color: white; font-weight: 600;">Foreign Name</th>');
//     headerRow.append('<th style="width: 8%; padding: 12px; background: #1a365d; color: white; font-weight: 600; text-align: center;">SKU</th>');
//     headerRow.append('<th style="width: 8%; padding: 12px; background: #1a365d; color: white; font-weight: 600; text-align: right;">Price (USD)</th>');
//     headerRow.append('<th style="width: 10%; padding: 12px; background: #1a365d; color: white; font-weight: 600; text-align: right;">Price (IQD)</th>');
//     headerRow.append('<th style="width: 6%; padding: 12px; background: #1a365d; color: white; font-weight: 600; text-align: center;">Discount</th>');
//     headerRow.append('<th style="width: 10%; padding: 12px; background: #1a365d; color: white; font-weight: 600; text-align: right;">Final Price</th>');
//       headerRow.append('<th style="width: 5%; padding: 12px; background: #1a365d; color: white; font-weight: 600; text-align: center;">UOM</th>');
  
//     headerRow.append('<th style="width: 8%; padding: 12px; background: #1a365d; color: white; font-weight: 600; text-align: right;">Total Stock</th>');
//     headerRow.append('<th style="width: 25%; padding: 12px; background: #1a365d; color: white; font-weight: 600;">All Locations Stock</th>');
//     $thead.append(headerRow);

//     var currentIndex = -1;

  

//     dropdown.append($table);
//     $('body').append(dropdown);

//     // إعداد التنقل بلوحة المفاتيح
//     setupDropdownKeyboardNavigation(dropdown, input, row, rowIndex, products);

//     // نقل التركيز فوراً إلى جدول البحث
//     setTimeout(function() {
//         var firstRow = dropdown.find('tbody tr').first();
//         if (firstRow.length) {
//             firstRow.addClass('highlighted');
//             // التمرير لإظهار الصف الأول
//             dropdown.scrollTop(0);
//         }
//         // إضافة فئة للإشارة إلى وجود نتائج
//         input.addClass('has-results');
//     }, 50);
// }


// Old keyboard navigation - replaced by setupEnhancedKeyboardNavigation
// Keeping for reference but not used
/*
function setupDropdownKeyboardNavigation(dropdown, input, row, rowIndex, products) {
    var currentIndex = 0; // البدء من الصف الأول
    var $rows = dropdown.find('tbody tr');
    
    // تمييز الصف الأول فوراً
    if ($rows.length > 0) {
        highlightDropdownRow($rows, currentIndex);
    }
    
    // معالج لوحة المفاتيح على مستوى الوثيقة
    var keyHandler = function(e) {
        // التحقق من أن القائمة موجودة
        if (!dropdown.is(':visible')) {
            $(document).off('keydown.productDropdown');
            return;
        }
        
        // معالجة مفاتيح التنقل
        switch(e.keyCode) {
            case 40: // Arrow Down
                e.preventDefault();
                e.stopPropagation();
                currentIndex = currentIndex < $rows.length - 1 ? currentIndex + 1 : 0;
                highlightDropdownRow($rows, currentIndex);
                break;
                
            case 38: // Arrow Up
                e.preventDefault();
                e.stopPropagation();
                currentIndex = currentIndex > 0 ? currentIndex - 1 : $rows.length - 1;
                highlightDropdownRow($rows, currentIndex);
                break;
                
            case 13: // Enter
                e.preventDefault();
                e.stopPropagation();
                if (currentIndex >= 0 && currentIndex < $rows.length) {
                    var selectedProduct = $rows.eq(currentIndex).data('product-data');
                    if (selectedProduct) {
                        selectProduct(selectedProduct);
                    }
                }
                break;
                
            case 27: // Escape
                e.preventDefault();
                e.stopPropagation();
                closeDropdown();
                break;
                
            case 9: // Tab
                e.preventDefault();
                e.stopPropagation();
                // يمكنك إضافة التنقل بـ Tab إذا أردت
                break;
        }
    };
    
    // إضافة معالج الأحداث على مستوى الوثيقة
    $(document).off('keydown.productDropdown').on('keydown.productDropdown', keyHandler);
    
    // معالج النقر على الصفوف
    $rows.off('click').on('click', function() {
        var selectedProduct = $(this).data('product-data');
        if (selectedProduct) {
            selectProduct(selectedProduct);
        }
    });
    
    // معالج التحويم
    $rows.off('mouseenter').on('mouseenter', function() {
        currentIndex = $(this).index();
        highlightDropdownRow($rows, currentIndex);
    });
    
    // دالة اختيار المنتج
    function selectProduct(product) {
        populateRowWithProduct(row, product, rowIndex);
        closeDropdown();
        
        // التركيز على حقل الكمية بعد الاختيار
        setTimeout(function() {
            row.find('.warehouse-input').focus().select();
        }, 100);
    }
    
    // دالة إغلاق القائمة
    function closeDropdown() {
        dropdown.remove();
        $(document).off('keydown.productDropdown');
        $(document).off('click.productDropdown');
        input.removeClass('has-results');
        input.val('').focus();
    }
    
    // إغلاق عند النقر خارج القائمة
    $(document).off('click.productDropdown').on('click.productDropdown', function(e) {
        if (!$(e.target).closest('.product-dropdown, .product-search-input').length) {
            closeDropdown();
        }
    });
    
    // معالج تغيير حجم النافذة
    $(window).off('resize.product-dropdown').on('resize.product-dropdown', function() {
        dropdown.css({
            top: input.offset().top + input.outerHeight(),
            left: '10px',
            right: '10px'
        });
    });
}
*/

// Enhanced CSS for product dropdown
var enhancedDropdownStyles = `
<style>
/* Product dropdown container */
.product-dropdown {
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Controls container */
.product-dropdown .product-search-controls {
    position: sticky;
    top: 0;
    z-index: 10;
    background: #f3f4f6;
    border-bottom: 1px solid #e5e7eb;
    min-height: 50px;
}

/* Product count styling */
.product-dropdown .product-count {
    font-size: 12px;
    color: #6b7280;
    font-weight: 500;
    white-space: nowrap;
}

/* Product search table enhancements */
.product-dropdown .product-search-table {
    margin: 0;
    border: none;
    position: relative;
    border-collapse: separate;
    border-spacing: 0;
}

/* Table wrapper for proper scrolling */
.product-dropdown .table-wrapper {
    position: relative;
    max-height: calc(100vh - 150px);
    overflow: visible;
}

/* Sticky table header */
.product-dropdown .product-search-table thead {
    position: sticky;
    top: -1px; /* Slight negative to prevent gap */
    z-index: 20;
    background: #1a365d;
}

.product-dropdown .product-search-table thead th {
    background: #1a365d;
    color: white;
    font-weight: 600;
    padding: 12px;
    border: 1px solid #2c5282;
    white-space: nowrap;
    position: relative;
}

/* Remove table display hacks - use standard table layout */
.product-dropdown .product-search-table tbody {
    display: table-row-group;
}

.product-dropdown .product-search-table tr {
    display: table-row;
}

.product-dropdown .product-search-table td,
.product-dropdown .product-search-table th {
    display: table-cell;
}

.product-dropdown .product-search-table tbody tr {
    cursor: pointer;
    transition: all 0.15s ease;
}

.product-dropdown .product-search-table tbody tr:hover {
    background-color: #f3f4f6;
}

/* Highlighted row styling */
.product-dropdown .product-search-table tbody tr.highlighted {
    background-color: #e0e7ff !important;
    outline: 2px solid #4f46e5;
    outline-offset: -1px;
}

.product-dropdown .product-search-table tbody tr.highlighted td {
    background-color: transparent !important;
}

/* Out of stock styling */
.product-dropdown .product-search-table tr.out-of-stock {
    opacity: 0.6;
}

.product-dropdown .product-search-table tr.out-of-stock td {
    color: #dc2626;
}

/* Sort select styling */
.product-dropdown .product-sort-select {
    min-width: 200px;
}

.product-dropdown .product-sort-select:focus {
    outline: 2px solid #4f46e5;
    outline-offset: 2px;
}

/* Scrollbar styling */
.product-dropdown::-webkit-scrollbar {
    width: 10px;
}

.product-dropdown::-webkit-scrollbar-track {
    background: #f3f4f6;
}

.product-dropdown::-webkit-scrollbar-thumb {
    background: #9ca3af;
    border-radius: 5px;
}

.product-dropdown::-webkit-scrollbar-thumb:hover {
    background: #6b7280;
}

/* Ensure dropdown has relative positioning */
.product-dropdown {
    position: relative;
}
</style>
`;

// Inject enhanced styles on initialization
$(function() {
    if (!$('#enhanced-dropdown-styles').length) {
        $('head').append($('<div id="enhanced-dropdown-styles">').html(enhancedDropdownStyles));
    }
});

// Define dropdown highlight styles
var dropdownHighlightStyles = `
<style>
/* تمييز الصف المحدد بحدود فقط */
.product-dropdown tr.highlighted {
    outline: 3px solid #2563eb;
    outline-offset: -1px;
}

/* أو يمكن استخدام border */
.product-dropdown tr.highlighted td {
    border-top: 3px solid #2563eb !important;
    border-bottom: 3px solid #2563eb !important;
}

.product-dropdown tr.highlighted td:first-child {
    border-left: 3px solid #2563eb !important;
}

.product-dropdown tr.highlighted td:last-child {
    border-right: 3px solid #2563eb !important;
}

/* تحسين hover بسيط */
.product-dropdown tr:hover:not(.highlighted) {
    outline: 2px solid #ddd;
    outline-offset: -1px;
}

/* رسالة التعليمات في الأعلى */
.product-dropdown::before {
    content: "↑↓ Navigate • Enter Select • Esc Close";
    display: block;
    padding: 10px;
    background: #f3f4f6;
    color: #374151;
    font-size: 13px;
    text-align: center;
    font-weight: 600;
    border-bottom: 1px solid #e5e7eb;
}

/* تحسين الانتقالات */
.product-dropdown tr {
    transition: outline 0.2s ease;
}
</style>
`;

// أضف الأنماط عند تحميل الصفحة
$(document).ready(function() {
    $('head').append(dropdownHighlightStyles);
});

// Enhanced highlight function with better visibility
function highlightDropdownRow($rows, index) {
    // Remove all previous highlights
    $rows.removeClass('highlighted').css({
        'background-color': '',
        'outline': ''
    });
    
    if (index >= 0 && index < $rows.length) {
        var $targetRow = $rows.eq(index);
        
        // Add highlighted class and ensure visibility with inline styles
        $targetRow.addClass('highlighted');
        $targetRow.css({
            'background-color': '#e0e7ff',
            'outline': '2px solid #4f46e5'
        });
        
        // Make text more visible on highlighted row
        $targetRow.find('td').css({
            'background-color': 'transparent'
        });
    }
}

// إذا استمرت المشكلة، جرب هذا الحل البديل:
function forceHighlightStyle() {
    // أضف هذا بعد إضافة dropdown للصفحة
    var style = document.createElement('style');
    style.innerHTML = `
        .product-dropdown tr.highlighted {
            background-color: #2563eb !important;
            color: white !important;
        }
        .product-dropdown tr.highlighted td {
            color: white !important;
        }
    `;
    document.head.appendChild(style);
}
function populateRowWithProduct(row, product, rowIndex) {
    row.removeClass('empty-row');
    row.addClass('product_row');
    
    var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
    var selling_price_base = parseFloat(product.selling_price || product.default_sell_price || 0);
    var selling_price_inc_tax = selling_price_base * exchange_rate;
    var default_quantity = 1;
    
    // 1. تحديث خلية المنتج
    var productCell = row.find('td:eq(1)');
    productCell.html(`
        <div class="product-info" style="position: relative;">
            <strong>${product.sku} |${product.name} | ${product.product_custom_field1}</strong>
            ${product.type === 'variable' ? '<br><small class="text-muted">' + product.variation + '</small>' : ''}
            <i class="fa fa-edit" style="position: absolute; right: 5px; top: 50%; transform: translateY(-50%); color: #6b7280; font-size: 12px; opacity: 0.5;"></i>
        </div>
        <input type="hidden" class="product_id" name="products[${rowIndex}][product_id]" value="${product.id || product.product_id}">
        <input type="hidden" class="variation_id row_variation_id" name="products[${rowIndex}][variation_id]" value="${product.variation_id || product.id}">
        <input type="hidden" class="product_type" name="products[${rowIndex}][product_type]" value="${product.type || 'single'}">
        <input type="hidden" class="enable_sr_no" value="${product.enable_sr_no || 0}">
        <input type="hidden" class="enable_stock" value="${product.enable_stock || 0}">
        <input type="hidden" class="product_warehouses" value='${JSON.stringify(product.warehouse_details || [])}'>
        <input type="hidden" class="product_unit_id" name="products[${rowIndex}][product_unit_id]" value="${product.unit_id || ''}">
        <input type="hidden" class="base_unit_multiplier" name="products[${rowIndex}][base_unit_multiplier]" value="1">
        <input type="hidden" class="hidden_base_unit_sell_price" value="${selling_price_base}">
        <input type="hidden" class="modifiers_exist" value="0">
        <input type="hidden" name="products[${rowIndex}][transaction_sell_lines_id]" value="">
        <input type="hidden" name="products[${rowIndex}][lot_no_line_id]" value="">

    `);
    
    // 2. تحديث حقل المستودع
    var warehouseCell = row.find('td:eq(2)');
    var defaultWarehouse = 'W01';
    if (product.warehouse_details && product.warehouse_details.length > 0) {
        var bestWarehouse = product.warehouse_details.reduce((prev, current) => 
            (prev.available || prev.quantity) > (current.available || current.quantity) ? prev : current
        );
        if (bestWarehouse && bestWarehouse.code) {
            defaultWarehouse = bestWarehouse.code;
        }
    }
    
    warehouseCell.html(`
        <input type="text" 
               class="form-control warehouse-input excel-input" 
               name="products[${rowIndex}][warehouse_code]"
               placeholder="W01"
               value="${defaultWarehouse}"
               style="text-align: center;">
        <input type="hidden" 
               class="warehouse_id" 
               name="products[${rowIndex}][warehouse_id]" 
               value="${defaultWarehouse}">
    `);
    
    // 3. تحديث حقل الوحدة - تحويل إلى input بدلاً من dropdown
    var unitCell = row.find('td:eq(3)');
    var unitInputHTML = '';
    
    // معالجة الوحدات الفرعية لتحديد القيمة الافتراضية
    var defaultUnit = 'EA';
    var defaultMultiplier = 1;
    var defaultUnitId = '';
    var availableUnits = [];
    
    // جمع بيانات الوحدات المتاحة
    if (product.processed_units && product.processed_units.length > 0) {
        availableUnits = product.processed_units;
    } else if (product.units && product.units.length > 0) {
        availableUnits = product.units;
    } else if (product.sub_units && product.sub_units.length > 0) {
        availableUnits = product.sub_units;
    }
    
    // البحث عن الوحدة الافتراضية (الوحدة الأساسية)
    if (availableUnits.length > 0) {
        var baseUnit = availableUnits.find(unit => unit.is_base_unit == 1);
        if (baseUnit) {
            defaultUnit = baseUnit.unit_name || baseUnit.name || 'EA';
            defaultMultiplier = baseUnit.multiplier || 1;
            defaultUnitId = baseUnit.id || '';
            
            console.log('🎯 Base unit detected:', {
                unit: defaultUnit,
                multiplier: defaultMultiplier,
                id: defaultUnitId,
                will_use_price_array: true
            });
        } else {
            // إذا لم توجد وحدة أساسية، استخدم الوحدة الأولى
            var firstUnit = availableUnits[0];
            defaultUnit = firstUnit.unit_name || firstUnit.name || 'EA';
            defaultMultiplier = firstUnit.multiplier || 1;
            defaultUnitId = firstUnit.id || '';
        }
    } else if (product.unit) {
        defaultUnit = product.unit;
    }
    
    // تسجيل للتأكد من البيانات
    console.log('Product units data:', {
        product_name: product.name,
        available_units: availableUnits,
        default_unit: defaultUnit,
        default_multiplier: defaultMultiplier,
        default_unit_id: defaultUnitId
    });
    
    // إضافة معلومات إضافية للوحدات لتسهيل التعرف على الكارتون والوحدات الخاصة
    var isCartonBaseUnit = defaultUnit && (
        defaultUnit.toLowerCase().includes('كارتون') || 
        defaultUnit.toLowerCase().includes('كرتون') ||
        defaultUnit.toLowerCase().includes('carton') ||
        defaultUnit.toLowerCase().includes('ctn')
    );
    
    // إنشاء input field للوحدة مع autocomplete
    unitInputHTML = `
        <input type="text" 
               class="form-control unit-input excel-input" 
               name="products[${rowIndex}][unit_name]"
               placeholder="EA"
               value="${defaultUnit}"
               style="text-align: center;"
               data-available-units='${JSON.stringify(availableUnits)}'
               data-is-carton-base="${isCartonBaseUnit}"
               data-default-multiplier="${defaultMultiplier}"
               autocomplete="off"
               list="units-list-${rowIndex}">
               
        <!-- قائمة الوحدات المتاحة للاقتراح -->
        <datalist id="units-list-${rowIndex}">
    `;
    
    // إضافة الوحدات المتاحة كخيارات في datalist
    availableUnits.forEach(function(unit) {
        var displayText = unit.unit_name || unit.name;
        if (unit.multiplier && unit.multiplier > 1) {
            displayText += ' (×' + unit.multiplier + ')';
        } else if (unit.multiplier && unit.multiplier < 1) {
            displayText += ' (÷' + (1/unit.multiplier) + ')';
        }
        
        unitInputHTML += `
            <option value="${unit.unit_name || unit.name}" 
                    data-unit-id="${unit.id}"
                    data-multiplier="${unit.multiplier || 1}"
                    data-allow-decimal="${unit.allow_decimal || 0}"
                    data-is-base-unit="${unit.is_base_unit || 0}">
                ${displayText}
            </option>
        `;
    });
    
    unitInputHTML += `
        </datalist>
        
        <!-- حقول مخفية لبيانات الوحدة المختارة -->
        <input type="hidden" 
               class="sub_unit_id" 
               name="products[${rowIndex}][sub_unit_id]" 
               value="${defaultUnitId}">
        <input type="hidden" 
               class="unit_multiplier" 
               name="products[${rowIndex}][unit_multiplier]" 
               value="${defaultMultiplier}">
        <input type="hidden" 
               class="allow_decimal" 
               value="1">
        <input type="hidden" 
               class="is_base_unit" 
               value="1">
    `;
    
    unitCell.html(unitInputHTML);
    
    // 4. تحديث حقل الكمية
    var qtyCell = row.find('td:eq(4)');
    qtyCell.html(`
        <input type="text" 
               class="form-control pos_quantity excel-input number-input" 
               name="products[${rowIndex}][quantity]"
               value="${default_quantity}"
               data-rule-required="true"
               data-msg-required="Please enter quantity"
               data-qty_available="${product.total_stock || product.qty_available || 0}"
               data-rule-max-value="${product.total_stock || product.qty_available || 999999}"
               data-msg-max-value="Max available: ${product.total_stock || product.qty_available || 999999}"
               data-decimal="${product.unit_allow_decimal || 1}"
               data-allow-overselling="${$('input#is_overselling_allowed').length > 0 ? 1 : 0}">
    `);
    
    // 5. تحديث السعر الأساسي
    row.find('td:eq(5) input').val(formatNumber(selling_price_base, 2));
    
    // 6. تحديث السعر بالدينار
   row.find('td:eq(6)').html(`
        <input type="text" 
               class="form-control excel-input number-input iqd-price-display" 
               readonly 
               value="${formatNumber(convertUSDToIQD(selling_price_base), 0)}"
               style="background-color: #f8f9fa; cursor: not-allowed; text-align: center;">
    `);
    
    // 7. تحديث حقل الخصم
    row.find('td:eq(7)').html(`
        <input type="text" class=" form-control excel-input number-input discount_percent" value="0">
        <select class="row_discount_type hide" name="products[${rowIndex}][line_discount_type]">
            <option value="percentage" selected>Percentage</option>
            <option value="fixed">Fixed</option>
        </select>
        <input type="hidden" class="row_discount_amount" name="products[${rowIndex}][line_discount_amount]" value="0">
    `);
    
    // 8. تحديث السعر شامل الضريبة
    row.find('td:eq(8)').html(`
        <input type="text" 
               class="form-control pos_unit_price_inc_tax excel-input number-input" 
               name="products[${rowIndex}][unit_price_inc_tax]"
               value="${formatNumber(selling_price_inc_tax, 2)}">
        <input type="hidden" 
               class="pos_unit_price" 
               name="products[${rowIndex}][unit_price]"
               value="${selling_price_base}">
        <input type="hidden" 
               class="item_tax" 
               name="products[${rowIndex}][item_tax]" 
               value="0">
    `);
    
    // 9. تحديث المجموع
    var line_total = default_quantity * selling_price_inc_tax;
    row.find('td:eq(9)').html(`
        <input type="hidden" 
               class="pos_line_total" 
               name="products[${rowIndex}][line_total]"
               value="${line_total}">
        <span class="pos_line_total_text">${__currency_trans_from_en(line_total, true)}</span>
    `);
    
    // 10. تحديث معلومات المخزون
    var stockInfo = product.total_stock || product.qty_available || 0;
    row.find('td:eq(10)').html(`<span class="stock-info text-center">${formatNumber(stockInfo, 0)}</span>`);
    
    // إضافة حقل الضريبة المخفي
    if (row.find('.tax_id').length === 0) {
        row.append(`
            <td class="hide">
                <select class="tax_id" name="products[${rowIndex}][tax_id]">
                    <option value="" data-rate="0" selected>No Tax</option>
                </select>
            </td>
        `);
    }
    
    // إضافة معالجات الأحداث
    attachRowEventListeners(row);
    
    // إضافة معالج خاص للـ unit input
    attachUnitInputEventHandlers(row, availableUnits);
    
    // إضافة معالجات للمستودع
    attachUnitWarehouseEventListeners(row);
    
    // تحديث المجاميع
    pos_total_row();
    
    // التركيز على حقل الكمية
    setTimeout(function() {
        row.find('.pos_quantity').focus().select();
    }, 100);
    
    // إضافة صف فارغ جديد إذا كان هذا آخر صف
    if (row.is('#pos_table tbody tr:last-child')) {
        addEmptyProductRow();
    }
}

// دالة معالجة أحداث حقل الوحدة
function attachUnitInputEventHandlers(row, availableUnits) {
    var unitInput = row.find('.unit-input');
    var originalValue = '';
    
    // حفظ القيمة الأصلية عند التركيز
    unitInput.on('focus', function() {
        originalValue = $(this).val();
        restoreUnitValue(row);
        $(this).select();
    });
    
    // معالج الكتابة مع البحث الذكي
    unitInput.on('input', function() {
        var enteredValue = $(this).val().trim().toUpperCase();
        
        if (enteredValue === '') {
            return;
        }
        
        // البحث عن أقرب وحدة
        var matchedUnit = findBestMatchingUnit(enteredValue, availableUnits);
        
        if (matchedUnit) {
            // عرض اقتراحات للمستخدم
            showUnitSuggestion(row, matchedUnit, enteredValue);
        }
    });
    
    // معالج تغيير الوحدة عند انتهاء الكتابة
    unitInput.on('blur', function() {
        var enteredValue = $(this).val().trim().toUpperCase();
        
        if (enteredValue === '') {
            // استعادة القيمة الأصلية إذا كان الحقل فارغ
            $(this).val(originalValue);
            return;
        }
        
        // البحث عن أقرب وحدة مطابقة
        var matchedUnit = findBestMatchingUnit(enteredValue, availableUnits);
        
        if (matchedUnit) {
            // تطبيق الوحدة المطابقة
            applySelectedUnit(row, matchedUnit);
            // تحديث القيمة لتكون الاسم الصحيح
            $(this).val(matchedUnit.unit_name || matchedUnit.name);
        } else {
            // إذا لم توجد وحدة مطابقة، أظهر رسالة تحذير واستعد الوحدة الأساسية
            showUnitValidationMessage(row, enteredValue, availableUnits);
            // استعادة الوحدة الأساسية
            var baseUnit = findBaseUnit(availableUnits);
            if (baseUnit) {
                $(this).val(baseUnit.unit_name || baseUnit.name);
                applySelectedUnit(row, baseUnit);
            }
        }
        
        persistUnitValue(row);
         
    });
    
    // معالج الضغط على Enter
    unitInput.on('keydown', function(e) {
        if (e.which === 13) { // Enter key
            e.preventDefault();
            $(this).blur(); // تشغيل معالج blur
            
            setTimeout(function() {
                row.find('.pos_quantity').focus().select();
            }, 100);
        }
        
        // معالج الضغط على Tab
        if (e.which === 9) { // Tab key
            $(this).blur(); // تشغيل معالج blur
        }
    });
    
    // معالج لإظهار قائمة الوحدات المتاحة عند الضغط على السهم للأسفل
    unitInput.on('keydown', function(e) {
        if (e.which === 40) { // Arrow Down
            e.preventDefault();
            showAvailableUnitsDropdown(row, availableUnits);
        }
    });
    
    // استعادة القيمة المحفوظة إذا وجدت
    restoreUnitValue(row);
      

}


function showAvailableUnitsDropdown(row, availableUnits) {
    $('.unit-dropdown').remove();
    
    if (!availableUnits || availableUnits.length === 0) return;
    
    // ترتيب الوحدات: الأساسية أولاً، ثم الفرعية حسب المضاعف
    var sortedUnits = availableUnits.slice().sort(function(a, b) {
        // الوحدة الأساسية أولاً
        if (a.is_base_unit && !b.is_base_unit) return -1;
        if (!a.is_base_unit && b.is_base_unit) return 1;
        
        // ترتيب حسب المضاعف
        var multA = parseFloat(a.multiplier || 1);
        var multB = parseFloat(b.multiplier || 1);
        return multA - multB;
    });
    
    var dropdownHTML = '<div class="unit-dropdown" style="' +
        'position: absolute;' +
        'background: white;' +
        'border: 2px solid #007bff;' +
        'border-radius: 6px;' +
        'box-shadow: 0 4px 12px rgba(0,0,0,0.15);' +
        'z-index: 1000;' +
        'top: 100%;' +
        'left: 0;' +
        'right: 0;' +
        'max-height: 200px;' +
        'overflow-y: auto;' +
        '">';
    
    // إضافة رأس القائمة
    dropdownHTML += `
        <div style="
            padding: 8px 12px;
            background: #007bff;
            color: white;
            font-size: 11px;
            font-weight: 600;
            text-align: center;
        ">
            اختر الوحدة المناسبة
        </div>
    `;
    
    sortedUnits.forEach(function(unit, index) {
        var unitName = unit.unit_name || unit.name;
        var multiplier = parseFloat(unit.multiplier || 1);
        var isBaseUnit = unit.is_base_unit == 1;
        
        var multiplierText = '';
        var badgeClass = '';
        
        if (isBaseUnit) {
            multiplierText = ' (أساسية)';
            badgeClass = 'background: #28a745; color: white;';
        } else if (multiplier !== 1) {
            if (multiplier < 1) {
                multiplierText = ` (÷${(1/multiplier).toFixed(0)})`;
            } else {
                multiplierText = ` (×${multiplier})`;
            }
            badgeClass = 'background: #17a2b8; color: white;';
        }
        
        dropdownHTML += `
            <div class="unit-option" data-unit-index="${index}" style="
                padding: 10px 12px;
                cursor: pointer;
                border-bottom: 1px solid #eee;
                font-size: 13px;
                display: flex;
                justify-content: space-between;
                align-items: center;
            " onmouseover="this.style.backgroundColor='#f8f9fa'" 
               onmouseout="this.style.backgroundColor='white'">
                <span style="font-weight: 500;">${unitName}</span>
                ${multiplierText ? `<small style="padding: 2px 6px; border-radius: 3px; font-size: 10px; ${badgeClass}">${multiplierText}</small>` : ''}
            </div>
        `;
    });
    
    // إضافة تعليمات في أسفل القائمة
    dropdownHTML += `
        <div style="
            padding: 6px 12px;
            background: #f8f9fa;
            font-size: 10px;
            color: #6c757d;
            text-align: center;
            border-top: 1px solid #e9ecef;
        ">
            استخدم ↑↓ للتنقل، Enter للاختيار، Esc للإغلاق
        </div>
    `;
    
    dropdownHTML += '</div>';
    
    var unitInput = row.find('.unit-input');
    unitInput.parent().css('position', 'relative').append(dropdownHTML);
    
    // معالج النقر على الخيارات
    row.find('.unit-option').on('click', function() {
        var unitIndex = $(this).data('unit-index');
        var selectedUnit = sortedUnits[unitIndex];
        
        unitInput.val(selectedUnit.unit_name || selectedUnit.name);
        applySelectedUnit(row, selectedUnit);
        persistUnitValue(row);
        
        $('.unit-dropdown').remove();
        
        // إضافة تأثير بصري
        addUnitAppliedEffect(row);
        
        // الانتقال للحقل التالي
        setTimeout(function() {
            row.find('.warehouse-input').focus().select();
        }, 100);
    });
    
    // إغلاق القائمة عند النقر خارجها أو الضغط على Escape
    $(document).on('click.unitDropdown keydown.unitDropdown', function(e) {
        if (e.type === 'click' && !$(e.target).closest('.unit-dropdown, .unit-input').length) {
            $('.unit-dropdown').remove();
            $(document).off('click.unitDropdown keydown.unitDropdown');
        } else if (e.type === 'keydown' && e.which === 27) { // Escape
            $('.unit-dropdown').remove();
            $(document).off('click.unitDropdown keydown.unitDropdown');
            unitInput.focus();
        }
    });
}


function findBestMatchingUnit(enteredValue, availableUnits) {
    if (!enteredValue || !availableUnits || availableUnits.length === 0) {
        return null;
    }
    
    enteredValue = enteredValue.toString().trim();
    var enteredValueUpper = enteredValue.toUpperCase();
    var enteredValueLower = enteredValue.toLowerCase();
    
    // الأولوية 1: مطابقة تامة (حساسة وغير حساسة للحالة)
    for (var i = 0; i < availableUnits.length; i++) {
        var unit = availableUnits[i];
        var unitName = (unit.unit_name || unit.name || '').toString();
        
        if (unitName === enteredValue || 
            unitName.toUpperCase() === enteredValueUpper ||
            unitName.toLowerCase() === enteredValueLower) {
            return unit;
        }
    }
    
    // الأولوية 2: يبدأ بنفس النص
    for (var i = 0; i < availableUnits.length; i++) {
        var unit = availableUnits[i];
        var unitName = (unit.unit_name || unit.name || '').toString().toUpperCase();
        if (unitName.startsWith(enteredValueUpper)) {
            return unit;
        }
    }
    
    // الأولوية 3: يحتوي على النص
    for (var i = 0; i < availableUnits.length; i++) {
        var unit = availableUnits[i];
        var unitName = (unit.unit_name || unit.name || '').toString().toUpperCase();
        if (unitName.includes(enteredValueUpper)) {
            return unit;
        }
    }
    
    // الأولوية 4: مطابقة الاختصارات والمرادفات الشائعة
    var unitMappings = {
        // الوحدات الإنجليزية
        'EA': ['EA', 'EACH', 'PCS', 'PIECE', 'PIECES', 'PC', 'واحدة', 'قطعة', 'حبة'],
        'PCS': ['PCS', 'PIECE', 'PIECES', 'EA', 'EACH', 'PC', 'قطعة', 'قطع', 'حبة'],
        'KG': ['KG', 'KILO', 'KILOGRAM', 'KILOGRAMS', 'كيلو', 'كيلوغرام'],
        'GM': ['GM', 'GRAM', 'GRAMS', 'GR', 'غرام', 'غم'],
        'LTR': ['LTR', 'LITER', 'LITRE', 'LITERS', 'LITRES', 'L', 'لتر', 'ليتر'],
        'ML': ['ML', 'MILLILITER', 'MILLILITRE', 'MILLILITERS', 'مل', 'مليلتر'],
        'BOX': ['BOX', 'BOXES', 'CTN', 'CARTON', 'CARTONS', 'صندوق', 'كرتون', 'علبة'],
        'CTN': ['CTN', 'CARTON', 'CARTONS', 'BOX', 'BOXES', 'كرتون', 'صندوق'],
        'DZ': ['DZ', 'DOZEN', 'DOZENS', 'DOZ', 'دزينة', 'دستة'],
        'PACK': ['PACK', 'PACKET', 'PACKETS', 'PKT', 'حزمة', 'باكيت', 'علبة'],
        
        // الوحدات الكسرية
        '½': ['0.5', 'HALF', 'نصف', '1/2', 'نص'],
        '¼': ['0.25', 'QUARTER', 'ربع', '1/4'],
        '⅛': ['0.125', 'EIGHTH', 'ثمن', '1/8'],
        
        // وحدات خاصة
        'فل بلاستك': ['فل بلاستك', 'FULL PLASTIC', 'FL PLASTIC', 'FULLPLASTIC'],
        'FULL PLASTIC': ['FULL PLASTIC', 'فل بلاستك', 'FL PLASTIC', 'FULLPLASTIC']
    };
    
    // البحث في المرادفات
    for (var mainUnit in unitMappings) {
        var synonyms = unitMappings[mainUnit];
        for (var j = 0; j < synonyms.length; j++) {
            if (synonyms[j].toUpperCase() === enteredValueUpper) {
                // البحث عن الوحدة الحقيقية في القائمة المتاحة
                for (var k = 0; k < availableUnits.length; k++) {
                    var unit = availableUnits[k];
                    var unitName = (unit.unit_name || unit.name || '').toString().toUpperCase();
                    if (synonyms.includes(unitName) || unitName === mainUnit) {
                        return unit;
                    }
                }
            }
        }
    }
    
    // الأولوية 5: مطابقة الكسور العشرية مع المضاعفات
    var decimalValue = parseFloat(enteredValue);
    if (!isNaN(decimalValue)) {
        for (var i = 0; i < availableUnits.length; i++) {
            var unit = availableUnits[i];
            var multiplier = parseFloat(unit.multiplier || 1);
            if (Math.abs(multiplier - decimalValue) < 0.001) {
                return unit;
            }
        }
    }
    
    // الأولوية 6: البحث الضبابي (Fuzzy Search) للأخطاء الإملائية
    var bestMatch = null;
    var bestScore = 0;
    
    for (var i = 0; i < availableUnits.length; i++) {
        var unit = availableUnits[i];
        var unitName = (unit.unit_name || unit.name || '').toString().toUpperCase();
        var similarity = calculateStringSimilarity(enteredValueUpper, unitName);
        
        if (similarity > bestScore && similarity > 0.6) { // 60% تشابه على الأقل
            bestScore = similarity;
            bestMatch = unit;
        }
    }
    
    return bestMatch;
}

// دالة حساب التشابه بين النصوص (Levenshtein Distance)
function calculateStringSimilarity(str1, str2) {
    if (str1.length === 0) return str2.length === 0 ? 1 : 0;
    if (str2.length === 0) return 0;
    
    var matrix = [];
    
    // إنشاء المصفوفة
    for (var i = 0; i <= str2.length; i++) {
        matrix[i] = [i];
    }
    
    for (var j = 0; j <= str1.length; j++) {
        matrix[0][j] = j;
    }
    
    // ملء المصفوفة
    for (var i = 1; i <= str2.length; i++) {
        for (var j = 1; j <= str1.length; j++) {
            if (str2.charAt(i - 1) === str1.charAt(j - 1)) {
                matrix[i][j] = matrix[i - 1][j - 1];
            } else {
                matrix[i][j] = Math.min(
                    matrix[i - 1][j - 1] + 1, // substitution
                    matrix[i][j - 1] + 1,     // insertion
                    matrix[i - 1][j] + 1      // deletion
                );
            }
        }
    }
    
    var distance = matrix[str2.length][str1.length];
    var maxLength = Math.max(str1.length, str2.length);
    
    return (maxLength - distance) / maxLength;
}


// دالة تطبيق الوحدة المختارة - مُعاد تفعيلها
window.applySelectedUnit = function(row, unit) {
    if (!row || !unit) {
        console.log('⚠️ applySelectedUnit: Missing row or unit data');
        return;
    }
    
    console.log('🔧 Applying selected unit:', unit.unit_name || unit.name);
    
    // تحديث الحقول المخفية
    row.find('.sub_unit_id').val(unit.id || '');
    row.find('.unit_multiplier').val(unit.multiplier || 1);
    row.find('.allow_decimal').val(unit.allow_decimal || 1);
    row.find('.is_base_unit').val(unit.is_base_unit || 0);

    // تحديث بيانات الإرسال
    if (typeof updateUnitSubmissionData === 'function') {
        updateUnitSubmissionData(row, unit);
    }
    
    // تحديث السعر بناءً على المضاعف مع الإضافات والتعامل الخاص مع السيت والوحدات الخاصة
    var baseSellPrice = parseFloat(row.find('.hidden_base_unit_sell_price').val()) || 0;
    var multiplier = parseFloat(unit.multiplier || 1);
    var unitName = unit.unit_name || unit.name || '';
    var isSetUnit = unitName.toLowerCase().includes('سيت') || unitName.toLowerCase().includes('set');
    var isSpecialUnit = isSpecialBaseUnit(unitName);
    
    var newPrice;
    if (isSetUnit) {
        // للسيت: استخدم السعر الأساسي مباشرة بدون ضرب في المضاعف
        newPrice = baseSellPrice;
        console.log('🎯 Set unit in applySelectedUnit - using base price directly');
    } else if (isSpecialUnit && (unit.is_base_unit == 1 || multiplier === 1)) {
        // للوحدات الخاصة (فل بلاستك، لك، درزن) كوحدة رئيسية: استخدم السعر الأساسي مباشرة
        newPrice = baseSellPrice;
        console.log('🎯 Special base unit (فل بلاستك/لك/درزن) in applySelectedUnit - using base price directly');
    } else {
        // للوحدات الأخرى: استخدم المضاعف كالمعتاد
        newPrice = baseSellPrice * multiplier;
    }
    
    // إضافة المبلغ الإضافي بناءً على المضاعف (حالياً معطل)
    var additionalAmount = 0;
    
    // تطبيق المبلغ الإضافي
    newPrice = newPrice + additionalAmount;
    
    if (newPrice > 0) {
        var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
        var newPriceIncTax = newPrice * exchange_rate;
        
        // تحديث حقول السعر
        try {
            row.find('td:eq(5) input').val(formatNumber(newPrice, 2));
            
            if (typeof __write_number === 'function') {
                __write_number(row.find('.pos_unit_price_inc_tax'), newPriceIncTax);
                __write_number(row.find('.pos_unit_price'), newPrice);
            } else {
                row.find('.pos_unit_price_inc_tax').val(newPriceIncTax);
                row.find('.pos_unit_price').val(newPrice);
            }
            
            // تحديث السعر بالدينار العراقي
            if (typeof convertUSDToIQD === 'function') {
                var iqrPrice = convertUSDToIQD(newPrice);
                row.find('td:eq(6) input').val(formatNumber(iqrPrice, 0));
            }
            
            // تحديث المجموع
            var quantity = parseFloat(row.find('.pos_quantity').val()) || 1;
            var lineTotal = quantity * newPriceIncTax;
            
            if (typeof __write_number === 'function') {
                __write_number(row.find('.pos_line_total'), lineTotal, false);
            } else {
                row.find('.pos_line_total').val(lineTotal);
            }
            
            if (typeof __currency_trans_from_en === 'function') {
                row.find('.pos_line_total_text').text(__currency_trans_from_en(lineTotal, true));
            }
            
            // إعادة حساب المجاميع
            if (typeof pos_total_row === 'function') {
                pos_total_row();
            }
            
            console.log('✅ Unit applied successfully:', {
                unit_name: unit.unit_name || unit.name,
                multiplier: multiplier,
                base_price: baseSellPrice,
                additional_amount: additionalAmount,
                final_price: newPrice
            });
            
        } catch (e) {
            console.log('❌ Error applying unit:', e);
        }
    } else {
        console.log('⚠️ Invalid price calculated for unit');
    }
};

// دالة العثور على الوحدة الأساسية
function findBaseUnit(availableUnits) {
    // البحث عن الوحدة الأساسية
    for (var i = 0; i < availableUnits.length; i++) {
        var unit = availableUnits[i];
        if (unit.is_base_unit == 1 || unit.multiplier == 1) {
            return unit;
        }
    }
    
    // إذا لم توجد، أرجع الوحدة الأولى
    return availableUnits.length > 0 ? availableUnits[0] : null;
}

// دالة عرض اقتراح الوحدة
function showUnitSuggestion(row, matchedUnit, enteredValue) {
    // إزالة الاقتراحات السابقة
    row.find('.unit-suggestion').remove();
    
    var unitName = matchedUnit.unit_name || matchedUnit.name;
    if (unitName.toUpperCase() !== enteredValue.toUpperCase()) {
        var suggestionHTML = `
            <div class="unit-suggestion" style="
                position: absolute;
                background: #17a2b8;
                color: white;
                padding: 4px 8px;
                border-radius: 4px;
                font-size: 11px;
                z-index: 1000;
                top: -25px;
                left: 0;
                white-space: nowrap;
                box-shadow: 0 2px 4px rgba(0,0,0,0.2);
            ">
                هل تقصد: ${unitName}?
            </div>
        `;
        
        row.find('.unit-input').parent().css('position', 'relative').append(suggestionHTML);
        
        // إخفاء الاقتراح بعد 3 ثوان
        setTimeout(function() {
            row.find('.unit-suggestion').fadeOut();
        }, 3000);
    }
}

// دالة عرض رسالة التحقق من صحة الوحدة
function showUnitValidationMessage(row, enteredValue, availableUnits) {
    // إزالة الرسائل السابقة
    row.find('.unit-validation-message').remove();
    
    // إنشاء قائمة بالوحدات المتاحة
    var availableUnitNames = availableUnits.map(function(unit) {
        return unit.unit_name || unit.name;
    }).join(', ');
    
    var messageHTML = `
        <div class="unit-validation-message" style="
            position: absolute;
            background: #dc3545;
            color: white;
            padding: 6px 10px;
            border-radius: 4px;
            font-size: 11px;
            z-index: 1000;
            top: 30px;
            left: 0;
            white-space: nowrap;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
            max-width: 250px;
            white-space: normal;
        ">
            الوحدة "${enteredValue}" غير متاحة.<br>
            الوحدات المتاحة: ${availableUnitNames}
        </div>
    `;
    
    row.find('.unit-input').parent().css('position', 'relative').append(messageHTML);
    
    // إخفاء الرسالة بعد 5 ثوان
    setTimeout(function() {
        row.find('.unit-validation-message').fadeOut(function() {
            $(this).remove();
        });
    }, 5000);
    
    // عرض toastr كذلك
    toastr.warning(`الوحدة "${enteredValue}" غير متاحة. سيتم استخدام الوحدة الأساسية.`);
}

// دالة للتحقق من صحة الوحدة والتعامل مع الإدخال الخاطئ
function validateAndCorrectUnitInput(row, enteredValue, availableUnits) {
    // البحث عن أقرب وحدة مطابقة
    var matchedUnit = findBestMatchingUnit(enteredValue, availableUnits);
    
    if (matchedUnit) {
        // تطبيق الوحدة المطابقة
        return {
            isValid: true,
            correctedUnit: matchedUnit,
            originalInput: enteredValue,
            correctedValue: matchedUnit.unit_name || matchedUnit.name
        };
    } else {
        // الوحدة غير صحيحة، العودة للوحدة الأساسية
        var baseUnit = findBaseUnit(availableUnits);
        return {
            isValid: false,
            correctedUnit: baseUnit,
            originalInput: enteredValue,
            correctedValue: baseUnit ? (baseUnit.unit_name || baseUnit.name) : 'EA',
            errorMessage: `الوحدة "${enteredValue}" غير متاحة`
        };
    }
}



function persistUnitValue(row) {
    var unitInput = row.find('.unit-input');
    var subUnitId = row.find('.sub_unit_id').val();
    var multiplier = row.find('.unit_multiplier').val();
    var unitName = unitInput.val();
    
    // Store all unit data as a JSON string in a data attribute
    var unitData = {
        unit_name: unitName,
        sub_unit_id: subUnitId,
        multiplier: multiplier
    };
    
    row.data('persisted-unit', JSON.stringify(unitData));
    
    // Double-check that form fields are properly named
    var rowIndex = row.data('row_index') || row.index();
    row.find('.sub_unit_id').attr('name', 'products[' + rowIndex + '][sub_unit_id]');
    row.find('.unit_multiplier').attr('name', 'products[' + rowIndex + '][unit_multiplier]');
    
    // Add missing hidden field for unit name if it doesn't exist
    if (row.find('input[name="products[' + rowIndex + '][unit_name]"]').length === 0) {
        row.append('<input type="hidden" name="products[' + rowIndex + '][unit_name]" value="' + unitName + '">');
    } else {
        row.find('input[name="products[' + rowIndex + '][unit_name]"]').val(unitName);
    }

     updateUnitSubmissionData(row, unit);

}

// دالة لاستعادة قيمة الوحدة
function restoreUnitValue(row) {
    var unitInfo = row.data('unit-info');
    
    if (unitInfo && unitInfo.value) {
        // استعادة القيمة المرئية
        row.find('.unit-input').val(unitInfo.value);
        
        // استعادة الحقول المخفية
        row.find('.sub_unit_id').val(unitInfo.id);
        row.find('.unit_multiplier').val(unitInfo.multiplier);
        row.find('.allow_decimal').val(unitInfo.allow_decimal);
        row.find('.is_base_unit').val(unitInfo.is_base_unit);
    }
}
// 5. دالة مساعدة لتنسيق الأرقام
function formatNumber(number, decimals = 2) {
    return parseFloat(number || 0).toFixed(decimals);
}
function processProductUnits(product) {
    // استخراج الوحدات الفرعية من بيانات المنتج
    var units = [];
    
    if (product.product_locations && product.product_locations.length > 0) {
        var location = product.product_locations[0];
        
        // الوحدة الأساسية
        if (product.unit) {
            units.push({
                id: product.unit_id || '',
                unit_name: product.unit,
                multiplier: 1,
                is_base_unit: 1,
                allow_decimal: product.unit_allow_decimal || 1
            });
        }
        
        // الوحدات الفرعية
        if (location.sub_units && location.sub_units.length > 0) {
            location.sub_units.forEach(function(subUnit) {
                units.push({
                    id: subUnit.id,
                    unit_name: subUnit.name || subUnit.unit_name,
                    multiplier: parseFloat(subUnit.multiplier) || 1,
                    is_base_unit: 0,
                    allow_decimal: subUnit.allow_decimal || 0
                });
            });
        }
    }
    
    // إضافة الوحدات للمنتج
    product.units = units;
}

// إضافة أنماط CSS للحقول الجديدة
var inputFieldStyles = `
<style>
/* أنماط حقول الكتابة للوحدة والمستودع */
.warehouse-input, .unit-input {
    width: 100%;
    height: 28px;
    border: none;
    outline: none;
    padding: 2px 6px;
    font-size: 12px;
    background: transparent;
    text-align: center;
    text-transform: uppercase;
}

.warehouse-input:focus, .unit-input:focus {
    background: white;
    border: 2px solid #2563eb;
    box-shadow: 0 0 0 2px rgba(37, 99, 235, 0.1);
}

.warehouse-input.error-input, .unit-input.error-input {
    border: 2px solid #dc3545 !important;
    background-color: #f8d7da !important;
}

.warehouse-input::placeholder, .unit-input::placeholder {
    color: #9ca3af;
    font-style: italic;
    text-transform: uppercase;
}

/* تلميحات الأدوات للوحدات */
.unit-hint {
    position: absolute;
    background: #333;
    color: white;
    padding: 5px 10px;
    border-radius: 4px;
    font-size: 11px;
    white-space: nowrap;
    z-index: 1000;
    top: -30px;
    left: 50%;
    transform: translateX(-50%);
    display: none;
}

.unit-input:focus + .unit-hint {
    display: block;
}

/* قائمة اقتراحات الوحدات */
.unit-suggestions {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: white;
    border: 1px solid #d1d5db;
    border-top: none;
    border-radius: 0 0 4px 4px;
    max-height: 150px;
    overflow-y: auto;
    z-index: 100;
    display: none;
}

.unit-suggestions.show {
    display: block;
}

.unit-suggestion-item {
    padding: 4px 8px;
    cursor: pointer;
    font-size: 12px;
}

.unit-suggestion-item:hover {
    background: #e5e7eb;
}

.unit-suggestion-item.selected {
    background: #3b82f6;
    color: white;
}
</style>
`;

// إضافة الأنماط عند تحميل الصفحة
$(document).ready(function() {
    $('head').append(inputFieldStyles);
});

// دالة لإضافة اقتراحات الوحدات (اختياري)
function addUnitSuggestions(inputElement) {
    var commonUnits = ['EA', 'PCS', 'BOX', 'CTN', 'DZ', 'PACK', 'KG', 'GM', 'LTR', 'ML'];
    var suggestionsDiv = $('<div class="unit-suggestions"></div>');
    
    commonUnits.forEach(function(unit) {
        var item = $('<div class="unit-suggestion-item">' + unit + '</div>');
        item.on('click', function() {
            inputElement.val(unit).trigger('change');
            suggestionsDiv.removeClass('show');
        });
        suggestionsDiv.append(item);
    });
    
    inputElement.after(suggestionsDiv);
    
    // إظهار/إخفاء الاقتراحات
    inputElement.on('focus', function() {
        suggestionsDiv.addClass('show');
    });
    
    inputElement.on('blur', function() {
        setTimeout(function() {
            suggestionsDiv.removeClass('show');
        }, 200);
    });
    
    // البحث في الاقتراحات
    inputElement.on('input', function() {
        var searchTerm = $(this).val().toUpperCase();
        suggestionsDiv.find('.unit-suggestion-item').each(function() {
            var unit = $(this).text();
            if (unit.startsWith(searchTerm)) {
                $(this).show();
            } else {
                $(this).hide();
            }
        });
    });
}

// دالة لتحديث السعر بناءً على الوحدة
// function updatePriceBasedOnUnit(row, multiplier) {
//     var basePrice = parseFloat(row.find('.hidden_base_unit_sell_price').val()) || 0;
//     var newPrice = basePrice * multiplier;
    
//     __write_number(row.find('.pos_unit_price'), newPrice);
//     row.find('.pos_unit_price').trigger('change');
// }

// دالة لتحديث معلومات المخزون للمستودع المحدد
function updateStockInfoForWarehouse(row, warehouseCode) {
    var productWarehouses = row.find('.product_warehouses').val();
    var stockCell = row.find('.stock-info');
    
    if (productWarehouses) {
        try {
            var warehouses = JSON.parse(productWarehouses);
            var warehouse = warehouses.find(w => w.code === warehouseCode);
            
            if (warehouse) {
                var qty = warehouse.available || warehouse.quantity || 0;
                var stockClass = qty > 10 ? 'stock-available' : (qty > 0 ? 'stock-warning' : 'stock-error');
                
                stockCell.html(formatNumber(qty, 0))
                        .removeClass('stock-available stock-warning stock-error')
                        .addClass(stockClass);
                
                // تحديث قيود الكمية
                var qtyInput = row.find('.pos_quantity');
                qtyInput.attr('data-qty_available', qty);
                qtyInput.attr('data-rule-max-value', qty);
                qtyInput.attr('data-msg-max-value', 'Max available in ' + warehouseCode + ': ' + qty);
            } else {
                stockCell.html('0').addClass('stock-error');
            }
        } catch (e) {
            console.error('Error parsing warehouse data:', e);
        }
    }
}


function updateStockInfoForAllLocations(row, product) {
    var stockCell = row.find('td:eq(10) .stock-info');
    if (!stockCell.length) {
        stockCell = row.find('td:eq(10)').html('<span class="stock-info text-center">-</span>').find('.stock-info');
    }
    
    var warehouseSelect = row.find('.warehouse-selector');
    var quantityInput = row.find('.pos_quantity');
    
    function updateStockDisplay() {
        var selectedWarehouse = warehouseSelect.val();
        var stockInfo = '';
        var stockClass = 'stock-available';
        var availableQty = 0;
        
        
        
        stockCell.html(stockInfo)
                .removeClass('stock-available stock-warning stock-error')
                .addClass(stockClass);
    }
    
    warehouseSelect.off('change.stock').on('change.stock', updateStockDisplay);
    updateStockDisplay();
}
function setupWarehouseChangeHandlerForAllLocations(row, product) {
    var warehouseSelect = row.find('.warehouse-selector');
    var quantityInput = row.find('.pos_quantity');
    
    warehouseSelect.on('change', function() {
        var selectedWarehouse = $(this).val();
        
        // تحديث معلومات المخزون
        updateStockInfoForAllLocations(row, product);
        
        // التحقق من الكمية
        var requestedQty = parseFloat(quantityInput.val()) || 0;
        var availableQty = parseFloat(quantityInput.attr('data-qty_available')) || 0;
        
        if (requestedQty > availableQty && availableQty > 0) {
            var locationName = selectedWarehouse ? 
                warehouseSelect.find('option:selected').text().split(' - ')[1].split(' (')[0] : 
                'All Locations';
            
            toastr.warning(`Requested quantity (${requestedQty}) exceeds available stock (${availableQty}) in ${locationName}`);
            
            if (!$('input#is_overselling_allowed').length) {
                quantityInput.val(availableQty).trigger('change');
            }
        }
        
        // إظهار تلميح بالمواقع الأخرى إذا كان المخزون منخفض
        if (selectedWarehouse && availableQty < requestedQty && product.warehouse_details) {
            var otherLocations = product.warehouse_details.filter(loc => 
                loc.id != selectedWarehouse && (loc.available || loc.quantity) >= requestedQty
            );
            
            if (otherLocations.length > 0) {
                var suggestions = otherLocations.map(loc => 
                    loc.name + ' (' + formatNumber(loc.available || loc.quantity, 0) + ')'
                ).join(', ');
                
                toastr.info('Available in other locations: ' + suggestions, '', {
                    timeOut: 5000,
                    extendedTimeOut: 2000
                });
            }
        }
    });
}

// إضافة أنماط CSS محسنة
var enhancedStyles = `
<style>
/* تحسينات للبحث في جميع المواقع */
.product-dropdown {
    backdrop-filter: blur(5px);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.product-search-table tr.highlighted {
    background-color: #dbeafe !important;
    box-shadow: 0 2px 8px rgba(59, 130, 246, 0.15);
}

.product-search-table tr.out-of-stock {
    background-color: #fee2e2 !important;
}

.locations-grid {
    max-width: 100%;
    padding: 4px;
}

.location-badge {
    display: inline-flex;
    align-items: center;
    white-space: nowrap;
    transition: all 0.2s;
}

.location-badge:hover {
    transform: scale(1.05);
    box-shadow: 0 2px 4px rgba(0,0,0,0.2);
}

.badge-success {
    background-color: #10b981;
    color: white;
}

.badge-warning {
    background-color: #f59e0b;
    color: white;
}

.badge-danger {
    background-color: #ef4444;
    color: white;
}

.badge-info {
    background-color: #3b82f6;
    color: white;
}

.warehouse-selector {
    font-size: 11px;
    padding: 2px 6px;
    height: 28px;
}

.warehouse-selector option {
    padding: 4px 8px;
    line-height: 1.4;
}

.warehouse-selector option:hover {
    background-color: #e0e7ff;
}

.product-stock-summary {
    margin-top: 4px;
    padding: 3px 0;
    border-top: 1px dashed #e5e7eb;
}

/* تحسين مؤشرات المخزون */
.stock-available {
    color: #059669;
    font-weight: 600;
}

.stock-warning {
    color: #d97706;
    font-weight: 600;
}

.stock-error {
    color: #dc2626;
    font-weight: 600;
}

.stock-info {
    line-height: 1.3;
    padding: 4px;
}

.stock-info small {
    font-size: 9px;
    opacity: 0.8;
    display: block;
}

/* تحسين الجدول الرئيسي */
#pos_table .product-info {
    line-height: 1.4;
}

#pos_table .badge {
    padding: 2px 6px;
    font-size: 10px;
    font-weight: 600;
}

/* رسائل التنبيه المحسنة */
.toast-info {
    background-color: #3b82f6 !important;
}

.toast-warning {
    background-color: #f59e0b !important;
}

/* تحسين التحميل */
.cell-loading {
    position: relative;
    opacity: 0.7;
}

.cell-loading::after {
    content: '';
    position: absolute;
    right: 8px;
    top: 50%;
    transform: translateY(-50%);
    width: 16px;
    height: 16px;
    border: 2px solid #e5e7eb;
    border-top-color: #3b82f6;
    border-radius: 50%;
    animation: spin 0.8s linear infinite;
}

/* تحسين placeholder */
.product-search-input::placeholder {
    color: #9ca3af;
    font-style: italic;
}

.product-search-input:focus::placeholder {
    color: #60a5fa;
}
</style>
`;

$(document).ready(function() {
    $('head').append(enhancedStyles);
    
    // تحديث نص placeholder للبحث
    $('.product-search-input').attr('placeholder', 'Search products across all locations...');
});

function updateStockInfo(row, product) {
    var stockCell = row.find('td:eq(10) .stock-info');
    if (!stockCell.length) {
        stockCell = row.find('td:eq(10)').html('<span class="stock-info text-center">-</span>').find('.stock-info');
    }
    
    var warehouseSelect = row.find('.warehouse-selector');
    var quantityInput = row.find('.pos_quantity');
    
    function updateStockDisplay() {
        var selectedWarehouse = warehouseSelect.val();
        var stockInfo = '';
        var stockClass = 'stock-available';
        var availableQty = 0;
        
        if (product.enable_stock == 1) {
            if (selectedWarehouse && product.warehouse_details) {
                // البحث عن المستودع المحدد
                var warehouse = product.warehouse_details.find(wh => wh.id == selectedWarehouse);
                if (warehouse) {
                    availableQty = warehouse.available || warehouse.quantity || 0;
                    stockInfo = formatNumber(availableQty, 0);
                    
                    if (availableQty <= 0) {
                        stockClass = 'stock-error';
                        stockInfo += ' <small>(Out)</small>';
                    } else if (availableQty <= 5) {
                        stockClass = 'stock-warning';
                        stockInfo += ' <small>(Low)</small>';
                    } else {
                        stockClass = 'stock-available';
                    }
                } else {
                    stockInfo = '0 <small>(N/A)</small>';
                    stockClass = 'stock-error';
                }
            } else {
                // عرض إجمالي المخزون
                availableQty = product.total_stock || 0;
                stockInfo = formatNumber(availableQty, 0);
                
                if (availableQty <= 0) {
                    stockClass = 'stock-error';
                    stockInfo += ' <small>(Total - Out)</small>';
                } else if (availableQty <= 10) {
                    stockClass = 'stock-warning';
                    stockInfo += ' <small>(Total - Low)</small>';
                } else {
                    stockClass = 'stock-available';
                    stockInfo += ' <small>(Total)</small>';
                }
            }
            
            // تحديث قيود الكمية
          
            
            // إضافة قاعدة التحقق إذا لزم الأمر
           
        } else {
            stockInfo = '<span class="text-muted">N/A</span>';
        }
        
        stockCell.html(stockInfo)
                .removeClass('stock-available stock-warning stock-error')
                .addClass(stockClass);
    }
    
    // تحديث العرض عند تغيير المستودع
    warehouseSelect.off('change.stock').on('change.stock', updateStockDisplay);
    
    // تحديث العرض الأولي
    updateStockDisplay();
}


function setupWarehouseChangeHandler(row, product) {
    var warehouseSelect = row.find('.warehouse-selector');
    var quantityInput = row.find('.pos_quantity');
    
    warehouseSelect.on('change', function() {
        var selectedWarehouse = $(this).val();
        
        // تحديث معلومات المخزون
        updateStockInfo(row, product);
        
        // إظهار تنبيه إذا كانت الكمية المطلوبة أكبر من المتوفرة
        var requestedQty = parseFloat(quantityInput.val()) || 0;
        var availableQty = parseFloat(quantityInput.attr('data-qty_available')) || 0;
        
        if (requestedQty > availableQty && availableQty > 0) {
            toastr.warning(`Requested quantity (${requestedQty}) exceeds available stock (${availableQty}) in selected warehouse`);
            
            // تعيين الكمية للحد الأقصى المتاح
            if (!$('input#is_overselling_allowed').length) {
                quantityInput.val(availableQty).trigger('change');
            }
        }
    });
}

// دالة مساعدة لتنسيق الأرقام
function formatNumber(number, decimals = 2) {
    return parseFloat(number || 0).toFixed(decimals);
}

// إضافة أنماط CSS إضافية
var additionalStyles = `
<style>
/* تحسين عرض معلومات المخازن */
.warehouse-details {
    max-height: 100px;
    overflow-y: auto;
    padding: 4px;
    background: #f8f9fa;
    border-radius: 4px;
}

.warehouse-item {
    padding: 2px 4px;
    border-bottom: 1px solid #e9ecef;
}

.warehouse-item:last-child {
    border-bottom: none;
}

.warehouse-selector {
    font-size: 11px;
    padding: 2px 4px;
}

.warehouse-selector option {
    padding: 4px;
}

.product-stock-info {
    margin-top: 2px;
    font-style: italic;
}

/* تحسين dropdown */
.product-dropdown {
    max-width: 95vw;
    overflow: hidden;
}

.product-search-table {
    font-size: 12px;
}

.product-search-table th {
    background: linear-gradient(to bottom, #4a5568, #2d3748) !important;
    color: white !important;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    border: 1px solid #1a202c !important;
}

.product-search-table td {
    vertical-align: top;
    border: 1px solid #e2e8f0;
}

.product-search-table tr:hover {
    background-color: #edf2f7 !important;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.product-search-table .out-of-stock {
    background-color: #fed7d7 !important;
}

.product-search-table .out-of-stock:hover {
    background-color: #fc8181 !important;
}

/* تحسين مؤشرات المخزون */
.stock-available {
    color: #22c55e;
    font-weight: 600;
}

.stock-warning {
    color: #f59e0b;
    font-weight: 600;
}

.stock-error {
    color: #ef4444;
    font-weight: 600;
}

.stock-info small {
    font-size: 9px;
    opacity: 0.8;
    font-weight: normal;
}

/* تحسين التحميل */
.cell-loading::after {
    content: '';
    position: absolute;
    right: 4px;
    top: 50%;
    transform: translateY(-50%);
    width: 14px;
    height: 14px;
    border: 2px solid #cbd5e0;
    border-top-color: #3b82f6;
    border-radius: 50%;
    animation: spin 0.8s linear infinite;
}

@keyframes spin {
    to {
        transform: translateY(-50%) rotate(360deg);
    }
}

/* تحسين تخطيط الجدول */
#pos_table {
    table-layout: fixed;
}

#pos_table th,
#pos_table td {
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

#pos_table td.product-info-cell {
    white-space: normal;
}

/* تحسين الصفوف الفارغة */
.empty-row {
    background-color: #fffbeb !important;
}

.empty-row:hover {
    background-color: #fef3c7 !important;
}

.empty-row td {
    border-style: dashed !important;
    border-color: #fbbf24 !important;
}
</style>
`;

// إضافة الأنماط عند تحميل الصفحة
$(document).ready(function() {
    $('head').append(additionalStyles);
});

// دالة مساعدة لإعداد أحداث القائمة المنسدلة
function setupDropdownEvents(dropdown, tbody, currentIndex, row, rowIndex) {
    // معالج لوحة المفاتيح
    $(document).on('keydown.product-dropdown', function(e) {
        var $rows = tbody.find('tr[data-index]');
        var maxIndex = $rows.length - 1;
        
        switch(e.which) {
            case 38: // سهم لأعلى
                e.preventDefault();
                currentIndex = currentIndex <= 0 ? maxIndex : currentIndex - 1;
                highlightRow($rows, currentIndex);
                break;
                
            case 40: // سهم لأسفل
                e.preventDefault();
                currentIndex = currentIndex >= maxIndex ? 0 : currentIndex + 1;
                highlightRow($rows, currentIndex);
                break;
                
            case 13: // Enter
                e.preventDefault();
                if (currentIndex >= 0 && currentIndex <= maxIndex) {
                    $rows.eq(currentIndex).trigger('click');
                }
                break;
                
            case 27: // Escape
                e.preventDefault();
                dropdown.remove();
                $(document).off('click.product-dropdown keydown.product-dropdown');
                $(window).off('resize.product-dropdown');
                row.find('.product-search-input').focus();
                break;
        }
    });
    
    // معالج النقر خارج القائمة
    $(document).on('click.product-dropdown', function(e) {
        if (!$(e.target).closest('.product-dropdown, .product-search-input').length) {
            dropdown.remove();
            $(document).off('click.product-dropdown keydown.product-dropdown');
            $(window).off('resize.product-dropdown');
        }
    });
    
    // معالج تغيير حجم النافذة
    $(window).on('resize.product-dropdown', function() {
        var inputOffset = row.find('.product-search-input').offset();
        dropdown.css({
            top: inputOffset.top + row.find('.product-search-input').outerHeight(),
            left: '20px',
            right: '20px'
        });
    });
}

// دالة لتمييز الصف المحدد
function highlightRow($rows, index) {
    $rows.removeClass('highlighted selected').css('background-color', '');
    
    if (index >= 0 && index < $rows.length) {
        var $targetRow = $rows.eq(index);
        $targetRow.addClass('highlighted').css('background-color', '#e0f2fe');
        
        // التمرير إذا لزم الأمر
        var dropdown = $targetRow.closest('.product-dropdown');
        var dropdownHeight = dropdown.height();
        var rowTop = $targetRow.position().top;
        var rowHeight = $targetRow.outerHeight();
        var scrollTop = dropdown.scrollTop();
        
        if (rowTop < 0) {
            dropdown.scrollTop(scrollTop + rowTop);
        } else if (rowTop + rowHeight > dropdownHeight) {
            dropdown.scrollTop(scrollTop + rowTop + rowHeight - dropdownHeight);
        }
    }
}


// Clear row data
function clearRowData(row) {
    row.find('input[type="text"]').not('.product-search-input').val('');
    row.find('select').prop('disabled', true);
    row.find('.product_id, .variation_id').val('');
}

// Handle quantity batch input (multiple products at once)
function handleQuantityBatchInput(e) {
    var input = $(this);
    var value = input.val().trim();
    var row = input.closest('tr');
    
    // Check if it's a batch input (contains newlines or multiple lines)
    if (value.includes('\n') || value.includes('\t')) {
        e.preventDefault();
        processBatchInput(value, row);
        input.val('');
    }
}

// Process batch input (paste multiple products/quantities)
function processBatchInput(batchData, startRow) {
    var lines = batchData.split('\n').filter(line => line.trim());
    var currentRowIndex = parseInt(startRow.data('row_index'));
    
    $('.paste-indicator').show();
    
    lines.forEach(function(line, index) {
        var parts = line.split('\t').map(part => part.trim());
        var targetRow = $('#pos_table tbody tr').eq(currentRowIndex + index);
        
        if (targetRow.length === 0) {
            addEmptyProductRow();
            targetRow = $('#pos_table tbody tr').last();
        }
        
        if (parts.length === 1) {
            // Single value - could be product name or quantity
            if (isNaN(parts[0])) {
                // It's a product name
                targetRow.find('.product-search-input').val(parts[0]).trigger('input');
            } else {
                // It's a quantity
                targetRow.find('.quantity-batch-input, .excel-input.number-input').first().val(parts[0]);
            }
        } else if (parts.length >= 2) {
            // Multiple values - product name and quantity
            targetRow.find('.product-search-input').val(parts[0]).trigger('input');
            setTimeout(function() {
                targetRow.find('.quantity-batch-input, .excel-input.number-input').first().val(parts[1]);
            }, 500);
        }
    });
    
    setTimeout(function() {
        $('.paste-indicator').hide();
        updateTotals();
    }, 1000);
}

// Handle global paste event
function handleGlobalPaste(e) {
    var target = $(e.target);
    
    // Only handle paste in POS table
    if (!target.closest('#pos_table').length) return;
    
    var clipboardData = e.originalEvent.clipboardData || window.clipboardData;
    var pastedData = clipboardData.getData('text/plain');
    
    if (pastedData && (pastedData.includes('\n') || pastedData.includes('\t'))) {
        e.preventDefault();
        var currentRow = target.closest('tr');
        processBatchInput(pastedData, currentRow);
    }
}

// Handle keyboard navigation
function handleKeyboardNavigation(e) {
    var current = $(this);
    var row = current.closest('tr');
    var cell = current.closest('td');
    var cellIndex = cell.index();
    var rowIndex = row.index();
    var rows = $('#pos_table tbody tr');
    var isLastRow = rowIndex === rows.length - 1;
    
    // Add Ctrl+K shortcut for deleting selected row
    if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
        e.preventDefault();
        deleteSelectedProductRow();
        return;
    }
    
    switch(e.key) {
        case 'Tab':
            // السماح بالسلوك الافتراضي للـ Tab مع تحسينات
            if (!e.shiftKey && isLastCell(current) && isLastRow) {
                e.preventDefault();
                // الانتقال لحقل البحث عند الوصول لآخر خلية في آخر صف
                $('#search_product').focus().select();
            }
            break;
            
        case 'Enter':
            e.preventDefault();
            if (current.hasClass('product-search-input')) {
                // لا تضف المنتج تلقائياً، انتظر الاختيار الصريح
                return;
            }
            navigateToNextRow(current);
            break;
            
        case 'ArrowUp':
            e.preventDefault();
            navigateVertically(current, -1);
            break;
            
        case 'ArrowDown':
            e.preventDefault();
            // معالجة خاصة للصف الأخير
            if (isLastRow && !row.hasClass('empty-row')) {
                // إذا كنا في الصف الأخير وليس فارغاً، انتقل لحقل البحث
                $('#search_product').focus().select();
            } else {
                navigateVertically(current, 1);
            }
            break;
            
        case 'ArrowLeft':
            if (current.get(0).selectionStart === 0 || current.is('select')) {
                e.preventDefault();
                navigateHorizontally(current, -1);
            }
            break;
            
        case 'ArrowRight':
            if (current.get(0).selectionStart === current.val().length || current.is('select')) {
                e.preventDefault();
                navigateHorizontally(current, 1);
            }
            break;
            
        case 'Delete':
            if (e.ctrlKey) {
                e.preventDefault();
                deleteCurrentRow(current);
            }
            break;
    }
}

// ============================================
// التعديل 2: أضف هذه الدوال المساعدة بعد دالة handleKeyboardNavigation
// ============================================

// دالة للتحقق من آخر خلية قابلة للتعديل
function isLastCell(current) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible:not([readonly]), select:visible:not([disabled])');
    var currentIndex = editableInputs.index(current);
    return currentIndex === editableInputs.length - 1;
}

// تحسين التنقل العمودي
function navigateVertically(current, direction) {
    var row = current.closest('tr');
    var cellIndex = current.closest('td').index();
    var targetRowIndex = row.index() + direction;
    var targetRow = $('#pos_table tbody tr').eq(targetRowIndex);
    
    if (targetRow.length) {
        var targetCell = targetRow.find('td').eq(cellIndex);
        var targetInput = targetCell.find('input:visible:not([readonly]), select:visible:not([disabled])').first();
        
        if (targetInput.length) {
            targetInput.focus();
            if (targetInput.is('input[type="text"], input[type="number"]')) {
                targetInput.select();
            }
        }
    }
}

// تحسين التنقل الأفقي
function navigateHorizontally(current, direction) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible:not([readonly]), select:visible:not([disabled])');
    var currentIndex = editableInputs.index(current);
    var targetIndex = currentIndex + direction;
    
    if (targetIndex >= 0 && targetIndex < editableInputs.length) {
        var targetInput = editableInputs.eq(targetIndex);
        targetInput.focus();
        if (targetInput.is('input[type="text"], input[type="number"]')) {
            targetInput.select();
        }
    }
}


// Navigation helper functions
function navigateToNextCell(current, reverse = false) {
    var cells = $('#pos_table tbody').find('input:visible, select:visible');
    var currentIndex = cells.index(current);
    var nextIndex = reverse ? currentIndex - 1 : currentIndex + 1;
    
    if (nextIndex >= 0 && nextIndex < cells.length) {
        cells.eq(nextIndex).focus().select();
    }
}

function navigateToNextRow(current) {
    var row = current.closest('tr');
    var cellIndex = current.closest('td').index();
    var nextRow = row.next();
    
    if (nextRow.length === 0) {
        addEmptyProductRow();
        nextRow = $('#pos_table tbody tr').last();
    }
    
    var nextCell = nextRow.find('td').eq(cellIndex).find('input:visible, select:visible').first();
    if (nextCell.length) {
        nextCell.focus().select();
    }
}

function navigateToRow(current, targetRowIndex) {
    var cellIndex = current.closest('td').index();
    var targetRow = $('#pos_table tbody tr').eq(targetRowIndex);
    
    if (targetRow.length) {
        var targetCell = targetRow.find('td').eq(cellIndex).find('input:visible, select:visible').first();
        if (targetCell.length) {
            targetCell.focus().select();
        }
    }
}

function navigateToCell(current, targetCellIndex) {
    var row = current.closest('tr');
    var targetCell = row.find('td').eq(targetCellIndex).find('input:visible, select:visible').first();
    
    if (targetCell.length) {
        targetCell.focus().select();
    }
}

// Delete current row
function deleteCurrentRow(current) {
    var row = current.closest('tr');
    if (!row.hasClass('empty-row')) {
        row.remove();
        updateSerialNumbers();
        updateTotals();
    }
}

// Delete selected product row (Ctrl+K)
function deleteSelectedProductRow() {
    // Check if we're in a POS table row
    var focusedElement = $(document.activeElement);
    var currentRow = focusedElement.closest('#pos_table tbody tr');
    
    if (currentRow.length > 0 && !currentRow.hasClass('empty-row')) {
        // Show confirmation dialog
        swal({
            title: LANG.sure,
            text: LANG.delete_product_confirm || "Are you sure you want to delete this product?",
            icon: "warning",
            buttons: true,
            dangerMode: true,
        }).then((willDelete) => {
            if (willDelete) {
                currentRow.remove();
                updateSerialNumbers();
                pos_total_row();
                toastr.success(LANG.deleted_success || 'Product deleted successfully');
                
                // Focus on the next row or search field
                var nextRow = $('#pos_table tbody tr').not('.empty-row').first();
                if (nextRow.length > 0) {
                    nextRow.find('input').first().focus();
                } else {
                    $('#search_product').focus();
                }
            }
        });
    }
}

// Handle click on product row to edit
function handleProductRowClick(clickedCell, event) {
    // Don't trigger if clicking on input elements or buttons
    if ($(event.target).is('input, select, button, .remove-row, .row-checkbox')) {
        return;
    }
    
    var row = clickedCell.closest('tr');
    var cellIndex = clickedCell.index();
    
    // Only handle click on product name column (column 1)
    if (cellIndex === 1 && !row.hasClass('empty-row')) {
        // Get current product info
        var productName = row.find('.product-info strong').text();
        var productId = row.find('.product_id').val();
        var variationId = row.find('.variation_id').val();
        var rowIndex = row.attr('data-row_index') || row.index();
        
        // Show product search modal or convert to search input
        showProductEditModal(row, rowIndex);
    }
}

// Show product edit modal
function showProductEditModal(row, rowIndex) {
    // Create a temporary search input in the product cell
    var productCell = row.find('td:eq(1)');
    var originalContent = productCell.html();
    
    // Create search input
    var searchInput = $('<input>')
        .attr({
            type: 'text',
            class: 'form-control product-edit-search',
            placeholder: 'Search for another product...',
            'data-row-index': rowIndex
        })
        .css({
            width: '100%',
            border: '2px solid #007bff',
            borderRadius: '4px'
        });
    
    // Replace cell content with search input
    productCell.html('').append(searchInput);
    searchInput.focus();
    
    // Handle search
    var searchTimeout;
    searchInput.on('input', function() {
        clearTimeout(searchTimeout);
        var searchTerm = $(this).val();
        
        if (searchTerm.length >= 2) {
            searchTimeout = setTimeout(function() {
                searchProductsForEdit(searchTerm, row, rowIndex, originalContent);
            }, 300);
        }
    });
    
    // Handle escape key to cancel edit
    searchInput.on('keydown', function(e) {
        if (e.key === 'Escape') {
            productCell.html(originalContent);
            attachRowEventListeners(row);
        }
    });
    
    // Handle blur event to revert if no product selected
    var blurTimeout;
    searchInput.on('blur', function(e) {
        // Use timeout to allow click events on dropdown to fire first
        blurTimeout = setTimeout(function() {
            // Check if dropdown is still visible or if we clicked on it
            if (!$('.product-dropdown').length && productCell.find('.product-edit-search').length > 0) {
                productCell.html(originalContent);
                attachRowEventListeners(row);
            }
        }, 200);
    });
    
    // Clear blur timeout if focusing back on input
    searchInput.on('focus', function() {
        clearTimeout(blurTimeout);
    });
    
    // Handle click outside to cancel edit
    $(document).one('click', function(e) {
        if (!$(e.target).closest('.product-edit-search, .product-dropdown').length) {
            clearTimeout(blurTimeout); // Clear any pending blur timeout
            if (productCell.find('.product-edit-search').length > 0) {
                productCell.html(originalContent);
                attachRowEventListeners(row);
            }
        }
    });
}

// Search products for editing existing row
function searchProductsForEdit(searchTerm, row, rowIndex, originalContent) {
    var input = row.find('.product-edit-search');
    
    $.ajax({
        url: base_path + '/products/list',
        method: 'GET',
        dataType: 'json',
        data: {
            term: searchTerm,
            not_for_selling: 0,
            price_group: $('#price_group').val() || 0,
            product_types: ['single', 'variable', 'combo'],
            limit: 50,
            page: 1,
            location_id: $('#location_id').val(),
            with_warehouse_details: true,
            with_sub_units: true
        },
        success: function(result) {
            if (result && result.length > 0) {
                showProductDropdownForEdit(input, result, row, rowIndex, originalContent);
            } else {
                // No products found
                input.css('border-color', '#dc3545');
            }
        },
        error: function() {
            toastr.error('Error searching products');
        }
    });
}

// Show product dropdown for editing
function showProductDropdownForEdit(input, products, row, rowIndex, originalContent) {
    // Remove existing dropdown
    $('.product-dropdown').remove();
    
    // Use the existing showProductDropdown function
    showProductDropdown(input, products, row, rowIndex);
    
    // Override the select behavior to restore content if cancelled
    var dropdown = $('.product-dropdown');
    
    // Store original content for restoration
    dropdown.data('original-content', originalContent);
    
    // Override close behavior
    $(document).off('click.productDropdown').on('click.productDropdown', function(e) {
        if (!$(e.target).closest('.product-dropdown, .product-edit-search').length) {
            var productCell = row.find('td:eq(1)');
            if (productCell.find('.product-edit-search').length > 0) {
                productCell.html(originalContent);
                attachRowEventListeners(row);
            }
            dropdown.remove();
            $(document).off('click.productDropdown');
        }
    });
}

// Handle cell focus and blur events
function handleCellFocus(e) {
    $(this).select();
}

function handleCellBlur(e) {
    var input = $(this);
    var row = input.closest('tr');
    
    // Trigger calculation if it's a number input
    if (input.hasClass('number-input')) {
        calculateRowTotal(row);
    }
}

// Attach event listeners to a specific row
function attachRowEventListeners(row) {
    var rowIndex = row.index();
    
    // حدث تغيير الكمية
    row.find('.pos_quantity').off('input change').on('input change', function() {
        if (pos_form_validator) {
            pos_form_validator.element($(this));
        }
        
        var quantity = __read_number($(this)) || 0;
        var unit_price_inc_tax = __read_number(row.find('.pos_unit_price_inc_tax'));
        var line_total = quantity * unit_price_inc_tax;
        
        __write_number(row.find('.pos_line_total'), line_total, false);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        // تحديث كمية المعدّلات
        row.find('.modifier_qty_text').each(function(){
            $(this).text(__currency_trans_from_en(quantity, false));
        });
        row.find('.modifiers_quantity').each(function(){
            $(this).val(quantity);
        });
        
        pos_total_row();
      //  adjustComboQty(row);
    });
    
    // حدث تغيير السعر الأساسي
    row.find('.pos_unit_price').off('input change').on('input change', function() {
        var unit_price = __read_number($(this));
        
        // حساب السعر بعد الخصم
        var discounted_unit_price = calculate_discounted_unit_price(row);
        
        var tax_rate = row.find('.tax_id').find(':selected').data('rate') || 0;
        var quantity = __read_number(row.find('.pos_quantity'));
        
        var unit_price_inc_tax = __add_percent(discounted_unit_price, tax_rate);
        var line_total = quantity * unit_price_inc_tax;
        
        __write_number(row.find('.pos_unit_price_inc_tax'), unit_price_inc_tax);
        __write_number(row.find('.pos_line_total'), line_total);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        // تحديث الأسعار المعروضة
        var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
        row.find('td:eq(5) input').val(formatNumber(unit_price, 2));
        row.find('td:eq(6) input').val(formatNumber(convertUSDToIQD(unit_price), 0));
        
        pos_each_row(row);
        pos_total_row();
        round_row_to_iraqi_dinnar(row);
    });
    
    // حدث تغيير السعر شامل الضريبة
    row.find('.pos_unit_price_inc_tax').off('input change').on('input change', function() {
        var unit_price_inc_tax = __read_number($(this));
        
        if (iraqi_selling_price_adjustment) {
            unit_price_inc_tax = round_to_iraqi_dinnar(unit_price_inc_tax);
            __write_number($(this), unit_price_inc_tax);
        }
        
        var tax_rate = row.find('.tax_id').find(':selected').data('rate') || 0;
        var quantity = __read_number(row.find('.pos_quantity'));
        
        var line_total = quantity * unit_price_inc_tax;
        var discounted_unit_price = __get_principle(unit_price_inc_tax, tax_rate);
        var unit_price = get_unit_price_from_discounted_unit_price(row, discounted_unit_price);
        
        __write_number(row.find('.pos_unit_price'), unit_price);
        __write_number(row.find('.pos_line_total'), line_total, false);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        // تحديث الأسعار المعروضة
        var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
        var unit_price_usd = unit_price_inc_tax / exchange_rate;
        row.find('td:eq(5) input').val(formatNumber(unit_price_usd, 2));
        row.find('td:eq(6) input').val(formatNumber(convertUSDToIQD(unit_price_usd), 0));
        
        pos_each_row(row);
        pos_total_row();
    });
    
    // حدث تغيير نوع الضريبة
    row.find('.tax_id').off('change').on('change', function() {
        var tax_rate = $(this).find(':selected').data('rate') || 0;
        var unit_price_inc_tax = __read_number(row.find('.pos_unit_price_inc_tax'));
        
        var discounted_unit_price = __get_principle(unit_price_inc_tax, tax_rate);
        var unit_price = get_unit_price_from_discounted_unit_price(row, discounted_unit_price);
        __write_number(row.find('.pos_unit_price'), unit_price);
        pos_each_row(row);
    });
    
    // حدث تغيير الخصم
    row.find('.row_discount_type, .row_discount_amount, .discount_percent').off('change input').on('change input', function() {
        var discountPercent = parseFloat(row.find('.discount_percent').val()) || 0;
        
        // تحديث نوع ومقدار الخصم
        row.find('.row_discount_type').val('percentage');
        __write_number(row.find('.row_discount_amount'), discountPercent);
        
        // إعادة حساب السعر
        var discounted_unit_price = calculate_discounted_unit_price(row);
        var tax_rate = row.find('.tax_id').find(':selected').data('rate') || 0;
        var quantity = __read_number(row.find('.pos_quantity'));
        
        var unit_price_inc_tax = __add_percent(discounted_unit_price, tax_rate);
        var line_total = quantity * unit_price_inc_tax;
        
        __write_number(row.find('.pos_unit_price_inc_tax'), unit_price_inc_tax);
        __write_number(row.find('.pos_line_total'), line_total, false);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(line_total, true));
        
        pos_each_row(row);
        pos_total_row();
        round_row_to_iraqi_dinnar(row);
    });
    
    // حدث تغيير الوحدة الفرعية
    row.find('.sub_unit').off('change').on('change', function() {
        var base_unit_selling_price = row.find('.hidden_base_unit_sell_price').val();
        var selected_option = $(this).find(':selected');
        var multiplier = parseFloat(selected_option.data('multiplier')) || 1;
        var allow_decimal = parseInt(selected_option.data('allow_decimal'));
        
        row.find('.base_unit_multiplier').val(multiplier);
        
        var unit_sp = base_unit_selling_price * multiplier;
        var sp_element = row.find('.pos_unit_price');
        __write_number(sp_element, unit_sp);
        sp_element.trigger('change');
        
        // تحديث قواعد التحقق للكمية
        var qty_element = row.find('.pos_quantity');
        var base_max_avlbl = qty_element.data('qty_available');
        
        qty_element.attr('data-decimal', allow_decimal);
        var abs_digit = !allow_decimal;
        qty_element.rules('add', {
            abs_digit: abs_digit,
        });
        
        if (base_max_avlbl) {
            var max_avlbl = parseFloat(base_max_avlbl) / multiplier;
            var formated_max_avlbl = __number_f(max_avlbl);
            var unit_name = selected_option.data('unit_name');
            var max_err_msg = __translate('pos_max_qty_error', {
                max_val: formated_max_avlbl,
                unit_name: unit_name,
            });
            qty_element.attr('data-rule-max-value', max_avlbl);
            qty_element.attr('data-msg-max-value', max_err_msg);
            qty_element.rules('add', {
                'max-value': max_avlbl,
                messages: {
                    'max-value': max_err_msg,
                },
            });
            qty_element.trigger('change');
        }
       // adjustComboQty(row);
    });
    
    // حدث زر الحذف
    row.find('.remove-row').off('click').on('click', function() {
        if (confirm('Are you sure you want to remove this item?')) {
            row.remove();
            updateSerialNumbers();
            pos_total_row();
        }
    });
    
    // أحداث تغيير الأسعار للعرض (للدولار والدينار)
   row.find('td:eq(5) input, td:eq(6) input').off('input change blur').on('blur change', function() {
    var usdInput = row.find('td:eq(5) input');
    var iqrInput = row.find('td:eq(6) input');
    var current = $(this);
    var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
    
    if (current.is(usdInput)) {
        // تغير الدولار
        var usdPrice = parseFloat(current.val()) || 0;
        var iqrPrice = convertUSDToIQD(usdPrice);
        var localPrice = usdPrice * exchange_rate;
        
        iqrInput.val(formatNumber(iqrPrice, 0));
        __write_number(row.find('.pos_unit_price'), usdPrice);
        row.find('.pos_unit_price').trigger('change');
    } else {
        // تغير الدينار
        var iqrPrice = parseFloat(current.val()) || 0;
        var usdPrice = convertIQDToUSD(iqrPrice);
        var localPrice = usdPrice * exchange_rate;
        
        usdInput.val(formatNumber(usdPrice, 2));
        __write_number(row.find('.pos_unit_price'), usdPrice);
        row.find('.pos_unit_price').trigger('change');
    }
});

// إضافة معالج لتحسين تجربة المستخدم
row.find('td:eq(5) input, td:eq(6) input').on('focus', function() {
    $(this).select(); // تحديد النص بالكامل عند التركيز
});

// إضافة معالج للضغط على Enter
row.find('td:eq(5) input, td:eq(6) input').on('keypress', function(e) {
    if (e.which === 13) { // Enter key
        e.preventDefault();
        $(this).blur(); // تشغيل حدث blur لحفظ القيمة
        
        // الانتقال للحقل التالي
        var nextInput = $(this).closest('td').next('td').find('input:visible, select:visible').first();
        if (nextInput.length) {
            nextInput.focus();
        } else {
            // إذا كان آخر حقل، انتقل للصف التالي
            var nextRow = $(this).closest('tr').next('tr');
            if (nextRow.length) {
                nextRow.find('input:visible, select:visible').first().focus();
            }
        }
    }
});
}



function validateStockQuantity(row) {
    var quantity = parseFloat(row.find('.excel-input.number-input').first().val()) || 0;
    var stockInfo = row.find('td:eq(11) .stock-info').text();
    var stockQty = parseFloat(stockInfo.match(/\d+\.?\d*/)) || 0;
    var qtyInput = row.find('.excel-input.number-input').first();
    
    // إزالة الفئات السابقة
    qtyInput.removeClass('error-quantity warning-quantity');
    
    if (quantity > stockQty && stockQty > 0) {
        qtyInput.addClass('error-quantity');
        if (!$('#is_overselling_allowed').length || !$('#is_overselling_allowed').prop('checked')) {
            toastr.warning(`Requested quantity (${quantity}) exceeds available stock (${stockQty})`);
        }
    } else if (quantity > stockQty * 0.8 && stockQty > 0) {
        qtyInput.addClass('warning-quantity');
    }
}

// تعديل CSS لإضافة أنماط التحقق من الكمية
function applyExcelStyling() {
    var styles = `
        <style>
        /* ... الأنماط السابقة ... */
        
        /* أنماط التحقق من الكمية */
        .error-quantity {
            border: 2px solid #dc3545 !important;
            background-color: #f8d7da !important;
        }
        
        .warning-quantity {
            border: 2px solid #ffc107 !important;
            background-color: #fff3cd !important;
        }
        
        /* أنماط أداة اختيار المستودع */
        .warehouse-selector {
            width: 100%;
            height: 28px;
            border: none;
            outline: none;
            padding: 0 6px;
            font-size: 12px;
            background: transparent;
            cursor: pointer;
        }
        
        .warehouse-selector:focus {
            background: white;
            border: 2px solid #2563eb;
            box-shadow: 0 0 0 2px rgba(37, 99, 235, 0.1);
        }
        
        .warehouse-selector:disabled {
            cursor: not-allowed;
            color: #9ca3af;
        }
        </style>
    `;
    
    $('head').append(styles);
}


// Calculate row total
function calculateRowTotal(row) {
    var quantity = parseFloat(row.find('.pos_quantity').val()) || 0;
    var unitPrice = parseFloat(row.find('.pos_unit_price_inc_tax').val()) || 0;
    var discountPercent = parseFloat(row.find('td:eq(7) input').val()) || 0;
    
    // تطبيق الخصم
    var discountAmount = unitPrice * (discountPercent / 100);
    var priceAfterDiscount = unitPrice - discountAmount;
    
    // حساب المجموع الفرعي
    var subtotal = quantity * priceAfterDiscount;
    
    // تحديث حقل المجموع الفرعي
    row.find('.pos_line_total').val(formatNumber(subtotal, 2));
    row.find('.pos_line_total_text').text(__currency_trans_from_en(subtotal, true));
    
    // تحديث المجموع الإجمالي
    updateTotals();
}

// Update serial numbers for all rows
function updateSerialNumbers() {
    $('#pos_table tbody tr').each(function(index) {
        $(this).find('.serial-number').text(index + 1);
        $(this).attr('data-row_index', index);
        
        // Update input names
        var inputs = $(this).find('input, select');
        inputs.each(function() {
            var name = $(this).attr('name');
            if (name && name.includes('[') && name.includes(']')) {
                var newName = name.replace(/\[\d+\]/, '[' + index + ']');
                $(this).attr('name', newName);
            }
        });
    });
}

// Update totals (integrate with existing POS total calculation)
function updateTotals() {
    // استخدام دالة النظام الأصلي إذا كانت متوفرة
    if (typeof pos_total_row === 'function') {
        pos_total_row();
    } else {
        // حساب يدوي كبديل
        var totalQuantity = 0;
        var totalPrice = 0;
        
        $('#pos_table tbody tr.product_row').each(function() {
            var qty = parseFloat($(this).find('.pos_quantity').val()) || 0;
            var lineTotal = parseFloat($(this).find('.pos_line_total').val()) || 0;
            
            totalQuantity += qty;
            totalPrice += lineTotal;
        });
        
        // تحديث واجهة المستخدم
        $('.total_quantity').text(__number_f(totalQuantity));
        $('.price_total').text(__currency_trans_from_en(totalPrice, false));
    }
}



// Format number helper
function formatNumber(number, decimals = 2) {
    return parseFloat(number).toFixed(decimals);
}

// Format currency helper
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: 'USD',
        minimumFractionDigits: 2
    }).format(amount);
}

// Auto-resize input based on content
function autoResizeInput(input) {
    var value = input.val();
    var length = value.length;
    var minWidth = 60;
    var maxWidth = 200;
    var charWidth = 8; // Approximate character width in pixels
    
    var newWidth = Math.max(minWidth, Math.min(maxWidth, length * charWidth + 20));
    input.css('width', newWidth + 'px');
}

// Add keyboard shortcuts help
function addKeyboardShortcutsHelp() {
    // Check if help already exists
    if ($('.pos-shortcuts-help').length > 0) {
        return;
    }
    
    // Define helpHtml locally since it was commented out globally
    var helpHtml = `
        <div class="pos-shortcuts-help" style="
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 10px 15px;
            border-radius: 8px;
            font-size: 12px;
            z-index: 1000;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2);
        ">
            <div style="margin-bottom: 5px;">
                <strong>Keyboard Shortcuts:</strong>
            </div>
            <div>
                <kbd style="background: #444; padding: 2px 6px; border-radius: 3px;">Ctrl + K</kbd> 
                <span style="margin-left: 8px;">Delete selected product</span>
            </div>
            <div style="margin-top: 3px;">
                <i class="fa fa-mouse-pointer"></i> 
                <span style="margin-left: 8px;">Click product name to edit</span>
            </div>
            <div style="margin-top: 8px; padding-top: 8px; border-top: 1px solid #555;">
                <strong>Exchange Rate:</strong>
                <span id="current-iqd-rate" style="margin-left: 5px; color: #4ade80;">1 USD = ${getIQDExchangeRate()} IQD</span>
                <a href="#" id="change-iqd-rate" style="margin-left: 10px; color: #60a5fa; font-size: 11px;">
                    <i class="fa fa-edit"></i> Change
                </a>
            </div>
        </div>
    `;
    
    $('body').append(helpHtml);
    
    // Handle exchange rate change click
    $('#change-iqd-rate').on('click', function(e) {
        e.preventDefault();
        var currentRate = getIQDExchangeRate();
        var newRate = prompt('Enter new USD to IQD exchange rate:', currentRate);
        if (newRate && !isNaN(newRate) && parseFloat(newRate) > 0) {
            updateIQDExchangeRate(parseFloat(newRate));
            $('#current-iqd-rate').text('1 USD = ' + newRate + ' IQD');
            toastr.success('Exchange rate updated to ' + newRate + ' IQD');
            // Refresh the page to apply new rate
            if (confirm('Refresh page to apply new exchange rate?')) {
                location.reload();
            }
        }
    });
    
    // Auto-hide after 10 seconds
    setTimeout(function() {
        $('.pos-shortcuts-help').fadeOut(500);
    }, 10000);
}

// Copy selected cells functionality
function copySelectedCells() {
    var selectedCells = $('#pos_table').find('.selected');
    if (selectedCells.length === 0) return;
    
    var copyData = [];
    selectedCells.each(function() {
        var cellValue = $(this).is('input, select') ? $(this).val() : $(this).text();
        copyData.push(cellValue);
    });
    
    // Copy to clipboard (modern browsers)
    if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(copyData.join('\t'));
    } else {
        // Fallback for older browsers
        var textArea = document.createElement('textarea');
        textArea.value = copyData.join('\t');
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
    }
    
    toastr.success('Copied ' + copyData.length + ' cells');
}

// Enhanced paste functionality for Excel-like data
function handleExcelPaste(pastedData, startRow) {
    var rows = pastedData.split('\n').filter(row => row.trim());
    var startRowIndex = startRow.index();
    
    $('.paste-indicator').show();
    
    rows.forEach(function(rowData, rowOffset) {
        var cells = rowData.split('\t');
        var targetRowIndex = startRowIndex + rowOffset;
        var targetRow = $('#pos_table tbody tr').eq(targetRowIndex);
        
        // Create new row if needed
        if (targetRow.length === 0 || targetRow.hasClass('empty-row')) {
            if (targetRow.hasClass('empty-row')) {
                targetRow.removeClass('empty-row');
            } else {
                addEmptyProductRow();
                targetRow = $('#pos_table tbody tr').eq(targetRowIndex);
            }
        }
        
        // Process each cell in the row
        cells.forEach(function(cellData, cellOffset) {
            cellData = cellData.trim();
            if (!cellData) return;
            
            switch(cellOffset) {
                case 0: // Product name/search
                    if (cellData && !targetRow.find('.product_id').val()) {
                        targetRow.find('.product-search-input').val(cellData);
                        setTimeout(function() {
                            targetRow.find('.product-search-input').trigger('input');
                        }, rowOffset * 100);
                    }
                    break;
                    
                case 1: // SKU (read-only, skip)
                    break;
                    
                case 2: // Quantity
                    if (!isNaN(cellData) && parseFloat(cellData) > 0) {
                        targetRow.find('td:eq(3) input').val(cellData);
                    }
                    break;
                    
                case 3: // Unit (handle later when product is loaded)
                    break;
                    
                case 4: // Price USD
                    if (!isNaN(cellData) && parseFloat(cellData) > 0) {
                        targetRow.find('td:eq(5) input').val(cellData).trigger('input');
                    }
                    break;
                    
                case 5: // Price IQD
                    if (!isNaN(cellData) && parseFloat(cellData) > 0) {
                        targetRow.find('td:eq(6) input').val(cellData).trigger('input');
                    }
                    break;
                    
                case 6: // Discount %
                    if (!isNaN(cellData) && parseFloat(cellData) >= 0) {
                        targetRow.find('td:eq(7) input').val(cellData).trigger('input');
                    }
                    break;
            }
        });
        
        // Add new empty row if this was the last row
        if (targetRowIndex === $('#pos_table tbody tr').length - 1) {
            addEmptyProductRow();
        }
    });
    
    setTimeout(function() {
        $('.paste-indicator').hide();
        updateSerialNumbers();
        updateTotals();
    }, rows.length * 100 + 500);
}

// Integrate with original POS functions
function integrateWithOriginalPOS() {
    // Override the original pos_product_row function to work with our Excel-like table
    if (typeof window.original_pos_product_row === 'undefined') {
        window.original_pos_product_row = window.pos_product_row;
    }
    
    window.pos_product_row = function(variation_id, purchase_line_id, weighing_scale_barcode, quantity) {
        // Use our enhanced table instead
        var emptyRow = $('#pos_table tbody tr.empty-row').first();
        if (emptyRow.length === 0) {
            addEmptyProductRow();
            emptyRow = $('#pos_table tbody tr.empty-row').first();
        }
        
        // Get product details
       $.ajax({
    method: 'GET',
    url: '/sells/pos/get_product_row/' + variation_id + '/' + location_id,
    async: false,
    data: {
        product_row: product_row,
        customer_id: customer_id,
        is_direct_sell: is_direct_sell,
        is_serial_no: is_serial_no,
        price_group: price_group,
        purchase_line_id: purchase_line_id,
        weighing_scale_barcode: weighing_scale_barcode,
        quantity: quantity,
        is_sales_order: is_sales_order,
        disable_qty_alert: disable_qty_alert,
        is_draft: is_draft,
        with_sub_units: true  // إضافة هذا المعامل
    },
            dataType: 'json',
            success: function(result) {
                if (result.success && result.product) {
                    populateRowWithProduct(emptyRow, result.product, emptyRow.index());
                    
                    // Apply unit filter if active
                    if (window.activeUnitFilter) {
                        setTimeout(function() {
                            if (typeof window.applyUnitFilterToRow === 'function') {
                                window.applyUnitFilterToRow(emptyRow, window.activeUnitFilter);
                            } else if (typeof applyUnitFilterToRow === 'function') {
                                applyUnitFilterToRow(emptyRow, window.activeUnitFilter);
                            }
                        }, 150);
                    }
                    
                    addEmptyProductRow(); // Add new empty row
                } else {
                    toastr.error(result.msg || 'Error loading product');
                }
            }
        });
    };
    
    // Override original autocomplete to use our search
    if ($('#search_product').length) {
    $('#search_product').off('autocomplete');
    
    var searchTimeout = null;
    var minimumSearchLength = 3; // يمكنك تغيير هذا الرقم حسب الحاجة
    
    $('#search_product').on('input', function() {
        var $this = $(this);
        var searchTerm = $this.val().trim();
        
        // إلغاء المؤقت السابق إذا كان موجوداً
        if (searchTimeout) {
            clearTimeout(searchTimeout);
        }
        
        // إذا كان النص أقل من الحد الأدنى، لا تفعل شيئاً
        if (searchTerm.length < minimumSearchLength) {
            return;
        }
        
        // انتظر 500 مللي ثانية (نصف ثانية) بعد آخر حرف
        searchTimeout = setTimeout(function() {
            // Find the first empty row and populate its search
            var emptyRow = $('#pos_table tbody tr.empty-row').first();
            if (emptyRow.length === 0) {
                addEmptyProductRow();
                emptyRow = $('#pos_table tbody tr.empty-row').first();
            }
            
            // نقل النص كاملاً إلى حقل البحث في الجدول
            emptyRow.find('.product-search-input').val(searchTerm).trigger('input');
            
            // مسح حقل البحث الرئيسي
            $this.val('');
            
            // إعادة التركيز على حقل البحث الرئيسي (اختياري)
            // $this.focus();
            
        }, 500); // انتظر 500 مللي ثانية
    });
    
    // معالج إضافي للضغط على Enter
    $('#search_product').on('keypress', function(e) {
        if (e.which === 13) { // Enter key
            e.preventDefault();
            
            var searchTerm = $(this).val().trim();
            if (searchTerm.length >= minimumSearchLength) {
                // إلغاء المؤقت إذا كان موجوداً
                if (searchTimeout) {
                    clearTimeout(searchTimeout);
                }
                
                // نقل النص فوراً عند الضغط على Enter
                var emptyRow = $('#pos_table tbody tr.empty-row').first();
                if (emptyRow.length === 0) {
                    addEmptyProductRow();
                    emptyRow = $('#pos_table tbody tr.empty-row').first();
                }
                
                emptyRow.find('.product-search-input').val(searchTerm).trigger('input');
                $(this).val('');
            }
        }
    });
}
}

// Advanced search with filters
function showAdvancedProductSearch(input, row) {
    var modal = $(`
        <div class="modal fade" id="advancedProductSearch" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h4 class="modal-title">Advanced Product Search</h4>
                        <button type="button" class="close" data-dismiss="modal">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-4">
                                <input type="text" class="form-control" id="searchProductName" placeholder="Product Name">
                            </div>
                            <div class="col-md-4">
                                <input type="text" class="form-control" id="searchSKU" placeholder="SKU">
                            </div>
                            <div class="col-md-4">
                                <select class="form-control" id="searchCategory">
                                    <option value="">All Categories</option>
                                </select>
                            </div>
                        </div>
                        <div class="row mt-3">
                            <div class="col-md-12">
                                <button type="button" class="btn btn-primary" id="executeSearch">Search</button>
                                <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                            </div>
                        </div>
                        <div class="search-results mt-3" style="max-height: 400px; overflow-y: auto;">
                            <table class="table table-striped" id="searchResultsTable">
                                <thead>
                                    <tr>
                                        <th>Product</th>
                                        <th>SKU</th>
                                        <th>Price</th>
                                        <th>Stock</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody></tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `);
    
    $('body').append(modal);
    modal.modal('show');
    
    // Handle search
    modal.find('#executeSearch').on('click', function() {
        var searchData = {
            name: modal.find('#searchProductName').val(),
            sku: modal.find('#searchSKU').val(),
            category: modal.find('#searchCategory').val(),
            location_id: $('#location_id').val(),
            price_group: $('#price_group').val()
        };
        
        performAdvancedSearch(searchData, modal, row);
    });
    
    // Handle modal cleanup
    modal.on('hidden.bs.modal', function() {
        modal.remove();
    });
}

// Perform advanced product search
function performAdvancedSearch(searchData, modal, row) {
    $.ajax({
        url: '/products/advanced-search',
        method: 'GET',
        data: searchData,
        dataType: 'json',
        success: function(products) {
            var tbody = modal.find('#searchResultsTable tbody');
            tbody.empty();
            
            if (products.length === 0) {
                tbody.append('<tr><td colspan="5" class="text-center">No products found</td></tr>');
                return;
            }
            
            products.forEach(function(product) {
                var tr = $(`
                    <tr>
                        <td>${product.name}</td>
                        <td>${product.sub_sku || 'N/A'}</td>
                        <td>${formatCurrency(product.selling_price || 0)}</td>
                        <td>${product.qty_available || 'N/A'}</td>
                        <td>
                            <button type="button" class="btn btn-sm btn-primary select-product" 
                                    data-product='${JSON.stringify(product)}'>
                                Select
                            </button>
                        </td>
                    </tr>
                `);
                tbody.append(tr);
            });
            
            // Handle product selection
            tbody.find('.select-product').on('click', function() {
                var product = JSON.parse($(this).attr('data-product'));
                populateRowWithProduct(row, product, row.index());
                modal.modal('hide');
            });
        },
        error: function() {
            modal.find('#searchResultsTable tbody').html(
                '<tr><td colspan="5" class="text-center text-danger">Error performing search</td></tr>'
            );
        }
    });
}

// Bulk operations
function setupBulkOperations() {
    // Add bulk operations toolbar
  /*  var toolbar = $(`
        <div class="excel-toolbar" style="padding: 10px; background: #f8f9fa; border-bottom: 1px solid #dee2e6;">
            <div class="btn-group" role="group">
                <button type="button" class="btn btn-sm btn-outline-primary" id="selectAllRows">
                    <i class="fa fa-check-square"></i> Select All
                </button>
                <button type="button" class="btn btn-sm btn-outline-danger" id="deleteSelectedRows">
                    <i class="fa fa-trash"></i> Delete Selected
                </button>
                <button type="button" class="btn btn-sm btn-outline-info" id="bulkEditDiscount">
                    <i class="fa fa-percent"></i> Bulk Discount
                </button>
                <button type="button" class="btn btn-sm btn-outline-success" id="importFromExcel">
                    <i class="fa fa-file-excel-o"></i> Import Excel
                </button>
                <button type="button" class="btn btn-sm btn-outline-warning" id="exportToExcel">
                    <i class="fa fa-download"></i> Export Excel
                </button>
            </div>
            <div class="float-right">
                <span class="text-muted">
                    <small>Press Ctrl+A to select all, Ctrl+V to paste, Ctrl+C to copy</small>
                </span>
            </div>
        </div>
    `);  */
    
    $('.pos_product_div').prepend(toolbar);
    
    // Add checkbox column to table
    $('#pos_table thead tr').prepend('<th width="3%"><input type="checkbox" id="selectAllCheckbox"></th>');
    
    // Handle select all checkbox
    $(document).on('change', '#selectAllCheckbox', function() {
        var isChecked = $(this).prop('checked');
        $('#pos_table tbody tr:not(.empty-row)').find('.row-checkbox').prop('checked', isChecked);
    });
    
    // Handle individual row selection
    $(document).on('change', '.row-checkbox', function() {
        var totalRows = $('#pos_table tbody tr:not(.empty-row)').length;
        var selectedRows = $('#pos_table tbody .row-checkbox:checked').length;
        $('#selectAllCheckbox').prop('checked', totalRows === selectedRows);
    });
    
    // Bulk operations handlers
    $('#selectAllRows').on('click', function() {
        $('#selectAllCheckbox').prop('checked', true).trigger('change');
    });
    
    $('#deleteSelectedRows').on('click', function() {
        var selectedRows = $('#pos_table tbody .row-checkbox:checked').closest('tr');
        if (selectedRows.length === 0) {
            toastr.warning('Please select rows to delete');
            return;
        }
        
        if (confirm(`Are you sure you want to delete ${selectedRows.length} row(s)?`)) {
            selectedRows.remove();
            updateSerialNumbers();
            updateTotals();
            toastr.success(`${selectedRows.length} row(s) deleted`);
        }
    });
    
    $('#bulkEditDiscount').on('click', function() {
        var selectedRows = $('#pos_table tbody .row-checkbox:checked').closest('tr');
        if (selectedRows.length === 0) {
            toastr.warning('Please select rows to apply discount');
            return;
        }
        
        var discount = prompt('Enter discount percentage (0-100):');
        if (discount !== null && !isNaN(discount) && discount >= 0 && discount <= 100) {
            selectedRows.each(function() {
                $(this).find('td:eq(8) input').val(discount).trigger('input');
            });
            toastr.success(`Discount ${discount}% applied to ${selectedRows.length} row(s)`);
        }
    });
    
    $('#importFromExcel').on('click', function() {
        showExcelImportModal();
    });
    
    $('#exportToExcel').on('click', function() {
        exportTableToExcel();
    });
}

// Show Excel import modal
function showExcelImportModal() {
    var modal = $(`
        <div class="modal fade" id="excelImportModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h4 class="modal-title">Import from Excel</h4>
                        <button type="button" class="close" data-dismiss="modal">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="form-group">
                            <label>Select Excel File:</label>
                            <input type="file" class="form-control" id="excelFileInput" 
                                   accept=".xlsx,.xls,.csv">
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="hasHeaders" checked> 
                                First row contains headers
                            </label>
                        </div>
                        <div class="alert alert-info">
                            <strong>Expected format:</strong>
                            <ul class="mb-0">
                                <li>Column A: Product Name or SKU</li>
                                <li>Column B: Quantity</li>
                                <li>Column C: Unit Price (optional)</li>
                                <li>Column D: Discount % (optional)</li>
                            </ul>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-primary" id="processExcelFile">Import</button>
                        <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                    </div>
                </div>
            </div>
        </div>
    `);
    
    $('body').append(modal);
    modal.modal('show');
    
    modal.find('#processExcelFile').on('click', function() {
        processExcelFile();
    });
    
    modal.on('hidden.bs.modal', function() {
        modal.remove();
    });
}

// Process Excel file import
function processExcelFile() {
    var fileInput = document.getElementById('excelFileInput');
    var file = fileInput.files[0];
    
    if (!file) {
        toastr.error('Please select a file');
        return;
    }
    
    var reader = new FileReader();
    reader.onload = function(e) {
        try {
            var data = new Uint8Array(e.target.result);
            var workbook = XLSX.read(data, {type: 'array'});
            var firstSheet = workbook.Sheets[workbook.SheetNames[0]];
            var jsonData = XLSX.utils.sheet_to_json(firstSheet, {header: 1});
            
            var hasHeaders = $('#hasHeaders').prop('checked');
            var startRow = hasHeaders ? 1 : 0;
            
            // Process each row
            var processedRows = 0;
            for (var i = startRow; i < jsonData.length; i++) {
                var row = jsonData[i];
                if (row.length === 0 || !row[0]) continue;
                
                // Add new empty row
                addEmptyProductRow();
                var targetRow = $('#pos_table tbody tr.empty-row').last();
                targetRow.removeClass('empty-row');
                
                // Set product search
                if (row[0]) {
                    targetRow.find('.product-search-input').val(row[0]).trigger('input');
                }
                
                // Set quantity
                if (row[1] && !isNaN(row[1])) {
                    setTimeout(function(qty, tr) {
                        tr.find('td:eq(3) input').val(qty);
                    }, 200, row[1], targetRow);
                }
                
                // Set unit price
                if (row[2] && !isNaN(row[2])) {
                    setTimeout(function(price, tr) {
                        tr.find('td:eq(5) input').val(price).trigger('input');
                    }, 300, row[2], targetRow);
                }
                
                // Set discount
                if (row[3] && !isNaN(row[3])) {
                    setTimeout(function(discount, tr) {
                        tr.find('td:eq(7) input').val(discount).trigger('input');
                    }, 400, row[3], targetRow);
                }
                
                processedRows++;
            }
            
            $('#excelImportModal').modal('hide');
            toastr.success(`Successfully imported ${processedRows} rows`);
            
        } catch (error) {
            console.error('Error processing Excel file:', error);
            toastr.error('Error processing Excel file');
        }
    };
    
    reader.readAsArrayBuffer(file);
}

// Export table to Excel
function exportTableToExcel() {
    var tableData = [];
    var headers = ['Product', 'SKU', 'Quantity', 'Unit', 'Price USD', 'Price IQD', 'Discount %', 'Price inc. tax', 'Subtotal'];
    tableData.push(headers);
    
    $('#pos_table tbody tr:not(.empty-row)').each(function() {
        var row = [];
        var productName = $(this).find('td:eq(1)').text().trim() || $(this).find('.product-search-input').val();
        row.push(productName);
        row.push($(this).find('td:eq(2) input').val());
        row.push($(this).find('td:eq(3) input').val());
        row.push($(this).find('td:eq(4) select option:selected').text());
        row.push($(this).find('td:eq(5) input').val());
        row.push($(this).find('td:eq(6) input').val());
        row.push($(this).find('td:eq(7) input').val());
        row.push($(this).find('td:eq(8) input').val());
        row.push($(this).find('td:eq(9) input').val());
        tableData.push(row);
    });
    
    // Create workbook and worksheet
    var wb = XLSX.utils.book_new();
    var ws = XLSX.utils.aoa_to_sheet(tableData);
    
    // Add worksheet to workbook
    XLSX.utils.book_append_sheet(wb, ws, 'POS Items');
    
    // Generate Excel file and download
    var fileName = 'POS_Items_' + new Date().toISOString().slice(0, 10) + '.xlsx';
    XLSX.writeFile(wb, fileName);
    
    toastr.success('Table exported successfully');
}

// Initialize everything when document is ready
$(document).ready(function() {
    // Initialize core functionality
    initializeExcelLikePOSTable();
    addEmptyProductRow();
    setupBulkOperations();
    
    // Integrate with existing POS system
    setTimeout(function() {
        integrateWithOriginalPOS();
    }, 1000);
    
    // Add keyboard shortcuts help
    $(document).on('keydown', function(e) {
        if (e.key === 'F1') {
            e.preventDefault();
            showKeyboardShortcuts();
        }
    });
});

// Show keyboard shortcuts help
function showKeyboardShortcuts() {
    var modal = $(`
        <div class="modal fade" id="keyboardShortcuts" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h4 class="modal-title">Keyboard Shortcuts</h4>
                        <button type="button" class="close" data-dismiss="modal">&times;</button>
                    </div>
                    <div class="modal-body">
                        <table class="table table-sm">
                            <tr><td><kbd>Tab</kbd></td><td>Move to next cell</td></tr>
                            <tr><td><kbd>Shift + Tab</kbd></td><td>Move to previous cell</td></tr>
                            <tr><td><kbd>Enter</kbd></td><td>Move to next row</td></tr>
                            <tr><td><kbd>↑ ↓ ← →</kbd></td><td>Navigate between cells</td></tr>
                            <tr><td><kbd>Ctrl + C</kbd></td><td>Copy selected cells</td></tr>
                            <tr><td><kbd>Ctrl + V</kbd></td><td>Paste data</td></tr>
                            <tr><td><kbd>Ctrl + A</kbd></td><td>Select all rows</td></tr>
                            <tr><td><kbd>Ctrl + Del</kbd></td><td>Delete current row</td></tr>
                            <tr><td><kbd>F1</kbd></td><td>Show this help</td></tr>
                        </table>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                    </div>
                </div>
            </div>
        </div>
    `);
    
    $('body').append(modal);
    modal.modal('show');
    modal.on('hidden.bs.modal', function() {
        modal.remove();
    });
}



// Centralized utility functions
var UtilityFunctions = {
    removeEmptyRowsBeforeSubmit: function() {
        $('#pos_table tbody tr').each(function() {
            var row = $(this);
            var hasProduct = row.find('.product_id').val() || 
                            row.find('.variation_id').val() ||
                            row.find('.product-search-input').val().trim();
            
            var hasQuantity = row.find('.pos_quantity').val() && 
                             parseFloat(row.find('.pos_quantity').val()) > 0;
            
            // إذا كان الصف فارغ، احذفه
            if (!hasProduct || !hasQuantity || row.hasClass('empty-row')) {
                row.remove();
            }
        });
        
        // إعادة ترقيم الصفوف المتبقية
        if (typeof updateSerialNumbers === 'function') {
            updateSerialNumbers();
        }
    }
};

// Backward compatibility
function removeEmptyRowsBeforeSubmit() {
    return UtilityFunctions.removeEmptyRowsBeforeSubmit();
}


// Filter 









var originalPopulateRowWithProduct = window.populateRowWithProduct;
window.populateRowWithProduct = function(row, product, rowIndex) {
    // استدعاء الدالة الأصلية أولاً
    originalPopulateRowWithProduct.apply(this, arguments);
    
    // حفظ بيانات المنتج في الصف للاستخدام مع نظام التسعير المتقدم
    if (product && row) {
        row.data('product-data', product);
        console.log('💾 Product data saved to row:', {
            name: product.name,
            sku: product.sku,
            base_unit: product.product_custom_field2,
            has_price_array: !!product.product_custom_field3
        });
    }
    
    // تطبيق التسعير المتقدم فوراً لضمان قراءة السعر من مصفوفة الأسعار
    setTimeout(function() {
        if (typeof applyAdvancedPricingToRow === 'function') {
            applyAdvancedPricingToRow(row);
            console.log('🎯 Advanced pricing applied - will prioritize price array');
        }
    }, 100);
    
    // إذا كان هناك فلتر وحدة نشط، طبقه
    if (window.activeUnitFilter) {
        setTimeout(function() {
            applyUnitFilterToRow(row, window.activeUnitFilter);
        }, 150);
    }
};

// دالة لتطبيق فلتر الوحدة على الصف
function applyUnitFilterToRow(row, unitFilter) {
    var unitInput = row.find('.unit-input');
    var availableUnits = [];
    
    try {
        availableUnits = JSON.parse(unitInput.attr('data-available-units') || '[]');
    } catch (e) {
        console.error('Error parsing available units:', e);
        return;
    }
    
    // البحث عن الوحدة المطابقة
    var matchedUnit = null;
    
    for (var i = 0; i < availableUnits.length; i++) {
        var unit = availableUnits[i];
        var unitMultiplier = parseFloat(unit.multiplier || 1);
        
        // للـ 1KG، نبحث عن وحدة KG
        if (unitFilter.name === '1KG' || unitFilter.filter === '1KG') {
            if ((unit.unit_name || unit.name || '').toUpperCase() === 'KG') {
                matchedUnit = unit;
                break;
            }
        } 
        // للوحدات الأخرى، نبحث عن المضاعف المطابق
        else if (Math.abs(unitMultiplier - unitFilter.multiplier) < 0.001) {
            matchedUnit = unit;
            break;
        }
    }
    
    if (matchedUnit) {
        // تطبيق الوحدة المطابقة
        var unitName = matchedUnit.unit_name || matchedUnit.name || 'EA';
        unitInput.val(unitName).trigger('change');
        
        console.log('Applied unit filter:', {
            filter: unitFilter,
            matched_unit: matchedUnit,
            unit_name: unitName
        });
        
        toastr.success('تم تطبيق وحدة ' + unitFilter.filter + ' تلقائياً', '', {
            timeOut: 1000,
            positionClass: 'toast-bottom-right'
        });
    } else {
        console.warn('No matching unit found for filter:', unitFilter);
    }
}

// تحديث دالة البحث عن المنتجات لتطبيق فلتر فل بلاستك
var originalSearchProducts = window.searchProducts;
window.searchProducts = function(searchTerm, row, rowIndex) {
    var price_group = $('#price_group').val() || '';
    
    var searchData = {
        price_group: price_group,
        term: searchTerm,
        not_for_selling: 0,
        limit: 20,
        search_all_locations: true,
        include_all_warehouses: true,
        with_warehouse_stock: true,
        with_sub_units: true,
        include_unit_details: true,
        load_sub_units: true
    };
    
    $.ajax({
        url: base_path + '/products/list',
        method: 'GET',
        dataType: 'json',
        data: searchData,
        success: function(products) {
            var input = row.find('.product-search-input');
            input.removeClass('cell-loading');
            
            if (!products || !Array.isArray(products)) {
                console.warn('Invalid products data received');
                products = [];
            }
            
            // معالجة البيانات
            products.forEach(function(product) {
                try {
                    processProductUnitsData(product);
                    processProductWarehouseData(product);
                } catch (e) {
                    console.error('Error processing product data:', e);
                }
            });
            
            // تطبيق فلاتر العلامات التجارية
            if (window.activeFilters.length > 0) {
                products = products.filter(function(product) {
                    var sku = product.sub_sku || product.sku || '';
                    return window.activeFilters.some(filter => 
                        sku.toUpperCase().startsWith(filter.prefix.toUpperCase())
                    );
                });
            }
            
            // تطبيق فلتر فل بلاستك
            if (window.fullPlasticFilterActive === false) {
                // إذا كان الفلتر غير نشط، استبعد منتجات فل بلاستك
                products = products.filter(function(product) {
                    return !isFullPlasticProduct(product);
                });
            }
            // إذا كان الفلتر نشط، اعرض جميع المنتجات بما فيها فل بلاستك
            
            if (products.length === 1) {
                populateRowWithProduct(row, products[0], rowIndex);
            } else if (products.length > 1) {
                showProductDropdown(input, products, row, rowIndex);
            } else {
                var message = 'لم يتم العثور على منتجات';
                if (window.activeFilters.length > 0) {
                    var filterNames = window.activeFilters.map(f => f.name).join(' أو ');
                    message = 'لم يتم العثور على منتجات ' + filterNames;
                }
                toastr.warning(message);
                clearRowData(row);
            }
        },
        error: function(jqXHR, textStatus, errorThrown) {
            console.error('Search error:', textStatus, errorThrown);
            row.find('.product-search-input').removeClass('cell-loading');
            toastr.error('خطأ في البحث عن المنتجات');
        }
    });
};

// دالة للتحقق من منتج فل بلاستك
function isFullPlasticProduct(product) {
    // التحقق من وحدة القياس الأساسية
    var baseUnit = (product.unit || '').toLowerCase();
    if (baseUnit === 'فل بلاستك' || baseUnit === 'full plastic' || baseUnit === 'fl plastic') {
        return true;
    }
    
    // التحقق من الوحدات الفرعية
    if (product.processed_units && Array.isArray(product.processed_units)) {
        return product.processed_units.some(function(unit) {
            var unitName = (unit.unit_name || unit.name || '').toLowerCase();
            return unitName === 'فل بلاستك' || unitName === 'full plastic' || unitName === 'fl plastic';
        });
    }
    
    // التحقق من الوصف أو الاسم
    var productName = (product.name || '').toLowerCase();
    var productDescription = (product.description || '').toLowerCase();
    
    return productName.includes('فل بلاستك') || 
           productName.includes('full plastic') ||
           productDescription.includes('فل بلاستك') ||
           productDescription.includes('full plastic');
}

// تحديث دالة إعادة تحميل قائمة المنتجات


// ============================================
// نظام التحكم في تنسيق الصفحة - الإصدار المصحح
// ============================================

// تهيئة النظام عند تحميل الصفحة
$(document).ready(function() {
    // الانتظار قليلاً للتأكد من تحميل جميع العناصر
    setTimeout(function() {
        initializeLayoutControlSystem();
    }, 1500);
});

// تهيئة نظام التحكم في التنسيق
function initializeLayoutControlSystem() {
    console.log('Initializing Layout Control System...');
    
    // التحقق من وجود عناصر الفلترات
    if ($('.filter-buttons-wrapper').length === 0) {
        console.warn('Filter buttons wrapper not found, retrying...');
        setTimeout(initializeLayoutControlSystem, 1000);
        return;
    }
    
    // إضافة زر التحكم في التنسيق
    var controlButtonHTML = `
        <button type="button" class="btn btn-info btn-sm" id="layoutControlBtn" 
                style="margin-left: 10px; border-radius: 20px; padding: 5px 15px;">
            <i class="fa fa-cog"></i> تخصيص العرض
        </button>
    `;
    
    // إضافة الزر بجانب الفلترات
    if ($('#layoutControlBtn').length === 0) {
        $('.filter-buttons-wrapper').append(controlButtonHTML);
    }
    
    // إنشاء نافذة التحكم
    if ($('#layoutControlModal').length === 0) {
        createLayoutControlModal();
    }
    
    // إضافة معالجات الأحداث
    attachLayoutControlEventHandlers();
    
    // تحميل الإعدادات المحفوظة
    loadUserLayoutSettings();
}

// إنشاء نافذة التحكم في التنسيق
function createLayoutControlModal() {
    var modalHTML = `
    <div class="modal fade" id="layoutControlModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document" style="max-width: 90%;">
            <div class="modal-content">
                <div class="modal-header">
                    <h4 class="modal-title">
                        <i class="fa fa-cog"></i> تخصيص عرض الصفحة
                    </h4>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <!-- التبويبات -->
                    <ul class="nav nav-tabs" role="tablist">
                        <li class="nav-item active">
                            <a class="nav-link" data-toggle="tab" href="#generalSettings">
                                <i class="fa fa-sliders"></i> إعدادات عامة
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" data-toggle="tab" href="#posTableSettings">
                                <i class="fa fa-table"></i> جدول المنتجات
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" data-toggle="tab" href="#searchTableSettings">
                                <i class="fa fa-search"></i> جدول البحث
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" data-toggle="tab" href="#columnHeaders">
                                <i class="fa fa-header"></i> عناوين الأعمدة
                            </a>
                        </li>
                    </ul>
                    
                    <!-- محتوى التبويبات -->
                    <div class="tab-content mt-3">
                        <!-- إعدادات عامة -->
                        <div class="tab-pane fade in active" id="generalSettings">
                            <h5>حجم الخط العام</h5>
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <label>حجم خط الصفحة الأساسي</label>
                                    <div class="input-group">
                                        <input type="range" class="form-control-range" id="globalFontSize" 
                                               min="10" max="20" value="14" step="1">
                                        <span class="ml-2 font-size-display" id="globalFontSizeValue">14px</span>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label>حجم خط الأزرار</label>
                                    <div class="input-group">
                                        <input type="range" class="form-control-range" id="buttonFontSize" 
                                               min="10" max="18" value="13" step="1">
                                        <span class="ml-2 font-size-display" id="buttonFontSizeValue">13px</span>
                                    </div>
                                </div>
                            </div>
                            
                            <h5>الألوان والمظهر</h5>
                            <div class="row mb-4">
                                <div class="col-md-4">
                                    <label>نمط الجدول</label>
                                    <select class="form-control" id="tableTheme">
                                        <option value="default">افتراضي</option>
                                        <option value="dark">داكن</option>
                                        <option value="light">فاتح</option>
                                        <option value="blue">أزرق</option>
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label>كثافة العرض</label>
                                    <select class="form-control" id="displayDensity">
                                        <option value="comfortable">مريح</option>
                                        <option value="compact">مضغوط</option>
                                        <option value="spacious">واسع</option>
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label>عرض الحدود</label>
                                    <select class="form-control" id="borderStyle">
                                        <option value="all">جميع الحدود</option>
                                        <option value="horizontal">أفقي فقط</option>
                                        <option value="none">بدون حدود</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <!-- إعدادات جدول المنتجات -->
                        <div class="tab-pane fade" id="posTableSettings">
                            <h5>أعمدة جدول المنتجات</h5>
                            <div class="column-settings" id="posTableColumns">
                                <!-- سيتم ملؤها ديناميكياً -->
                            </div>
                            
                            <h5 class="mt-4">قياسات الجدول</h5>
                            <div class="row">
                                <div class="col-md-4">
                                    <label>ارتفاع الصف</label>
                                    <div class="input-group">
                                        <input type="range" class="form-control-range" id="posRowHeight" 
                                               min="25" max="50" value="32" step="1">
                                        <span class="ml-2 font-size-display" id="posRowHeightValue">32px</span>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <label>حجم خط الجدول</label>
                                    <div class="input-group">
                                        <input type="range" class="form-control-range" id="posTableFontSize" 
                                               min="10" max="16" value="12" step="1">
                                        <span class="ml-2 font-size-display" id="posTableFontSizeValue">12px</span>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <label>عرض الجدول</label>
                                    <select class="form-control" id="posTableWidth">
                                        <option value="auto">تلقائي</option>
                                        <option value="100%">100%</option>
                                        <option value="90%">90%</option>
                                        <option value="fixed">ثابت</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <!-- إعدادات جدول البحث -->
                        <div class="tab-pane fade" id="searchTableSettings">
                            <h5>أعمدة جدول البحث</h5>
                            <div class="column-settings" id="searchTableColumns">
                                <!-- سيتم ملؤها ديناميكياً -->
                            </div>
                            
                            <h5 class="mt-4">قياسات جدول البحث</h5>
                            <div class="row">
                                <div class="col-md-4">
                                    <label>ارتفاع الصف</label>
                                    <div class="input-group">
                                        <input type="range" class="form-control-range" id="searchRowHeight" 
                                               min="30" max="60" value="40" step="2">
                                        <span class="ml-2 font-size-display" id="searchRowHeightValue">40px</span>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <label>حجم خط البحث</label>
                                    <div class="input-group">
                                        <input type="range" class="form-control-range" id="searchTableFontSize" 
                                               min="11" max="18" value="13" step="1">
                                        <span class="ml-2 font-size-display" id="searchTableFontSizeValue">13px</span>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <label>عرض نافذة البحث</label>
                                    <select class="form-control" id="searchModalWidth">
                                        <option value="auto">تلقائي</option>
                                        <option value="80%">80%</option>
                                        <option value="90%">90%</option>
                                        <option value="full">ملء الشاشة</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <!-- عناوين الأعمدة المخصصة -->
                        <div class="tab-pane fade" id="columnHeaders">
                            <h5>تخصيص عناوين الأعمدة</h5>
                            <div class="custom-headers-container">
                                <!-- سيتم ملؤها ديناميكياً -->
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" id="resetToDefaults">
                        <i class="fa fa-refresh"></i> استعادة الافتراضي
                    </button>
                    <button type="button" class="btn btn-primary" id="saveLayoutSettings">
                        <i class="fa fa-save"></i> حفظ الإعدادات
                    </button>
                    <button type="button" class="btn btn-success" id="applyLayoutSettings">
                        <i class="fa fa-check"></i> تطبيق
                    </button>
                </div>
            </div>
        </div>
    </div>
    `;
    
    $('body').append(modalHTML);
    
    // إضافة الأنماط CSS
    addLayoutControlStyles();
}

// إضافة الأنماط CSS لنافذة التحكم
function addLayoutControlStyles() {
    if ($('#layoutControlStyles').length > 0) return;
    
    var styles = `
    <style id="layoutControlStyles">
    /* أنماط نافذة التحكم */
    #layoutControlModal .modal-dialog {
        max-height: 90vh;
    }
    
    #layoutControlModal .modal-body {
        max-height: 70vh;
        overflow-y: auto;
    }
    
    /* إصلاح عرض القيم */
    .font-size-display {
        display: inline-block;
        min-width: 50px;
        text-align: left;
        font-weight: bold;
        color: #333;
    }
    
    /* أنماط الأعمدة */
    .column-settings {
        max-height: 300px;
        overflow-y: auto;
        border: 1px solid #dee2e6;
        border-radius: 5px;
        padding: 10px;
    }
    
    .column-item {
        padding: 10px;
        margin-bottom: 5px;
        background: #f8f9fa;
        border-radius: 5px;
        cursor: move;
        transition: all 0.3s ease;
    }
    
    .column-item:hover {
        background: #e9ecef;
        transform: translateY(-2px);
        box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .column-item.sortable-ghost {
        opacity: 0.5;
        background: #6c757d;
    }
    
    .column-item .column-controls {
        display: flex;
        align-items: center;
        justify-content: space-between;
    }
    
    .column-visibility {
        width: 20px;
        height: 20px;
        cursor: pointer;
    }
    
    .column-width-slider {
        width: 100px;
        margin: 0 10px;
    }
    
    /* أنماط عناوين الأعمدة المخصصة */
    .custom-header-item {
        display: flex;
        align-items: center;
        margin-bottom: 10px;
        padding: 10px;
        background: #f8f9fa;
        border-radius: 5px;
    }
    
    .custom-header-item label {
        flex: 1;
        margin-bottom: 0;
        font-weight: 600;
        min-width: 150px;
    }
    
    .custom-header-item input {
        flex: 2;
        margin-left: 10px;
    }
    
    /* معاينة الإعدادات */
    .settings-preview {
        position: fixed;
        bottom: 20px;
        right: 20px;
        background: rgba(0,0,0,0.8);
        color: white;
        padding: 10px 20px;
        border-radius: 5px;
        display: none;
        z-index: 9999;
    }
    
    /* تأثيرات التحريك */
    .layout-transition {
        transition: all 0.3s ease !important;
    }
    
    /* مؤشر السحب */
    .drag-handle {
        cursor: move;
        color: #6c757d;
        margin-right: 10px;
    }
    
    .drag-handle:hover {
        color: #343a40;
    }
    
    /* شريط التقدم للتحميل */
    .layout-loading {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 3px;
        background: linear-gradient(90deg, #007bff 0%, #28a745 50%, #007bff 100%);
        background-size: 200% 100%;
        animation: loading 1.5s linear infinite;
        z-index: 9999;
        display: none;
    }
    
    @keyframes loading {
        0% {
            background-position: 0% 0%;
        }
        100% {
            background-position: 200% 0%;
        }
    }
    
    /* تحسينات للأجهزة المحمولة */
    @media (max-width: 768px) {
        #layoutControlModal .modal-dialog {
            max-width: 100%;
            margin: 0;
        }
        
        .column-item {
            font-size: 12px;
        }
        
        .column-width-slider {
            width: 80px;
        }
    }
    
    /* إضافة مساحة بين الأقسام */
    #layoutControlModal h5 {
        margin-top: 20px;
        margin-bottom: 15px;
        color: #333;
        border-bottom: 1px solid #dee2e6;
        padding-bottom: 5px;
    }
    
    #layoutControlModal h5:first-child {
        margin-top: 0;
    }
    </style>
    `;
    
    $('head').append(styles);
    
    // إضافة شريط التحميل
    if ($('.layout-loading').length === 0) {
        $('body').append('<div class="layout-loading"></div>');
    }
}

// إضافة معالجات الأحداث
function attachLayoutControlEventHandlers() {
    // فتح نافذة التحكم
    $(document).off('click', '#layoutControlBtn').on('click', '#layoutControlBtn', function() {
        loadTableColumnsSettings();
        $('#layoutControlModal').modal('show');
    });
    
    // معالجات تغيير القيم
    $(document).off('input', '#globalFontSize').on('input', '#globalFontSize', function() {
        $('#globalFontSizeValue').text($(this).val() + 'px');
        previewSetting('fontSize', $(this).val() + 'px');
    });
    
    $(document).off('input', '#buttonFontSize').on('input', '#buttonFontSize', function() {
        $('#buttonFontSizeValue').text($(this).val() + 'px');
        previewSetting('buttonFontSize', $(this).val() + 'px');
    });
    
    $(document).off('input', '#posRowHeight').on('input', '#posRowHeight', function() {
        $('#posRowHeightValue').text($(this).val() + 'px');
        previewSetting('posRowHeight', $(this).val() + 'px');
    });
    
    $(document).off('input', '#posTableFontSize').on('input', '#posTableFontSize', function() {
        $('#posTableFontSizeValue').text($(this).val() + 'px');
        previewSetting('posTableFontSize', $(this).val() + 'px');
    });
    
    $(document).off('input', '#searchRowHeight').on('input', '#searchRowHeight', function() {
        $('#searchRowHeightValue').text($(this).val() + 'px');
    });
    
    $(document).off('input', '#searchTableFontSize').on('input', '#searchTableFontSize', function() {
        $('#searchTableFontSizeValue').text($(this).val() + 'px');
    });
    
    // معالج تغيير عرض العمود
    $(document).off('input', '.column-width-slider').on('input', '.column-width-slider', function() {
        var width = $(this).val();
        $(this).siblings('span').text(width + 'px');
    });
    
    // حفظ الإعدادات
    $(document).off('click', '#saveLayoutSettings').on('click', '#saveLayoutSettings', function() {
        saveAllLayoutSettings();
    });
    
    // تطبيق الإعدادات
    $(document).off('click', '#applyLayoutSettings').on('click', '#applyLayoutSettings', function() {
        applyAllLayoutSettings();
        $('#layoutControlModal').modal('hide');
    });
    
    // استعادة الإعدادات الافتراضية
    $(document).off('click', '#resetToDefaults').on('click', '#resetToDefaults', function() {
        if (confirm('هل أنت متأكد من استعادة الإعدادات الافتراضية؟')) {
            resetToDefaultSettings();
        }
    });
    
    // معالج تغيير النمط
    $(document).off('change', '#tableTheme').on('change', '#tableTheme', function() {
        previewTheme($(this).val());
    });
    
    // معالج تغيير كثافة العرض
    $(document).off('change', '#displayDensity').on('change', '#displayDensity', function() {
        previewDensity($(this).val());
    });
}

// تحميل إعدادات الأعمدة للجداول
function loadTableColumnsSettings() {
    // أعمدة جدول المنتجات (محدث حسب الجدول الفعلي)
    var posTableColumns = [
        { id: 'serial', name: '#', visible: true, width: 30, order: 0 },
        { id: 'product', name: 'Product', visible: true, width: 200, order: 1 },
        { id: 'quantity', name: 'Quantity', visible: true, width: 80, order: 2 },
        { id: 'unit', name: 'Unit', visible: true, width: 80, order: 3 },
        { id: 'price_usd', name: 'Price (USD)', visible: true, width: 80, order: 4 },
        { id: 'price_iqd', name: 'Price (IQD)', visible: true, width: 80, order: 5 },
        { id: 'discount', name: 'Discount %', visible: true, width: 60, order: 6 },
        { id: 'price_inc_tax', name: 'Price inc. tax', visible: true, width: 80, order: 7 },
        { id: 'subtotal', name: 'Subtotal', visible: true, width: 80, order: 8 },
        { id: 'warehouse', name: 'Warehouse', visible: true, width: 100, order: 9 },
        { id: 'stock', name: 'Stock Info', visible: true, width: 80, order: 10 },
        { id: 'actions', name: 'Actions', visible: true, width: 40, order: 11 }
    ];
    
    // أعمدة جدول البحث
    var searchTableColumns = [
        { id: 'product_name', name: 'Product', visible: true, width: 200, order: 0 },
        { id: 'category', name: 'Category', visible: true, width: 120, order: 3},
        { id: 'foreign_name', name: 'Foreign Name', visible: true, width: 150, order: 1 },
        { id: 'sku', name: 'SKU', visible: true, width: 100, order: 10 },
        { id: 'price_usd', name: 'Price (USD)', visible: true, width: 100, order: 4 },
        { id: 'price_iqd', name: 'Price (IQD)', visible: true, width: 120, order: 5 },
        { id: 'discount', name: 'Discount', visible: true, width: 80, order: 6 },
        { id: 'final_price', name: 'Final Price', visible: true, width: 120, order: 7 },
        { id: 'uom', name: 'UOM', visible: true, width: 80, order: 2 },
        { id: 'total_stock', name: 'Total Stock', visible: true, width: 100, order: 8 },
        { id: 'locations', name: 'All Locations Stock', visible: true, width: 250, order: 9 }
    ];
    
    // تحميل الإعدادات المحفوظة
    var savedPosColumns = localStorage.getItem('posTableColumnsSettings');
    if (savedPosColumns) {
        try {
            posTableColumns = JSON.parse(savedPosColumns);
        } catch (e) {
            console.error('Error parsing saved POS columns:', e);
        }
    }
    
    var savedSearchColumns = localStorage.getItem('searchTableColumnsSettings');
    if (savedSearchColumns) {
        try {
            searchTableColumns = JSON.parse(savedSearchColumns);
        } catch (e) {
            console.error('Error parsing saved search columns:', e);
        }
    }
    
    // عرض الأعمدة في الواجهة
    displayColumnSettings('posTableColumns', posTableColumns);
    displayColumnSettings('searchTableColumns', searchTableColumns);
    
    // عرض عناوين الأعمدة المخصصة
    displayCustomHeaders(posTableColumns, searchTableColumns);
}

// عرض إعدادات الأعمدة
function displayColumnSettings(containerId, columns) {
    var container = $('#' + containerId);
    container.empty();
    
    // ترتيب الأعمدة حسب الترتيب المحفوظ
    columns.sort((a, b) => a.order - b.order);
    
    columns.forEach(function(column) {
        var columnHTML = `
            <div class="column-item" data-column-id="${column.id}">
                <div class="column-controls">
                    <div class="d-flex align-items-center">
                        <i class="fa fa-bars drag-handle"></i>
                        <input type="checkbox" class="column-visibility" 
                               ${column.visible ? 'checked' : ''} 
                               id="${containerId}_${column.id}">
                        <label for="${containerId}_${column.id}" class="mb-0 ml-2">
                            ${column.name}
                        </label>
                    </div>
                    <div class="d-flex align-items-center">
                        <small class="mr-2">العرض:</small>
                        <input type="range" class="column-width-slider" 
                               min="50" max="300" value="${column.width}" 
                               data-column-id="${column.id}">
                        <span class="ml-2">${column.width}px</span>
                    </div>
                </div>
            </div>
        `;
        container.append(columnHTML);
    });
    
    // تفعيل السحب والإفلات
    if (typeof Sortable !== 'undefined') {
        new Sortable(container[0], {
            animation: 150,
            ghostClass: 'sortable-ghost',
            handle: '.drag-handle',
            onEnd: function(evt) {
                updateColumnOrder(containerId);
            }
        });
    }
}

// عرض عناوين الأعمدة المخصصة
function displayCustomHeaders(posColumns, searchColumns) {
    var container = $('.custom-headers-container');
    container.empty();
    
    // دمج الأعمدة من الجدولين
    var allColumns = [...posColumns, ...searchColumns];
    var uniqueColumns = [];
    var seenIds = new Set();
    
    allColumns.forEach(col => {
        if (!seenIds.has(col.id)) {
            seenIds.add(col.id);
            uniqueColumns.push(col);
        }
    });
    
    uniqueColumns.forEach(function(column) {
        var savedHeaders = JSON.parse(localStorage.getItem('customColumnHeaders') || '{}');
        var currentValue = savedHeaders[column.id] || column.name;
        
        var headerHTML = `
            <div class="custom-header-item">
                <label>${column.id}</label>
                <input type="text" class="form-control custom-header-input" 
                       data-column-id="${column.id}" 
                       value="${currentValue}" 
                       placeholder="أدخل العنوان المخصص">
            </div>
        `;
        container.append(headerHTML);
    });
}

// معاينة الإعدادات
function previewSetting(setting, value) {
    // إضافة فئة للانتقال السلس
    $('body').addClass('layout-transition');
    
    switch(setting) {
        case 'fontSize':
            $('body').css('font-size', value);
            break;
        case 'buttonFontSize':
            $('button, .btn').css('font-size', value);
            break;
        case 'posRowHeight':
            $('#pos_table tbody tr').css('height', value);
            $('#pos_table tbody td').css('height', value);
            break;
        case 'posTableFontSize':
            $('#pos_table').css('font-size', value);
            break;
    }
    
    // إظهار رسالة المعاينة
    showPreviewMessage('معاينة: ' + setting);
}

// معاينة النمط
function previewTheme(theme) {
    $('#pos_table').removeClass('theme-default theme-dark theme-light theme-blue');
    $('#pos_table').addClass('theme-' + theme);
    
    // إضافة أنماط النمط
    applyThemeStyles(theme);
}

// معاينة كثافة العرض
function previewDensity(density) {
    var padding = {
        'comfortable': '12px',
        'compact': '6px',
        'spacious': '16px'
    };
    
    $('#pos_table td, #pos_table th').css('padding', padding[density]);
}

// حفظ جميع الإعدادات
function saveAllLayoutSettings() {
    $('.layout-loading').show();
    
    var settings = {
        general: {
            globalFontSize: $('#globalFontSize').val(),
            buttonFontSize: $('#buttonFontSize').val(),
            tableTheme: $('#tableTheme').val(),
            displayDensity: $('#displayDensity').val(),
            borderStyle: $('#borderStyle').val()
        },
        posTable: {
            rowHeight: $('#posRowHeight').val(),
            fontSize: $('#posTableFontSize').val(),
            width: $('#posTableWidth').val(),
            columns: collectColumnSettings('posTableColumns')
        },
        searchTable: {
            rowHeight: $('#searchRowHeight').val(),
            fontSize: $('#searchTableFontSize').val(),
            modalWidth: $('#searchModalWidth').val(),
            columns: collectColumnSettings('searchTableColumns')
        },
        customHeaders: collectCustomHeaders()
    };
    
    // حفظ في localStorage
    localStorage.setItem('userLayoutSettings', JSON.stringify(settings));
    localStorage.setItem('posTableColumnsSettings', JSON.stringify(settings.posTable.columns));
    localStorage.setItem('searchTableColumnsSettings', JSON.stringify(settings.searchTable.columns));
    localStorage.setItem('customColumnHeaders', JSON.stringify(settings.customHeaders));
    
    // إظهار رسالة النجاح
    setTimeout(function() {
        $('.layout-loading').hide();
        toastr.success('تم حفظ الإعدادات بنجاح');
    }, 500);
}

// جمع إعدادات الأعمدة
function collectColumnSettings(containerId) {
    var columns = [];
    var order = 0;
    
    $('#' + containerId + ' .column-item').each(function() {
        var $item = $(this);
        var columnId = $item.data('column-id');
        var isVisible = $item.find('.column-visibility').prop('checked');
        var width = $item.find('.column-width-slider').val();
        var customName = $('.custom-header-input[data-column-id="' + columnId + '"]').val();
        
        columns.push({
            id: columnId,
            name: customName || $item.find('label').text().trim(),
            visible: isVisible,
            width: parseInt(width),
            order: order++
        });
    });
    
    return columns;
}

// جمع العناوين المخصصة
function collectCustomHeaders() {
    var headers = {};
    
    $('.custom-header-input').each(function() {
        var columnId = $(this).data('column-id');
        var customName = $(this).val().trim();
        if (customName) {
            headers[columnId] = customName;
        }
    });
    
    return headers;
}

// تطبيق جميع الإعدادات
function applyAllLayoutSettings() {
    var settings = JSON.parse(localStorage.getItem('userLayoutSettings') || '{}');
    
    if (!settings || Object.keys(settings).length === 0) {
        toastr.warning('لا توجد إعدادات محفوظة');
        return;
    }
    
    $('.layout-loading').show();
    
    // تطبيق الإعدادات العامة
    if (settings.general) {
        applyGeneralSettings(settings.general);
    }
    
    // تطبيق إعدادات جدول المنتجات
    if (settings.posTable) {
        applyPosTableSettings(settings.posTable);
    }
    
    // تطبيق إعدادات جدول البحث
    if (settings.searchTable) {
        applySearchTableSettings(settings.searchTable);
    }
    
    // تطبيق العناوين المخصصة
    if (settings.customHeaders) {
        applyCustomHeaders(settings.customHeaders);
    }
    
    setTimeout(function() {
        $('.layout-loading').hide();
        toastr.success('تم تطبيق الإعدادات بنجاح');
    }, 500);
}

// تطبيق الإعدادات العامة
function applyGeneralSettings(settings) {
    // حجم الخط
    if (settings.globalFontSize) {
        $('body').css('font-size', settings.globalFontSize + 'px');
    }
    
    if (settings.buttonFontSize) {
        $('button, .btn').css('font-size', settings.buttonFontSize + 'px');
    }
    
    // النمط
    if (settings.tableTheme) {
        applyThemeStyles(settings.tableTheme);
    }
    
    // كثافة العرض
    if (settings.displayDensity) {
        var padding = {
            'comfortable': '12px',
            'compact': '6px',
            'spacious': '16px'
        }[settings.displayDensity];
        
        $('#pos_table td, #pos_table th').css('padding', padding);
        $('.product-dropdown td, .product-dropdown th').css('padding', padding);
    }
    
    // نمط الحدود
    if (settings.borderStyle) {
        applyBorderStyle(settings.borderStyle);
    }
}

// تطبيق إعدادات جدول المنتجات
function applyPosTableSettings(settings) {
    // ارتفاع الصف
    if (settings.rowHeight) {
        $('#pos_table tbody tr').css('height', settings.rowHeight + 'px');
        $('#pos_table tbody td').css('height', settings.rowHeight + 'px');
    }
    
    // حجم الخط
    if (settings.fontSize) {
        $('#pos_table').css('font-size', settings.fontSize + 'px');
    }
    
    // عرض الجدول
    if (settings.width) {
        if (settings.width === 'fixed') {
            $('#pos_table').css({
                'width': '1200px',
                'table-layout': 'fixed'
            });
        } else {
            $('#pos_table').css({
                'width': settings.width,
                'table-layout': 'auto'
            });
        }
    }
    
    // تطبيق إعدادات الأعمدة
    if (settings.columns) {
        applyColumnSettings('pos_table', settings.columns);
    }
}

// تطبيق إعدادات جدول البحث
function applySearchTableSettings(settings) {
    // إنشاء أنماط CSS للبحث
    var searchStyles = `
        <style id="searchTableCustomStyles">
            .product-search-table tr {
                height: ${settings.rowHeight}px !important;
            }
            
            .product-search-table {
                font-size: ${settings.fontSize}px !important;
            }
            
            .product-dropdown {
                ${settings.modalWidth === 'full' ? 
                    'left: 10px !important; right: 10px !important; width: auto !important;' : 
                    'width: ' + settings.modalWidth + ' !important;'}
            }
        </style>
    `;
    
    // إزالة الأنماط القديمة وإضافة الجديدة
    $('#searchTableCustomStyles').remove();
    $('head').append(searchStyles);
    
    // حفظ إعدادات الأعمدة للاستخدام عند فتح البحث
    window.searchTableColumnSettings = settings.columns;
}

// تطبيق إعدادات الأعمدة
function applyColumnSettings(tableId, columns) {
    var table = $('#' + tableId);
    
    // إعادة ترتيب الأعمدة
    columns.sort((a, b) => a.order - b.order);
    
    // إنشاء أنماط CSS للأعمدة
    var columnStyles = '<style id="' + tableId + 'ColumnStyles">';
    
    // خريطة الأعمدة لجدول POS
    var columnMap = {
        'serial': 1,
        'product': 2,
        'quantity': 3,
        'unit': 4,
        'price_usd': 5,
        'price_iqd': 6,
        'discount': 7,
        'price_inc_tax': 8,
        'subtotal': 9,
        'warehouse': 10,
        'stock': 11,
        'actions': 12
    };
    
    Object.keys(columnMap).forEach(function(columnId) {
        var column = columns.find(c => c.id === columnId);
        if (column) {
            var index = columnMap[columnId];
            
            // إخفاء/إظهار الأعمدة
            if (!column.visible) {
                columnStyles += `
                    #${tableId} th:nth-child(${index}),
                    #${tableId} td:nth-child(${index}) {
                        display: none !important;
                    }
                `;
            }
            
            // تعيين عرض الأعمدة
            if (column.width && column.visible) {
                columnStyles += `
                    #${tableId} th:nth-child(${index}),
                    #${tableId} td:nth-child(${index}) {
                        width: ${column.width}px !important;
                        max-width: ${column.width}px !important;
                    }
                `;
            }
        }
    });
    
    columnStyles += '</style>';
    
    // إزالة الأنماط القديمة وإضافة الجديدة
    $('#' + tableId + 'ColumnStyles').remove();
    $('head').append(columnStyles);
}

// تطبيق العناوين المخصصة
function applyCustomHeaders(customHeaders) {
    // تطبيق على جدول المنتجات
    var columnMap = {
        'serial': 1,
        'product': 2,
        'quantity': 3,
        'unit': 4,
        'price_usd': 5,
        'price_iqd': 6,
        'discount': 7,
        'price_inc_tax': 8,
        'subtotal': 9,
        'warehouse': 10,
        'stock': 11,
        'actions': 12
    };
    
    Object.keys(customHeaders).forEach(function(columnId) {
        var index = columnMap[columnId];
        if (index) {
            $('#pos_table thead th:nth-child(' + index + ')').text(customHeaders[columnId]);
        }
    });
    
    // حفظ العناوين لتطبيقها على جدول البحث
    window.customSearchHeaders = customHeaders;
}

// تطبيق أنماط النمط
function applyThemeStyles(theme) {
    var themeStyles = {
        'default': {
            headerBg: '#556270',
            headerColor: '#ffffff',
            rowBg: '#ffffff',
            rowAltBg: '#f8f9fa',
            borderColor: '#e0e0e0'
        },
        'dark': {
            headerBg: '#1f2937',
            headerColor: '#f9fafb',
            rowBg: '#374151',
            rowAltBg: '#4b5563',
            borderColor: '#6b7280'
        },
        'light': {
            headerBg: '#ffffff',
            headerColor: '#1f2937',
            rowBg: '#ffffff',
            rowAltBg: '#f3f4f6',
            borderColor: '#d1d5db'
        },
        'blue': {
            headerBg: '#3b82f6',
            headerColor: '#ffffff',
            rowBg: '#eff6ff',
            rowAltBg: '#dbeafe',
            borderColor: '#93c5fd'
        }
    };
    
    var styles = themeStyles[theme] || themeStyles['default'];
    
    var themeCSS = `
        <style id="tableThemeStyles">
            #pos_table th,
            .product-search-table th {
                background: ${styles.headerBg} !important;
                color: ${styles.headerColor} !important;
            }
            
            #pos_table tbody tr:nth-child(odd) td,
            .product-search-table tbody tr:nth-child(odd) {
                background: ${styles.rowBg} !important;
            }
            
            #pos_table tbody tr:nth-child(even) td,
            .product-search-table tbody tr:nth-child(even) {
                background: ${styles.rowAltBg} !important;
            }
            
            #pos_table td,
            #pos_table th,
            .product-search-table td,
            .product-search-table th {
                border-color: ${styles.borderColor} !important;
            }
            
            ${theme === 'dark' ? `
                #pos_table,
                #pos_table input,
                #pos_table select,
                .product-search-table,
                .product-search-table td {
                    color: #f9fafb !important;
                }
                
                #pos_table input,
                #pos_table select {
                    background: #4b5563 !important;
                    border-color: #6b7280 !important;
                }
            ` : ''}
        </style>
    `;
    
    $('#tableThemeStyles').remove();
    $('head').append(themeCSS);
}

// تطبيق نمط الحدود
function applyBorderStyle(style) {
    var borderCSS = `
        <style id="tableBorderStyles">
            ${style === 'none' ? `
                #pos_table td,
                #pos_table th {
                    border: none !important;
                }
            ` : ''}
            
            ${style === 'horizontal' ? `
                #pos_table td,
                #pos_table th {
                    border-left: none !important;
                    border-right: none !important;
                }
            ` : ''}
        </style>
    `;
    
    $('#tableBorderStyles').remove();
    $('head').append(borderCSS);
}

// استعادة الإعدادات الافتراضية
function resetToDefaultSettings() {
    // مسح جميع الإعدادات المحفوظة
    localStorage.removeItem('userLayoutSettings');
    localStorage.removeItem('posTableColumnsSettings');
    localStorage.removeItem('searchTableColumnsSettings');
    localStorage.removeItem('customColumnHeaders');
    
    // إزالة جميع الأنماط المخصصة
    $('#tableThemeStyles').remove();
    $('#tableBorderStyles').remove();
    $('#pos_tableColumnStyles').remove();
    $('#searchTableCustomStyles').remove();
    
    // إعادة تعيين الأنماط الافتراضية
    $('body').css('font-size', '');
    $('button, .btn').css('font-size', '');
    $('#pos_table').css({
        'font-size': '',
        'width': ''
    });
    $('#pos_table tbody tr').css('height', '');
    $('#pos_table td, #pos_table th').css({
        'padding': '',
        'border': ''
    });
    
    // إعادة تحميل الصفحة لضمان إعادة التعيين الكامل
    toastr.success('تم استعادة الإعدادات الافتراضية');
    setTimeout(function() {
        location.reload();
    }, 1000);
}

// تحميل إعدادات المستخدم عند بدء التشغيل
function loadUserLayoutSettings() {
    var settings = JSON.parse(localStorage.getItem('userLayoutSettings') || '{}');
    
    if (settings && Object.keys(settings).length > 0) {
        // تطبيق الإعدادات تلقائياً
        applyAllLayoutSettings();
        
        // تحديث قيم الحقول في النافذة
        if (settings.general) {
            $('#globalFontSize').val(settings.general.globalFontSize || 14);
            $('#buttonFontSize').val(settings.general.buttonFontSize || 13);
            $('#tableTheme').val(settings.general.tableTheme || 'default');
            $('#displayDensity').val(settings.general.displayDensity || 'comfortable');
            $('#borderStyle').val(settings.general.borderStyle || 'all');
        }
        
        if (settings.posTable) {
            $('#posRowHeight').val(settings.posTable.rowHeight || 32);
            $('#posTableFontSize').val(settings.posTable.fontSize || 12);
            $('#posTableWidth').val(settings.posTable.width || 'auto');
        }
        
        if (settings.searchTable) {
            $('#searchRowHeight').val(settings.searchTable.rowHeight || 40);
            $('#searchTableFontSize').val(settings.searchTable.fontSize || 13);
            $('#searchModalWidth').val(settings.searchTable.modalWidth || 'auto');
        }
    }
}

// إظهار رسالة المعاينة
function showPreviewMessage(message) {
    var preview = $('.settings-preview');
    if (preview.length === 0) {
        preview = $('<div class="settings-preview"></div>');
        $('body').append(preview);
    }
    
    preview.text(message).fadeIn(300);
    
    setTimeout(function() {
        preview.fadeOut(300);
    }, 2000);
}

// تحديث ترتيب الأعمدة
function updateColumnOrder(containerId) {
    var columns = collectColumnSettings(containerId);
    
    // حفظ الترتيب الجديد
    if (containerId === 'posTableColumns') {
        localStorage.setItem('posTableColumnsSettings', JSON.stringify(columns));
    } else if (containerId === 'searchTableColumns') {
        localStorage.setItem('searchTableColumnsSettings', JSON.stringify(columns));
    }
}

// تعديل دالة showProductDropdown لتطبيق الإعدادات المخصصة
$(document).ready(function() {
    // حفظ الدالة الأصلية إذا كانت موجودة
    if (typeof window.showProductDropdown === 'function') {
        window.originalShowProductDropdown = window.showProductDropdown;
        
        // تعديل الدالة
        window.showProductDropdown = function(input, products, row, rowIndex) {
            // استدعاء الدالة الأصلية
            if (window.originalShowProductDropdown) {
                window.originalShowProductDropdown.apply(this, arguments);
            }
            
            // تطبيق إعدادات الأعمدة المخصصة
            setTimeout(function() {
                if (window.searchTableColumnSettings) {
                    applySearchTableColumnsOnDropdown(window.searchTableColumnSettings);
                }
                
                if (window.customSearchHeaders) {
                    applyCustomHeadersOnDropdown(window.customSearchHeaders);
                }
            }, 100);
        };
    }
});

// تطبيق إعدادات أعمدة جدول البحث على القائمة المنسدلة
function applySearchTableColumnsOnDropdown(columns) {
    var dropdown = $('.product-dropdown');
    if (dropdown.length === 0) return;
    
    var table = dropdown.find('.product-search-table');
    var headerRow = table.find('thead tr');
    var bodyRows = table.find('tbody tr');
    
    // إنشاء خريطة للأعمدة
    var columnMap = {
        'product_name': 0,
        'category': 1,
        'foreign_name': 2,
        'sku': 3,
        'price_usd': 4,
        'price_iqd': 5,
        'discount': 6,
        'final_price': 7,
        'uom': 8,
        'total_stock': 9,
        'locations': 10
    };
    
    // تطبيق الترتيب والإخفاء/الإظهار
    columns.forEach(function(column) {
        var colIndex = columnMap[column.id];
        if (colIndex !== undefined) {
            // إخفاء/إظهار العمود
            if (!column.visible) {
                headerRow.find('th').eq(colIndex).hide();
                bodyRows.each(function() {
                    $(this).find('td').eq(colIndex).hide();
                });
            }
            
            // تطبيق العرض
            if (column.width) {
                headerRow.find('th').eq(colIndex).css('width', column.width + 'px');
            }
        }
    });
}

// تطبيق العناوين المخصصة على القائمة المنسدلة
function applyCustomHeadersOnDropdown(customHeaders) {
    var dropdown = $('.product-dropdown');
    if (dropdown.length === 0) return;
    
    var headerMap = {
        'product_name': 0,
        'category': 1,
        'foreign_name': 2,
        'sku': 3,
        'price_usd': 4,
        'price_iqd': 5,
        'discount': 6,
        'final_price': 7,
        'uom': 8,
        'total_stock': 9,
        'locations': 10
    };
    
    var headerRow = dropdown.find('.product-search-table thead tr');
    
    Object.keys(customHeaders).forEach(function(columnId) {
        var colIndex = headerMap[columnId];
        if (colIndex !== undefined) {
            headerRow.find('th').eq(colIndex).text(customHeaders[columnId]);
        }
    });
}

// إضافة مكتبة Sortable إذا لم تكن محملة
$(document).ready(function() {
    if (typeof Sortable === 'undefined') {
        var script = document.createElement('script');
        script.src = 'https://cdn.jsdelivr.net/npm/sortablejs@latest/Sortable.min.js';
        script.onload = function() {
            console.log('Sortable.js loaded successfully');
        };
        document.head.appendChild(script);
    }
});

// Note: Loading indicators removed as pagination is no longer used

// Simple keyboard navigation for single scrollable list
function setupSimpleKeyboardNavigation(dropdown, input, row, rowIndex, products) {
    var currentIndex = 0;
    var $rows = dropdown.find('tbody tr');
    var $tableWrapper = dropdown.find('.table-wrapper');
    
    // Highlight first row immediately
    if ($rows.length > 0) {
        highlightDropdownRow($rows, currentIndex);
        ensureRowVisible($tableWrapper, $rows.eq(currentIndex));
    }
    
    // Keyboard handler
    var keyHandler = function(e) {
        if (!dropdown.is(':visible')) {
            $(document).off('keydown.productDropdown');
            return;
        }
        
        var handled = false;
        
        switch(e.keyCode) {
            case 40: // Arrow Down
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (currentIndex < $rows.length - 1) {
                    currentIndex++;
                } else {
                    currentIndex = 0; // Wrap to beginning
                }
                
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible($tableWrapper, $rows.eq(currentIndex));
                break;
                
            case 38: // Arrow Up
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (currentIndex > 0) {
                    currentIndex--;
                } else {
                    currentIndex = $rows.length - 1; // Wrap to end
                }
                
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible($tableWrapper, $rows.eq(currentIndex));
                break;
                
            case 33: // Page Up
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                currentIndex = Math.max(0, currentIndex - 10);
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible($tableWrapper, $rows.eq(currentIndex));
                break;
                
            case 34: // Page Down
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                currentIndex = Math.min($rows.length - 1, currentIndex + 10);
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible($tableWrapper, $rows.eq(currentIndex));
                break;
                
            case 36: // Home
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                currentIndex = 0;
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible($tableWrapper, $rows.eq(currentIndex));
                break;
                
            case 35: // End
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                currentIndex = $rows.length - 1;
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible($tableWrapper, $rows.eq(currentIndex));
                break;
                
            case 13: // Enter
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (currentIndex >= 0 && currentIndex < $rows.length) {
                    var selectedProduct = $rows.eq(currentIndex).data('product-data');
                    if (selectedProduct) {
                        selectProduct(selectedProduct);
                    }
                }
                break;
                
            case 27: // Escape
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                closeDropdown();
                break;
                
            case 9: // Tab
                closeDropdown();
                break;
        }
        
        if (handled) {
            return false;
        }
    };
    
    // Ensure row is visible in the scrollable container
    function ensureRowVisible($container, $row) {
        if (!$row || $row.length === 0) return;
        
        var containerTop = $container.offset().top;
        var containerBottom = containerTop + $container.height();
        var rowTop = $row.offset().top;
        var rowBottom = rowTop + $row.outerHeight();
        
        // Account for sticky header height
        var headerHeight = $container.find('thead').outerHeight() || 0;
        
        if (rowTop < containerTop + headerHeight) {
            // Row is above visible area
            var scrollTop = $container.scrollTop() - (containerTop + headerHeight - rowTop) - 10;
            $container.scrollTop(scrollTop);
        } else if (rowBottom > containerBottom) {
            // Row is below visible area
            var scrollTop = $container.scrollTop() + (rowBottom - containerBottom) + 10;
            $container.scrollTop(scrollTop);
        }
    }
    
    // Add keyboard event handler
    $(document).off('keydown.productDropdown').on('keydown.productDropdown', keyHandler);
    
    // Mouse events
    $rows.off('click').on('click', function() {
        var selectedProduct = $(this).data('product-data');
        if (selectedProduct) {
            selectProduct(selectedProduct);
        }
    });
    
    $rows.off('mouseenter').on('mouseenter', function() {
        currentIndex = $(this).index();
        highlightDropdownRow($rows, currentIndex);
    });
    
    // Select product function
    function selectProduct(product) {
        populateRowWithProduct(row, product, rowIndex);
        closeDropdown();
        
        // Apply unit filter if active
        if (window.activeUnitFilter) {
            console.log('🎯 Applying unit filter after product selection:', window.activeUnitFilter);
            setTimeout(function() {
                if (typeof window.applyUnitFilterToRow === 'function') {
                    window.applyUnitFilterToRow(row, window.activeUnitFilter);
                } else if (typeof applyUnitFilterToRow === 'function') {
                    applyUnitFilterToRow(row, window.activeUnitFilter);
                } else {
                    console.error('❌ applyUnitFilterToRow function not found!');
                }
            }, 150);
        } else {
            console.log('ℹ️ No active unit filter to apply');
        }
        
        setTimeout(function() {
            row.find('.warehouse-input').focus().select();
        }, 100);
    }
    
    // Close dropdown function
    function closeDropdown() {
        // إزالة مستمعات منع فقدان التركيز
        $(document).off('focusout.searchTable');
        dropdown.off('mousedown.keepFocus');
        dropdown.off('wheel.keepFocus');
        dropdown.find('tbody tr').off('mousedown.keepRowFocus');
        dropdown.find('thead th').off('mousedown.keepHeaderFocus');
        
        dropdown.remove();
        $(document).off('keydown.productDropdown');
        $(document).off('click.productDropdown');
        $(window).off('resize.productDropdown');
        input.removeClass('has-results');
        input.val('').focus();
    }
    
    // Click outside to close
    $(document).off('click.productDropdown').on('click.productDropdown', function(e) {
        if (!$(e.target).closest('.product-dropdown, .product-search-input').length) {
            closeDropdown();
        }
    });
    
    // Window resize handler
    $(window).off('resize.productDropdown').on('resize.productDropdown', function() {
        dropdown.css({
            top: input.offset().top + input.outerHeight(),
            left: '10px',
            right: '10px'
        });
    });
}

// دالة لمنع فقدان التركيز من جدول البحث
function preventFocusLoss($tableWrapper, dropdown) {
    // منع فقدان التركيز عند النقر خارج الجدول
    $(document).off('focusout.searchTable').on('focusout.searchTable', function(e) {
        // تحقق من أن الجدول مفتوح
        if (!dropdown.is(':visible')) {
            $(document).off('focusout.searchTable');
            return;
        }
        
        // تحقق من أن العنصر الجديد المركز عليه ليس جزءاً من الجدول
        var newFocusElement = e.relatedTarget;
        if (!newFocusElement || !$(newFocusElement).closest('.product-dropdown').length) {
            // إعادة التركيز للجدول بعد تأخير قصير
            setTimeout(function() {
                if (dropdown.is(':visible')) {
                    $tableWrapper.focus();
                }
            }, 10);
        }
    });
    
    // منع فقدان التركيز عند النقر داخل الجدول
    dropdown.off('mousedown.keepFocus').on('mousedown.keepFocus', function(e) {
        // السماح بالنقر على أزرار الإغلاق والتنقل
        if ($(e.target).closest('.close-search, .navigation-hints').length) {
            return;
        }
        e.preventDefault(); // منع فقدان التركيز
        $tableWrapper.focus();
    });
    
    // التأكد من بقاء التركيز عند النقر على الصفوف
    dropdown.find('tbody tr').off('mousedown.keepRowFocus').on('mousedown.keepRowFocus', function(e) {
        e.preventDefault();
        $tableWrapper.focus();
    });
    
    // التأكد من بقاء التركيز عند النقر على رأس الجدول
    dropdown.find('thead th').off('mousedown.keepHeaderFocus').on('mousedown.keepHeaderFocus', function(e) {
        e.preventDefault();
        $tableWrapper.focus();
    });
    
    // إضافة مستمع لمنع فقدان التركيز عند استخدام العجلة
    dropdown.off('wheel.keepFocus').on('wheel.keepFocus', function() {
        setTimeout(function() {
            if (dropdown.is(':visible')) {
                $tableWrapper.focus();
            }
        }, 10);
    });
    
    // إزالة المستمعات عند إغلاق الجدول
    dropdown.on('remove', function() {
        $(document).off('focusout.searchTable');
        dropdown.off('mousedown.keepFocus');
        dropdown.off('wheel.keepFocus');
        dropdown.find('tbody tr').off('mousedown.keepRowFocus');
        dropdown.find('thead th').off('mousedown.keepHeaderFocus');
    });
}

// Enhanced keyboard navigation with pagination support (keeping for compatibility)
function setupEnhancedKeyboardNavigation(dropdown, input, row, rowIndex, products, paginationData) {
    var currentIndex = 0;
    var $rows = dropdown.find('tbody tr');
    var pageSize = 10; // Number of visible rows before scrolling
    
    // Extract pagination data and make it accessible
    var currentPage = paginationData ? paginationData.currentPage : 0;
    var itemsPerPage = paginationData ? paginationData.itemsPerPage : products.length;
    var totalPages = paginationData ? paginationData.totalPages : 1;
    var allProducts = paginationData ? paginationData.allProducts : products;
    var renderFunction = paginationData ? paginationData.renderFunction : null;
    
    // Store reference to dropdown for updates
    dropdown.data('paginationData', paginationData);
    
    // Handle initial selection
    if (paginationData && paginationData.selectLast) {
        currentIndex = $rows.length - 1;
    }
    
    // Highlight selected row immediately and ensure it's visible
    if ($rows.length > 0) {
        highlightDropdownRow($rows, currentIndex);
        ensureRowVisible(dropdown, $rows.eq(currentIndex));
    }
    
    // Keyboard handler
    var keyHandler = function(e) {
        if (!dropdown.is(':visible')) {
            $(document).off('keydown.productDropdown');
            return;
        }
        
        var handled = false;
        
        switch(e.keyCode) {
            case 40: // Arrow Down
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (currentIndex < $rows.length - 1) {
                    currentIndex++;
                    highlightDropdownRow($rows, currentIndex);
                    ensureRowVisible(dropdown, $rows.eq(currentIndex));
                } else if (paginationData && paginationData.currentPage < paginationData.totalPages - 1) {
                    // Load next page
                    var nextPage = paginationData.currentPage + 1;
                    paginationData.currentPage = nextPage;
                    
                    console.log('Moving to next page:', nextPage + 1, 'of', paginationData.totalPages);
                    
                    // Show loading indicator
                    showLoadingIndicator(dropdown);
                    
                    renderFunction(paginationData.allProducts || allProducts, nextPage);
                    
                    // Re-setup navigation with new page
                    setTimeout(function() {
                        hideLoadingIndicator(dropdown);
                        setupEnhancedKeyboardNavigation(dropdown, input, row, rowIndex, products, paginationData);
                    }, 50);
                } else {
                    // Wrap to beginning of first page
                    if (paginationData && paginationData.currentPage > 0) {
                        paginationData.currentPage = 0;
                        
                        showLoadingIndicator(dropdown);
                        renderFunction(paginationData.allProducts || allProducts, 0);
                        
                        setTimeout(function() {
                            hideLoadingIndicator(dropdown);
                            setupEnhancedKeyboardNavigation(dropdown, input, row, rowIndex, products, paginationData);
                        }, 50);
                    } else {
                        currentIndex = 0;
                        highlightDropdownRow($rows, currentIndex);
                        ensureRowVisible(dropdown, $rows.eq(currentIndex));
                    }
                }
                break;
                
            case 38: // Arrow Up
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (currentIndex > 0) {
                    currentIndex--;
                    highlightDropdownRow($rows, currentIndex);
                    ensureRowVisible(dropdown, $rows.eq(currentIndex));
                } else if (paginationData && paginationData.currentPage > 0) {
                    // Load previous page
                    var prevPage = paginationData.currentPage - 1;
                    paginationData.currentPage = prevPage;
                    
                    showLoadingIndicator(dropdown);
                    renderFunction(paginationData.allProducts || allProducts, prevPage);
                    
                    // Re-setup navigation with new page and select last item
                    setTimeout(function() {
                        hideLoadingIndicator(dropdown);
                        var newPaginationData = Object.assign({}, paginationData);
                        newPaginationData.selectLast = true;
                        setupEnhancedKeyboardNavigation(dropdown, input, row, rowIndex, products, newPaginationData);
                    }, 50);
                } else {
                    // Wrap to end of last page
                    if (paginationData && paginationData.totalPages > 1) {
                        var lastPage = paginationData.totalPages - 1;
                        paginationData.currentPage = lastPage;
                        
                        showLoadingIndicator(dropdown);
                        renderFunction(paginationData.allProducts || allProducts, lastPage);
                        
                        setTimeout(function() {
                            hideLoadingIndicator(dropdown);
                            var newPaginationData = Object.assign({}, paginationData);
                            newPaginationData.selectLast = true;
                            setupEnhancedKeyboardNavigation(dropdown, input, row, rowIndex, products, newPaginationData);
                        }, 50);
                    } else {
                        currentIndex = $rows.length - 1;
                        highlightDropdownRow($rows, currentIndex);
                        ensureRowVisible(dropdown, $rows.eq(currentIndex));
                    }
                }
                break;
                
            case 33: // Page Up
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (paginationData && paginationData.currentPage > 0) {
                    // Go to previous page
                    var prevPage = paginationData.currentPage - 1;
                    paginationData.currentPage = prevPage;
                    
                    showLoadingIndicator(dropdown);
                    renderFunction(paginationData.allProducts || allProducts, prevPage);
                    
                    setTimeout(function() {
                        hideLoadingIndicator(dropdown);
                        setupEnhancedKeyboardNavigation(dropdown, input, row, rowIndex, products, paginationData);
                    }, 50);
                } else {
                    currentIndex = 0;
                    highlightDropdownRow($rows, currentIndex);
                    ensureRowVisible(dropdown, $rows.eq(currentIndex));
                }
                break;
                
            case 34: // Page Down
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (paginationData && paginationData.currentPage < paginationData.totalPages - 1) {
                    // Go to next page
                    var nextPage = paginationData.currentPage + 1;
                    paginationData.currentPage = nextPage;
                    
                    showLoadingIndicator(dropdown);
                    renderFunction(paginationData.allProducts || allProducts, nextPage);
                    
                    setTimeout(function() {
                        hideLoadingIndicator(dropdown);
                        setupEnhancedKeyboardNavigation(dropdown, input, row, rowIndex, products, paginationData);
                    }, 50);
                } else {
                    currentIndex = $rows.length - 1;
                    highlightDropdownRow($rows, currentIndex);
                    ensureRowVisible(dropdown, $rows.eq(currentIndex));
                }
                break;
                
            case 36: // Home
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                currentIndex = 0;
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible(dropdown, $rows.eq(currentIndex));
                break;
                
            case 35: // End
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                currentIndex = $rows.length - 1;
                highlightDropdownRow($rows, currentIndex);
                ensureRowVisible(dropdown, $rows.eq(currentIndex));
                break;
                
            case 13: // Enter
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                
                if (currentIndex >= 0 && currentIndex < $rows.length) {
                    var selectedProduct = $rows.eq(currentIndex).data('product-data');
                    if (selectedProduct) {
                        selectProduct(selectedProduct);
                    }
                }
                break;
                
            case 27: // Escape
                e.preventDefault();
                e.stopPropagation();
                handled = true;
                closeDropdown();
                break;
                
            case 9: // Tab
                // Allow default tab behavior to move to next field
                closeDropdown();
                break;
        }
        
        // Prevent default behavior for handled keys
        if (handled) {
            return false;
        }
    };
    
    // Ensure row is visible in the viewport
    function ensureRowVisible(dropdown, $row) {
        if (!$row || $row.length === 0) return;
        
        var dropdownScrollTop = dropdown.scrollTop();
        var dropdownHeight = dropdown.height();
        var dropdownTop = dropdown.offset().top;
        
        // Account for controls container height
        var controlsHeight = dropdown.find('.product-search-controls').outerHeight() || 0;
        
        var rowTop = $row.position().top + controlsHeight;
        var rowHeight = $row.outerHeight();
        var rowBottom = rowTop + rowHeight;
        
        // Scroll if row is above visible area
        if (rowTop < controlsHeight) {
            dropdown.scrollTop(dropdownScrollTop + rowTop - controlsHeight - 10);
        }
        // Scroll if row is below visible area
        else if (rowBottom > dropdownHeight) {
            dropdown.scrollTop(dropdownScrollTop + rowBottom - dropdownHeight + 10);
        }
    }
    
    // Add keyboard event handler
    $(document).off('keydown.productDropdown').on('keydown.productDropdown', keyHandler);
    
    // Mouse events
    $rows.off('click').on('click', function() {
        var selectedProduct = $(this).data('product-data');
        if (selectedProduct) {
            selectProduct(selectedProduct);
        }
    });
    
    $rows.off('mouseenter').on('mouseenter', function() {
        currentIndex = $(this).index();
        highlightDropdownRow($rows, currentIndex);
    });
    
    // Select product function
    function selectProduct(product) {
        populateRowWithProduct(row, product, rowIndex);
        closeDropdown();
        
        // Focus on quantity field after selection
        setTimeout(function() {
            row.find('.warehouse-input').focus().select();
        }, 100);
    }
    
    // Close dropdown function
    function closeDropdown() {
        // إزالة مستمعات منع فقدان التركيز
        $(document).off('focusout.searchTable');
        dropdown.off('mousedown.keepFocus');
        dropdown.off('wheel.keepFocus');
        dropdown.find('tbody tr').off('mousedown.keepRowFocus');
        dropdown.find('thead th').off('mousedown.keepHeaderFocus');
        
        dropdown.remove();
        $(document).off('keydown.productDropdown');
        $(document).off('click.productDropdown');
        $(window).off('resize.productDropdown');
        input.removeClass('has-results');
        input.val('').focus();
    }
    
    // Click outside to close
    $(document).off('click.productDropdown').on('click.productDropdown', function(e) {
        if (!$(e.target).closest('.product-dropdown, .product-search-input').length) {
            closeDropdown();
        }
    });
    
    // Window resize handler
    $(window).off('resize.productDropdown').on('resize.productDropdown', function() {
        dropdown.css({
            top: input.offset().top + input.outerHeight(),
            left: '10px',
            right: '10px'
        });
    });
}


function enhanceProductDropdownWithCustomLayout() {
    // حفظ الدالة الأصلية
    if (typeof window.showProductDropdown === 'function' && !window.originalShowProductDropdown) {
        window.originalShowProductDropdown = window.showProductDropdown;
    }
    
    // استبدال الدالة
    window.showProductDropdown = function(input, products, row, rowIndex) {
        $('.product-dropdown').remove();

        var currentProducts = products; // Store current products (may be sorted)

        var dropdown = $('<div class="product-dropdown product-search-container"></div>');
        dropdown.css({
            position: 'fixed',
            top: input.offset().top + input.outerHeight(),
            left: '10px',
            right: '10px',
            width: 'auto',
            maxHeight: '600px',
            overflow: 'visible', // For sticky header
            background: 'white',
            border: '2px solid #d1d5db',
            borderRadius: '8px',
            boxShadow: '0 10px 30px rgba(0, 0, 0, 0.2)',
            zIndex: 1000
        });

        // Add sorting controls container
        var $controlsContainer = $('<div class="product-search-controls"></div>');
        $controlsContainer.css({
            padding: '10px',
            background: '#f3f4f6',
            borderBottom: '1px solid #e5e7eb',
            display: 'flex',
            justifyContent: 'space-between',
            alignItems: 'center'
        });

        // Add instructions
        var $instructions = $('<div class="search-instructions"></div>');
        $instructions.css({
            fontSize: '13px',
            color: '#374151',
            fontWeight: '600'
        });
        $instructions.html('↑↓ Navigate • Enter Select • Esc Close');

        // Add sorting dropdown
        var $sortContainer = $('<div class="sort-container"></div>');
        $sortContainer.css({
            display: 'flex',
            alignItems: 'center',
            gap: '10px'
        });

        var $sortLabel = $('<label></label>');
        $sortLabel.css({
            fontSize: '13px',
            color: '#374151',
            fontWeight: '600',
            margin: '0'
        });
        $sortLabel.text('Sort by:');

        var $sortSelect = $('<select class="product-sort-select"></select>');
        $sortSelect.css({
            padding: '5px 10px',
            borderRadius: '4px',
            border: '1px solid #d1d5db',
            fontSize: '13px',
            background: 'white',
            cursor: 'pointer'
        });

        // Add sort options
        $sortSelect.append('<option value="name_asc">Name (A-Z)</option>');
        $sortSelect.append('<option value="name_desc">Name (Z-A)</option>');
        $sortSelect.append('<option value="unit_asc">Unit (A-Z)</option>');
        $sortSelect.append('<option value="unit_desc">Unit (Z-A)</option>');
        $sortSelect.append('<option value="price_asc">Price (Low to High)</option>');
        $sortSelect.append('<option value="price_desc">Price (High to Low)</option>');

        $sortContainer.append($sortLabel);
        $sortContainer.append($sortSelect);

        $controlsContainer.append($instructions);
        $controlsContainer.append($sortContainer);

        dropdown.append($controlsContainer);

        // Create table wrapper for better sticky header handling
        var $tableWrapper = $('<div class="table-wrapper"></div>');
        $tableWrapper.css({
            'position': 'relative',
            'overflow-y': 'auto',
            'overflow-x': 'auto',
            'width': '100%',
          //  'max-height': '500px' // Limit height for scrolling
        });

        var $table = $('<table class="product-search-table table table-bordered table-striped">');
        $table.css({
            'table-layout': 'fixed',
            'width': '100%',
            'min-width': '1200px',
            'border-collapse': 'separate',
            'border-spacing': '0',
            'margin': '0',
            'font-size': '12px'
        });
        
        var $thead = $('<thead>').appendTo($table);
        var $tbody = $('<tbody>').appendTo($table);

        // الحصول على إعدادات الأعمدة المحفوظة
        var columnSettings = window.searchTableColumnSettings || getDefaultSearchColumns();
        var customHeaders = window.customSearchHeaders || {};
        
        // ترتيب الأعمدة حسب الإعدادات المحفوظة
        columnSettings.sort((a, b) => a.order - b.order);
        
        // بناء رأس الجدول مع الترتيب المخصص
        var headerRow = $('<tr>');
        var columnIndexMap = {}; // لتتبع موقع كل عمود
        
        columnSettings.forEach(function(colSetting, newIndex) {
            if (colSetting.visible) {
                var headerText = customHeaders[colSetting.id] || colSetting.name;
                var $th = $('<th>')
                    .text(headerText)
                    .css({
                        'width': colSetting.width + 'px',
                        'padding': '12px',
                        'background': '#1a365d',
                        'color': 'white',
                        'font-weight': '600',
                        'text-align': getColumnAlignment(colSetting.id)
                    });
                headerRow.append($th);
                columnIndexMap[colSetting.id] = newIndex;
            }
        });
        $thead.append(headerRow);

        // Function to render all table rows
        function renderTableRows(productsToRender) {
            $tbody.empty();
            
            productsToRender.forEach(function(product, index) {
                var $tr = $('<tr style="cursor: pointer; transition: all 0.2s;">');
                $tr.attr('data-index', index);
                
                // إضافة البيانات بنفس ترتيب الأعمدة
                columnSettings.forEach(function(colSetting) {
                    if (colSetting.visible) {
                        var $td = createTableCell(product, colSetting.id);
                        $td.css('width', colSetting.width + 'px');
                        $tr.append($td);
                    }
                });
                
                // حفظ بيانات المنتج
                $tr.data('product-data', product);
                $tr.addClass('ui-menu-item');
                
                $tbody.append($tr);
            });
            
            // Update product count
            updateProductCount();
        }
        
        // Function to update product count
        function updateProductCount() {
            var $productCount = dropdown.find('.product-count');
            if ($productCount.length === 0) {
                $productCount = $('<div class="product-count"></div>');
                $productCount.css({
                    fontSize: '12px',
                    color: '#6b7280',
                    marginLeft: '10px'
                });
                dropdown.find('.search-instructions').after($productCount);
            }
            
            $productCount.text(`${currentProducts.length} products found`);
        }

        // Initial render
        renderTableRows(products);

        // Sort functionality
        $sortSelect.on('change', function() {
            var sortValue = $(this).val();
            var sortedProducts = [...products]; // Create a copy

            switch(sortValue) {
                case 'name_asc':
                    sortedProducts.sort((a, b) => (a.name || '').localeCompare(b.name || ''));
                    break;
                case 'name_desc':
                    sortedProducts.sort((a, b) => (b.name || '').localeCompare(a.name || ''));
                    break;
                case 'unit_asc':
                    sortedProducts.sort((a, b) => (a.unit || '').localeCompare(b.unit || ''));
                    break;
                case 'unit_desc':
                    sortedProducts.sort((a, b) => (b.unit || '').localeCompare(a.unit || ''));
                    break;
                case 'price_asc':
                    sortedProducts.sort((a, b) => {
                        var priceA = parseFloat(a.selling_price) || 0;
                        var priceB = parseFloat(b.selling_price) || 0;
                        return priceA - priceB;
                    });
                    break;
                case 'price_desc':
                    sortedProducts.sort((a, b) => {
                        var priceA = parseFloat(a.selling_price) || 0;
                        var priceB = parseFloat(b.selling_price) || 0;
                        return priceB - priceA;
                    });
                    break;
            }

            currentProducts = sortedProducts;
            renderTableRows(currentProducts);
            
            // Re-setup keyboard navigation after re-rendering
            setupSimpleKeyboardNavigation(dropdown, input, row, rowIndex, currentProducts);
        });

        $tableWrapper.append($table);
        dropdown.append($tableWrapper);
        $('body').append(dropdown);

        // Set focus to the table wrapper for better scrolling
        $tableWrapper.attr('tabindex', '-1');
        setTimeout(function() {
            $tableWrapper.focus();
            // إضافة مستمع لمنع فقدان التركيز من الجدول
            preventFocusLoss($tableWrapper, dropdown);
        }, 50);

        // Setup simple keyboard navigation
        setupSimpleKeyboardNavigation(dropdown, input, row, rowIndex, currentProducts);
        
        // تطبيق الإعدادات الإضافية
        applySearchTableCustomSettings(dropdown);
    };
}

// دالة لإنشاء خلية الجدول حسب نوع العمود

// دالة للحصول على محاذاة العمود
function getColumnAlignment(columnId) {
    var rightAlignColumns = ['price_usd', 'price_iqd', 'final_price'];
    var centerAlignColumns = ['sku', 'discount', 'uom', 'total_stock'];
    
    if (rightAlignColumns.includes(columnId)) {
        return 'right';
    } else if (centerAlignColumns.includes(columnId)) {
        return 'center';
    }
    return 'left';
}

// دالة للحصول على الأعمدة الافتراضية
function getDefaultSearchColumns() {
    return [
           { id: 'product_name', name: 'Product', visible: true, width: 200, order: 0 },
        { id: 'category', name: 'Category', visible: true, width: 120, order: 3},
        { id: 'foreign_name', name: 'Foreign Name', visible: true, width: 150, order: 1 },
        { id: 'sku', name: 'SKU', visible: true, width: 100, order: 6 },
        { id: 'price_usd', name: 'Price (USD)', visible: true, width: 100, order: 4 },
        { id: 'price_iqd', name: 'Price (IQD)', visible: true, width: 120, order: 5 },
        { id: 'discount', name: 'Discount', visible: true, width: 80, order: 9 },
        { id: 'final_price', name: 'Final Price', visible: true, width: 120, order:10 },
        { id: 'uom', name: 'UOM', visible: true, width: 80, order: 2 },
        { id: 'total_stock', name: 'Total Stock', visible: true, width: 100, order: 8 },
        { id: 'locations', name: 'All Locations Stock', visible: true, width: 250, order: 7 }
    ];
}

// تطبيق الإعدادات المخصصة على جدول البحث
function applySearchTableCustomSettings(dropdown) {
    // تطبيق إعدادات الحجم والنمط من الإعدادات المحفوظة
    var settings = JSON.parse(localStorage.getItem('userLayoutSettings') || '{}');
    
    if (settings.searchTable) {
        // ارتفاع الصف
        if (settings.searchTable.rowHeight) {
            dropdown.find('tr').css('height', settings.searchTable.rowHeight + 'px');
        }
        
        // حجم الخط
        if (settings.searchTable.fontSize) {
            dropdown.find('.product-search-table').css('font-size', settings.searchTable.fontSize + 'px');
        }
        
        // عرض النافذة
        if (settings.searchTable.modalWidth) {
            if (settings.searchTable.modalWidth === 'full') {
                dropdown.css({
                    'left': '10px',
                    'right': '10px',
                    'width': 'auto'
                });
            } else if (settings.searchTable.modalWidth !== 'auto') {
                dropdown.css('width', settings.searchTable.modalWidth);
            }
        }
    }
    
    // تطبيق النمط العام
    if (settings.general) {
        if (settings.general.displayDensity) {
            var padding = {
                'comfortable': '12px',
                'compact': '6px',
                'spacious': '16px'
            }[settings.general.displayDensity];
            
            dropdown.find('td, th').css('padding', padding);
        }
    }
}

// تحديث دالة حفظ ترتيب الأعمدة
function updateSearchColumnOrder() {
    var columns = [];
    var order = 0;
    
    $('#searchTableColumns .column-item').each(function() {
        var $item = $(this);
        var columnId = $item.data('column-id');
        
        columns.push({
            id: columnId,
            name: $item.find('label').text().trim(),
            visible: $item.find('.column-visibility').prop('checked'),
            width: parseInt($item.find('.column-width-slider').val()),
            order: order++
        });
    });
    
    // حفظ الترتيب الجديد
    window.searchTableColumnSettings = columns;
    localStorage.setItem('searchTableColumnsSettings', JSON.stringify(columns));
}

// إضافة معالج لحفظ الترتيب عند السحب والإفلات
$(document).ready(function() {
    // الانتظار حتى يتم تحميل النظام
    setTimeout(function() {
        // تحسين معالج السحب والإفلات لأعمدة البحث
        $(document).on('sortupdate', '#searchTableColumns', function(event, ui) {
            updateSearchColumnOrder();
        });
        
        // استبدال دالة showProductDropdown
        enhanceProductDropdownWithCustomLayout();
    }, 2000);
});

// إضافة أنماط CSS إضافية للجدول المحسن
var enhancedSearchTableStyles = `
<style>
/* تحسين مظهر جدول البحث مع تأثيرات أوضح */
.product-search-table {
    border-collapse: separate !important;
    border-spacing: 0 !important;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    border-radius: 8px;
    overflow: hidden;
}

.product-search-table thead th {
    position: sticky;
    top: 0;
    z-index: 10;
    background: linear-gradient(135deg, #2196f3, #1976d2);
    color: white !important;
    box-shadow: 0 4px 8px rgba(0,0,0,0.15);
    font-weight: 600;
    letter-spacing: 0.5px;
    text-shadow: 0 1px 2px rgba(0,0,0,0.2);
    padding: 12px 8px;
}

.product-search-table tbody tr {
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    cursor: pointer;
    border-bottom: 1px solid rgba(0,0,0,0.08);
}

/* تأثير التمرير العادي - أكثر وضوحاً */
.product-search-table tbody tr:hover {
    background: linear-gradient(135deg, #e3f2fd, #bbdefb) !important;
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(33, 150, 243, 0.15);
    border-radius: 6px;
    position: relative;
    z-index: 5;
}

/* تأثير إضافي للخلايا عند التمرير */
.product-search-table tbody tr:hover td {
    color: #1565c0 !important;
    font-weight: 500;
    text-shadow: 0 1px 2px rgba(0,0,0,0.1);
}

/* الصف المحدد - تأثير أقوى */
.product-search-table tbody tr.highlighted {
    background: linear-gradient(135deg, #2196f3, #1976d2) !important;
    color: black !important;
    transform: scale(1.02);
    box-shadow: 0 12px 30px rgba(33, 150, 243, 0.3);
    border-radius: 8px;
    position: relative;
    z-index: 10;
    animation: highlight-pulse 2s ease-in-out infinite;
}

.product-search-table tbody tr.highlighted td {
    color: black !important;
    font-weight: 600;
    text-shadow: 0 1px 3px rgba(0,0,0,0.3);
}

.product-search-table tbody tr.highlighted .badge {
    background-color: #6c757d !important;
    color: black !important;
    border: 3px solid rgba(255,255,255,0.3);
    text-shadow: none;
    font-weight: 600;
}

/* انيميشن للصف المحدد */
@keyframes highlight-pulse {
    0%, 100% { 
        box-shadow: 0 12px 30px rgba(33, 150, 243, 0.3);
    }
    50% { 
        box-shadow: 0 15px 40px rgba(33, 150, 243, 0.4);
    }
}

/* تحسين شكل الـ badges - مربعة بلون واحد */
.product-search-table .badge {
    font-weight: 500;
    padding: 6px 12px;
    font-size: 11px;
    border-radius: 4px;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    background: #6c757d !important;
    color: black !important;
    border: 2px solid #6c757d;
    position: relative;
    z-index: 2;
}

.product-search-table tbody tr:hover .badge {
    transform: scale(1.15);
    font-weight: 700;
    font-size: 12px;
    border-width: 3px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
    color: black !important;
}

.product-search-table .badge-success,
.product-search-table .badge-warning,
.product-search-table .badge-danger,
.product-search-table .badge-secondary,
.product-search-table .badge-dark {
    background: #6c757d !important;
    color: black !important;
    border-color: #6c757d;
}

/* تحسين عرض المواقع - مربعة بلون واحد */
.locations-grid {
    max-width: 100%;
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.location-badge {
    white-space: nowrap;
    display: inline-flex;
    align-items: center;
    transition: all 0.3s ease;
    border-radius: 4px;
    font-weight: 500;
    position: relative;
    z-index: 2;
    overflow: hidden;
    background: transparent !important;
    border: 2px solid #6c757d;
    padding: 0;
}

.location-badge .first-part {
    background: #6c757d;
    color: white;
    font-weight: bold;
    width:20px;
  
    display: inline-block;
}

.location-badge .second-part {
  
    color: #8a0c0cff;
    display: inline-block;
    border-left: 2px solid #6c757d;
    font-weight: bold;
}

.location-badge:hover {
    transform: scale(1.15);
    font-weight: 700;
    border-width: 3px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
    color: black !important;
}

/* تأثير التحميل المحسن */
.product-dropdown.loading::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, 
        transparent 0%, 
        #2196f3 25%, 
        #4caf50 50%, 
        #ff9800 75%, 
        transparent 100%);
    background-size: 200% 100%;
    animation: loading 1.5s ease-in-out infinite;
    border-radius: 2px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.2);
}

@keyframes loading {
    0% { 
        background-position: -200% 0;
        opacity: 0.8;
    }
    50% {
        opacity: 1;
    }
    100% { 
        background-position: 200% 0;
        opacity: 0.8;
    }
}

/* تحسين التمرير مع تأثيرات أوضح */
.product-dropdown {
    scrollbar-width: thin;
    scrollbar-color: #2196f3 #f1f1f1;
    border-radius: 8px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.15);
}

.product-dropdown::-webkit-scrollbar {
    width: 12px;
}

.product-dropdown::-webkit-scrollbar-track {
    background: #f8f9fa;
    border-radius: 10px;
    border: 1px solid #e9ecef;
}

.product-dropdown::-webkit-scrollbar-thumb {
    background: linear-gradient(135deg, #2196f3, #1976d2);
    border-radius: 10px;
    border: 2px solid #f8f9fa;
    transition: all 0.3s ease;
}

.product-dropdown::-webkit-scrollbar-thumb:hover {
    background: linear-gradient(135deg, #1976d2, #1565c0);
    transform: scale(1.1);
    box-shadow: 0 2px 8px rgba(33, 150, 243, 0.3);
}

/* تأثيرات إضافية للتفاعل */
.product-search-table tbody tr:active {
    transform: scale(0.98);
    transition: transform 0.1s ease;
}

/* تحسين الحدود والظلال */
.product-search-table td {
    padding: 12px 8px;
    border-bottom: 1px solid rgba(0,0,0,0.05);
    transition: all 0.3s ease;
}

.product-search-table tbody tr:hover td {
    border-color: rgba(33, 150, 243, 0.2);
}

/* تأثير التركيز للوصولية */
.product-search-table tbody tr:focus {
    outline: 3px solid rgba(33, 150, 243, 0.5);
    outline-offset: 2px;
}

/* تحسين الاستجابة للشاشات الصغيرة */
@media (max-width: 768px) {
    .product-search-table tbody tr:hover {
        transform: none;
        box-shadow: 0 4px 15px rgba(33, 150, 243, 0.1);
    }
    
    .product-search-table tbody tr.highlighted {
        transform: none;
        box-shadow: 0 6px 20px rgba(33, 150, 243, 0.25);
    }
}
</style>
`;

// إضافة الأنماط عند التحميل
$(document).ready(function() {
    $('head').append(enhancedSearchTableStyles);
});

// ============================================
// نهاية نظام التحكم في تنسيق الصفحة
// ============================================


// ============================================
// نظام إرسال الفاتورة عبر WhatsApp مباشرة - نسخة مُصححة
// ============================================

// تهيئة النظام عند تحميل الصفحة
$(document).ready(function() {
    setTimeout(function() {
        initializeDirectWhatsAppSystem();
    }, 1000);
});

// تهيئة نظام WhatsApp المباشر
function initializeDirectWhatsAppSystem() {
    // إضافة زر WhatsApp المباشر
  
    
    // إضافة الأنماط CSS
    addDirectWhatsAppStyles();
    
    // إضافة معالجات الأحداث
    attachDirectWhatsAppEventHandlers();
}



// إضافة الأنماط CSS للزر المباشر
function addDirectWhatsAppStyles() {
    var directWhatsappStyles = `
    <style id="directWhatsappStyles">
    /* أنماط زر WhatsApp المباشر */
    .direct-whatsapp-btn {
        background: #25D366;
        border-color: #25D366;
        color: white;
        padding: 12px 25px;
        border-radius: 30px;
        font-weight: 600;
        font-size: 16px;
        transition: all 0.3s ease;
        margin-left: 10px;
        box-shadow: 0 4px 15px rgba(37, 211, 102, 0.3);
    }
    
    .direct-whatsapp-btn:hover {
        background: #1ea952;
        border-color: #1ea952;
        color: white;
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(37, 211, 102, 0.4);
    }
    
    .direct-whatsapp-btn:active {
        transform: translateY(0);
    }
    
    .direct-whatsapp-btn:disabled {
        background: #95a5a6;
        border-color: #95a5a6;
        transform: none;
        box-shadow: none;
        cursor: not-allowed;
    }
    
    .direct-whatsapp-btn i {
        margin-right: 8px;
        font-size: 18px;
    }
    
    .direct-whatsapp-container {
        margin-top: 15px;
        text-align: center;
    }
    
    /* مؤشر التحميل */
    .loading-spinner {
        margin-left: 10px;
    }
    
    /* مؤشر التقدم */
    .progress-indicator {
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        z-index: 9999;
        background: rgba(0,0,0,0.85);
        color: white;
        padding: 25px 35px;
        border-radius: 15px;
        text-align: center;
        min-width: 250px;
        box-shadow: 0 10px 30px rgba(0,0,0,0.3);
    }
    
    .progress-indicator i {
        font-size: 24px;
        margin-bottom: 10px;
        display: block;
        color: #25D366;
    }
    
    .progress-indicator .progress-text {
        font-size: 14px;
        margin-top: 10px;
    }
    
    /* نافذة إدخال رقم الهاتف */
    .phone-input-modal {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 9998;
        display: flex;
        align-items: center;
        justify-content: center;
    }
    
    .phone-input-content {
        background: white;
        padding: 30px;
        border-radius: 15px;
        max-width: 400px;
        width: 90%;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.3);
    }
    
    .phone-input-content h4 {
        color: #25D366;
        margin-bottom: 20px;
        font-size: 18px;
    }
    
    .phone-input-content input {
        width: 100%;
        padding: 12px;
        border: 2px solid #e9ecef;
        border-radius: 8px;
        font-size: 16px;
        text-align: center;
        margin-bottom: 20px;
        direction: ltr;
    }
    
    .phone-input-content input:focus {
        border-color: #25D366;
        outline: none;
        box-shadow: 0 0 0 3px rgba(37, 211, 102, 0.1);
    }
    
    .phone-input-content .btn {
        margin: 0 5px;
        padding: 10px 20px;
        border-radius: 8px;
    }
    
    .phone-input-content .btn-success {
        background: #25D366;
        border-color: #25D366;
    }
    
    .phone-input-content .btn-success:hover {
        background: #1ea952;
        border-color: #1ea952;
    }
    
    /* رسائل النجاح والخطأ */
    .whatsapp-message {
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        z-index: 9999;
        min-width: 300px;
        text-align: center;
    }
    
    .whatsapp-message.success {
        background: #28a745;
        border-left: 4px solid #1e7e34;
    }
    
    .whatsapp-message.error {
        background:rgb(255, 255, 255);
        border-left: 4px solid #c82333;
    }
    
    .whatsapp-message i {
        margin-right: 10px;
        font-size: 16px;
    }
    
    /* تحسينات للأجهزة المحمولة */
    @media (max-width: 768px) {
        .direct-whatsapp-btn {
            font-size: 14px;
            padding: 10px 20px;
        }
        
        .phone-input-content {
            margin: 20px;
            padding: 20px;
        }
        
        .progress-indicator {
            margin: 20px;
            padding: 20px;
        }
    }
    </style>
    `;
    
    $('head').append(directWhatsappStyles);
}

// إضافة معالجات الأحداث المباشرة
function attachDirectWhatsAppEventHandlers() {
    // معالج النقر على زر الإرسال المباشر
    $(document).off('click', '#sendDirectToWhatsApp').on('click', '#sendDirectToWhatsApp', function() {
        sendInvoiceDirectly();
    });
}

// إرسال الفاتورة مباشرة
// إرسال الفاتورة مباشرة مع أخذ رقم العميل تلقائياً
function sendInvoiceDirectly() {
    // التحقق من وجود منتجات
    var products = collectCurrentProducts();
    if (products.length === 0) {
        showWhatsAppMessage('لا توجد منتجات في الفاتورة لإرسالها', 'error');
        return;
    }
    
    // محاولة الحصول على رقم هاتف العميل المحدد
    var customerPhone = getCustomerPhoneNumber();
    
    if (customerPhone) {
        // تنسيق الرقم تلقائياً
        var formattedPhone = formatPhoneNumberAuto(customerPhone);
        
        if (formattedPhone) {
            console.log('Customer phone found:', customerPhone, '-> formatted:', formattedPhone);
            processDirectInvoiceSend(formattedPhone);
        } else {
            // إذا لم يتم تنسيق الرقم بنجاح، اطلب من المستخدم إدخاله
            showPhoneInputModal(function(phoneNumber) {
                if (phoneNumber) {
                    processDirectInvoiceSend(phoneNumber);
                }
            });
        }
    } else {
        // لا يوجد رقم محفوظ، اطلب من المستخدم إدخاله
        showPhoneInputModal(function(phoneNumber) {
            if (phoneNumber) {
                processDirectInvoiceSend(phoneNumber);
            }
        });
    }
}

// دالة للحصول على رقم هاتف العميل من البيانات المحفوظة - محسنة
function getCustomerPhoneNumber() {
    var phoneNumber = null;
    
    try {
        // البحث في العميل المحدد في dropdown
        var customerSelect = $('#customer_id');
        if (customerSelect.length > 0 && customerSelect.val()) {
            var selectedOption = customerSelect.find('option:selected');
            
            // الطريقة الأولى: البحث في data attributes
            var customerData = selectedOption.data();
            if (customerData && customerData.mobile) {
                phoneNumber = customerData.mobile;
                console.log('Phone found in data attributes:', phoneNumber);
            }
            
            // الطريقة الثانية: البحث في Select2 data إذا كان يستخدم Select2
            if (!phoneNumber && customerSelect.hasClass('select2-hidden-accessible')) {
                try {
                    var select2Data = customerSelect.select2('data');
                    if (select2Data && select2Data.length > 0) {
                        var customerInfo = select2Data[0];
                        if (customerInfo.mobile) {
                            phoneNumber = customerInfo.mobile;
                            console.log('Phone found in Select2 data:', phoneNumber);
                        } else if (customerInfo.phone) {
                            phoneNumber = customerInfo.phone;
                            console.log('Phone found in Select2 data (phone field):', phoneNumber);
                        }
                    }
                } catch (select2Error) {
                    console.log('Select2 not available or error:', select2Error);
                }
            }
            
            // الطريقة الثالثة: البحث في نص الخيار المحدد
            if (!phoneNumber) {
                var customerText = selectedOption.text();
                // البحث عن أنماط أرقام الهواتف العراقية والدولية
                var phonePatterns = [
                    /(\+964[7-9]\d{8})/g,      // +96477xxxxxxxx
                    /(\+964\d{10})/g,          // +964xxxxxxxxxx
                    /(964[7-9]\d{8})/g,        // 96477xxxxxxxx
                    /(07[0-9]\d{7})/g,         // 077xxxxxxxx
                    /(7[0-9]\d{7})/g,          // 77xxxxxxxx
                    /(\+\d{10,15})/g           // أرقام دولية
                ];
                
                for (var i = 0; i < phonePatterns.length; i++) {
                    var phoneMatch = customerText.match(phonePatterns[i]);
                    if (phoneMatch) {
                        phoneNumber = phoneMatch[0];
                        console.log('Phone found in option text:', phoneNumber);
                        break;
                    }
                }
            }
        }
        
        // إذا لم نجد الرقم، نبحث في الحقول المخفية والمدخلات
        if (!phoneNumber) {
            // البحث في حقول الهاتف المختلفة
            var phoneFields = [
                'input[name="customer_mobile"]',
                'input[name="mobile"]', 
                '#customer_mobile',
                'input[name="customer_phone"]',
                'input[name="phone"]',
                '#customer_phone',
                '.customer-mobile',
                '.customer-phone'
            ];
            
            for (var j = 0; j < phoneFields.length; j++) {
                var fieldValue = $(phoneFields[j]).val();
                if (fieldValue && fieldValue.trim()) {
                    phoneNumber = fieldValue.trim();
                    console.log('Phone found in field', phoneFields[j], ':', phoneNumber);
                    break;
                }
            }
        }
        
        // البحث في بيانات العميل المحفوظة في الصفحة (JSON objects)
        if (!phoneNumber) {
            // البحث في متغيرات JavaScript العامة
            if (typeof window.customerData !== 'undefined' && window.customerData.mobile) {
                phoneNumber = window.customerData.mobile;
                console.log('Phone found in window.customerData:', phoneNumber);
            }
            
            // البحث في البيانات المحفوظة في localStorage
            try {
                var savedCustomer = localStorage.getItem('selectedCustomer');
                if (savedCustomer) {
                    var customerObj = JSON.parse(savedCustomer);
                    if (customerObj.mobile) {
                        phoneNumber = customerObj.mobile;
                        console.log('Phone found in localStorage:', phoneNumber);
                    }
                }
            } catch (localStorageError) {
                console.log('localStorage not available:', localStorageError);
            }
        }
        
        // البحث في الجدول أو القائمة إذا كانت بيانات العميل معروضة
        if (!phoneNumber) {
            // البحث في جدول معلومات العميل
            var customerTable = $('.customer-info-table, .customer-details, .customer-info');
            if (customerTable.length > 0) {
                var phoneCell = customerTable.find('td:contains("هاتف"), td:contains("موبايل"), td:contains("Phone"), td:contains("Mobile")').next();
                if (phoneCell.length > 0) {
                    var phoneText = phoneCell.text().trim();
                    if (phoneText) {
                        phoneNumber = phoneText;
                        console.log('Phone found in customer table:', phoneNumber);
                    }
                }
            }
        }
        
        console.log('Final found customer phone:', phoneNumber);
        return phoneNumber;
        
    } catch (error) {
        console.error('Error getting customer phone:', error);
        return null;
    }
}

// دالة تنسيق رقم الهاتف التلقائي المحسنة
function formatPhoneNumberAuto(phone) {
    if (!phone) return null;
    
    // تنظيف الرقم من الرموز غير المرغوبة
    var cleanPhone = phone.toString().replace(/[\s\-\(\)\+]/g, '');
    
    // إزالة الأصفار المكررة في البداية
    cleanPhone = cleanPhone.replace(/^0+/, '');
    
    console.log('Cleaning phone:', phone, '->', cleanPhone);
    
    // التعامل مع الأنماط المختلفة للأرقام العراقية
    var formattedPhone = null;
    
    // نمط: 9647xxxxxxxxx (بدون +)
    if (cleanPhone.match(/^964[7-9]\d{8}$/)) {
        formattedPhone = '+' + cleanPhone;
    }
    // نمط: 7xxxxxxxxx (رقم عراقي بدون كود الدولة)
    else if (cleanPhone.match(/^[7-9]\d{8}$/)) {
        formattedPhone = '+964' + cleanPhone;
    }
    // نمط: 77xxxxxxxx أو 78xxxxxxxx أو 79xxxxxxxx (بدون الصفر الأول)
    else if (cleanPhone.match(/^7[0-9]\d{7}$/)) {
        formattedPhone = '+964' + cleanPhone;
    }
    // نمط: أرقام أخرى تبدأ بـ 8 أو 9 (شبكات عراقية أخرى)
    else if (cleanPhone.match(/^[8-9]\d{8}$/)) {
        formattedPhone = '+964' + cleanPhone;
    }
    // محاولة التعامل مع أرقام بـ 11 رقم (قد تحتوي على صفر إضافي)
    else if (cleanPhone.match(/^0?7[0-9]\d{7}$/)) {
        var withoutZero = cleanPhone.replace(/^0/, '');
        formattedPhone = '+964' + withoutZero;
    }
    // نمط: 07xxxxxxxx (النمط العراقي الشائع)
    else if (cleanPhone.match(/^07[0-9]\d{7}$/)) {
        formattedPhone = '+964' + cleanPhone.substring(1); // إزالة الصفر الأول
    }
    // أرقام دولية أخرى (غير عراقية)
    else if (cleanPhone.length >= 10 && cleanPhone.length <= 15) {
        // إذا لم يبدأ برمز دولة معروف، افترض أنه عراقي
        if (!cleanPhone.match(/^(1|44|33|49|81|86|91|92|93|94|95|98)/)) {
            // إذا كان الرقم 10 أرقام أو أكثر وليس له رمز دولة، أضف رمز العراق
            if (cleanPhone.length >= 9) {
                formattedPhone = '+964' + cleanPhone;
            }
        } else {
            formattedPhone = '+' + cleanPhone;
        }
    }
    
    // التحقق من صحة الرقم النهائي
    if (formattedPhone && isValidPhoneNumber(formattedPhone)) {
        console.log('Phone formatted successfully:', phone, '->', formattedPhone);
        return formattedPhone;
    } else {
        console.warn('Could not format phone number:', phone);
        return null;
    }
}

// دالة للتحقق من صحة رقم الهاتف - محسنة
function isValidPhoneNumber(phone) {
    if (!phone) return false;
    
    // أنماط صحيحة لأرقام الهواتف
    var patterns = [
        /^\+964[7-9]\d{8}$/,     // عراقي: +96477xxxxxxxx
        /^\+964[0-9]\d{8}$/,     // عراقي عام: +964xxxxxxxxx  
        /^\+964\d{10}$/,         // عراقي: +964xxxxxxxxxx
        /^\+[1-9]\d{10,14}$/,    // دولي عام
        /^\+1\d{10}$/,           // أمريكي/كندي
        /^\+44\d{10}$/,          // بريطاني
        /^\+33\d{9}$/,           // فرنسي
        /^\+49\d{10,11}$/,       // ألماني
        /^\+81\d{10,11}$/,       // ياباني
        /^\+86\d{11}$/,          // صيني
        /^\+91\d{10}$/           // هندي
    ];
    
    return patterns.some(pattern => pattern.test(phone));
}

// دالة مساعدة لاستخراج رقم الهاتف من النص
function extractPhoneFromText(text) {
    if (!text) return null;
    
    // أنماط البحث عن أرقام الهواتف
    var phonePatterns = [
        /\+964[7-9]\d{8}/g,        // +96477xxxxxxxx
        /\+964\d{10}/g,            // +964xxxxxxxxxx
        /964[7-9]\d{8}/g,          // 96477xxxxxxxx (بدون +)
        /07[0-9]\d{7}/g,           // 077xxxxxxxx
        /7[0-9]\d{7}/g,            // 77xxxxxxxx
        /\+\d{10,15}/g             // أرقام دولية عامة
    ];
    
    for (var i = 0; i < phonePatterns.length; i++) {
        var matches = text.match(phonePatterns[i]);
        if (matches && matches.length > 0) {
            return matches[0]; // إرجاع أول رقم موجود
        }
    }
    
    return null;
}

// دالة للبحث المتقدم عن رقم الهاتف في DOM
function findPhoneInDOM() {
    var phoneNumber = null;
    
    // البحث في العناصر التي قد تحتوي على رقم الهاتف
    var selectors = [
        '.phone', '.mobile', '.tel', '.customer-phone', '.customer-mobile',
        '[data-phone]', '[data-mobile]', '[data-tel]',
        'span:contains("Phone")', 'span:contains("Mobile")', 'span:contains("هاتف")', 'span:contains("موبايل")'
    ];
    
    for (var i = 0; i < selectors.length; i++) {
        var elements = $(selectors[i]);
        elements.each(function() {
            var text = $(this).text() || $(this).val() || $(this).data('phone') || $(this).data('mobile');
            if (text) {
                var extractedPhone = extractPhoneFromText(text);
                if (extractedPhone) {
                    phoneNumber = extractedPhone;
                    return false; // توقف عند العثور على أول رقم
                }
            }
        });
        
        if (phoneNumber) break;
    }
    
    return phoneNumber;
}

// دالة محسنة لعرض نافذة إدخال الرقم (مع الرقم المقترح)
function showPhoneInputModal(callback, suggestedPhone = null) {
    var modalHTML = `
    <div class="phone-input-modal" id="phoneInputModal">
        <div class="phone-input-content">
            <h4><i class="fab fa-whatsapp"></i> إرسال الفاتورة</h4>
            <p>أدخل رقم هاتف العميل:</p>
            <input type="text" id="customerPhoneInput" placeholder="+9647xxxxxxxx" dir="ltr" maxlength="15" value="${suggestedPhone || ''}">
            <div style="margin-top: 10px;">
                <small style="color: #6c757d;">سيتم تنسيق الرقم تلقائياً لإضافة كود العراق (+964)</small>
            </div>
            <div style="margin-top: 20px;">
                <button type="button" class="btn btn-success" id="confirmSendBtn">
                    <i class="fab fa-whatsapp"></i> إرسال
                </button>
                <button type="button" class="btn btn-secondary" id="cancelSendBtn">إلغاء</button>
            </div>
        </div>
    </div>
    `;
    
    $('body').append(modalHTML);
    
    // التركيز على حقل الهاتف وتحديد النص إذا كان هناك رقم مقترح
    var phoneInput = $('#customerPhoneInput');
    phoneInput.focus();
    if (suggestedPhone) {
        phoneInput.select();
    }
    
    // معالج الإرسال
    $('#confirmSendBtn').on('click', function() {
        var phone = phoneInput.val().trim();
        var formattedPhone = formatPhoneNumberAuto(phone);
        
        if (!formattedPhone) {
            showWhatsAppMessage('رقم الهاتف غير صحيح. يرجى إدخال رقم صحيح', 'error');
            phoneInput.focus().select();
            return;
        }
        
        $('#phoneInputModal').remove();
        callback(formattedPhone);
    });
    
    // معالج الإلغاء
    $('#cancelSendBtn').on('click', function() {
        $('#phoneInputModal').remove();
        callback(null);
    });
    
    // معالج الضغط على Enter
    phoneInput.on('keypress', function(e) {
        if (e.which === 13) {
            $('#confirmSendBtn').click();
        }
    });
    
    // معالج الضغط على Escape
    $(document).on('keydown.phoneModal', function(e) {
        if (e.which === 27) {
            $('#phoneInputModal').remove();
            $(document).off('keydown.phoneModal');
            callback(null);
        }
    });
    
    // تنسيق الرقم أثناء الكتابة
    phoneInput.on('input', function() {
        var currentValue = $(this).val();
        // لا نقوم بالتنسيق أثناء الكتابة لتجنب مقاطعة المستخدم
        // فقط نظهر معاينة في placeholder أو نص مساعد
    });
}

// دالة مساعدة لاستخراج رقم الهاتف من النص
function extractPhoneFromText(text) {
    if (!text) return null;
    
    // أنماط مختلفة لاستخراج أرقام الهواتف
    var patterns = [
        /(\+964\d{10})/g,           // +964xxxxxxxxxx
        /(964\d{10})/g,             // 964xxxxxxxxxx
        /(\+964\s?\d{3}\s?\d{3}\s?\d{4})/g, // +964 xxx xxx xxxx
        /(07\d{8})/g,               // 07xxxxxxxx
        /(\d{11})/g                 // xxxxxxxxxxx
    ];
    
    for (var i = 0; i < patterns.length; i++) {
        var matches = text.match(patterns[i]);
        if (matches && matches.length > 0) {
            return matches[0].replace(/\s/g, ''); // إزالة المسافات
        }
    }
    
    return null;
}

// دالة للحصول على معلومات العميل الكاملة (إضافية)
function getCustomerFullInfo() {
    var customerInfo = {
        name: 'عميل نقدي',
        phone: null,
        email: null
    };
    
    try {
        var customerSelect = $('#customer_id');
        if (customerSelect.length > 0 && customerSelect.val()) {
            var selectedOption = customerSelect.find('option:selected');
            
            // الحصول على الاسم
            customerInfo.name = selectedOption.text().split(' - ')[0] || 'عميل نقدي';
            
            // الحصول على الهاتف
            customerInfo.phone = getCustomerPhoneNumber();
            
            // محاولة الحصول على الإيميل من بيانات العميل
            var customerData = selectedOption.data();
            if (customerData && customerData.email) {
                customerInfo.email = customerData.email;
            }
        }
    } catch (error) {
        console.error('Error getting customer info:', error);
    }
    
    return customerInfo;
}

// عرض نافذة إدخال رقم الهاتف
function showPhoneInputModal(callback) {
    var modalHTML = `
    <div class="phone-input-modal" id="phoneInputModal">
        <div class="phone-input-content">
            <h4><i class="fab fa-whatsapp"></i> إرسال الفاتورة</h4>
            <p>أدخل رقم هاتف العميل:</p>
            <input type="text" id="customerPhoneInput" placeholder="+9647xxxxxxxx" dir="ltr" maxlength="15">
            <div>
                <button type="button" class="btn btn-success" id="confirmSendBtn">
                    <i class="fab fa-whatsapp"></i> إرسال
                </button>
                <button type="button" class="btn btn-secondary" id="cancelSendBtn">إلغاء</button>
            </div>
        </div>
    </div>
    `;
    
    $('body').append(modalHTML);
    
    // التركيز على حقل الهاتف
    $('#customerPhoneInput').focus();
    
    // معالج الإرسال
    $('#confirmSendBtn').on('click', function() {
        var phone = $('#customerPhoneInput').val().trim();
        var formattedPhone = formatPhoneNumber(phone);
        
        if (!formattedPhone) {
            showWhatsAppMessage('رقم الهاتف غير صحيح. يرجى إدخال رقم صحيح مع رمز الدولة', 'error');
            $('#customerPhoneInput').focus().select();
            return;
        }
        
        $('#phoneInputModal').remove();
        callback(formattedPhone);
    });
    
    // معالج الإلغاء
    $('#cancelSendBtn').on('click', function() {
        $('#phoneInputModal').remove();
        callback(null);
    });
    
    // معالج الضغط على Enter
    $('#customerPhoneInput').on('keypress', function(e) {
        if (e.which === 13) {
            $('#confirmSendBtn').click();
        }
    });
    
    // معالج الضغط على Escape
    $(document).on('keydown.phoneModal', function(e) {
        if (e.which === 27) {
            $('#phoneInputModal').remove();
            $(document).off('keydown.phoneModal');
            callback(null);
        }
    });
}

// معالجة إرسال الفاتورة مباشرة - مُصحح
function processDirectInvoiceSend(phoneNumber) {
    // تعطيل الزر وإظهار مؤشر التحميل
    $('#sendDirectToWhatsApp').prop('disabled', true);
    $('.loading-spinner').show();
    showProgressIndicator('جاري إنشاء صورة الفاتورة...');
    
    // إنشاء HTML للفاتورة بنفس تصميم التمبليت
    createInvoiceHTML().then(function(invoiceHTML) {
        showProgressIndicator('جاري تحويل الفاتورة إلى صورة عالية الجودة...');
        
        // تحويل HTML إلى صورة بدقة عالية
        convertInvoiceToHighQualityImage(invoiceHTML).then(function(imageData) {
            showProgressIndicator('جاري إرسال الفاتورة عبر WhatsApp...');
            
            // إرسال الصورة - الإصلاح هنا
            sendImageToWhatsApp(phoneNumber, imageData);
            
        }).catch(function(error) {
            console.error('Error converting to image:', error);
            hideProgressIndicator();
            $('#sendDirectToWhatsApp').prop('disabled', false);
            $('.loading-spinner').hide();
            showWhatsAppMessage('فشل في تحويل الفاتورة إلى صورة', 'error');
        });
        
    }).catch(function(error) {
        console.error('Error creating invoice HTML:', error);
        hideProgressIndicator();
        $('#sendDirectToWhatsApp').prop('disabled', false);
        $('.loading-spinner').hide();
        showWhatsAppMessage('فشل في إنشاء الفاتورة', 'error');
    });
}

// إرسال الصورة إلى WhatsApp - نسخة مُبسطة
function sendImageToWhatsApp(phoneNumber, imageData) {
    // التحقق من صحة البيانات قبل الإرسال
    if (!phoneNumber || !imageData) {
        hideProgressIndicator();
        $('#sendDirectToWhatsApp').prop('disabled', false);
        $('.loading-spinner').hide();
        showWhatsAppMessage('بيانات غير مكتملة للإرسال', 'error');
        return;
    }

    // تنظيف رقم الهاتف
    var cleanPhone = phoneNumber.trim();
    
    // تحضير البيانات للإرسال
    var requestData = {
        phone: cleanPhone,
        image_data: imageData,
        caption: '  عرض سعر من شركة نور النبراس   - ' + new Date().toLocaleDateString('en-US')
    };

    // التحقق من حجم البيانات
    var dataSize = JSON.stringify(requestData).length;
    console.log('Request data size:', (dataSize / 1024 / 1024).toFixed(2) + ' MB');

    $.ajax({
        url: 'whatsapp/send-invoice-image',
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content') || $('input[name="_token"]').val()
        },
        data: JSON.stringify(requestData),
        timeout: 120000, // 2 دقيقة
        beforeSend: function() {
            console.log('Sending invoice to:', cleanPhone);
            console.log('Image data length:', imageData.length);
        },
        success: function(response) {
            console.log('WhatsApp response:', response);
            
            hideProgressIndicator();
            $('#sendDirectToWhatsApp').prop('disabled', false);
            $('.loading-spinner').hide();
            
            if (response.success) {
                showWhatsAppMessage('تم إرسال الفاتورة بنجاح عبر WhatsApp! 🎉', 'success');
            } else {
                showWhatsAppMessage('فشل في إرسال الفاتورة: ' + (response.message || 'خطأ غير معروف'), 'error');
            }
        },
        error: function(xhr, status, error) {
            console.error('WhatsApp send error:', {
                status: status,
                error: error,
                responseText: xhr.responseText,
                responseJSON: xhr.responseJSON
            });
            
            hideProgressIndicator();
            $('#sendDirectToWhatsApp').prop('disabled', false);
            $('.loading-spinner').hide();
            
            var errorMessage = getErrorMessage(xhr);
            showWhatsAppMessage(errorMessage + ' (كود: ' + xhr.status + ')', 'error');
        }
    });
}

// استخراج رسالة الخطأ من استجابة الخادم
function getErrorMessage(xhr) {
    var errorMessage = 'خطأ في إرسال الفاتورة';
    
    if (xhr.status === 422) {
        errorMessage = 'بيانات الطلب غير صحيحة';
    } else if (xhr.status === 500) {
        errorMessage = 'خطأ في الخادم';
    } else if (xhr.responseJSON && xhr.responseJSON.message) {
        errorMessage = xhr.responseJSON.message;
    } else if (xhr.responseText) {
        try {
            var errorData = JSON.parse(xhr.responseText);
            errorMessage = errorData.message || errorMessage;
        } catch (e) {
            // استخدام الرسالة الافتراضية
        }
    }
    
    return errorMessage;
}

// إنشاء HTML للفاتورة بنفس تصميم التمبليت
function createInvoiceHTML() {
    return new Promise(function(resolve, reject) {
        try {
            // جمع بيانات الفاتورة
            var invoiceData = collectInvoiceData();
            
            // التحقق من صحة البيانات
            if (!invoiceData || !invoiceData.products || invoiceData.products.length === 0) {
                reject(new Error('لا توجد بيانات فاتورة صحيحة'));
                return;
            }
            
            // إنشاء HTML بنفس تصميم التمبليت المرسل
            var invoiceHTML = generateInvoiceTemplate(invoiceData);
            
            resolve(invoiceHTML);
        } catch (error) {
            reject(error);
        }
    });
}

// جمع بيانات الفاتورة الحالية
function collectInvoiceData() {
    var customerName = 'عميل نقدي';
    var customerSelect = $('#customer_id');
    if (customerSelect.length > 0 && customerSelect.val()) {
        var selectedOption = customerSelect.find('option:selected');
        customerName = selectedOption.text() || 'عميل نقدي';
    }
    
    // جمع المنتجات
    var products = [];
    $('#pos_table tbody tr:not(.empty-row)').each(function() {
        var row = $(this);
        var productName = row.find('.product-search-input').val() || 
                         row.find('td:eq(1)').text().trim() ||
                         row.find('[name*="product_name"]').val();
        
        if (productName) {
            var quantity = parseFloat(row.find('.pos_quantity').val()) || 0;
            var unit = row.find('.unit-input').val() || 'EA';
            var unitPrice = parseFloat(row.find('.pos_unit_price_inc_tax').val()) || 0;
            var lineTotal = parseFloat(row.find('.pos_line_total').val()) || 0;
            var discount = parseFloat(row.find('.discount_percent').val()) || 0;
            
            if (quantity > 0) {
                products.push({
                    name: productName,
                    quantity: quantity,
                    unit: unit,
                    unitPrice: unitPrice,
                    lineTotal: lineTotal,
                    discount: discount
                });
            }
        }
    });
    
    // حساب الإجماليات
    var subtotal = products.reduce((sum, product) => sum + product.lineTotal, 0);
    var generalDiscount = parseFloat($('#discount_amount').val()) || 0;
    var tax = parseFloat($('#tax_calculation_amount').val()) || 0;
    var shipping = parseFloat($('#shipping_charges').val()) || 0;
    var total = subtotal - generalDiscount + (subtotal * tax / 100) + shipping;
    
    return {
        invoiceNo: 'INV-' + Date.now(),
        date: new Date().toLocaleDateString('en-US'),
        dueDate: new Date(Date.now() + 30*24*60*60*1000).toLocaleDateString('en-US'),
        customerName: customerName,
        products: products,
        subtotal: subtotal,
        discount: generalDiscount,
        tax: tax,
        shipping: shipping,
        total: total,
        exchangeRate: getIQDExchangeRate()
    };
}

// إنشاء تمبليت الفاتورة بنفس التصميم المرسل
function generateInvoiceTemplate(data) {
    return `
    <!DOCTYPE html>
    <html dir="rtl" lang="ar">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <link href="https://fonts.googleapis.com/css2?family=Almarai:wght@300;400;700;800&display=swap" rel="stylesheet">
        <style>
            /* نفس الأنماط من التمبليت الأصلي */
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }
            
            body {
                font-family: 'Almarai', Arial, sans-serif;
                color: #2c3e50;
                direction: rtl;
                font-size: 14px;
                line-height: 1.3;
                background-color: white;
                width: 800px;
                margin: 0 auto;
                padding: 20px;
            }
            
            .invoice-wrapper {
                width: 100%;
                background: white;
            }
            
            .invoice-wrapper table {
                width: 100%;
                border-collapse: collapse;
                border-spacing: 0;
            }
            
            .invoice-wrapper td, .invoice-wrapper th {
                vertical-align: top;
                padding: 0;
            }
            
            /* Header styles */
            .header-section {
                background: linear-gradient(135deg, #00a99d 0%, #00c9b7 100%);
                color: white;
                padding: 15px;
                border-radius: 8px;
                margin-bottom: 15px;
                width: 100%;
            }
            
            .header-section h1 {
                font-size: 24px;
                margin: 0;
                color: white;
                font-weight: 800;
                text-align: center;
            }
            
            .header-section p {
                margin: 5px 0 0 0;
                font-size: 14px;
                color: white;
                text-align: center;
            }
            
            /* Logo styles */
            .logo-container {
                text-align: right;
                vertical-align: middle;
            }
            
            .company-logo {
                max-width: 80px;
                max-height: 80px;
                border-radius: 8px;
                background-color: rgba(255,255,255,0.1);
                padding: 5px;
            }
            
            /* Info boxes */
            .info-box {
                background-color: #f8f9fa;
                border: 1px solid #e9ecef;
                border-radius: 5px;
                padding: 8px;
                margin: 2px;
                text-align: center;
                display: inline-block;
                width: calc(100% - 4px);
            }
            
            .info-box .label {
                font-size: 11px;
                color: #6c757d;
                display: block;
                margin-bottom: 3px;
            }
            
            .info-box .value {
                font-size: 13px;
                font-weight: 600;
                color: #2c3e50;
                display: block;
            }
            
            /* Info cards */
            .info-card {
                background-color: #f8f9fa;
                border-radius: 6px;
                padding: 12px;
                border-right: 3px solid #00a99d;
                margin-bottom: 10px;
                width: 100%;
            }
            
            .info-card h4 {
                color: #00a99d;
                font-size: 14px;
                margin: 0 0 8px 0;
                font-weight: 700;
            }
            
            .info-card p {
                margin: 3px 0;
                font-size: 13px;
                color: #2c3e50;
            }
            
            .info-card strong {
                font-weight: 600;
                color: #495057;
            }
            
            /* Products table */
            .products-table {
                width: 100%;
                border-collapse: separate;
                border-spacing: 0;
                margin: 15px 0;
                font-size: 13px;
                overflow: hidden;
                border-radius: 8px;
            }
            
            .products-table thead tr {
                background: linear-gradient(135deg, #00a99d 0%, #00c9b7 100%);
            }
            
            .products-table th {
                color: white;
                padding: 10px 5px;
                font-weight: 600;
                font-size: 13px;
                text-align: center;
                border: none;
            }
            
            .products-table td {
                padding: 8px 5px;
                border-bottom: 1px solid #e9ecef;
                font-size: 13px;
                color: #2c3e50;
                text-align: center;
            }
            
            .products-table tbody tr:nth-child(even) {
                background-color: #f8f9fa;
            }
            
            .products-table td:first-child {
                font-weight: 600;
                color: #00a99d;
            }
            
            .products-table td:nth-child(2) {
                text-align: right;
                padding-right: 10px;
            }
            
            .products-table td:nth-child(2) strong {
                font-weight: 600;
                color: #2c3e50;
            }
            
            .products-table td:last-child {
                font-weight: 600;
                background-color: #e7f6f5;
            }
            
            /* Summary box */
            .summary-box {
                background-color: #f8f9fa;
                border-radius: 6px;
                padding: 12px;
                font-size: 13px;
                width: 100%;
            }
            
            .summary-box h5 {
                font-size: 13px;
                color: #00a99d;
                margin: 0 0 8px 0;
                font-weight: 600;
            }
            
            .summary-row {
                display: flex;
                justify-content: space-between;
                padding: 4px 0;
                border-bottom: 1px solid #e9ecef;
                align-items: center;
            }
            
            .summary-row span {
                font-size: 13px;
                color: #2c3e50;
            }
            
            .summary-row span:last-child {
                font-weight: 600;
            }
            
            .total-box {
                background: linear-gradient(135deg, #00a99d 0%, #00c9b7 100%);
                color: white;
                padding: 12px;
                border-radius: 5px;
                margin-top: 8px;
                font-size: 16px;
                font-weight: 700;
                text-align: center;
            }
            
            /* Footer */
            .footer-section {
                background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
                color: white;
                padding: 20px;
                border-radius: 8px;
                margin-top: 20px;
                font-size: 13px;
            }
            
            .signature-box {
                border-top: 1px solid #333;
                width: 120px;
                margin: 20px auto 5px;
                padding-top: 5px;
                text-align: center;
            }
            
            .signature-box small {
                font-size: 12px;
                color: #6c757d;
            }
        </style>
    </head>
    <body>
        <div class="invoice-wrapper">
            <table>
                <!-- Header Section -->
                <tr>
                    <td>
                        <div class="header-section">
                            <table>
                                <tr>
                                    <td style="width: 20%; text-align: right;" class="logo-container">
                                        <img src="http://192.168.116.181/public/uploads/business_logos/1749638837_Asset%2059@72x.png" 
                                             alt="شعار الشركة" 
                                             class="company-logo">
                                    </td>
                                    <td style="width: 55%; text-align: center;">
                                        <h1> عرض سعر</h1>
                                        <p>COMBO SAS</p>
                                    </td>
                                    <td style="width: 25%; text-align: left;">
                                        <div style="background-color: rgba(255,255,255,0.2); padding: 8px; border-radius: 5px; display: inline-block;">
                                            <span style="font-size: 10px; color: white; display: block;">رقم الفاتورة</span>
                                            <strong style="font-size: 16px; color: white; display: block;">${data.invoiceNo}</strong>
                                        </div>
                                    </td>
                                </tr>
                            </table>
                        </div>
                    </td>
                </tr>
                
                <!-- Invoice Details -->
                <tr>
                    <td>
                        <table style="margin-bottom: 10px;">
                            <tr>
                                <td style="width: 33.33%; padding: 0 3px;">
                                    <div class="info-box">
                                        <span class="label">التاريخ</span>
                                        <span class="value">${data.date}</span>
                                    </div>
                                </td>
                                <td style="width: 33.33%; padding: 0 3px;">
                                    <div class="info-box">
                                        <span class="label">الاستحقاق</span>
                                        <span class="value">${data.dueDate}</span>
                                    </div>
                                </td>
                                <td style="width: 33.33%; padding: 0 3px;">
                                    <div class="info-box">
                                        <span class="label">طريقة الدفع</span>
                                        <span class="value">نقدي</span>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
                
                <!-- Customer Info -->
                <tr>
                    <td>
                        <div class="info-card">
                            <h4>معلومات العميل</h4>
                            <p><strong>الاسم:</strong> ${data.customerName}</p>
                        </div>
                    </td>
                </tr>
                
                <!-- Products Table -->
                <tr>
                    <td>
                        <table class="products-table">
                            <thead>
                                <tr>
                                    <th style="width: 5%;">#</th>
                                    <th style="width: 45%;">الصنف</th>
                                    <th style="width: 10%;">الكمية</th>
                                    <th style="width: 10%;">الوحدة</th>
                                    <th style="width: 15%;">السعر</th>
                                    <th style="width: 15%;">الإجمالي</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${data.products.map((product, index) => `
                                <tr>
                                    <td>${index + 1}</td>
                                    <td><strong>${product.name}</strong></td>
                                    <td><strong>${product.quantity}</strong></td>
                                    <td>${product.unit}</td>
                                    <td>${formatNumber(product.unitPrice * data.exchangeRate, 0)} د.ع</td>
                                    <td>${formatNumber(product.lineTotal * data.exchangeRate, 0)} د.ع</td>
                                </tr>
                                `).join('')}
                                
                                ${Array.from({length: Math.max(0, 8 - data.products.length)}, (_, i) => `
                                <tr>
                                    <td style="color: #f8f9fa;">${data.products.length + i + 1}</td>
                                    <td>&nbsp;</td>
                                    <td>&nbsp;</td>
                                    <td>&nbsp;</td>
                                    <td>&nbsp;</td>
                                    <td style="background-color: #f8f9fa;">&nbsp;</td>
                                </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </td>
                </tr>
                
                <!-- Summary Section -->
                <tr>
                    <td>
                        <table style="margin-top: 15px;">
                            <tr>
                                <td style="width: 55%; vertical-align: top; padding-left: 15px;">
                                    <!-- يمكن إضافة ملاحظات هنا -->
                                </td>
                                <td style="width: 45%; vertical-align: top; padding-right: 15px;">
                                    <div class="summary-box">
                                        <div class="summary-row">
                                            <span>المجموع الفرعي:</span>
                                            <span>${formatNumber(data.subtotal * data.exchangeRate, 0)} د.ع</span>
                                        </div>
                                        
                                        ${data.discount > 0 ? `
                                        <div class="summary-row">
                                            <span style="color: #dc3545;">الخصم:</span>
                                            <span style="color: #dc3545;">(-) ${formatNumber(data.discount * data.exchangeRate, 0)} د.ع</span>
                                        </div>
                                        ` : ''}
                                        
                                        ${data.tax > 0 ? `
                                        <div class="summary-row">
                                            <span>الضريبة (${data.tax}%):</span>
                                            <span>(+) ${formatNumber((data.subtotal * data.tax / 100) * data.exchangeRate, 0)} د.ع</span>
                                        </div>
                                        ` : ''}
                                        
                                        ${data.shipping > 0 ? `
                                        <div class="summary-row">
                                            <span>الشحن:</span>
                                            <span>(+) ${formatNumber(data.shipping * data.exchangeRate, 0)} د.ع</span>
                                        </div>
                                        ` : ''}
                                        
                                        <div class="total-box">
                                            الإجمالي: ${formatNumber(data.total * data.exchangeRate, 0)} دينار عراقي
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
                
                <!-- Signature -->
                <tr>
                    <td>
                        <table style="margin-top: 15px;">
                            <tr>
                                <td style="width: 50%; text-align: center;">
                                    <div class="signature-box">
                                        <small>التوقيع المعتمد</small>
                                    </div>
                                </td>
                                <td style="width: 50%; text-align: center;">
                                    <div class="signature-box">
                                        <small>توقيع العميل</small>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
                
                <!-- Company Info Footer -->
                <tr>
                    <td>
                        <div class="footer-section">
                            <div style="display: flex; justify-content: space-between; align-items: center; text-align: center;">
                                <div style="flex: 1;">
                                    <span style="display: block; margin-bottom: 3px;">📞 07700004462</span>
                                    <span style="display: block;">📞 07902406969</span>
                                </div>
                                <div style="flex: 1;">
                                    <span style="display: block; margin-bottom: 3px;">✉️ nbsiqofficial@gmail.com</span>
                                    <span style="display: block;">🌐 nbs.iq</span>
                                </div>
                                <div style="flex: 1;">
                                    <span style="display: block; font-size: 11px;">📍 بغداد - الصرافية - مقابل جامع عادلة خاتون</span>
                                </div>
                            </div>
                        </div>
                    </td>
                </tr>
            </table>
        </div>
    </body>
    </html>
    `;
}

// تحويل الفاتورة إلى صورة عالية الجودة
function convertInvoiceToHighQualityImage(invoiceHTML) {
    return new Promise(function(resolve, reject) {
        try {
            // إنشاء عنصر مؤقت
            var tempContainer = document.createElement('div');
            tempContainer.style.cssText = `
                position: absolute;
                top: -9999px;
                left: -9999px;
                width: 800px;
                background: white;
                padding: 0;
                margin: 0;
                font-family: 'Almarai', Arial, sans-serif;
                direction: rtl;
            `;
            
            tempContainer.innerHTML = invoiceHTML;
            document.body.appendChild(tempContainer);
            
            // انتظار تحميل الخطوط
            setTimeout(function() {
                // إعدادات عالية الجودة لـ html2canvas
                var options = {
                    allowTaint: false,
                    useCORS: true,
                    scale: 2, // دقة عالية
                    backgroundColor: '#ffffff',
                    width: 800,
                    height: null,
                    scrollX: 0,
                    scrollY: 0,
                    logging: false,
                    imageTimeout: 0,
                    removeContainer: true,
                    foreignObjectRendering: false,
                    onclone: function(clonedDoc) {
                        // التأكد من تطبيق الخطوط في النسخة المستنسخة
                        var clonedContainer = clonedDoc.querySelector('div');
                        if (clonedContainer) {
                            clonedContainer.style.fontFamily = "'Almarai', Arial, sans-serif";
                            clonedContainer.style.direction = 'rtl';
                        }
                    }
                };
                
                html2canvas(tempContainer, options).then(function(canvas) {
                    // تنظيف العنصر المؤقت
                    document.body.removeChild(tempContainer);
                    
                    // تحويل إلى صورة مع جودة عالية وحجم محسن
                    var imageDataURL = canvas.toDataURL('image/jpeg', 0.85); // جودة 85%
                    
                    // التحقق من حجم الصورة
                    var sizeInBytes = Math.round((imageDataURL.length - 'data:image/jpeg;base64,'.length) * 3/4);
                    var sizeInMB = sizeInBytes / (1024 * 1024);
                    
                    console.log('Invoice image size:', sizeInMB.toFixed(2) + ' MB');
                    
                    // تقليل الجودة إذا كان الحجم كبير
                    if (sizeInMB > 4) {
                        imageDataURL = canvas.toDataURL('image/jpeg', 0.7); // جودة 70%
                        sizeInBytes = Math.round((imageDataURL.length - 'data:image/jpeg;base64,'.length) * 3/4);
                        sizeInMB = sizeInBytes / (1024 * 1024);
                        console.log('Compressed image size:', sizeInMB.toFixed(2) + ' MB');
                    }
                    
                    resolve(imageDataURL);
                    
                }).catch(function(error) {
                    document.body.removeChild(tempContainer);
                    reject(error);
                });
                
            }, 1000); // انتظار ثانية واحدة لتحميل الخطوط
            
        } catch (error) {
            reject(error);
        }
    });
}

// جمع المنتجات الحالية
function collectCurrentProducts() {
    var products = [];
    
    $('#pos_table tbody tr:not(.empty-row)').each(function() {
        var row = $(this);
        var productName = row.find('.product-search-input').val() || 
                         row.find('td:eq(1)').text().trim();
        
        if (productName) {
            var quantity = parseFloat(row.find('.pos_quantity').val()) || 0;
            if (quantity > 0) {
                products.push({
                    name: productName,
                    quantity: quantity
                });
            }
        }
    });
    
    return products;
}

// تنسيق رقم الهاتف
function formatPhoneNumber(phone) {
    phone = phone.replace(/[\s\-\(\)]/g, '');
    
    if (phone.startsWith('+964')) {
        return phone;
    } else if (phone.startsWith('964')) {
        return '+' + phone;
    } else if (phone.startsWith('07')) {
        return '+964' + phone.substring(1);
    } else if (phone.startsWith('7')) {
        return '+964' + phone;
    } else if (phone.startsWith('+')) {
        return phone;
    }
    
    return null;
}

// عرض مؤشر التقدم
function showProgressIndicator(message) {
    var progressHTML = `
    <div class="progress-indicator">
        <i class="fa fa-spinner fa-spin"></i>
        <div class="progress-text">${message}</div>
    </div>
    `;
    
    $('.progress-indicator').remove();
    $('body').append(progressHTML);
}

// إخفاء مؤشر التقدم
function hideProgressIndicator() {
    $('.progress-indicator').remove();
}

// عرض رسالة WhatsApp
function showWhatsAppMessage(message, type) {
    var icon = type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle';
    var messageHTML = `
    <div class="whatsapp-message ${type}">
        <i class="fa ${icon}"></i>
        ${message}
    </div>
    `;
    
    $('.whatsapp-message').remove();
    $('body').append(messageHTML);
    
    // إزالة الرسالة بعد 5 ثوان
    setTimeout(function() {
        $('.whatsapp-message').fadeOut(function() {
            $(this).remove();
        });
    }, 5000);
}

// تنسيق الأرقام
function formatNumber(number, decimals = 0) {
    return parseFloat(number || 0).toLocaleString('en-US', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals
    });
}

// إضافة مكتبة html2canvas إذا لم تكن محملة
$(document).ready(function() {
    if (typeof html2canvas === 'undefined') {
        var script = document.createElement('script');
        script.src = 'https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js';
        script.onload = function() {
            console.log('html2canvas loaded successfully');
        };
        document.head.appendChild(script);
    }
    
    // التأكد من وجود CSRF token
    if ($('meta[name="csrf-token"]').length === 0) {
        $('head').append('<meta name="csrf-token" content="' + $('input[name="_token"]').val() + '">');
    }
});

// ============================================
// نهاية نظام إرسال الفاتورة عبر WhatsApp مباشرة
// ============================================


// إصلاح معالجات التنقل من حقل IQD
function fixNavigationFromIQDField() {
    // إزالة المعالجات القديمة وإضافة الجديدة
    $(document).off('keydown', '.iqd-price-display').on('keydown', '.iqd-price-display', function(e) {
        var current = $(this);
        var row = current.closest('tr');
        
        switch(e.which) {
            case 9: // Tab key
                e.preventDefault();
                if (e.shiftKey) {
                    // Shift + Tab - الانتقال للحقل السابق (USD)
                    navigateToField(row, 'previous');
                } else {
                    // Tab - الانتقال للحقل التالي (Discount)
                    navigateToField(row, 'next');
                }
                break;
                
            case 13: // Enter key
                e.preventDefault();
                navigateToField(row, 'next');
                break;
                
            case 37: // Arrow Left
                e.preventDefault();
                navigateToField(row, 'previous');
                break;
                
            case 39: // Arrow Right
                e.preventDefault();
                navigateToField(row, 'next');
                break;
                
            case 38: // Arrow Up
                e.preventDefault();
                navigateVerticallyFromIQD(row, -1);
                break;
                
            case 40: // Arrow Down
                e.preventDefault();
                navigateVerticallyFromIQD(row, 1);
                break;
                
            default:
                // منع أي إدخال آخر
                if (!isNavigationKey(e.which)) {
                    e.preventDefault();
                    return false;
                }
        }
    });
    
    // معالج للنقر على حقل IQD - توجيه التركيز للحقل التالي
    $(document).off('click focus', '.iqd-price-display').on('click focus', '.iqd-price-display', function(e) {
        var row = $(this).closest('tr');
        
        // إضافة تأخير قصير للسماح بإكمال حدث focus
        setTimeout(function() {
            // الانتقال للحقل التالي (Discount)
            navigateToField(row, 'next');
        }, 50);
    });
    
    // معالج خاص لمنع التحديد والتعديل
    $(document).off('select mouseup', '.iqd-price-display').on('select mouseup', '.iqd-price-display', function(e) {
        e.preventDefault();
        return false;
    });
}

// دالة التنقل بين الحقول من IQD
function navigateToField(row, direction) {
    var targetInput = null;
    
    if (direction === 'next') {
        // الانتقال للحقل التالي (Discount)
        targetInput = row.find('td:eq(7) input.discount_percent');
        
        if (!targetInput.length || !targetInput.is(':visible')) {
            // إذا لم يوجد حقل الخصم، انتقل لحقل السعر شامل الضريبة
            targetInput = row.find('td:eq(8) input.pos_unit_price_inc_tax');
        }
    } else if (direction === 'previous') {
        // الانتقال للحقل السابق (USD)
        targetInput = row.find('td:eq(5) input');
        
        if (!targetInput.length || !targetInput.is(':visible')) {
            // إذا لم يوجد حقل USD، انتقل للكمية
            targetInput = row.find('td:eq(4) input.pos_quantity');
        }
    }
    
    if (targetInput && targetInput.length && targetInput.is(':visible') && !targetInput.prop('readonly')) {
        targetInput.focus().select();
    }
}

// التنقل العمودي من حقل IQD
function navigateVerticallyFromIQD(currentRow, direction) {
    var targetRowIndex = currentRow.index() + direction;
    var targetRow = $('#pos_table tbody tr').eq(targetRowIndex);
    
    if (targetRow.length) {
        // محاولة العثور على حقل IQD في الصف المستهدف
        var targetIQDField = targetRow.find('.iqd-price-display');
        
        if (targetIQDField.length && targetIQDField.is(':visible')) {
            // إذا وُجد حقل IQD، انتقل إليه ثم للحقل التالي
            setTimeout(function() {
                navigateToField(targetRow, 'next');
            }, 50);
        } else {
            // إذا لم يوجد، انتقل لأول حقل قابل للتعديل في الصف
            var firstEditableInput = targetRow.find('input:visible:not([readonly]):not(.iqd-price-display)').first();
            if (firstEditableInput.length) {
                firstEditableInput.focus().select();
            }
        }
    }
}

// فحص ما إذا كان المفتاح مفتاح تنقل
function isNavigationKey(keyCode) {
    var navigationKeys = [
        9,   // Tab
        13,  // Enter
        27,  // Escape
        37,  // Arrow Left
        38,  // Arrow Up
        39,  // Arrow Right
        40,  // Arrow Down
        35,  // End
        36,  // Home
        33,  // Page Up
        34   // Page Down
    ];
    
    return navigationKeys.includes(keyCode);
}

// تحديث دالة handleKeyboardNavigation لتتعامل مع IQD
function enhancedHandleKeyboardNavigation(e) {
    var current = $(this);
    var row = current.closest('tr');
    var cell = current.closest('td');
    var cellIndex = cell.index();
    var rowIndex = row.index();
    var rows = $('#pos_table tbody tr');
    var isLastRow = rowIndex === rows.length - 1;
    
    // إذا كان الحقل الحالي هو IQD، استخدم المعالج الخاص
    if (current.hasClass('iqd-price-display')) {
        return; // المعالج الخاص سيتولى الأمر
    }
    
    switch(e.key) {
        case 'Tab':
            // تحسين التنقل بـ Tab للتعامل مع IQD
            if (!e.shiftKey && isNextFieldIQD(current)) {
                e.preventDefault();
                // تخطي IQD والانتقال للحقل الذي بعده
                skipIQDAndNavigate(current, 'forward');
            } else if (e.shiftKey && isPreviousFieldIQD(current)) {
                e.preventDefault();
                // تخطي IQD والانتقال للحقل الذي قبله
                skipIQDAndNavigate(current, 'backward');
            }
            // السماح بالسلوك الافتراضي في باقي الحالات
            break;
            
        case 'Enter':
            e.preventDefault();
            if (current.hasClass('product-search-input')) {
                return;
            }
            navigateToNextRowEnhanced(current);
            break;
            
        case 'ArrowUp':
            e.preventDefault();
            navigateVertically(current, -1);
            break;
            
        case 'ArrowDown':
            e.preventDefault();
            if (isLastRow && !row.hasClass('empty-row')) {
                $('#search_product').focus().select();
            } else {
                navigateVertically(current, 1);
            }
            break;
            
        case 'ArrowLeft':
            if (current.get(0).selectionStart === 0 || current.is('select')) {
                e.preventDefault();
                navigateHorizontallyEnhanced(current, -1);
            }
            break;
            
        case 'ArrowRight':
            if (current.get(0).selectionStart === current.val().length || current.is('select')) {
                e.preventDefault();
                navigateHorizontallyEnhanced(current, 1);
            }
            break;
            
        case 'Delete':
            if (e.ctrlKey) {
                e.preventDefault();
                deleteCurrentRow(current);
            }
            break;
    }
}

// فحص ما إذا كان الحقل التالي هو IQD
function isNextFieldIQD(current) {
    var row = current.closest('tr');
    var currentCellIndex = current.closest('td').index();
    var editableInputs = row.find('input:visible, select:visible');
    var currentIndex = editableInputs.index(current);
    var nextInput = editableInputs.eq(currentIndex + 1);
    
    return nextInput.length > 0 && nextInput.hasClass('iqd-price-display');
}

// فحص ما إذا كان الحقل السابق هو IQD
function isPreviousFieldIQD(current) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible, select:visible');
    var currentIndex = editableInputs.index(current);
    var previousInput = editableInputs.eq(currentIndex - 1);
    
    return previousInput.length > 0 && previousInput.hasClass('iqd-price-display');
}

// تخطي IQD والانتقال للحقل المناسب
function skipIQDAndNavigate(current, direction) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible:not(.iqd-price-display), select:visible');
    var currentIndex = editableInputs.index(current);
    
    var targetIndex = direction === 'forward' ? currentIndex + 1 : currentIndex - 1;
    
    if (targetIndex >= 0 && targetIndex < editableInputs.length) {
        var targetInput = editableInputs.eq(targetIndex);
        targetInput.focus();
        if (targetInput.is('input[type="text"], input[type="number"]')) {
            targetInput.select();
        }
    } else if (direction === 'forward') {
        // إذا وصلنا لنهاية الصف، انتقل للصف التالي
        navigateToNextRowEnhanced(current);
    } else {
        // إذا كنا في بداية الصف، انتقل للصف السابق
        navigateToPreviousRow(current);
    }
}

// تحسين التنقل الأفقي
function navigateHorizontallyEnhanced(current, direction) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible:not(.iqd-price-display), select:visible');
    var currentIndex = editableInputs.index(current);
    var targetIndex = currentIndex + direction;
    
    if (targetIndex >= 0 && targetIndex < editableInputs.length) {
        var targetInput = editableInputs.eq(targetIndex);
        targetInput.focus();
        if (targetInput.is('input[type="text"], input[type="number"]')) {
            targetInput.select();
        }
    }
}

// تحسين الانتقال للصف التالي
function navigateToNextRowEnhanced(current) {
    var row = current.closest('tr');
    var cellIndex = current.closest('td').index();
    var nextRow = row.next();
    
    if (nextRow.length === 0) {
        addEmptyProductRow();
        nextRow = $('#pos_table tbody tr').last();
    }
    
    // البحث عن أول حقل قابل للتعديل في نفس العمود أو قريب منه
    var targetInput = findBestInputInRow(nextRow, cellIndex);
    
    if (targetInput && targetInput.length) {
        targetInput.focus();
        if (targetInput.is('input[type="text"], input[type="number"]')) {
            targetInput.select();
        }
    }
}

// البحث عن أفضل حقل في الصف للانتقال إليه
function findBestInputInRow(row, preferredCellIndex) {
    // محاولة العثور على حقل في نفس العمود
    var preferredCell = row.find('td').eq(preferredCellIndex);
    var preferredInput = preferredCell.find('input:visible:not(.iqd-price-display):not([readonly]), select:visible');
    
    if (preferredInput.length > 0) {
        return preferredInput.first();
    }
    
    // إذا لم يوجد، ابحث عن أول حقل قابل للتعديل
    return row.find('input:visible:not(.iqd-price-display):not([readonly]), select:visible').first();
}

// الانتقال للصف السابق
function navigateToPreviousRow(current) {
    var row = current.closest('tr');
    var cellIndex = current.closest('td').index();
    var previousRow = row.prev();
    
    if (previousRow.length > 0) {
        var targetInput = findBestInputInRow(previousRow, cellIndex);
        if (targetInput && targetInput.length) {
            targetInput.focus();
            if (targetInput.is('input[type="text"], input[type="number"]')) {
                targetInput.select();
            }
        }
    }
}

// إعداد معالجات التنقل المحسنة
function setupEnhancedNavigationHandlers() {
    // إزالة المعالجات القديمة
    $(document).off('keydown', 'table#pos_table input, table#pos_table select');
    
    // إضافة المعالجات الجديدة
    $(document).on('keydown', 'table#pos_table input:not(.iqd-price-display), table#pos_table select', enhancedHandleKeyboardNavigation);
    
    // إعداد معالجات IQD الخاصة
    fixNavigationFromIQDField();
    
    // إضافة معالج للتأكد من عدم التعديل على IQD
    $(document).on('input paste', '.iqd-price-display', function(e) {
        e.preventDefault();
        return false;
    });
    
    // معالج لإظهار رسالة توضيحية عند محاولة التعديل
    $(document).on('keypress', '.iqd-price-display', function(e) {
        if (!isNavigationKey(e.which)) {
            e.preventDefault();
            showTooltipMessage($(this), 'هذا الحقل محسوب تلقائياً - استخدم Tab للانتقال');
            return false;
        }
    });
}

// عرض رسالة توضيحية مؤقتة
function showTooltipMessage(element, message) {
    var tooltip = $('<div class="iqd-tooltip">' + message + '</div>');
    
    tooltip.css({
        position: 'absolute',
        background: 'rgba(0,0,0,0.8)',
        color: 'white',
        padding: '5px 10px',
        borderRadius: '4px',
        fontSize: '12px',
        zIndex: 9999,
        whiteSpace: 'nowrap',
        top: element.offset().top - 35,
        left: element.offset().left,
        transform: 'translateX(-50%)'
    });
    
    $('body').append(tooltip);
    
    setTimeout(function() {
        tooltip.fadeOut(function() {
            tooltip.remove();
        });
    }, 2000);
}

// تهيئة النظام عند تحميل الصفحة
$(document).ready(function() {
    // انتظار قليل للتأكد من تحميل جميع العناصر
    setTimeout(function() {
        setupEnhancedNavigationHandlers();
    }, 1000);
});

// CSS إضافي لتحسين المظهر
var enhancedNavigationStyles = `
<style>
/* تحسين مظهر حقل IQD */
.iqd-price-display {
    position: relative;
}

.iqd-price-display:focus {
    outline: 2px solid #007bff !important;
    outline-offset: 1px;
}

/* أنماط الرسالة التوضيحية */
.iqd-tooltip {
    font-family: Arial, sans-serif;
    box-shadow: 0 2px 8px rgba(0,0,0,0.3);
    animation: fadeInTooltip 0.3s ease-in;
}

@keyframes fadeInTooltip {
    from {
        opacity: 0;
        transform: translateX(-50%) translateY(-5px);
    }
    to {
        opacity: 1;
        transform: translateX(-50%) translateY(0);
    }
}

/* تحسين مظهر التركيز */
.excel-input:focus {
    border-color: #007bff !important;
    box-shadow: 0 0 0 2px rgba(0, 123, 255, 0.25) !important;
}

/* مؤشر بصري للحقول غير القابلة للتعديل */
.iqd-price-display::before {
    content: "🔒";
    position: absolute;
    right: 5px;
    top: 50%;
    transform: translateY(-50%);
    font-size: 12px;
    opacity: 0.5;
    pointer-events: none;
}
</style>
`;

// إضافة الأنماط
$(document).ready(function() {
    if ($('#enhancedNavigationStyles').length === 0) {
        $('head').append('<style id="enhancedNavigationStyles">' + enhancedNavigationStyles + '</style>');
    }
});





// ============================================
// إصلاح نظام فلاتر العلامات التجارية - نسخة مُصححة نهائياً
// ============================================

$(document).ready(function() {
    // إزالة التهيئة السابقة وإعادة تهيئة النظام
    setTimeout(function() {
        reinitializeFilterSystem();
    }, 1000);
});

// إعادة تهيئة نظام الفلترة بالكامل
function reinitializeFilterSystem() {
    console.log('🔄 Reinitializing filter system...');
    
    // إزالة الفلاتر الموجودة
    $('.pos-filter-container').remove();
    
    // تهيئة المتغيرات العامة
    window.activeFilters = [];
    window.activeUnitFilter = null;
    window.fullPlasticFilterActive = null; // تغيير من undefined إلى null
    
    // إعادة إنشاء الفلاتر
    createBrandFiltersHTML();
    
    // إضافة معالجات الأحداث الجديدة
    attachBrandFilterEventHandlers();
    
    // تحديث دوال البحث
    overrideSearchFunctions();
    
    console.log('✅ Filter system reinitialized successfully');
}

// إنشاء HTML الفلاتر بالتصميم المدمج
function createBrandFiltersHTML() {
    var filterHTML = `
        <div class="row" style="margin-top: 10px;">
            <div class="col-md-12">
                <div class="cyber-filter-container">
                    <!-- محتوى الفلاتر -->
                    <div class="cyber-filter-content">
                        <!-- القسم الأول: العلامات التجارية -->
                        <div class="filter-section brands-section">
                    <!--        <span class="section-label">العلامات:</span>  -->
                            <div class="filter-buttons">
                                <button type="button" class="cyber-filter-btn brand-filter" data-filter="ADF" data-sku-prefix="ADF">
                                    <img src="http://192.168.116.181/public/images/ADF.png" alt="ADF">
                                    <span class="btn-label">ADF</span>
                                </button>
                                
                                <button type="button" class="cyber-filter-btn brand-filter" data-filter="ROYAL" data-sku-prefix="R">
                                    <img src="http://192.168.116.181/public/images/R.png" alt="ROYAL">
                                    <span class="btn-label">ROYAL</span>
                                </button>
                                
                                <button type="button" class="cyber-filter-btn brand-filter" data-filter="GOVIDAN" data-sku-prefix="G">
                                    <img src="http://192.168.116.181/public/images/G.png" alt="GOVIDAN">
                                    <span class="btn-label">GIVAUDAN</span>
                                </button>
                                
                                <button type="button" class="cyber-filter-btn brand-filter" data-filter="EURO" data-sku-prefix="EURO">
                                    <img src="http://192.168.116.181/public/images/N1.png" alt="EURO">
                                    <span class="btn-label">EURO</span>
                                </button>

                                
                                <button type="button" class="cyber-filter-btn brand-filter" data-filter="FLOR" data-sku-prefix="FL">
                                    <img src="http://192.168.116.181/public/images/FL.png" alt="FLOR">
                                    <span class="btn-label">FLOR</span>
                                </button>
                            </div>
                        </div>
                        
                        <!-- الفاصل -->
                        <div class="section-divider"></div>
                        
                        <!-- القسم الثاني: الأوزان -->
                        <div class="filter-section weights-section">
                <!--            <span class="section-label">الأوزان:</span>  -->
                            <div class="filter-buttons">
                                <!-- الأوزان الصغيرة -->
                                <div class="cyber-stack-container">
                                    <button type="button" class="cyber-stack-btn unit-filter" data-filter="50" data-unit-multiplier="0.05" data-unit-name="50 غم">
                                        <span>50g</span>
                                    </button>
                                    <button type="button" class="cyber-stack-btn unit-filter" data-filter="100" data-unit-multiplier="0.1" data-unit-name="100 غم">
                                        <span>100g</span>
                                    </button>
                                    <button type="button" class="cyber-stack-btn unit-filter" data-filter="125" data-unit-multiplier="0.125" data-unit-name="125غم">
                                        <span>125g</span>
                                    </button>
                                </div>
                                
                                <button type="button" class="cyber-filter-btn unit-filter weight-btn" data-filter="0.25" data-unit-multiplier="0.25">
                                    <span class="weight-value">¼</span>
                                    <span class="weight-label">ربع كيلو</span>
                                </button>
                                
                                <button type="button" class="cyber-filter-btn unit-filter weight-btn" data-filter="0.5" data-unit-multiplier="0.5">
                                    <span class="weight-value">½</span>
                                    <span class="weight-label">نصف كيلو</span>
                                </button>
                                
                                <button type="button" class="cyber-filter-btn unit-filter weight-btn" data-filter="1KG" data-unit-multiplier="1" data-unit-name="KG">
                                    <span class="weight-value">1</span>
                                    <span class="weight-label">كيلو</span>
                                </button>
                            </div>
                        </div>
                        
                        <!-- الفاصل -->
                        <div class="section-divider"></div>
                        
                        <!-- القسم الثالث: الإجراءات -->
                        <div class="filter-section actions-section">
                            <button type="button" class="cyber-action-btn special-filter" data-filter="FULL_PLASTIC" data-special-type="full-plastic">
                                <span class="action-icon">📦</span>
                                <span>فل بلاستك</span>
                            </button>
                            
                            <button type="button" class="cyber-action-btn excel-paste" id="excelPasteBtn" title="لصق من Excel">
                                <i class="fa fa-paste action-icon"></i>
                                <span>لصق Excel</span>
                            </button>
                            
                            <!-- أزرار النسخ المدمجة -->
                            <button type="button" class="cyber-action-btn copy-btn" id="select-all-btn" title="تحديد جميع الصفوف">
                                <i class="fa fa-check-square action-icon"></i>
                                <span>تحديد الكل</span>
                            </button>
                            
                            <button type="button" class="cyber-action-btn copy-btn" id="copy-selected-btn" title="نسخ الصفوف المحددة">
                                <i class="fa fa-copy action-icon"></i>
                                <span>نسخ المحدد</span>
                            </button>
                            
                            <button type="button" class="cyber-action-btn copy-btn" id="copy-all-btn" title="نسخ الجدول بالكامل">
                                <i class="fa fa-table action-icon"></i>
                                <span>نسخ الكل</span>
                            </button>
                            
                            <button type="button" class="cyber-action-btn copy-btn" id="export-excel-btn" title="تصدير إلى Excel">
                                <i class="fa fa-file-excel-o action-icon"></i>
                                <span>Excel</span>
                            </button>
                            
                            <button type="button" class="cyber-action-btn cyber-clear filter-clear">
                                <span class="action-icon">✕</span>
                                <span>مسح</span>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // العثور على أفضل مكان لإدراج الفلاتر
    var insertLocation = findFilterInsertLocation();
    if (insertLocation) {
        insertLocation.after(filterHTML);
    } else {
        console.warn('⚠️ Could not find suitable location for filters');
        $('#pos_table').closest('.row').before(filterHTML);
    }
    
    // إضافة الأنماط المدمجة
    addCompactCyberFilterStyles();
}

// إضافة الأنماط المدمجة
function addCompactCyberFilterStyles() {
    if ($('#cyberFilterStyles').length > 0) return;
    
    var styles = `
    <style id="cyberFilterStyles">
    /* الحاوي الرئيسي المدمج */
    .cyber-filter-container {
        background: linear-gradient(135deg, #0f1419 0%, #1a1f2e 100%);
        border-radius: 10px;
        padding: 10px 15px;
        position: relative;
        overflow: hidden;
        border: 1px solid rgba(56, 189, 248, 0.2);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.3);
        max-height: 52px;
        height: 52px;
    }
    
    /* محتوى الفلاتر */
    .cyber-filter-content {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 0;
        height: 100%;
    }
    
    /* أقسام الفلاتر */
    .filter-section {
        display: flex;
        align-items: center;
        gap: 8px;
        flex: 1;
        padding: 0 12px;
    }
    
    .filter-section:first-child {
        padding-left: 0;
    }
    
    .filter-section:last-child {
        padding-right: 0;
        flex: 0 0 auto;
    }
    
    /* الفواصل بين الأقسام */
    .section-divider {
        width: 1px;
        height: 50px;
        background: linear-gradient(to bottom, 
            transparent, 
            rgba(56, 189, 248, 0.3), 
            rgba(56, 189, 248, 0.3),
            transparent);
        margin: 0 8px;
    }
    
    .section-label {
        color: #94a3b8;
        font-size: 11px;
        font-weight: 600;
        white-space: nowrap;
        min-width: 45px;
    }
    
    .filter-buttons {
        display: flex;
        align-items: center;
        gap: 6px;
        flex: 1;
    }
    
    /* أزرار الفلاتر المدمجة */
    .cyber-filter-btn {
        position: relative;
        padding: 4px 8px;
        background: rgba(15, 23, 42, 0.8);
        border: 1px solid rgba(56, 189, 248, 0.3);
        border-radius: 6px;
        color: #cbd5e1;
        cursor: pointer;
        transition: all 0.2s ease;
        display: flex;
        align-items: center;
        gap: 4px;
        height: 32px;
        overflow: hidden;
    }
    
    .cyber-filter-btn:hover {
        background: rgba(56, 189, 248, 0.1);
        border-color: #38bdf8;
        transform: translateY(-1px);
        box-shadow: 0 2px 6px rgba(56, 189, 248, 0.2);
    }
    
    .cyber-filter-btn img {
        height: 18px;
        width: auto;
        object-fit: contain;
        filter: brightness(0.9);
    }
    
    .btn-label {
        font-size: 10px;
        font-weight: 500;
        white-space: nowrap;
    }
    
    /* أزرار الوزن المحسنة */
    .weight-btn {
        flex-direction: column;
        padding: 6px 12px;
        min-width: 55px;
        height: 42px !important;
        justify-content: center;
    }
    
    .weight-value {
        font-size: 18px;
        font-weight: 700;
        color: #38bdf8;
        line-height: 1;
        display: block;
    }
    
    .weight-label {
        font-size: 10px;
        font-weight: 500;
        color: #cbd5e1;
        line-height: 1.2;
        margin-top: 3px;
        display: block;
    }
    
    .weight-btn:hover .weight-label {
        color: #38bdf8;
    }
    
    .weight-btn.active .weight-value {
        color: #fff;
        text-shadow: 0 0 8px #38bdf8;
    }
    
    .weight-btn.active .weight-label {
        color:rgb(255, 255, 255);
    }
    
    /* حالة النشط */
    .cyber-filter-btn.active {
        background: linear-gradient(135deg, 
            rgba(56, 191, 248, 0.75), 
            rgba(129, 141, 248, 0.69));
        border-color: #38bdf8;
        color:rgb(255, 255, 255);
        box-shadow: 0 0 12px rgba(56, 189, 248, 0.3),
                    inset 0 0 12px rgba(56, 189, 248, 0.1);
    }
    
    .cyber-filter-btn.active img {
        filter: brightness(1.2) drop-shadow(0 0 3px #38bdf8);
    }
    
    /* حاوي الأوزان المكدسة المدمج */
    .cyber-stack-container {
        display: flex;
        background: rgba(15, 23, 42, 0.8);
        border: 1px solid rgba(56, 189, 248, 0.3);
        border-radius: 6px;
        overflow: hidden;
        height: 32px;
    }
    
    .cyber-stack-btn {
        flex: 1;
        padding: 0 8px;
        background: transparent;
        border: none;
        border-right: 1px solid rgba(56, 189, 248, 0.2);
        color: #cbd5e1;
        cursor: pointer;
        transition: all 0.2s ease;
        font-size: 10px;
        font-weight: 500;
        white-space: nowrap;
    }
    
    .cyber-stack-btn:last-child {
        border-right: none;
    }
    
    .cyber-stack-btn:hover {
        background: rgba(56, 189, 248, 0.1);
        color: #38bdf8;
    }
    
    .cyber-stack-btn.active {
        background: rgba(56, 191, 248, 0.82);
        color:rgb(255, 255, 255);
        font-weight: 700;
    }
    
    /* أزرار الإجراءات المدمجة */
    .cyber-action-btn {
        padding: 6px 12px;
        background: linear-gradient(135deg, #0f172a, #1e293b);
        border: 1px solid rgba(56, 189, 248, 0.3);
        border-radius: 6px;
        color: #cbd5e1;
        cursor: pointer;
        transition: all 0.2s ease;
        display: flex;
        align-items: center;
        gap: 4px;
        font-size: 10px;
        font-weight: 500;
        height: 32px;
    }
    
    .action-icon {
        font-size: 14px;
    }
    
    .cyber-action-btn:hover {
        background: linear-gradient(135deg, #1e293b, #334155);
        border-color: #38bdf8;
        transform: translateY(-1px);
        box-shadow: 0 2px 6px rgba(56, 189, 248, 0.2);
    }
    
    .cyber-action-btn.special-filter.active {
        background: linear-gradient(135deg, #f97316, #fb923c);
        border-color: #f97316;
        color: white;
        box-shadow: 0 0 12px rgba(249, 115, 22, 0.3);
    }
    
    /* زر لصق Excel */
    .cyber-action-btn.excel-paste {
        background: linear-gradient(135deg, #1e293b, #0f172a);
        border-color: rgba(34, 197, 94, 0.3);
    }
    
    .cyber-action-btn.excel-paste:hover {
        background: linear-gradient(135deg, #16a34a, #22c55e);
        border-color: #22c55e;
        color: white;
        box-shadow: 0 0 12px rgba(34, 197, 94, 0.3);
    }
    
    .cyber-action-btn.excel-paste i {
        font-size: 13px;
    }
    
    .cyber-clear {
        background: linear-gradient(135deg, #1e293b, #0f172a);
        border-color: rgba(239, 68, 68, 0.3);
    }
    
    .cyber-clear:hover {
        background: linear-gradient(135deg, #dc2626, #ef4444);
        border-color: #ef4444;
        color: white;
        box-shadow: 0 0 12px rgba(239, 68, 68, 0.3);
    }
    
    /* تأثير التحميل */
    .cyber-filter-btn.loading,
    .cyber-action-btn.loading {
        pointer-events: none;
        opacity: 0.6;
    }
    
    /* التجاوب */
    @media (max-width: 768px) {
        .cyber-filter-container {
            padding: 6px 10px;
            height: auto;
            max-height: none;
        }
        
        .cyber-filter-content {
            flex-direction: column;
            gap: 8px;
        }
        
        .section-divider {
            width: 80%;
            height: 1px;
            margin: 4px auto;
            background: linear-gradient(to right, 
                transparent, 
                rgba(56, 189, 248, 0.3), 
                rgba(56, 189, 248, 0.3),
                transparent);
        }
        
        .filter-section {
            width: 100%;
            justify-content: center;
            padding: 4px 0;
        }
        
        .section-label {
            font-size: 10px;
            min-width: 40px;
        }
        
        .cyber-filter-btn {
            height: 28px;
            padding: 4px 6px;
        }
        
        .cyber-filter-btn img {
            height: 16px;
        }
        
        .cyber-stack-container {
            height: 28px;
        }
        
        .cyber-action-btn {
            height: 28px;
            padding: 4px 8px;
        }
        
        .weight-btn {
            min-width: 50px;
            height: 36px !important;
        }
        
        .weight-value {
            font-size: 16px;
        }
        
        .weight-label {
            font-size: 9px;
        }
    }
    </style>
    `;
    
    $('head').append(styles);
}

// دالة helper للعثور على موقع إدراج الفلاتر
function findFilterInsertLocation() {
    // البحث عن شريط البحث أولاً
    var searchInput = $('#search_product');
    if (searchInput.length > 0) {
        var searchRow = searchInput.closest('.row');
        if (searchRow.length > 0) {
            return searchRow;
        }
    }
    
    // البحث عن أي container يحتوي على حقل البحث
    var searchContainer = $('.col-md-8').has('#search_product').closest('.row');
    if (searchContainer.length > 0) {
        return searchContainer;
    }
    
    // كبديل، البحث عن form container
    var formContainer = $('form#add_pos_sell_form, form#edit_pos_sell_form').find('.row').first();
    if (formContainer.length > 0) {
        return formContainer;
    }
    
    return null;
}

// إضافة معالجات الأحداث (نفس الكود السابق)
function attachBrandFilterEventHandlers() {
    console.log('🔗 Attaching cyber filter event handlers...');
    
    // إزالة جميع المعالجات السابقة
    $(document).off('click', '.cyber-filter-btn.brand-filter');
    $(document).off('click', '.cyber-filter-btn.unit-filter, .cyber-stack-btn.unit-filter');
    $(document).off('click', '.cyber-action-btn.special-filter');
    $(document).off('click', '.cyber-clear');
    
    // معالج النقر على فلاتر العلامات التجارية
    $(document).on('click', '.cyber-filter-btn.brand-filter', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $btn = $(this);
        var filterName = $btn.data('filter');
        var skuPrefix = $btn.data('sku-prefix');
        
        console.log('🎯 Brand filter clicked:', filterName, skuPrefix);
        
        // إضافة تأثير التحميل
        $btn.addClass('loading');
        
        setTimeout(function() {
            if ($btn.hasClass('active')) {
                // إلغاء تفعيل الفلتر
                $btn.removeClass('active');
                removeBrandFilter(skuPrefix, filterName);
                console.log('❌ Filter deactivated:', filterName);
            } else {
                // تفعيل الفلتر
                $btn.addClass('active');
                addBrandFilter(skuPrefix, filterName);
                console.log('✅ Filter activated:', filterName);
            }
            
            // تطبيق الفلاتر
            applyAllFilters();
            
            // إزالة تأثير التحميل
            $btn.removeClass('loading');
            
        }, 300);
    });
    
    // معالج النقر على فلاتر الوحدات
    $(document).on('click', '.cyber-filter-btn.unit-filter, .cyber-stack-btn.unit-filter', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $btn = $(this);
        var filterValue = $btn.data('filter');
        var multiplier = $btn.data('unit-multiplier');
        var unitName = $btn.data('unit-name') || filterValue;
        
        console.log('🎯 Unit filter clicked:', filterValue, multiplier);
        
        // إضافة تأثير التحميل للأزرار العادية فقط
        if ($btn.hasClass('cyber-filter-btn')) {
            $btn.addClass('loading');
        }
        
        setTimeout(function() {
            if ($btn.hasClass('active')) {
                // إلغاء تفعيل الفلتر الحالي
                $btn.removeClass('active');
                window.activeUnitFilter = null;
                console.log('❌ Unit filter deactivated:', filterValue);
            } else {
                // إلغاء تفعيل فلاتر الوحدات الأخرى
                $('.cyber-filter-btn.unit-filter, .cyber-stack-btn.unit-filter').removeClass('active');
                
                // تفعيل الفلتر الحالي
                $btn.addClass('active');
                window.activeUnitFilter = {
                    multiplier: parseFloat(multiplier),
                    name: unitName,
                    filter: filterValue
                };
                console.log('✅ Unit filter activated:', window.activeUnitFilter);
            }
            
            // تطبيق الفلاتر
            applyAllFilters();
            
            // إزالة تأثير التحميل
            if ($btn.hasClass('cyber-filter-btn')) {
                $btn.removeClass('loading');
            }
            
        }, 300);
    });
    
    // معالج النقر على فلتر فل بلاستك
    $(document).on('click', '.cyber-action-btn.special-filter', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $btn = $(this);
        
        // إضافة تأثير التحميل
        $btn.addClass('loading');
        
        setTimeout(function() {
            if ($btn.hasClass('active')) {
                $btn.removeClass('active');
                window.fullPlasticFilterActive = false;
                console.log('❌ Full plastic filter deactivated');
            } else {
                $btn.addClass('active');
                window.fullPlasticFilterActive = true;
                console.log('✅ Full plastic filter activated');
            }
            
            // تطبيق الفلاتر
            applyAllFilters();
            
            // إزالة تأثير التحميل
            $btn.removeClass('loading');
            
        }, 300);
    });
    
    // معالج زر المسح
    $(document).on('click', '.cyber-clear', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $btn = $(this);
        $btn.addClass('loading');
        
        setTimeout(function() {
            // مسح جميع الفلاتر
            clearAllFilters();
            
            // إزالة جميع الحالات النشطة
            $('.cyber-filter-btn, .cyber-stack-btn, .cyber-action-btn').removeClass('active');
            
            // إزالة تأثير التحميل
            $btn.removeClass('loading');
            
        }, 300);
    });
    
    // معالج النقر على زر لصق Excel
    $(document).on('click', '.cyber-action-btn.excel-paste', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $btn = $(this);
        
        // إضافة تأثير التحميل
        $btn.addClass('loading');
        
        setTimeout(function() {
            // إزالة تأثير التحميل
            $btn.removeClass('loading');
            
            // استدعاء الوظيفة الأصلية للصق من Excel إذا كانت موجودة
            if (typeof handleExcelPaste === 'function') {
                handleExcelPaste();
            } else {
                console.log('📋 Excel paste function triggered');
                // يمكن إضافة كود اللصق من Excel هنا
            }
        }, 300);
    });
    
    console.log('✅ Cyber event handlers attached successfully');
}

// إضافة فلتر العلامة التجارية
function addBrandFilter(skuPrefix, filterName) {
    // التحقق من عدم وجود الفلتر مسبقاً
    var existingFilter = window.activeFilters.find(f => f.prefix === skuPrefix);
    if (!existingFilter) {
        window.activeFilters.push({
            prefix: skuPrefix,
            name: filterName,
            type: 'brand'
        });
        console.log('➕ Added brand filter:', filterName, window.activeFilters);
    }
}

// إزالة فلتر العلامة التجارية
function removeBrandFilter(skuPrefix, filterName) {
    window.activeFilters = window.activeFilters.filter(f => f.prefix !== skuPrefix);
    console.log('➖ Removed brand filter:', filterName, window.activeFilters);
}

// تطبيق جميع الفلاتر (العلامات التجارية + الوحدات + فل بلاستك)
function applyAllFilters() {
    console.log('🔍 Applying all filters:', {
        brandFilters: window.activeFilters,
        unitFilter: window.activeUnitFilter,
        fullPlastic: window.fullPlasticFilterActive,
        hasSearchResults: shouldMaintainSearchResults && lastSearchResults.length > 0
    });
    
    var hasAnyFilters = window.activeFilters.length > 0 || 
                       window.activeUnitFilter !== null || 
                       window.fullPlasticFilterActive !== null;
    
    if (!hasAnyFilters) {
        // لا توجد فلاتر نشطة، استعادة البحث الأصلي
        restoreOriginalProductSearch();
        // إذا كانت هناك نتائج بحث محفوظة، إعادة تطبيقها بدلاً من إعادة التحميل
        if (shouldMaintainSearchResults && lastSearchResults.length > 0) {
            console.log('🔄 إعادة تطبيق نتائج البحث المحفوظة بدون فلاتر');
            updateAutocompleteWithResults(lastSearchResults);
        } else {
            reloadProductSuggestionList();
        }
        showSuccessMessage('تم مسح جميع الفلاتر');
    } else {
        // تطبيق الفلاتر
        if (shouldMaintainSearchResults && lastSearchResults.length > 0) {
            // تطبيق الفلاتر على النتائج المحفوظة
            console.log('🔄 تطبيق الفلاتر على النتائج المحفوظة');
            var filteredResults = applyAllFiltersToProducts([...lastSearchResults]);
            updateAutocompleteWithResults(filteredResults);
        } else {
            updateProductSearchWithAllFilters();
            reloadProductSuggestionList();
        }
        
        // إظهار رسالة الفلاتر النشطة
        var activeFiltersText = [];
        
        if (window.activeFilters.length > 0) {
            var brandNames = window.activeFilters.map(f => f.name).join(', ');
            activeFiltersText.push('العلامات: ' + brandNames);
        }
        
        if (window.activeUnitFilter) {
            activeFiltersText.push('الوحدة: ' + window.activeUnitFilter.filter);
        }
        
        if (window.fullPlasticFilterActive === true) {
            activeFiltersText.push('عرض جميع المنتجات بما فيها فل بلاستك');
        } else if (window.fullPlasticFilterActive === false) {
            activeFiltersText.push('إخفاء منتجات فل بلاستك');
        }
        
        showSuccessMessage('الفلاتر النشطة: ' + activeFiltersText.join(' | '));
    }
}

// تحديث البحث عن المنتجات مع جميع الفلاتر
function updateProductSearchWithAllFilters() {
    console.log('🔄 Updating product search with all filters...');
    
    // حفظ الدالة الأصلية إذا لم تكن محفوظة
    if (!window.originalSearchProducts) {
        window.originalSearchProducts = window.searchProducts;
    }
    
    // استبدال دالة البحث
    window.searchProducts = function(searchTerm, row, rowIndex) {
        console.log('🔍 Filtered search called with all filters:', {
            term: searchTerm,
            brandFilters: window.activeFilters,
            unitFilter: window.activeUnitFilter,
            fullPlastic: window.fullPlasticFilterActive
        });
        
        var price_group = $('#price_group').val() || '';
        
        var searchData = {
            price_group: price_group,
            term: searchTerm,
            not_for_selling: 0,
            limit: 100, // زيادة العدد للحصول على نتائج أكثر قبل الفلترة
            search_all_locations: true,
            include_all_warehouses: true,
            with_warehouse_stock: true,
            with_sub_units: true,
            include_unit_details: true,
            load_sub_units: true
        };
        
        $.ajax({
            url: base_path + '/products/list',
            method: 'GET',
            dataType: 'json',
            data: searchData,
            success: function(products) {
                var input = row.find('.product-search-input');
                input.removeClass('cell-loading');
                
                if (!products || !Array.isArray(products)) {
                    console.warn('Invalid products data received');
                    products = [];
                }
                
                console.log('📦 Products received before filtering:', products.length);
                
                // معالجة البيانات
                products.forEach(function(product) {
                    try {
                        processProductUnitsData(product);
                        processProductWarehouseData(product);
                    } catch (e) {
                        console.error('Error processing product data:', e);
                    }
                });
                
                // تطبيق جميع الفلاتر
                products = applyAllFiltersToProducts(products);
                console.log('📦 Products after all filtering:', products.length);
                
                // تطبيق فلتر الوحدة على المنتجات إذا لزم الأمر
                if (window.activeUnitFilter && products.length > 0) {
                    products = filterProductsByUnits(products);
                    console.log('📦 Products after unit filtering:', products.length);
                }
                
                // عرض النتائج
                if (products.length === 1) {
                    populateRowWithProduct(row, products[0], rowIndex);
                    
                    // تطبيق فلتر الوحدة تلقائياً
                    if (window.activeUnitFilter) {
                        setTimeout(function() {
                            applyUnitFilterToRow(row, window.activeUnitFilter);
                        }, 300);
                    }
                } else if (products.length > 1) {
                    showProductDropdown(input, products, row, rowIndex);
                } else {
                    var message = buildNoResultsMessage(searchTerm);
                    showWarningMessage(message);
                    clearRowData(row);
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                console.error('Search error:', textStatus, errorThrown);
                row.find('.product-search-input').removeClass('cell-loading');
                showErrorMessage('خطأ في البحث عن المنتجات');
            }
        });
    };
}

// متغير للتحكم في debug
window.debug025KgFilter = false; // تعيين true لتفعيل الـ debug

// دالة لتفعيل/إلغاء تفعيل debug
window.toggle025KgDebug = function(enable) {
    if (typeof enable === 'undefined') {
        window.debug025KgFilter = !window.debug025KgFilter;
    } else {
        window.debug025KgFilter = !!enable;
    }
    console.log('🔧 0.25kg Filter Debug:', window.debug025KgFilter ? 'ENABLED' : 'DISABLED');
    return window.debug025KgFilter;
};

// دالة اختبار سريع للفلترة
window.testUnitBasedFilter = function() {
    console.log('🧪 Testing unit-based filter (show only لك/فل_بلاستك)...');
    console.log('Current activeUnitFilter:', window.activeUnitFilter);
    console.log('Current fullPlasticFilterActive:', window.fullPlasticFilterActive);
    
    if (!window.activeUnitFilter) {
        console.log('❌ No active unit filter');
        return;
    }
    
    var filtersToApply = ['1KG', '0.5', '50', '100', '125', 50, 100, 125, 0.5];
    var activeFilterStr = String(window.activeUnitFilter.filter);
    var shouldApplyFiltering = filtersToApply.some(function(filter) {
        return String(filter) === activeFilterStr;
    });
    
    console.log('Filter value:', window.activeUnitFilter.filter, '(type:', typeof window.activeUnitFilter.filter, ')');
    console.log('Should apply unit-based filtering:', shouldApplyFiltering);
    console.log('Available filters:', filtersToApply);
    
    if (shouldApplyFiltering) {
        console.log('📋 When filtering is active:');
        console.log('  ✅ لк products: Always shown');
        console.log('  ' + (window.fullPlasticFilterActive === true ? '✅' : '🚫') + ' فل بلاستك products: ' + (window.fullPlasticFilterActive === true ? 'Shown (filter active)' : 'Hidden (filter not active)'));
        console.log('  🚫 Other products: Hidden');
    }
};

// دالة لاختبار وحدة أساسية لمنتج معين
window.testProductMainUnit = function(productName) {
    console.log('🧪 Testing main unit for product:', productName);
    
    // محاولة العثور على المنتج في البيانات المخزنة مؤقتاً
    if (window.lastSearchProducts && Array.isArray(window.lastSearchProducts)) {
        var product = window.lastSearchProducts.find(function(p) {
            return (p.name && p.name.toLowerCase().includes(productName.toLowerCase())) ||
                   (p.sku && p.sku.toLowerCase().includes(productName.toLowerCase()));
        });
        
        if (product) {
            console.log('Found product:', product.name || product.sku);
            var shouldHide = shouldHideProduct(product);
            console.log('Should hide product:', shouldHide);
        } else {
            console.log('❌ Product not found in last search results');
        }
    } else {
        console.log('❌ No search results available. Search for products first.');
    }
};

// دالة مساعدة لفحص بنية بيانات المنتج
function debugProductStructure(product) {
    if (!window.debug025KgFilter) return;
    
    console.log('🔍 Product structure debug:', {
        hasUnits: !!product.units,
        unitsType: typeof product.units,
        unitsIsArray: Array.isArray(product.units),
        unitKeys: product.units ? Object.keys(product.units) : 'N/A',
        productKeys: Object.keys(product),
        sampleData: {
            name: product.name,
            sku: product.sku,
            sub_sku: product.sub_sku,
            units: product.units
        }
    });
}

// دالة للتحقق من أن المنتج يجب إخفاؤه عند تفعيل فلاتر الوحدات
function shouldHideProduct(product) {
    try {
        var productName = (product.name || product.sku || '').toLowerCase();
        
        if (window.debug025KgFilter) {
            console.log('🔍 Checking if product should be hidden:', {
                name: productName,
                fullPlasticFilter: window.fullPlasticFilterActive,
                main_unit: product.unit,
                units: product.units
            });
        }
        
        // دالة مساعدة للحصول على اسم الوحدة الأساسية
        function getMainUnitName(product) {
            // فحص الوحدة الأساسية أولاً
            if (product.unit && typeof product.unit === 'object') {
                return (product.unit.unit_name || product.unit.name || product.unit.actual_name || '').trim().toLowerCase();
            }
            
            // إذا لم توجد، فحص الوحدة الأولى في القائمة
            var units = product.units || product.sub_units || [];
            if (units && Array.isArray(units) && units.length > 0) {
                var firstUnit = units[0];
                return (firstUnit.unit_name || firstUnit.name || firstUnit.actual_name || '').trim().toLowerCase();
            }
            
            return '';
        }
        
        var mainUnitName = getMainUnitName(product);
        
        if (window.debug025KgFilter) {
            console.log('🔍 Product main unit name:', mainUnitName);
        }
        
        // تحديد نوع المنتج
        var isLakProduct = mainUnitName.includes('لك');
        var isFullPlasticProduct = mainUnitName.includes('فل بلاستك');
        
        if (window.debug025KgFilter) {
            console.log('🔍 Product classification:', {
                isLak: isLakProduct,
                isFullPlastic: isFullPlasticProduct,
                fullPlasticFilterActive: window.fullPlasticFilterActive
            });
        }
        
        // المنطق الجديد:
        // 1. منتجات "لك": تظهر دائماً عند تفعيل فلاتر الوحدات
        if (isLakProduct) {
            if (window.debug025KgFilter) {
                console.log('✅ Showing لك product');
            }
            return false; // لا تخفي منتجات لك
        }
        
        // 2. منتجات "فل بلاستك": تظهر فقط إذا كان فلتر فل بلاستك مفعل
        if (isFullPlasticProduct) {
            var shouldShow = window.fullPlasticFilterActive === true;
            if (window.debug025KgFilter) {
                console.log(shouldShow ? '✅ Showing فل بلاستك product (filter active)' : '🚫 Hiding فل بلاستك product (filter not active)');
            }
            return !shouldShow; // اخف إذا لم يكن الفلتر مفعل
        }
        
        // 3. باقي المنتجات: تُخفى عند تفعيل فلاتر الوحدات (إظهار فقط لك وفل بلاستك)
        if (window.debug025KgFilter) {
            console.log('🚫 Hiding other product (not لك or فل بلاستك)');
        }
        return true; // اخف باقي المنتجات
        
    } catch (e) {
        console.error('Error checking if product should be hidden:', e);
        return false; // في حالة الخطأ، لا تخفي المنتج
    }
}

// دالة لفلترة المنتجات لإظهار فقط لك وفل بلاستك عند تفعيل فلاتر الوحدات
function filterProductsBy025KgUnit(products) {
    console.log('🔄 Starting unit-based filtering (show only لك/فل_بلاستك)...', {
        totalProducts: products.length,
        activeUnitFilter: window.activeUnitFilter,
        fullPlasticFilterActive: window.fullPlasticFilterActive
    });
    
    if (!window.activeUnitFilter) {
        console.log('ℹ️ No active unit filter, skipping unit-based filtering');
        return products;
    }
    
    // الفلاتر التي يجب تطبيق الفلترة عندها (إظهار فقط لك وفل بلاستك)
    var filtersToApply = ['1KG', '0.5', '50', '100', '125', 50, 100, 125, 0.5];
    
    // التحقق من أن الفلتر النشط من الفلاتر المحددة (تحويل إلى نص للمقارنة)
    var activeFilterStr = String(window.activeUnitFilter.filter);
    var shouldApplyFiltering = filtersToApply.some(function(filter) {
        return String(filter) === activeFilterStr;
    });
    
    if (!shouldApplyFiltering) {
        console.log('ℹ️ Active filter not in list to apply unit-based filtering:', window.activeUnitFilter.filter, '(type:', typeof window.activeUnitFilter.filter, ')');
        return products;
    }
    
    console.log('🔍 Applying unit-based filtering for filter:', window.activeUnitFilter.filter);
    console.log('📋 Full plastic filter status:', window.fullPlasticFilterActive);
    
    var hiddenCount = 0;
    var filteredProducts = products.filter(function(product) {
        var shouldHide = shouldHideProduct(product);
        if (shouldHide) {
            hiddenCount++;
            console.log('🚫 Hiding product (showing only لك/فل_بلاستك):', product.name || product.sku);
            return false; // إخفاء المنتج
        }
        return true; // إظهار المنتج
    });
    
    console.log('✅ Unit-based filtering complete (showing only لك/فل_بلاستك):', {
        originalCount: products.length,
        filteredCount: filteredProducts.length,
        hiddenCount: hiddenCount,
        fullPlasticFilterActive: window.fullPlasticFilterActive
    });
    
    return filteredProducts;
}

// تطبيق جميع الفلاتر على المنتجات
function applyAllFiltersToProducts(products) {
    console.log('🔄 applyAllFiltersToProducts called with:', {
        productsCount: products.length,
        activeFilters: window.activeFilters,
        activeUnitFilter: window.activeUnitFilter,
        fullPlasticFilterActive: window.fullPlasticFilterActive
    });
    
    var filteredProducts = products;
    
    // 1. فلاتر العلامات التجارية
    if (window.activeFilters.length > 0) {
        console.log('📋 Applying brand filters...');
        filteredProducts = filterProductsByBrand(filteredProducts);
        console.log('📋 After brand filtering:', filteredProducts.length);
    }
    
    // 2. فلتر فل بلاستك - إصلاح المنطق
    if (window.fullPlasticFilterActive === false) {
        console.log('📦 Applying plastic filter...');
        // إخفاء منتجات فل بلاستك فقط عندما يكون الفلتر غير مفعل
        filteredProducts = filterProductsByPlastic(filteredProducts);
        console.log('📦 After plastic filtering:', filteredProducts.length);
    }
    // إذا كان fullPlasticFilterActive === true أو null، نعرض جميع المنتجات
    
    // 3. فلتر إخفاء منتجات 0.25 كغم عند فلاتر معينة
    console.log('⚖️ About to apply 0.25kg filtering...');
    filteredProducts = filterProductsBy025KgUnit(filteredProducts);
    console.log('⚖️ After 0.25kg filtering:', filteredProducts.length);
    
    console.log('✅ applyAllFiltersToProducts complete. Final count:', filteredProducts.length);
    return filteredProducts;
}

// فلترة المنتجات حسب فل بلاستك - منطق مُصحح
function filterProductsByPlastic(products) {
    if (window.fullPlasticFilterActive === false) {
        // إخفاء منتجات فل بلاستك عندما يكون الفلتر غير مفعل
        return products.filter(function(product) {
            return !isFullPlasticProduct(product);
        });
    }
    
    // عرض جميع المنتجات عندما يكون الفلتر مفعل أو null
    return products;
}

// فلترة المنتجات حسب الوحدات المتاحة
function filterProductsByUnits(products) {
    if (!window.activeUnitFilter) {
        return products;
    }
    
    return products.filter(function(product) {
        return productHasRequiredUnit(product, window.activeUnitFilter);
    });
}

// التحقق من وجود الوحدة المطلوبة في المنتج - مُصحح
function productHasRequiredUnit(product, unitFilter) {
    // استخراج الوحدات المتاحة للمنتج
    var availableUnits = [];
    
    if (product.processed_units && product.processed_units.length > 0) {
        availableUnits = product.processed_units;
    } else if (product.units && product.units.length > 0) {
        availableUnits = product.units;
    } else if (product.sub_units && product.sub_units.length > 0) {
        availableUnits = product.sub_units;
    }
    
    // إضافة الوحدة الأساسية
    if (product.unit) {
        availableUnits.push({
            unit_name: product.unit,
            multiplier: 1,
            is_base_unit: 1
        });
    }
    
    // البحث عن الوحدة المطابقة
    return availableUnits.some(function(unit) {
        var unitMultiplier = parseFloat(unit.multiplier || 1);
        var unitName = (unit.unit_name || unit.name || '').trim();
        
        // تنظيف وتوحيد أسماء الوحدات
        var normalizedUnitName = unitName.replace(/\s+/g, ' ').toLowerCase();
        
        // للـ 1KG
        if (unitFilter.filter === '1KG') {
            return normalizedUnitName === 'kg' || normalizedUnitName.includes('kg');
        }
        
        // للوحدات 50، 100، 125 - استثناء الكيلوغرام
        if (unitFilter.filter === '50') {
            // التحقق من أن الوحدة تحتوي على 50 وليست كيلوغرام
            if (normalizedUnitName.includes('كغم') || normalizedUnitName.includes('كجم') || 
                normalizedUnitName.includes('kg') || normalizedUnitName.includes('كيلو')) {
                return false;
            }
            return normalizedUnitName === '50 غم' || 
                   normalizedUnitName === '50غم' || 
                   normalizedUnitName === '50 جم' ||
                   normalizedUnitName === '50جم' ||
                   normalizedUnitName === '50 جرام' ||
                   normalizedUnitName === '50جرام' ||
                   (normalizedUnitName.includes('50') && (normalizedUnitName.includes('غم') || normalizedUnitName.includes('جم') || normalizedUnitName.includes('جرام')));
        }
        
        if (unitFilter.filter === '100') {
            // التحقق من أن الوحدة تحتوي على 100 وليست كيلوغرام
            if (normalizedUnitName.includes('كغم') || normalizedUnitName.includes('كجم') || 
                normalizedUnitName.includes('kg') || normalizedUnitName.includes('كيلو')) {
                return false;
            }
            return normalizedUnitName === '100 غم' || 
                   normalizedUnitName === '100غم' || 
                   normalizedUnitName === '100 جم' ||
                   normalizedUnitName === '100جم' ||
                   normalizedUnitName === '100 جرام' ||
                   normalizedUnitName === '100جرام' ||
                   (normalizedUnitName.includes('100') && (normalizedUnitName.includes('غم') || normalizedUnitName.includes('جم') || normalizedUnitName.includes('جرام')));
        }
        
        if (unitFilter.filter === '125') {
            // التحقق من أن الوحدة تحتوي على 125 وليست كيلوغرام
            if (normalizedUnitName.includes('كغم') || normalizedUnitName.includes('كجم') || 
                normalizedUnitName.includes('kg') || normalizedUnitName.includes('كيلو')) {
                return false;
            }
            return normalizedUnitName === '125 غم' || 
                   normalizedUnitName === '125غم' || 
                   normalizedUnitName === '125 جم' ||
                   normalizedUnitName === '125جم' ||
                   normalizedUnitName === '125 جرام' ||
                   normalizedUnitName === '125جرام' ||
                   (normalizedUnitName.includes('125') && (normalizedUnitName.includes('غم') || normalizedUnitName.includes('جم') || normalizedUnitName.includes('جرام')));
        }
        
        // للوحدات الأخرى (0.25, 0.5)
        return Math.abs(unitMultiplier - unitFilter.multiplier) < 0.001;
    });
}

// التحقق من منتج فل بلاستك - محسن
function isFullPlasticProduct(product) {
    // التحقق من وحدة القياس الأساسية
    var baseUnit = (product.unit || '').toLowerCase();
    if (baseUnit.includes('فل بلاستك') || 
        baseUnit.includes('فل بلاستيك') || 
        baseUnit.includes('full plastic') || 
        baseUnit.includes('fl plastic')) {
        return true;
    }
    
    // التحقق من الوحدات الفرعية
    var allUnits = [];
    
    if (product.processed_units && Array.isArray(product.processed_units)) {
        allUnits = allUnits.concat(product.processed_units);
    }
    
    if (product.units && Array.isArray(product.units)) {
        allUnits = allUnits.concat(product.units);
    }
    
    if (product.sub_units && Array.isArray(product.sub_units)) {
        allUnits = allUnits.concat(product.sub_units);
    }
    
    var hasFullPlasticUnit = allUnits.some(function(unit) {
        var unitName = (unit.unit_name || unit.name || '').toLowerCase();
        return unitName.includes('فل بلاستك') || 
               unitName.includes('فل بلاستيك') ||
               unitName.includes('full plastic') || 
               unitName.includes('fl plastic');
    });
    
    if (hasFullPlasticUnit) {
        return true;
    }
    
    // التحقق من الوصف أو الاسم
    var productName = (product.name || '').toLowerCase();
    var productDescription = (product.description || '').toLowerCase();
    
    return productName.includes('فل بلاستك') || 
           productName.includes('فل بلاستيك') ||
           productName.includes('full plastic') ||
           productDescription.includes('فل بلاستك') ||
           productDescription.includes('فل بلاستيك') ||
           productDescription.includes('full plastic');
}

// تطبيق فلتر الوحدة على صف المنتج - مُصحح مع اختيار الوحدة الأقرب
function applyUnitFilterToRow(row, unitFilter) {
    var unitInput = row.find('.unit-input');
    var availableUnits = [];
    
    try {
        availableUnits = JSON.parse(unitInput.attr('data-available-units') || '[]');
    } catch (e) {
        console.error('Error parsing available units:', e);
        return;
    }
    
    // البحث عن الوحدة المطابقة
    var matchedUnit = null;
    var closestUnit = null;
    var closestDifference = Infinity;
    
    for (var i = 0; i < availableUnits.length; i++) {
        var unit = availableUnits[i];
        var unitMultiplier = parseFloat(unit.multiplier || 1);
        var unitName = (unit.unit_name || unit.name || '').trim().toLowerCase();
        
        // استثناء وحدات الكيلوغرام للوحدات 50، 100، 125
        var isKilogram = unitName.includes('كغم') || unitName.includes('كجم') || 
                        unitName.includes('kg') || unitName.includes('كيلو');
        
        // للـ 1KG
        if (unitFilter.filter === '1KG') {
            if (unitName === 'kg' || unitName.includes('kg')) {
                matchedUnit = unit;
                break;
            }
        }
        // للوحدات 50، 100، 125
        else if (unitFilter.filter === '50' && !isKilogram) {
            if (unitName.includes('50') && (unitName.includes('غم') || unitName.includes('جم') || unitName.includes('جرام'))) {
                matchedUnit = unit;
                break;
            }
            // البحث عن الوحدة الأقرب بناءً على المضاعف
            var targetMultiplier = 0.05; // 50 جرام = 0.05 كيلوجرام
            var difference = Math.abs(unitMultiplier - targetMultiplier);
            if (difference < closestDifference && !isKilogram) {
                closestUnit = unit;
                closestDifference = difference;
            }
        }
        else if (unitFilter.filter === '100' && !isKilogram) {
            if (unitName.includes('100') && (unitName.includes('غم') || unitName.includes('جم') || unitName.includes('جرام'))) {
                matchedUnit = unit;
                break;
            }
            // البحث عن الوحدة الأقرب بناءً على المضاعف
            var targetMultiplier = 0.1; // 100 جرام = 0.1 كيلوجرام
            var difference = Math.abs(unitMultiplier - targetMultiplier);
            if (difference < closestDifference && !isKilogram) {
                closestUnit = unit;
                closestDifference = difference;
            }
        }
        else if (unitFilter.filter === '125' && !isKilogram) {
            if (unitName.includes('125') && (unitName.includes('غم') || unitName.includes('جم') || unitName.includes('جرام'))) {
                matchedUnit = unit;
                break;
            }
            // البحث عن الوحدة الأقرب بناءً على المضاعف
            var targetMultiplier = 0.125; // 125 جرام = 0.125 كيلوجرام
            var difference = Math.abs(unitMultiplier - targetMultiplier);
            if (difference < closestDifference && !isKilogram) {
                closestUnit = unit;
                closestDifference = difference;
            }
        }
        // للوحدات الأخرى
        else if (Math.abs(unitMultiplier - unitFilter.multiplier) < 0.001) {
            matchedUnit = unit;
            break;
        }
    }
    
    // إذا لم نجد مطابقة دقيقة، نستخدم الوحدة الأقرب
    if (!matchedUnit && closestUnit) {
        matchedUnit = closestUnit;
        console.log('Using closest unit match:', {
            filter: unitFilter,
            closest_unit: closestUnit,
            difference: closestDifference
        });
    }
    
    if (matchedUnit) {
        // تطبيق الوحدة المطابقة
        var unitName = matchedUnit.unit_name || matchedUnit.name || 'EA';
        unitInput.val(unitName).trigger('change');
        
        console.log('Applied unit filter:', {
            filter: unitFilter,
            matched_unit: matchedUnit,
            unit_name: unitName
        });
        
        showSuccessMessage('تم تطبيق وحدة ' + unitFilter.filter + ' تلقائياً', 'info');
    } else {
        console.warn('No matching unit found for filter:', unitFilter);
    }
}

// بناء رسالة عدم وجود نتائج
function buildNoResultsMessage(searchTerm) {
    var messageParts = ['لم يتم العثور على منتجات'];
    
    if (window.activeFilters.length > 0) {
        var brandNames = window.activeFilters.map(f => f.name).join(' أو ');
        messageParts.push('من العلامات: ' + brandNames);
    }
    
    if (window.activeUnitFilter) {
        messageParts.push('بوحدة: ' + window.activeUnitFilter.filter);
    }
    
    if (window.fullPlasticFilterActive === false) {
        messageParts.push('(بدون منتجات فل بلاستك)');
    }
    
    if (searchTerm) {
        messageParts.push('تحتوي على: ' + searchTerm);
    }
    
    return messageParts.join(' ');
}

// إعادة تحميل قائمة اقتراحات المنتجات
function reloadProductSuggestionList() {
    if (typeof get_product_suggestion_list === 'function') {
        console.log('🔄 Reloading product suggestion list...');
        
        $('input#suggestion_page').val(1);
        $('div#product_list_body').html('');
        
        var location_id = $('input#location_id').val();
        var url = '/sells/pos/get-product-suggestion';
        
        // إضافة فلاتر العلامات التجارية إلى URL
        if (window.activeFilters.length > 0) {
            var prefixes = window.activeFilters.map(f => f.prefix).join(',');
            url += '?sku_filters=' + encodeURIComponent(prefixes);
        }
        
        get_product_suggestion_list(
            global_p_category_id,
            global_brand_id,
            location_id,
            url
        );
    }
}

// استعادة البحث الأصلي
function restoreOriginalProductSearch() {
    if (window.originalSearchProducts) {
        window.searchProducts = window.originalSearchProducts;
        console.log('🔄 Original search function restored');
    }
}

// مسح جميع الفلاتر (العلامات التجارية + الوحدات + فل بلاستك)
function clearAllFilters() {
    console.log('🧹 Clearing all filters...');
    
    // إزالة جميع الفلاتر النشطة
    window.activeFilters = [];
    window.activeUnitFilter = null;
    window.fullPlasticFilterActive = null;
    
    // إزالة فئة active من جميع الأزرار
    $('.filter-btn').removeClass('active');
    $('.stacked-filter-container').removeClass('has-active');
    
    // استعادة البحث الأصلي
    restoreOriginalProductSearch();
    
    // إعادة تحميل قائمة المنتجات
    reloadProductSuggestionList();
    
    // تحديث مؤشر الفلاتر النشطة
    updateActiveFiltersIndicator();
    
    showSuccessMessage('تم مسح جميع الفلاتر');
}

// تحديث مؤشر الفلاتر النشطة - محسن
function updateActiveFiltersIndicator() {
    var indicator = $('.active-filters-indicator');
    var totalCount = window.activeFilters.length;
    
    if (window.activeUnitFilter) {
        totalCount++;
    }
    
    if (window.fullPlasticFilterActive !== null) {
        totalCount++;
    }
    
    if (totalCount > 0) {
        indicator.show();
        $('#activeFiltersCount').text(totalCount);
    } else {
        indicator.hide();
    }
}

// فلترة المنتجات حسب العلامة التجارية - محسن
function filterProductsByBrand(products) {
    if (!window.activeFilters || window.activeFilters.length === 0) {
        return products;
    }
    
    return products.filter(function(product) {
        var sku = product.sub_sku || product.sku || '';
        var productName = product.name || '';
        
        // التحقق من مطابقة SKU أو اسم المنتج مع الفلاتر النشطة
        return window.activeFilters.some(function(filter) {
            var prefix = filter.prefix.toUpperCase();
            var skuUpper = sku.toUpperCase();
            var nameUpper = productName.toUpperCase();
            
            // فحص SKU
            if (skuUpper.startsWith(prefix)) {
                return true;
            }
            
            // فحص اسم المنتج للحالات الخاصة
            if (prefix === 'EURO' && (nameUpper.includes('EURO') || skuUpper.includes('N1'))) {
                return true;
            }
            
            if (prefix === 'FL' && nameUpper.includes('FLOR')) {
                return true;
            }
            
            return false;
        });
    });
}

// تحديث البحث الرئيسي للعمل مع الفلاتر
function overrideSearchFunctions() {
    // تحديث autocomplete للبحث الرئيسي
    setTimeout(function() {
        if ($('#search_product').data('ui-autocomplete')) {
            var autocomplete = $('#search_product').data('ui-autocomplete');
            
            // حفظ المصدر الأصلي
            if (!window.originalAutocompleteSource) {
                window.originalAutocompleteSource = autocomplete.options.source;
            }
            
            autocomplete.options.source = function(request, response) {
                var searchData = {
                    price_group: $('#price_group').val() || '',
                    location_id: $('input#location_id').val(),
                    term: request.term,
                    not_for_selling: 0,
                    search_fields: [],
                    include_warehouse_details: 1
                };
                
                // جمع حقول البحث
                $('.search_fields:checked').each(function(i){
                    searchData.search_fields[i] = $(this).val();
                });
                
                $.getJSON('/products/list', searchData, function(data) {
                    // تطبيق فلاتر العلامات التجارية
                    if (window.activeFilters.length > 0) {
                        data = filterProductsByBrand(data);
                    }
                    
                    // تطبيق فلتر فل بلاستك
                    if (window.fullPlasticFilterActive === false) {
                        data = filterProductsByPlastic(data);
                    }
                    
                    response(data);
                });
            };
        }
    }, 1500);
}

// دوال الرسائل المحسنة
function showSuccessMessage(message) {
    if (typeof toastr !== 'undefined') {
        toastr.success(message);
    } else {
        console.log('✅ ' + message);
    }
}

function showWarningMessage(message) {
    if (typeof toastr !== 'undefined') {
        toastr.warning(message);
    } else {
        console.log('⚠️ ' + message);
    }
}

function showErrorMessage(message) {
    if (typeof toastr !== 'undefined') {
        toastr.error(message);
    } else {
        console.log('❌ ' + message);
    }
}

// دالة تشخيص المشاكل - محسنة
function debugFilterSystem() {
    console.log('🔍 Enhanced Filter System Debug Info:');
    console.log('='.repeat(50));
    
    // معلومات الفلاتر النشطة
    console.log('📊 Active Filters:');
    console.log('  Brand filters:', window.activeFilters);
    console.log('  Unit filter:', window.activeUnitFilter);
    console.log('  Full plastic filter:', window.fullPlasticFilterActive);
    
    // معلومات أزرار الفلاتر
    console.log('🎛️ Filter Buttons:');
    console.log('  Brand filter buttons found:', $('.filter-btn.brand-filter').length);
    console.log('  Unit filter buttons found:', $('.filter-btn.unit-filter').length);
    console.log('  Special filter buttons found:', $('.filter-btn.special-filter').length);
    
    // حالة الأزرار
    console.log('🔘 Button States:');
    $('.filter-btn.brand-filter').each(function(index) {
        var $btn = $(this);
        console.log(`  Brand Button ${index + 1}:`, {
            filter: $btn.data('filter'),
            prefix: $btn.data('sku-prefix'),
            active: $btn.hasClass('active'),
            visible: $btn.is(':visible')
        });
    });
    
    $('.filter-btn.unit-filter').each(function(index) {
        var $btn = $(this);
        console.log(`  Unit Button ${index + 1}:`, {
            filter: $btn.data('filter'),
            multiplier: $btn.data('unit-multiplier'),
            unitName: $btn.data('unit-name'),
            active: $btn.hasClass('active'),
            visible: $btn.is(':visible')
        });
    });
    
    // معلومات النظام
    console.log('🔧 System Info:');
    console.log('  Search function overridden:', typeof window.originalSearchProducts !== 'undefined');
    console.log('  jQuery version:', $.fn.jquery);
    console.log('  Page URL:', window.location.href);
    console.log('  Filter container found:', $('.pos-filter-container').length > 0);
    
    // اختبار الأحداث
    console.log('⚡ Event Test:');
    var testBtn = $('.filter-btn.brand-filter').first();
    if (testBtn.length > 0) {
        console.log('  Test button found:', testBtn.data('filter'));
        console.log('  Click handler attached:', $._data(testBtn[0], 'events') !== undefined);
    }
    
    console.log('='.repeat(50));
    
    // اختبار تلقائي للفلاتر
    if (window.location.search.includes('debug=filters')) {
        setTimeout(function() {
            console.log('🧪 Running automatic filter test...');
            testFilterFunctionality();
        }, 1000);
    }
}

// اختبار تلقائي لوظائف الفلاتر
function testFilterFunctionality() {
    console.log('🧪 Testing filter functionality...');
    
    // اختبار تفعيل فلتر العلامة التجارية
    var adfBtn = $('.filter-btn.brand-filter[data-filter="ADF"]');
    if (adfBtn.length > 0) {
        console.log('Testing ADF filter...');
        adfBtn.click();
        
        setTimeout(function() {
            console.log('ADF filter active:', adfBtn.hasClass('active'));
            console.log('Active filters:', window.activeFilters);
            
            // اختبار إلغاء التفعيل
            adfBtn.click();
            setTimeout(function() {
                console.log('ADF filter deactivated:', !adfBtn.hasClass('active'));
                console.log('Filters cleared:', window.activeFilters.length === 0);
            }, 500);
        }, 500);
    }
}

// إضافة دالة للوحة التحكم (اختيارية)
window.filterDebug = debugFilterSystem;

// إضافة رسالة تأكيد التحميل
$(document).ready(function() {
    setTimeout(function() {
        if ($('.pos-filter-container').length > 0) {
            console.log('✅ Brand filters loaded successfully!');
            console.log('📌 Full plastic filter behavior:');
            console.log('   - When active (ON): Show all products including full plastic');
            console.log('   - When inactive (OFF): Hide full plastic products');
            console.log('   - Default: Show all products');
        } else {
            console.warn('⚠️ Brand filters failed to load');
        }
    }, 2000);
});

















// ============================================
// تحسينات جدول البحث - نسخة محسنة مع عرض ثابت في الأعلى
// ============================================

// إضافة الأنماط المحسنة أولاً
var enhancedSearchTableStyles = `
<style id="enhancedSearchTableStyles">
/* حاوي البحث الثابت في الأعلى */
.product-dropdown {
    position: fixed !important;
    top: 50px !important;
    left: 50% !important;
    transform: translateX(-50%) !important;
    width: 95vw !important;
    max-width: 1400px !important;
    height: 70vh !important;
    max-height: 80vh !important;
    z-index: 9999 !important;
    background: white;
    border: 1px solid #666;
    border-radius: 4px;
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    overflow: hidden;
    display: flex;
    flex-direction: column;
}

/* شريط المعلومات العلوي */
.search-info-bar {
    background: #333;
    color: white;
    padding: 12px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-weight: normal;
    font-size: 14px;
    border-bottom: 1px solid #666;
}

.search-info-bar .search-term {
    font-size: 16px;
    background: #555;
    padding: 4px 12px;
    border-radius: 4px;
}

.search-info-bar .results-count {
    background: #ddd;
    color: #333;
    padding: 4px 12px;
    border-radius: 4px;
    font-weight: normal;
}

.search-info-bar .close-search {
    background: #555;
    border: none;
    color: white;
    width: 30px;
    height: 30px;
    border-radius: 4px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
}

.search-info-bar .close-search:hover {
    background: #777;
}

/* حاوي الجدول مع التمرير */
.table-container {
    flex: 1;
    overflow: auto;
    position: relative;
    background: white;
}

/* الجدول المحسن */
.product-search-table {
    width: 100% !important;
    border-collapse: collapse !important;
    margin: 0 !important;
    background: white;
    position: relative;
}

/* رأس الجدول الثابت */
.product-search-table thead {
    position: sticky !important;
    top: 0 !important;
    z-index: 100 !important;
    background: white;
}

.product-search-table thead th {
    background: #f5f5f5 !important;
    color: #333 !important;
    padding: 15px 8px !important;
    font-weight: bold !important;
    text-align: center !important;
    font-size: 13px !important;
    border: 1px solid #ddd !important;
    position: relative;
}

.product-search-table thead th:first-child {
    border-left: 1px solid #ddd;
}

.product-search-table thead th:last-child {
    border-right: 1px solid #ddd;
}

/* إزالة مؤشر الترتيب */
.product-search-table thead th::after {
    display: none;
}

/* جسم الجدول */
.product-search-table tbody {
    background: white;
}

.product-search-table tbody tr {
    cursor: pointer;
    border-bottom: 1px solid #ddd;
}

/* حالة التمرير - إضافة بوردر فقط */
.product-search-table tbody tr:hover {
    border: 2px solid #333 !important;
}

/* الصف المحدد بالتنقل - بوردر فقط */
.product-search-table tbody tr.highlighted {
    border: 2px solid #666 !important;
}

.product-search-table tbody tr.highlighted td {
    color: inherit !important;
    font-weight: normal;
    border-color: #ddd !important;
}

.product-search-table tbody tr.highlighted .badge {
    background: #666 !important;
    color: white !important;
    font-weight: normal;
}

/* خلايا الجدول */
.product-search-table td {
    padding: 12px 8px !important;
    vertical-align: middle !important;
    font-size: 13px !important;
    border: 1px solid #ddd !important;
    position: relative;
}

/* تحسين المؤشرات والبيانات - ألوان رمادية */
.badge {
    font-weight: normal;
    padding: 6px 10px;
    font-size: 11px;
    border-radius: 4px;
    display: inline-flex;
    align-items: center;
    gap: 4px;
}

.badge-success { background: #666; color: white; }
.badge-warning { background: #999; color: white; }
.badge-danger { background: #333; color: white; }
.badge-info { background: #777; color: white; }
.badge-secondary { background: #555; color: white; }

/* تحسين عرض المواقع */
.locations-grid {
    display: flex;
    flex-wrap: wrap;
    gap: 6px;
    max-width: 100%;
}

.location-badge {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    white-space: nowrap;
    font-size: 12px;
    padding: 3px 8px;
    border-radius: 4px;
    background: #ddd;
    color: #333;
}

/* شريط التمرير البسيط */
.table-container {
    scrollbar-width: thin;
    scrollbar-color: #999 #f1f1f1;
}

.table-container::-webkit-scrollbar {
    width: 12px;
    height: 12px;
}

.table-container::-webkit-scrollbar-track {
    background: #f1f1f1;
}

.table-container::-webkit-scrollbar-thumb {
    background: #999;
    border-radius: 4px;
}

.table-container::-webkit-scrollbar-thumb:hover {
    background: #777;
}

.table-container::-webkit-scrollbar-corner {
    background: #f1f1f1;
}

/* إرشادات التنقل */
.navigation-hints {
    position: absolute;
    bottom: 10px;
    right: 20px;
    background: rgba(0,0,0,0.8);
    color: white;
    padding: 8px 12px;
    border-radius: 4px;
    font-size: 11px;
    z-index: 200;
    display: flex;
    gap: 10px;
}

.navigation-hints .hint {
    display: flex;
    align-items: center;
    gap: 4px;
}

.navigation-hints .key {
    background: #555;
    padding: 2px 6px;
    border-radius: 4px;
    font-family: monospace;
    font-size: 10px;
}

/* حالة التحميل */
.search-loading {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255,255,255,0.9);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.search-loading .spinner {
    width: 50px;
    height: 50px;
    border: 4px solid #ddd;
    border-top: 4px solid #666;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* تحسينات للشاشات الصغيرة */
@media (max-width: 768px) {
    .product-dropdown {
        top: 20px !important;
        left: 10px !important;
        right: 10px !important;
        width: auto !important;
        transform: none !important;
        height: calc(100vh - 40px) !important;
    }
    
    .product-search-table thead th {
        padding: 10px 4px !important;
        font-size: 11px !important;
    }
    
    .product-search-table td {
        padding: 8px 4px !important;
        font-size: 11px !important;
    }
    
    .search-info-bar {
        padding: 8px 12px;
        font-size: 12px;
    }
    
    .navigation-hints {
        display: none;
    }
}
</style>
`;

// إضافة الأنماط إلى الصفحة
if ($('#enhancedSearchTableStyles').length === 0) {
    $('head').append(enhancedSearchTableStyles);
}



// ============================================
// نظام تصنيف المنتجات التلقائي حسب الكود
// ============================================

// خريطة أكواد المنتجات والفئات المقابلة لها
const productCategoryMap = {
    'adf': 'ADF',
    'r': 'ROYAL',
    'g': 'Givaudan', 
    'n1': 'EURO',
    'euro': 'EURO',
    'fl': 'FLOR'
};

// ألوان الفئات المختلفة للعرض
const categoryColors = {
    'ADF': { bg: '#007bff', text: 'white' },
    'ROYAL': { bg: '#28a745', text: 'white' },
    'Givaudan': { bg: '#ffc107', text: 'black' },
    'EURO': { bg: '#17a2b8', text: 'white' },
    'FLOR': { bg: '#dc3545', text: 'white' },
    'default': { bg: '#6c757d', text: 'white' }
};

// ============================================
// دالة لتحديد الفئة بناء على رمز المنتج
// ============================================
function getProductCategory(productCode, productName) {
    if (!productCode || typeof productCode !== 'string') {
        productCode = '';
    }
    
    // تحويل رمز المنتج إلى أحرف صغيرة للمقارنة
    const lowerCode = productCode.toLowerCase();
    
    // البحث عن التطابق في بداية الرمز
    for (const [prefix, category] of Object.entries(productCategoryMap)) {
        if (lowerCode.startsWith(prefix)) {
            return category;
        }
    }
    
    // طرق إضافية لتحديد الفئة - تحليل الاسم إذا كان الـ SKU غير واضح
    if (typeof productName === 'string' && productName) {
        const lowerName = productName.toLowerCase();
        
        if (lowerName.includes('adf')) return 'ADF';
        if (lowerName.includes('royal')) return 'ROYAL';
        if (lowerName.includes('givaudan')) return 'Givaudan';
        if (lowerName.includes('euro')) return 'EURO';
        if (lowerName.includes('flor')) return 'FLOR';
    }
    
    return null; // لا يوجد تطابق
}

// ============================================
// دالة لإضافة الفئة إلى المنتج
// ============================================
function addCategoryToProduct(product) {
    if (!product) return product;
    
    // تجنب إعادة التصنيف إذا كان المنتج مصنف بالفعل بشكل صحيح
    if (product.category && product.auto_categorized) {
        return product;
    }
    
    // استخدام sub_sku أولاً، ثم sku
    var productCode = product.sub_sku || product.sku || '';
    var productName = product.name || '';
    
    // محاولة تحديد الفئة من الكود أولاً
    var category = getProductCategory(productCode, productName);
    
    // تطبيق الفئة إذا وجدت
    if (category) {
        product.category = category;
        // إضافة معلومات إضافية للتتبع
        product.auto_categorized = true;
        product.category_source = productCode ? 'sku_prefix' : 'product_name';
    }
    
    return product;
}

// ============================================
// دالة لمعالجة مجموعة من المنتجات
// ============================================
function processProductsWithCategories(products) {
    if (!Array.isArray(products)) {
        return products;
    }
    
    return products.map(product => addCategoryToProduct(product));
}

// ============================================
// تكامل مع نظام الفلاتر الموجود
// ============================================
function processProductDataWithCategories(product) {
    try {
        // تطبيق التصنيف التلقائي أولاً
        product = addCategoryToProduct(product);
        
        // معالجة بيانات الوحدات (إذا كانت الدالة موجودة)
        if (typeof processProductUnitsData === 'function') {
            processProductUnitsData(product);
        }
        
        // معالجة بيانات المستودعات (إذا كانت الدالة موجودة)
        if (typeof processProductWarehouseData === 'function') {
            processProductWarehouseData(product);
        }
        
        return product;
    } catch (e) {
        console.error('Error processing product data with categories:', e);
        return product;
    }
}

// ============================================
// تحديث دالة createTableCell لتحسين عرض الفئات
// ============================================

// ============================================
// تحديث دالة البحث لتشمل التصنيف التلقائي
// ============================================

// تحديث دالة البحث الأصلية
function enhanceSearchWithAutoCategories() {
    // حفظ الدالة الأصلية إذا لم تكن محفوظة
    if (!window.originalSearchProductsWithCategories && window.searchProducts) {
        window.originalSearchProductsWithCategories = window.searchProducts;
    }
    
    // إذا كانت دالة البحث المحسنة موجودة، نحديثها
    if (window.originalSearchProducts) {
        var originalEnhancedSearch = window.originalSearchProducts;
        
        window.searchProducts = function(searchTerm, row, rowIndex) {
            console.log('🔍 Enhanced search with auto-categories called:', searchTerm);
            
            var price_group = $('#price_group').val() || '';
            
            var searchData = {
                price_group: price_group,
                term: searchTerm,
                not_for_selling: 0,
                limit: 100,
                search_all_locations: true,
                include_all_warehouses: true,
                with_warehouse_stock: true,
                with_sub_units: true,
                include_unit_details: true,
                load_sub_units: true
            };
            
            $.ajax({
                url: base_path + '/products/list',
                method: 'GET',
                dataType: 'json',
                data: searchData,
                success: function(products) {
                    var input = row.find('.product-search-input');
                    input.removeClass('cell-loading');
                    
                    if (!products || !Array.isArray(products)) {
                        console.warn('Invalid products data received');
                        products = [];
                    }
                    
                    console.log('📦 Products received before processing:', products.length);
                    
                    // معالجة البيانات مع التصنيف التلقائي
                    products.forEach(function(product) {
                        product = processProductDataWithCategories(product);
                    });
                    
                    console.log('📋 Products after auto-categorization:', products.length);
                    
                    // تطبيق الفلاتر إذا كانت موجودة
                    if (typeof applyAllFiltersToProducts === 'function') {
                        products = applyAllFiltersToProducts(products);
                        console.log('📦 Products after filtering:', products.length);
                    }
                    
                    // تطبيق فلتر الوحدة إذا كان نشطاً
                    if (window.activeUnitFilter && products.length > 0) {
                        if (typeof filterProductsByUnits === 'function') {
                            products = filterProductsByUnits(products);
                            console.log('📦 Products after unit filtering:', products.length);
                        }
                    }
                    
                    // عرض النتائج
                    if (products.length === 1) {
                        populateRowWithProduct(row, products[0], rowIndex);
                        
                        // تطبيق فلتر الوحدة تلقائياً
                        if (window.activeUnitFilter && typeof applyUnitFilterToRow === 'function') {
                            setTimeout(function() {
                                applyUnitFilterToRow(row, window.activeUnitFilter);
                            }, 300);
                        }
                    } else if (products.length > 1) {
                        if (typeof showProductDropdown === 'function') {
                            showProductDropdown(input, products, row, rowIndex);
                        }
                    } else {
                        var message = 'لم يتم العثور على منتجات';
                        if (searchTerm) {
                            message += ' تحتوي على: ' + searchTerm;
                        }
                        if (typeof showWarningMessage === 'function') {
                            showWarningMessage(message);
                        }
                        if (typeof clearRowData === 'function') {
                            clearRowData(row);
                        }
                    }
                },
                error: function(jqXHR, textStatus, errorThrown) {
                    console.error('Search error:', textStatus, errorThrown);
                    row.find('.product-search-input').removeClass('cell-loading');
                    if (typeof showErrorMessage === 'function') {
                        showErrorMessage('خطأ في البحث عن المنتجات');
                    }
                }
            });
        };
    }
}

// ============================================
// تحديث جدول البحث لعرض الفئة
// ============================================

// تحديث إعدادات أعمدة البحث لتشمل الفئة
function updateSearchTableColumnSettings() {
    var currentSettings = getSearchTableColumnSettings();
    
    // التحقق من وجود عمود الفئة
    var categoryColumnExists = currentSettings.some(col => col.id === 'category');
    
    if (!categoryColumnExists) {
        // إضافة عمود الفئة
        currentSettings.splice(1, 0, {
            id: 'category',
            name: 'الفئة',
            visible: true,
            width: 100,
            order: 1
        });
        
        // إعادة ترقيم الأعمدة
        currentSettings.forEach((col, index) => {
            col.order = index;
        });
        
        // حفظ الإعدادات المحدثة
        localStorage.setItem('searchTableColumnsSettings', JSON.stringify(currentSettings));
        
        console.log('📋 Category column added to search table settings');
    }
}

// تحديث دالة إنشاء خلية الجدول لتشمل الفئة
function createTableCellWithCategory(product, columnId) {
    var td = $('<td>');
    
    switch(columnId) {
        case 'product_name':
            var productName = product.name;
            if (product.type === 'variable') {
                productName += '<br><small class="text-primary">(' + product.variation + ')</small>';
            }
            if (product.has_multiple_locations) {
                productName;
            }
            td.html(productName);
            break;
            
        case 'category':
            var category = product.category || '-';
            if (category !== '-') {
                var badgeClass = 'badge-secondary';
                
                // تخصيص لون البادج حسب الفئة
                switch(category) {
                    case 'ADF':
                        badgeClass = 'badge-primary';
                        break;
                    case 'ROYAL':
                        badgeClass = 'badge-success';
                        break;
                    case 'GOVIDAN':
                        badgeClass = 'badge-warning';
                        break;
                    case 'EURO':
                        badgeClass = 'badge-info';
                        break;
                    case 'FLOR':
                        badgeClass = 'badge-danger';
                        break;
                }
                
                var categoryHtml = '<span class="badge ' + badgeClass + '">' + category + '</span>';
                
                // إضافة مؤشر للتصنيف التلقائي
                if (product.auto_categorized) {
                    categoryHtml += ' <small class="text-muted" title="تم التصنيف تلقائياً">  </small>';
                }
                
                td.html(categoryHtml);
            } else {
                td.html('<span class="text-muted">-</span>');
            }
            break;
            
        case 'foreign_name':
            td.text(product.product_custom_field1 || '-');
            break;
            
        case 'sku':
            var sku = product.sub_sku || product.sku || '-';
            td.html('<span class="badge badge-secondary">' + sku + '</span>');
            break;
            
        case 'price_usd':
            var priceUSD = parseFloat(product.variation_group_price || product.selling_price || 0);
            td.html('$' + formatNumber(priceUSD, 2))
              .css({'text-align': 'right', 'font-weight': '600'});
            break;
            
        case 'price_iqd':
            var priceUSD = parseFloat(product.variation_group_price || product.selling_price || 0);
            var priceIQD = convertUSDToIQD(priceUSD);
            td.html(formatNumber(priceIQD, 0) + ' د.ع')
              .css({'text-align': 'right', 'font-weight': '600'});
            break;
            
        case 'discount':
            var discount = product.discount_percent || 0;
            if (discount > 0) {
                td.html('<span class="badge badge-success">' + discount + '%</span>');
            } else {
                td.html('<span class="text-muted">-</span>');
            }
            break;
            
        case 'uom':
            td.html('<span class="badge badge-secondary">' + (product.unit || 'قطعة') + '</span>');
            break;
            
        case 'total_stock':
            var totalStock = product.total_stock || product.qty_available || 0;
            var stockBadgeClass = '';
            
            if (product.enable_stock == 1) {
                if (totalStock <= 0) {
                    stockBadgeClass = 'badge-danger';
                } else if (totalStock <= 10) {
                    stockBadgeClass = 'badge-warning';
                } else {
                    stockBadgeClass = 'badge-success';
                }
                
                td.html('<span class="badge ' + stockBadgeClass + '">' + formatNumber(totalStock, 0) + '</span>');
            } else {
                td.html('<span class="text-muted">غير محدود</span>');
            }
            break;
            
        case 'locations':
            var warehouseDetails = product.warehouse_details || [];
            if (warehouseDetails.length > 0) {
                var locationsHTML = '<div class="locations-grid">';
                
                warehouseDetails.forEach(function(location) {
                    var qty = location.available || location.quantity || 0;
                 

                    if (qty > 0) {
    var badgeClass = qty <= 5 ? 'badge-warning' : 'badge-success';
    
    // تحويل الكمية إلى نص
    var qtyString = formatNumber(qty, 0);
    
    locationsHTML += '<span class="location-badge ' + badgeClass + '">' +
                     '<span class="first-part" style="background:#6c757d;color: white;font-weight: bold;width:20px;">' + location.code + '</span>' +
                     '<span class="second-part" style="   color: #8a0c0cff;display: inline-block;font-weight: bold;">' + qtyString + '</span>' +
                     '</span> ';
}


                });
                
                locationsHTML += '</div>';
                td.html(locationsHTML);
            } else {
                td.html('<span class="text-muted">غير متوفر</span>');
            }
            break;
            
        default:
            td.text('-');
    }
    
    return td;
}
function enhanceSearchWithAutoCategories() {
    // حفظ الدالة الأصلية إذا لم تكن محفوظة
    if (!window.originalSearchProductsWithCategories && window.searchProducts) {
        window.originalSearchProductsWithCategories = window.searchProducts;
    }
    
    // تحديث دالة البحث
    window.searchProducts = function(searchTerm, row, rowIndex) {
        console.log('🔍 Enhanced search with auto-categories called:', searchTerm);
        
        var price_group = $('#price_group').val() || '';
        
        var searchData = {
            price_group: price_group,
            term: searchTerm,
            not_for_selling: 0,
            limit: 100,
            search_all_locations: true,
            include_all_warehouses: true,
            with_warehouse_stock: true,
            with_sub_units: true,
            include_unit_details: true,
            load_sub_units: true
        };
        
        $.ajax({
            url: base_path + '/products/list',
            method: 'GET',
            dataType: 'json',
            data: searchData,
            success: function(products) {
                var input = row.find('.product-search-input');
                input.removeClass('cell-loading');
                
                if (!products || !Array.isArray(products)) {
                    console.warn('Invalid products data received');
                    products = [];
                }
                
                console.log('📦 Products received before processing:', products.length);
                
                // معالجة البيانات مع التصنيف التلقائي
                products.forEach(function(product) {
                    processProductDataWithCategories(product);
                });
                
                console.log('📋 Products after auto-categorization:', products.length);
                
                // تطبيق الفلاتر إذا كانت موجودة
                if (typeof applyAllFiltersToProducts === 'function') {
                    products = applyAllFiltersToProducts(products);
                    console.log('📦 Products after filtering:', products.length);
                }
                
                // تطبيق فلتر الوحدة إذا كان نشطاً
                if (window.activeUnitFilter && products.length > 0 && typeof filterProductsByUnits === 'function') {
                    products = filterProductsByUnits(products);
                    console.log('📦 Products after unit filtering:', products.length);
                }
                
                // عرض النتائج
                if (products.length === 1) {
                    populateRowWithProduct(row, products[0], rowIndex);
                    
                    // تطبيق فلتر الوحدة تلقائياً
                    if (window.activeUnitFilter && typeof applyUnitFilterToRow === 'function') {
                        setTimeout(function() {
                            applyUnitFilterToRow(row, window.activeUnitFilter);
                        }, 300);
                    }
                } else if (products.length > 1) {
                    // استخدام الدالة المحسنة إذا كانت متاحة
                    if (typeof showProductDropdownEnhanced === 'function') {
                        showProductDropdownEnhanced(input, products, row, rowIndex);
                    } else if (typeof showProductDropdown === 'function') {
                        showProductDropdown(input, products, row, rowIndex);
                    }
                } else {
                    var message = 'لم يتم العثور على منتجات';
                    if (searchTerm) {
                        message += ' تحتوي على: ' + searchTerm;
                    }
                    if (typeof showWarningMessage === 'function') {
                        showWarningMessage(message);
                    }
                    if (typeof clearRowData === 'function') {
                        clearRowData(row);
                    }
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                console.error('Search error:', textStatus, errorThrown);
                row.find('.product-search-input').removeClass('cell-loading');
                if (typeof showErrorMessage === 'function') {
                    showErrorMessage('خطأ في البحث عن المنتجات');
                }
            }
        });
    };
}
// ============================================
// تحديث نظام الفلاتر لتشمل فلترة حسب الفئة
// ============================================

// ============================================
// تهيئة النظام
// ============================================

// تهيئة نظام التصنيف التلقائي
function initializeAutoCategorySystem() {
    console.log('🚀 Initializing auto-category system...');
    
    // تحديث إعدادات أعمدة الجدول
    updateSearchTableColumnSettings();
    
    // إضافة فلاتر الفئات
    setTimeout(function() {
        addCategoryFilter();
        attachCategoryFilterEventHandlers();
    }, 1000);
    
    // تحديث دالة البحث
    setTimeout(function() {
        enhanceSearchWithAutoCategories();
    }, 1500);
    
    // تحديث دالة إنشاء خلايا الجدول
    if (typeof window.createTableCell === 'undefined') {
        window.createTableCell = createTableCellWithCategory;
    }
    
    // تهيئة متغيرات النظام
    if (!window.activeCategoryFilters) {
        window.activeCategoryFilters = [];
    }
    
    console.log('✅ Auto-category system initialized successfully');
}

// تحديث دالة تطبيق جميع الفلاتر لتشمل فلاتر الفئات

// ============================================
// تشغيل النظام عند تحميل الصفحة
// ============================================

$(document).ready(function() {
    setTimeout(function() {
        initializeAutoCategorySystem();
        updateApplyAllFilters();
    }, 2000);
    
    console.log('📋 Product auto-categorization system loaded');
});

// تصدير الدوال للاستخدام الخارجي
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        getProductCategory,
        addCategoryToProduct,
        processProductsWithCategories,
        productCategoryMap,
        filterProductsByCategory,
        addActiveCategoryFilter,
        removeCategoryFilter
    };
}

// إضافة الدوال للكونسول للاختبار
window.categorySystem = {
    getProductCategory,
    addCategoryToProduct,
    processProductsWithCategories,
    productCategoryMap,
    test: function(sku) {
        console.log('Testing SKU:', sku);
        console.log('Category:', getProductCategory(sku));
    }
};


/**
 * تعديل نظام فلاتر الوحدات - استبدال الدوال مباشرة
 * الغرض: إزالة فلترة المنتجات حسب الوحدة مع الاحتفاظ بتطبيق الوحدة تلقائياً عند اختيار المنتج
 */

$(document).ready(function() {
    // انتظار تحميل النظام الأصلي
    setTimeout(function() {
        // تطبيق التعديلات
        applyUnitFilterFixes();
        
        console.log('✅ تم تطبيق تعديلات فلتر الوحدات بنجاح');
    }, 2000);
});

/**
 * تطبيق جميع التعديلات المطلوبة على نظام فلاتر الوحدات
 */
function applyUnitFilterFixes() {
    // إضافة دوال مساعدة لتحديث السعر
    window.updatePriceBasedOnUnitSelection = function(row, unit) {
        console.log('Updating price based on unit selection:', unit);
        
        // الحصول على السعر الأساسي
        var basePrice = parseFloat(row.find('.hidden_base_unit_sell_price').val()) || 0;
        var multiplier = parseFloat(unit.multiplier || 1);
        
        // حساب السعر الجديد بناءً على المضاعف
        var newPrice = basePrice * multiplier;
        
        // التعديلات الإضافية على السعر حسب الوحدة
        var additionalAmount = 0;
        
        if (Math.abs(multiplier - 0.5) < 0.001) {
            additionalAmount = 0; // إضافة دولار واحد للنصف
        } else if (Math.abs(multiplier - 0.25) < 0.001) {
            additionalAmount = 0; // إضافة دولارين للربع
        } else if (Math.abs(multiplier - 0.125) < 0.001) {
            additionalAmount = 0; // إضافة دولار واحد للثمن
        }
         else if (Math.abs(multiplier - 0.100) < 0.001) {
            additionalAmount = 0; // إضافة دولار واحد للثمن
        }
        
        // تطبيق المبلغ الإضافي
        newPrice = newPrice + additionalAmount;
        
        console.log('Price calculation:', {
            basePrice: basePrice,
            multiplier: multiplier,
            additionalAmount: additionalAmount,
            finalPrice: newPrice
        });
        
        // تحديث حقول السعر
        var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
        var newPriceIncTax = newPrice * exchange_rate;
        
        // تحديث السعر الأساسي
        __write_number(row.find('.pos_unit_price'), newPrice);
        
        // تحديث السعر شامل الضريبة
        __write_number(row.find('.pos_unit_price_inc_tax'), newPriceIncTax);
        
        // تحديث حقول العرض
        if (row.find('td:eq(5) input').length > 0) {
            row.find('td:eq(5) input').val(formatNumber(newPrice, 2));
        }
        
        // تحديث السعر بالدينار العراقي
        var iqrPrice = convertUSDToIQD(newPrice);
        if (row.find('td:eq(6) input').length > 0) {
            row.find('td:eq(6) input').val(formatNumber(iqrPrice, 0));
        }
        
        // تحديث المجموع
        var quantity = parseFloat(row.find('.pos_quantity').val()) || 0;
        var lineTotal = quantity * newPriceIncTax;
        
        __write_number(row.find('.pos_line_total'), lineTotal, false);
        row.find('.pos_line_total_text').text(__currency_trans_from_en(lineTotal, true));
        
        // إعادة حساب المجاميع
        if (typeof pos_total_row === 'function') {
            pos_total_row();
        }
        
        return newPrice;
    };
    
    // دالة للتأكد من تحديث السعر بشكل صحيح
    window.ensurePriceIsUpdated = function(row, multiplier) {
        // التحقق من أن السعر قد تم تحديثه بشكل صحيح
        var basePrice = parseFloat(row.find('.hidden_base_unit_sell_price').val()) || 0;
        var currentPrice = __read_number(row.find('.pos_unit_price'));
        var expectedBasePrice = basePrice * multiplier;
        
        // إذا كان السعر لا يزال قريباً من السعر الأساسي، نقوم بإعادة تحديثه
        if (Math.abs(currentPrice - expectedBasePrice) < 0.1 || currentPrice < 0.01) {
            console.log('Price update not detected, forcing update:', {
                currentPrice: currentPrice,
                expectedBasePrice: expectedBasePrice,
                multiplier: multiplier
            });
            
            // استخدام دالة مساعدة من النظام الأصلي إذا كانت موجودة
            if (typeof updatePriceBasedOnUnitWithAddition === 'function') {
                updatePriceBasedOnUnitWithAddition(row, multiplier);
            } else {
                // محاولة إعادة تطبيق الوحدة وتحديث السعر
                var unitSelection = row.find('select.sub_unit');
                if (unitSelection.length > 0) {
                    unitSelection.trigger('change');
                } else {
                    // تحديث السعر يدوياً
                    var newPrice = basePrice * multiplier;
                    
                    // إضافات السعر الخاصة
                    if (Math.abs(multiplier - 0.5) < 0.001) {
                   //     newPrice += 1; // +$1 للنصف
                    } else if (Math.abs(multiplier - 0.25) < 0.001) {
                     //   newPrice += 2; // +$2 للربع
                    } else if (Math.abs(multiplier - 0.125) < 0.001) {
                   //     newPrice += 1; // +$1 للثمن
                    } else if (Math.abs(multiplier - 0.100) < 0.001) {
                    //    newPrice += 1; // +$1 للثمن
                    }

                    
                    // تطبيق السعر الجديد
                    __write_number(row.find('.pos_unit_price'), newPrice);
                    row.find('.pos_unit_price').trigger('change');
                }
            }
        }
    };
    
    // دالة مساعدة لتنسيق الأرقام إذا لم تكن موجودة
    if (typeof formatNumber !== 'function') {
        window.formatNumber = function(number, decimals = 2) {
            return parseFloat(number || 0).toFixed(decimals);
        };
    }
    // 1. تعديل دالة applyAllFilters
    window.applyAllFilters = function() {
        console.log('🔍 Applying all filters:', {
            brandFilters: window.activeFilters,
            unitFilter: window.activeUnitFilter,
            fullPlastic: window.fullPlasticFilterActive
        });
        
        var hasAnyFilters = window.activeFilters.length > 0 || 
                           window.fullPlasticFilterActive !== undefined;
        
        // ملاحظة: أزلنا window.activeUnitFilter !== null من شرط hasAnyFilters
        
        if (!hasAnyFilters) {
            // لا توجد فلاتر نشطة، استعادة البحث الأصلي
            restoreOriginalProductSearch();
            reloadProductSuggestionList();
            showSuccessMessage('تم مسح جميع الفلاتر');
        } else {
            // تطبيق الفلاتر
            updateProductSearchWithAllFilters();
            reloadProductSuggestionList();
            
            // إظهار رسالة الفلاتر النشطة
            var activeFiltersText = [];
            
            if (window.activeFilters.length > 0) {
                var brandNames = window.activeFilters.map(f => f.name).join(', ');
                activeFiltersText.push('العلامات: ' + brandNames);
            }
            
            if (window.activeUnitFilter) {
                activeFiltersText.push('الوحدة: ' + window.activeUnitFilter.filter);
            }
            
            if (window.fullPlasticFilterActive === true) {
                activeFiltersText.push('فل بلاستك');
            } else if (window.fullPlasticFilterActive === false) {
                activeFiltersText.push('إخفاء فل بلاستك');
            }
            
            if (activeFiltersText.length > 0) {
                showSuccessMessage('الفلاتر النشطة: ' + activeFiltersText.join(' | '));
            }
        }
    };

    // 2. تعديل دالة applyAllFiltersToProducts
    window.applyAllFiltersToProducts = function(products) {
        var filteredProducts = products;
        
        // 1. فلاتر العلامات التجارية
        if (window.activeFilters.length > 0) {
            filteredProducts = filterProductsByBrand(filteredProducts);
        }
        
        // 2. فلتر فل بلاستك
        if (window.fullPlasticFilterActive !== undefined) {
            filteredProducts = filterProductsByPlastic(filteredProducts);
        }
        
        // ملاحظة: أزلنا فلترة الوحدات هنا
        
        return filteredProducts;
    };

    // 3. تعديل دالة updateProductSearchWithAllFilters
    window.updateProductSearchWithAllFilters = function() {
        console.log('🔄 Updating product search with all filters...');
        
        // حفظ الدالة الأصلية إذا لم تكن محفوظة
        if (!window.originalSearchProducts) {
            window.originalSearchProducts = window.searchProducts;
        }
        
        // استبدال دالة البحث
        window.searchProducts = function(searchTerm, row, rowIndex) {
            console.log('🔍 Filtered search called with all filters:', {
                term: searchTerm,
                brandFilters: window.activeFilters,
                unitFilter: window.activeUnitFilter,
                fullPlastic: window.fullPlasticFilterActive
            });
            
            var price_group = $('#price_group').val() || '';
            
            var searchData = {
                price_group: price_group,
                term: searchTerm,
                not_for_selling: 0,
                limit: 100, // زيادة العدد للحصول على نتائج أكثر قبل الفلترة
                search_all_locations: true,
                include_all_warehouses: true,
                with_warehouse_stock: true,
                with_sub_units: true,
                include_unit_details: true,
                load_sub_units: true
            };
            
            $.ajax({
                url: base_path + '/products/list',
                method: 'GET',
                dataType: 'json',
                data: searchData,
                success: function(products) {
                    var input = row.find('.product-search-input');
                    input.removeClass('cell-loading');
                    
                    if (!products || !Array.isArray(products)) {
                        console.warn('Invalid products data received');
                        products = [];
                    }
                    
                    console.log('📦 Products received before filtering:', products.length);
                    
                    // معالجة البيانات
                    products.forEach(function(product) {
                        try {
                            processProductUnitsData(product);
                            processProductWarehouseData(product);
                        } catch (e) {
                            console.error('Error processing product data:', e);
                        }
                    });
                    
                    // تطبيق جميع الفلاتر (بدون فلتر الوحدة)
                    products = applyAllFiltersToProducts(products);
                    console.log('📦 Products after all filtering:', products.length);
                    
                    // ملاحظة: أزلنا فلترة الوحدات هنا
                    
                    // عرض النتائج
                    if (products.length === 1) {
                        populateRowWithProduct(row, products[0], rowIndex);
                        
                        // تطبيق فلتر الوحدة على المنتج إذا كان الفلتر نشطاً
                        if (window.activeUnitFilter) {
                            setTimeout(function() {
                                applyUnitFilterToRow(row, window.activeUnitFilter);
                            }, 300);
                        }
                    } else if (products.length > 1) {
                        if (typeof showProductDropdown === 'function') {
                            showProductDropdown(input, products, row, rowIndex);
                        } else if (typeof showProductDropdownEnhanced === 'function') {
                            showProductDropdownEnhanced(input, products, row, rowIndex);
                        }
                    } else {
                        var message = buildNoResultsMessage(searchTerm);
                        showWarningMessage(message);
                        clearRowData(row);
                    }
                },
                error: function(jqXHR, textStatus, errorThrown) {
                    console.error('Search error:', textStatus, errorThrown);
                    row.find('.product-search-input').removeClass('cell-loading');
                    showErrorMessage('خطأ في البحث عن المنتجات');
                }
            });
        };
    };

    // 4. تحسين دالة applyUnitFilterToRow مع التأكيد على تحديث السعر
    window.applyUnitFilterToRow = function(row, unitFilter) {
        var unitInput = row.find('.unit-input');
        var availableUnits = [];
        
        try {
            availableUnits = JSON.parse(unitInput.attr('data-available-units') || '[]');
        } catch (e) {
            console.error('Error parsing available units:', e);
            return;
        }
        
        // البحث عن الوحدة المطابقة
        var matchedUnit = null;
        
        for (var i = 0; i < availableUnits.length; i++) {
            var unit = availableUnits[i];
            var unitMultiplier = parseFloat(unit.multiplier || 1);
            
            // للـ 1KG، نبحث عن وحدة KG
            if (unitFilter.name === '1KG' || unitFilter.filter === '1KG') {
                if ((unit.unit_name || unit.name || '').toUpperCase() === 'KG') {
                    matchedUnit = unit;
                    break;
                }
            } 
            // للوحدات الأخرى، نبحث عن المضاعف المطابق
            else if (Math.abs(unitMultiplier - unitFilter.multiplier) < 0.001) {
                matchedUnit = unit;
                break;
            }
        }
        
        if (matchedUnit) {
            // تطبيق الوحدة المطابقة
            var unitName = matchedUnit.unit_name || matchedUnit.name || 'EA';
            var unitMultiplier = parseFloat(matchedUnit.multiplier || 1);
            
            // تطبيق الوحدة وتحديث السعر بطريقتين لضمان التوافق مع النظام
            
            // الطريقة 1: باستخدام حقل الإدخال والـ trigger (الطريقة الأصلية)
            unitInput.val(unitName);
            
            // نسجل بيانات الوحدة في الحقول المخفية قبل تشغيل trigger
            row.find('.sub_unit_id').val(matchedUnit.id || '');
            row.find('.unit_multiplier').val(unitMultiplier);
            row.find('.base_unit_multiplier').val(unitMultiplier);
            row.find('.allow_decimal').val(matchedUnit.allow_decimal || 1);
            row.find('.is_base_unit').val(matchedUnit.is_base_unit || 0);
            
            // تحديث السعر بناءً على المضاعف
            updatePriceBasedOnUnitSelection(row, matchedUnit);
            
            // تشغيل الحدث بعد تحديث البيانات (يمكن أن يؤدي هذا إلى تحديث إضافي للسعر)
            unitInput.trigger('change');
            
            // الطريقة 2: إذا كانت هناك select.sub_unit متاحة
            var subUnitSelect = row.find('select.sub_unit');
            if (subUnitSelect.length > 0) {
                // البحث عن الخيار المناسب
                var optionToSelect = subUnitSelect.find('option').filter(function() {
                    return parseFloat($(this).data('multiplier')) === unitMultiplier;
                });
                
                if (optionToSelect.length > 0) {
                    subUnitSelect.val(optionToSelect.val()).trigger('change');
                }
            }
            
            // تأكيد تحديث السعر بعد كل الخطوات السابقة
            setTimeout(function() {
                ensurePriceIsUpdated(row, unitMultiplier);
            }, 300);
            
            console.log('Applied unit filter with price update:', {
                filter: unitFilter,
                matched_unit: matchedUnit,
                unit_name: unitName,
                multiplier: unitMultiplier
            });
            
            if (typeof showSuccessMessage === 'function') {
                showSuccessMessage('تم تطبيق وحدة ' + unitFilter.filter + ' وتحديث السعر تلقائياً', 'info');
            } else if (typeof toastr !== 'undefined') {
                toastr.success('تم تطبيق وحدة ' + unitFilter.filter + ' وتحديث السعر تلقائياً');
            }
        } else {
            console.warn('No matching unit found for filter:', unitFilter);
            // إظهار رسالة توضح أن الوحدة المطلوبة غير متوفرة لهذا المنتج
            if (typeof showWarningMessage === 'function') {
                showWarningMessage('الوحدة ' + unitFilter.filter + ' غير متوفرة لهذا المنتج');
            } else if (typeof toastr !== 'undefined') {
                toastr.warning('الوحدة ' + unitFilter.filter + ' غير متوفرة لهذا المنتج');
            }
        }
    };

    // 5. تعديل دالة buildNoResultsMessage
    window.buildNoResultsMessage = function(searchTerm) {
        var messageParts = ['لم يتم العثور على منتجات'];
        
        if (window.activeFilters.length > 0) {
            var brandNames = window.activeFilters.map(f => f.name).join(' أو ');
            messageParts.push('من العلامات: ' + brandNames);
        }
        
        // ملاحظة: أزلنا الوحدة من رسالة عدم وجود نتائج
        
        if (window.fullPlasticFilterActive === true) {
            messageParts.push('من نوع فل بلاستك');
        } else if (window.fullPlasticFilterActive === false) {
            messageParts.push('(باستثناء فل بلاستك)');
        }
        
        if (searchTerm) {
            messageParts.push('تحتوي على: ' + searchTerm);
        }
        
        return messageParts.join(' ');
    };

    // 6. تعديل دالة updateActiveFiltersIndicator
    window.updateActiveFiltersIndicator = function() {
        var indicator = $('.active-filters-indicator');
        var totalCount = window.activeFilters.length;
        
        if (window.activeUnitFilter) {
            totalCount++; // نحسب فلتر الوحدة ضمن العدد الإجمالي
        }
        
        if (window.fullPlasticFilterActive !== undefined) {
            totalCount++;
        }
        
        if (totalCount > 0) {
            indicator.show();
            $('#activeFiltersCount').text(totalCount);
        } else {
            indicator.hide();
        }
    };

    // 7. تعديل لمنع فلترة المنتجات في عرض showProductDropdown
    if (window.originalShowProductDropdown && typeof window.showProductDropdownEnhanced === 'function') {
        window.showProductDropdownEnhanced = function(input, products, row, rowIndex) {
            // نستدعي الدالة الأصلية ولكن نمرر لها المنتجات بدون فلترة الوحدة
            var originalFunction = window.originalShowProductDropdown;
            originalFunction.call(window, input, products, row, rowIndex);
            
            // تهيئة النافذة لتطبيق الوحدة عند الاختيار
            setTimeout(function() {
                $('.product-dropdown tr').off('click.unitFilter').on('click.unitFilter', function() {
                    var productData = $(this).data('product-data');
                    if (productData && window.activeUnitFilter) {
                        setTimeout(function() {
                            applyUnitFilterToRow(row, window.activeUnitFilter);
                        }, 300);
                    }
                });
            }, 100);
        };
    }

    // 8. تعديل لإضافة معالج أحداث النقر على الصفوف في جدول البحث
    if (typeof window.selectProduct === 'function') {
        var originalSelectProduct = window.selectProduct;
        window.selectProduct = function(product, row, rowIndex) {
            // استدعاء الدالة الأصلية
            originalSelectProduct.call(window, product, row, rowIndex);
            
            // تطبيق فلتر الوحدة إذا كان نشطاً
            if (window.activeUnitFilter) {
                setTimeout(function() {
                    applyUnitFilterToRow(row, window.activeUnitFilter);
                }, 300);
            }
        };
    }

    // 9. تعديل دالة populateRowWithProduct لتطبيق الوحدة مباشرة
    if (typeof window.populateRowWithProduct === 'function') {
        var originalPopulateRowWithProduct = window.populateRowWithProduct;
        window.populateRowWithProduct = function(row, product, rowIndex) {
            // استدعاء الدالة الأصلية
            originalPopulateRowWithProduct.call(window, row, product, rowIndex);
            
            // تطبيق فلتر الوحدة إذا كان نشطاً
            if (window.activeUnitFilter) {
                setTimeout(function() {
                    applyUnitFilterToRow(row, window.activeUnitFilter);
                }, 100);
            }
        };
    }

    // 10. دوال مساعدة في حالة عدم وجودها
    if (typeof window.showSuccessMessage !== 'function') {
        window.showSuccessMessage = function(message, type) {
            if (typeof toastr !== 'undefined') {
                if (type === 'info') {
                    toastr.info(message);
                } else {
                    toastr.success(message);
                }
            } else {
                console.log('✅ ' + message);
            }
        };
    }

    if (typeof window.showWarningMessage !== 'function') {
        window.showWarningMessage = function(message) {
            if (typeof toastr !== 'undefined') {
                toastr.warning(message);
            } else {
                console.log('⚠️ ' + message);
            }
        };
    }

    if (typeof window.showErrorMessage !== 'function') {
        window.showErrorMessage = function(message) {
            if (typeof toastr !== 'undefined') {
                toastr.error(message);
            } else {
                console.log('❌ ' + message);
            }
        };
    }
}





/**
 * إعداد التحقق المباشر من حقول صف المنتجات أثناء الكتابة
 */
function setupLiveFieldValidation() {
    console.log('Setting up live field validation...');
    
    // إضافة أنماط CSS للتحقق المباشر
    addLiveValidationStyles();
    
    // إزالة معالجات الأحداث القديمة لتجنب التكرار
    $(document).off('input change', '.pos_quantity, .warehouse-input, .unit-input, .pos_unit_price, .discount_percent');
    
    // التحقق من الكمية أثناء الكتابة
    $(document).on('input change', '.pos_quantity', function() {
        validateQuantityField($(this));
    });
    
    // التحقق من حقل المستودع أثناء الكتابة
    $(document).on('input change blur', '.warehouse-input', function() {
        validateWarehouseField($(this));
    });
    
    // التحقق من حقل وحدة القياس أثناء الكتابة
    $(document).on('input change blur', '.unit-input', function() {
        validateUnitField($(this));
    });
    
    // التحقق من حقل السعر أثناء الكتابة
    $(document).on('input change', '.pos_unit_price, .pos_unit_price_inc_tax', function() {
        validatePriceField($(this));
    });
    
    // التحقق من حقل التخفيض أثناء الكتابة
    $(document).on('input change', '.discount_percent', function() {
        validateDiscountField($(this));
    });
    
    // التحقق الشامل عند تغيير أي حقل في الصف
    $(document).on('change', '#pos_table tbody tr td input, #pos_table tbody tr td select', function() {
        var row = $(this).closest('tr');
        validateRowCompleteness(row);
    });
    
    // تطبيق التحقق على الصفوف الموجودة حاليًا
    $('#pos_table tbody tr').each(function() {
        var row = $(this);
        
        // تحقق من حقول الصف الحالي
        row.find('.pos_quantity').each(function() {
            validateQuantityField($(this));
        });
        
        row.find('.warehouse-input').each(function() {
            validateWarehouseField($(this));
        });
        
        row.find('.unit-input').each(function() {
            validateUnitField($(this));
        });
        
        row.find('.pos_unit_price, .pos_unit_price_inc_tax').each(function() {
            validatePriceField($(this));
        });
        
        row.find('.discount_percent').each(function() {
            validateDiscountField($(this));
        });
        
        // تحقق من اكتمال الصف
        validateRowCompleteness(row);
    });
    
    console.log('Live field validation setup complete');
}

/**
 * إضافة أنماط CSS للتحقق المباشر
 */
function addLiveValidationStyles() {
    if ($('#liveValidationStyles').length === 0) {
        var styles = `
        <style id="liveValidationStyles">
        /* أنماط حقول الخطأ */
        .field-error {
            border: 2px solid #dc3545 !important;
            background-color: rgba(220, 53, 69, 0.05) !important;
            transition: all 0.3s ease;
        }
        
        /* أنماط حقول التحذير */
        .field-warning {
            border: 2px solid #ffc107 !important;
            background-color: rgba(255, 193, 7, 0.05) !important;
        }
        
        /* أنماط حقول صحيحة */
        .field-success {
            border: 2px solid #28a745 !important;
            background-color: rgba(40, 167, 69, 0.05) !important;
            transition: all 0.3s ease;
        }
        
        /* مؤشر الحالة */
        .field-status-indicator {
            position: absolute;
            top: 50%;
            right: 10px;
            transform: translateY(-50%);
            font-size: 14px;
        }
        
        .field-error-indicator {
            color: #dc3545;
        }
        
        .field-warning-indicator {
            color: #ffc107;
        }
        
        .field-success-indicator {
            color: #28a745;
        }
        
        /* رسائل الخطأ */
        .field-error-message {
            position: absolute;
            bottom: 100%;
            left: 0;
            width: 200px;
            background-color: #dc3545;
            color: white;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 12px;
            z-index: 1000;
            box-shadow: 0 3px 6px rgba(0,0,0,0.2);
            display: none;
            pointer-events: none;
        }
        
        .field-error-message::after {
            content: '';
            position: absolute;
            top: 100%;
            left: 10px;
            border-width: 5px;
            border-style: solid;
            border-color: #dc3545 transparent transparent transparent;
        }
        
        /* أنماط الصف المكتمل أو غير المكتمل */
        tr.row-incomplete {
            background-color: rgba(255, 193, 7, 0.05) !important;
        }
        
        tr.row-complete {
            background-color: rgba(40, 167, 69, 0.05) !important;
        }
        
        /* أنماط الحقول المعطلة */
        input[readonly], input:disabled {
            background-color: #f8f9fa !important;
            cursor: not-allowed;
            opacity: 0.7;
        }
        </style>
        `;
        
        $('head').append(styles);
    }
}

/**
 * التحقق من حقل الكمية
 * @param {jQuery} field - حقل الكمية
 */
function validateQuantityField(field) {
    if (!field || field.length === 0) return;
    
    var row = field.closest('tr');
    var value = field.val().trim();
    var quantity = parseFloat(value);
    
    // إزالة جميع التصنيفات والمؤشرات
    clearFieldStatus(field);
    
    // التحقق من أن القيمة رقم
    if (value === '') {
        // الحقل فارغ - لا شيء للتحقق منه بعد
        return;
    } else if (isNaN(quantity)) {
        // القيمة ليست رقمًا
        setFieldError(field, 'يرجى إدخال رقم صحيح');
        return;
    } else if (quantity <= 0) {
        // القيمة صفر أو سالبة
        setFieldError(field, 'يجب أن تكون الكمية أكبر من 0');
        return;
    }
    
    // التحقق من حدود الكمية (إذا كان التحقق من المخزون مفعل)
    if (row.find('.enable_stock').val() == '1' && !$('input#is_overselling_allowed').length) {
        var maxQuantity = parseFloat(field.data('qty_available')) || 0;
        var unitMultiplier = parseFloat(row.find('.base_unit_multiplier').val()) || 1;
        var adjustedMaxQty = maxQuantity / unitMultiplier;
        
        if (quantity > adjustedMaxQty) {
            // الكمية المطلوبة أكبر من المتاحة
           
            return;
        }
    }
    
    // الكمية صحيحة
    setFieldSuccess(field);
}

/**
 * التحقق من حقل المستودع
 * @param {jQuery} field - حقل المستودع
 */
function validateWarehouseField(field) {
    if (!field || field.length === 0) return;
    
    var value = field.val().trim();
    
    // إزالة جميع التصنيفات والمؤشرات
    clearFieldStatus(field);
    
    // المستودع اختياري في معظم الأنظمة، لذا فهو صحيح إذا كان فارغًا
    if (value === '0') {
        return;
    }
    
    // التحقق من تنسيق المستودع (مثال: W01)
    var warehousePattern = /^W\d{2}$/i;
    if (!warehousePattern.test(value)) {
        // تحويل التنسيق تلقائيًا إذا أمكن
        if (/^\d+$/.test(value)) {
            // إذا كان رقم فقط، أضف W في البداية
            var formattedValue = 'W' + value.padStart(1, '0');
            field.val(formattedValue);
            setFieldSuccess(field);
        } else {
            // تنسيق غير صحيح
           
        }
        return;
    }
    
    // التحقق من أن المستودع موجود (اختياري - يمكن تنفيذه إذا كان لديك قائمة بالمستودعات المتاحة)
    // اعتبره صحيح الآن
    setFieldSuccess(field);
}

/**
 * التحقق من حقل وحدة القياس
 * @param {jQuery} field - حقل وحدة القياس
 */
function validateUnitField(field) {
    if (!field || field.length === 0) return;
    
    var row = field.closest('tr');
    var value = field.val().trim();
    
    // إزالة جميع التصنيفات والمؤشرات
    clearFieldStatus(field);
    
    // وحدة القياس اختيارية في بعض الأنظمة، لذا فهي صحيحة إذا كانت فارغة
    if (value === '') {
        return;
    }
    
    // قائمة الوحدات الشائعة
    var commonUnits = ['EA', 'PCS', 'BOX', 'CTN', 'DZ', 'PACK', 'KG', 'GM', 'LTR', 'ML'];
    
    // التحقق من أن الوحدة موجودة في القائمة
    var unitExists = commonUnits.some(function(unit) {
        return unit.toUpperCase() === value.toUpperCase();
    });
    
    if (!unitExists) {
        // وحدة غير معروفة
        
        return;
    }
    
    // وحدة القياس صحيحة
    setFieldSuccess(field);
}

/**
 * التحقق من حقل السعر
 * @param {jQuery} field - حقل السعر
 */
function validatePriceField(field) {
    if (!field || field.length === 0) return;
    
    var value = field.val().trim();
    var price = parseFloat(value);
    
    // إزالة جميع التصنيفات والمؤشرات
    clearFieldStatus(field);
    
    // التحقق من أن القيمة رقم
    if (value === '') {
        // الحقل فارغ - لا شيء للتحقق منه بعد
        return;
    } else if (isNaN(price)) {
        // القيمة ليست رقمًا
        setFieldError(field, 'يرجى إدخال رقم صحيح');
        return;
    } else if (price < 0) {
        // السعر سالب
        setFieldError(field, 'يجب أن يكون السعر 0 أو أكبر');
        return;
    } else if (price === 0) {
        // السعر صفر - قد يكون صحيحًا ولكن يفضل التحذير
        setFieldWarning(field, 'السعر صفر');
        return;
    }
    
    // السعر صحيح
    setFieldSuccess(field);
}

/**
 * التحقق من حقل التخفيض
 * @param {jQuery} field - حقل التخفيض
 */
function validateDiscountField(field) {
    if (!field || field.length === 0) return;
    
    var value = field.val().trim();
    var discount = parseFloat(value);
    
    // إزالة جميع التصنيفات والمؤشرات
    clearFieldStatus(field);
    
    // التحقق من أن القيمة رقم
    if (value === '' || value === '0') {
        // الحقل فارغ أو صفر - لا شيء للتحقق منه بعد
        return;
    } else if (isNaN(discount)) {
        // القيمة ليست رقمًا
        setFieldError(field, 'يرجى إدخال رقم صحيح');
        return;
    } else if (discount < 0) {
        // التخفيض سالب
        setFieldError(field, 'يجب أن يكون التخفيض 0 أو أكبر');
        return;
    } else if (discount > 100) {
        // التخفيض أكبر من 100%
        setFieldError(field, 'يجب أن يكون التخفيض 100% أو أقل');
        return;
    }
    
    // التخفيض صحيح
    setFieldSuccess(field);
}

/**
 * التحقق من اكتمال الصف
 * @param {jQuery} row - صف المنتج
 */
function validateRowCompleteness(row) {
    if (!row || row.length === 0) return;
    
    // التحقق فقط من الصفوف غير الفارغة
    if (row.hasClass('empty-row')) {
        return;
    }
    
    var isComplete = true;
    var errorCount = 0;
    
    // التحقق من وجود منتج
    var productId = row.find('.product_id').val() || row.find('.variation_id').val();
    var productName = row.find('.product_name').val() || row.find('.product-search-input').val() || 
                     row.find('td:eq(1)').text().trim();
    
    if (!productId && !productName) {
        isComplete = false;
    }
    
    // عد حقول الخطأ
    errorCount += row.find('.field-error').length;
    
    // تحديد حالة الصف
    if (errorCount > 0) {
        row.removeClass('row-complete').addClass('row-incomplete');
    } else if (!isComplete) {
        row.removeClass('row-complete').addClass('row-incomplete');
    } else {
        row.removeClass('row-incomplete').addClass('row-complete');
    }
}

/**
 * إزالة جميع تصنيفات ومؤشرات الحالة من الحقل
 * @param {jQuery} field - الحقل المراد تنظيفه
 */
function clearFieldStatus(field) {
    field.removeClass('field-error field-warning field-success');
    field.siblings('.field-status-indicator, .field-error-message').remove();
}

/**
 * تعيين حالة خطأ للحقل
 * @param {jQuery} field - الحقل المراد تعيين حالة خطأ له
 * @param {string} message - رسالة الخطأ
 */
function setFieldError(field, message) {
    clearFieldStatus(field);
    field.addClass('field-error');
    
    // إضافة مؤشر الخطأ
    field.after('<span class="field-status-indicator field-error-indicator"><i class="fa fa-exclamation-circle"></i></span>');
    
    // إضافة رسالة الخطأ
    if (message) {
        var errorMessage = $('<div class="field-error-message">' + message + '</div>');
        field.before(errorMessage);
        
        // عرض رسالة الخطأ عند تحويم الماوس
        field.hover(function() {
            errorMessage.fadeIn(200);
        }, function() {
            errorMessage.fadeOut(200);
        });
    }
}

/**
 * تعيين حالة تحذير للحقل
 * @param {jQuery} field - الحقل المراد تعيين حالة تحذير له
 * @param {string} message - رسالة التحذير
 */
function setFieldWarning(field, message) {
    clearFieldStatus(field);
   
    
    // إضافة مؤشر التحذير
    field.after('<span class="field-status-indicator field-warning-indicator"><i class="fa fa-exclamation-triangle"></i></span>');
    
    // إضافة رسالة التحذير
    if (message) {
        var warningMessage = $('<div class="field-error-message" style="background-color: #ffc107; color: #000;">' + message + '</div>');
        field.before(warningMessage);
        
        // عرض رسالة التحذير عند تحويم الماوس
        field.hover(function() {
            warningMessage.fadeIn(200);
        }, function() {
            warningMessage.fadeOut(200);
        });
    }
}

/**
 * تعيين حالة نجاح للحقل
 * @param {jQuery} field - الحقل المراد تعيين حالة نجاح له
 */
function setFieldSuccess(field) {
    clearFieldStatus(field);
    field.addClass('field-success');
    
    // إضافة مؤشر النجاح
    field.after('<span class="field-status-indicator field-success-indicator"><i class="fa fa-check-circle"></i></span>');
}

/**
 * التحقق من أن القيمة هي رقم صحيح
 * @param {mixed} value - القيمة المراد التحقق منها
 * @returns {boolean} - نتيجة التحقق
 */
function isValidNumber(value) {
    // إزالة الفواصل وأي أحرف غير رقمية (باستثناء النقطة العشرية والعلامة السالبة)
    if (typeof value === 'string') {
        value = value.replace(/[^\d.-]/g, '');
    }
    
    // محاولة تحويل القيمة إلى رقم
    var number = parseFloat(value);
    
    // التحقق من أن القيمة رقم وليست NaN
    return !isNaN(number) && isFinite(number);
}

// تفعيل التحقق المباشر عند تحميل الصفحة
$(document).ready(function() {
    // انتظار قليلاً لضمان تحميل جميع المكونات
    setTimeout(function() {
        setupLiveFieldValidation();
        
        // إعادة تطبيق التحقق عند إضافة صف جديد
        $(document).on('row-added', function(e, newRow) {
            if (newRow && newRow.length > 0) {
                setTimeout(function() {
                    validateRowCompleteness(newRow);
                }, 100);
            }
        });
    }, 1000);
});






/**
 * تعطيل كل تفاعلات حقل سعر الدينار العراقي (IQD Price)
 * لجعله غير قابل للنقر أو التفاعل تمامًا
 */
function disableIQDPriceFieldInteractions() {
    console.log('Disabling all IQD Price field interactions...');
    
    // إضافة الأنماط اللازمة
    addDisabledIQDStyles();
    
    // إزالة جميع معالجات الأحداث المرتبطة بحقل IQD
    $(document).off('click mousedown focus mouseenter mouseleave change input keydown keyup', '.iqd-price-display');
    
    // منع جميع التفاعلات مع حقل IQD
    $(document).on('click mousedown focus mouseenter mouseleave change input keydown keyup', '.iqd-price-display', function(e) {
        // إيقاف جميع الأحداث تمامًا
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        return false;
    });
    
    // تحديث جميع حقول IQD الموجودة
    $('.iqd-price-display').each(function() {
        makeFieldCompletelyDisabled($(this));
    });
    
    // إضافة معالج لتعطيل أي حقول IQD جديدة تضاف لاحقًا
    $(document).on('DOMNodeInserted', function(e) {
        if ($(e.target).hasClass('iqd-price-display') || $(e.target).find('.iqd-price-display').length > 0) {
            setTimeout(function() {
                $(e.target).find('.iqd-price-display').each(function() {
                    makeFieldCompletelyDisabled($(this));
                });
            }, 100);
        }
    });
    
    console.log('IQD Price field completely disabled');
}

/**
 * إضافة أنماط CSS لتعطيل حقل IQD تمامًا
 */
function addDisabledIQDStyles() {
    if ($('#disabledIQDStyles').length === 0) {
        var styles = `
        <style id="disabledIQDStyles">
        /* أنماط تعطيل حقل سعر الدينار العراقي تمامًا */
        .iqd-price-display {
            background-color: #f1f3f5 !important;
            color: #6c757d !important;
            border: 1px solid #dee2e6 !important;
            cursor: default !important;
            pointer-events: none !important;
            user-select: none !important;
            -webkit-user-select: none !important;
            -moz-user-select: none !important;
            -ms-user-select: none !important;
            opacity: 0.8 !important;
            box-shadow: none !important;
        }
        
        /* إزالة أي مؤشرات تفاعل */
        .iqd-price-display:hover,
        .iqd-price-display:focus,
        .iqd-price-display:active {
            outline: none !important;
            border-color: #dee2e6 !important;
            box-shadow: none !important;
            background-color: #f1f3f5 !important;
        }
        
        /* تمييز بصري بسيط ليظهر أنه للعرض فقط */
        .iqd-price-display::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: repeating-linear-gradient(
                45deg,
                rgba(0, 0, 0, 0.03),
                rgba(0, 0, 0, 0.03) 10px,
                rgba(0, 0, 0, 0.06) 10px,
                rgba(0, 0, 0, 0.06) 20px
            );
            pointer-events: none;
            opacity: 0.3;
            z-index: 1;
        }
        
        /* تعطيل التحديد */
        .iqd-price-display::selection {
            background: transparent !important;
        }
        
        .iqd-price-display::-moz-selection {
            background: transparent !important;
        }
        </style>
        `;
        
        $('head').append(styles);
    }
}

/**
 * جعل الحقل معطل تمامًا بجميع الطرق الممكنة
 * @param {jQuery} field - الحقل المراد تعطيله
 */
function makeFieldCompletelyDisabled(field) {
    if (!field || field.length === 0) return;
    
    // تعطيل الحقل باستخدام خاصية disabled
    field.prop('disabled', true);
    
    // تعطيل الحقل باستخدام خاصية readonly
    field.prop('readonly', true);
    
    // تعطيل الحقل باستخدام خاصية tabindex سالبة
    field.attr('tabindex', '-1');
    
    // إضافة السمة aria-disabled
    field.attr('aria-disabled', 'true');
    
    // إضافة السمة data-disabled
    field.attr('data-disabled', 'true');
    
    // تعطيل التفاعل عبر CSS
    field.css({
        'pointer-events': 'none',
        'user-select': 'none',
        'cursor': 'default'
    });
    
    // إزالة أي معالجات أحداث مرتبطة بالحقل
    field.off();
}

// تعطيل التفاعل مع حقل IQD عند تحميل الصفحة
$(document).ready(function() {
    // انتظار قليلاً لضمان تحميل جميع المكونات
    setTimeout(function() {
        disableIQDPriceFieldInteractions();
        
        // تعطيل الحقول الموجودة حاليًا
        $('.iqd-price-display').each(function() {
            makeFieldCompletelyDisabled($(this));
        });
        
        // معالجة المشكلة في دالة التنقل بلوحة المفاتيح
        fixKeyboardNavigationForIQD();
    }, 1000);
});

/**
 * إصلاح التنقل بلوحة المفاتيح لتجاوز حقل IQD
 */
function fixKeyboardNavigationForIQD() {
    // استبدال أي دالة تنقل موجودة للتأكد من تجاهل حقل IQD
    if (typeof enhancedHandleKeyboardNavigation === 'function') {
        // حفظ الدالة الأصلية
        var originalKeyboardNav = enhancedHandleKeyboardNavigation;
        
        // استبدالها بدالة محسنة
        window.enhancedHandleKeyboardNavigation = function(e) {
            var current = $(this);
            
            // التحقق من أن الحقل التالي ليس IQD
            if (isNextFieldIQD(current) && (e.key === 'Tab' && !e.shiftKey || e.key === 'ArrowRight')) {
                e.preventDefault();
                skipIQDAndNavigate(current, 'forward');
                return;
            }
            
            // التحقق من أن الحقل السابق ليس IQD
            if (isPreviousFieldIQD(current) && (e.key === 'Tab' && e.shiftKey || e.key === 'ArrowLeft')) {
                e.preventDefault();
                skipIQDAndNavigate(current, 'backward');
                return;
            }
            
            // استدعاء الدالة الأصلية
            originalKeyboardNav.call(this, e);
        };
    }
    
    // تحديث معالجات لوحة المفاتيح للتنقل
    $(document).off('keydown', 'table#pos_table input:not(.iqd-price-display), table#pos_table select');
    $(document).on('keydown', 'table#pos_table input:not(.iqd-price-display), table#pos_table select', window.enhancedHandleKeyboardNavigation || function(e) {
        // دالة بديلة بسيطة إذا لم تكن الدالة المحسنة موجودة
        if ((e.key === 'Tab' || e.key === 'ArrowRight' || e.key === 'ArrowLeft') && 
            (isNextFieldIQD($(this)) || isPreviousFieldIQD($(this)))) {
            
            e.preventDefault();
            var direction = (e.key === 'Tab' && !e.shiftKey) || e.key === 'ArrowRight' ? 'forward' : 'backward';
            skipIQDAndNavigate($(this), direction);
        }
    });
}

/**
 * التحقق مما إذا كان الحقل التالي هو IQD
 * @param {jQuery} current - الحقل الحالي
 * @returns {boolean} - نتيجة التحقق
 */
function isNextFieldIQD(current) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible, select:visible');
    var currentIndex = editableInputs.index(current);
    var nextInput = editableInputs.eq(currentIndex + 1);
    
    return nextInput.length > 0 && nextInput.hasClass('iqd-price-display');
}

/**
 * التحقق مما إذا كان الحقل السابق هو IQD
 * @param {jQuery} current - الحقل الحالي
 * @returns {boolean} - نتيجة التحقق
 */
function isPreviousFieldIQD(current) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible, select:visible');
    var currentIndex = editableInputs.index(current);
    var previousInput = editableInputs.eq(currentIndex - 1);
    
    return previousInput.length > 0 && previousInput.hasClass('iqd-price-display');
}

/**
 * تجاوز حقل IQD والانتقال للحقل المناسب
 * @param {jQuery} current - الحقل الحالي
 * @param {string} direction - اتجاه التنقل ('forward' أو 'backward')
 */
function skipIQDAndNavigate(current, direction) {
    var row = current.closest('tr');
    var editableInputs = row.find('input:visible:not(.iqd-price-display), select:visible');
    var allInputs = row.find('input:visible, select:visible');
    var currentIndex = allInputs.index(current);
    
    // البحث عن الحقل التالي أو السابق (غير IQD)
    var targetIndex;
    
    if (direction === 'forward') {
        // البحث للأمام
        for (var i = currentIndex + 1; i < allInputs.length; i++) {
            if (!$(allInputs[i]).hasClass('iqd-price-display') && 
                !$(allInputs[i]).prop('disabled') && 
                !$(allInputs[i]).prop('readonly')) {
                targetIndex = i;
                break;
            }
        }
    } else {
        // البحث للخلف
        for (var i = currentIndex - 1; i >= 0; i--) {
            if (!$(allInputs[i]).hasClass('iqd-price-display') && 
                !$(allInputs[i]).prop('disabled') && 
                !$(allInputs[i]).prop('readonly')) {
                targetIndex = i;
                break;
            }
        }
    }
    
    // الانتقال للحقل المستهدف
    if (targetIndex !== undefined) {
        var targetInput = $(allInputs[targetIndex]);
        targetInput.focus();
        
        if (targetInput.is('input[type="text"], input[type="number"]')) {
            targetInput.select();
        }
    } else if (direction === 'forward') {
        // إذا وصلنا لنهاية الصف، انتقل للصف التالي
        var nextRow = row.next('tr');
        if (nextRow.length) {
            var firstInput = nextRow.find('input:visible:not(.iqd-price-display), select:visible').first();
            if (firstInput.length) {
                firstInput.focus();
                if (firstInput.is('input[type="text"], input[type="number"]')) {
                    firstInput.select();
                }
            }
        }
    } else {
        // إذا وصلنا لبداية الصف، انتقل للصف السابق
        var prevRow = row.prev('tr');
        if (prevRow.length) {
            var lastInput = prevRow.find('input:visible:not(.iqd-price-display), select:visible').last();
            if (lastInput.length) {
                lastInput.focus();
                if (lastInput.is('input[type="text"], input[type="number"]')) {
                    lastInput.select();
                }
            }
        }
    }
}

/**
 * التحقق من نوع المفتاح المضغوط
 * @param {number} keyCode - كود المفتاح
 * @returns {boolean} - هل هو مفتاح تنقل
 */
function isNavigationKey(keyCode) {
    var navigationKeys = [
        9,   // Tab
        13,  // Enter
        27,  // Escape
        37,  // Arrow Left
        38,  // Arrow Up
        39,  // Arrow Right
        40,  // Arrow Down
        35,  // End
        36,  // Home
        33,  // Page Up
        34   // Page Down
    ];
    
    return navigationKeys.includes(keyCode);
}








function updateApplyAllFilters() {
    // حفظ النسخة الأصلية من الدالة
    if (typeof window.applyAllFiltersToProducts === 'function' && !window.originalApplyAllFiltersToProducts) {
        window.originalApplyAllFiltersToProducts = window.applyAllFiltersToProducts;
    }
    
    // تعديل الدالة
    window.applyAllFiltersToProducts = function(products) {
        // تأكد أن جميع المنتجات مصنفة قبل التطبيق
        var productsWithCategories = processProductsWithCategories(products);
        
        // استدعاء الدالة الأصلية إذا كانت موجودة
        if (window.originalApplyAllFiltersToProducts) {
            return window.originalApplyAllFiltersToProducts(productsWithCategories);
        }
        
        // تنفيذ الفلترة الأساسية إذا لم تكن الدالة الأصلية موجودة
        var filteredProducts = productsWithCategories;
        
        // 1. فلاتر العلامات التجارية
        if (window.activeFilters && window.activeFilters.length > 0) {
            filteredProducts = filterProductsByBrand(filteredProducts);
        }
        
        // 2. فلتر فل بلاستك
        if (window.fullPlasticFilterActive !== undefined) {
            filteredProducts = filterProductsByPlastic(filteredProducts);
        }
        
        // 3. فلتر إخفاء منتجات 0.25 كغم عند فلاتر معينة
        filteredProducts = filterProductsBy025KgUnit(filteredProducts);
        
        return filteredProducts;
    };
}

// ============================================
// تحديث دالة إنشاء صف المنتج في جدول البحث
// ============================================
function updateCreateProductRow() {
    if (typeof window.createProductRow === 'function' && !window.originalCreateProductRow) {
        window.originalCreateProductRow = window.createProductRow;
    }
    
    window.createProductRow = function(product, index, row, rowIndex) {
        // تأكد من تطبيق التصنيف التلقائي
        product = addCategoryToProduct(product);
        
        // استدعاء الدالة الأصلية إذا كانت موجودة
        if (window.originalCreateProductRow) {
            return window.originalCreateProductRow(product, index, row, rowIndex);
        }
        
        // تنفيذ إنشاء الصف إذا لم تكن الدالة الأصلية موجودة
        var tr = $('<tr>')
            .attr('data-index', index)
            .attr('tabindex', '0')
            .addClass('product-row');
        
        // الحصول على إعدادات الأعمدة
        var columnSettings;
        if (typeof getSearchTableColumnSettings === 'function') {
            columnSettings = getSearchTableColumnSettings();
        } else {
            // إعدادات افتراضية إذا لم تكن الدالة موجودة
            columnSettings = [
                { id: 'product_name', name: 'اسم المنتج', visible: true },
                { id: 'category', name: 'الفئة', visible: true },
                { id: 'sku', name: 'الكود', visible: true },
                { id: 'price_usd', name: 'السعر (دولار)', visible: true },
                { id: 'total_stock', name: 'المخزون', visible: true }
            ];
        }
        
        columnSettings.forEach(function(column) {
            if (column.visible) {
                var td = createTableCellWithCategory(product, column.id);
                tr.append(td);
            }
        });
        
        // حفظ بيانات المنتج
        tr.data('product-data', product);
        
        // معالج النقر
        tr.on('click', function() {
            if (typeof selectProduct === 'function') {
                selectProduct(product, row, rowIndex);
            }
        });
        
        return tr;
    };
}

// ============================================
// تحديث دالة populateRowWithProduct لضمان التصنيف
// ============================================
function updatePopulateRowWithProduct() {
    if (typeof window.populateRowWithProduct === 'function' && !window.originalPopulateRowWithProduct) {
        window.originalPopulateRowWithProduct = window.populateRowWithProduct;
    }
    
    window.populateRowWithProduct = function(row, product, rowIndex) {
        // تطبيق التصنيف التلقائي
        product = addCategoryToProduct(product);
        
        // استدعاء الدالة الأصلية
        if (window.originalPopulateRowWithProduct) {
            window.originalPopulateRowWithProduct(row, product, rowIndex);
        }
        
        // ضمان وجود بيانات الفئة في الحقول المخفية
        if (product.category) {
            // إضافة فئة المنتج إلى الصف للاستخدام لاحقاً
            row.find('.product_category').val(product.category);
            row.attr('data-category', product.category);
            
            // إضافة علامة مرئية للفئة (اختياري)
            if (!row.find('.row-category-indicator').length) {
                var colorInfo = categoryColors[product.category] || categoryColors['default'];
                var indicator = $('<span class="row-category-indicator">')
                    .css({
                        'position': 'absolute',
                        'top': '0',
                        'right': '0',
                        'width': '8px',
                        'height': '100%',
                        'background-color': colorInfo.bg,
                        'opacity': '0.5'
                    });
                row.css('position', 'relative').append(indicator);
            }
        }
    };
}

// ============================================
// تحديث دالة showProductDropdown مع ضمان التصنيف
// ============================================
function updateShowProductDropdown() {
    // تحديث النسخة العادية
    if (typeof window.showProductDropdown === 'function' && !window.originalShowProductDropdown) {
        window.originalShowProductDropdown = window.showProductDropdown;
        
        window.showProductDropdown = function(input, products, row, rowIndex) {
            // تطبيق التصنيف التلقائي على جميع المنتجات
            products = processProductsWithCategories(products);
            
            // استدعاء الدالة الأصلية
            window.originalShowProductDropdown(input, products, row, rowIndex);
        };
    }
    
    // تحديث النسخة المحسنة إذا كانت موجودة
    if (typeof window.showProductDropdownEnhanced === 'function' && !window.originalShowProductDropdownEnhanced) {
        window.originalShowProductDropdownEnhanced = window.showProductDropdownEnhanced;
        
        window.showProductDropdownEnhanced = function(input, products, row, rowIndex) {
            // تطبيق التصنيف التلقائي على جميع المنتجات
            products = processProductsWithCategories(products);
            
            // استدعاء الدالة الأصلية
            window.originalShowProductDropdownEnhanced(input, products, row, rowIndex);
            
            // تعديل إضافي - استبدال دالة إنشاء الخلية
            setTimeout(function() {
                // إذا كان هناك جدول بحث مفتوح، استبدل وظائف العرض
                $('.product-search-table').find('td[data-column="category"]').each(function() {
                    var cell = $(this);
                    var productData = cell.closest('tr').data('product-data');
                    
                    if (productData) {
                        // تأكد من تطبيق التصنيف
                        productData = addCategoryToProduct(productData);
                        
                        // تحديث المحتوى
                        if (productData.category) {
                            var colorInfo = categoryColors[productData.category] || categoryColors['default'];
                            
                            var categoryHtml = '<span class="category-badge" style="display: inline-block; padding: 4px 8px; ' +
                                'background-color: ' + colorInfo.bg + '; color: ' + colorInfo.text + '; ' +
                                'border-radius: 4px; font-size: 11px; font-weight: 500;">' + 
                                productData.category + '</span>';
                            
                            cell.html(categoryHtml);
                        }
                    }
                });
            }, 100);
        };
    }
}

// ============================================
// دالة مساعدة لتنسيق الأرقام
// ============================================
function formatNumber(number, decimals = 0) {
    if (typeof number !== 'number') {
        number = parseFloat(number || 0);
    }
    
    if (typeof Intl !== 'undefined' && Intl.NumberFormat) {
        return new Intl.NumberFormat('en-US', {
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        }).format(number);
    } else {
        // البديل إذا لم تكن NumberFormat متاحة
        return number.toFixed(decimals).replace(/\B(?=(\d{3})+(?!\d))/g, ",");
    }
}

// ============================================
// دالة التهيئة الرئيسية
// ============================================
function initializeAutoCategorySystem() {
    console.log('🚀 Initializing auto-category system...');
    
    // إضافة أنماط CSS للفئات
    addCategoryStyles();
    
    // تحديث الدوال
    updateCreateTableCell();
    updateApplyAllFilters();
    updateCreateProductRow();
    updatePopulateRowWithProduct();
    updateShowProductDropdown();
    enhanceSearchWithAutoCategories();
    
    // تصنيف المنتجات الموجودة في الصفحة الحالية
    categorizeExistingProducts();
    
    console.log('✅ Auto-category system initialized successfully');
}

// ============================================
// إضافة أنماط CSS للفئات
// ============================================
function addCategoryStyles() {
    if ($('#categoryStyles').length === 0) {
        var styles = `
        <style id="categoryStyles">
        /* أنماط بطاقات الفئات */
        .category-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 500;
            box-shadow: 0 1px 3px rgba(0,0,0,0.12);
            transition: all 0.2s ease;
        }
        
        .category-badge:hover {
            transform: translateY(-1px);
            box-shadow: 0 3px 6px rgba(0,0,0,0.15);
        }
        
        /* أنماط مؤشر الصف */
        .row-category-indicator {
            transition: opacity 0.3s ease;
        }
        
        tr:hover .row-category-indicator {
            opacity: 0.8 !important;
        }
        </style>
        `;
        
        $('head').append(styles);
    }
}

// ============================================
// تحديث دالة إنشاء خلايا الجدول
// ============================================
function updateCreateTableCell() {
    if (typeof window.createTableCell === 'function' && !window.originalCreateTableCell) {
        window.originalCreateTableCell = window.createTableCell;
    }
    
    window.createTableCell = createTableCellWithCategory;
}

// ============================================
// تصنيف المنتجات الموجودة في الصفحة الحالية
// ============================================
function categorizeExistingProducts() {
    // تصنيف المنتجات في جدول البحث الحالي
    $('.product-search-table tr.product-row').each(function() {
        var row = $(this);
        var productData = row.data('product-data');
        
        if (productData) {
            productData = addCategoryToProduct(productData);
            row.data('product-data', productData);
            
            // تحديث خلية الفئة
            var categoryCell = row.find('td[data-column="category"]');
            if (categoryCell.length && productData.category) {
                var colorInfo = categoryColors[productData.category] || categoryColors['default'];
                
                var categoryHtml = '<span class="category-badge" style="display: inline-block; padding: 4px 8px; ' +
                    'background-color: ' + colorInfo.bg + '; color: ' + colorInfo.text + '; ' +
                    'border-radius: 4px; font-size: 11px; font-weight: 500;">' + 
                    productData.category + '</span>';
                
                categoryCell.html(categoryHtml);
            }
        }
    });
    
    // تصنيف المنتجات في جدول POS الحالي
    $('#pos_table tbody tr').each(function() {
        var row = $(this);
        if (row.hasClass('empty-row')) return;
        
        var productId = row.find('.product_id').val();
        var productName = row.find('.product_name').val() || row.find('td:eq(1)').text().trim();
        var productSku = row.find('.product_sku').val() || '';
        
        if (productId && productName) {
            var category = getProductCategory(productSku, productName);
            
            if (category) {
                // حفظ الفئة في الصف
                row.find('.product_category').val(category);
                row.attr('data-category', category);
                
                // إضافة علامة مرئية للفئة (اختياري)
                if (!row.find('.row-category-indicator').length) {
                    var colorInfo = categoryColors[category] || categoryColors['default'];
                    var indicator = $('<span class="row-category-indicator">')
                        .css({
                            'position': 'absolute',
                            'top': '0',
                            'right': '0',
                            'width': '8px',
                            'height': '100%',
                            'background-color': colorInfo.bg,
                            'opacity': '0.5'
                        });
                    row.css('position', 'relative').append(indicator);
                }
            }
        }
    });
}

// ============================================
// تنفيذ النظام عند تحميل الصفحة
// ============================================
$(document).ready(function() {
    // انتظار لضمان تحميل جميع المكونات
    setTimeout(function() {
        initializeAutoCategorySystem();
    }, 2000);
    
    console.log('📋 Product auto-categorization system loaded');
});

// ============================================
// تصدير وظائف النظام للاستخدام المباشر
// ============================================
window.categorySystem = {
    initialize: initializeAutoCategorySystem,
    getCategory: getProductCategory,
    addCategoryToProduct: addCategoryToProduct,
    processProducts: processProductsWithCategories,
    categorizeExisting: categorizeExistingProducts,
    updateFilters: updateApplyAllFilters,
    
    // دالة تشخيص
    debug: function() {
        console.log('=== نظام التصنيف التلقائي - معلومات التشخيص ===');
        console.log('خريطة الفئات:', productCategoryMap);
        console.log('الدوال المتاحة:', Object.keys(window.categorySystem));
        console.log('تم تحديث دالة applyAllFilters:', !!window.originalApplyAllFiltersToProducts);
        console.log('تم تحديث دالة createTableCell:', !!window.originalCreateTableCell);
        console.log('تم تحديث دالة createProductRow:', !!window.originalCreateProductRow);
        console.log('تم تحديث دالة populateRowWithProduct:', !!window.originalPopulateRowWithProduct);
        console.log('تم تحديث دالة showProductDropdown:', !!window.originalShowProductDropdown);
        console.log('تم تحديث دالة showProductDropdownEnhanced:', !!window.originalShowProductDropdownEnhanced);
        
        // اختبار التصنيف
        var testCodes = ['ADF123', 'R456', 'G789', 'N1234', 'FL567', 'XYZ'];
        testCodes.forEach(function(code) {
            console.log('الكود:', code, '- الفئة:', getProductCategory(code));
        });
    }
};



/**
 * Updates all unit-related form fields to ensure proper submission of unit data
 * 
 * This function ensures that the correct unit information is stored in the form fields
 * for proper submission, handling both base units and sub-units correctly.
 * 
 * @param {jQuery} row - The jQuery object representing the table row
 * @param {Object} unit - The unit object containing unit details
 */
function updateUnitSubmissionData(row, selectedUnit) {
    var rowIndex = row.data('row_index') || row.index();
    
    // التأكد من أن جميع الحقول موجودة
    if (!row.find('input[name="products[' + rowIndex + '][unit_name]"]').length) {
        row.append('<input type="hidden" name="products[' + rowIndex + '][unit_name]" value="">');
    }
    
    if (!row.find('input[name="products[' + rowIndex + '][sub_unit_id]"]').length) {
        row.append('<input type="hidden" name="products[' + rowIndex + '][sub_unit_id]" value="">');
    }
    
    if (!row.find('input[name="products[' + rowIndex + '][unit_multiplier]"]').length) {
        row.append('<input type="hidden" name="products[' + rowIndex + '][unit_multiplier]" value="1">');
    }
    
    // تحديث قيم الوحدة
    row.find('input[name="products[' + rowIndex + '][unit_name]"]').val(selectedUnit.unit_name || selectedUnit.name || 'EA');
    row.find('input[name="products[' + rowIndex + '][sub_unit_id]"]').val(selectedUnit.id || '');
    row.find('input[name="products[' + rowIndex + '][unit_multiplier]"]').val(parseFloat(selectedUnit.multiplier || 1));
    
    console.log('Unit submission data updated:', {
        unit_name: selectedUnit.unit_name || selectedUnit.name,
        sub_unit_id: selectedUnit.id,
        multiplier: selectedUnit.multiplier
    });
}

/**
 * Helper function to ensure a field has the proper name attribute
 */
function ensureFieldHasName(row, selector, fieldName) {
    var field = row.find(selector);
    if (field.length > 0) {
        if (!field.attr('name')) {
            field.attr('name', fieldName);
        }
    } else {
        // If field doesn't exist, create it
        var value = '';
        if (selector === '.sub_unit_id') value = row.data('unit-info')?.id || '';
        if (selector === '.product_unit_id') value = row.find('.product_unit_id').val() || '';
        if (selector === '.unit_multiplier') value = row.data('unit-info')?.multiplier || 1;
        if (selector === '.base_unit_multiplier') value = row.data('unit-info')?.multiplier || 1;
        if (selector === '.is_base_unit') value = row.data('unit-info')?.is_base_unit || 0;
        
        row.append('<input type="hidden" class="' + selector.substring(1) + '" name="' + fieldName + '" value="' + value + '">');
    }
}



/**
 * Enhanced function to fix unit submission issues
 * This ensures the selected sub-unit is properly sent with the form
 */
function fixUnitSubmissionIssue() {
    console.log("🔧 Starting unit submission fix...");
    
    // Apply this fix right before form submission
    $('#pos-finalize, #pos-quotation, #pos-draft, .pos-express-finalize, #submit-sell, #save-and-print').on('click', function(e) {
        // Don't interfere with other event handlers
        if (window.unitFixApplied) return;
        
        // Flag to prevent double-application
        window.unitFixApplied = true;
        console.log("🔍 Applying unit submission fix...");
        
        // Process each product row
        $('#pos_table tbody tr.product_row').each(function() {
            var row = $(this);
            var rowIndex = row.data('row_index') || row.index();
            
            // Get the currently displayed unit name from the UI
            var selectedUnitName = row.find('.unit-input').val();
            console.log(`Row ${rowIndex}: Current unit name displayed: "${selectedUnitName}"`);
            
            // Check stored unit info in the row data
            var unitInfo = row.data('unit-info');
            console.log(`Row ${rowIndex}: Stored unit info:`, unitInfo);
            
            if (unitInfo && unitInfo.value === selectedUnitName) {
                // If stored unit matches displayed unit, use stored unit info
                var isBaseUnit = unitInfo.is_base_unit == 1;
                var unitId = unitInfo.id || '';
                var multiplier = unitInfo.multiplier || 1;
                
                console.log(`Row ${rowIndex}: Using stored unit info: ${selectedUnitName} (ID: ${unitId}, Base: ${isBaseUnit}, Mult: ${multiplier})`);
                
                // Update form fields for this row with correct unit info
                updateUnitFormFields(row, {
                    unit_name: selectedUnitName,
                    id: unitId,
                    is_base_unit: isBaseUnit,
                    multiplier: multiplier
                });
            } else {
                // If no stored info or mismatch, look through available units
                console.log(`Row ${rowIndex}: No stored unit info, searching for unit: "${selectedUnitName}"`);
                
                // Try to find unit info in the available units data
                var availableUnitsData = row.find('.unit-input').attr('data-available-units');
                
                if (availableUnitsData) {
                    try {
                        var availableUnits = JSON.parse(availableUnitsData);
                        var matchedUnit = availableUnits.find(unit => 
                            (unit.unit_name || unit.name) === selectedUnitName
                        );
                        
                        if (matchedUnit) {
                            console.log(`Row ${rowIndex}: Found matching unit in available units:`, matchedUnit);
                            
                            // Update form fields with matched unit info
                            updateUnitFormFields(row, {
                                unit_name: selectedUnitName,
                                id: matchedUnit.id || '',
                                is_base_unit: matchedUnit.is_base_unit == 1,
                                multiplier: parseFloat(matchedUnit.multiplier || 1)
                            });
                        } else {
                            console.warn(`Row ${rowIndex}: Could not find unit "${selectedUnitName}" in available units`);
                        }
                    } catch (e) {
                        console.error(`Row ${rowIndex}: Error parsing available units:`, e);
                    }
                }
            }
            
            // Final verification of form fields
            console.log(`Row ${rowIndex}: Final unit values:`, {
                unit_name: row.find(`input[name="products[${rowIndex}][unit_name]"]`).val(),
                product_unit_id: row.find(`input[name="products[${rowIndex}][product_unit_id]"]`).val(),
                sub_unit_id: row.find(`input[name="products[${rowIndex}][sub_unit_id]"]`).val(),
                base_unit_multiplier: row.find(`input[name="products[${rowIndex}][base_unit_multiplier]"]`).val(),
                unit_multiplier: row.find(`input[name="products[${rowIndex}][unit_multiplier]"]`).val(),
                is_base_unit: row.find(`input[name="products[${rowIndex}][is_base_unit]"]`).val()
            });
        });
        
        console.log("✅ Unit submission fix applied successfully!");
        // Reset flag for next submission
        setTimeout(() => { window.unitFixApplied = false; }, 100);
    });
    
    console.log("🔧 Unit submission fix initialized");
}

/**
 * Helper function to update unit-related form fields
 */
function updateUnitFormFields(row, unitData) {
    var rowIndex = row.data('row_index') || row.index();
    
    // Update visible unit name field
    row.find('.unit-input').val(unitData.unit_name);
    
    // Ensure unit name field exists and is set
    ensureFormField(row, rowIndex, 'unit_name', unitData.unit_name);
    
    // Update sub_unit_id field based on whether this is a base unit
    if (!unitData.is_base_unit) {
        // For sub-unit: set the sub-unit ID
        ensureFormField(row, rowIndex, 'sub_unit_id', unitData.id);
    } else {
        // For base unit: clear sub_unit_id
        ensureFormField(row, rowIndex, 'sub_unit_id', '');
    }
    
    // Set multiplier values
    ensureFormField(row, rowIndex, 'unit_multiplier', unitData.multiplier);
    ensureFormField(row, rowIndex, 'base_unit_multiplier', unitData.multiplier);
    
    // Set is_base_unit flag
    ensureFormField(row, rowIndex, 'is_base_unit', unitData.is_base_unit ? 1 : 0);
    
    // Store updated unit info in row data
    row.data('unit-info', {
        value: unitData.unit_name,
        id: unitData.id,
        multiplier: unitData.multiplier,
        is_base_unit: unitData.is_base_unit ? 1 : 0
    });
}

/**
 * Helper function to ensure a form field exists with the correct value
 */
function ensureFormField(row, rowIndex, fieldName, fieldValue) {
    var fullFieldName = `products[${rowIndex}][${fieldName}]`;
    var field = row.find(`input[name="${fullFieldName}"]`);
    
    if (field.length > 0) {
        // Field exists, update its value
        field.val(fieldValue);
    } else {
        // Field doesn't exist, create it
        row.append(`<input type="hidden" name="${fullFieldName}" value="${fieldValue}">`);
    }
}




/// ================================================
// نظام التسعير المتقدم - النسخة المحدثة V4.2
// ================================================






// ============================================
// إصلاح مشاكل الأداء في نظام اللصق من Excel
// ============================================

// 1. إضافة متغيرات للتحكم في الأداء
var processingQueue = [];
var isProcessingActive = false;
var maxConcurrentRequests = 2; // الحد الأقصى للطلبات المتزامنة
var activeRequests = 0;
var processingDelay = 500; // تأخير بين العمليات

// 2. إصلاح دالة معالجة الصف التالي
async function processNextRowOptimized() {
    // فحص حالة الإيقاف أولاً
    if (shouldStop) {
        completeProcessing();
        return;
    }
    
    if (isPaused) {
        showPasteStatus('المعالجة متوقفة مؤقتاً', 'info');
        return;
    }
    
    // فحص الفهرس
    if (currentProcessIndex >= processedData.length) {
        completeProcessing();
        return;
    }
    
    var item = processedData[currentProcessIndex];
    
    // تخطي الصفوف غير الصالحة أو المعالجة مسبقاً
    if (!item.isValid || item.processed) {
        currentProcessIndex++;
        // استدعاء فوري للصف التالي بدون تأخير
        processNextRowOptimized();
        return;
    }
    
    // التحقق من عدد الطلبات النشطة
    if (activeRequests >= maxConcurrentRequests) {
        // انتظار وإعادة المحاولة
        setTimeout(() => {
            processNextRowOptimized();
        }, 100);
        return;
    }
    
    // تحديث واجهة المستخدم
    updateProcessingUI(currentProcessIndex, 'processing');
    updateProgress();
    
    activeRequests++;
    
    try {
        // معالجة الصف مع تحديد timeout
        await processDataRowOptimized(item);
        
        // تحديث الحالة
        item.processed = true;
        item.status = 'success';
        updateProcessingUI(currentProcessIndex, 'completed', '✓ تمت الإضافة');
        
    } catch (error) {
        console.error('Error processing row:', error);
        item.processed = true;
        item.status = 'error';
        item.error = error.message || 'خطأ غير معروف';
        updateProcessingUI(currentProcessIndex, 'error', '✗ ' + item.error);
    } finally {
        activeRequests--;
    }
    
    // تحديث الإحصائيات
    updateSummaryStats();
    
    // الانتقال للصف التالي
    currentProcessIndex++;
    
    // تأخير محكوم قبل المعالجة التالية
    setTimeout(() => {
        processNextRowOptimized();
    }, processingDelay);
}

// 3. إصلاح دالة معالجة البيانات مع timeout
async function processDataRowOptimized(item) {
    return new Promise((resolve, reject) => {
        // تعيين timeout للطلب
        var timeoutId = setTimeout(() => {
            reject(new Error('انتهت مهلة الانتظار'));
        }, 10000); // 10 ثواني timeout
        
        // إضافة صف فارغ إذا لزم الأمر
        ensureEmptyRow();
        
        // الحصول على الصف الفارغ
        var emptyRow = $('#pos_table tbody tr.empty-row').first();
        
        if (emptyRow.length === 0) {
            addEmptyProductRow();
            emptyRow = $('#pos_table tbody tr.empty-row').first();
        }
        
        var rowIndex = emptyRow.data('row_index') || emptyRow.index();
        
        // البحث عن المنتج باستخدام SKU مع تحسين الطلب
        $.ajax({
            url: base_path + '/products/list',
            method: 'GET',
            dataType: 'json',
            timeout: 8000, // 8 ثواني timeout للطلب
            data: {
                term: item.sku,
                search_fields: ['sku', 'sub_sku'],
                not_for_selling: 0,
                limit: 5, // تقليل عدد النتائج
                with_sub_units: false, // تسريع الطلب
                include_unit_details: false
            },
            success: function(products) {
                clearTimeout(timeoutId);
                
                if (products && products.length > 0) {
                    var matchedProduct = null;
                    
                    // البحث عن تطابق دقيق للـ SKU
                    for (var i = 0; i < products.length; i++) {
                        var product = products[i];
                        if (product.sub_sku === item.sku || product.sku === item.sku) {
                            matchedProduct = product;
                            break;
                        }
                    }
                    
                    if (matchedProduct) {
                        // ملء الصف بالمنتج
                        populateRowWithProduct(emptyRow, matchedProduct, rowIndex);
                        
                        // تعيين الكمية فقط (بدون معالجة معقدة للوحدات)
                        setTimeout(function() {
                            emptyRow.find('.pos_quantity').val(item.quantity).trigger('change');
                            resolve();
                        }, 100);
                        
                    } else {
                        reject(new Error('المنتج غير موجود'));
                    }
                } else {
                    reject(new Error('المنتج غير موجود'));
                }
            },
            error: function(xhr, status, error) {
                clearTimeout(timeoutId);
                reject(new Error('خطأ في البحث: ' + error));
            }
        });
    });
}

// 4. إصلاح وتحسين معالجات الأحداث
function optimizeEventHandlers() {
    // إزالة جميع المعالجات القديمة
    $(document).off('change.excel-paste');
    $(document).off('blur.excel-paste');
    $(document).off('paste.excel-paste');
    
    // إضافة معالج واحد محسن
    var processingTimeout;
    
    $(document).on('change.excel-paste', '.unit-input', function() {
        var $input = $(this);
        var row = $input.closest('tr');
        
        // إلغاء المعالجة السابقة
        if (processingTimeout) {
            clearTimeout(processingTimeout);
        }
        
        // تأخير المعالجة لتجنب التكرار
        processingTimeout = setTimeout(function() {
            if (!$input.data('processing')) {
                $input.data('processing', true);
                
                try {
                    persistUnitValue(row);
                    // تطبيق التسعير بشكل محدود
                    if (typeof applyAdvancedPricingToRow === 'function') {
                        applyAdvancedPricingToRow(row);
                    }
                } catch (e) {
                    console.error('Error in unit processing:', e);
                } finally {
                    $input.data('processing', false);
                }
            }
        }, 300);
    });
}

// 5. إصلاح مراقب DOM
function optimizeDOMObserver() {
    if (window.posTableObserver) {
        window.posTableObserver.disconnect();
    }
    
    var tableBody = document.querySelector('#pos_table tbody');
    if (!tableBody) return;
    
    // مراقب محسن مع throttling
    var observerTimeout;
    
    window.posTableObserver = new MutationObserver(function(mutations) {
        if (observerTimeout) {
            clearTimeout(observerTimeout);
        }
        
        observerTimeout = setTimeout(function() {
            var hasNewRows = false;
            
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    hasNewRows = true;
                }
            });
            
            if (hasNewRows) {
                // معالجة محدودة للصفوف الجديدة
                $('#pos_table tbody tr:not(.processed)').each(function() {
                    var $row = $(this);
                    if ($row.find('.product_id').length > 0) {
                        $row.addClass('processed');
                        // معالجة بسيطة بدون تعقيد
                        setTimeout(function() {
                            if (typeof applyAdvancedPricingToRow === 'function') {
                                applyAdvancedPricingToRow($row);
                            }
                        }, 200);
                    }
                });
            }
        }, 500);
    });
    
    window.posTableObserver.observe(tableBody, {
        childList: true,
        subtree: false
    });
}

// 6. إصلاح الفحص الدوري
function optimizePeriodicChecks() {
    // إزالة الفحص القديم
    if (window.periodicCheckInterval) {
        clearInterval(window.periodicCheckInterval);
    }
    
    // فحص محسن كل 5 ثواني بدلاً من ثانيتين
    window.periodicCheckInterval = setInterval(function() {
        if (!isProcessing && !isPaused) {
            var unprocessedRows = $('#pos_table tbody tr:not(.processed)').length;
            if (unprocessedRows > 0 && unprocessedRows < 3) {
                // معالجة محدودة للصفوف المتبقية
                $('#pos_table tbody tr:not(.processed)').first().addClass('processed');
            }
        }
    }, 5000);
}

// 7. دالة تحسين الذاكرة
function optimizeMemoryUsage() {
    // تنظيف البيانات القديمة
    if (window.productDataStore) {
        var storeSize = Object.keys(window.productDataStore).length;
        if (storeSize > 100) {
            // الاحتفاظ بآخر 50 منتج فقط
            var keys = Object.keys(window.productDataStore);
            var keysToRemove = keys.slice(0, keys.length - 50);
            keysToRemove.forEach(key => {
                delete window.productDataStore[key];
            });
        }
    }
    
    // تنظيف معالجات الأحداث غير المستخدمة
    $('*').off('.unused-events');
}

// 8. دالة بدء المعالجة المحسنة
function startProcessingOptimized() {
    var validData = processedData.filter(item => item.isValid);
    
    if (validData.length === 0) {
        showPasteStatus('لا توجد بيانات صالحة للمعالجة', 'error');
        return;
    }
    
    // تحسين إعدادات المعالجة حسب حجم البيانات
    if (validData.length > 50) {
        maxConcurrentRequests = 1;
        processingDelay = 1000;
    } else if (validData.length > 20) {
        maxConcurrentRequests = 2;
        processingDelay = 500;
    } else {
        maxConcurrentRequests = 3;
        processingDelay = 300;
    }
    
    // تعيين حالة المعالجة
    isProcessingActive = true;
    isProcessing = true;
    isPaused = false;
    shouldStop = false;
    currentProcessIndex = 0;
    activeRequests = 0;
    
    // تحديث واجهة المستخدم
    $('#processPasteBtn').hide();
    $('#previewPasteBtn').hide();
    $('#excelDataInput').prop('readonly', true);
    $('#processControlButtons').css('display', 'flex');
    $('#pasteProgress').show();
    $('#processSummary').show();
    $('#processStatusIndicator').show().addClass('processing');
    
    // تحسين الذاكرة قبل البدء
    optimizeMemoryUsage();
    
    // بدء المعالجة المحسنة
    processNextRowOptimized();
}

// 9. دالة الإيقاف المحسنة
function stopProcessingOptimized() {
    shouldStop = true;
    isProcessing = false;
    isPaused = false;
    isProcessingActive = false;
    
    // إيقاف جميع الطلبات النشطة
    if (window.activeAjaxRequests) {
        window.activeAjaxRequests.forEach(request => {
            if (request.readyState !== 4) {
                request.abort();
            }
        });
    }
    
    // تنظيف المتغيرات
    activeRequests = 0;
    processingQueue = [];
    
    // تحديث واجهة المستخدم
    $('#processStatusIndicator').removeClass('processing paused').addClass('stopped');
    showPasteStatus('تم إيقاف المعالجة نهائياً', 'error');
    
    // تنظيف الذاكرة
    setTimeout(optimizeMemoryUsage, 1000);
}

// 10. دالة التهيئة المحسنة
function initializeOptimizedExcelPaste() {
    console.log('🚀 Initializing optimized Excel paste system...');
    
    // تطبيق التحسينات
    optimizeEventHandlers();
    optimizeDOMObserver();
    optimizePeriodicChecks();
    
    // إعداد متغيرات الأداء
    window.activeAjaxRequests = [];
    
    // اعتراض طلبات AJAX لتتبعها
    var originalAjax = $.ajax;
    $.ajax = function(options) {
        var jqXHR = originalAjax.apply(this, arguments);
        
        // تتبع الطلبات النشطة
        window.activeAjaxRequests.push(jqXHR);
        
        // تنظيف الطلبات المكتملة
        jqXHR.always(function() {
            var index = window.activeAjaxRequests.indexOf(jqXHR);
            if (index > -1) {
                window.activeAjaxRequests.splice(index, 1);
            }
        });
        
        return jqXHR;
    };
    
    console.log('✅ Optimized Excel paste system initialized');
}

// 11. دالة تنظيف شاملة
function cleanupExcelPasteSystem() {
    // إيقاف جميع العمليات
    shouldStop = true;
    isProcessing = false;
    isPaused = false;
    isProcessingActive = false;
    
    // إيقاف المراقبات
    if (window.posTableObserver) {
        window.posTableObserver.disconnect();
    }
    
    // إيقاف الفحص الدوري
    if (window.periodicCheckInterval) {
        clearInterval(window.periodicCheckInterval);
    }
    
    // إيقاف الطلبات النشطة
    if (window.activeAjaxRequests) {
        window.activeAjaxRequests.forEach(request => {
            if (request.readyState !== 4) {
                request.abort();
            }
        });
        window.activeAjaxRequests = [];
    }
    
    // تنظيف معالجات الأحداث
    $(document).off('.excel-paste');
    
    // تنظيف البيانات
    processedData = [];
    processingQueue = [];
    activeRequests = 0;
    
    // تنظيف الذاكرة
    optimizeMemoryUsage();
    
    console.log('🧹 Excel paste system cleaned up');
}

// 12. استبدال الدوال الأصلية
function replaceOriginalFunctions() {
    // استبدال الدوال الأصلية بالمحسنة
    if (typeof window.processNextRow !== 'undefined') {
        window.processNextRow = processNextRowOptimized;
    }
    
    if (typeof window.startProcessing !== 'undefined') {
        window.startProcessing = startProcessingOptimized;
    }
    
    if (typeof window.stopProcess !== 'undefined') {
        window.stopProcess = stopProcessingOptimized;
    }
    
    if (typeof window.processDataRow !== 'undefined') {
        window.processDataRow = processDataRowOptimized;
    }
}

// 13. إعداد مراقب استخدام الذاكرة
function setupMemoryMonitor() {
    if (performance.memory) {
        setInterval(function() {
            var memoryUsage = performance.memory.usedJSHeapSize / 1024 / 1024;
            
            if (memoryUsage > 100) { // إذا تجاوز 100 MB
                console.warn('⚠️ High memory usage detected:', memoryUsage.toFixed(2) + ' MB');
                
                // تنظيف الذاكرة
                optimizeMemoryUsage();
                
                // إيقاف المعالجة إذا كانت نشطة
                if (isProcessing && memoryUsage > 200) {
                    console.error('❌ Memory usage too high, stopping processing');
                    stopProcessingOptimized();
                }
            }
        }, 5000);
    }
}

// 14. دالة إعادة التعيين الكاملة
function resetExcelPasteSystem() {
    console.log('🔄 Resetting Excel paste system...');
    
    // تنظيف كامل
    cleanupExcelPasteSystem();
    
    // إعادة تهيئة
    setTimeout(function() {
        initializeOptimizedExcelPaste();
        replaceOriginalFunctions();
        setupMemoryMonitor();
        console.log('✅ Excel paste system reset complete');
    }, 1000);
}

// 15. تطبيق التحسينات عند التحميل
$(document).ready(function() {
    setTimeout(function() {
        try {
            initializeOptimizedExcelPaste();
            replaceOriginalFunctions();
            setupMemoryMonitor();
            
            // إضافة زر إعادة التعيين للطوارئ
            $('body').append(`
                <button id="resetExcelPasteBtn" style="
                    position: fixed;
                    bottom: 20px;
                    right: 20px;
                    z-index: 9999;
                    background: #dc3545;
                    color: white;
                    border: none;
                    padding: 10px;
                    border-radius: 5px;
                    display: none;
                " onclick="resetExcelPasteSystem()">
                    إعادة تعيين النظام
                </button>
            `);
            
            // إظهار زر الإعادة عند حدوث مشاكل
            window.addEventListener('error', function() {
                $('#resetExcelPasteBtn').show();
            });
            
        } catch (error) {
            console.error('❌ Failed to initialize optimized system:', error);
        }
    }, 2000);
});

// 16. تصدير الدوال للاستخدام الخارجي
window.optimizedExcelPaste = {
    start: startProcessingOptimized,
    stop: stopProcessingOptimized,
    reset: resetExcelPasteSystem,
    cleanup: cleanupExcelPasteSystem,
    monitor: setupMemoryMonitor
};
// ================================================
// نظام التسعير المتقدم - النسخة المحدثة V4.3
// ================================================
// ================================================
// نظام التسعير المتقدم - النسخة المحدثة V4.2
// ================================================
(function() {
    'use strict';
    
    console.log('🚀 =================================');
    console.log('🚀 ADVANCED PRICING SYSTEM V4.3');
    console.log('🚀 Updated with Dozen Unit Support');
    console.log('🚀 =================================');
    
    // ==========================================
    // الجزء 1: الإعدادات والمتغيرات العامة
    // ==========================================
    
    window.advancedPricingDebug = true;
    window.productDataStore = window.productDataStore || {};
    window.lastAjaxProductData = window.lastAjaxProductData || {};
    
    // جدول ربط الوحدات بأكوادها المحدثة
    window.UNIT_MAPPING = {
        'Manual': 1,
        'قطعة': 2,
        'كارتون 120 ق': 3,
        'لتر': 4,
        'غم 250': 5,      // 250 غرام
        // ID 6 محجوز لـ كغم (تم حذفه لتجنب الخلط)
        '50غم': 7,
        '100غم': 8,
        '125غم': 9,
        '0.5 كيلو': 10,
        'كغم 0.25': 11,  // ✅ المعرف الصحيح لوحدة 0.25 كغم
        'دبة 5 لتر': 12,
        'ك غم 25': 13,
        'كغم 100': 14,
        'كغم 500': 15,
        'كغم 1000': 16,
        'درزن': 17,
        'كارتون': 18,
        'كارتون$10': 19,
        'باكيت': 20,
        'سيت 50ق': 21,
        'برميل 200 لتر': 22,
        'دبة 20 لتر': 23,
        'برميل 30 لتر': 24,
        'سيت': 21,
        'كغم 250': 26,
        'ربطة': 27,
        'لك': 28,  // إضافة وحدة لك
        'فل بلاستك': 29  // إضافة وحدة فل بلاستك
    };
    
    // دالة التسجيل
    window.logPricing = function(message, data, type = 'info') {
        if (!window.advancedPricingDebug) return;
        
        var timestamp = new Date().toLocaleTimeString();
        var prefix = '💰 [APricing ' + timestamp + '] ';
        
        console.log(prefix + message, data || '');
        
        if (type === 'error') {
            console.trace();
        }
    };
    
    // ==========================================
    // الجزء 2: إصلاح دوال الوحدات
    // ==========================================
    
    window.persistUnitValue = function(row) {
        if (!row || row.length === 0) return;
        
        try {
            var unitInput = row.find('.unit-input');
            var unitName = unitInput.val() || '';
            var subUnitId = row.find('.sub_unit_id').val() || '';
            var multiplier = row.find('.unit_multiplier').val() || '1';
            
            var unitData = {
                unit_name: unitName,
                sub_unit_id: subUnitId,
                multiplier: multiplier
            };
            
            row.data('persisted-unit', JSON.stringify(unitData));
            
            var rowIndex = row.data('row_index') || row.index();
            
            // تحديث أسماء الحقول
            row.find('.sub_unit_id').attr('name', 'products[' + rowIndex + '][sub_unit_id]');
            row.find('.unit_multiplier').attr('name', 'products[' + rowIndex + '][unit_multiplier]');
            
            // حقل اسم الوحدة
            var unitNameFieldName = 'products[' + rowIndex + '][unit_name]';
            var unitNameField = row.find('input[name="' + unitNameFieldName + '"]');
            
            if (unitNameField.length === 0) {
                $('<input>').attr({
                    type: 'hidden',
                    name: unitNameFieldName,
                    value: unitName
                }).appendTo(row);
            } else {
                unitNameField.val(unitName);
            }
            
            logPricing('Unit persisted:', unitData);
            
        } catch (e) {
            logPricing('Error in persistUnitValue:', e, 'error');
        }
    };
    
    window.updateUnitSubmissionData = function(row, unitDetails) {
        if (!row || !unitDetails) return;
        
        try {
            var rowIndex = row.data('row_index') || row.index();
            
            // تحديث الحقول المخفية
            if (unitDetails.id !== undefined) {
                row.find('.sub_unit_id').val(unitDetails.id);
            }
            if (unitDetails.sub_unit_id !== undefined) {
                row.find('.sub_unit_id').val(unitDetails.sub_unit_id);
            }
            if (unitDetails.multiplier !== undefined) {
                row.find('.unit_multiplier').val(unitDetails.multiplier);
            }
            
            // أسماء الحقول
            row.find('.sub_unit_id').attr('name', 'products[' + rowIndex + '][sub_unit_id]');
            row.find('.unit_multiplier').attr('name', 'products[' + rowIndex + '][unit_multiplier]');
            
            // اسم الوحدة
            var unitName = unitDetails.unit_name || unitDetails.name || 'EA';
            var unitNameFieldName = 'products[' + rowIndex + '][unit_name]';
            var unitNameField = row.find('input[name="' + unitNameFieldName + '"]');
            
            if (unitNameField.length === 0) {
                $('<input>').attr({
                    type: 'hidden',
                    name: unitNameFieldName,
                    value: unitName
                }).appendTo(row);
            } else {
                unitNameField.val(unitName);
            }
            
            logPricing('Unit submission data updated:', unitDetails);
            
        } catch (e) {
            logPricing('Error in updateUnitSubmissionData:', e, 'error');
        }
    };
    
    // ==========================================
    // الجزء 3: اعتراض AJAX وحفظ البيانات
    // ==========================================
    
    (function interceptAjax() {
        var originalAjax = $.ajax;
        
        $.ajax = function(options) {
            var originalSuccess = options.success;
            
            options.success = function(data, textStatus, jqXHR) {
                // التحقق من طلبات المنتجات
                if (options.url && (
                    options.url.includes('/products/list') ||
                    options.url.includes('/get_product_row') ||
                    options.url.includes('/sells/pos/get_product_row')
                )) {
                    logPricing('AJAX Product Response intercepted');
                    processProductAjaxResponse(data, options);
                }
                
                if (originalSuccess) {
                    return originalSuccess.apply(this, arguments);
                }
            };
            
            return originalAjax.call(this, options);
        };
    })();
    
    function processProductAjaxResponse(data, ajaxOptions) {
        try {
            if (Array.isArray(data)) {
                data.forEach(function(product) {
                    storeProductData(product);
                });
            } else if (data && typeof data === 'object') {
                if (data.html_content && data.product) {
                    storeProductData(data.product);
                } else if (data.product) {
                    storeProductData(data.product);
                } else if (data.variation_id || data.id) {
                    storeProductData(data);
                }
            }
        } catch (e) {
            logPricing('Error processing AJAX response:', e, 'error');
        }
    }
    
    function storeProductData(product) {
        if (!product) return;
        
        var id = product.variation_id || product.id;
        if (!id) return;
        
        // حفظ البيانات
        window.productDataStore[id] = product;
        window.lastAjaxProductData[id] = product;
        
        // حفظ باستخدام SKU
        var sku = product.sub_sku || product.sku;
        if (sku) {
            window.productDataStore['sku_' + sku] = product;
        }
        
        logPricing('Product stored:', {
            id: id,
            sku: sku,
            name: product.name,
            custom_field3: product.product_custom_field3
        });
    }
    
    // ==========================================
    // الجزء 4: دالة الحصول على بيانات المنتج
    // ==========================================
    
    window.getProductDataForRow = function(row) {
        logPricing('Getting product data for row...');
        
        if (!row || row.length === 0) return null;
        
        // 1. من البيانات المحفوظة في الصف
        var savedData = row.data('product-data');
        if (savedData && savedData.product_custom_field3) {
            logPricing('Found data in row');
            return savedData;
        }
        
        // 2. البحث بالمعرفات
        var variationId = row.find('.variation_id, .row_variation_id, input[name*="[variation_id]"]').val();
        var productId = row.find('.product_id, input[name*="[product_id]"]').val();
        
        var productData = null;
        
        if (variationId) {
            productData = window.productDataStore[variationId] || window.lastAjaxProductData[variationId];
        }
        
        if (!productData && productId) {
            productData = window.productDataStore[productId] || window.lastAjaxProductData[productId];
        }
        
        // 3. البحث بـ SKU
        if (!productData) {
            var sku = extractSKUFromRow(row);
            if (sku) {
                productData = window.productDataStore['sku_' + sku];
            }
        }
        
        // 4. بناء البيانات من الصف
        if (!productData) {
            productData = buildProductDataFromRow(row);
        }
        
        if (productData) {
            row.data('product-data', productData);
            logPricing('Product data found');
        } else {
            logPricing('No product data found', null, 'warn');
        }
        
        return productData;
    };
    
    function extractSKUFromRow(row) {
        var sku = row.find('.product_sku, input[name*="[sku]"]').val();
        
        if (!sku) {
            var labelText = row.find('.label:contains(":")').text();
            if (labelText) {
                sku = labelText.split(':')[1];
            }
        }
        
        if (!sku) {
            var productInfo = row.find('.product-info').text();
            var skuMatch = productInfo.match(/\b[A-Z]\d+\b/);
            if (skuMatch) {
                sku = skuMatch[0];
            }
        }
        
        return sku ? sku.trim() : null;
    }
    
    function buildProductDataFromRow(row) {
        var productData = {
            variation_id: row.find('.variation_id, .row_variation_id').val(),
            product_id: row.find('.product_id').val(),
            name: row.find('.product-info strong').text() || row.find('.product_name').text()
        };
        
        productData.sub_sku = extractSKUFromRow(row);
        
        // استخراج custom_field3
        var customField3 = row.find('.product_custom_field3, input[name*="[product_custom_field3]"]').val();
        if (!customField3) {
            var productText = row.find('.product-info').text();
            var match = productText.match(/\|([^|]+)$/);
            if (match) {
                customField3 = match[1].trim();
            }
        }
        
        productData.product_custom_field3 = customField3;
        
        if (productData.variation_id || productData.product_id) {
            return productData;
        }
        
        return null;
    }
    
    // ==========================================
    // الجزء 5: نظام استخراج الأسعار المحسن
    // ==========================================
    
    window.extractPricesFromCustomField3 = function(customField3Data) {
        logPricing('=== بدء استخراج الأسعار من custom_field3 ===');
        
        if (!customField3Data) {
            logPricing('❌ لا توجد بيانات في custom_field3', null, 'warn');
            return null;
        }
        
        var priceData = {
            mainPrice: null,
            unitPrices: {}
        };
        
        try {
            var dataArray = [];
            
            // تحويل البيانات إلى مصفوفة
            if (typeof customField3Data === 'string') {
                try {
                    // محاولة إصلاح JSON غير المكتمل
                    var fixedJson = customField3Data.trim();
                    
                    // فحص وإصلاح JSON
                    if (!fixedJson.endsWith(']') && !fixedJson.endsWith('}')) {
                        logPricing('⚠️ JSON غير مكتمل، محاولة الإصلاح...');
                        
                        // عد الأقواس
                        var openBrackets = (fixedJson.match(/\[/g) || []).length;
                        var closeBrackets = (fixedJson.match(/\]/g) || []).length;
                        var openBraces = (fixedJson.match(/\{/g) || []).length;
                        var closeBraces = (fixedJson.match(/\}/g) || []).length;
                        
                        // إضافة الأقواس الناقصة
                        var missingBraces = openBraces - closeBraces;
                        var missingBrackets = openBrackets - closeBrackets;
                        
                        // إزالة النص غير المكتمل في النهاية
                        var lastComma = fixedJson.lastIndexOf(',');
                        var lastBrace = fixedJson.lastIndexOf('{');
                        var lastBracket = fixedJson.lastIndexOf('[');
                        
                        if (lastComma > Math.max(lastBrace, lastBracket)) {
                            fixedJson = fixedJson.substring(0, lastComma);
                        }
                        
                        // إضافة الأقواس المفقودة
                        for (var i = 0; i < missingBraces; i++) {
                            fixedJson += '}';
                        }
                        for (var i = 0; i < missingBrackets; i++) {
                            fixedJson += ']';
                        }
                    }
                    
                    dataArray = JSON.parse(fixedJson);
                    logPricing('✅ تم تحليل JSON بنجاح');
                    
                } catch (parseError) {
                    logPricing('❌ فشل تحليل JSON: ' + parseError.message);
                    return null;
                }
            } else if (Array.isArray(customField3Data)) {
                dataArray = customField3Data;
            } else if (typeof customField3Data === 'object') {
                dataArray = [customField3Data];
            }
            
            // معالجة المصفوفة لاستخراج الأسعار
            if (Array.isArray(dataArray) && dataArray.length > 0) {
                // البحث عن قائمة الأسعار الأولى (PriceList: 1)
                var priceList1 = dataArray.find(function(item) {
                    return item && item.PriceList === 1;
                });
                
                if (priceList1) {
                    // استخراج السعر الأساسي
                    if (priceList1.Price && parseFloat(priceList1.Price) > 0) {
                        priceData.mainPrice = parseFloat(priceList1.Price);
                        logPricing('✓ السعر الأساسي: ' + priceData.mainPrice);
                    }
                    
                    // استخراج أسعار الوحدات الفرعية
                    if (priceList1.UoMPrices && Array.isArray(priceList1.UoMPrices)) {
                        priceList1.UoMPrices.forEach(function(uomItem) {
                            if (uomItem && uomItem.UoMEntry && uomItem.Price && parseFloat(uomItem.Price) > 0) {
                                var uomEntry = parseInt(uomItem.UoMEntry);
                                var price = parseFloat(uomItem.Price);
                                priceData.unitPrices[uomEntry] = price;
                                logPricing('✓ سعر الوحدة ' + uomEntry + ': ' + price);
                            }
                        });
                    }
                }
            }
            
        } catch (e) {
            logPricing('❌ خطأ في استخراج الأسعار: ' + e.message, e, 'error');
            return null;
        }
        
        return priceData;
    };
    
    // ==========================================
    // الجزء 6: دوال التحقق من الوحدات
    // ==========================================
    
    // دالة للتحقق من وحدة الدرزن
    window.isDozenUnit = function(unitName) {
        if (!unitName) return false;
        
        var dozenVariations = ['درزن', 'دزن', 'dozen', 'dz'];
        var normalizedUnit = unitName.toLowerCase().trim();
        
        for (var i = 0; i < dozenVariations.length; i++) {
            if (normalizedUnit === dozenVariations[i].toLowerCase()) {
                return true;
            }
        }
        
        return false;
    };
    
    // دالة للتحقق من وحدة لك
    window.isLakUnit = function(unitName) {
        if (!unitName) return false;
        
        var lakVariations = ['لك', 'lak', 'LAK', 'Lak'];
        var normalizedUnit = unitName.toLowerCase().trim();
        
        for (var i = 0; i < lakVariations.length; i++) {
            if (normalizedUnit === lakVariations[i].toLowerCase()) {
                return true;
            }
        }
        
        return false;
    };
    
    // دالة للتحقق من وحدة فل بلاستك
    window.isFullPlasticUnit = function(unitName) {
        if (!unitName) return false;
        
        var fullPlasticVariations = [
            'فل بلاستك', 
            'فل بلاستيك',
            'full plastic',
            'fullplastic',
            'فل',
            'بلاستك'
        ];
        
        var normalizedUnit = unitName.toLowerCase().trim();
        
        for (var i = 0; i < fullPlasticVariations.length; i++) {
            if (normalizedUnit === fullPlasticVariations[i].toLowerCase() || 
                normalizedUnit.includes('فل') && normalizedUnit.includes('بلاست')) {
                return true;
            }
        }
        
        return false;
    };
    
    // دالة مجمعة للتحقق من الوحدات الخاصة
    window.isSpecialBaseUnit = function(unitName) {
        return isLakUnit(unitName) || isFullPlasticUnit(unitName) || isDozenUnit(unitName);
    };
    
    // دالة للتحقق من الكارتون
    window.isCartonUnit = function(unitName) {
        if (!unitName) return false;
        
        var cartonVariations = ['كارتون', 'كرتون', 'carton', 'كارتونة', 'كرتونة'];
        var normalizedUnit = unitName.toLowerCase().trim();
        
        for (var i = 0; i < cartonVariations.length; i++) {
            if (normalizedUnit.includes(cartonVariations[i])) {
                return true;
            }
        }
        
        return false;
    };
    
    // دالة للتحقق من الباكيت
    window.isPacketUnit = function(unitName) {
        if (!unitName) return false;
        
        var packetVariations = ['باكيت', 'باكت', 'packet', 'pack', 'باك'];
        var normalizedUnit = unitName.toLowerCase().trim();
        
        for (var i = 0; i < packetVariations.length; i++) {
            if (normalizedUnit.includes(packetVariations[i])) {
                return true;
            }
        }
        
        var similarity = calculateSimilarity(unitName, 'باكيت');
        return similarity >= 80;
    };
    
    // ==========================================
    // الجزء 7: تحديد السعر حسب الوحدة (محدث لدعم الدرزن)
    // ==========================================
    
    window.determinePriceByUnit = function(product, unitName, row) {
        logPricing('Determining price for unit (V4.3):', unitName);
        
        if (!product) {
            logPricing('No product data provided', null, 'warn');
            return null;
        }
        
        // ============================================
        // تعامل خاص مع وحدة السيت: استخدم السعر الأساسي مباشرة
        // ============================================
        var isSetUnit = unitName.toLowerCase().includes('سيت') || unitName.toLowerCase().includes('set');
        if (isSetUnit) {
            logPricing('🎯 Set unit detected - using base selling price directly');
            
            // الحصول على السعر الأساسي للبيع من المنتج
            var baseSellPrice = parseFloat(product.selling_price || product.default_sell_price || 0);
            
            if (baseSellPrice > 0) {
                return {
                    price: baseSellPrice,
                    shouldUpdateUnit: false,
                    isMainUnit: true,
                    unitType: 'Set Unit (Base Price)'
                };
            }
        }
        
        // استخراج الأسعار
        var priceData = extractPricesFromCustomField3(product.product_custom_field3);
        
        if (!priceData) {
            logPricing('No price data found', null, 'warn');
            return null;
        }
        
        // ============================================
        // معالجة خاصة للوحدات الخاصة (لك، فل بلاستك، درزن)
        // ============================================
        
        // تحديد الوحدة الأساسية للمنتج
        var productMainUnit = getProductMainUnit(product);
        
        // أولوية عالية: إذا كانت الوحدة الخاصة هي نفسها الوحدة الرئيسية
        if (isSpecialBaseUnit(unitName) && productMainUnit && isUnitMatch(unitName, productMainUnit.name)) {
            var unitType = '';
            if (isLakUnit(unitName)) unitType = 'LAK';
            else if (isFullPlasticUnit(unitName)) unitType = 'Full Plastic';
            else if (isDozenUnit(unitName)) unitType = 'Dozen';
            
            logPricing('🎯 Special unit is main unit - priority handling for ' + unitType);
            
            if (priceData.mainPrice && priceData.mainPrice > 0) {
                logPricing('Using main price for ' + unitType + ' as main unit: ' + priceData.mainPrice);
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: false,
                    isMainUnit: true,
                    isSpecialUnit: true,
                    unitType: unitType + ' (Main Unit - Priority)'
                };
            }
        }
        
        // معالجة عامة للوحدات الخاصة (في حالة عدم كونها الوحدة الرئيسية)
        if (isSpecialBaseUnit(unitName)) {
            var unitType = '';
            if (isLakUnit(unitName)) unitType = 'LAK';
            else if (isFullPlasticUnit(unitName)) unitType = 'Full Plastic';
            else if (isDozenUnit(unitName)) unitType = 'Dozen';
            
            logPricing('Special handling for ' + unitType + ' unit (not main unit)');
            
            // للوحدات الخاصة، السعر الأساسي هو سعرها
            if (priceData.mainPrice && priceData.mainPrice > 0) {
                logPricing('Using main price as ' + unitType + ' price: ' + priceData.mainPrice);
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: false,
                    isSpecialUnit: true,
                    unitType: unitType + ' (Special Unit)'
                };
            }
        }
        
        // ============================================
        // التحقق من وحدة الدرزن كوحدة رئيسية للمنتج
        // ============================================
        var dozenCode = 17; // كود الدرزن في UNIT_MAPPING
        var hasDozenAsMainUnit = false;
        var productMainUnit = getProductMainUnit(product);
        
        if (productMainUnit && (isDozenUnit(productMainUnit.name) || productMainUnit.id === dozenCode)) {
            hasDozenAsMainUnit = true;
            logPricing('Product has Dozen as main unit');
        }
        
        // ============================================
        // إذا كانت الوحدة الرئيسية درزن والمستخدم يطلب كارتون
        // ============================================
        if (hasDozenAsMainUnit && isCartonUnit(unitName)) {
            // التحقق من وجود وحدة كارتون في وحدات المنتج
            var cartonCode = 18; // كود الكارتون
            var hasCartonUnit = false;
            
            if (priceData.unitPrices[cartonCode] && priceData.unitPrices[cartonCode] > 0) {
                hasCartonUnit = true;
            } else if (product.units && Array.isArray(product.units)) {
                for (var i = 0; i < product.units.length; i++) {
                    if (isCartonUnit(product.units[i].name) || product.units[i].id === cartonCode) {
                        hasCartonUnit = true;
                        break;
                    }
                }
            }
            
            if (!hasCartonUnit) {
                logPricing('Main unit is Dozen and Carton not available - using Dozen price');
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: true,
                    baseUnit: { name: 'درزن', id: dozenCode, multiplier: 1 },
                    message: 'الوحدة الرئيسية هي درزن - لا يوجد كارتون'
                };
            }
        }
        
        // ============================================
        // التحقق من وجود وحدة الباكيت في المنتج
        // ============================================
        var packetCode = 20; // كود الباكيت في UNIT_MAPPING
        var hasPacketUnit = false;
        
        // التحقق من وجود سعر للباكيت أو أن المنتج يدعم وحدة الباكيت
        if (priceData.unitPrices[packetCode] && priceData.unitPrices[packetCode] > 0) {
            hasPacketUnit = true;
            logPricing('Product has explicit packet price');
        } else if (product.units && Array.isArray(product.units)) {
            // التحقق من وجود وحدة الباكيت في قائمة وحدات المنتج
            for (var i = 0; i < product.units.length; i++) {
                if (isPacketUnit(product.units[i].name) || product.units[i].id === packetCode) {
                    hasPacketUnit = true;
                    logPricing('Product has packet unit in units list');
                    break;
                }
            }
        }
        
        // البحث عن أفضل مطابقة للوحدة في جدول UNIT_MAPPING - محسن لدقة أكبر
        var bestMatch = null;
        var bestScore = 0;
        var threshold = 80;
        
        // المرحلة 1: البحث عن المطابقة الكاملة أولاً
        for (var unitKey in window.UNIT_MAPPING) {
            if (unitName.toLowerCase().trim() === unitKey.toLowerCase().trim()) {
                bestMatch = unitKey;
                bestScore = 100;
                logPricing('🎯 Exact unit match found (V4.3):', unitKey);
                break;
            }
        }
        
        // المرحلة 2: إذا لم نجد مطابقة كاملة، ابحث عن أفضل مطابقة تقريبية
        if (!bestMatch) {
            for (var unitKey in window.UNIT_MAPPING) {
                // تجنب مطابقة الوحدات التي تحتوي على أرقام مع وحدات بدون أرقام
                var inputHasNumbers = /\d/.test(unitName);
                var keyHasNumbers = /\d/.test(unitKey);
                
                if (inputHasNumbers !== keyHasNumbers) {
                    // تخط هذه المطابقة لتجنب خلط "0.25 كغم" مع "كغم"
                    logPricing('⚠️ Skipping mixed match (V4.3):', unitName + ' vs ' + unitKey);
                    continue;
                }
                
                var score = calculateSimilarity(unitName, unitKey);
                if (score > bestScore && score >= threshold) {
                    bestScore = score;
                    bestMatch = unitKey;
                }
            }
        }
        
        // إذا وجدنا مطابقة وكان لها سعر
        if (bestMatch) {
            var unitCode = window.UNIT_MAPPING[bestMatch];
            logPricing('Best unit mapping match:', {
                input: unitName,
                matched: bestMatch,
                code: unitCode,
                score: bestScore + '%'
            });
            
            if (priceData.unitPrices[unitCode]) {
                logPricing('✅ Found specific price for unit code ' + unitCode + ' (' + bestMatch + '): ' + priceData.unitPrices[unitCode]);
                return {
                    price: priceData.unitPrices[unitCode],
                    shouldUpdateUnit: false
                };
            } else {
                logPricing('⚠️ Unit code ' + unitCode + ' found but no price available in unitPrices');
            }
        }
        
        // ============================================
        // معالجة عدم وجود سعر للوحدة المطلوبة
        // ============================================
        
        // إذا كان المنتج يحتوي على وحدة باكيت ولم نجد سعر للوحدة المطلوبة
        if (hasPacketUnit && !priceData.unitPrices[unitCode]) {
            var hasExplicitPacketPrice = priceData.unitPrices[packetCode] && priceData.unitPrices[packetCode] > 0;
            var packetPrice = hasExplicitPacketPrice ? priceData.unitPrices[packetCode] : priceData.mainPrice;
            
            logPricing('Product has packet unit, returning packet price with update flag');
            
            return {
                price: packetPrice,
                shouldUpdateUnit: !isPacketUnit(unitName),
                baseUnit: { name: 'باكيت', id: packetCode, multiplier: 1 }
            };
        }
        
        // إذا لم يكن المنتج يحتوي على وحدة باكيت ولم نجد سعر للوحدة المطلوبة
        if (!hasPacketUnit && !priceData.unitPrices[unitCode]) {
            // إرجاع خطأ - لا نريد العودة للوحدة الأساسية إلا في حالة الباكيت
            logPricing('No matching unit price found and product does not have packet unit', null, 'error');
            return {
                error: false,
               message: 'لم يتم العثور على سعر للوحدة المطلوبة: ' + unitName,
                requestedUnit: unitName,
                availableUnits: Object.keys(priceData.unitPrices).map(function(code) {
                    // البحث عن اسم الوحدة من جدول UNIT_MAPPING
                    for (var unit in window.UNIT_MAPPING) {
                        if (window.UNIT_MAPPING[unit] == code) {
                            return unit + ' (كود: ' + code + ')';
                        }
                    }
                    return 'كود: ' + code;
                })
            };
        }
        
        // إذا كان لدينا سعر أساسي فقط
        if (priceData.mainPrice && !Object.keys(priceData.unitPrices).length) {
            return {
                price: priceData.mainPrice,
                shouldUpdateUnit: false
            };
        }
        
        logPricing('No price found for unit: ' + unitName, null, 'warn');
        return null;
    };
    
    // ==========================================
    // دالة للحصول على الوحدة الرئيسية للمنتج
    // ==========================================
    window.getProductMainUnit = function(product) {
        // البحث في الوحدات المعالجة
        if (product.processed_units && product.processed_units.length > 0) {
            var mainUnit = product.processed_units.find(unit => unit.is_base_unit == 1);
            if (mainUnit) {
                return {
                    name: mainUnit.unit_name || mainUnit.name,
                    id: mainUnit.id,
                    multiplier: 1
                };
            }
        }
        
        // البحث في الوحدات العادية
        if (product.units && Array.isArray(product.units)) {
            var mainUnit = product.units.find(unit => unit.is_base_unit == 1 || unit.multiplier == 1);
            if (mainUnit) {
                return {
                    name: mainUnit.unit_name || mainUnit.name,
                    id: mainUnit.id,
                    multiplier: 1
                };
            }
            
            // إذا لم نجد وحدة أساسية، نأخذ الأولى
            if (product.units[0]) {
                return {
                    name: product.units[0].unit_name || product.units[0].name,
                    id: product.units[0].id,
                    multiplier: product.units[0].multiplier || 1
                };
            }
        }
        
        return null;
    };
    
    // ==========================================
    // دالة لتحديث الصف إلى وحدة الدرزن
    // ==========================================
    window.updateRowToDozenUnit = function(row, dozenUnit) {
        logPricing('Updating row to dozen unit');
        
        // تحديث حقل الوحدة
        var unitInput = row.find('.unit-input');
        if (unitInput.length > 0) {
            unitInput.val(dozenUnit.name);
            
            // تحديث الحقول المخفية
            row.find('.sub_unit_id').val(dozenUnit.id || '');
            row.find('.unit_multiplier').val(dozenUnit.multiplier || 1);
            row.find('.base_unit_multiplier').val(dozenUnit.multiplier || 1);
            row.find('.is_base_unit').val(1);
            
            // حفظ معلومات الوحدة
            row.data('unit-info', {
                value: dozenUnit.name,
                id: dozenUnit.id || '',
                multiplier: dozenUnit.multiplier || 1,
                is_base_unit: 1
            });
            
            // تطبيق تأثير بصري
            unitInput.addClass('dozen-unit-highlight unit-updated');
            
            // عرض رسالة للمستخدم
            addUnitAppliedEffect(row, 'تم التحديث إلى وحدة الدرزن - لا يوجد كارتون');
            
            logPricing('Row updated to dozen unit:', dozenUnit);
        }
    };
    
    // ==========================================
    // دالة لتحديث الصف إلى الوحدة الأساسية
    // ==========================================
    window.updateRowToBaseUnit = function(row, product) {
        logPricing('Updating row to base unit');
        
        // الحصول على الوحدة الأساسية
        var baseUnit = getProductBaseUnit(product);
        if (!baseUnit) {
            logPricing('No base unit found', null, 'warn');
            return;
        }
        
        // تحديث حقل الوحدة
        var unitInput = row.find('.unit-input');
        if (unitInput.length > 0) {
            unitInput.val(baseUnit.name);
            
            // تحديث الحقول المخفية
            row.find('.sub_unit_id').val(baseUnit.id || '');
            row.find('.unit_multiplier').val(1);
            row.find('.base_unit_multiplier').val(1);
            row.find('.is_base_unit').val(1);
            
            // حفظ معلومات الوحدة
            row.data('unit-info', {
                value: baseUnit.name,
                id: baseUnit.id || '',
                multiplier: 1,
                is_base_unit: 1
            });
            
            // تطبيق تأثير بصري للإشارة إلى التغيير
            unitInput.css('background-color', '#fff3cd')
                     .animate({backgroundColor: 'white'}, 1000);
            
            // عرض رسالة للمستخدم
            toastr.info('تم تحديث الوحدة إلى الوحدة الأساسية: ' + baseUnit.name, '', {
                timeOut: 3000,
                positionClass: 'toast-bottom-right'
            });
            
            logPricing('Row updated to base unit:', baseUnit);
        }
    };
    
    // ==========================================
    // دالة لتحديث الصف إلى وحدة الباكيت
    // ==========================================
    window.updateRowToPacketUnit = function(row, packetUnit) {
        logPricing('Updating row to packet unit');
        
        // تحديث حقل الوحدة
        var unitInput = row.find('.unit-input');
        if (unitInput.length > 0) {
            unitInput.val(packetUnit.name);
            
            // تحديث الحقول المخفية
            row.find('.sub_unit_id').val(packetUnit.id || '');
            row.find('.unit_multiplier').val(packetUnit.multiplier || 1);
            row.find('.base_unit_multiplier').val(packetUnit.multiplier || 1);
            
            // حفظ معلومات الوحدة
            row.data('unit-info', {
                value: packetUnit.name,
                id: packetUnit.id || '',
                multiplier: packetUnit.multiplier || 1
            });
            
            // تطبيق تأثير بصري للإشارة إلى التغيير
            unitInput.addClass('packet-unit-highlight unit-updated');
            
            // عرض رسالة للمستخدم
            addUnitAppliedEffect(row, 'تم التحديث إلى وحدة الباكيت');
            
            logPricing('Row updated to packet unit:', packetUnit);
        }
    };
    
    // ==========================================
    // الجزء 8: تطبيق السعر والأخطاء
    // ==========================================
    
    window.applyAdvancedPrice = function(row, price) {
        if (!price || price <= 0) return;
        
        logPricing('Applying price:', price);
        
        try {
            var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
            var priceIncTax = price * exchange_rate;
            
            // تحديث الحقول
            if (typeof __write_number === 'function') {
                __write_number(row.find('.pos_unit_price'), price);
                __write_number(row.find('.pos_unit_price_inc_tax'), priceIncTax);
            } else {
                row.find('.pos_unit_price').val(price).trigger('change');
                row.find('.pos_unit_price_inc_tax').val(priceIncTax).trigger('change');
            }
            
            // تحديث العرض
            row.find('td:eq(5) input').val(price.toFixed(2));
            row.find('td:eq(6) input').val((convertUSDToIQD(price)).toFixed(0)); // تحديث سعر الصرف
            
            // المجموع
            var quantity = parseFloat(row.find('.pos_quantity').val()) || 1;
            var lineTotal = quantity * priceIncTax;
            
            if (typeof __write_number === 'function') {
                __write_number(row.find('.pos_line_total'), lineTotal, false);
            }
            row.find('.pos_line_total_text').text(__currency_trans_from_en(lineTotal, true));
            
            row.addClass('advanced-pricing-applied');
            
            if (typeof pos_total_row === 'function') {
                pos_total_row();
            }
            
            logPricing('Price applied successfully');
        } catch (e) {
            logPricing('Error applying price:', e, 'error');
        }
    };
    
    // دالة لعرض خطأ الوحدة
    window.showUnitError = function(row, errorMessage) {
        logPricing('Showing unit error:', errorMessage);
        
        // تلوين حقل الوحدة بالأحمر
        var unitInput = row.find('.unit-input');
        unitInput.addClass('unit-error-field');
        
        // تلوين حقل السعر بالأحمر
        row.find('.pos_unit_price, .pos_unit_price_inc_tax, td:eq(5) input, td:eq(6) input')
           .addClass('price-error-field');
        
        // إضافة class للصف
        row.addClass('unit-error-row');
        
        // إضافة tooltip للحقل بطريقة آمنة
        try {
            unitInput.attr('title', errorMessage)
                     .attr('data-toggle', 'tooltip')
                     .attr('data-placement', 'top');
            
            // تفعيل tooltip إذا كان متاحاً
            if (typeof unitInput.tooltip === 'function' && typeof $.fn.tooltip !== 'undefined') {
                try {
                    unitInput.tooltip();
                } catch (e) {
                    // تجاهل خطأ tooltip
                    logPricing('Could not initialize tooltip:', e);
                }
            }
        } catch (e) {
            logPricing('Error setting up tooltip:', e);
        }
        
        // إضافة رسالة خطأ بجانب الحقل
        var errorIcon = $('<i class="fa fa-exclamation-circle unit-error-icon" style="color: #dc3545; margin-left: 5px;"></i>');
        errorIcon.attr('title', errorMessage);
        unitInput.after(errorIcon);
    };
    
    // دالة لإزالة أخطاء الوحدة
    window.clearUnitError = function(row) {
        logPricing('Clearing unit error');
        
        // إزالة التلوين
        row.find('.unit-input').removeClass('unit-error-field');
        row.find('.pos_unit_price, .pos_unit_price_inc_tax, td:eq(5) input, td:eq(6) input')
           .removeClass('price-error-field');
        
        // إزالة class الصف
        row.removeClass('unit-error-row');
        
        // إزالة tooltip بطريقة آمنة
        var unitInput = row.find('.unit-input');
        try {
            // التحقق من وجود tooltip أولاً
            if (unitInput.data('bs.tooltip') || unitInput.data('tooltip')) {
                if (typeof unitInput.tooltip === 'function') {
                    // محاولة إزالة tooltip
                    try {
                        unitInput.tooltip('dispose');
                    } catch (e1) {
                        try {
                            unitInput.tooltip('destroy');
                        } catch (e2) {
                            // إذا فشلت كل المحاولات، إزالة البيانات مباشرة
                            unitInput.removeData('bs.tooltip');
                            unitInput.removeData('tooltip');
                        }
                    }
                }
            }
        } catch (e) {
            // تجاهل أي أخطاء في tooltip
            logPricing('Could not remove tooltip:', e);
        }
        
        // إزالة الخصائص
        unitInput.removeAttr('title data-toggle data-placement data-original-title');
        
        // إزالة أيقونة الخطأ
        row.find('.unit-error-icon').remove();
    };
    
    window.getProductBaseUnit = function(product) {
        // البحث في الوحدات المعالجة
        if (product.processed_units && product.processed_units.length > 0) {
            var baseUnit = product.processed_units.find(unit => unit.is_base_unit == 1);
            if (baseUnit && baseUnit.unit_name) {
                return {
                    name: baseUnit.unit_name || baseUnit.name,
                    id: baseUnit.id,
                    multiplier: 1
                };
            }
        }
        
        // البحث في الوحدات العادية
        if (product.units && product.units.length > 0) {
            var baseUnit = product.units.find(unit => unit.is_base_unit == 1 || unit.multiplier == 1);
            if (baseUnit && (baseUnit.unit_name || baseUnit.name)) {
                return {
                    name: baseUnit.unit_name || baseUnit.name,
                    id: baseUnit.id,
                    multiplier: 1
                };
            }
        }
        
        // إرجاع خطأ إذا لم يتم العثور على وحدة مطابقة
        return {
            error: true,
            message: 'لم يتم العثور على وحدة أساسية للمنتج',
            errorMessage: 'No base unit found for product',
            productId: product.id || 'unknown',
            productName: product.name || 'unknown'
        };
    };
    
    // ============================================
    // دالة تطبيق التسعير المتقدم على الصف
    // ============================================
    window.applyAdvancedPricingToRow = function(row) {
        var product = getProductDataForRow(row);
        if (!product) return;
        
        var unitName = row.find('.unit-input').val() || 'قطعة';
        var priceResult = determinePriceByUnit(product, unitName, row);
        
        if (priceResult) {
            // إذا كان هناك خطأ
            if (priceResult.error) {
                showUnitError(row, priceResult.message);
                return;
            }
            
            // إذا كان كائن يحتوي على price
            if (typeof priceResult === 'object' && priceResult.price) {
                // إزالة أي أخطاء سابقة
                clearUnitError(row);
                
                applyAdvancedPrice(row, priceResult.price);
                
                // إذا كان يجب تحديث الوحدة
                if (priceResult.shouldUpdateUnit && priceResult.baseUnit) {
                    if (isDozenUnit(priceResult.baseUnit.name)) {
                        updateRowToDozenUnit(row, priceResult.baseUnit);
                    } else if (isPacketUnit(priceResult.baseUnit.name)) {
                        updateRowToPacketUnit(row, priceResult.baseUnit);
                    } else {
                        updateRowToBaseUnit(row, product);
                    }
                }
            } 
            // إذا كان رقم مباشر (للتوافق مع الكود القديم)
            else if (typeof priceResult === 'number' && priceResult > 0) {
                clearUnitError(row);
                applyAdvancedPrice(row, priceResult);
            }
        }
    };
    
    // ============================================
    // دالة لإضافة تأثير بصري عند تحديث الوحدة
    // ============================================
    window.addUnitAppliedEffect = function(row, message) {
        var unitInput = row.find('.unit-input');
        
        // إضافة class للتأثير
        unitInput.addClass('unit-updated');
        
        // إضافة رسالة تحذيرية مؤقتة
        if (message) {
            var warning = $('<div class="unit-update-warning">' + message + '</div>');
            unitInput.parent().css('position', 'relative').append(warning);
            
            setTimeout(function() {
                warning.fadeOut(function() {
                    $(this).remove();
                });
            }, 3000);
        }
        
        // إزالة class التأثير بعد انتهاء الأنيميشن
        setTimeout(function() {
            unitInput.removeClass('unit-updated');
        }, 1000);
    };
    
    // ==========================================
    // الجزء 9: معالجات الأحداث
    // ==========================================
    
    // إزالة جميع المعالجات القديمة
    $(document).off('change', '.unit-input');
    $(document).off('blur', '.unit-input');
    $(document).off('change.unit');
    $(document).off('blur.unit');
    $(document).off('change.unitfix');
    $(document).off('blur.unitfix');
    
    // دالة مساعدة لحساب التشابه بين نصين - جعلها في النطاق العالمي
    window.calculateSimilarity = function(str1, str2) {
        if (!str1 || !str2) return 0;
        
        str1 = str1.toLowerCase().trim();
        str2 = str2.toLowerCase().trim();
        
        // إذا كانا متطابقين تماماً
        if (str1 === str2) return 100;
        
        // إذا كان أحدهما يحتوي على الآخر
        if (str1.includes(str2) || str2.includes(str1)) return 90;
        
        // إزالة المسافات والرموز الخاصة للمقارنة
        var clean1 = str1.replace(/[\s\-_\.]/g, '');
        var clean2 = str2.replace(/[\s\-_\.]/g, '');
        
        if (clean1 === clean2) return 85;
        if (clean1.includes(clean2) || clean2.includes(clean1)) return 80;
        
        // حساب عدد الأحرف المشتركة
        var longer = str1.length > str2.length ? str1 : str2;
        var shorter = str1.length > str2.length ? str2 : str1;
        
        if (longer.length === 0) return 100;
        
        var editDistance = getEditDistance(longer, shorter);
        var similarity = ((longer.length - editDistance) / longer.length) * 100;
        
        return Math.round(similarity);
    };
    
    // دالة حساب المسافة بين نصين (Levenshtein Distance) - جعلها في النطاق العالمي
    window.getEditDistance = function(str1, str2) {
        var matrix = [];
        
        for (var i = 0; i <= str2.length; i++) {
            matrix[i] = [i];
        }
        
        for (var j = 0; j <= str1.length; j++) {
            matrix[0][j] = j;
        }
        
        for (var i = 1; i <= str2.length; i++) {
            for (var j = 1; j <= str1.length; j++) {
                if (str2.charAt(i - 1) === str1.charAt(j - 1)) {
                    matrix[i][j] = matrix[i - 1][j - 1];
                } else {
                    matrix[i][j] = Math.min(
                        matrix[i - 1][j - 1] + 1, // substitution
                        matrix[i][j - 1] + 1,     // insertion
                        matrix[i - 1][j] + 1      // deletion
                    );
                }
            }
        }
        
        return matrix[str2.length][str1.length];
    };
    
    // دالة للعثور على أفضل تطابق للوحدة - محسنة لمنع الخلط بين الوحدات المتشابهة
    window.findBestUnitMatch = function(inputUnit, availableUnits) {
        if (!inputUnit || !availableUnits || availableUnits.length === 0) return null;
        
        var bestMatch = null;
        var bestScore = 0;
        var threshold = 80; // الحد الأدنى للتشابه المقبول
        
        // المرحلة 1: البحث عن المطابقة الكاملة أولاً
        for (var i = 0; i < availableUnits.length; i++) {
            var unit = availableUnits[i];
            var unitName = unit.unit_name || unit.name || '';
            var shortName = unit.short_name || '';
            
            if (inputUnit.toLowerCase().trim() === unitName.toLowerCase().trim() ||
                inputUnit.toLowerCase().trim() === shortName.toLowerCase().trim()) {
                bestMatch = unit;
                bestScore = 100;
                logPricing('🎯 Exact unit match found in available units:', unitName);
                break;
            }
        }
        
        // المرحلة 2: إذا لم نجد مطابقة كاملة، ابحث عن أفضل مطابقة تقريبية
        if (!bestMatch) {
            availableUnits.forEach(function(unit) {
                var unitName = unit.unit_name || unit.name || '';
                var shortName = unit.short_name || '';
                
                // تجنب مطابقة الوحدات التي تحتوي على أرقام مع وحدات بدون أرقام
                var inputHasNumbers = /\d/.test(inputUnit);
                var nameHasNumbers = /\d/.test(unitName);
                var shortHasNumbers = /\d/.test(shortName);
                
                // حساب التشابه مع الاسم الكامل
                var nameScore = 0;
                if (inputHasNumbers === nameHasNumbers) {
                    nameScore = calculateSimilarity(inputUnit, unitName);
                } else {
                    logPricing('⚠️ Skipping mixed match in available units:', inputUnit + ' vs ' + unitName);
                }
                
                // حساب التشابه مع الاسم المختصر
                var shortScore = 0;
                if (inputHasNumbers === shortHasNumbers) {
                    shortScore = calculateSimilarity(inputUnit, shortName);
                } else {
                    logPricing('⚠️ Skipping mixed match in available units (short):', inputUnit + ' vs ' + shortName);
                }
                
                // أخذ أعلى نقاط
                var score = Math.max(nameScore, shortScore);
                
                if (score > bestScore && score >= threshold) {
                    bestScore = score;
                    bestMatch = unit;
                }
            });
        }
        
        if (bestMatch) {
            logPricing('Best unit match found:', {
                input: inputUnit,
                matched: bestMatch.unit_name || bestMatch.name,
                score: bestScore + '%'
            });
        }
        
        return bestMatch;
    };
    
    // معالج واحد نظيف للوحدات
    var unitChangeHandler = function() {
        var $input = $(this);
        var row = $input.closest('tr');
        
        if ($input.data('unit-processing')) {
            return;
        }
        
        $input.data('unit-processing', true);
        
        setTimeout(function() {
            try {
                var unitName = $input.val();
                logPricing('Unit changed to:', unitName);
                
                // حفظ قيمة الوحدة
                persistUnitValue(row);
                
                // البحث عن تفاصيل الوحدة
                var availableUnits = [];
                try {
                    var unitsData = $input.attr('data-available-units');
                    if (unitsData) {
                        availableUnits = JSON.parse(unitsData);
                    }
                } catch (err) {
                    // ignore
                }
                
                // البحث عن أفضل وحدة مطابقة
                var matchedUnit = findBestUnitMatch(unitName, availableUnits);
                
                // تحديث بيانات الإرسال
                if (matchedUnit) {
                    updateUnitSubmissionData(row, matchedUnit);
                } else {
                    logPricing('No matching unit found for: ' + unitName, null, 'warn');
                }
                
                // تطبيق التسعير المتقدم
                applyAdvancedPricingToRow(row);
                
            } catch (e) {
                logPricing('Error in unit change handler:', e, 'error');
            } finally {
                $input.data('unit-processing', false);
            }
        }, 100);
    };
    
    // تطبيق المعالج
    $(document).on('change', '.unit-input', unitChangeHandler);
    $(document).on('blur', '.unit-input', unitChangeHandler);
    
    // معالج إضافي لتغيير الوحدة يدوياً
    $(document).on('change', '.unit-input', function() {
        var row = $(this).closest('tr');
        var unitName = $(this).val();
        
        // تطبيق التسعير المتقدم مع التحقق من الوحدة
        setTimeout(function() {
            applyAdvancedPricingToRow(row);
        }, 100);
    });
    
    // ==========================================
    // مراقب DOM للصفوف الجديدة
    // ==========================================
    
    window.posTableObserver = null;
    
    function startPosTableObserver() {
        if (window.posTableObserver) {
            window.posTableObserver.disconnect();
        }
        
        var tableBody = document.querySelector('#pos_table tbody');
        if (!tableBody) {
            logPricing('POS table body not found, retrying...', null, 'warn');
            setTimeout(startPosTableObserver, 1000);
            return;
        }
        
        window.posTableObserver = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList') {
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === 1 && node.tagName === 'TR') {
                            var $row = $(node);
                            
                            if ($row.find('.product_id').length > 0 || 
                                $row.find('.variation_id').length > 0 ||
                                $row.hasClass('product_row')) {
                                
                                logPricing('New product row detected via DOM observer');
                                
                                setTimeout(function() {
                                    applyAdvancedPricingToRow($row);
                                }, 300);
                            }
                        }
                    });
                }
            });
        });
        
        window.posTableObserver.observe(tableBody, {
            childList: true,
            subtree: false
        });
        
        logPricing('DOM observer started for POS table');
    }
    
    // معالج لأحداث jQuery المخصصة
    $(document).on('product-added-to-table', function(e, data) {
        logPricing('Product added event triggered');
        
        if (data && data.row) {
            setTimeout(function() {
                applyAdvancedPricingToRow(data.row);
            }, 200);
        } else {
            setTimeout(function() {
                var lastRow = $('#pos_table tbody tr').not('.empty-row').last();
                if (lastRow.length > 0) {
                    applyAdvancedPricingToRow(lastRow);
                }
            }, 300);
        }
    });
    
    // اعتراض نجاح AJAX للمنتجات
    $(document).ajaxSuccess(function(event, xhr, settings) {
        if (settings.url && (
            settings.url.includes('/products/list') ||
            settings.url.includes('/get_product_row') ||
            settings.url.includes('get-product-row') ||
            settings.url.includes('product_row')
        )) {
            logPricing('Product AJAX request completed');
            
            setTimeout(function() {
                $('#pos_table tbody tr').not('.empty-row').each(function() {
                    var $row = $(this);
                    
                    if (!$row.hasClass('advanced-pricing-applied')) {
                        var productId = $row.find('.product_id').val() || $row.find('.variation_id').val();
                        if (productId) {
                            applyAdvancedPricingToRow($row);
                        }
                    }
                });
            }, 500);
        }
    });
    
    // معالج عند اختيار منتج من البحث
    $(document).on('select2:select', '#search_product', function(e) {
        logPricing('Product selected from search');
        
        setTimeout(function() {
            var lastRow = $('#pos_table tbody tr').not('.empty-row').last();
            if (lastRow.length > 0) {
                applyAdvancedPricingToRow(lastRow);
            }
        }, 500);
    });
    
    // معالج الباركود
    $(document).on('barcode-scanned', function(e, barcode) {
        logPricing('Barcode scanned:', barcode);
        
        setTimeout(function() {
            var lastRow = $('#pos_table tbody tr').not('.empty-row').last();
            if (lastRow.length > 0) {
                applyAdvancedPricingToRow(lastRow);
            }
        }, 500);
    });
    
    // دالة فحص دورية (كحل احتياطي)
    var lastProcessedRowCount = 0;
    
    function periodicPricingCheck() {
        var currentRows = $('#pos_table tbody tr').not('.empty-row');
        var currentRowCount = currentRows.length;
        
        if (currentRowCount > lastProcessedRowCount) {
            logPricing('Row count increased, checking new rows');
            
            currentRows.each(function() {
                var $row = $(this);
                if (!$row.hasClass('advanced-pricing-applied')) {
                    var productId = $row.find('.product_id').val() || $row.find('.variation_id').val();
                    if (productId) {
                        applyAdvancedPricingToRow($row);
                    }
                }
            });
            
            lastProcessedRowCount = currentRowCount;
        }
    }
    
    // تشغيل الفحص كل ثانيتين
    setInterval(periodicPricingCheck, 2000);
    
    // تهيئة المعالجات عند التحميل
    $(document).ready(function() {
        logPricing('Initializing event handlers');
        
        setTimeout(startPosTableObserver, 500);
        
        setTimeout(function() {
            $('#pos_table tbody tr').not('.empty-row').each(function() {
                var $row = $(this);
                var productId = $row.find('.product_id').val() || $row.find('.variation_id').val();
                if (productId && !$row.hasClass('advanced-pricing-applied')) {
                    applyAdvancedPricingToRow($row);
                }
            });
        }, 1000);
    });
    
    // ==========================================
    // الجزء 10: تحسين populateRowWithProduct
    // ==========================================
    
    var originalPopulateRow = window.populateRowWithProduct;
    
    window.populateRowWithProduct = function(row, product, rowIndex) {
        logPricing('Populating row with product:', product);
        
        if (product) {
            storeProductData(product);
            row.data('product-data', product);
            
            if (product.product_custom_field3) {
                var fieldName = 'products[' + rowIndex + '][product_custom_field3]';
                var existingField = row.find('input[name="' + fieldName + '"]');
                
                if (existingField.length === 0) {
                    $('<input>').attr({
                        type: 'hidden',
                        name: fieldName,
                        class: 'product_custom_field3',
                        value: product.product_custom_field3
                    }).appendTo(row);
                } else {
                    existingField.val(product.product_custom_field3);
                }
            }
        }
        
        if (originalPopulateRow && typeof originalPopulateRow === 'function') {
            originalPopulateRow.apply(this, arguments);
        }
        
        setTimeout(function() {
            // تفعيل نظام التسعير المتقدم للصف الجديد
            row.addClass('advanced-pricing-active');
            applyAdvancedPricingToRow(row);
        }, 200);
    };
    
    // ==========================================
    // الجزء 11: أدوات التشخيص والاختبار
    // ==========================================
    
    window.testAdvancedPricing = function() {
        console.clear();
        console.log('🧪 === TESTING ADVANCED PRICING SYSTEM V4.3 ===');
        
        // اختبار استخراج الأسعار مع سعر الباكيت
        var testData = `[{"PriceList":1,"Price":18,"Currency":"USD","AdditionalPrice1":27000,"AdditionalCurrency1":"IQD","UoMPrices":[{"PriceList":1,"UoMEntry":17,"Price":2,"Currency":"USD"},{"PriceList":1,"UoMEntry":20,"Price":15,"Currency":"USD"}]},{"PriceList":2,"Price":19}]`;
        
        console.log('\n📊 Test Data (with packet price):', testData);
        
        var priceData = extractPricesFromCustomField3(testData);
        console.log('\n💰 Extracted Price Data:', priceData);
        
        // اختبار المطابقة التقريبية
        console.log('\n🔍 Testing Fuzzy Matching:');
        var testMatches = [
            ['قطعة', 'قطعه'],
            ['كارتون', 'كرتون'],
            ['درزن', 'دزن'],
            ['كغم', 'كيلو'],
            ['غم 50', '50 غم'],
            ['كغم 0.25', 'كغم 0,25'],
            ['باكيت', 'باكت'],
            ['packet', 'باكيت'],
            ['لك', 'LAK']
        ];

        testMatches.forEach(function(pair) {
            var similarity = calculateSimilarity(pair[0], pair[1]);
            console.log(`  "${pair[0]}" vs "${pair[1]}": ${similarity}% similarity`);
        });
        
        // اختبار التحقق من الدرزن
        console.log('\n📦 Testing Dozen Detection:');
        var dozenTests = ['درزن', 'دزن', 'dozen', 'كارتون', 'قطعة'];
        dozenTests.forEach(function(unit) {
            console.log(`  "${unit}": ${isDozenUnit(unit) ? '✓ IS dozen' : '✗ NOT dozen'}`);
        });
        
        // اختبار التحقق من الباكيت
        console.log('\n📦 Testing Packet Detection:');
        var packetTests = ['باكيت', 'باكت', 'packet', 'pack', 'كارتون', 'قطعة'];
        packetTests.forEach(function(unit) {
            console.log(`  "${unit}": ${isPacketUnit(unit) ? '✓ IS packet' : '✗ NOT packet'}`);
        });
        
        // اختبار التحقق من وحدة لك
        console.log('\n💎 Testing LAK Detection:');
        var lakTests = ['لك', 'LAK', 'lak', 'قطعة', 'باكيت'];
        lakTests.forEach(function(unit) {
            console.log(`  "${unit}": ${isLakUnit(unit) ? '✓ IS LAK' : '✗ NOT LAK'}`);
        });
        
        // اختبار التحقق من وحدة فل بلاستك
        console.log('\n🏭 Testing Full Plastic Detection:');
        var fullPlasticTests = ['فل بلاستك', 'فل بلاستيك', 'full plastic', 'فل', 'بلاستك', 'قطعة'];
        fullPlasticTests.forEach(function(unit) {
            console.log(`  "${unit}": ${isFullPlasticUnit(unit) ? '✓ IS Full Plastic' : '✗ NOT Full Plastic'}`);
        });
        
        // اختبار سيناريو الدرزن مع الكارتون
        console.log('\n🎯 Testing Dozen as Main Unit Scenario:');
        
        var testProduct = {
            sub_sku: 'TEST123',
            product_custom_field3: `[{"PriceList":1,"Price":24,"Currency":"USD","UoMPrices":[{"PriceList":1,"UoMEntry":17,"Price":24,"Currency":"USD"}]}]`,
            units: [
                { name: 'درزن', id: 17, is_base_unit: 1, multiplier: 1 }
            ]
        };
        
        console.log('Product with Dozen as main unit (no Carton):');
        var testUnits = ['درزن', 'كارتون', 'قطعة', 'باكيت'];
        
        testUnits.forEach(function(unit) {
            var priceResult = determinePriceByUnit(testProduct, unit);
            if (priceResult && typeof priceResult === 'object') {
                if (priceResult.error) {
                    console.log(`  "${unit}": ❌ ERROR - ${priceResult.message}`);
                } else {
                    console.log(`  "${unit}": ${priceResult.price} (Update: ${priceResult.shouldUpdateUnit}, Message: ${priceResult.message || 'N/A'})`);
                }
            }
        });
        
        // عرض جدول الوحدات
        console.log('\n📋 Unit Mapping Table:');
        console.table(window.UNIT_MAPPING);
        
        return true;
    };
    
    window.debugProductData = function() {
        console.clear();
        console.log('🔍 === PRODUCT DATA DEBUG ===');
        
        console.log('\n📦 Stored Products:', Object.keys(window.productDataStore).length);
        
        console.log('\n📋 Table Rows:');
        $('#pos_table tbody tr').each(function(index) {
            var row = $(this);
            var data = getProductDataForRow(row);
            var hasError = row.hasClass('unit-error-row');
            
            console.log('Row ' + (index + 1) + ':', {
                hasData: !!data,
                sku: data ? data.sub_sku : 'N/A',
                custom_field3: data ? (data.product_custom_field3 ? 'YES' : 'NO') : 'N/A',
                unit: row.find('.unit-input').val(),
                unitCode: window.UNIT_MAPPING[row.find('.unit-input').val()],
                isPacket: isPacketUnit(row.find('.unit-input').val()),
                isDozen: isDozenUnit(row.find('.unit-input').val()),
                hasError: hasError,
                errorIcon: row.find('.unit-error-icon').length > 0
            });
        });
        
        console.log('\n❌ Error Rows:', $('.unit-error-row').length);
        
        return true;
    };
    
    window.diagnoseSystem = function() {
        console.clear();
        console.log('🔧 === SYSTEM DIAGNOSIS V4.3 ===');
        
        console.log('\n✅ Functions Status:');
        console.log('persistUnitValue:', typeof window.persistUnitValue);
        console.log('updateUnitSubmissionData:', typeof window.updateUnitSubmissionData);
        console.log('extractPricesFromCustomField3:', typeof window.extractPricesFromCustomField3);
        console.log('determinePriceByUnit:', typeof window.determinePriceByUnit);
        console.log('applyAdvancedPrice:', typeof window.applyAdvancedPrice);
        console.log('isPacketUnit:', typeof window.isPacketUnit);
        console.log('isDozenUnit:', typeof window.isDozenUnit);
        console.log('showUnitError:', typeof window.showUnitError);
        console.log('clearUnitError:', typeof window.clearUnitError);
        
        console.log('\n📊 Data Stores:');
        console.log('productDataStore entries:', Object.keys(window.productDataStore).length);
        console.log('lastAjaxProductData entries:', Object.keys(window.lastAjaxProductData).length);
        console.log('Unit mapping entries:', Object.keys(window.UNIT_MAPPING).length);
        
        console.log('\n📦 Special Units Support:');
        console.log('Packet code in UNIT_MAPPING:', window.UNIT_MAPPING['باكيت']);
        console.log('Dozen code in UNIT_MAPPING:', window.UNIT_MAPPING['درزن']);
        console.log('LAK code in UNIT_MAPPING:', window.UNIT_MAPPING['لك']);
        console.log('Full Plastic code in UNIT_MAPPING:', window.UNIT_MAPPING['فل بلاستك']);
        
        console.log('\n❌ Error Handling:');
        console.log('Error rows:', $('.unit-error-row').length);
        console.log('Error fields:', $('.unit-error-field').length);
        console.log('Error icons:', $('.unit-error-icon').length);
        
        return true;
    };
    
    // ==========================================
    // الجزء 12: التهيئة
    // ==========================================
    
    function initialize() {
        logPricing('Initializing Advanced Pricing System V4.3...');
        
        // إضافة الأزرار
        if ($('.advanced-pricing-buttons').length === 0) {
            $('.pos-form-actions').append(`
                <div class="advanced-pricing-buttons" style="display: inline-block; margin-left: 10px;">
                    <button type="button" class="btn btn-warning btn-sm" onclick="testAdvancedPricing()">
                        <i class="fa fa-flask"></i> Test
                    </button>
                    <button type="button" class="btn btn-info btn-sm" onclick="debugProductData()">
                        <i class="fa fa-bug"></i> Debug
                    </button>
                    <button type="button" class="btn btn-danger btn-sm" onclick="diagnoseSystem()">
                        <i class="fa fa-stethoscope"></i> Diagnose
                    </button>
                    <button type="button" class="btn btn-success btn-sm" onclick="clearAllErrors()">
                        <i class="fa fa-refresh"></i> Clear Errors
                    </button>
                </div>
            `);
        }
        
        // تطبيق على الصفوف الموجودة
        setTimeout(function() {
            $('#pos_table tbody tr').each(function() {
                var row = $(this);
                if (row.find('.product_id').val()) {
                    applyAdvancedPricingToRow(row);
                }
            });
        }, 500);
        
        logPricing('System initialized successfully');
    }
    
    // ==========================================
    // دالة إضافية لمسح جميع الأخطاء
    // ==========================================
    window.clearAllErrors = function() {
        $('#pos_table tbody tr').each(function() {
            try {
                clearUnitError($(this));
            } catch (e) {
                // تجاهل أي أخطاء أثناء المسح
                console.log('Error clearing row:', e);
            }
        });
        
        if (typeof toastr !== 'undefined') {
            toastr.success('تم مسح جميع الأخطاء', '', {
                timeOut: 2000,
                positionClass: 'toast-bottom-right'
            });
        }
        
        logPricing('All errors cleared');
    };
    
    // ==========================================
    // دالة تسجيل رسالة تحذيرية عند عدم وجود وحدة مطابقة
    // ==========================================
    window.logUnitMismatchWarning = function(requestedUnit, availableUnits) {
        console.warn('⚠️ Unit Mismatch Warning:', {
            requested: requestedUnit,
            available: availableUnits,
            message: 'No price found for requested unit'
        });
    };
    
    // ==========================================
    // الجزء 13: الأنماط
    // ==========================================
    
    // إضافة أنماط CSS محدثة
    var unitUpdateStyles = `
    <style>
    /* تأثير تحديث الوحدة */
    .unit-input.unit-updated {
        animation: unitUpdatePulse 1s ease-in-out;
    }
    
    @keyframes unitUpdatePulse {
        0% { background-color: #fff3cd; transform: scale(1); }
        50% { background-color: #ffeaa7; transform: scale(1.05); }
        100% { background-color: white; transform: scale(1); }
    }
    
    /* رسالة تحذيرية عند تحديث الوحدة */
    .unit-update-warning {
        position: absolute;
        top: -25px;
        left: 0;
        background: #ff9800;
        color: white;
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 11px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        z-index: 1000;
        white-space: nowrap;
    }
    
    .unit-update-warning::after {
        content: '';
        position: absolute;
        bottom: -5px;
        left: 10px;
        width: 0;
        height: 0;
        border-left: 5px solid transparent;
        border-right: 5px solid transparent;
        border-top: 5px solid #ff9800;
    }
    
    /* تأثير خاص للباكيت */
    .packet-unit-highlight {
        background-color: #e3f2fd !important;
        border: 1px solid #2196F3 !important;
    }
    
    /* تأثير خاص للدرزن */
    .dozen-unit-highlight {
        background-color: #f3e5f5 !important;
        border: 1px solid #9c27b0 !important;
    }
    
    /* أنماط الأخطاء الجديدة */
    .unit-error-field {
        background-color: #ffebee !important;
        border: 2px solid #dc3545 !important;
        color: #721c24 !important;
    }
    
    .price-error-field {
        background-color: #ffebee !important;
        border: 1px solid #dc3545 !important;
        color: #721c24 !important;
    }
    
    .unit-error-row {
        background-color: #fff5f5 !important;
    }
    
    .unit-error-icon {
        cursor: help;
        animation: errorPulse 2s infinite;
    }
    
    @keyframes errorPulse {
        0% { opacity: 1; }
        50% { opacity: 0.5; }
        100% { opacity: 1; }
    }
    
    /* تحسين مظهر رسائل الخطأ */
    .toast-error {
        background-color: #dc3545 !important;
    }
    
    .tooltip.show {
        opacity: 1 !important;
    }
    
    .tooltip-inner {
        background-color: #dc3545;
        max-width: 300px;
    }
    
    .tooltip.bs-tooltip-top .arrow::before {
        border-top-color: #dc3545;
    }
    
    /* تمييز صفوف الدرزن */
    tr:has(.unit-input[value*="درزن"]) {
        border-left: 3px solid #9c27b0;
    }
    
    /* تمييز صفوف الباكيت */
    tr:has(.unit-input[value*="باكيت"]) {
        border-left: 3px solid #2196F3;
    }
    </style>
    `;
    
    // إضافة الأنماط عند التحميل
    $(document).ready(function() {
        $('head').append(unitUpdateStyles);
    });
    
    var styles = `
        <style id="advancedPricingStyles">
        .advanced-pricing-applied {
            background-color: #e8f5e9 !important;
        }
        
        .advanced-pricing-applied.unit-error-row {
            background-color: #fff5f5 !important;
        }
        
        .advanced-pricing-error {
            background-color: #ffebee !important;
        }
        
        .advanced-pricing-buttons button {
            margin-left: 5px;
        }
        
        .advanced-pricing-buttons .btn-warning {
            background-color: #ff9800;
            border-color: #ff9800;
        }
        
        .advanced-pricing-buttons .btn-warning:hover {
            background-color: #e68900;
            border-color: #e68900;
        }
        
        .advanced-pricing-buttons .btn-success {
            background-color: #4caf50;
            border-color: #4caf50;
        }
        
        .advanced-pricing-buttons .btn-success:hover {
            background-color: #45a049;
            border-color: #45a049;
        }
        
        /* تمييز صفوف الباكيت */
        tr:has(.unit-input[value*="باكيت"]) {
            border-left: 3px solid #2196F3;
        }
        
        /* تمييز صفوف الدرزن */
        tr:has(.unit-input[value*="درزن"]) {
            border-left: 3px solid #9c27b0;
        }
        
        /* تأكد من أن الأخطاء مرئية */
        .unit-error-field {
            background-color: #ffebee !important;
            border: 2px solid #dc3545 !important;
        }
        
        .price-error-field {
            background-color: #ffebee !important;
            border: 1px solid #dc3545 !important;
        }
        </style>
    `;
    
    if ($('#advancedPricingStyles').length === 0) {
        $('head').append(styles);
    }
    
    // ==========================================
    // الجزء 14: البدء
    // ==========================================
    
    $(document).ready(function() {
        setTimeout(function() {
            try {
                initialize();
                console.log('✅ Advanced Pricing System V4.3 loaded successfully');
                console.log('📌 New Feature: Dozen unit support - no Carton fallback');
                console.log('📌 When main unit is Dozen and Carton not available, use Dozen price');
                console.log('📌 Special units (LAK, Full Plastic, Dozen) use main price');
                console.log('📌 Use buttons to test: Test | Debug | Diagnose | Clear Errors');
            } catch (e) {
                console.error('❌ Failed to initialize:', e);
            }
        }, 1500);
    });
    
})();
// ============================================
// نظام معالجة SKU المتعدد - نسخة معزولة وقابلة للإيقاف
// ============================================
// إضافة معالج للتأكد من عدم تعطيل الأزرار بشكل دائم
    function ensureButtonsEnabled() {
        // التحقق كل ثانية من حالة الأزرار الرئيسية
        setInterval(function() {
            if (!isProcessing) {
                $('#pos-draft, #pos-quotation, #pos-finalize').each(function() {
                    if ($(this).prop('disabled') && !$(this).hasClass('manual-sku-disabled')) {
                        // إذا كان الزر معطل وليس بسبب النظام، أعد تفعيله
                        $(this).prop('disabled', false).removeAttr('disabled');
                    }
                });
            }
        }, 1000);
    }    // البحث المباشر وإضافة المنتج
    function searchAndAddProduct(sku, callback) {
        var price_group = $('#price_group').val() || '';
        
        // البحث عن المنتج
        $.ajax({
            url: base_path + '/products/list',
            method: 'GET',
            dataType: 'json',
            data: {
                price_group: price_group,
                term: sku,
                not_for_selling: 0,
                limit: 10,
                search_all_locations: true,
                include_all_warehouses: true,
                with_warehouse_stock: true,
                with_sub_units: true,
                include_unit_details: true,
                load_sub_units: true
            },
            success: function(products) {
                if (products && products.length > 0) {
                    // البحث عن تطابق دقيق
                    var exactMatch = null;
                    for (var i = 0; i < products.length; i++) {
                        if (products[i].sub_sku === sku || products[i].sku === sku) {
                            exactMatch = products[i];
                            break;
                        }
                    }
                    
                    if (exactMatch) {
                        // إضافة المنتج مباشرة
                        if (typeof pos_product_row === 'function') {
                            pos_product_row(exactMatch.variation_id);
                            callback(true);
                        } else {
                            console.error('pos_product_row function not found');
                            callback(false);
                        }
                    } else {
                        // استخدام أول نتيجة
                        if (typeof pos_product_row === 'function') {
                            pos_product_row(products[0].variation_id);
                            callback(true);
                        } else {
                            console.error('pos_product_row function not found');
                            callback(false);
                        }
                    }
                } else {
                    console.warn('No products found for SKU:', sku);
                    callback(false);
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                console.error('Search error:', textStatus, errorThrown);
                callback(false);
            }
        });
    }
    
    // تطبيق الوحدة على آخر صف
    function applyUnitToLastRow(selectedUnit) {
        var lastRow = $('#pos_table tbody tr:last');
        if (lastRow.length === 0) return;
        
        // البحث عن select الوحدة
        var unitSelect = lastRow.find('select.sub_unit');
        if (unitSelect.length > 0) {
            // البحث عن الخيار المطابق
            var targetOption = null;
            
            unitSelect.find('option').each(function() {
                var optionText = $(this).text().trim();
                var optionMultiplier = parseFloat($(this).data('multiplier')) || 1;
                
                // مطابقة حسب الاسم
                if (selectedUnit.name && optionText === selectedUnit.name) {
                    targetOption = $(this);
                    return false;
                }
                
                // مطابقة حسب المضاعف
                if (selectedUnit.multiplier && Math.abs(optionMultiplier - selectedUnit.multiplier) < 0.001) {
                    targetOption = $(this);
                    return false;
                }
            });
            
            if (targetOption) {
                unitSelect.val(targetOption.val()).trigger('change');
                console.log('Applied unit:', selectedUnit.name || selectedUnit.filter);
            }
        }
    }// ============================================
// نظام معالجة SKU المتعدد - نسخة محدّثة مع دعم الوحدات والفلاتر
// ===========================================




// ============================================
// نظام لصق البيانات من Excel مع زر الإيقاف - النسخة المحسنة
// ============================================

// متغيرات عامة للتحكم في المعالجة
var isProcessing = false;
var isPaused = false;
var shouldStop = false;
var currentProcessIndex = 0;
var processedData = [];

// تهيئة النظام عند تحميل الصفحة
$(document).ready(function() {
    setTimeout(function() {
        initializeExcelPasteSystem();
    }, 1500);
});

// تهيئة نظام لصق البيانات من Excel
function initializeExcelPasteSystem() {
    console.log('🚀 Initializing Excel paste system with stop control...');
    
    // إضافة زر اللصق المباشر

    
    // إضافة معالجات الأحداث
    attachExcelPasteEventHandlers();
    
    // إضافة الأنماط المحسنة
    addExcelPasteStyles();
    
    console.log('✅ Excel paste system initialized');
}

// إضافة زر اللصق من Excel


// إضافة الأنماط المحسنة
function addExcelPasteStyles() {
    var styles = `
    <style id="excelPasteStyles">
    /* نافذة لصق البيانات */
    .excel-paste-modal {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 9999;
        display: flex;
        align-items: center;
        justify-content: center;
    }
    
    .excel-paste-content {
        background: white;
        padding: 30px;
        border-radius: 15px;
        max-width: 900px;
        width: 90%;
        max-height: 85vh;
        overflow-y: auto;
        box-shadow: 0 10px 30px rgba(0,0,0,0.3);
    }
    
    .excel-paste-content h3 {
        margin-bottom: 20px;
        color: #333;
        display: flex;
        align-items: center;
        justify-content: space-between;
    }
    
    .excel-paste-textarea {
        width: 100%;
        height: 300px;
        font-family: monospace;
        font-size: 13px;
        border: 2px solid #dee2e6;
        border-radius: 8px;
        padding: 10px;
        resize: vertical;
    }
    
    .excel-paste-textarea:focus {
        border-color: #007bff;
        outline: none;
        box-shadow: 0 0 0 3px rgba(0,123,255,0.1);
    }
    
    .paste-instructions {
        background: #f8f9fa;
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 20px;
        border-left: 4px solid #007bff;
    }
    
    .paste-instructions h5 {
        margin-bottom: 10px;
        color: #007bff;
    }
    
    .paste-instructions ul {
        margin-bottom: 0;
        padding-left: 20px;
    }
    
    .paste-instructions li {
        margin-bottom: 5px;
    }
    
    .paste-preview {
        margin-top: 20px;
        max-height: 250px;
        overflow-y: auto;
        border: 1px solid #dee2e6;
        border-radius: 8px;
    }
    
    .paste-preview table {
        width: 100%;
        font-size: 12px;
    }
    
    .paste-preview th {
        background: #f8f9fa;
        padding: 8px;
        text-align: center;
        font-weight: 600;
        position: sticky;
        top: 0;
        z-index: 10;
    }
    
    .paste-preview td {
        padding: 6px;
        border-bottom: 1px solid #dee2e6;
        text-align: center;
    }
    
    /* حالات الصفوف */
    .paste-preview tr.row-processing {
        background-color: #cfe2ff !important;
        animation: pulse 1.5s infinite;
    }
    
    .paste-preview tr.row-completed {
        background-color: #d1e7dd !important;
    }
    
    .paste-preview tr.row-error {
        background-color: #f8d7da !important;
    }
    
    .paste-preview tr.row-pending {
        background-color: #fff3cd !important;
    }
    
    @keyframes pulse {
        0% { opacity: 1; }
        50% { opacity: 0.6; }
        100% { opacity: 1; }
    }
    
    /* مؤشر التقدم */
    .paste-progress {
        margin-top: 20px;
        display: none;
    }
    
    .paste-progress-bar {
        height: 30px;
        background: #e9ecef;
        border-radius: 15px;
        overflow: hidden;
        position: relative;
        box-shadow: inset 0 2px 4px rgba(0,0,0,0.1);
    }
    
    .paste-progress-fill {
        height: 100%;
        background: linear-gradient(90deg, #007bff, #0056b3);
        width: 0%;
        transition: width 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 13px;
        font-weight: 600;
        position: relative;
        overflow: hidden;
    }
    
    .paste-progress-fill::after {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        bottom: 0;
        right: 0;
        background: linear-gradient(
            45deg,
            transparent 25%,
            rgba(255,255,255,0.2) 25%,
            rgba(255,255,255,0.2) 50%,
            transparent 50%,
            transparent 75%,
            rgba(255,255,255,0.2) 75%,
            rgba(255,255,255,0.2)
        );
        background-size: 30px 30px;
        animation: progress-animation 1s linear infinite;
    }
    
    @keyframes progress-animation {
        0% { background-position: 0 0; }
        100% { background-position: 30px 30px; }
    }
    
    .paste-progress-text {
        text-align: center;
        margin-top: 10px;
        font-size: 13px;
        color: #6c757d;
    }
    
    /* أزرار التحكم في المعالجة */
    .process-control-buttons {
        display: none;
        gap: 10px;
        margin-top: 15px;
        justify-content: center;
    }
    
    .process-control-buttons .btn {
        min-width: 120px;
    }
    
    #stopProcessBtn {
        background: #dc3545;
        border-color: #dc3545;
        color: white;
    }
    
    #stopProcessBtn:hover {
        background: #c82333;
        border-color: #bd2130;
    }
    
    #pauseProcessBtn {
        background: #ffc107;
        border-color: #ffc107;
        color: #212529;
    }
    
    #pauseProcessBtn:hover {
        background: #e0a800;
        border-color: #d39e00;
    }
    
    #pauseProcessBtn.btn-resume {
        background: #28a745;
        border-color: #28a745;
        color: white;
    }
    
    #pauseProcessBtn.btn-resume:hover {
        background: #218838;
        border-color: #1e7e34;
    }
    
    /* رسائل الحالة */
    .paste-status {
        margin-top: 15px;
        padding: 12px;
        border-radius: 8px;
        font-size: 14px;
        display: none;
        position: relative;
    }
    
    .paste-status.success {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }
    
    .paste-status.error {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }
    
    .paste-status.warning {
        background: #fff3cd;
        color: #856404;
        border: 1px solid #ffeaa7;
    }
    
    .paste-status.info {
        background: #d1ecf1;
        color: #0c5460;
        border: 1px solid #bee5eb;
    }
    
    /* ملخص المعالجة */
    .process-summary {
        display: none;
        margin-top: 20px;
        padding: 15px;
        background: #f8f9fa;
        border-radius: 8px;
        border: 1px solid #dee2e6;
    }
    
    .process-summary h5 {
        margin-bottom: 15px;
        color: #495057;
    }
    
    .summary-stats {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
        gap: 15px;
    }
    
    .summary-stat {
        text-align: center;
        padding: 10px;
        background: white;
        border-radius: 8px;
        border: 1px solid #e9ecef;
    }
    
    .summary-stat .stat-value {
        font-size: 24px;
        font-weight: 700;
        display: block;
        margin-bottom: 5px;
    }
    
    .summary-stat .stat-label {
        font-size: 12px;
        color: #6c757d;
    }
    
    .summary-stat.success .stat-value {
        color: #28a745;
    }
    
    .summary-stat.error .stat-value {
        color: #dc3545;
    }
    
    .summary-stat.pending .stat-value {
        color: #ffc107;
    }
    
    /* تحسينات الأزرار */
    .excel-paste-buttons {
        margin-top: 20px;
        display: flex;
        gap: 10px;
        justify-content: flex-end;
    }
    
    .excel-paste-buttons .btn {
        padding: 10px 20px;
        border-radius: 8px;
        font-weight: 500;
        min-width: 100px;
    }
    
    /* مؤشر الحالة */
    .status-indicator {
        display: inline-block;
        width: 10px;
        height: 10px;
        border-radius: 50%;
        margin-right: 5px;
        animation: blink 1s infinite;
    }
    
    .status-indicator.processing {
        background: #007bff;
    }
    
    .status-indicator.paused {
        background: #ffc107;
        animation: none;
    }
    
    .status-indicator.stopped {
        background: #dc3545;
        animation: none;
    }
    
    @keyframes blink {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.3; }
    }
    </style>
    `;
    
    $('head').append(styles);
}

// إضافة معالجات الأحداث
function attachExcelPasteEventHandlers() {
    // معالج زر اللصق
    $(document).on('click', '#excelPasteBtn', function() {
        resetProcessState();
        showExcelPasteModal();
    });
    
    // معالج اللصق المباشر في أي مكان في الجدول
    $(document).on('paste', '#pos_table', function(e) {
        e.preventDefault();
        handleDirectTablePaste(e);
    });
}

// إعادة تعيين حالة المعالجة
function resetProcessState() {
    isProcessing = false;
    isPaused = false;
    shouldStop = false;
    currentProcessIndex = 0;
    processedData = [];
}

// عرض نافذة لصق البيانات
function showExcelPasteModal() {
    var modalHTML = `
    <div class="excel-paste-modal" id="excelPasteModal">
        <div class="excel-paste-content">
            <h3>
                <span><i class="fa fa-file-excel-o"></i> لصق البيانات من Excel</span>
                <span class="status-indicator" id="processStatusIndicator" style="display: none;"></span>
            </h3>
            
            <div class="paste-instructions">
                <h5>تعليمات اللصق:</h5>
                <ul>
                    <li>الصق البيانات بالترتيب التالي: <strong>كود المنتج (SKU) | الكمية | وحدة القياس</strong></li>
                    <li>يمكن لصق أعمدة متعددة مفصولة بـ Tab</li>
                    <li>يمكن لصق صفوف متعددة (كل صف في سطر جديد)</li>
                    <li>مثال: <code>ADF001⇥10⇥EA</code> أو <code>R502⇥5⇥½</code></li>
                    <li>الوحدات المدعومة: EA, ½, ¼, ⅛, KG, 100غم, 125غم, 50غم</li>
                </ul>
            </div>
            
            <textarea class="excel-paste-textarea" id="excelDataInput" 
                      placeholder="الصق البيانات هنا من Excel..."></textarea>
            
            <div class="paste-preview" id="pastePreview" style="display: none;">
                <h5>معاينة البيانات:</h5>
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th width="5%">#</th>
                            <th width="30%">كود المنتج</th>
                            <th width="15%">الكمية</th>
                            <th width="15%">الوحدة</th>
                            <th width="20%">الحالة</th>
                            <th width="15%">النتيجة</th>
                        </tr>
                    </thead>
                    <tbody id="previewTableBody"></tbody>
                </table>
            </div>
            
            <div class="paste-progress" id="pasteProgress">
                <div class="paste-progress-bar">
                    <div class="paste-progress-fill" id="progressFill">0%</div>
                </div>
                <div class="paste-progress-text" id="progressText">جاري المعالجة...</div>
            </div>
            
            <div class="process-control-buttons" id="processControlButtons">
                <button type="button" class="btn btn-warning" id="pauseProcessBtn">
                    <i class="fa fa-pause"></i> إيقاف مؤقت
                </button>
                <button type="button" class="btn btn-danger" id="stopProcessBtn">
                    <i class="fa fa-stop"></i> إيقاف نهائي
                </button>
            </div>
            
            <div class="paste-status" id="pasteStatus"></div>
            
            <div class="process-summary" id="processSummary">
                <h5>ملخص المعالجة</h5>
                <div class="summary-stats">
                    <div class="summary-stat success">
                        <span class="stat-value" id="successCount">0</span>
                        <span class="stat-label">نجح</span>
                    </div>
                    <div class="summary-stat error">
                        <span class="stat-value" id="errorCount">0</span>
                        <span class="stat-label">فشل</span>
                    </div>
                    <div class="summary-stat pending">
                        <span class="stat-value" id="pendingCount">0</span>
                        <span class="stat-label">متبقي</span>
                    </div>
                </div>
            </div>
            
            <div class="excel-paste-buttons">
                <button type="button" class="btn btn-secondary" id="cancelPasteBtn">إلغاء</button>
                <button type="button" class="btn btn-primary" id="previewPasteBtn">معاينة</button>
                <button type="button" class="btn btn-success" id="processPasteBtn" style="display: none;">
                    <i class="fa fa-check"></i> بدء المعالجة
                </button>
                <button type="button" class="btn btn-info" id="continueProcessBtn" style="display: none;">
                    <i class="fa fa-play"></i> متابعة المعالجة
                </button>
            </div>
        </div>
    </div>
    `;
    
    $('body').append(modalHTML);
    
    // التركيز على منطقة اللصق
    $('#excelDataInput').focus();
    
    // معالجات الأزرار
    setupModalEventHandlers();
}

// إعداد معالجات أحداث النافذة
function setupModalEventHandlers() {
    // معالج الإلغاء
    $('#cancelPasteBtn, #excelPasteModal').on('click', function(e) {
        if (e.target === this || $(e.target).attr('id') === 'cancelPasteBtn') {
            if (isProcessing && !shouldStop) {
                if (confirm('هل أنت متأكد من إيقاف المعالجة وإغلاق النافذة؟')) {
                    shouldStop = true;
                    setTimeout(() => {
                        $('#excelPasteModal').remove();
                    }, 500);
                }
            } else {
                $('#excelPasteModal').remove();
            }
        }
    });
    
    // معالج المعاينة
    $('#previewPasteBtn').on('click', function() {
        previewPastedData();
    });
    
    // معالج بدء المعالجة
    $('#processPasteBtn').on('click', function() {
        startProcessing();
    });
    
    // معالج متابعة المعالجة
    $('#continueProcessBtn').on('click', function() {
        continueProcessing();
    });
    
    // معالج الإيقاف المؤقت/الاستئناف
    $('#pauseProcessBtn').on('click', function() {
        togglePauseProcess();
    });
    
    // معالج الإيقاف النهائي
    $('#stopProcessBtn').on('click', function() {
        stopProcess();
    });
    
    // معالج اللصق في منطقة النص
    $('#excelDataInput').on('paste', function(e) {
        setTimeout(function() {
            previewPastedData();
        }, 100);
    });
    
    // معالج تغيير النص
    $('#excelDataInput').on('input', function() {
        if ($(this).val().trim()) {
            $('#previewPasteBtn').prop('disabled', false);
        } else {
            $('#previewPasteBtn').prop('disabled', true);
            $('#pastePreview').hide();
            $('#processPasteBtn').hide();
        }
    });
}

// معاينة البيانات الملصقة
function previewPastedData() {
    var pastedText = $('#excelDataInput').val().trim();
    
    if (!pastedText) {
        showPasteStatus('يرجى لصق البيانات أولاً', 'warning');
        return;
    }
    
    var parsedData = parseExcelData(pastedText);
    
    if (parsedData.length === 0) {
        showPasteStatus('لم يتم العثور على بيانات صالحة', 'error');
        return;
    }
    
    // حفظ البيانات للمعالجة
    processedData = parsedData;
    
    // عرض المعاينة
    displayPreview(parsedData);
    
    // إظهار زر المعالجة
    $('#processPasteBtn').show();
}

// تحليل بيانات Excel
function parseExcelData(text) {
    var rows = text.split('\n').filter(row => row.trim());
    var parsedData = [];
    
    rows.forEach(function(row, index) {
        var columns = row.split('\t').map(col => col.trim());
        
        if (columns.length >= 1) {
            var data = {
                row: index + 1,
                sku: columns[0] || '',
                quantity: parseFloat(columns[1]) || 1,
                unit: columns[2] || 'EA',
                status: 'pending',
                processed: false,
                error: null
            };
            
            // التحقق من صحة البيانات
            if (data.sku) {
                data.isValid = true;
                data.statusText = 'جاهز للمعالجة';
            } else {
                data.isValid = false;
                data.statusText = 'كود المنتج مفقود';
            }
            
            parsedData.push(data);
        }
    });
    
    return parsedData;
}

// عرض المعاينة
function displayPreview(data) {
    var tbody = $('#previewTableBody');
    tbody.empty();
    
    data.forEach(function(item, index) {
        var statusClass = item.isValid ? 'text-success' : 'text-danger';
        var statusIcon = item.isValid ? 'fa-check-circle' : 'fa-exclamation-circle';
        
        var row = `
            <tr id="previewRow${index}" class="row-pending">
                <td>${item.row}</td>
                <td><strong>${item.sku}</strong></td>
                <td>${item.quantity}</td>
                <td>${item.unit}</td>
                <td class="${statusClass}">
                    <i class="fa ${statusIcon}"></i> 
                    <span class="status-text">${item.statusText}</span>
                </td>
                <td class="result-cell">
                    <span class="result-text">-</span>
                </td>
            </tr>
        `;
        
        tbody.append(row);
    });
    
    $('#pastePreview').show();
    
    // عرض ملخص
    var validCount = data.filter(item => item.isValid).length;
    var totalCount = data.length;
    
    showPasteStatus(`تم العثور على ${totalCount} صف، ${validCount} صالح للمعالجة`, 'success');
}

// بدء المعالجة
function startProcessing() {
    var validData = processedData.filter(item => item.isValid);
    
    if (validData.length === 0) {
        showPasteStatus('لا توجد بيانات صالحة للمعالجة', 'error');
        return;
    }
    
    // تعيين حالة المعالجة
    isProcessing = true;
    isPaused = false;
    shouldStop = false;
    currentProcessIndex = 0;
    
    // تحديث واجهة المستخدم
    $('#processPasteBtn').hide();
    $('#previewPasteBtn').hide();
    $('#excelDataInput').prop('readonly', true);
    $('#processControlButtons').css('display', 'flex');
    $('#pasteProgress').show();
    $('#processSummary').show();
    $('#processStatusIndicator').show().addClass('processing');
    
    // بدء المعالجة
    processNextRow();
}

// معالجة الصف التالي
async function processNextRow() {
    if (shouldStop || currentProcessIndex >= processedData.length) {
        completeProcessing();
        return;
    }
    
    if (isPaused) {
        showPasteStatus('المعالجة متوقفة مؤقتاً', 'info');
        return;
    }
    
    var item = processedData[currentProcessIndex];
    
    // تخطي الصفوف غير الصالحة أو المعالجة مسبقاً
    if (!item.isValid || item.processed) {
        currentProcessIndex++;
        processNextRow();
        return;
    }
    
    // تحديث واجهة المستخدم
    updateProcessingUI(currentProcessIndex, 'processing');
    updateProgress();
    
    try {
        // معالجة الصف
        await processDataRow(item);
        
        // تحديث الحالة
        item.processed = true;
        item.status = 'success';
        updateProcessingUI(currentProcessIndex, 'completed', '✓ تمت الإضافة');
        
    } catch (error) {
        console.error('Error processing row:', error);
        item.processed = true;
        item.status = 'error';
        item.error = error.message || 'خطأ غير معروف';
        updateProcessingUI(currentProcessIndex, 'error', '✗ ' + item.error);
    }
    
    // تحديث الإحصائيات
    updateSummaryStats();
    
    // الانتقال للصف التالي
    currentProcessIndex++;
    
    // تأخير صغير ثم معالجة الصف التالي
    setTimeout(() => {
        processNextRow();
    }, 300);
}

// معالجة صف البيانات
async function processDataRow(item) {
    return new Promise((resolve, reject) => {
        // إضافة صف فارغ إذا لزم الأمر
        ensureEmptyRow();
        
        // الحصول على الصف الفارغ
        var emptyRow = $('#pos_table tbody tr.empty-row').first();
        
        if (emptyRow.length === 0) {
            addEmptyProductRow();
            emptyRow = $('#pos_table tbody tr.empty-row').first();
        }
        
        var rowIndex = emptyRow.data('row_index') || emptyRow.index();
        
        // البحث عن المنتج باستخدام SKU
        $.ajax({
            url: base_path + '/products/list',
            method: 'GET',
            dataType: 'json',
            data: {
                term: item.sku,
                search_fields: ['sku', 'sub_sku'],
                not_for_selling: 0,
                limit: 10,
                with_sub_units: true,
                include_unit_details: true
            },
            success: function(products) {
                if (products && products.length > 0) {
                    // معالجة بيانات المنتج
                    products.forEach(function(product) {
                        processProductUnitsData(product);
                        processProductWarehouseData(product);
                    });
                    
                    var matchedProduct = null;
                    
                    // البحث عن تطابق دقيق للـ SKU
                    for (var i = 0; i < products.length; i++) {
                        var product = products[i];
                        if (product.sub_sku === item.sku || product.sku === item.sku) {
                            matchedProduct = product;
                            break;
                        }
                    }
                    
                    if (matchedProduct) {
                        // ملء الصف بالمنتج
                        populateRowWithProduct(emptyRow, matchedProduct, rowIndex);
                        
                        // تعيين الكمية والوحدة
                        setTimeout(function() {
                            emptyRow.find('.pos_quantity').val(item.quantity).trigger('change');
                            applyUnitToRow(emptyRow, item.unit);
                            resolve();
                        }, 200);
                        
                    } else {
                        reject(new Error('المنتج غير موجود'));
                    }
                } else {
                    reject(new Error('المنتج غير موجود'));
                }
            },
            error: function(xhr, status, error) {
                reject(new Error('خطأ في البحث'));
            }
        });
    });
}

// تبديل الإيقاف المؤقت
function togglePauseProcess() {
    isPaused = !isPaused;
    
    if (isPaused) {
        $('#pauseProcessBtn').html('<i class="fa fa-play"></i> استئناف').addClass('btn-resume');
        $('#processStatusIndicator').removeClass('processing').addClass('paused');
        showPasteStatus('تم إيقاف المعالجة مؤقتاً', 'info');
    } else {
        $('#pauseProcessBtn').html('<i class="fa fa-pause"></i> إيقاف مؤقت').removeClass('btn-resume');
        $('#processStatusIndicator').removeClass('paused').addClass('processing');
        showPasteStatus('تم استئناف المعالجة', 'info');
        processNextRow(); // استئناف المعالجة
    }
}

// إيقاف المعالجة نهائياً
function stopProcess() {
    if (confirm('هل أنت متأكد من إيقاف المعالجة نهائياً؟')) {
        shouldStop = true;
        isProcessing = false;
        isPaused = false;
        
        $('#processStatusIndicator').removeClass('processing paused').addClass('stopped');
        showPasteStatus('تم إيقاف المعالجة نهائياً', 'error');
        
        // إظهار زر المتابعة إذا كانت هناك صفوف متبقية
        var remainingRows = processedData.filter(item => item.isValid && !item.processed).length;
        if (remainingRows > 0) {
            $('#continueProcessBtn').show();
        }
        
        completeProcessing();
    }
}

// متابعة المعالجة
function continueProcessing() {
    // البحث عن أول صف غير معالج
    currentProcessIndex = processedData.findIndex(item => item.isValid && !item.processed);
    
    if (currentProcessIndex === -1) {
        showPasteStatus('لا توجد صفوف متبقية للمعالجة', 'info');
        return;
    }
    
    // إعادة تعيين الحالة
    isProcessing = true;
    isPaused = false;
    shouldStop = false;
    
    // تحديث واجهة المستخدم
    $('#continueProcessBtn').hide();
    $('#processControlButtons').css('display', 'flex');
    $('#processStatusIndicator').show().removeClass('stopped').addClass('processing');
    
    // استئناف المعالجة
    processNextRow();
}

// تحديث واجهة المعالجة
function updateProcessingUI(index, status, resultText) {
    var row = $(`#previewRow${index}`);
    
    // إزالة جميع الفئات السابقة
    row.removeClass('row-pending row-processing row-completed row-error');
    
    // إضافة الفئة الجديدة
    row.addClass(`row-${status}`);
    
    // تحديث نص الحالة
    if (status === 'processing') {
        row.find('.status-text').html('<i class="fa fa-spinner fa-spin"></i> جاري المعالجة...');
    }
    
    // تحديث نص النتيجة
    if (resultText) {
        row.find('.result-text').html(resultText);
    }
    
    // التمرير للصف الحالي
    var container = $('#pastePreview');
    var rowTop = row.position().top;
    var containerHeight = container.height();
    var scrollTop = container.scrollTop();
    
    if (rowTop < 0 || rowTop > containerHeight) {
        container.animate({
            scrollTop: scrollTop + rowTop - (containerHeight / 2)
        }, 200);
    }
}

// تحديث شريط التقدم
function updateProgress() {
    var total = processedData.filter(item => item.isValid).length;
    var processed = processedData.filter(item => item.isValid && item.processed).length;
    var percentage = total > 0 ? Math.round((processed / total) * 100) : 0;
    
    $('#progressFill').css('width', percentage + '%').text(percentage + '%');
    
    var currentItem = processedData[currentProcessIndex];
    if (currentItem) {
        $('#progressText').text(`معالجة ${currentItem.sku} (${processed + 1} من ${total})`);
    }
}

// تحديث إحصائيات الملخص
function updateSummaryStats() {
    var validData = processedData.filter(item => item.isValid);
    var successCount = validData.filter(item => item.status === 'success').length;
    var errorCount = validData.filter(item => item.status === 'error').length;
    var pendingCount = validData.filter(item => !item.processed).length;
    
    $('#successCount').text(successCount);
    $('#errorCount').text(errorCount);
    $('#pendingCount').text(pendingCount);
}

// إكمال المعالجة
function completeProcessing() {
    // تحديث واجهة المستخدم
    $('#processControlButtons').hide();
    $('#pauseProcessBtn').html('<i class="fa fa-pause"></i> إيقاف مؤقت').removeClass('btn-resume');
    
    // تحديث الإحصائيات النهائية
    updateSummaryStats();
    
    var validData = processedData.filter(item => item.isValid);
    var successCount = validData.filter(item => item.status === 'success').length;
    var errorCount = validData.filter(item => item.status === 'error').length;
    var totalCount = validData.length;
    
    // عرض رسالة الإكمال
    if (shouldStop) {
        showPasteStatus(`تم إيقاف المعالجة: ${successCount} نجح، ${errorCount} فشل من أصل ${totalCount}`, 'warning');
    } else {
        var message = `اكتملت المعالجة: ${successCount} نجح، ${errorCount} فشل من أصل ${totalCount}`;
        var statusType = errorCount > 0 ? 'warning' : 'success';
        showPasteStatus(message, statusType);
    }
     if ($('#status').length === 0) {
        $('form#add_pos_sell_form, form#edit_pos_sell_form')
            .append('<input type="hidden" name="status" id="status" value="final">');
    }
    
    // إزالة الصفوف الفارغة
     $('#pos_table tbody tr').each(function() {
        var row = $(this);
        var hasValidProduct = false;
        
        // التحقق من وجود معرف منتج صالح
        var productId = row.find('input[name*="[product_id]"]').val();
        var variationId = row.find('input[name*="[variation_id]"]').val();
        
        if (productId || variationId) {
            hasValidProduct = true;
        }
        
        // إزالة الصف إذا لم يكن صالحاً
        if (!hasValidProduct || row.hasClass('empty-row')) {
            row.remove();
        }
    });
    
    // التأكد من وجود حقل الحالة
    var forms = $('form#add_pos_sell_form, form#edit_pos_sell_form');
    forms.each(function() {
        var form = $(this);
        if (form.find('[name="status"]').length === 0) {
            form.append('<input type="hidden" name="status" value="final">');
        }
    });
    
    // إعادة تهيئة النموذج
    if (typeof pos_form_validator !== 'undefined') {
        pos_form_validator.resetForm();
    }
    
    // تحديث الأرقام والمجاميع
    updateSerialNumbers();
    pos_total_row();
    
    // إظهار زر الإغلاق
    $('#cancelPasteBtn').text('إغلاق');
     

}

// تطبيق الوحدة على الصف
function applyUnitToRow(row, unitName) {
    var unitInput = row.find('.unit-input');
    var availableUnits = [];
    
    try {
        availableUnits = JSON.parse(unitInput.attr('data-available-units') || '[]');
    } catch (e) {
        console.error('Error parsing units:', e);
        return;
    }
    
    // تحويل أسماء الوحدات الخاصة
    var unitMappings = {
        '½': '0.5',
        '¼': '0.25',
        '⅛': '0.125',
        '50غم': '0.05',
        '100غم': '0.1',
        '125غم': '0.125',
        'نصف': '0.5',
        'ربع': '0.25',
        'ثمن': '0.125'
    };
    
    var mappedUnit = unitMappings[unitName] || unitName;
    
    // البحث عن الوحدة المطابقة
    var matchedUnit = findMatchingUnit(availableUnits, mappedUnit);
    
    if (matchedUnit) {
        unitInput.val(matchedUnit.unit_name || matchedUnit.name);
        applySelectedUnit(row, matchedUnit);
        persistUnitValue(row);
    }
}

// البحث عن الوحدة المطابقة
function findMatchingUnit(units, searchUnit) {
    // البحث عن تطابق مباشر
    for (var i = 0; i < units.length; i++) {
        var unit = units[i];
        var unitName = (unit.unit_name || unit.name || '').toUpperCase();
        
        if (unitName === searchUnit.toUpperCase()) {
            return unit;
        }
        
        // البحث بالمضاعف
        if (parseFloat(unit.multiplier) === parseFloat(searchUnit)) {
            return unit;
        }
    }
    
    // إذا لم يجد، استخدم الوحدة الأساسية
    return units.find(u => u.is_base_unit == 1) || units[0];
}

// التأكد من وجود صف فارغ
function ensureEmptyRow() {
    if ($('#pos_table tbody tr.empty-row').length === 0) {
        addEmptyProductRow();
    }
}

// معالج اللصق المباشر في الجدول
function handleDirectTablePaste(e) {
    var clipboardData = e.originalEvent.clipboardData || window.clipboardData;
    var pastedData = clipboardData.getData('text/plain');
    
    if (!pastedData || !pastedData.trim()) {
        return;
    }
    
    // فحص ما إذا كانت البيانات تحتوي على أسطر متعددة
    if (pastedData.includes('\n') || pastedData.includes('\t')) {
        e.preventDefault();
        
        // إعادة تعيين الحالة
        resetProcessState();
        
        // عرض نافذة اللصق مع البيانات
        showExcelPasteModal();
        setTimeout(function() {
            $('#excelDataInput').val(pastedData);
            previewPastedData();
        }, 100);
    }
}

// عرض رسالة الحالة
function showPasteStatus(message, type) {
    var status = $('#pasteStatus');
    status.removeClass('success error warning info').addClass(type);
    status.html(message).show();
    
    if (type !== 'info') {
        setTimeout(function() {
            status.fadeOut();
        }, 5000);
    }
}

// دالة التأخير
function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

// اختصار لوحة المفاتيح Ctrl+Shift+V
$(document).on('keydown', function(e) {
    if (e.ctrlKey && e.shiftKey && e.key === 'V') {
        e.preventDefault();
        resetProcessState();
        showExcelPasteModal();
    }
});

// إضافة تلميح للمستخدم
$(document).ready(function() {
    setTimeout(function() {
        if ($('#excelPasteBtn').length > 0) {
            $('#excelPasteBtn').tooltip({
                title: 'اضغط Ctrl+Shift+V للصق من Excel',
                placement: 'bottom',
                trigger: 'hover'
            });
        }
    }, 2000);
});



// إضافة معالج إضافي لأزرار POS
$(document).on('click', '#pos-finalize, #pos-quotation, #pos-draft, .pos-express-finalize', function(e) {
    // إزالة أي صفوف فارغة
    $('#pos_table tbody tr').each(function() {
        var row = $(this);
        var productId = row.find('.product_id').val();
        var variationId = row.find('.variation_id').val();
        var quantity = parseFloat(row.find('.pos_quantity').val()) || 0;
        
        // إزالة الصف إذا كان فارغاً أو بدون كمية
        if ((!productId && !variationId) || quantity <= 0) {
            row.remove();
        }
    });
    
    // إعادة ترقيم الصفوف
    updateSerialNumbers();
    
    // تحديث المجاميع
    pos_total_row();
});


// إضافة معالج لضمان وجود حقل الحالة قبل الإرسال
$(document).ready(function() {
    // التأكد من وجود حقل الحالة في النموذج
    if ($('input[name="status"]').length === 0) {
        var statusField = '<input type="hidden" name="status" id="status" value="final">';
        $('form#add_pos_sell_form, form#edit_pos_sell_form').append(statusField);
    }
});

// تحديث حقل الحالة عند الضغط على أزرار مختلفة
$(document).on('click', '#pos-finalize, .pos-express-finalize', function() {
    $('#status').val('final');
});

$(document).on('click', '#pos-draft', function() {
    $('#status').val('draft');
});

$(document).on('click', '#pos-quotation', function() {
    $('#status').val('quotation');
});

// إضافة معالج للتحقق من البيانات قبل الإرسال
function validatePOSFormBeforeSubmit() {
    // التأكد من وجود حقل الحالة
    if (!$('input[name="status"]').val()) {
        $('input[name="status"]').val('final');
    }
    
    // إزالة الصفوف الفارغة
    $('#pos_table tbody tr').each(function() {
        var row = $(this);
        var hasProduct = row.find('.product_id').val() || 
                        row.find('.variation_id').val();
        var quantity = parseFloat(row.find('.pos_quantity').val()) || 0;
        
        if (!hasProduct || quantity <= 0 || row.hasClass('empty-row')) {
            row.remove();
        }
    });
    
    // إعادة ترقيم الصفوف
    updateSerialNumbers();
    
    return true;
}

// تعديل معالج إرسال النموذج
$(document).ready(function() {
    // Wait for pos_form_obj to be initialized
    if (typeof pos_form_obj !== 'undefined' && pos_form_obj) {
        var originalPosFormSubmit = pos_form_obj.submit;
        pos_form_obj.submit = function(e) {
            // التحقق من البيانات قبل الإرسال
            if (!validatePOSFormBeforeSubmit()) {
        e.preventDefault();
        return false;
    }
    
    // التأكد من وجود حقل الحالة
    var formData = $(this).serializeArray();
    var hasStatus = formData.some(field => field.name === 'status');
    
    if (!hasStatus) {
        $(this).append('<input type="hidden" name="status" value="final">');
    }
    
    // استدعاء الدالة الأصلية
    if (originalPosFormSubmit) {
        return originalPosFormSubmit.apply(this, arguments);
    }
};
    }
});



// إصلاح مشكلة حقل الحالة المفقود
$(document).ready(function() {
    // 1. التأكد من وجود حقل الحالة في النموذج
    function ensureStatusField() {
        var forms = ['#add_pos_sell_form', '#edit_pos_sell_form'];
        forms.forEach(function(formSelector) {
            var form = $(formSelector);
            if (form.length > 0) {
                // البحث عن حقل الحالة
                var statusField = form.find('input[name="status"], select[name="status"]');
                
                if (statusField.length === 0) {
                    // إضافة حقل مخفي للحالة
                    form.prepend('<input type="hidden" name="status" id="hidden_status" value="final">');
                }
            }
        });
    }
    
    // تنفيذ عند التحميل
    ensureStatusField();
    
    // 2. تعديل معالجات الأزرار لتحديث حقل الحالة
    
    // زر البيع النهائي
    $(document).on('click', '#pos-finalize', function() {
        updateStatusField('final');
    });
    
    // أزرار البيع السريع
    $(document).on('click', '.pos-express-finalize', function() {
        updateStatusField('final');
    });
    
    // زر المسودة
    $(document).on('click', '#pos-draft', function() {
        updateStatusField('draft');
    });
    
    // زر عرض السعر
    $(document).on('click', '#pos-quotation', function() {
        updateStatusField('quotation');
    });
    
    // 3. دالة تحديث حقل الحالة
    function updateStatusField(status) {
        // تحديث جميع حقول الحالة الموجودة
        $('input[name="status"], select[name="status"], #hidden_status').val(status);
        
        // إذا لم يوجد أي حقل، أضف واحد
        if ($('[name="status"]').length === 0) {
            var activeForm = $('form#add_pos_sell_form').length > 0 ? 
                           $('form#add_pos_sell_form') : 
                           $('form#edit_pos_sell_form');
            
            if (activeForm.length > 0) {
                activeForm.prepend('<input type="hidden" name="status" value="' + status + '">');
            }
        }
        
        console.log('Status updated to:', status);
    }
    
    // 4. اعتراض إرسال النموذج للتحقق من البيانات
    $('form#add_pos_sell_form, form#edit_pos_sell_form').on('submit', function(e) {
        var form = $(this);
        
        // التحقق من وجود حقل الحالة
        var statusExists = form.find('[name="status"]').length > 0;
        if (!statusExists) {
            // إيقاف الإرسال
            e.preventDefault();
            
            // إضافة حقل الحالة
            form.append('<input type="hidden" name="status" value="final">');
            
            // إعادة الإرسال
            setTimeout(function() {
                form.off('submit').submit();
            }, 100);
            
            return false;
        }
        
        // إزالة الصفوف الفارغة
        cleanEmptyRows();
        
        // التحقق من وجود منتجات
        if ($('#pos_table tbody tr:not(.empty-row)').length === 0) {
            e.preventDefault();
            toastr.error('لا توجد منتجات في الفاتورة');
            return false;
        }
        
        console.log('Form data being submitted:', form.serialize());
    });
    
    // 5. دالة تنظيف الصفوف الفارغة
    function cleanEmptyRows() {
        $('#pos_table tbody tr').each(function() {
            var row = $(this);
            var productId = row.find('[name*="[product_id]"]').val();
            var variationId = row.find('[name*="[variation_id]"]').val();
            var quantity = parseFloat(row.find('[name*="[quantity]"]').val()) || 0;
            
            // إزالة الصف إذا كان فارغاً
            if ((!productId && !variationId) || quantity <= 0) {
                row.remove();
            }
        });
        
        // إعادة ترقيم الصفوف
        $('#pos_table tbody tr').each(function(index) {
            var row = $(this);
            // تحديث جميع أسماء الحقول
            row.find('input, select, textarea').each(function() {
                var name = $(this).attr('name');
                if (name) {
                    var newName = name.replace(/\[\d+\]/, '[' + index + ']');
                    $(this).attr('name', newName);
                }
            });
        });
    }
});

// تعديل معالج إرسال النموذج الرئيسي
if (typeof pos_form_validator !== 'undefined') {
    var originalSubmitHandler = pos_form_validator.settings.submitHandler;
    
    pos_form_validator.settings.submitHandler = function(form) {
        // التحقق من حقل الحالة قبل الإرسال
        var $form = $(form);
        var data = $form.serialize();
        
        // إضافة الحالة إذا لم تكن موجودة
        if (data.indexOf('status=') === -1) {
            var status = 'final';
            
            // تحديد الحالة بناءً على الزر المضغوط
            if ($('#is_suspend').val() == '1') {
                status = 'suspend';
            } else if ($form.find('[name="is_quotation"]').val() == '1') {
                status = 'quotation';
            } else if ($form.find('[name="is_draft"]').val() == '1') {
                status = 'draft';
            }
            
            data += '&status=' + status;
        }
        
        // إزالة الصفوف الفارغة
        $('#pos_table tbody tr.empty-row').remove();
        
        // استدعاء المعالج الأصلي
        if (originalSubmitHandler) {
            return originalSubmitHandler.call(this, form);
        } else {
            form.submit();
        }
    };
}


async function processDataRow(item) {
    return new Promise((resolve, reject) => {
        // إضافة صف فارغ إذا لزم الأمر
        ensureEmptyRow();
        
        // الحصول على الصف الفارغ
        var emptyRow = $('#pos_table tbody tr.empty-row').first();
        
        if (emptyRow.length === 0) {
            addEmptyProductRow();
            emptyRow = $('#pos_table tbody tr.empty-row').first();
        }
        
        var rowIndex = emptyRow.data('row_index') || emptyRow.index();
        
        // البحث عن المنتج باستخدام SKU
        $.ajax({
            url: base_path + '/products/list',
            method: 'GET',
            dataType: 'json',
            data: {
                term: item.sku,
                search_fields: ['sku', 'sub_sku'],
                not_for_selling: 0,
                limit: 10,
                with_sub_units: true,
                include_unit_details: true
            },
            success: function(products) {
                if (products && products.length > 0) {
                    // معالجة بيانات المنتج
                    products.forEach(function(product) {
                        processProductUnitsData(product);
                        processProductWarehouseData(product);
                        
                        // حفظ بيانات المنتج لنظام التسعير المتقدم
                        if (typeof storeProductData === 'function') {
                            storeProductData(product);
                        }
                    });
                    
                    var matchedProduct = null;
                    
                    // البحث عن تطابق دقيق للـ SKU
                    for (var i = 0; i < products.length; i++) {
                        var product = products[i];
                        if (product.sub_sku === item.sku || product.sku === item.sku) {
                            matchedProduct = product;
                            break;
                        }
                    }
                    
                    if (matchedProduct) {
                        // حفظ بيانات المنتج في الصف
                        emptyRow.data('product-data', matchedProduct);
                        
                        // ملء الصف بالمنتج
                        populateRowWithProduct(emptyRow, matchedProduct, rowIndex);
                        
                        // تعيين الكمية والوحدة
                        setTimeout(function() {
                            // تعيين الكمية
                            emptyRow.find('.pos_quantity').val(item.quantity).trigger('change');
                            
                            // تطبيق الوحدة
                            applyUnitToRowEnhanced(emptyRow, item.unit);
                            
                            // تطبيق نظام التسعير المتقدم
                            if (typeof applyAdvancedPricingToRow === 'function') {
                                setTimeout(function() {
                                    applyAdvancedPricingToRow(emptyRow);
                                    
                                    // التحقق من وجود أخطاء في الوحدة
                                    // if (emptyRow.hasClass('unit-error-row')) {
                                    //     reject(new Error('خطأ في الوحدة - لا يوجد سعر للوحدة المطلوبة'));
                                    //     return;
                                    // }
                                    
                                    resolve();
                                }, 300);
                            } else {
                                resolve();
                            }
                        }, 200);
                        
                    } else {
                        reject(new Error('المنتج غير موجود'));
                    }
                } else {
                    reject(new Error('المنتج غير موجود'));
                }
            },
            error: function(xhr, status, error) {
                reject(new Error('خطأ في البحث'));
            }
        });
    });
}

// تحسين دالة تطبيق الوحدة على الصف
function applyUnitToRowEnhanced(row, unitName) {
    var unitInput = row.find('.unit-input');
    var availableUnits = [];
    
    try {
        availableUnits = JSON.parse(unitInput.attr('data-available-units') || '[]');
    } catch (e) {
        console.error('Error parsing units:', e);
        return;
    }
    
    // تحويل أسماء الوحدات الخاصة
    var unitMappings = {
        '½': '0.5',
        '¼': '0.25',
        '⅛': '0.125',
        '50غم': '50غم',
        '100غم': '100غم',
        '125غم': '125غم',
        'نصف': '0.5',
        'ربع': '0.25',
        'ثمن': '0.125',
        'باكيت': 'باكيت',
        'باكت': 'باكيت',
        'packet': 'باكيت',
        'كارتون': 'كارتون',
        'كرتون': 'كارتون',
        'درزن': 'درزن',
        'دزن': 'درزن',
    //    'كغم': 'كغم',
        'كيلو': 'كغم',
        'قطعة': 'قطعة',
        'قطعه': 'قطعة'
    };
    
    var mappedUnit = unitMappings[unitName] || unitName;
    
    // تطبيق الوحدة مباشرة
    unitInput.val(mappedUnit);
    
    // حفظ قيمة الوحدة
    if (typeof persistUnitValue === 'function') {
        persistUnitValue(row);
    }
    
    // البحث عن الوحدة المطابقة
    var matchedUnit = findMatchingUnit(availableUnits, mappedUnit);
    
    if (matchedUnit) {
        // تطبيق الوحدة المحددة
        applySelectedUnit(row, matchedUnit);
        
        // تحديث بيانات الإرسال
        if (typeof updateUnitSubmissionData === 'function') {
            updateUnitSubmissionData(row, matchedUnit);
        }
    }
}

// تحديث دالة معالجة الصف التالي لعرض المزيد من تفاصيل الأخطاء
async function processNextRowEnhanced() {
    if (shouldStop || currentProcessIndex >= processedData.length) {
        completeProcessing();
        return;
    }
    
    if (isPaused) {
        showPasteStatus('المعالجة متوقفة مؤقتاً', 'info');
        return;
    }
    
    var item = processedData[currentProcessIndex];
    
    // تخطي الصفوف غير الصالحة أو المعالجة مسبقاً
    if (!item.isValid || item.processed) {
        currentProcessIndex++;
        processNextRowEnhanced();
        return;
    }
    
    // تحديث واجهة المستخدم
    updateProcessingUI(currentProcessIndex, 'processing');
    updateProgress();
    
    try {
        // معالجة الصف
        await processDataRow(item);
        
        // تحديث الحالة
        item.processed = true;
        item.status = 'success';
        updateProcessingUI(currentProcessIndex, 'completed', '✓ تمت الإضافة');
        
    } catch (error) {
        // console.error('Error processing row:', error);
        // item.processed = true;
        // item.status = 'error';
        // item.error = error.message || 'خطأ غير معروف';
        
        // // تفصيل أكثر للأخطاء
        // var errorMessage = '✗ ' + item.error;
        // if (item.error.includes('خطأ في الوحدة')) {
        //     errorMessage = '✗ الوحدة "' + item.unit + '" غير متوفرة أو ليس لها سعر';
        // }
        
        // updateProcessingUI(currentProcessIndex, 'error', errorMessage);
    }
    
    // تحديث الإحصائيات
    updateSummaryStats();
    
    // الانتقال للصف التالي
    currentProcessIndex++;
    
    // تأخير صغير ثم معالجة الصف التالي
    setTimeout(() => {
        processNextRowEnhanced();
    }, 300);
}

// استبدال processNextRow بـ processNextRowEnhanced في دالة startProcessing
function startProcessingEnhanced() {
    var validData = processedData.filter(item => item.isValid);
    
    if (validData.length === 0) {
        showPasteStatus('لا توجد بيانات صالحة للمعالجة', 'error');
        return;
    }
    
    // تعيين حالة المعالجة
    isProcessing = true;
    isPaused = false;
    shouldStop = false;
    currentProcessIndex = 0;
    
    // تحديث واجهة المستخدم
    $('#processPasteBtn').hide();
    $('#previewPasteBtn').hide();
    $('#excelDataInput').prop('readonly', true);
    $('#processControlButtons').css('display', 'flex');
    $('#pasteProgress').show();
    $('#processSummary').show();
    $('#processStatusIndicator').show().addClass('processing');
    
    // بدء المعالجة باستخدام الدالة المحسنة
    processNextRowEnhanced();
}

// تحديث دالة إكمال المعالجة
function completeProcessingEnhanced() {
    // تحديث واجهة المستخدم
    $('#processControlButtons').hide();
    $('#pauseProcessBtn').html('<i class="fa fa-pause"></i> إيقاف مؤقت').removeClass('btn-resume');
    
    // تحديث الإحصائيات النهائية
    updateSummaryStats();
    
    var validData = processedData.filter(item => item.isValid);
    var successCount = validData.filter(item => item.status === 'success').length;
    var errorCount = validData.filter(item => item.status === 'error').length;
    var totalCount = validData.length;
    
    // عرض رسالة الإكمال
    if (shouldStop) {
        showPasteStatus(`تم إيقاف المعالجة: ${successCount} نجح، ${errorCount} فشل من أصل ${totalCount}`, 'warning');
    } else {
        var message = `اكتملت المعالجة: ${successCount} نجح، ${errorCount} فشل من أصل ${totalCount}`;
        var statusType = errorCount > 0 ? 'warning' : 'success';
        showPasteStatus(message, statusType);
    }
    
    // التأكد من وجود حقل الحالة
    if ($('#status').length === 0) {
        $('form#add_pos_sell_form, form#edit_pos_sell_form')
            .append('<input type="hidden" name="status" id="status" value="final">');
    }
    
    // إزالة الصفوف الفارغة والصفوف التي بها أخطاء
    $('#pos_table tbody tr').each(function() {
        var row = $(this);
        var hasValidProduct = false;
        
        // التحقق من وجود معرف منتج صالح
        var productId = row.find('input[name*="[product_id]"]').val();
        var variationId = row.find('input[name*="[variation_id]"]').val();
        
        if (productId || variationId) {
            hasValidProduct = true;
        }
        
        // إزالة الصف إذا لم يكن صالحاً أو إذا كان به خطأ في الوحدة
        if (!hasValidProduct || row.hasClass('empty-row') || row.hasClass('unit-error-row')) {
            // إذا كان به خطأ في الوحدة، أضف تنبيه
            if (row.hasClass('unit-error-row')) {
                var sku = row.find('.product-info').text().match(/\b[A-Z]\d+\b/);
                if (sku) {
                    console.warn('تم حذف المنتج ' + sku[0] + ' بسبب خطأ في الوحدة');
                }
            }
            row.remove();
        }
    });
    
    // مسح جميع الأخطاء المرئية
    if (typeof clearAllErrors === 'function') {
        clearAllErrors();
    }
    
    // التأكد من وجود حقل الحالة
    var forms = $('form#add_pos_sell_form, form#edit_pos_sell_form');
    forms.each(function() {
        var form = $(this);
        if (form.find('[name="status"]').length === 0) {
            form.append('<input type="hidden" name="status" value="final">');
        }
    });
    
    // إعادة تهيئة النموذج
    if (typeof pos_form_validator !== 'undefined') {
        pos_form_validator.resetForm();
    }
    
    // تحديث الأرقام والمجاميع
    updateSerialNumbers();
    pos_total_row();
    
    // إظهار زر الإغلاق
    $('#cancelPasteBtn').text('إغلاق');
}

// تحديث معالجات الأحداث لاستخدام الدوال المحسنة
function setupModalEventHandlersEnhanced() {
    // معالج الإلغاء
    $('#cancelPasteBtn, #excelPasteModal').on('click', function(e) {
        if (e.target === this || $(e.target).attr('id') === 'cancelPasteBtn') {
            if (isProcessing && !shouldStop) {
                if (confirm('هل أنت متأكد من إيقاف المعالجة وإغلاق النافذة؟')) {
                    shouldStop = true;
                    setTimeout(() => {
                        $('#excelPasteModal').remove();
                    }, 500);
                }
            } else {
                $('#excelPasteModal').remove();
            }
        }
    });
    
    // معالج المعاينة
    $('#previewPasteBtn').on('click', function() {
        previewPastedData();
    });
    
    // معالج بدء المعالجة - استخدام الدالة المحسنة
    $('#processPasteBtn').on('click', function() {
        startProcessingEnhanced();
    });
    
    // معالج متابعة المعالجة
    $('#continueProcessBtn').on('click', function() {
        continueProcessingEnhanced();
    });
    
    // معالج الإيقاف المؤقت/الاستئناف
    $('#pauseProcessBtn').on('click', function() {
        togglePauseProcess();
    });
    
    // معالج الإيقاف النهائي
    $('#stopProcessBtn').on('click', function() {
        stopProcess();
    });
    
    // معالج اللصق في منطقة النص
    $('#excelDataInput').on('paste', function(e) {
        setTimeout(function() {
            previewPastedData();
        }, 100);
    });
    
    // معالج تغيير النص
    $('#excelDataInput').on('input', function() {
        if ($(this).val().trim()) {
            $('#previewPasteBtn').prop('disabled', false);
        } else {
            $('#previewPasteBtn').prop('disabled', true);
            $('#pastePreview').hide();
            $('#processPasteBtn').hide();
        }
    });
}

// دالة متابعة المعالجة المحسنة
function continueProcessingEnhanced() {
    // البحث عن أول صف غير معالج
    currentProcessIndex = processedData.findIndex(item => item.isValid && !item.processed);
    
    if (currentProcessIndex === -1) {
        showPasteStatus('لا توجد صفوف متبقية للمعالجة', 'info');
        return;
    }
    
    // إعادة تعيين الحالة
    isProcessing = true;
    isPaused = false;
    shouldStop = false;
    
    // تحديث واجهة المستخدم
    $('#continueProcessBtn').hide();
    $('#processControlButtons').css('display', 'flex');
    $('#processStatusIndicator').show().removeClass('stopped').addClass('processing');
    
    // استئناف المعالجة باستخدام الدالة المحسنة
    processNextRowEnhanced();
}

// استبدال setupModalEventHandlers بـ setupModalEventHandlersEnhanced
// واستبدال completeProcessing بـ completeProcessingEnhanced
// في دالة showExcelPasteModal

// تحديث دالة showExcelPasteModal لاستخدام الدوال المحسنة
$(document).ready(function() {
    // استبدال الدوال الأصلية بالدوال المحسنة
    if (typeof processNextRow !== 'undefined') {
        window.processNextRow = processNextRowEnhanced;
    }
    
    if (typeof startProcessing !== 'undefined') {
        window.startProcessing = startProcessingEnhanced;
    }
    
    if (typeof completeProcessing !== 'undefined') {
        window.completeProcessing = completeProcessingEnhanced;
    }
    
    if (typeof setupModalEventHandlers !== 'undefined') {
        window.setupModalEventHandlers = setupModalEventHandlersEnhanced;
    }
    
    if (typeof continueProcessing !== 'undefined') {
        window.continueProcessing = continueProcessingEnhanced;
    }
    
    if (typeof applyUnitToRow !== 'undefined') {
        window.applyUnitToRow = applyUnitToRowEnhanced;
    }
    
    console.log('✅ Excel integration with Advanced Pricing System V4.2 initialized');
});



// ================================================
// نظام التسعير المتقدم - النسخة المحدثة V4.4
// دعم الوحدة الأساسية (درزن، كارتون، قطعة)
// ================================================
(function() {
    'use strict';
    
    console.log('🚀 =================================');
    console.log('🚀 ADVANCED PRICING SYSTEM V4.4');
    console.log('🚀 Base Unit Priority Support');
    console.log('🚀 =================================');
    
    // ==========================================
    // تحديث دالة تحديد السعر حسب الوحدة
    // ==========================================
    
    window.determinePriceByUnit = function(product, unitName, row) {
        logPricing('=== Determining price for unit (V4.4): ' + unitName + ' ===');
        
        if (!product) {
            logPricing('No product data provided', null, 'warn');
            return null;
        }
        
        // ============================================
        // تعامل خاص مع وحدة السيت: استخدم السعر الأساسي مباشرة
        // ============================================
        var isSetUnit = unitName.toLowerCase().includes('سيت') || unitName.toLowerCase().includes('set');
        if (isSetUnit) {
            logPricing('🎯 Set unit detected - using base selling price directly');
            
            // الحصول على السعر الأساسي للبيع من المنتج
            var baseSellPrice = parseFloat(product.selling_price || product.default_sell_price || 0);
            
            if (baseSellPrice > 0) {
                return {
                    price: baseSellPrice,
                    shouldUpdateUnit: false,
                    isMainUnit: true,
                    unitType: 'Set Unit (Base Price)'
                };
            }
        }
        
        // استخراج الأسعار
        var priceData = extractPricesFromCustomField3(product.product_custom_field3);
        
        if (!priceData) {
            logPricing('No price data found', null, 'warn');
            return null;
        }
        
        // تحديد الوحدة الأساسية للمنتج
        var productMainUnit = getProductMainUnit(product);
        logPricing('Product main unit:', productMainUnit);
        
                // ============================================
        // 1. معالجة خاصة للوحدات الخاصة كوحدة رئيسية (أولوية عالية)
        // ============================================
        if (productMainUnit && isSpecialBaseUnit(unitName) && isUnitMatch(unitName, productMainUnit.name)) {
            var unitType = '';
            if (isLakUnit(unitName)) unitType = 'LAK';
            else if (isFullPlasticUnit(unitName)) unitType = 'Full Plastic';
            else if (isDozenUnit(unitName)) unitType = 'Dozen';
            
            logPricing('🎯 Special unit is main unit - using main price directly for ' + unitType);
            
            if (priceData.mainPrice && priceData.mainPrice > 0) {
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: false,
                    isMainUnit: true,
                    isSpecialUnit: true,
                    unitType: unitType + ' (Main Unit - Priority)'
                };
            }
        }

        // ============================================
        // 2. إذا كانت الوحدة المطلوبة هي نفس الوحدة الأساسية للمنتج (معالجة عامة)
        // ============================================
        if (productMainUnit && isUnitMatch(unitName, productMainUnit.name)) {
            logPricing('Requested unit matches main unit - using main price');
            
            if (priceData.mainPrice && priceData.mainPrice > 0) {
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: false,
                    isMainUnit: true,
                    unitType: 'Main Unit: ' + productMainUnit.name
                };
            }
        }

        // ============================================
        // 3. معالجة خاصة للوحدات الخاصة (في حالة عدم كونها الوحدة الرئيسية)
        // ============================================
        if (isSpecialBaseUnit(unitName)) {
            var unitType = '';
            if (isLakUnit(unitName)) unitType = 'LAK';
            else if (isFullPlasticUnit(unitName)) unitType = 'Full Plastic';
            else if (isDozenUnit(unitName)) unitType = 'Dozen';
            
            logPricing('Special handling for ' + unitType + ' unit (not main unit)');
            
            // للوحدات الخاصة، استخدم السعر الأساسي إذا توفر
            if (priceData.mainPrice && priceData.mainPrice > 0) {
                logPricing('Using main price for special unit: ' + priceData.mainPrice);
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: false,
                    isSpecialUnit: true,
                    unitType: unitType + ' (Special Unit)'
                };
            }
        }
        
        // ============================================
        // 3. معالجة سيناريو الدرزن مع الكارتون
        // ============================================
        var dozenCode = 17; // كود الدرزن في UNIT_MAPPING
        var cartonCode = 18; // كود الكارتون
        
        var hasDozenAsMainUnit = productMainUnit && (
            isDozenUnit(productMainUnit.name) || 
            productMainUnit.id === dozenCode
        );
        
        if (hasDozenAsMainUnit && isCartonUnit(unitName)) {
            logPricing('Main unit is Dozen, checking for Carton availability');
            
            // التحقق من وجود وحدة كارتون في أسعار المنتج
            var hasCartonPrice = priceData.unitPrices[cartonCode] && priceData.unitPrices[cartonCode] > 0;
            
            if (!hasCartonPrice) {
                logPricing('Main unit is Dozen and Carton price not available - using Dozen price');
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: true,
                    baseUnit: { name: 'درزن', id: dozenCode, multiplier: 1 },
                    message: 'الوحدة الرئيسية هي درزن - لا يوجد سعر للكارتون'
                };
            } else {
                logPricing('Found Carton price for Dozen-based product: ' + priceData.unitPrices[cartonCode]);
                return {
                    price: priceData.unitPrices[cartonCode],
                    shouldUpdateUnit: false
                };
            }
        }
        
        // ============================================
        // 4. معالجة سيناريو الكارتون كوحدة أساسية - PRIORITY للمصفوفة
        // ============================================
        var hasCartonAsMainUnit = productMainUnit && (
            isCartonUnit(productMainUnit.name) || 
            productMainUnit.id === cartonCode
        );
        
        if (hasCartonAsMainUnit) {
            logPricing('Product has Carton as main unit - checking price array first');
            
            if (isCartonUnit(unitName)) {
                // المطلوب كارتون والوحدة الأساسية كارتون
                // البحث أولاً في مصفوفة الأسعار للكارتون
                var cartonPriceFromArray = priceData.unitPrices[cartonCode];
                
                if (cartonPriceFromArray && cartonPriceFromArray > 0) {
                    logPricing('🎯 Using Carton price from UoMPrices array: ' + cartonPriceFromArray);
                    return {
                        price: cartonPriceFromArray,
                        shouldUpdateUnit: false,
                        source: 'unit_prices_array',
                        unitType: 'Carton from Array'
                    };
                } else {
                    // إذا لم يوجد في المصفوفة، استخدم السعر الأساسي
                    logPricing('🔄 No Carton in array - using main price: ' + priceData.mainPrice);
                    return {
                        price: priceData.mainPrice,
                        shouldUpdateUnit: false,
                        isMainUnit: true,
                        unitType: 'Main Unit: Carton'
                    };
                }
            } else if (isDozenUnit(unitName)) {
                // المطلوب درزن والوحدة الأساسية كارتون - ابحث في مصفوفة الأسعار
                var dozenPrice = priceData.unitPrices[dozenCode];
                if (dozenPrice && dozenPrice > 0) {
                    logPricing('🎯 Found Dozen price in unit prices array: ' + dozenPrice);
                    return {
                        price: dozenPrice,
                        shouldUpdateUnit: false,
                        source: 'unit_prices_array'
                    };
                } else {
                    logPricing('❌ No Dozen price found in array for Carton-based product');
                    return {
                        error: true,
                        message: 'لم يتم العثور على سعر الدرزن لهذا المنتج في مصفوفة الأسعار',
                        requestedUnit: unitName,
                        mainUnit: productMainUnit.name
                    };
                }
            }
        }
        
        // ============================================
        // 5. معالجة سيناريو القطعة كوحدة أساسية
        // ============================================
        var pieceVariations = ['قطعة', 'قطعه', 'EA', 'PCS', 'piece'];
        var hasPieceAsMainUnit = productMainUnit && pieceVariations.some(variation => 
            productMainUnit.name.toLowerCase().includes(variation.toLowerCase())
        );
        
        if (hasPieceAsMainUnit) {
            logPricing('Product has Piece as main unit');
            
            if (isPieceUnit(unitName)) {
                // المطلوب قطعة والوحدة الأساسية قطعة - استخدم السعر الأساسي
                logPricing('Requested Piece matches main unit - using main price');
                return {
                    price: priceData.mainPrice,
                    shouldUpdateUnit: false,
                    isMainUnit: true,
                    unitType: 'Main Unit: Piece'
                };
            }
        }
        
        // ============================================
        // 6. البحث في مصفوفة الأسعار للوحدات الأخرى
        // ============================================
        
        // التحقق من وجود وحدة الباكيت في المنتج
        var packetCode = 20; // كود الباكيت في UNIT_MAPPING
        var hasPacketUnit = false;
        
        if (priceData.unitPrices[packetCode] && priceData.unitPrices[packetCode] > 0) {
            hasPacketUnit = true;
            logPricing('Product has explicit packet price');
        } else if (product.units && Array.isArray(product.units)) {
            for (var i = 0; i < product.units.length; i++) {
                if (isPacketUnit(product.units[i].name) || product.units[i].id === packetCode) {
                    hasPacketUnit = true;
                    logPricing('Product has packet unit in units list');
                    break;
                }
            }
        }
        
        // البحث عن أفضل مطابقة للوحدة في جدول UNIT_MAPPING
        var bestMatch = null;
        var bestScore = 0;
        var threshold = 80;
        
        // المرحلة 1: البحث عن المطابقة الكاملة أولاً
        for (var unitKey in window.UNIT_MAPPING) {
            if (unitName.toLowerCase().trim() === unitKey.toLowerCase().trim()) {
                bestMatch = unitKey;
                bestScore = 100;
                logPricing('🎯 Exact unit match found:', unitKey);
                break;
            }
        }
        
        // المرحلة 2: إذا لم نجد مطابقة كاملة، ابحث عن أفضل مطابقة تقريبية
        if (!bestMatch) {
            for (var unitKey in window.UNIT_MAPPING) {
                // تجنب مطابقة الوحدات التي تحتوي على أرقام مع وحدات بدون أرقام
                var inputHasNumbers = /\d/.test(unitName);
                var keyHasNumbers = /\d/.test(unitKey);
                
                if (inputHasNumbers !== keyHasNumbers) {
                    // تخط هذه المطابقة لتجنب خلط "0.25 كغم" مع "كغم"
                    logPricing('⚠️ Skipping mixed match:', unitName + ' vs ' + unitKey);
                    continue;
                }
                
                var score = calculateSimilarity(unitName, unitKey);
                if (score > bestScore && score >= threshold) {
                    bestScore = score;
                    bestMatch = unitKey;
                }
            }
        }
        
        // إذا وجدنا مطابقة وكان لها سعر
        if (bestMatch) {
            var unitCode = window.UNIT_MAPPING[bestMatch];
            logPricing('Best unit mapping match:', {
                input: unitName,
                matched: bestMatch,
                code: unitCode,
                score: bestScore + '%'
            });
            
            if (priceData.unitPrices[unitCode]) {
                logPricing('✅ Found specific price for unit code ' + unitCode + ' (' + bestMatch + '): ' + priceData.unitPrices[unitCode]);
                return {
                    price: priceData.unitPrices[unitCode],
                    shouldUpdateUnit: false
                };
            } else {
                logPricing('⚠️ Unit code ' + unitCode + ' found but no price available in unitPrices');
            }
        }
        
        // ============================================
        // 7. معالجة عدم وجود سعر للوحدة المطلوبة
        // ============================================
        
        // إذا كان المنتج يحتوي على وحدة باكيت ولم نجد سعر للوحدة المطلوبة
        if (hasPacketUnit && !priceData.unitPrices[unitCode]) {
            var hasExplicitPacketPrice = priceData.unitPrices[packetCode] && priceData.unitPrices[packetCode] > 0;
            var packetPrice = hasExplicitPacketPrice ? priceData.unitPrices[packetCode] : priceData.mainPrice;
            
            logPricing('Product has packet unit, returning packet price with update flag');
            
            return {
                price: packetPrice,
                shouldUpdateUnit: !isPacketUnit(unitName),
                baseUnit: { name: 'باكيت', id: packetCode, multiplier: 1 }
            };
        }
        
        // إذا لم يكن المنتج يحتوي على وحدة باكيت ولم نجد سعر للوحدة المطلوبة
        if (!hasPacketUnit && !priceData.unitPrices[unitCode]) {
            logPricing('No matching unit price found and product does not have packet unit', null, 'error');
            return {
                error: false,
          //      message: 'لم يتم العثور على سعر للوحدة المطلوبة: ' + unitName,
                requestedUnit: unitName,
                mainUnit: productMainUnit ? productMainUnit.name : 'غير محدد',
                availableUnits: Object.keys(priceData.unitPrices).map(function(code) {
                    for (var unit in window.UNIT_MAPPING) {
                        if (window.UNIT_MAPPING[unit] == code) {
                            return unit + ' (كود: ' + code + ')';
                        }
                    }
                    return 'كود: ' + code;
                })
            };
        }
        
        // إذا كان لدينا سعر أساسي فقط
        if (priceData.mainPrice && !Object.keys(priceData.unitPrices).length) {
            return {
                price: priceData.mainPrice,
                shouldUpdateUnit: false
            };
        }
        
        logPricing('No price found for unit: ' + unitName, null, 'warn');
        return null;
    };
    
    // ==========================================
    // دوال مساعدة جديدة
    // ==========================================
    
    // دالة للتحقق من تطابق الوحدات - محسنة لمنع الخلط بين الوحدات المتشابهة
    window.isUnitMatch = function(unitName1, unitName2) {
        if (!unitName1 || !unitName2) return false;
        
        var normalized1 = unitName1.toLowerCase().trim();
        var normalized2 = unitName2.toLowerCase().trim();
        
        // تطابق مباشر
        if (normalized1 === normalized2) return true;
        
        // حماية خاصة للوحدات التي تحتوي على أرقام
        // منع مطابقة "0.25 كغم" مع "كغم" مثلاً
        var hasNumbers1 = /\d/.test(normalized1);
        var hasNumbers2 = /\d/.test(normalized2);
        
        if (hasNumbers1 || hasNumbers2) {
            // إذا كانت إحدى الوحدتين تحتوي على أرقام، اشترط المطابقة الكاملة فقط
            return normalized1 === normalized2;
        }
        
        // تطابق تقريبي للوحدات المشابهة (للوحدات بدون أرقام فقط)
        var similarityScore = calculateSimilarity(normalized1, normalized2);
        return similarityScore >= 85; // 85% تشابه أو أكثر
    };
    
    // دالة للتحقق من وحدة القطعة
    window.isPieceUnit = function(unitName) {
        if (!unitName) return false;
        
        var pieceVariations = [
            'قطعة', 'قطعه', 'قطع', 
            'EA', 'ea', 'PCS', 'pcs', 
            'piece', 'pieces'
        ];
        
        var normalizedUnit = unitName.toLowerCase().trim();
        
        return pieceVariations.some(variation => 
            normalizedUnit === variation.toLowerCase() ||
            normalizedUnit.includes(variation.toLowerCase())
        );
    };
    
    // تحديث دالة getProductMainUnit لاستخراج من product_custom_field2
    window.getProductMainUnit = function(product) {
        logPricing('Getting main unit for product...');
        
        // 1. الأولوية الأولى: استخراج من product_custom_field2
        if (product.product_custom_field2) {
            var mainUnitName = extractMainUnitFromCustomField2(product.product_custom_field2);
            if (mainUnitName) {
                logPricing('Found main unit in custom_field2:', mainUnitName);
                return {
                    name: mainUnitName,
                    id: getUnitIdByName(mainUnitName),
                    multiplier: 1,
                    source: 'custom_field2'
                };
            }
        }
        
        // 2. البحث في الوحدات المعالجة
        if (product.processed_units && product.processed_units.length > 0) {
            var mainUnit = product.processed_units.find(unit => unit.is_base_unit == 1);
            if (mainUnit) {
                logPricing('Found main unit in processed_units:', mainUnit);
                return {
                    name: mainUnit.unit_name || mainUnit.name,
                    id: mainUnit.id,
                    multiplier: 1,
                    source: 'processed_units'
                };
            }
        }
        
        // 3. البحث في الوحدات العادية
        if (product.units && Array.isArray(product.units)) {
            var mainUnit = product.units.find(unit => unit.is_base_unit == 1);
            if (mainUnit) {
                logPricing('Found main unit in units:', mainUnit);
                return {
                    name: mainUnit.unit_name || mainUnit.name,
                    id: mainUnit.id,
                    multiplier: 1,
                    source: 'units'
                };
            }
            
            // إذا لم نجد وحدة أساسية، نأخذ الأولى مع multiplier = 1
            var firstUnit = product.units.find(unit => parseFloat(unit.multiplier || 1) === 1);
            if (firstUnit) {
                logPricing('Found first unit with multiplier 1:', firstUnit);
                return {
                    name: firstUnit.unit_name || firstUnit.name,
                    id: firstUnit.id,
                    multiplier: 1,
                    source: 'units_first'
                };
            }
            
            // إذا لم نجد أي شيء، نأخذ الوحدة الأولى
            if (product.units[0]) {
                logPricing('Using first available unit:', product.units[0]);
                return {
                    name: product.units[0].unit_name || product.units[0].name,
                    id: product.units[0].id,
                    multiplier: product.units[0].multiplier || 1,
                    source: 'units_fallback'
                };
            }
        }
        
        // 4. البحث في الوحدة الأساسية للمنتج (product.unit)
        if (product.unit) {
            logPricing('Using product.unit as fallback:', product.unit);
            return {
                name: product.unit,
                id: null,
                multiplier: 1,
                source: 'product_unit'
            };
        }
        
        logPricing('No main unit found for product', null, 'warn');
        return null;
    };
    
    // ==========================================
    // دالة استخراج الوحدة الأساسية من custom_field2
    // ==========================================
    
    window.extractMainUnitFromCustomField2 = function(customField2Data) {
        logPricing('=== Extracting main unit from custom_field2 ===');
        
        if (!customField2Data) {
            logPricing('❌ No custom_field2 data provided', null, 'warn');
            return null;
        }
        
        try {
            var unitName = null;
            
            // إذا كان النص مباشر (اسم الوحدة)
            if (typeof customField2Data === 'string') {
                var trimmedData = customField2Data.trim();
                
                // إذا كان نص بسيط بدون JSON
                if (!trimmedData.startsWith('{') && !trimmedData.startsWith('[')) {
                    unitName = trimmedData;
                    logPricing('✅ Found direct unit name:', unitName);
                    return unitName;
                }
                
                // محاولة تحليل JSON
                try {
                    var parsedData = JSON.parse(trimmedData);
                    
                    // إذا كان كائن واحد
                    if (typeof parsedData === 'object' && !Array.isArray(parsedData)) {
                        unitName = extractUnitFromObject(parsedData);
                    }
                    // إذا كان مصفوفة
                    else if (Array.isArray(parsedData) && parsedData.length > 0) {
                        unitName = extractUnitFromObject(parsedData[0]);
                    }
                    
                } catch (parseError) {
                    logPricing('⚠️ JSON parsing failed, treating as text:', parseError.message);
                    // إذا فشل التحليل، استخدم النص مباشرة
                    unitName = trimmedData;
                }
            }
            // إذا كان كائن مباشر
            else if (typeof customField2Data === 'object') {
                if (Array.isArray(customField2Data) && customField2Data.length > 0) {
                    unitName = extractUnitFromObject(customField2Data[0]);
                } else {
                    unitName = extractUnitFromObject(customField2Data);
                }
            }
            
            if (unitName) {
                // تنظيف اسم الوحدة
                unitName = cleanUnitName(unitName);
                logPricing('✅ Extracted and cleaned unit name:', unitName);
                return unitName;
            }
            
        } catch (error) {
            logPricing('❌ Error extracting main unit:', error.message, 'error');
        }
        
        logPricing('❌ Could not extract main unit from custom_field2');
        return null;
    };
    
    // دالة مساعدة لاستخراج الوحدة من كائن
    function extractUnitFromObject(obj) {
        if (!obj || typeof obj !== 'object') return null;
        
        // البحث في خصائص مختلفة قد تحتوي على اسم الوحدة
        var possibleFields = [
            'main_unit', 'base_unit', 'unit_name', 'unit', 
            'UnitName', 'BaseUnit', 'MainUnit',
            'وحدة_أساسية', 'وحدة', 'الوحدة'
        ];
        
        for (var i = 0; i < possibleFields.length; i++) {
            var field = possibleFields[i];
            if (obj[field] && typeof obj[field] === 'string') {
                return obj[field];
            }
        }
        
        // إذا لم نجد، نبحث في القيم المباشرة
        var keys = Object.keys(obj);
        for (var i = 0; i < keys.length; i++) {
            var value = obj[keys[i]];
            if (typeof value === 'string' && value.trim() && 
                (isKnownUnitName(value) || value.length < 20)) {
                return value;
            }
        }
        
        return null;
    }
    
    // دالة للتحقق من أسماء الوحدات المعروفة
    function isKnownUnitName(unitName) {
        var knownUnits = [
            'قطعة', 'قطعه', 'درزن', 'دزن', 'كارتون', 'كرتون', 'باكيت', 'باكت',
            'كغم', 'كيلو', 'غم', 'جرام', 'لتر', 'مل', 'لك',
            'EA', 'PCS', 'KG', 'GM', 'LTR', 'ML', 'DZ', 'CTN', 'PKT', 'LAK'
        ];
        
        var normalizedUnit = unitName.toLowerCase().trim();
        return knownUnits.some(known => 
            normalizedUnit === known.toLowerCase() ||
            normalizedUnit.includes(known.toLowerCase())
        );
    }
    
    // دالة تنظيف اسم الوحدة
    function cleanUnitName(unitName) {
        if (!unitName || typeof unitName !== 'string') return unitName;
        
        return unitName.trim()
                      .replace(/[""'']/g, '') // إزالة علامات الاقتباس
                      .replace(/\s+/g, ' ')    // توحيد المسافات
                      .trim();
    }
    
    // دالة للحصول على معرف الوحدة بالاسم
    function getUnitIdByName(unitName) {
        if (!unitName) return null;
        
        // البحث في جدول UNIT_MAPPING
        for (var unit in window.UNIT_MAPPING) {
            if (isUnitMatch(unitName, unit)) {
                return window.UNIT_MAPPING[unit];
            }
        }
        
        return null;
    }
    
    // ==========================================
    // تحديث دالة تطبيق التسعير المتقدم
    // ==========================================
    
    window.applyAdvancedPricingToRow = function(row) {
        logPricing('=== Applying advanced pricing to row ===');
        
        var product = getProductDataForRow(row);
        if (!product) {
            logPricing('No product data found for row', null, 'warn');
            return;
        }
        
        var unitName = row.find('.unit-input').val() || 'قطعة';
        logPricing('Current unit in row:', unitName);
        
        var priceResult = determinePriceByUnit(product, unitName, row);
        
        if (priceResult) {
            // إذا كان هناك خطأ
            if (priceResult.error) {
                logPricing('Price determination error:', priceResult.message);
                showUnitError(row, priceResult.message);
                return;
            }
            
            // إذا كان كائن يحتوي على price
            if (typeof priceResult === 'object' && priceResult.price) {
                // إزالة أي أخطاء سابقة
                clearUnitError(row);
                
                // تطبيق السعر
                applyAdvancedPrice(row, priceResult.price);
                
                // إضافة تمييز للوحدة الأساسية
                if (priceResult.isMainUnit) {
                    addMainUnitHighlight(row, priceResult.unitType);
                }
                
                // إذا كان يجب تحديث الوحدة
                if (priceResult.shouldUpdateUnit && priceResult.baseUnit) {
                    if (isDozenUnit(priceResult.baseUnit.name)) {
                        updateRowToDozenUnit(row, priceResult.baseUnit);
                    } else if (isPacketUnit(priceResult.baseUnit.name)) {
                        updateRowToPacketUnit(row, priceResult.baseUnit);
                    } else {
                        updateRowToBaseUnit(row, product);
                    }
                }
                
                logPricing('Price applied successfully:', priceResult.price);
            } 
            // إذا كان رقم مباشر (للتوافق مع الكود القديم)
            else if (typeof priceResult === 'number' && priceResult > 0) {
                clearUnitError(row);
                applyAdvancedPrice(row, priceResult);
                logPricing('Direct price applied:', priceResult);
            }
        } else {
            logPricing('No price result returned', null, 'warn');
        }
    };
    
    // ==========================================
    // دالة جديدة لتمييز الوحدة الأساسية
    // ==========================================
    
    window.addMainUnitHighlight = function(row, unitType) {
        logPricing('Adding main unit highlight:', unitType);
        
        var unitInput = row.find('.unit-input');
        
        // إضافة class للتمييز
        unitInput.addClass('main-unit-highlight');
        
        // إضافة tooltip
        unitInput.attr('title', unitType || 'الوحدة الأساسية للمنتج')
                 .attr('data-toggle', 'tooltip')
                 .attr('data-placement', 'top');
        
        // إضافة أيقونة مرئية
        if (!row.find('.main-unit-indicator').length) {
            var indicator = $('<span class="main-unit-indicator">★</span>');
            unitInput.after(indicator);
        }
        
        // تفعيل tooltip إذا كان متاحاً
        if (typeof unitInput.tooltip === 'function') {
            try {
                unitInput.tooltip();
            } catch (e) {
                // تجاهل خطأ tooltip
            }
        }
    };
    
    // ==========================================
    // دالة محسنة لعرض معلومات تشخيص المنتج
    // ==========================================
    
    window.debugProductPricing = function(productData) {
        console.clear();
        console.log('🔍 === PRODUCT PRICING DEBUG V4.4 ===');
        
        if (!productData) {
            console.log('❌ No product data provided');
            return;
        }
        
        console.log('\n📦 Product Info:');
        console.log('Name:', productData.name);
        console.log('SKU:', productData.sub_sku || productData.sku);
        console.log('ID:', productData.variation_id || productData.id);
        console.log('Custom Field 2 (Main Unit):', productData.product_custom_field2);
        console.log('Custom Field 3 (Prices):', productData.product_custom_field3);
        
        // معلومات الوحدة الأساسية من custom_field2
        console.log('\n🎯 Main Unit Analysis (from custom_field2):');
        if (productData.product_custom_field2) {
            var extractedUnit = extractMainUnitFromCustomField2(productData.product_custom_field2);
            console.log('Raw custom_field2:', productData.product_custom_field2);
            console.log('Extracted Unit:', extractedUnit);
            
            if (extractedUnit) {
                console.log('Is Dozen:', isDozenUnit(extractedUnit));
                console.log('Is Carton:', isCartonUnit(extractedUnit));
                console.log('Is Piece:', isPieceUnit(extractedUnit));
                console.log('Is Packet:', isPacketUnit(extractedUnit));
                console.log('Is LAK:', isLakUnit(extractedUnit));
                console.log('Is Full Plastic:', isFullPlasticUnit(extractedUnit));
            }
        } else {
            console.log('❌ No custom_field2 data found');
        }
        
        // الوحدة الأساسية النهائية
        console.log('\n🎯 Final Main Unit:');
        var mainUnit = getProductMainUnit(productData);
        if (mainUnit) {
            console.log('Main Unit:', mainUnit);
        } else {
            console.log('❌ No main unit found');
        }
        
        // معلومات التسعير
        console.log('\n💰 Pricing Analysis:');
        var priceData = extractPricesFromCustomField3(productData.product_custom_field3);
        if (priceData) {
            console.log('Main Price:', priceData.mainPrice);
            console.log('Unit Prices:', priceData.unitPrices);
            
            // تحليل الوحدات المتاحة
            console.log('\n📋 Available Units with Prices:');
            Object.keys(priceData.unitPrices).forEach(function(code) {
                var unitName = 'Unknown';
                for (var unit in window.UNIT_MAPPING) {
                    if (window.UNIT_MAPPING[unit] == code) {
                        unitName = unit;
                        break;
                    }
                }
                console.log(`  ${unitName} (${code}): ${priceData.unitPrices[code]}`);
            });
        } else {
            console.log('❌ No price data found');
        }
        
        // اختبار تحديد السعر لوحدات مختلفة
        console.log('\n🧪 Price Determination Tests:');
        var testUnits = ['قطعة', 'درزن', 'كارتون', 'باكيت', 'كغم', '0.5', '0.25'];
        
        testUnits.forEach(function(unit) {
            var result = determinePriceByUnit(productData, unit);
            if (result && typeof result === 'object') {
                if (result.error) {
                    console.log(`  ${unit}: ❌ ${result.message}`);
                } else {
                    var status = result.isMainUnit ? '(Base Unit)' : 
                                result.shouldUpdateUnit ? '(Will Update)' : '';
                    console.log(`  ${unit}: ${result.price} ${status}`);
                }
            } else if (typeof result === 'number') {
                console.log(`  ${unit}: ${result}`);
            } else {
                console.log(`  ${unit}: ❌ No price found`);
            }
        });
        
        return true;
    };
    
    // ==========================================
    // تحديث دالة الاختبار مع سيناريوهات المعامل
    // ==========================================
    
    window.testAdvancedPricing = function() {
        console.clear();
        console.log('🧪 === TESTING ADVANCED PRICING SYSTEM V4.4 ===');
        console.log('📌 Priority: custom_field2 → unit_prices_array → multiplier_calculation');
        
        // اختبار 1: منتج بوحدة أساسية درزن مع حساب بالمعامل
        console.log('\n📦 Test 1: Dozen Base Unit with Multiplier Calculation');
        var dozenProduct = {
            name: 'Test Dozen Product',
            sub_sku: 'TEST-DOZ',
            product_custom_field2: 'درزن', // الوحدة الأساسية
            product_custom_field3: `[{"PriceList":1,"Price":24,"Currency":"USD","UoMPrices":[{"PriceList":1,"UoMEntry":18,"Price":120,"Currency":"USD"}]}]`,
            units: [
                { name: 'درزن', id: 17, is_base_unit: 1, multiplier: 1 },
                { name: 'كارتون', id: 18, is_base_unit: 0, multiplier: 5 }, // 5 درزن = 1 كارتون
                { name: 'قطعة', id: 2, is_base_unit: 0, multiplier: 0.08333 } // 12 قطعة = 1 درزن
            ]
        };
        
        console.log('Scenario: درزن base unit ($24), كارتون in array ($120), قطعة calculated');
        console.log('Testing units:');
        ['درزن', 'كارتون', 'قطعة'].forEach(function(unit) {
            var result = determinePriceByUnit(dozenProduct, unit);
            if (result && result.price) {
                var source = result.isMainUnit ? 'Main Price' : 
                            result.source === 'unit_prices_array' ? 'Array Price' :
                            result.calculatedByMultiplier ? 'Calculated' : 'Other';
                console.log(`  ${unit}: ${result.price} (${source})`);
                if (result.calculatedByMultiplier) {
                    console.log(`    Calculation: $24 × ${result.multiplier} = ${result.price}`);
                }
            } else if (result && result.error) {
                console.log(`  ${unit}: ❌ ${result.message}`);
            }
        });
        
        // اختبار 2: منتج بوحدة أساسية قطعة مع حساب بالمعامل
        console.log('\n📦 Test 2: Piece Base Unit with Multiplier Calculation');
        var pieceProduct = {
            name: 'Test Piece Product',
            sub_sku: 'TEST-PC',
            product_custom_field2: 'قطعة', // الوحدة الأساسية
            product_custom_field3: `[{"PriceList":1,"Price":2,"Currency":"USD","UoMPrices":[{"PriceList":1,"UoMEntry":17,"Price":24,"Currency":"USD"}]}]`,
            units: [
                { name: 'قطعة', id: 2, is_base_unit: 1, multiplier: 1 },
                { name: 'درزن', id: 17, is_base_unit: 0, multiplier: 12 }, // 12 قطعة = 1 درزن
                { name: 'كارتون', id: 18, is_base_unit: 0, multiplier: 144 } // 144 قطعة = 1 كارتون
            ]
        };
        
        console.log('Scenario: قطعة base unit ($2), درزن in array ($24), كارتون calculated');
        console.log('Testing units:');
        ['قطعة', 'درزن', 'كارتون'].forEach(function(unit) {
            var result = determinePriceByUnit(pieceProduct, unit);
            if (result && result.price) {
                var source = result.isMainUnit ? 'Main Price' : 
                            result.source === 'unit_prices_array' ? 'Array Price' :
                            result.calculatedByMultiplier ? 'Calculated' : 'Other';
                console.log(`  ${unit}: ${result.price} (${source})`);
                if (result.calculatedByMultiplier) {
                    console.log(`    Calculation: $2 × ${result.multiplier} = ${result.price}`);
                }
            } else if (result && result.error) {
                console.log(`  ${unit}: ❌ ${result.message}`);
            }
        });
        
        // اختبار 3: منتج بوحدة أساسية كارتون مع حساب للوحدات الأصغر
        console.log('\n📦 Test 3: Carton Base Unit with Smaller Units Calculation');
        var cartonProduct = {
            name: 'Test Carton Product',
            sub_sku: 'TEST-CTN',
            product_custom_field2: 'كارتون', // الوحدة الأساسية
            product_custom_field3: `[{"PriceList":1,"Price":144,"Currency":"USD","UoMPrices":[{"PriceList":1,"UoMEntry":20,"Price":120,"Currency":"USD"}]}]`,
            units: [
                { name: 'كارتون', id: 18, is_base_unit: 1, multiplier: 1 },
                { name: 'درزن', id: 17, is_base_unit: 0, multiplier: 0.08333 }, // 1 درزن = 1/12 كارتون
                { name: 'قطعة', id: 2, is_base_unit: 0, multiplier: 0.00694 }, // 1 قطعة = 1/144 كارتون
                { name: 'باكيت', id: 20, is_base_unit: 0, multiplier: 0.83333 } // 1 باكيت = 5/6 كارتون
            ]
        };
        
        console.log('Scenario: كارتون base unit ($144), باكيت in array ($120), others calculated');
        console.log('Testing units:');
        ['كارتون', 'درزن', 'قطعة', 'باكيت'].forEach(function(unit) {
            var result = determinePriceByUnit(cartonProduct, unit);
            if (result && result.price) {
                var source = result.isMainUnit ? 'Main Price' : 
                            result.source === 'unit_prices_array' ? 'Array Price' :
                            result.calculatedByMultiplier ? 'Calculated' : 'Other';
                console.log(`  ${unit}: ${result.price.toFixed(2)} (${source})`);
                if (result.calculatedByMultiplier) {
                    console.log(`    Calculation: $144 × ${result.multiplier} = ${result.price.toFixed(2)}`);
                }
            } else if (result && result.error) {
                console.log(`  ${unit}: ❌ ${result.message}`);
            }
        });
        
        // اختبار 4: منتج بدون معاملات (يجب أن يفشل في الحساب)
        console.log('\n📦 Test 4: Product without Multipliers');
        var noMultiplierProduct = {
            name: 'Test No Multiplier Product',
            sub_sku: 'TEST-NO-MULT',
            product_custom_field2: 'قطعة',
            product_custom_field3: `[{"PriceList":1,"Price":5,"Currency":"USD","UoMPrices":[]}]`,
            units: [
                { name: 'قطعة', id: 2, is_base_unit: 1, multiplier: 1 }
                // لا توجد وحدات أخرى
            ]
        };
        
        console.log('Scenario: قطعة base unit ($5), no other units or multipliers');
        console.log('Testing units:');
        ['قطعة', 'درزن', 'كارتون'].forEach(function(unit) {
            var result = determinePriceByUnit(noMultiplierProduct, unit);
            if (result && result.price) {
                console.log(`  ${unit}: ${result.price} (${result.isMainUnit ? 'Main Price' : 'Other'})`);
            } else if (result && result.error) {
                console.log(`  ${unit}: ❌ ${result.message}`);
                if (result.suggestions && result.suggestions.length > 0) {
                    console.log(`    💡 Suggestions: ${result.suggestions.join(', ')}`);
                }
            }
        });
        
        // اختبار 5: اختبار دوال المساعدة
        console.log('\n🔧 Test 5: Helper Functions');
        
        console.log('Unit Code Mapping:');
        ['قطعة', 'درزن', 'كارتون', 'باكيت', 'كغم'].forEach(function(unit) {
            var code = getUnitCodeByName(unit);
            console.log(`  ${unit} → Code: ${code || 'Not found'}`);
        });
        
        console.log('\nUnit Detection:');
        var testUnits = ['قطعة', 'قطعه', 'EA', 'درزن', 'dz', 'كارتون', 'carton'];
        testUnits.forEach(function(unit) {
            var isPiece = isPieceUnit(unit);
            var isDozen = isDozenUnit(unit);
            var isCarton = isCartonUnit(unit);
            
            if (isPiece) console.log(`  ${unit} → Piece ✅`);
            if (isDozen) console.log(`  ${unit} → Dozen ✅`);
            if (isCarton) console.log(`  ${unit} → Carton ✅`);
        });
        
        return true;
    };
    
    // ==========================================
    // إضافة أنماط CSS للتمييز
    // ==========================================
    
    var mainUnitStyles = `
    <style id="mainUnitStyles">
    /* تمييز الوحدة الأساسية */
    .main-unit-highlight {
        background: linear-gradient(45deg, #e3f2fd, #bbdefb) !important;
        border: 2px solid #2196F3 !important;
        font-weight: 600 !important;
        position: relative;
    }
    
    .main-unit-highlight::before {
        content: "★";
        position: absolute;
        top: -5px;
        right: -5px;
        background: #2196F3;
        color: white;
        border-radius: 50%;
        width: 16px;
        height: 16px;
        font-size: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10;
    }
    
    .main-unit-indicator {
        color: #2196F3;
        font-size: 14px;
        margin-left: 5px;
        animation: starPulse 2s infinite;
    }
    
    @keyframes starPulse {
        0%, 100% { opacity: 1; transform: scale(1); }
        50% { opacity: 0.7; transform: scale(1.1); }
    }
    
    /* تمييز الصفوف حسب نوع الوحدة الأساسية */
    tr:has(.main-unit-highlight[value*="درزن"]) {
        border-left: 4px solid #9c27b0;
        background-color: rgba(156, 39, 176, 0.02);
    }
    
    tr:has(.main-unit-highlight[value*="كارتون"]) {
        border-left: 4px solid #ff9800;
        background-color: rgba(255, 152, 0, 0.02);
    }
    
    tr:has(.main-unit-highlight[value*="قطعة"]) {
        border-left: 4px solid #4caf50;
        background-color: rgba(76, 175, 80, 0.02);
    }
    
    /* تحسين tooltip للوحدة الأساسية */
    .tooltip.main-unit-tooltip .tooltip-inner {
        background-color: #2196F3;
        color: white;
        font-weight: 500;
    }
    
    .tooltip.main-unit-tooltip.bs-tooltip-top .arrow::before {
        border-top-color: #2196F3;
    }
    </style>
    `;
    
    // إضافة الأنماط
    if ($('#mainUnitStyles').length === 0) {
        $('head').append(mainUnitStyles);
    }
    
    // ==========================================
    // دالة تحديث الأزرار للنسخة الجديدة
    // ==========================================
    
    function updateAdvancedPricingButtons() {
        // تحديث الأزرار إذا كانت موجودة
        if ($('.advanced-pricing-buttons').length > 0) {
            $('.advanced-pricing-buttons').html(`
                <button type="button" class="btn btn-warning btn-sm" onclick="testAdvancedPricing()">
                    <i class="fa fa-flask"></i> Test V4.4
                </button>
                <button type="button" class="btn btn-info btn-sm" onclick="debugProductData()">
                    <i class="fa fa-bug"></i> Debug
                </button>
                <button type="button" class="btn btn-primary btn-sm" onclick="testCurrentRow()">
                    <i class="fa fa-search"></i> Test Row
                </button>
                <button type="button" class="btn btn-success btn-sm" onclick="clearAllErrors()">
                    <i class="fa fa-refresh"></i> Clear
                </button>
            `);
        }
    }
    
    // دالة اختبار الصف الحالي
    window.testCurrentRow = function() {
        var lastRow = $('#pos_table tbody tr:not(.empty-row)').last();
        if (lastRow.length === 0) {
            console.log('❌ No product rows found');
            return;
        }
        
        var productData = getProductDataForRow(lastRow);
        if (productData) {
            debugProductPricing(productData);
        } else {
            console.log('❌ No product data found for last row');
        }
    };
    
    // ==========================================
    // التهيئة النهائية
    // ==========================================
    
    $(document).ready(function() {
        setTimeout(function() {
            try {
                updateAdvancedPricingButtons();
                console.log('✅ Advanced Pricing System V4.4 loaded successfully');
                console.log('📌 New Features:');
                console.log('   - Base Unit Priority: Main unit gets main price');
                console.log('   - Smart Unit Detection: Dozen, Carton, Piece support');
                console.log('   - Visual Indicators: Base units are highlighted');
                console.log('   - Enhanced Testing: Use Test V4.4 button');
                console.log('📌 Use debugProductPricing(product) for detailed analysis');
            } catch (e) {
                console.error('❌ Failed to initialize V4.4:', e);
            }
        }, 2000);
        
        // ========================================
        // إصلاح مشكلة عدم تحديث السعر عند تغيير الوحدة
        // Unit Price Update Fix for Unit Changes
        // ========================================
        
        // معالج نظام التسعير المتقدم - أولوية قصوى
        $(document).off('change.advanced-pricing', '.unit-input').on('change.advanced-pricing', '.unit-input', function() {
            var $unitInput = $(this);
            var row = $unitInput.closest('tr');
            var unitName = $unitInput.val().trim();
            
            // منع التشغيل المتعدد
            if ($unitInput.data('advanced-pricing-processing')) {
                return;
            }
            
            if (!unitName) return;
            
            $unitInput.data('advanced-pricing-processing', true);
            console.log('🚀 Advanced pricing system activated for unit:', unitName);
            
            // إضافة علامة للإشارة إلى أن نظام التسعير المتقدم نشط
            row.addClass('advanced-pricing-active');
            
            // الحصول على بيانات المنتج
            var product = null;
            try {
                if (typeof getProductDataForRow === 'function') {
                    product = getProductDataForRow(row);
                }
                
                if (!product) {
                    // محاولة الحصول على البيانات من الصف مباشرة
                    product = row.data('product-data');
                }
            } catch (e) {
                console.log('Error getting product data:', e);
            }
            
            if (product) {
                console.log('📦 Product data found, applying advanced pricing...');
                
                // استخدام نظام التسعير المتقدم
                if (typeof applyAdvancedPricingToRow === 'function') {
                    try {
                        applyAdvancedPricingToRow(row);
                        console.log('✅ Advanced pricing applied successfully');
                        
                        // إنهاء المعالجة بنجاح
                        setTimeout(function() {
                            $unitInput.data('advanced-pricing-processing', false);
                        }, 200);
                        return;
                        
                    } catch (e) {
                        console.log('❌ Error in advanced pricing:', e);
                    }
                }
            }
            
            // إذا فشل نظام التسعير المتقدم، استخدم النظام البديل
            console.log('⚠️ Falling back to basic pricing calculation...');
            
            // الحصول على السعر الأساسي
            var basePriceField = row.find('.hidden_base_unit_sell_price');
            var basePrice = parseFloat(basePriceField.val()) || 0;
            
            if (basePrice <= 0) {
                console.log('⚠️ Base price not found or zero');
                $unitInput.data('advanced-pricing-processing', false);
                return;
            }
            
            // حساب السعر مع التعامل الخاص لوحدة السيت
            var newPrice;
            var isSetUnit = unitName.toLowerCase().includes('سيت') || unitName.toLowerCase().includes('set');
            
            if (isSetUnit) {
                // للسيت: استخدم السعر الأساسي مباشرة
                newPrice = basePrice;
                console.log('🎯 Set unit detected - using base price directly');
            } else {
                // للوحدات الأخرى: محاولة الحصول على المضاعف
                var multiplier = 1;
                try {
                    var availableUnits = [];
                    var unitsData = $unitInput.attr('data-available-units');
                    if (unitsData) {
                        availableUnits = JSON.parse(unitsData);
                        
                        for (var i = 0; i < availableUnits.length; i++) {
                            var unit = availableUnits[i];
                            var unitDisplayName = (unit.unit_name || unit.name || '').toLowerCase().trim();
                            
                            if (unitDisplayName === unitName.toLowerCase().trim()) {
                                multiplier = parseFloat(unit.multiplier) || 1;
                                break;
                            }
                        }
                    }
                } catch (e) {
                    console.log('Error parsing unit data:', e);
                }
                
                newPrice = basePrice * multiplier;
            }
            
            // تطبيق السعر الجديد
            if (newPrice > 0) {
                var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
                var priceIncTax = newPrice * exchange_rate;
                
                // تحديث الحقول
                if (typeof __write_number === 'function') {
                    __write_number(row.find('.pos_unit_price'), newPrice);
                    __write_number(row.find('.pos_unit_price_inc_tax'), priceIncTax);
                }
                
                // تحديث العرض
                row.find('td:eq(5) input').val(newPrice.toFixed(2));
                if (typeof convertUSDToIQD === 'function') {
                    row.find('td:eq(6) input').val(convertUSDToIQD(newPrice).toFixed(0));
                }
                
                // تحديث المجموع
                var quantity = parseFloat(row.find('.pos_quantity').val()) || 1;
                var lineTotal = quantity * priceIncTax;
                
                if (typeof __write_number === 'function') {
                    __write_number(row.find('.pos_line_total'), lineTotal, false);
                }
                
                if (typeof pos_total_row === 'function') {
                    pos_total_row();
                }
                
                console.log('✅ Basic pricing applied:', newPrice);
            }
            
            // إنهاء معالجة نظام التسعير المتقدم
            setTimeout(function() {
                $unitInput.data('advanced-pricing-processing', false);
            }, 300);
        });
        
        // معالج إضافي عند فقدان التركيز
        $(document).off('blur.advanced-pricing', '.unit-input').on('blur.advanced-pricing', '.unit-input', function() {
            var $this = $(this);
            setTimeout(function() {
                $this.trigger('change.advanced-pricing');
            }, 150);
        });
        
        // حماية من تداخل المعالجات القديمة
        $(document).off('change.old-handlers', '.unit-input').on('change.old-handlers', '.unit-input', function(e) {
            var row = $(this).closest('tr');
            
            // إذا كان نظام التسعير المتقدم نشطاً، منع المعالجات الأخرى
            if (row.hasClass('advanced-pricing-active')) {
                console.log('🛡️ Blocking old pricing handlers - Advanced pricing is active');
                e.stopImmediatePropagation();
                return false;
            }
        });
        
        console.log('🔧 Unit price update fix loaded');
        
        // إضافة اختبار سريع للتأكد من الدوال
        setTimeout(function() {
            console.log('🧪 Testing required functions:');
            console.log('   ✓ calculateSimilarity:', typeof window.calculateSimilarity);
            console.log('   ✓ getEditDistance:', typeof window.getEditDistance);
            console.log('   ✓ findBestUnitMatch:', typeof window.findBestUnitMatch);
            console.log('   ✓ applySelectedUnit:', typeof window.applySelectedUnit);
            console.log('   ✓ updateUnitSubmissionData:', typeof window.updateUnitSubmissionData);
            console.log('   ✓ logPricing:', typeof window.logPricing);
            console.log('   ✓ persistUnitValue:', typeof window.persistUnitValue);
            console.log('   ✓ formatNumber:', typeof window.formatNumber);
            console.log('   ✓ convertUSDToIQD:', typeof convertUSDToIQD);
            
            if (typeof window.calculateSimilarity === 'function' && 
                typeof window.applySelectedUnit === 'function') {
                console.log('✅ All unit processing functions are available');
            } else {
                console.log('❌ Some functions are still missing');
            }
            
            console.log('🎯 Advanced pricing system improvements:');
            console.log('   - ⭐ PRIORITY: Advanced pricing system has highest priority');
            console.log('   - 🛡️ PROTECTION: Old handlers blocked when advanced pricing is active');
            console.log('   - 🎯 SET UNITS: Use base price directly (no multiplier)');
            console.log('   - 📝 APPLIES TO: Units containing "سيت" or "set"');
            console.log('   - 🔄 AUTO-ACTIVATION: New products automatically use advanced pricing');
            console.log('   - 🎯 EXACT MATCHING: Fixed "0.25 كغم" vs "كغم" confusion');
            console.log('   - 🔢 NUMBER PROTECTION: Units with numbers require exact match');
            console.log('   - ✅ UNIT ID FIX: "كغم 0.25" now correctly maps to ID 11');
            
            // دالة لتنظيف علامات نظام التسعير المتقدم
            window.clearAdvancedPricingFlags = function() {
                $('.advanced-pricing-active').removeClass('advanced-pricing-active');
                $('.unit-input').removeData('advanced-pricing-processing');
                console.log('🧹 Advanced pricing flags cleared');
            };
            
            // دالة لفرض استخدام نظام التسعير المتقدم
            window.forceAdvancedPricing = function(row) {
                if (!row || !row.length) return;
                
                row.addClass('advanced-pricing-active');
                var unitInput = row.find('.unit-input');
                if (unitInput.length > 0) {
                    setTimeout(function() {
                        unitInput.trigger('change.advanced-pricing');
                    }, 100);
                }
                console.log('🚀 Forced advanced pricing for row');
            };
            
            // دالة لاختبار وحدة معينة والحصول على معرفها
            window.testUnitID = function(unitName) {
                console.log('🔍 Testing unit: "' + unitName + '"');
                
                var unitId = window.UNIT_MAPPING[unitName];
                if (unitId) {
                    console.log('   ✅ Found ID:', unitId);
                } else {
                    console.log('   ❌ Unit not found in UNIT_MAPPING');
                    
                    // محاولة البحث التقريبي
                    console.log('   🔍 Trying fuzzy search...');
                    for (var key in window.UNIT_MAPPING) {
                        if (key.includes(unitName) || unitName.includes(key)) {
                            console.log('     Possible match:', key, '→ ID:', window.UNIT_MAPPING[key]);
                        }
                    }
                }
                
                return unitId;
            };
            
            // دالة لعرض ملخص الإصلاحات
            window.showUnitFixSummary = function() {
                console.log('🎯 Unit Pricing Fix Summary:');
                console.log('══════════════════════════════════════');
                console.log('✅ FIXED: "كغم 0.25" now maps to correct ID 11');
                console.log('✅ REMOVED: "كغم" mapping to prevent confusion');
                console.log('✅ ENHANCED: Exact matching for units with numbers');
                console.log('✅ PROTECTED: Mixed number/non-number unit matching blocked');
                console.log('══════════════════════════════════════');
                console.log('💡 Test commands:');
                console.log('   testUnitID("كغم 0.25")  // Should return 11');
                console.log('   testUnitMatching()      // Test matching logic');
                console.log('   testAdvancedPricing()   // Full system test');
            };
            
            // دالة لاختبار المطابقة المحسنة
            window.testUnitMatching = function() {
                console.log('🧪 Testing improved unit matching:');
                
                var testCases = [
                    ['كغم 0.25', 'كغم 0.25'], // يجب أن تتطابق تماماً
                    ['كغم 0.25', 'كغم'],       // يجب ألا تتطابق (هذه كانت المشكلة!)
                    ['قطعة', 'قطعه'],         // يجب أن تتطابق تقريبياً
                    ['درزن', 'دزن'],          // يجب أن تتطابق تقريبياً
                    ['0.25 كغم', 'كغم 0.25'], // يجب أن تتطابق تماماً
                    ['50غم', '100غم'],        // يجب ألا تتطابق (وحدات مختلفة بأرقام)
                ];
                
                testCases.forEach(function(testCase) {
                    var unit1 = testCase[0];
                    var unit2 = testCase[1];
                    var result = isUnitMatch(unit1, unit2);
                    
                    console.log('   ' + unit1 + ' vs ' + unit2 + ': ' + (result ? '✅ Match' : '❌ No Match'));
                });
                
                console.log('');
                console.log('🎯 Key improvement: "كغم 0.25" no longer matches with "كغم"');
                console.log('   This prevents wrong pricing when selecting 0.25 kg units!');
                
                // اختبار المعرف الصحيح لوحدة كغم 0.25
                console.log('');
                console.log('🔍 Testing unit ID mapping:');
                var unitId = window.UNIT_MAPPING['كغم 0.25'];
                console.log('   "كغم 0.25" maps to ID:', unitId, unitId === 11 ? '✅ CORRECT!' : '❌ WRONG!');
                
                // اختبار الوحدات الأخرى للتأكد
                console.log('   "غم 250" maps to ID:', window.UNIT_MAPPING['غم 250'], '(should be 5)');
                console.log('   "درزن" maps to ID:', window.UNIT_MAPPING['درزن'], '(should be 17)');
                

// ==========================================
// تحسين خاص للكارتون - ضمان التطبيق الفوري للسعر الصحيح
// ==========================================

// دالة للتحقق من وجود وحدة كارتون كوحدة أساسية وتطبيق سعرها من المصفوفة
window.ensureCartonBasePricing = function(row, product) {
    if (!row || !product) return;
    
    // التحقق من الوحدات المتاحة
    var availableUnits = [];
    if (product.processed_units && product.processed_units.length > 0) {
        availableUnits = product.processed_units;
    } else if (product.units && product.units.length > 0) {
        availableUnits = product.units;
    } else if (product.sub_units && product.sub_units.length > 0) {
        availableUnits = product.sub_units;
    }
    
    // البحث عن الوحدة الأساسية
    var baseUnit = availableUnits.find(unit => unit.is_base_unit == 1);
    
    if (baseUnit && isCartonUnit(baseUnit.unit_name || baseUnit.name)) {
        console.log('🎯 Detected carton as base unit - will use price array prioritization');
        
        // التحقق من وجود مصفوفة الأسعار
        if (product.product_custom_field3) {
            try {
                var priceArray = JSON.parse(product.product_custom_field3);
                if (priceArray && priceArray[0] && priceArray[0].UoMPrices) {
                    console.log('📊 Price array found:', priceArray[0].UoMPrices);
                }
            } catch (e) {
                console.log('⚠️ Could not parse price array');
            }
        }
        
        // التأكد من أن الحقل يحتوي على اسم الوحدة الصحيح
        var unitInput = row.find('.unit-input');
        if (unitInput.length > 0) {
            unitInput.val(baseUnit.unit_name || baseUnit.name);
        }
        
        // إجبار تطبيق التسعير المتقدم مع أولوية المصفوفة
        setTimeout(function() {
            if (typeof applyAdvancedPricingToRow === 'function') {
                applyAdvancedPricingToRow(row);
                console.log('🎯 Carton base unit pricing from array applied successfully');
            }
        }, 50);
        
        return true;
    }
    
    return false;
};

// تعزيز معالج إضافة المنتج من البحث للتأكد من التعامل الصحيح مع الكارتون
$(document).on('select2:select', '#search_product', function(e) {
    console.log('🔍 Product selected from select2 search - will be handled by pos_product_row');
    // لا نحتاج لمعالجة إضافية هنا لأن pos_product_row سيتولى الأمر
});

// دالة مساعدة للتأكد من تطبيق التسعير المباشر للكارتون
window.forceCartonPricingFromArray = function(row) {
    var productData = row.data('product-data');
    if (!productData) return false;
    
    // التحقق من أن الوحدة الحالية كارتون أو أن الوحدة الأساسية كارتون
    var currentUnit = row.find('.unit-input').val() || '';
    var baseUnit = productData.product_custom_field2 || '';
    
    var isCurrentUnitCarton = isCartonUnit(currentUnit);
    var isBaseUnitCarton = isCartonUnit(baseUnit);
    
    // إذا لم تكن أي من الوحدات كارتون، لا تطبق
    if (!isCurrentUnitCarton && !isBaseUnitCarton) return false;
    
    // إذا كانت الوحدة الأساسية كارتون، قم بتطبيق الوحدة والسعر
    if (isBaseUnitCarton && !isCurrentUnitCarton) {
        console.log('🎯 Base unit is carton - setting unit input to carton');
        row.find('.unit-input').val(baseUnit);
    }
    
    // التحقق من وجود مصفوفة الأسعار
    if (!productData.product_custom_field3) return false;
    
    try {
        var priceArray = JSON.parse(productData.product_custom_field3);
        if (priceArray && priceArray[0] && priceArray[0].UoMPrices) {
            var cartonCode = 18; // كود الكارتون
            var cartonPrice = null;
            
            // البحث عن سعر الكارتون في المصفوفة
            priceArray[0].UoMPrices.forEach(function(uomPrice) {
                if (uomPrice.UoMEntry === cartonCode && uomPrice.Price > 0) {
                    cartonPrice = parseFloat(uomPrice.Price);
                }
            });
            
            if (cartonPrice) {
                console.log('🎯 Found carton price in array:', cartonPrice, '- applying directly');
                
                // تطبيق السعر مباشرة
                var exchange_rate = parseFloat($('#exchange_rate').val()) || 1;
                var priceInIQD = cartonPrice * exchange_rate;
                
                // تحديث حقول السعر
                row.find('.pos_unit_price').val(cartonPrice);
                row.find('.pos_unit_price_inc_tax').val(cartonPrice);
                
                // إعادة حساب الإجماليات
                var quantity = parseFloat(row.find('.pos_quantity').val()) || 1;
                var lineTotal = quantity * cartonPrice;
                row.find('.pos_line_total').val(lineTotal);
                
                // تحديث النص المعروض
                row.find('.pos_line_total_text').text(__currency_trans_from_en(lineTotal, true));
                
                // إعادة حساب الإجماليات العامة
                pos_each_row(row);
                pos_total_row();
                
                console.log('✅ Carton price applied successfully from array');
                return true;
            }
        }
    } catch (e) {
        console.log('⚠️ Error parsing price array:', e);
    }
    
    return false;
};

                
                // عرض ملخص الإصلاحات
                console.log('');
                showUnitFixSummary();
            };
        }, 1000);
    });
    
})();

// ==========================================
// تحسين نهائي لضمان التسعير الفوري للكارتون
// ==========================================

// اعتراض معالج autocomplete select للتأكد من التسعير الفوري
$(document).ready(function() {
    // التأكد من تطبيق التسعير فوراً بعد إضافة أي منتج
    var originalPosProductRow = window.pos_product_row;
    
    if (originalPosProductRow && typeof originalPosProductRow === 'function') {
        window.pos_product_row = function(variation_id, purchase_line_id, weighing_scale_barcode, quantity) {
            console.log('🚀 Enhanced pos_product_row called for variation_id:', variation_id);
            
            // استدعاء الدالة الأصلية
            var result = originalPosProductRow.apply(this, arguments);
            
            // التأكد من تطبيق التسعير الفوري للمنتجات التي وحدتها الأساسية كارتون
            setTimeout(function() {
                var lastRow = $('#pos_table tbody tr').not('.empty-row').last();
                if (lastRow.length > 0) {
                    var variationId = lastRow.find('.variation_id').val();
                    
                    if (variationId) {
                        console.log('🔍 Post-addition check for carton pricing on row with variation_id:', variationId);
                        
                        // محاولة تطبيق التسعير المباشر إذا توفرت البيانات
                        if (typeof forceCartonPricingFromArray === 'function') {
                            var applied = forceCartonPricingFromArray(lastRow);
                            if (applied) {
                                console.log('✅ Direct carton pricing applied successfully');
                            }
                        }
                    }
                }
            }, 500); // وقت إضافي للتأكد من تحميل البيانات
            
            return result;
        };
        
                 console.log('✅ Enhanced pos_product_row wrapper installed');
     }
});

// ==========================================
// ميزة نسخ الجدول - Table Copy Feature
// ==========================================

$(document).ready(function() {
    console.log('🚀 DOM Ready - Initializing table copy feature...');
    addTableCopyFeature();
    
    // إشعار بأن الميزة جاهزة
    setTimeout(function() {
        console.log('📋 Table Copy Feature Status:');
        console.log('   - Copy buttons:', $('.cyber-action-btn.copy-btn').length);
        console.log('   - Table rows:', $('#pos_table tbody tr').length);
        console.log('   - Checkboxes:', $('.pos-row-checkbox').length);
        
        if ($('.cyber-action-btn.copy-btn').length > 0) {
            console.log('✅ All copy buttons are ready!');
        } else {
            console.log('❌ Copy buttons not found!');
        }
    }, 1000);
    
    // مراقب إضافي لتغييرات DOM
    var domObserver = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
                // تحديث الأزرار عند تغيير DOM
                setTimeout(function() {
                    updateButtonStates();
                    forceEnableButtons();
                }, 100);
            }
        });
    });
    
    // مراقبة تغييرات الجدول
    if ($('#pos_table').length) {
        domObserver.observe($('#pos_table')[0], { 
            childList: true, 
            subtree: true 
        });
    }
});

function addTableCopyFeature() {
    console.log('🔄 Adding table copy feature...');
    
    // إضافة أزرار التحكم فوق الجدول
    addCopyControls();
    
    // إضافة اختصارات لوحة المفاتيح
    setupCopyShortcuts();
    
    // إضافة معالجات الأحداث
    setupCopyEvents();
    
    // تفحل إجباري لتفعيل الأزرار
    forceEnableButtons();
    
    // التأكد من التحديث الدوري لحالة الأزرار
    var updateInterval = setInterval(function() {
        updateButtonStates();
        forceEnableButtons();
    }, 2000);
    
    // تحديثات متعددة لضمان التفعيل
    setTimeout(updateButtonStates, 100);
    setTimeout(forceEnableButtons, 200);
    setTimeout(updateButtonStates, 500);
    setTimeout(forceEnableButtons, 1000);
    setTimeout(updateButtonStates, 2000);
    
    console.log('✅ Table copy feature added successfully');
}

// دالة إجبار تفعيل الأزرار إذا كان هناك بيانات
function forceEnableButtons() {
    var tableRows = $('#pos_table tbody tr').not('.empty-row').length;
    
    if (tableRows > 0) {
        console.log('🔧 Force enabling buttons - found', tableRows, 'rows');
        $('#copy-all-btn, #export-excel-btn, #select-all-btn').removeClass('disabled').addClass('has-data');
        
        // تأكد من الأنماط البصرية
        $('.cyber-action-btn.copy-btn').addClass('has-data');
        
        // تحديث النصوص
        $('#copy-all-btn span').text(`نسخ الكل (${tableRows})`);
        $('#export-excel-btn span').text(`Excel (${tableRows})`);
        $('#select-all-btn span').text('تحديد الكل');
        
        console.log('✅ Buttons force-enabled successfully');
    } else {
        console.log('⚠️ No table rows found for force enabling');
    }
}

// إضافة أنماط النسخ المدمجة
function addCopyControls() {
    // إضافة الأنماط المطلوبة فقط
    var stylesHTML = `
        <style id="copy-table-styles">
            .row-selected {
                background-color: #e3f2fd !important;
                border-left: 3px solid #2196F3;
            }
            
            .row-selected td {
                background-color: #e3f2fd !important;
            }
            
            .copy-notification {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 9999;
                background: #4CAF50;
                color: white;
                padding: 10px 15px;
                border-radius: 4px;
                box-shadow: 0 2px 8px rgba(0,0,0,0.2);
                animation: slideIn 0.3s ease-out;
            }
            
            @keyframes slideIn {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            
            /* أنماط إضافية لأزرار النسخ */
            .cyber-action-btn.copy-btn {
                opacity: 1;
                transition: all 0.2s ease;
                cursor: pointer;
            }
            
            .cyber-action-btn.copy-btn:hover:not(.disabled) {
                opacity: 1;
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.2);
            }
            
            .cyber-action-btn.copy-btn.disabled {
                opacity: 0.4;
                pointer-events: none;
                cursor: not-allowed;
                filter: grayscale(100%);
            }
            
            .cyber-action-btn.copy-btn.active {
                background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
                box-shadow: 0 0 10px rgba(76, 175, 80, 0.4);
                color: white;
            }
            
            /* تأكيد أن الأزرار تبدو مفعلة */
            .cyber-action-btn.copy-btn:not(.disabled) {
                background: linear-gradient(135deg, #6366f1 0%, #5b5bd6 100%);
                border: 1px solid #5b5bd6;
                color: white;
                box-shadow: 0 2px 8px rgba(99, 102, 241, 0.3);
            }
            
            /* تمييز إضافي للأزرار التي لديها بيانات */
            .cyber-action-btn.copy-btn.has-data:not(.disabled) {
                background: linear-gradient(135deg, #10b981 0%, #059669 100%);
                border: 1px solid #059669;
                box-shadow: 0 2px 8px rgba(16, 185, 129, 0.4);
                animation: pulse-ready 2s infinite;
            }
            
            @keyframes pulse-ready {
                0%, 100% { 
                    box-shadow: 0 2px 8px rgba(16, 185, 129, 0.4);
                }
                50% { 
                    box-shadow: 0 4px 16px rgba(16, 185, 129, 0.6);
                }
            }
        </style>
    `;
    
    // إضافة الأنماط فقط إذا لم تكن موجودة
    if (!$('#copy-table-styles').length) {
        $('head').append(stylesHTML);
    }
}

// إعداد اختصارات لوحة المفاتيح
function setupCopyShortcuts() {
    $(document).on('keydown', function(e) {
        // Ctrl+A - تحديد جميع الصفوف
        if (e.ctrlKey && e.keyCode === 65 && $(e.target).closest('#pos_table').length) {
            e.preventDefault();
            selectAllRows();
        }
        
        // Ctrl+C - نسخ الصفوف المحددة
        else if (e.ctrlKey && !e.shiftKey && e.keyCode === 67 && $(e.target).closest('#pos_table').length) {
            e.preventDefault();
            copySelectedRows();
        }
        
        // Ctrl+Shift+C - نسخ الجدول بالكامل
        else if (e.ctrlKey && e.shiftKey && e.keyCode === 67 && $(e.target).closest('#pos_table').length) {
            e.preventDefault();
            copyAllTable();
        }
    });
}

// إعداد معالجات الأحداث
function setupCopyEvents() {
    console.log('🔄 Setting up copy events...');
    
    // معالجات الأزرار - التأكد من عدم تكرار ربط الأحداث
    $(document).off('click', '#select-all-btn').on('click', '#select-all-btn', function(e) {
        e.preventDefault();
        console.log('✅ Select All button clicked');
        selectAllRows();
    });
    
    $(document).off('click', '#copy-selected-btn').on('click', '#copy-selected-btn', function(e) {
        e.preventDefault();
        if (!$(this).hasClass('disabled')) {
            console.log('✅ Copy Selected button clicked');
            copySelectedRows();
        }
    });
    
    $(document).off('click', '#copy-all-btn').on('click', '#copy-all-btn', function(e) {
        e.preventDefault();
        if (!$(this).hasClass('disabled')) {
            console.log('✅ Copy All button clicked');
            copyAllTable();
        }
    });
    
    $(document).off('click', '#export-excel-btn').on('click', '#export-excel-btn', function(e) {
        e.preventDefault();
        if (!$(this).hasClass('disabled')) {
            console.log('✅ Export Excel button clicked');
            exportToExcel();
        }
    });
    
    // إضافة checkboxes للصفوف الموجودة
    addCheckboxesToExistingRows();
    
    // مراقبة إضافة صفوف جديدة
    var observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1 && node.tagName === 'TR') {
                        addCheckboxToRow($(node));
                    }
                });
                updateButtonStates();
            }
        });
    });
    
    if ($('#pos_table tbody').length) {
        observer.observe($('#pos_table tbody')[0], { childList: true });
    }
    
    // معالج تغيير الـ checkboxes
    $(document).on('change', '.pos-row-checkbox', function() {
        var row = $(this).closest('tr');
        if ($(this).is(':checked')) {
            row.addClass('row-selected');
        } else {
            row.removeClass('row-selected');
        }
        updateButtonStates();
    });
    
    // النقر على الصف لتحديده
    $(document).on('click', '#pos_table tbody tr', function(e) {
        if (!$(e.target).is('input, select, button, a, .fa') && !$(e.target).closest('input, select, button, a').length) {
            var checkbox = $(this).find('.pos-row-checkbox');
            if (checkbox.length) {
                checkbox.prop('checked', !checkbox.prop('checked')).trigger('change');
            }
        }
    });
    
    // تحديث حالة الأزرار بشكل فوري ومتكرر
    updateButtonStates();
    setTimeout(updateButtonStates, 500);
    setTimeout(updateButtonStates, 1000);
    setTimeout(updateButtonStates, 2000);
    
    console.log('✅ Copy events setup complete');
}

// إضافة checkboxes للصفوف الموجودة
function addCheckboxesToExistingRows() {
    console.log('🔄 Adding checkboxes to existing rows...');
    var rowsProcessed = 0;
    
    $('#pos_table tbody tr').not('.empty-row').each(function() {
        var $row = $(this);
        if (!$row.find('.pos-row-checkbox').length) {
            addCheckboxToRow($row);
            rowsProcessed++;
        }
    });
    
    console.log(`✅ Processed ${rowsProcessed} rows, total checkboxes: ${$('.pos-row-checkbox').length}`);
    updateButtonStates();
}

// إضافة checkbox لصف واحد
function addCheckboxToRow(row) {
    if (row.hasClass('empty-row') || row.find('.pos-row-checkbox').length > 0) {
        console.log('🔶 Skipping row (empty or already has checkbox)');
        return;
    }
    
    var firstCell = row.find('td:first');
    if (firstCell.length) {
        var currentContent = firstCell.html();
        firstCell.html(`<input type="checkbox" class="pos-row-checkbox" style="margin-right: 5px;"> ${currentContent}`);
        console.log('✅ Added checkbox to row');
    } else {
        console.log('❌ No first cell found in row');
    }
}

// تحديد جميع الصفوف
window.selectAllRows = function() {
    console.log('🔄 selectAllRows called');
    var checkboxes = $('.pos-row-checkbox');
    console.log('📋 Found checkboxes:', checkboxes.length);
    
    if (checkboxes.length === 0) {
        // إذا لم توجد checkboxes، قم بإضافتها أولاً
        console.log('⚠️ No checkboxes found, adding them...');
        addCheckboxesToExistingRows();
        
        // جرب مرة أخرى
        checkboxes = $('.pos-row-checkbox');
        if (checkboxes.length === 0) {
            showNotification('لا توجد صفوف للتحديد', 'warning');
            return;
        }
    }
    
    var allChecked = checkboxes.filter(':checked').length === checkboxes.length;
    
    if (allChecked) {
        // إلغاء تحديد الكل
        checkboxes.prop('checked', false).trigger('change');
        showNotification('تم إلغاء تحديد جميع الصفوف');
        console.log('✅ All rows deselected');
    } else {
        // تحديد الكل
        checkboxes.prop('checked', true).trigger('change');
        showNotification(`تم تحديد ${checkboxes.length} صف`);
        console.log('✅ All rows selected');
    }
};

// نسخ الصفوف المحددة
window.copySelectedRows = function() {
    console.log('🔄 copySelectedRows called');
    var selectedRows = $('.pos-row-checkbox:checked').closest('tr');
    console.log('📋 Selected rows:', selectedRows.length);
    
    if (selectedRows.length === 0) {
        // إذا لم توجد صفوف محددة، جرب نسخ جميع الصفوف
        console.log('⚠️ No selected rows, trying to copy all rows instead...');
        copyAllTable();
        return;
    }
    
    var tableData = extractRowsData(selectedRows);
    copyToClipboard(tableData);
    showNotification(`تم نسخ ${selectedRows.length} صف بنجاح (أسماء المنتجات مفصلة إلى 3 أعمدة)`);
    console.log('✅ Copy selected completed');
};

// نسخ الجدول بالكامل
window.copyAllTable = function() {
    console.log('🔄 copyAllTable called');
    var allRows = $('#pos_table tbody tr').not('.empty-row');
    console.log('📋 Total rows found:', allRows.length);
    
    if (allRows.length === 0) {
        showNotification('الجدول فارغ', 'warning');
        return;
    }
    
    var tableData = extractRowsData(allRows, true);
    copyToClipboard(tableData);
    showNotification(`تم نسخ الجدول بالكامل (${allRows.length} صف) - أسماء المنتجات مفصلة إلى 3 أعمدة`);
    console.log('✅ Copy all completed');
};

// استخراج بيانات الصفوف مع فصل أجزاء اسم المنتج (دائماً مفعل)
function extractRowsData(rows, includeHeaders = false) {
    var data = [];
    
    // إضافة العناوين المحدثة
    if (includeHeaders) {
        var headers = [];
        var productColumnFound = false;
        
        $('#pos_table thead th').each(function(index) {
            var headerText = $(this).text().trim();
            headerText = cleanTextForExport(headerText);
            
            // إذا كان عمود المنتج، قم بفصله إلى ثلاثة أعمدة
            if ((headerText.toLowerCase().includes('product') || headerText.includes('منتج') || index === 1)) {
                if (!productColumnFound) {
                    headers.push('Item Code');
                    headers.push('Name');
                    headers.push('Foreign Name');
                    productColumnFound = true;
                }
            } else {
                headers.push(headerText);
            }
        });
        data.push(headers);
    }
    
    // استخراج بيانات الصفوف
    rows.each(function() {
        var row = $(this);
        var rowData = [];
        
        row.find('td').each(function(cellIndex) {
            var cell = $(this);
            var cellText = '';
            
            // استخراج النص من أنواع مختلفة من الخلايا
            if (cell.find('input[type="text"], input[type="number"]').length) {
                cellText = cell.find('input[type="text"], input[type="number"]').val() || '';
            } else if (cell.find('select').length) {
                cellText = cell.find('select option:selected').text() || '';
            } else if (cell.find('.product-info strong').length) {
                // معالجة خاصة لخلية المنتج - فصل الأجزاء دائماً
                var productText = cell.find('.product-info strong').text() || '';
                var productParts = parseProductName(productText);
                
                rowData.push(cleanTextForExport(productParts.itemCode));
                rowData.push(cleanTextForExport(productParts.name));
                rowData.push(cleanTextForExport(productParts.foreignName));
                return; // تجاهل معالجة هذه الخلية بالطريقة العادية
            } else {
                // إزالة checkbox من النص
                var tempText = cell.clone();
                tempText.find('.pos-row-checkbox').remove();
                cellText = tempText.text().trim();
            }
            
            // تنظيف النص من الأحرف التي تسبب مشاكل في التصدير
            cellText = cleanTextForExport(cellText);
            rowData.push(cellText);
        });
        
        if (rowData.some(cell => cell !== '')) {
            data.push(rowData);
        }
    });
    
    return data;
}

// دالة لفصل أجزاء اسم المنتج
function parseProductName(productText) {
    var parts = {
        itemCode: '',
        name: '',
        foreignName: ''
    };
    
    if (!productText) return parts;
    
    // فصل النص باستخدام الفاصل العمودي |
    var segments = productText.split('|').map(function(part) {
        return part.trim();
    });
    
    // توزيع الأجزاء
    if (segments.length >= 3) {
        parts.itemCode = segments[0] || '';
        parts.name = segments[1] || '';
        parts.foreignName = segments[2] || '';
    } else if (segments.length === 2) {
        parts.itemCode = segments[0] || '';
        parts.name = segments[1] || '';
        parts.foreignName = '';
    } else if (segments.length === 1) {
        parts.itemCode = segments[0] || '';
        parts.name = '';
        parts.foreignName = '';
    }
    
    return parts;
}

// نسخ إلى الحافظة
function copyToClipboard(data) {
    // تحويل إلى TSV للتوافق مع Excel
    var content = data.map(row => row.join('\t')).join('\n');
    
    if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(content).then(function() {
            console.log('✅ تم النسخ بنجاح');
        }).catch(function(err) {
            console.error('❌ فشل النسخ:', err);
            fallbackCopy(content);
        });
    } else {
        fallbackCopy(content);
    }
}

// نسخ احتياطي
function fallbackCopy(text) {
    var textarea = document.createElement('textarea');
    textarea.value = text;
    textarea.style.position = 'fixed';
    textarea.style.opacity = '0';
    document.body.appendChild(textarea);
    textarea.select();
    
    try {
        document.execCommand('copy');
        console.log('✅ تم النسخ الاحتياطي');
    } catch (err) {
        console.error('❌ فشل النسخ الاحتياطي:', err);
    }
    
    document.body.removeChild(textarea);
}

// تحديث تسميات الأزرار ديناميكياً
function updateButtonLabels() {
    var selectedCount = $('.pos-row-checkbox:checked').length;
    var totalCount = $('.pos-row-checkbox').length;
    var tableRows = $('#pos_table tbody tr').not('.empty-row').length;
    
    // إذا لم توجد checkboxes، استخدم عدد صفوف الجدول
    var actualRows = totalCount > 0 ? totalCount : tableRows;
    
    console.log('🔄 Updating button labels:', { selectedCount, totalCount, tableRows, actualRows });
    
    // تحديث نص الأزرار لإظهار العدد
    if (selectedCount > 0) {
        $('#copy-selected-btn span').text(`نسخ المحدد (${selectedCount})`);
    } else {
        $('#copy-selected-btn span').text('نسخ المحدد');
    }
    
    if (actualRows > 0) {
        $('#copy-all-btn span').text(`نسخ الكل (${actualRows})`);
        $('#export-excel-btn span').text(`Excel (${actualRows})`);
        $('#select-all-btn span').text(selectedCount === totalCount && totalCount > 0 ? 'إلغاء التحديد' : 'تحديد الكل');
    } else {
        $('#copy-all-btn span').text('نسخ الكل');
        $('#export-excel-btn span').text('Excel');
        $('#select-all-btn span').text('تحديد الكل');
    }
}

// دالة تنظيف النص للتصدير
function cleanTextForExport(text) {
    if (!text || typeof text !== 'string') return text;
    
    return text
        // إزالة أحرف التبويب
        .replace(/\t/g, ' ')
        // إزالة أسطر جديدة واستبدالها بمسافات
        .replace(/\r?\n/g, ' ')
        // إزالة المسافات الزائدة
        .replace(/\s+/g, ' ')
        // إزالة المسافات من البداية والنهاية
        .trim()
        // إزالة علامات الاقتباس المزدوجة لتجنب مشاكل CSV
        .replace(/"/g, "'");
}

// تصدير إلى Excel
window.exportToExcel = function() {
    console.log('🔄 exportToExcel called');
    var allRows = $('#pos_table tbody tr').not('.empty-row');
    console.log('📋 Rows for export:', allRows.length);
    
    if (allRows.length === 0) {
        showNotification('الجدول فارغ', 'warning');
        return;
    }
    
    var data = extractRowsData(allRows, true);
    var content = data.map(row => row.join('\t')).join('\n');
    
    var blob = new Blob([content], { type: 'application/vnd.ms-excel' });
    var url = URL.createObjectURL(blob);
    var a = document.createElement('a');
    a.href = url;
    a.download = 'pos-table-' + new Date().toISOString().slice(0,10) + '.xls';
    document.body.appendChild(a);
    a.click();
    URL.revokeObjectURL(url);
    document.body.removeChild(a);
    
    showNotification('تم تصدير Excel بنجاح - أسماء المنتجات مفصلة إلى أعمدة منفصلة');
    console.log('✅ Excel export completed');
};

// تحديث حالة الأزرار
function updateButtonStates() {
    var selectedCount = $('.pos-row-checkbox:checked').length;
    var totalCount = $('.pos-row-checkbox').length;
    var tableRows = $('#pos_table tbody tr').not('.empty-row').length;
    
    console.log('🔄 Button states update:', { 
        selectedCount, 
        totalCount, 
        tableRows,
        copyBtns: $('.cyber-action-btn.copy-btn').length 
    });
    
    // إذا لم توجد checkboxes، استخدم عدد صفوف الجدول
    var actualRows = totalCount > 0 ? totalCount : tableRows;
    
    // تمكين/تعطيل أزرار النسخ
    if (selectedCount === 0) {
        $('#copy-selected-btn').addClass('disabled');
    } else {
        $('#copy-selected-btn').removeClass('disabled');
    }
    
    // تمكين أزرار النسخ والتصدير إذا كان هناك صفوف في الجدول
    if (actualRows === 0) {
        $('#copy-all-btn, #export-excel-btn, #select-all-btn').addClass('disabled');
        console.log('❌ All buttons disabled - no data');
    } else {
        $('#copy-all-btn, #export-excel-btn, #select-all-btn').removeClass('disabled');
        console.log('✅ Buttons enabled - data found:', actualRows);
    }
    
    // تحديث حالة زر التحديد
    if (selectedCount === totalCount && totalCount > 0) {
        $('#select-all-btn').addClass('active');
    } else {
        $('#select-all-btn').removeClass('active');
    }
    
    // تحديث تسميات الأزرار
    updateButtonLabels();
    
    // إضافة كلاس has-data للأزرار إذا كان هناك بيانات
    if (actualRows > 0) {
        $('.cyber-action-btn.copy-btn').addClass('has-data');
        console.log('🎨 Added has-data class to buttons');
    } else {
        $('.cyber-action-btn.copy-btn').removeClass('has-data');
    }
}

// عرض الإشعارات
function showNotification(message, type = 'success') {
    var bgColor = type === 'warning' ? '#ff9800' : type === 'error' ? '#f44336' : '#4CAF50';
    
    var notification = $(`
        <div class="copy-notification" style="background-color: ${bgColor};">
            <i class="fa fa-${type === 'warning' ? 'exclamation-triangle' : type === 'error' ? 'times' : 'check'}"></i>
            ${message}
        </div>
    `);
    
    $('body').append(notification);
    
    setTimeout(function() {
        notification.fadeOut(300, function() {
            $(this).remove();
        });
    }, 3000);
}

// SAP Integration Functions
var SAPIntegration = {
    // Check SAP sync status for a transaction
    checkSyncStatus: function(transactionId, callback) {
        $.ajax({
            url: '/sells/pos/check-sap-sync-status/' + transactionId,
            method: 'GET',
            dataType: 'json',
            success: function(result) {
                if (result.success == 1) {
                    if (callback) {
                        callback(result);
                    }
                    
                    // Update UI if SAP document ID is available
                    if (result.sap_document_id) {
                        // Update any visible SAP document ID displays
                        $('.sap-document-id').text(result.sap_document_id);
                        
                        // Show success message if just synced
                        if (result.msg) {
                            toastr.success(result.msg);
                        }
                    } else if (result.sap_error) {
                        toastr.warning('SAP: ' + result.sap_error);
                    }
                } else {
                    toastr.error(result.msg || 'Failed to check SAP status');
                }
            },
            error: function() {
                console.error('Failed to check SAP sync status');
            }
        });
    },
    
    // Force sync with SAP
    forceSync: function(transactionId, callback) {
        $.ajax({
            url: '/sells/pos/force-sap-sync/' + transactionId,
            method: 'POST',
            dataType: 'json',
            success: function(result) {
                if (result.success == 1) {
                    toastr.success(result.msg);
                    if (callback) {
                        callback(result);
                    }
                } else {
                    toastr.error(result.msg);
                }
            },
            error: function() {
                toastr.error('Failed to sync with SAP');
            }
        });
    },
    
    // Auto-check SAP status after transaction creation
    autoCheckAfterSale: function(transactionId) {
        // Wait 2 seconds then check status
        setTimeout(function() {
            SAPIntegration.checkSyncStatus(transactionId);
        }, 2000);
    }
};

console.log('✅ Table Copy Feature loaded successfully');