/**
 * SAP Sync Batched Processing
 * يدير المزامنة على دفعات صغيرة للمتصفح
 */

class SapSyncBatched {
    constructor() {
        this.isRunning = false;
        this.currentProgress = 0;
        this.totalProcessed = 0;
        this.errors = [];
        this.progressCallback = null;
        this.csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
    }

    /**
     * بدء مزامنة جهات الاتصال على دفعات
     */
    async syncContactsBatched(options = {}) {
        const defaultOptions = {
            fullSync: false,
            batchSize: 100,
            maxBatches: 5,
            autoRetry: true,
            maxRetries: 3
        };
        
        const config = { ...defaultOptions, ...options };
        
        this.isRunning = true;
        this.currentProgress = 0;
        this.totalProcessed = 0;
        this.errors = [];
        
        try {
            this.updateProgress('بدء مزامنة جهات الاتصال...', 0);
            
            let hasMore = true;
            let attempt = 0;
            
            while (hasMore && this.isRunning && attempt < 50) { // حد أقصى 50 دفعة
                attempt++;
                
                this.updateProgress(`معالجة الدفعة ${attempt}...`, this.currentProgress);
                
                try {
                    const response = await this.makeRequest('/sap/sync/contacts-batched', {
                        full_sync: config.fullSync,
                        batch_size: config.batchSize,
                        max_batches: config.maxBatches
                    });
                    
                    if (response.success) {
                        const data = response.data;
                        this.totalProcessed += data.total;
                        hasMore = data.has_more;
                        
                        this.updateProgress(
                            `تم معالجة ${this.totalProcessed} جهة اتصال حتى الآن...`,
                            Math.min(90, attempt * 2)
                        );
                        
                        // تأخير قصير بين الدفعات
                        if (hasMore) {
                            await this.delay(1000);
                        }
                    } else {
                        throw new Error(response.message || 'خطأ غير معروف');
                    }
                } catch (error) {
                    this.errors.push(`خطأ في الدفعة ${attempt}: ${error.message}`);
                    
                    if (config.autoRetry && attempt <= config.maxRetries) {
                        this.updateProgress(`إعادة محاولة الدفعة ${attempt}...`, this.currentProgress);
                        await this.delay(2000);
                        continue;
                    } else {
                        throw error;
                    }
                }
            }
            
            this.updateProgress(`انتهت المزامنة بنجاح! تم معالجة ${this.totalProcessed} جهة اتصال`, 100);
            
            return {
                success: true,
                totalProcessed: this.totalProcessed,
                batches: attempt,
                errors: this.errors
            };
            
        } catch (error) {
            this.updateProgress(`فشلت المزامنة: ${error.message}`, -1);
            throw error;
        } finally {
            this.isRunning = false;
        }
    }

    /**
     * بدء مزامنة المنتجات على دفعات
     */
    async syncProductsBatched(options = {}) {
        const defaultOptions = {
            fullSync: false,
            batchSize: 50,
            maxBatches: 3,
            autoRetry: true,
            maxRetries: 3
        };
        
        const config = { ...defaultOptions, ...options };
        
        this.isRunning = true;
        this.currentProgress = 0;
        this.totalProcessed = 0;
        this.errors = [];
        
        try {
            this.updateProgress('بدء مزامنة المنتجات...', 0);
            
            let hasMore = true;
            let attempt = 0;
            
            while (hasMore && this.isRunning && attempt < 100) { // حد أقصى 100 دفعة للمنتجات
                attempt++;
                
                this.updateProgress(`معالجة دفعة المنتجات ${attempt}...`, this.currentProgress);
                
                try {
                    const response = await this.makeRequest('/sap/sync/products-batched', {
                        full_sync: config.fullSync,
                        batch_size: config.batchSize,
                        max_batches: config.maxBatches
                    });
                    
                    if (response.success) {
                        const data = response.data;
                        this.totalProcessed += data.total;
                        hasMore = data.has_more;
                        
                        this.updateProgress(
                            `تم معالجة ${this.totalProcessed} منتج حتى الآن...`,
                            Math.min(90, attempt * 1)
                        );
                        
                        // تأخير أطول بين دفعات المنتجات
                        if (hasMore) {
                            await this.delay(2000);
                        }
                    } else {
                        throw new Error(response.message || 'خطأ غير معروف');
                    }
                } catch (error) {
                    this.errors.push(`خطأ في دفعة المنتجات ${attempt}: ${error.message}`);
                    
                    if (config.autoRetry && attempt <= config.maxRetries) {
                        this.updateProgress(`إعادة محاولة دفعة المنتجات ${attempt}...`, this.currentProgress);
                        await this.delay(3000);
                        continue;
                    } else {
                        throw error;
                    }
                }
            }
            
            this.updateProgress(`انتهت مزامنة المنتجات بنجاح! تم معالجة ${this.totalProcessed} منتج`, 100);
            
            return {
                success: true,
                totalProcessed: this.totalProcessed,
                batches: attempt,
                errors: this.errors
            };
            
        } catch (error) {
            this.updateProgress(`فشلت مزامنة المنتجات: ${error.message}`, -1);
            throw error;
        } finally {
            this.isRunning = false;
        }
    }

    /**
     * إعادة تعيين مؤشرات المزامنة
     */
    async resetCursors(type = 'all') {
        try {
            const response = await this.makeRequest('/sap/sync/reset-cursors', { type });
            return response;
        } catch (error) {
            throw new Error(`فشل في إعادة تعيين المؤشرات: ${error.message}`);
        }
    }

    /**
     * إيقاف المزامنة
     */
    stop() {
        this.isRunning = false;
        this.updateProgress('تم إيقاف المزامنة بواسطة المستخدم', -1);
    }

    /**
     * تحديث شريط التقدم
     */
    updateProgress(message, percent) {
        this.currentProgress = percent;
        
        if (this.progressCallback) {
            this.progressCallback(message, percent);
        }
        
        console.log(`SAP Sync: ${message} (${percent}%)`);
    }

    /**
     * تعيين دالة callback لتحديثات التقدم
     */
    onProgress(callback) {
        this.progressCallback = callback;
    }

    /**
     * إرسال طلب HTTP
     */
    async makeRequest(url, data) {
        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': this.csrfToken,
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify(data)
        });

        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }

        return await response.json();
    }

    /**
     * تأخير
     */
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}

// إنشاء instance عامة
window.sapSyncBatched = new SapSyncBatched();

// دالة مساعدة لبدء مزامنة جهات الاتصال
window.startContactsSync = function(options = {}) {
    return window.sapSyncBatched.syncContactsBatched(options);
};

// دالة مساعدة لبدء مزامنة المنتجات
window.startProductsSync = function(options = {}) {
    return window.sapSyncBatched.syncProductsBatched(options);
};

// دالة مساعدة لإيقاف المزامنة
window.stopSync = function() {
    window.sapSyncBatched.stop();
};

// دالة مساعدة لإعادة تعيين المؤشرات
window.resetSyncCursors = function(type = 'all') {
    return window.sapSyncBatched.resetCursors(type);
};

console.log('SAP Sync Batched module loaded successfully'); 